package PT::PostalCode;

use 5.006;
use strict;
use warnings;

require Exporter;

our @ISA = qw(Exporter);

our %EXPORT_TAGS = ( 'all' => [ qw(
        code_is_from_area code_is_from_subarea code_is_from
        range_from_subarea
        code_is_valid areas_of_code subareas_of_code
) ] );

our @EXPORT_OK = ( @{ $EXPORT_TAGS{'all'} } );

our @EXPORT = qw(
        code_is_from_area code_is_from_subarea code_is_from
        range_from_subarea
        code_is_valid areas_of_code subareas_of_code
);

our $VERSION = '0.03';

=head1 NAME

PT::PostalCode - Validates Portuguese postal codes

=head1 SYNOPSIS

  use PT::PostalCode;

  validate($postalcode);

  code_is_from_area($postalcode,$city);
  code_is_from_subarea($postalcode,$district);

  code_is_from($postalcode,$district,$city);

  range_from_subarea($district);

  code_is_valid($code);
  areas_of_code($code);
  subareas_of_code($code);

=cut

my (%codes,%range);

BEGIN {
  %range = (

        'Santarm' => {
          'Santarm'                         => [ '2000', '2150' ],
          'Goleg'                         => [ '2150', '2150' ],
          'Rio Maior'                         => [ '2040', '2040' ],
          'Cartaxo'                         => [ '2070', '2070' ],
          'Mao'                         => [ '6120', '6120' ],
          'Torres Novas'                 => [ '2350', '2354' ],
          'Salvaterra de Magos'                 => [ '2120', '2125' ],
          'Sardoal'                         => [ '2230', '2230' ],
          'Ferreira do Zzere'                 => [ '2240', '2240' ],
          'Almeirim'                         => [ '2080', '2080' ],
          'Alpiara'                         => [ '2090', '2090' ],
          'Constncia'                         => [ '2250', '2250' ],
          'Vila Nova da Barquinha'         => [ '2260', '2260' ],
          'Alcanena'                         => [ '2380', '2395' ],
          'Chamusca'                         => [ '2140', '2140' ],
          'Benavente'                         => [ '2120', '2890' ],
          'Entroncamento'                 => [ '2330', '2330' ],
          'Tomar'                         => [ '2300', '2305' ],
          'Coruche'                         => [ '2100', '2100' ],
          'Abrantes'                         => [ '2200', '2230' ],
          'Ourm'                         => [ '2435', '2495' ],
        },

        'Lisboa' => {
          'Lisboa'                         => [ '1000', '1998' ],
          'Odivelas'                         => [ '1675', '2675' ],
          'Cadaval'                         => [ '2550', '2550' ],
          'Mafra'                         => [ '2640', '2669' ],
          'Sintra'                         => [ '2605', '2749' ],
          'Alenquer'                         => [ '2580', '2581' ],
          'Arruda dos Vinhos'                 => [ '2630', '2630' ],
          'Azambuja'                         => [ '2050', '2065' ],
          'Sobral Monte Agrao'                 => [ '2590', '2590' ],
          'Oeiras'                         => [ '1495', '2799' ],
          'Lourinh'                         => [ '2530', '2530' ],
          'Amadora'                         => [ '2610', '2724' ],
          'Vila Franca de Xira'                 => [ '2600', '2626' ],
          'Cascais'                         => [ '2645', '2789' ],
          'Torres Vedras'                 => [ '2560', '2565' ],
          'Loures'                         => [ '1885', '2699' ],
        },

        'Ilha da Madeira' => {
          'Ponta do Sol'                 => [ '9360', '9360' ],
          'Calheta (Madeira)'                 => [ '9370', '9385' ],
          'Santana'                         => [ '9230', '9230' ],
          'Santa Cruz'                         => [ '9100', '9135' ],
          'So Vicente'                         => [ '9240', '9240' ],
          'Machico'                         => [ '9200', '9225' ],
          'Ribeira Brava'                 => [ '9350', '9350' ],
          'Cmara de Lobos'                 => [ '9030', '9325' ],
          'Funchal'                         => [ '9000', '9064' ],
          'Porto Moniz'                         => [ '9270', '9270' ],
        },

        'vora' => {
          'Portel'                         => [ '7220', '7220' ],
          'Arraiolos'                         => [ '7040', '7040' ],
          'vora'                         => [ '7000', '7200' ],
          'Estremoz'                         => [ '7100', '7100' ],
          'Vila Viosa'                         => [ '7160', '7160' ],
          'Viana do Alentejo'                 => [ '7090', '7090' ],
          'Vendas Novas'                 => [ '2965', '7080' ],
          'Redondo'                         => [ '7170', '7200' ],
          'Borba'                         => [ '7150', '7150' ],
          'Alandroal'                         => [ '7200', '7250' ],
          'Reguengos de Monsaraz'         => [ '7200', '7200' ],
          'Mouro'                         => [ '7240', '7240' ],
          'Montemor-o-Novo'                 => [ '7050', '7050' ],
          'Mora'                         => [ '7490', '7490' ],
        },

        'Leiria' =>  {
           'bidos'                         => [ '2510', '2510' ],
           'Nazar'                         => [ '2450', '2450' ],
           'Leiria'                         => [ '2400', '2495' ],
           'Castanheira de Pra'         => [ '3280', '3280' ],
           'Bombarral'                         => [ '2540', '2540' ],
           'Alvaizere'                         => [ '3250', '3260' ],
           'Pombal'                         => [ '3100', '3105' ],
           'Peniche'                         => [ '2520', '2525' ],
           'Alcobaa'                         => [ '2445', '2475' ],
           'Marinha Grande'                 => [ '2430', '2445' ],
           'Porto de Ms'                 => [ '2480', '2485' ],
           'Batalha'                         => [ '2440', '2495' ],
           'Ansio'                         => [ '3240', '3240' ],
           'Pedrgo Grande'                 => [ '3270', '3270' ],
           'Figueir dos Vinhos'         => [ '3260', '3260' ],
           'Caldas da Rainha'                 => [ '2500', '2500' ],
         },

         'Ilha das Flores' =>  {
           'Lajes das Flores'                 => [ '9960', '9960' ],
           'Santa Cruz das Flores'         => [ '9970', '9970' ],
         },

         'Ilha do Pico' =>  {
           'Lajes do Pico'                 => [ '9930', '9930' ],
           'So Roque do Pico'                 => [ '9940', '9940' ],
           'Madalena'                         => [ '9950', '9950' ],
         },

         'Aveiro' =>  {
           'Mealhada'                         => [ '3050', '3050' ],
           'Ovar'                         => [ '3880', '3885' ],
           'Murtosa'                         => [ '3870', '3870' ],
           'Oliveira de Azemeis'         => [ '3700', '3720' ],
           'Aveiro'                         => [ '3800', '3814' ],
           'Albergaria-a-Velha'                 => [ '3850', '3850' ],
           'Oliveira do Bairro'                 => [ '3770', '3770' ],
           'Estarreja'                         => [ '3860', '3865' ],
           'So Joo da Madeira'         => [ '3700', '3701' ],
           'Espinho'                         => [ '4500', '4504' ],
           'Vale de Cambra'                 => [ '3730', '3730' ],
           'Vagos'                         => [ '3840', '3840' ],
           'lhavo'                         => [ '3830', '3830' ],
           'Anadia'                         => [ '3780', '3780' ],
           'Castelo de Paiva'                 => [ '4550', '4550' ],
           'gueda'                         => [ '3750', '3750' ],
           'Sever do Vouga'                 => [ '3740', '3740' ],
           'Arouca'                         => [ '4540', '4540' ],
           'Santa Maria da Feira'         => [ '3700', '4535' ],
         },

         'Bragana' =>  {
           'Vila Flor'                         => [ '5360', '5360' ],
           'Carrazeda de Ansies'         => [ '5140', '5140' ],
           'Miranda do Douro'                 => [ '5210', '5225' ],
           'Bragana'                         => [ '5300', '5301' ],
           'Macedo de Cavaleiros'         => [ '5340', '5340' ],
           'Torre de Moncorvo'                 => [ '5160', '5160' ],
           'Mirandela'                         => [ '5370', '5385' ],
           'Mogadouro'                         => [ '5200', '5350' ],
           'Vimioso'                         => [ '5230', '5230' ],
           'Freixo Espada  Cinta'         => [ '5180', '5180' ],
           'Vinhais'                         => [ '5320', '5335' ],
           'Alfandega da F'                 => [ '5350', '5350' ],
         },

         'Ilha de So Miguel' =>  {
           'Ribeira Grande'                 => [ '9600', '9625' ],
           'Vila Franca do Campo'         => [ '9680', '9680' ],
           'Povoao'                         => [ '9650', '9675' ],
           'Ponta Delgada'                 => [ '9500', '9555' ],
           'Lagoa (So Miguel)'                 => [ '9560', '9560' ],
           'Nordeste'                         => [ '9630', '9630' ],
         },

         'Ilha do Corvo' =>  {
           'Corvo'                         => [ '9980', '9980' ],
         },

         'Ilha de So Jorge' =>  {
           'Calheta (So Jorge)'         => [ '9850', '9875' ],
           'Velas'                         => [ '9800', '9800' ],
         },

         'Viana do Castelo' =>  {
           'Valena'                         => [ '4930', '4930' ],
           'Ponte de Lima'                 => [ '4990', '4990' ],
           'Mono'                         => [ '4950', '4950' ],
           'Vila Nova de Cerveira'         => [ '4920', '4920' ],
           'Ponte da Barca'                 => [ '4980', '4980' ],
           'Caminha'                         => [ '4910', '4910' ],
           'Arcos de Valdevez'                 => [ '4970', '4974' ],
           'Paredes de Coura'                 => [ '4940', '4940' ],
           'Viana do Castelo'                 => [ '4900', '4935' ],
           'Melgao'                         => [ '4960', '4960' ],
         },

         'Portalegre' =>  {
           'Castelo de Vide'                 => [ '7320', '7320' ],
           'Arronches'                         => [ '7340', '7340' ],
           'Monforte'                         => [ '7450', '7450' ],
           'Elvas'                         => [ '7350', '7350' ],
           'Sousel'                         => [ '7470', '7470' ],
           'Nisa'                         => [ '6050', '6050' ],
           'Crato'                         => [ '7430', '7430' ],
           'Gavio'                         => [ '6040', '6040' ],
           'Fronteira'                         => [ '7460', '7460' ],
           'Ponte de Sor'                 => [ '7400', '7425' ],
           'Campo Maior'                 => [ '7370', '7370' ],
           'Avis'                         => [ '7480', '7480' ],
           'Portalegre'                         => [ '7300', '7301' ],
           'Marvo'                         => [ '7330', '7330' ],
           'Alter do Cho'                 => [ '7440', '7440' ],
         },

         'Beja' =>  {
           'Serpa'                         => [ '7830', '7830' ],
           'Almodvar'                         => [ '7700', '7700' ],
           'Ourique'                         => [ '7670', '7670' ],
           'Mrtola'                         => [ '7750', '7750' ],
           'Alvito'                         => [ '7920', '7920' ],
           'Vidigueira'                         => [ '7960', '7960' ],
           'Aljustrel'                         => [ '7600', '7600' ],
           'Ferreira do Alentejo'         => [ '7900', '7900' ],
           'Barrancos'                         => [ '7230', '7230' ],
           'Cuba'                         => [ '7940', '7940' ],
           'Castro Verde'                 => [ '7780', '7780' ],
           'Moura'                         => [ '7860', '7885' ],
           'Beja'                         => [ '7800', '7801' ],
           'Odemira'                         => [ '7630', '7665' ],
         },

         'Viseu' =>  {
           'Penalva do Castelo'                 => [ '3550', '5385' ],
           'Mortgua'                         => [ '3450', '3450' ],
           'Sernancelhe'                 => [ '3620', '3640' ],
           'Nelas'                         => [ '3520', '3525' ],
           'So Joo da Pesqueira'         => [ '5130', '5130' ],
           'Castro Daire'                 => [ '3600', '3600' ],
           'Cinfes'                         => [ '4690', '4690' ],
           'Mangualde'                         => [ '3530', '3534' ],
           'Santa Comba Do'                 => [ '3440', '3440' ],
           'Armamar'                         => [ '5110', '5114' ],
           'Viseu'                         => [ '3500', '3519' ],
           'Tarouca'                         => [ '3515', '3610' ],
           'Vouzela'                         => [ '3670', '3670' ],
           'So Pedro do Sul'                 => [ '3660', '3660' ],
           'Sto'                         => [ '3560', '3650' ],
           'Lamego'                         => [ '5100', '5100' ],
           'Carregal do Sal'                 => [ '3430', '3430' ],
           'Moimenta da Beira'                 => [ '3620', '3620' ],
           'Penedono'                         => [ '3630', '3630' ],
           'Vila Nova de Paiva'                 => [ '3650', '3650' ],
           'Resende'                         => [ '4660', '4660' ],
           'Tondela'                         => [ '3460', '3475' ],
           'Tabuao'                         => [ '5120', '5120' ],
           'Oliveira de Frades'                 => [ '3475', '3680' ],
         },

         'Ilha de Santa Maria' =>  {
           'Vila do Porto'                 => [ '9580', '9580' ],
         },

         'Faro' =>  {
           'Vila do Bispo'                 => [ '8650', '8650' ],
           'Albufeira'                         => [ '8200', '8201' ],
           'Portimo'                         => [ '8500', '8501' ],
           'Tavira'                         => [ '8800', '8801' ],
           'Silves'                         => [ '8300', '8375' ],
           'Lagos'                         => [ '8600', '8601' ],
           'Aljezur'                         => [ '8670', '8670' ],
           'Vila Real de Santo Antnio'         => [ '8900', '8900' ],
           'So Brs de Alportel'         => [ '8150', '8150' ],
           'Alcoutim'                         => [ '8970', '8970' ],
           'Castro Marim'                 => [ '8950', '8950' ],
           'Lagoa (Algarve)'                 => [ '8400', '8401' ],
           'Loul'                         => [ '8100', '8135' ],
           'Faro'                         => [ '8000', '8700' ],
           'Monchique'                         => [ '8550', '8550' ],
           'Olho'                         => [ '8700', '8700' ],
         },

         'Coimbra' =>  {
           'Penacova'                         => [ '3360', '3360' ],
           'Figueira da Foz'                 => [ '3080', '3094' ],
           'Cantanhede'                         => [ '3060', '3060' ],
           'Pampilhosa da Serra'         => [ '3320', '3320' ],
           'Penela'                         => [ '3230', '3230' ],
           'Soure'                         => [ '3130', '3130' ],
           'Gis'                         => [ '3330', '3330' ],
           'Lous'                         => [ '3200', '3200' ],
           'Arganil'                         => [ '3300', '6285' ],
           'Coimbra'                         => [ '3000', '3049' ],
           'Montemor-o-Velho'                 => [ '3140', '3140' ],
           'Condeixa-a-Nova'                 => [ '3150', '3150' ],
           'Oliveira do Hospital'         => [ '3400', '3405' ],
           'Tbua'                         => [ '3420', '3420' ],
           'Vila Nova de Poiares'         => [ '3350', '3350' ],
           'Miranda do Corvo'                 => [ '3220', '3220' ],
           'Mira'                         => [ '3070', '3070' ],
         },

         'Ilha da Graciosa' =>  {
           'Santa Cruz da Graciosa'         => [ '9880', '9880' ],
         },

         'Vila Real' =>  {
           'Montalegre'                         => [ '5470', '5470' ],
           'Boticas'                         => [ '5460', '5460' ],
           'Santa Marta de Penaguio'         => [ '5030', '5030' ],
           'Vila Real'                         => [ '5000', '5004' ],
           'Alij'                         => [ '5070', '5085' ],
           'Ribeira de Pena'                 => [ '4870', '4870' ],
           'Chaves'                         => [ '5400', '5425' ],
           'Meso Frio'                         => [ '5040', '5040' ],
           'Vila Pouca de Aguiar'         => [ '5450', '5450' ],
           'Peso da Rgua'                 => [ '5040', '5054' ],
           'Valpaos'                         => [ '5430', '5445' ],
           'Mura'                         => [ '5090', '5090' ],
           'Sabrosa'                         => [ '5060', '5085' ],
           'Mondim de Basto'                 => [ '4880', '4880' ],
         },

         'Braga' =>  {
           'Terras de Bouro'                 => [ '4840', '4845' ],
           'Braga'                         => [ '4700', '4719' ],
           'Vizela'                         => [ '4620', '4815' ],
           'Vila Verde'                         => [ '4730', '4730' ],
           'Amares'                         => [ '4720', '4720' ],
           'Barcelos'                         => [ '4740', '4905' ],
           'Cabeceiras de Basto'         => [ '4860', '4860' ],
           'Vieira do Minho'                 => [ '4850', '4850' ],
           'Vila Nova de Famalico'         => [ '4760', '4775' ],
           'Esposende'                         => [ '4740', '4740' ],
           'Pvoa de Lanhoso'                 => [ '4830', '4830' ],
           'Fafe'                         => [ '4820', '4824' ],
           'Guimares'                         => [ '4765', '4839' ],
           'Celorico de Basto'                 => [ '4615', '4890' ],
         },

         'Setbal' =>  {
           'Moita'                         => [ '2835', '2864' ],
           'Setbal'                         => [ '2900', '2925' ],
           'Sesimbra'                         => [ '2970', '2975' ],
           'Palmela'                         => [ '2950', '2965' ],
           'Alcochete'                         => [ '2890', '2894' ],
           'Sines'                         => [ '7520', '7555' ],
           'Seixal'                         => [ '2840', '2865' ],
           'Montijo'                         => [ '2100', '2985' ],
           'Alccer do Sal'                 => [ '7580', '7595' ],
           'Barreiro'                         => [ '2830', '2835' ],
           'Grndola'                         => [ '7570', '7570' ],
           'Santiago do Cacm'                 => [ '7500', '7565' ],
           'Almada'                         => [ '2800', '2829' ],
         },

         'Ilha do Faial' =>  {
           'Horta'                         => [ '9900', '9901' ],
         },

         'Ilha Terceira' =>  {
           'Angra do Herosmo'                 => [ '9700', '9701' ],
           'Praia da Vitria'                 => [ '9760', '9760' ],
         },

         'Ilha de Porto Santo' =>  {
           'Porto Santo'                 => [ '9400', '9400' ],
         },

         'Porto' =>  {
           'Baio'                         => [ '4640', '5040' ],
           'Pvoa de Varzim'                 => [ '4490', '4570' ],
           'Valongo'                         => [ '4440', '4445' ],
           'Santo Tirso'                 => [ '4780', '4825' ],
           'Trofa'                         => [ '4745', '4785' ],
           'Marco de Canaveses'                 => [ '4575', '4635' ],
           'Maia'                         => [ '4425', '4479' ],
           'Vila Nova de Gaia'                 => [ '4400', '4434' ],
           'Matosinhos'                         => [ '4450', '4465' ],
           'Amarante'                         => [ '4600', '4615' ],
           'Penafiel'                         => [ '4560', '4575' ],
           'Paredes'                         => [ '4580', '4585' ],
           'Gondomar'                         => [ '4420', '4515' ],
           'Paos de Ferreira'                 => [ '4590', '4595' ],
           'Vila do Conde'                 => [ '4480', '4486' ],
           'Felgueiras'                         => [ '4610', '4815' ],
           'Porto'                         => [ '4000', '4369' ],
           'Lousada'                         => [ '4620', '4620' ],
         },

         'Castelo Branco' =>  {
           'Vila de Rei'                 => [ '6110', '6110' ],
           'Proena-a-Nova'                 => [ '6150', '6150' ],
           'Oleiros'                         => [ '6160', '6185' ],
           'Vila Velha de Rodo'         => [ '6030', '6030' ],
           'Fundo'                         => [ '6005', '6230' ],
           'Covilh'                         => [ '6200', '6230' ],
           'Penamacor'                         => [ '6090', '6320' ],
           'Castelo Branco'                 => [ '6000', '6005' ],
           'Sert'                         => [ '6100', '6100' ],
           'Belmonte'                         => [ '6250', '6250' ],
           'Idanha-a-Nova'                 => [ '6060', '6060' ],
         },

         'Guarda' =>  {
           'Pinhel'                         => [ '6400', '6400' ],
           'Meda'                         => [ '6430', '6430' ],
           'Sabugal'                         => [ '6250', '6324' ],
           'Gouveia'                         => [ '6290', '6290' ],
           'Seia'                         => [ '6270', '6285' ],
           'Fornos de Algodres'                 => [ '6370', '6370' ],
           'Vila Nova de Foz Coa'         => [ '5150', '5155' ],
           'Aguiar da Beira'                 => [ '3570', '3570' ],
           'Celorico da Beira'                 => [ '6360', '6360' ],
           'Trancoso'                         => [ '3640', '6420' ],
           'Almeida'                         => [ '6350', '6355' ],
           'Guarda'                         => [ '6300', '6301' ],
           'Figueira de Castelo Rodrigo' => [ '6440', '6440' ],
           'Manteigas'                         => [ '6260', '6300' ],
         },

  );

  %codes = (
    'Ilha da Madeira' => {
      'Calheta (Madeira)' => {
        'Lombo das Laranjeiras' => {
          '9370' => [
            [ '119', '122', 'CALHETA (MADEIRA)' ] ] },
        'Eirinhas' => {
          '9370' => [
            [ '302', '307', 'FAJ DA OVELHA' ] ] },
        'Lameiro' => {
          '9370' => [
            [ '208', '208', 'ESTREITO DA CALHETA' ] ] },
        'Fonte da Bugia' => {
          '9370' => [
            [ '024', '028', 'ARCO DA CALHETA' ] ] },
        'Loreto' => {
          '9370' => [
            [ '032', '038', 'ARCO DA CALHETA' ] ] },
        'Pombal' => {
          '9370' => [
            [ '040', '040', 'ARCO DA CALHETA' ] ] },
        'Florenas' => {
          '9370' => [
            [ '021', '022', 'ARCO DA CALHETA' ] ] },
        'Ribeira da Vaca' => {
          '9385' => [
            [ '100', '100', 'PONTA DO PARGO' ] ] },
        'Igreja' => {
          '9370' => [
            [ '402', '404', 'JARDIM DO MAR' ],
            [ '501', '505', 'PAUL DO MAR' ] ] },
        'Achada' => {
          '9370' => [
            [ '101', '106', 'CALHETA (MADEIRA)' ] ] },
        'Fazendinha' => {
          '9370' => [
            [ '203', '206', 'ESTREITO DA CALHETA' ] ] },
        'Castanheiros' => {
          '9370' => [
            [ '201', '201', 'ESTREITO DA CALHETA' ] ] },
        'Salo de Baixo' => {
          '9385' => [
            [ '110', '110', 'PONTA DO PARGO' ] ] },
        'Pedregal' => {
          '9385' => [
            [ '090', '090', 'PONTA DO PARGO' ] ] },
        'Corujeira' => {
          '9385' => [
            [ '040', '040', 'PONTA DO PARGO' ] ] },
        'Amparo' => {
          '9385' => [
            [ '020', '020', 'PONTA DO PARGO' ] ] },
        'Carreira' => {
          '9370' => [
            [ '601', '610', 'PRAZERES' ] ] },
        'Cabo' => {
          '9385' => [
            [ '030', '030', 'PONTA DO PARGO' ] ] },
        'Lombadinha' => {
          '9385' => [
            [ '060', '060', 'PONTA DO PARGO' ] ] },
        'Achada de Santo Anto' => {
          '9370' => [
            [ '011', '019', 'ARCO DA CALHETA' ] ] },
        'Canto da Levada' => {
          '9370' => [
            [ '108', '116', 'CALHETA (MADEIRA)' ] ] },
        'Salo de Cima' => {
          '9385' => [
            [ '120', '120', 'PONTA DO PARGO' ] ] },
        'Lombo Queimado' => {
          '9385' => [
            [ '080', '080', 'PONTA DO PARGO' ] ] },
        'Lombada Velha' => {
          '9385' => [
            [ '050', '050', 'PONTA DO PARGO' ] ] },
        'Moinhos Abaixo' => {
          '9370' => [
            [ '216', '217', 'ESTREITO DA CALHETA' ] ] },
        'Reis Abaixo' => {
          '9370' => [
            [ '219', '221', 'ESTREITO DA CALHETA' ] ] },
        'Serrado' => {
          '9385' => [
            [ '130', '130', 'PONTA DO PARGO' ] ] },
        'Serres Abaixo' => {
          '9370' => [
            [ '223', '224', 'ESTREITO DA CALHETA' ] ] },
        'Lombo' => {
          '9385' => [
            [ '070', '070', 'PONTA DO PARGO' ] ] },
        'Massapez' => {
          '9370' => [
            [ '309', '314', 'FAJ DA OVELHA' ] ] },
        'Miradouro' => {
          '9370' => [
            [ '125', '133', 'CALHETA (MADEIRA)' ] ] }
              },
      'Ponta do Sol' => {
        'Ponta do Sol' => {
          '9360' => [
            [ '101', '164', 'PONTA DO SOL' ],
            [ '166', '182', 'PONTA DO SOL' ],
            [ '184', '221', 'PONTA DO SOL' ] ] },
        'Canhas' => {
          '9360' => [
            [ '011', '076', 'CANHAS' ] ] },
        'Lombo So Joo' => {
          '9360' => [
            [ '223', '223', 'PONTA DO SOL' ] ] },
        'Banda Dlem' => {
          '9360' => [
            [ '401', '407', 'MADALENA DO MAR' ] ] }
               },
      'Santana' => {
        'Achada da Cruz' => {
          '9230' => [
            [ '080', '117', 'SANTANA' ] ] },
        'Achada do Folhadal' => {
          '9230' => [
            [ '200', '211', 'SO ROQUE DO FAIAL' ] ] },
        'Achada do Marques' => {
          '9230' => [
            [ '230', '239', 'ILHA STN' ] ] },
        'Achada de Antnio Teixeira' => {
          '9230' => [
            [ '140', '168', 'SO JORGE STN' ] ] },
        'gua de Alto' => {
          '9230' => [
            [ '040', '067', 'FAIAL' ] ] },
        'Vila' => {
          '9230' => [
            [ '119', '127', 'SANTANA' ] ] },
        'Arco de So Jorge' => {
          '9230' => [
            [ '011', '020', 'ARCO DE SO JORGE' ] ] }
                },
      'Santa Cruz' => {
        'Pico do Infante' => {
          '9135' => [
            [ '220', '220', 'CAMACHA' ] ] },
        'Achada da Rocha' => {
          '9100' => [
            [ '011', '024', 'GAULA' ] ] },
        'Carreiras' => {
          '9135' => [
            [ '080', '080', 'CAMACHA' ] ] },
        'Achadinha' => {
          '9135' => [
            [ '030', '030', 'CAMACHA' ] ] },
        'Cerquinha' => {
          '9135' => [
            [ '100', '100', 'CAMACHA' ] ] },
        'Canio' => {
          '9125' => [
            [ '001', '039', 'CANIO' ],
            [ '042', '055', 'CANIO' ],
            [ '057', '065', 'CANIO' ],
            [ '067', '087', 'CANIO' ],
            [ '089', '093', 'CANIO' ],
            [ '095', '098', 'CANIO' ],
            [ '100', '134', 'CANIO' ],
            [ '136', '170', 'CANIO' ],
            [ '172', '202', 'CANIO' ],
            [ '204', '228', 'CANIO' ],
            [ '230', '235', 'CANIO' ],
            [ '237', '242', 'CANIO' ],
            [ '244', '250', 'CANIO' ],
            [ '252', '255', 'CANIO' ] ] },
        'Camacha' => {
          '9135' => [
            [ '050', '062', 'CAMACHA' ] ] },
        'Pico da Silva' => {
          '9135' => [
            [ '210', '210', 'CAMACHA' ] ] },
        'Lombo Barreto' => {
          '9135' => [
            [ '170', '170', 'CAMACHA' ] ] },
        'Eira de Fora' => {
          '9135' => [
            [ '120', '120', 'CAMACHA' ] ] },
        'So Fernando' => {
          '9100' => [
            [ '180', '181', 'SANTA CRUZ' ] ] },
        'Rocho do Meio' => {
          '9135' => [
            [ '330', '330', 'CAMACHA' ] ] },
        'Santa Cruz' => {
          '9100' => [
            [ '150', '176', 'SANTA CRUZ' ] ] },
        'Eiras' => {
          '9100' => [
            [ '190', '223', 'SANTA CRUZ' ] ] },
        'Eira de Dentro' => {
          '9135' => [
            [ '110', '110', 'CAMACHA' ] ] },
        'Rocho de Baixo' => {
          '9135' => [
            [ '310', '310', 'CAMACHA' ] ] },
        'Piquetes' => {
          '9135' => [
            [ '240', '240', 'CAMACHA' ] ] },
        'Lamarejas' => {
          '9100' => [
            [ '121', '127', 'SANTA CRUZ' ] ] },
        'Gaula' => {
          '9100' => [
            [ '031', '046', 'GAULA' ] ] },
        'Casais d\' Alm' => {
          '9135' => [
            [ '351', '437', 'CAMACHA' ] ] },
        'Eiras Velhas' => {
          '9100' => [
            [ '116', '119', 'SANTA CRUZ' ] ] },
        'Pinheirinho' => {
          '9135' => [
            [ '230', '230', 'CAMACHA' ] ] },
        'Santa Catarina' => {
          '9100' => [
            [ '101', '103', 'SANTA CRUZ' ] ] },
        'Cano' => {
          '9100' => [
            [ '108', '114', 'SANTA CRUZ' ] ] },
        'Levada do Pico' => {
          '9135' => [
            [ '160', '160', 'CAMACHA' ] ] },
        'Lombo Centeio' => {
          '9135' => [
            [ '180', '180', 'CAMACHA' ] ] },
        'Aeroporto Santa Catarina' => {
          '9100' => [
            [ '105', '106', 'SANTA CRUZ' ] ] },
        'Lombo dos Policarpos' => {
          '9135' => [
            [ '190', '190', 'CAMACHA' ] ] },
        'Pedra Mole' => {
          '9100' => [
            [ '133', '134', 'SANTA CRUZ' ] ] },
        'Ribeiro Frances' => {
          '9100' => [
            [ '143', '146', 'SANTA CRUZ' ] ] },
        'Ribeiro Fernando' => {
          '9135' => [
            [ '280', '280', 'CAMACHA' ] ] },
        'Serrado da Ribeira' => {
          '9100' => [
            [ '184', '188', 'SANTA CRUZ' ] ] },
        'Morena' => {
          '9100' => [
            [ '131', '131', 'SANTA CRUZ' ] ] },
        'Salgados' => {
          '9135' => [
            [ '340', '340', 'CAMACHA' ] ] },
        'Lajes' => {
          '9100' => [
            [ '061', '079', 'GAULA' ] ] },
        'Rego' => {
          '9100' => [
            [ '136', '141', 'SANTA CRUZ' ] ] },
        'Figueirinha' => {
          '9135' => [
            [ '130', '130', 'CAMACHA' ] ] },
        'Malcavada' => {
          '9100' => [
            [ '129', '129', 'SANTA CRUZ' ] ] },
        'Boqueiro' => {
          '9135' => [
            [ '040', '040', 'CAMACHA' ] ] },
        'Ribeirinho' => {
          '9135' => [
            [ '270', '270', 'CAMACHA' ] ] },
        'Rocho de Cima' => {
          '9135' => [
            [ '320', '320', 'CAMACHA' ] ] },
        'Achada do Barro' => {
          '9100' => [
            [ '251', '267', 'STO ANTNIO DA SERRA SCR' ] ] }
             },
      'So Vicente' => {
        'Lugar' => {
          '9240' => [
            [ '110', '122', 'PONTA DELGADA SVC' ] ] },
        'Venda Nova - Faj da Areia' => {
          '9240' => [
            [ '050', '050', 'SO VICENTE' ] ] },
        'Esmoitada da Ribeira' => {
          '9240' => [
            [ '022', '023', 'BOA VENTURA' ] ] },
        'P da Corrida' => {
          '9240' => [
            [ '038', '040', 'SO VICENTE' ] ] },
        'Laranjal' => {
          '9240' => [
            [ '031', '031', 'SO VICENTE' ] ] },
        'Aougue' => {
          '9240' => [
            [ '101', '108', 'PONTA DELGADA SVC' ] ] },
        'Limoeiro' => {
          '9240' => [
            [ '034', '034', 'SO VICENTE' ] ] },
        'Baia - Faj da Areia' => {
          '9240' => [
            [ '016', '016', 'SO VICENTE' ] ] },
        'Faj Grande' => {
          '9240' => [
            [ '025', '028', 'BOA VENTURA' ] ] },
        'Levada de Baixo' => {
          '9240' => [
            [ '032', '033', 'BOA VENTURA' ] ] },
        'Lamaceiros' => {
          '9240' => [
            [ '030', '030', 'BOA VENTURA' ] ] },
        'Faj do Rente' => {
          '9240' => [
            [ '024', '024', 'SO VICENTE' ] ] },
        'Lombadinha' => {
          '9240' => [
            [ '035', '037', 'BOA VENTURA' ] ] },
        'Cabo da Ribeira' => {
          '9240' => [
            [ '017', '017', 'BOA VENTURA' ] ] },
        'Pomar' => {
          '9240' => [
            [ '041', '049', 'BOA VENTURA' ] ] },
        'Achada da Madeira' => {
          '9240' => [
            [ '011', '015', 'BOA VENTURA' ] ] },
        'Achada de Til' => {
          '9240' => [
            [ '201', '225', 'SO VICENTE' ] ] },
        'Juncos - Faj da Areia' => {
          '9240' => [
            [ '029', '029', 'SO VICENTE' ] ] },
        'Calhau' => {
          '9240' => [
            [ '018', '021', 'SO VICENTE' ] ] }
              },
      'Machico' => {
        'Larano' => {
          '9225' => [
            [ '170', '170', 'PORTO DA CRUZ' ] ] },
        'Maiata de Baixo' => {
          '9225' => [
            [ '190', '190', 'PORTO DA CRUZ' ] ] },
        'Maiata' => {
          '9225' => [
            [ '180', '180', 'PORTO DA CRUZ' ] ] },
        'Achadinha' => {
          '9225' => [
            [ '020', '020', 'PORTO DA CRUZ' ] ],
          '9200' => [
            [ '151', '162', 'STO ANTNIO DA SERRA MCH' ] ] },
        'Junqueira' => {
          '9225' => [
            [ '160', '160', 'PORTO DA CRUZ' ] ] },
        'Cova do Til' => {
          '9225' => [
            [ '060', '060', 'PORTO DA CRUZ' ] ] },
        'Maiata de Cima' => {
          '9225' => [
            [ '200', '200', 'PORTO DA CRUZ' ] ] },
        'Cruz' => {
          '9225' => [
            [ '070', '070', 'PORTO DA CRUZ' ] ] },
        'Banda da Silva' => {
          '9200' => [
            [ '031', '033', 'CANIAL' ] ] },
        'Banda de Alm' => {
          '9200' => [
            [ '071', '078', 'MACHICO' ] ] },
        'Achada' => {
          '9225' => [
            [ '010', '010', 'PORTO DA CRUZ' ] ] },
        'Referta' => {
          '9225' => [
            [ '220', '220', 'PORTO DA CRUZ' ] ] },
        'Casas Prximas' => {
          '9225' => [
            [ '050', '050', 'PORTO DA CRUZ' ] ] },
        'Cancela' => {
          '9200' => [
            [ '035', '047', 'CANIAL' ] ] },
        'Ladeira' => {
          '9200' => [
            [ '080', '104', 'MACHICO' ] ] },
        'Cruz da Guarda' => {
          '9225' => [
            [ '080', '080', 'PORTO DA CRUZ' ] ] },
        'Machico' => {
          '9200' => [
            [ '106', '129', 'MACHICO' ] ] },
        'Juncal' => {
          '9225' => [
            [ '150', '150', 'PORTO DA CRUZ' ] ] },
        'Gambo' => {
          '9225' => [
            [ '140', '140', 'PORTO DA CRUZ' ] ] },
        'Serrado' => {
          '9225' => [
            [ '240', '240', 'PORTO DA CRUZ' ] ] },
        'Massapez' => {
          '9225' => [
            [ '210', '210', 'PORTO DA CRUZ' ] ] },
        'Terra do Batista' => {
          '9225' => [
            [ '250', '250', 'PORTO DA CRUZ' ] ] },
        'Agua de Pena' => {
          '9200' => [
            [ '011', '018', 'GUA DE PENA' ] ] },
        'Ribeira Tem-Te-No-Caias' => {
          '9225' => [
            [ '230', '230', 'PORTO DA CRUZ' ] ] },
        'Faj de Milho' => {
          '9225' => [
            [ '100', '100', 'PORTO DA CRUZ' ] ] },
        'Folhadal' => {
          '9225' => [
            [ '120', '120', 'PORTO DA CRUZ' ] ] }
                },
      'Ribeira Brava' => {
        'Palmeira' => {
          '9350' => [
            [ '054', '055', 'CAMPANRIO' ] ] },
        'Praia' => {
          '9350' => [
            [ '412', '413', 'TBUA RBR' ] ] },
        'Achada dos Aparcios' => {
          '9350' => [
            [ '301', '310', 'SERRA DE GUA' ] ] },
        'Igreja' => {
          '9350' => [
            [ '037', '038', 'CAMPANRIO' ] ] },
        'Meia Lgua' => {
          '9350' => [
            [ '131', '131', 'RIBEIRA BRAVA' ] ] },
        'Lugar da Ribeira' => {
          '9350' => [
            [ '046', '047', 'CAMPANRIO' ] ] },
        'Pedregal' => {
          '9350' => [
            [ '057', '057', 'CAMPANRIO' ] ] },
        'Corujeira' => {
          '9350' => [
            [ '025', '026', 'CAMPANRIO' ] ] },
        'Quebrada' => {
          '9350' => [
            [ '067', '067', 'CAMPANRIO' ] ] },
        'Furnas' => {
          '9350' => [
            [ '035', '035', 'CAMPANRIO' ] ] },
        'Pinheiro' => {
          '9350' => [
            [ '060', '060', 'CAMPANRIO' ] ] },
        'Roda' => {
          '9350' => [
            [ '070', '073', 'CAMPANRIO' ] ] },
        'Lombo Cesteiro' => {
          '9350' => [
            [ '126', '126', 'RIBEIRA BRAVA' ] ] },
        'Massapez' => {
          '9350' => [
            [ '051', '051', 'CAMPANRIO' ] ] },
        'Chamorra' => {
          '9350' => [
            [ '022', '023', 'CAMPANRIO' ] ] },
        'Vale' => {
          '9350' => [
            [ '145', '145', 'RIBEIRA BRAVA' ] ] },
        'Lombo da Levada' => {
          '9350' => [
            [ '128', '129', 'RIBEIRA BRAVA' ] ] },
        'Heredias' => {
          '9350' => [
            [ '124', '124', 'RIBEIRA BRAVA' ] ] },
        'Porto da Ribeira' => {
          '9350' => [
            [ '064', '064', 'CAMPANRIO' ] ] },
        'Passal' => {
          '9350' => [
            [ '312', '320', 'SERRA DE GUA' ] ] },
        'Tera' => {
          '9350' => [
            [ '416', '418', 'TBUA RBR' ] ] },
        'Fonte Cruzada' => {
          '9350' => [
            [ '119', '122', 'RIBEIRA BRAVA' ] ] },
        'Tranqual' => {
          '9350' => [
            [ '078', '078', 'CAMPANRIO' ] ] },
        'Bica de Pau' => {
          '9350' => [
            [ '403', '403', 'TBUA RBR' ] ] },
        'Achada' => {
          '9350' => [
            [ '401', '401', 'TBUA RBR' ] ] },
        'Longueira' => {
          '9350' => [
            [ '044', '044', 'CAMPANRIO' ] ] },
        'Moreno' => {
          '9350' => [
            [ '133', '133', 'RIBEIRA BRAVA' ] ] },
        'Terreiros' => {
          '9350' => [
            [ '076', '076', 'CAMPANRIO' ] ] },
        'Carmo' => {
          '9350' => [
            [ '019', '019', 'CAMPANRIO' ] ] },
        'Vila' => {
          '9350' => [
            [ '201', '224', 'RIBEIRA BRAVA' ] ] },
        'Candelria' => {
          '9350' => [
            [ '405', '410', 'TBUA RBR' ] ] },
        'Lapa' => {
          '9350' => [
            [ '040', '041', 'CAMPANRIO' ] ] },
        'Pedra da Nossa Senhora' => {
          '9350' => [
            [ '135', '143', 'RIBEIRA BRAVA' ] ] },
        'Amoreira' => {
          '9350' => [
            [ '015', '016', 'CAMPANRIO' ] ] },
        'Barreiro' => {
          '9350' => [
            [ '103', '116', 'RIBEIRA BRAVA' ] ] },
        'Porta Nova' => {
          '9350' => [
            [ '062', '062', 'CAMPANRIO' ] ] },
        'Faj Velha' => {
          '9350' => [
            [ '032', '032', 'CAMPANRIO' ] ] },
        'Adega' => {
          '9350' => [
            [ '013', '013', 'CAMPANRIO' ] ] },
        'Vigia' => {
          '9350' => [
            [ '080', '081', 'CAMPANRIO' ] ] }
                },
      'Cmara de Lobos' => {
        'Cmara de Lobos' => {
          '9300' => [
            [ '002', '004', 'CMARA DE LOBOS' ],
            [ '006', '008', 'CMARA DE LOBOS' ],
            [ '010', '010', 'CMARA DE LOBOS' ],
            [ '012', '014', 'CMARA DE LOBOS' ],
            [ '018', '018', 'CMARA DE LOBOS' ],
            [ '020', '035', 'CMARA DE LOBOS' ],
            [ '037', '039', 'CMARA DE LOBOS' ],
            [ '041', '058', 'CMARA DE LOBOS' ],
            [ '060', '066', 'CMARA DE LOBOS' ],
            [ '068', '076', 'CMARA DE LOBOS' ],
            [ '079', '079', 'CMARA DE LOBOS' ],
            [ '085', '102', 'CMARA DE LOBOS' ],
            [ '105', '128', 'CMARA DE LOBOS' ],
            [ '130', '133', 'CMARA DE LOBOS' ],
            [ '136', '168', 'CMARA DE LOBOS' ],
            [ '210', '210', 'CMARA DE LOBOS' ],
            [ '303', '304', 'CMARA DE LOBOS' ],
            [ '306', '329', 'CMARA DE LOBOS' ],
            [ '331', '332', 'CMARA DE LOBOS' ],
            [ '334', '342', 'CMARA DE LOBOS' ],
            [ '344', '347', 'CMARA DE LOBOS' ],
            [ '349', '349', 'CMARA DE LOBOS' ],
            [ '351', '425', 'CMARA DE LOBOS' ] ],
          '9304' => [
            [ '001', '007', 'CMARA DE LOBOS' ] ] },
        'Estreito Cmara de Lobos' => {
          '9325' => [
            [ '002', '010', 'ESTREITO CMARA DE LOBOS' ],
            [ '017', '040', 'ESTREITO CMARA DE LOBOS' ],
            [ '042', '043', 'ESTREITO CMARA DE LOBOS' ],
            [ '046', '061', 'ESTREITO CMARA DE LOBOS' ],
            [ '073', '080', 'ESTREITO CMARA DE LOBOS' ],
            [ '082', '095', 'ESTREITO CMARA DE LOBOS' ],
            [ '097', '101', 'ESTREITO CMARA DE LOBOS' ],
            [ '116', '116', 'ESTREITO CMARA DE LOBOS' ],
            [ '310', '360', 'ESTREITO CMARA DE LOBOS' ] ] },
        'Quinta Grande' => {
          '9300' => [
            [ '077', '078', 'QUINTA GRANDE' ],
            [ '080', '080', 'QUINTA GRANDE' ],
            [ '082', '083', 'QUINTA GRANDE' ],
            [ '211', '226', 'QUINTA GRANDE' ],
            [ '228', '248', 'QUINTA GRANDE' ],
            [ '261', '289', 'QUINTA GRANDE' ],
            [ '291', '296', 'QUINTA GRANDE' ],
            [ '302', '302', 'QUINTA GRANDE' ],
            [ '426', '428', 'QUINTA GRANDE' ] ] },
        'Jardim da Serra' => {
          '9325' => [
            [ '120', '267', 'JARDIM DA SERRA' ] ] },
        'Curral das Freiras' => {
          '9030' => [
            [ '010', '010', 'CURRAL DAS FREIRAS' ],
            [ '020', '020', 'CURRAL DAS FREIRAS' ],
            [ '030', '030', 'CURRAL DAS FREIRAS' ],
            [ '040', '040', 'CURRAL DAS FREIRAS' ],
            [ '050', '050', 'CURRAL DAS FREIRAS' ],
            [ '060', '060', 'CURRAL DAS FREIRAS' ],
            [ '070', '070', 'CURRAL DAS FREIRAS' ],
            [ '080', '080', 'CURRAL DAS FREIRAS' ],
            [ '090', '090', 'CURRAL DAS FREIRAS' ],
            [ '100', '100', 'CURRAL DAS FREIRAS' ],
            [ '110', '110', 'CURRAL DAS FREIRAS' ],
            [ '120', '120', 'CURRAL DAS FREIRAS' ],
            [ '130', '130', 'CURRAL DAS FREIRAS' ],
            [ '140', '140', 'CURRAL DAS FREIRAS' ] ] }
                  },
      'Funchal' => {
        'Selvagem Pequena' => {
          '9000' => [
            [ '900', '900', 'FUNCHAL' ] ] },
        'Funchal' => {
          '9004' => [
            [ '501', '509', 'FUNCHAL' ],
            [ '511', '522', 'FUNCHAL' ],
            [ '524', '527', 'FUNCHAL' ],
            [ '529', '549', 'FUNCHAL' ],
            [ '551', '556', 'FUNCHAL' ],
            [ '558', '566', 'FUNCHAL' ],
            [ '600', '600', 'FUNCHAL' ] ],
          '9060' => [
            [ '001', '016', 'FUNCHAL' ],
            [ '019', '096', 'FUNCHAL' ],
            [ '098', '105', 'FUNCHAL' ],
            [ '107', '126', 'FUNCHAL' ],
            [ '128', '149', 'FUNCHAL' ],
            [ '152', '186', 'FUNCHAL' ],
            [ '188', '190', 'FUNCHAL' ],
            [ '194', '228', 'FUNCHAL' ],
            [ '231', '232', 'FUNCHAL' ],
            [ '235', '246', 'FUNCHAL' ],
            [ '248', '259', 'FUNCHAL' ],
            [ '261', '327', 'FUNCHAL' ],
            [ '329', '329', 'FUNCHAL' ],
            [ '333', '338', 'FUNCHAL' ],
            [ '340', '352', 'FUNCHAL' ],
            [ '354', '354', 'FUNCHAL' ],
            [ '356', '358', 'FUNCHAL' ],
            [ '361', '365', 'FUNCHAL' ],
            [ '367', '411', 'FUNCHAL' ] ],
          '9020' => [
            [ '001', '106', 'FUNCHAL' ],
            [ '108', '159', 'FUNCHAL' ],
            [ '161', '212', 'FUNCHAL' ],
            [ '214', '216', 'FUNCHAL' ],
            [ '218', '236', 'FUNCHAL' ],
            [ '238', '291', 'FUNCHAL' ],
            [ '293', '343', 'FUNCHAL' ],
            [ '345', '371', 'FUNCHAL' ],
            [ '373', '404', 'FUNCHAL' ] ],
          '9024' => [
            [ '500', '502', 'FUNCHAL' ] ],
          '9064' => [
            [ '501', '512', 'FUNCHAL' ] ],
          '9054' => [
            [ '502', '503', 'FUNCHAL' ],
            [ '505', '508', 'FUNCHAL' ],
            [ '510', '512', 'FUNCHAL' ],
            [ '514', '514', 'FUNCHAL' ],
            [ '516', '516', 'FUNCHAL' ],
            [ '518', '518', 'FUNCHAL' ],
            [ '520', '520', 'FUNCHAL' ],
            [ '525', '533', 'FUNCHAL' ],
            [ '535', '535', 'FUNCHAL' ],
            [ '550', '550', 'FUNCHAL' ],
            [ '559', '559', 'FUNCHAL' ] ],
          '9050' => [
            [ '011', '028', 'FUNCHAL' ],
            [ '045', '059', 'FUNCHAL' ],
            [ '061', '075', 'FUNCHAL' ],
            [ '077', '080', 'FUNCHAL' ],
            [ '082', '086', 'FUNCHAL' ],
            [ '088', '089', 'FUNCHAL' ],
            [ '091', '092', 'FUNCHAL' ],
            [ '095', '100', 'FUNCHAL' ],
            [ '190', '199', 'FUNCHAL' ],
            [ '205', '206', 'FUNCHAL' ],
            [ '208', '219', 'FUNCHAL' ],
            [ '221', '222', 'FUNCHAL' ],
            [ '225', '227', 'FUNCHAL' ],
            [ '231', '231', 'FUNCHAL' ],
            [ '233', '238', 'FUNCHAL' ],
            [ '281', '285', 'FUNCHAL' ],
            [ '287', '288', 'FUNCHAL' ],
            [ '290', '294', 'FUNCHAL' ],
            [ '328', '328', 'FUNCHAL' ],
            [ '331', '331', 'FUNCHAL' ],
            [ '401', '411', 'FUNCHAL' ],
            [ '414', '415', 'FUNCHAL' ],
            [ '417', '424', 'FUNCHAL' ],
            [ '426', '430', 'FUNCHAL' ],
            [ '432', '432', 'FUNCHAL' ],
            [ '434', '434', 'FUNCHAL' ],
            [ '436', '436', 'FUNCHAL' ],
            [ '438', '463', 'FUNCHAL' ],
            [ '466', '488', 'FUNCHAL' ],
            [ '490', '501', 'FUNCHAL' ],
            [ '503', '532', 'FUNCHAL' ],
            [ '534', '562', 'FUNCHAL' ] ],
          '9000' => [
            [ '001', '005', 'FUNCHAL' ],
            [ '011', '015', 'FUNCHAL' ],
            [ '017', '022', 'FUNCHAL' ],
            [ '024', '034', 'FUNCHAL' ],
            [ '036', '040', 'FUNCHAL' ],
            [ '042', '090', 'FUNCHAL' ],
            [ '092', '096', 'FUNCHAL' ],
            [ '098', '100', 'FUNCHAL' ],
            [ '102', '103', 'FUNCHAL' ],
            [ '105', '124', 'FUNCHAL' ],
            [ '126', '137', 'FUNCHAL' ],
            [ '140', '141', 'FUNCHAL' ],
            [ '143', '146', 'FUNCHAL' ],
            [ '149', '151', 'FUNCHAL' ],
            [ '154', '187', 'FUNCHAL' ],
            [ '189', '192', 'FUNCHAL' ],
            [ '198', '200', 'FUNCHAL' ],
            [ '202', '208', 'FUNCHAL' ],
            [ '210', '213', 'FUNCHAL' ],
            [ '215', '224', 'FUNCHAL' ],
            [ '226', '229', 'FUNCHAL' ],
            [ '231', '251', 'FUNCHAL' ],
            [ '254', '261', 'FUNCHAL' ],
            [ '263', '279', 'FUNCHAL' ],
            [ '281', '285', 'FUNCHAL' ],
            [ '290', '292', 'FUNCHAL' ],
            [ '296', '296', 'FUNCHAL' ],
            [ '309', '309', 'FUNCHAL' ],
            [ '427', '427', 'FUNCHAL' ],
            [ '500', '501', 'FUNCHAL' ],
            [ '503', '503', 'FUNCHAL' ],
            [ '512', '512', 'FUNCHAL' ],
            [ '601', '639', 'FUNCHAL' ],
            [ '641', '650', 'FUNCHAL' ],
            [ '652', '753', 'FUNCHAL' ] ] }
                },
      'Porto Moniz' => {
        'Ribeirinho' => {
          '9270' => [
            [ '091', '095', 'PORTO MONIZ' ] ] },
        'Cova' => {
          '9270' => [
            [ '121', '130', 'SEIXAL PMZ' ] ] },
        'Casais da Igreja' => {
          '9270' => [
            [ '101', '107', 'RIBEIRA DA JANELA' ] ] },
        'Cabo' => {
          '9270' => [
            [ '031', '042', 'PORTO MONIZ' ] ] },
        'Achada da Arruda' => {
          '9270' => [
            [ '011', '014', 'ACHADAS DA CRUZ' ] ] },
        'Porto Moniz' => {
          '9270' => [
            [ '051', '053', 'PORTO MONIZ' ] ] }
              }
             },
    'Santarm' => {
      'Goleg' => {
        'Goleg' => {
          '2150' => [
            [ '120', '128', 'GOLEG' ],
            [ '130', '130', 'GOLEG' ],
            [ '132', '133', 'GOLEG' ],
            [ '136', '181', 'GOLEG' ],
            [ '183', '210', 'GOLEG' ],
            [ '212', '221', 'GOLEG' ] ] },
        'Estao Caminho Ferro' => {
          '2150' => [
            [ '061', '073', 'AZINHAGA' ] ] },
        'Azinhaga' => {
          '2150' => [
            [ '001', '009', 'AZINHAGA' ],
            [ '011', '048', 'AZINHAGA' ] ] },
        'M-L' => {
          '2150' => [
            [ '261', '268', 'GOLEG' ] ] },
        'Baralha' => {
          '2150' => [
            [ '101', '107', 'GOLEG' ] ] }
              },
      'Santarm' => {
        'Santarm' => {
          '2005' => [
            [ '054', '054', 'SANTARM' ],
            [ '056', '058', 'SANTARM' ],
            [ '065', '071', 'SANTARM' ],
            [ '073', '075', 'SANTARM' ],
            [ '132', '134', 'SANTARM' ],
            [ '136', '150', 'SANTARM' ],
            [ '152', '184', 'SANTARM' ],
            [ '186', '186', 'SANTARM' ],
            [ '188', '188', 'SANTARM' ],
            [ '190', '194', 'SANTARM' ],
            [ '197', '201', 'SANTARM' ],
            [ '239', '240', 'SANTARM' ],
            [ '244', '247', 'SANTARM' ],
            [ '249', '252', 'SANTARM' ],
            [ '256', '259', 'SANTARM' ],
            [ '262', '280', 'SANTARM' ],
            [ '282', '288', 'SANTARM' ],
            [ '294', '294', 'SANTARM' ],
            [ '296', '304', 'SANTARM' ],
            [ '306', '306', 'SANTARM' ],
            [ '308', '348', 'SANTARM' ],
            [ '350', '366', 'SANTARM' ],
            [ '372', '372', 'SANTARM' ],
            [ '383', '383', 'SANTARM' ],
            [ '394', '394', 'SANTARM' ],
            [ '405', '405', 'SANTARM' ],
            [ '416', '416', 'SANTARM' ],
            [ '427', '427', 'SANTARM' ],
            [ '436', '439', 'SANTARM' ],
            [ '441', '449', 'SANTARM' ],
            [ '451', '458', 'SANTARM' ],
            [ '460', '461', 'SANTARM' ],
            [ '463', '479', 'SANTARM' ],
            [ '481', '506', 'SANTARM' ],
            [ '508', '512', 'SANTARM' ],
            [ '517', '521', 'SANTARM' ],
            [ '538', '538', 'SANTARM' ],
            [ '542', '542', 'SANTARM' ] ],
          '2004' => [
            [ '002', '007', 'SANTARM' ] ],
          '2000' => [
            [ '005', '005', 'SANTARM' ],
            [ '010', '011', 'SANTARM' ],
            [ '013', '016', 'SANTARM' ],
            [ '018', '021', 'SANTARM' ],
            [ '024', '024', 'SANTARM' ],
            [ '029', '031', 'SANTARM' ],
            [ '034', '128', 'SANTARM' ],
            [ '134', '135', 'SANTARM' ],
            [ '141', '141', 'SANTARM' ],
            [ '143', '143', 'SANTARM' ],
            [ '145', '145', 'SANTARM' ],
            [ '179', '185', 'SANTARM' ],
            [ '198', '232', 'SANTARM' ],
            [ '241', '241', 'SANTARM' ],
            [ '245', '245', 'SANTARM' ],
            [ '471', '478', 'SANTARM' ],
            [ '834', '842', 'SANTARM' ] ],
          '2009' => [
            [ '001', '002', 'SANTARM' ],
            [ '004', '005', 'SANTARM' ] ] },
        'Atalaia' => {
          '2025' => [
            [ '451', '455', 'AZOIA DE CIMA' ] ] },
        'Quinta das Chantas' => {
          '2000' => [
            [ '350', '401', 'ALCANHES' ] ] },
        'Casal Maria do Cu' => {
          '2005' => [
            [ '291', '293', 'ROMEIRA' ] ] },
        'Casal das Lamas' => {
          '2000' => [
            [ '671', '697', 'SO VICENTE DO PAUL' ] ] },
        'Quinta da Boavista' => {
          '2000' => [
            [ '710', '756', 'VALE DE FIGUEIRA STR' ] ] },
        'Alqueido do Mato' => {
          '2025' => [
            [ '140', '177', 'ALCANEDE' ] ] },
        'Abr' => {
          '2025' => [
            [ '011', '015', 'ABR' ] ] },
        'Valverde' => {
          '2025' => [
            [ '201', '242', 'ALCANEDE' ] ] },
        'Quinta Fonte da Pedra' => {
          '2005' => [
            [ '540', '541', 'VRZEA STR' ] ] },
        'Casal dos Clrigos' => {
          '2000' => [
            [ '451', '467', 'CASVEL STR' ] ] },
        'Ganaria' => {
          '2025' => [
            [ '601', '602', 'GANARIA' ] ] },
        'Casal Coelho' => {
          '2000' => [
            [ '791', '797', 'VAQUEIROS STR' ] ] },
        'Pernes' => {
          '2000' => [
            [ '698', '698', 'PERNES' ] ] },
        'Casal da Comenda' => {
          '2000' => [
            [ '334', '346', 'ACHETE' ] ] },
        'O Almeirim' => {
          '2000' => [
            [ '432', '434', 'ARNEIRO DAS MILHARIAS' ] ] },
        'Estrada da Isenta' => {
          '2005' => [
            [ '295', '295', 'PVOA DA ISENTA' ] ] },
        'gua Peneira' => {
          '2025' => [
            [ '501', '511', 'TREMS' ] ] },
        'Casal Laranjeiro' => {
          '2005' => [
            [ '522', '522', 'VALE DE SANTARM' ] ] },
        'Cabeo do Monte' => {
          '2000' => [
            [ '490', '500', 'PERNES' ] ] },
        'Trems' => {
          '2025' => [
            [ '550', '563', 'TREMS' ],
            [ '566', '585', 'TREMS' ] ] },
        'Casais do Maio' => {
          '2005' => [
            [ '523', '524', 'VRZEA STR' ] ] },
        'Vale de Santarm' => {
          '2005' => [
            [ '029', '037', 'VALE DE SANTARM' ],
            [ '039', '046', 'VALE DE SANTARM' ],
            [ '367', '370', 'VALE DE SANTARM' ],
            [ '373', '382', 'VALE DE SANTARM' ],
            [ '384', '393', 'VALE DE SANTARM' ],
            [ '395', '400', 'VALE DE SANTARM' ],
            [ '402', '404', 'VALE DE SANTARM' ],
            [ '406', '415', 'VALE DE SANTARM' ],
            [ '417', '426', 'VALE DE SANTARM' ],
            [ '428', '435', 'VALE DE SANTARM' ],
            [ '525', '537', 'VALE DE SANTARM' ] ] },
        'Casal dos Florindos' => {
          '2000' => [
            [ '320', '332', 'ACHETE' ] ] },
        'Urbanizao Casais do Mocho' => {
          '2005' => [
            [ '514', '515', 'VRZEA STR' ] ] },
        'Ribeira de Santarm' => {
          '2000' => [
            [ '007', '007', 'STA IRIA RIBEIRA SANTARM' ],
            [ '561', '623', 'STA IRIA RIBEIRA SANTARM' ] ] },
        'Amiais de Baixo' => {
          '2025' => [
            [ '300', '369', 'AMIAIS DE BAIXO' ],
            [ '390', '401', 'AMIAIS DE BAIXO' ] ] },
        'Louriceira' => {
          '2005' => [
            [ '104', '114', 'ALMOSTER STR' ] ] },
        'Moita' => {
          '2000' => [
            [ '504', '504', 'PERNES' ] ] },
        'Pvoa de Santarm' => {
          '2000' => [
            [ '531', '531', 'PVOA DE SANTARM' ] ] },
        'Casal do Fairro' => {
          '2000' => [
            [ '833', '833', 'ACHETE' ] ] },
        'Vale Moinhos' => {
          '2005' => [
            [ '539', '539', 'PVOA DA ISENTA' ] ] },
        'Moita do Lobo' => {
          '2005' => [
            [ '516', '516', 'ALMOSTER STR' ] ] },
        'Pombalinho' => {
          '2150' => [
            [ '080', '092', 'POMBALINHO STR' ] ] },
        'Casais da Ferreira' => {
          '2000' => [
            [ '429', '430', 'ARNEIRO DAS MILHARIAS' ] ] },
        'Vila Nova do Coito' => {
          '2005' => [
            [ '101', '102', 'ALMOSTER STR' ] ] },
        'Casal da Vindima' => {
          '2005' => [
            [ '289', '290', 'VALE DE SANTARM' ] ] },
        'Quinta da Marrafa' => {
          '2009' => [
            [ '003', '003', 'VRZEA STR' ] ] },
        'Assacaias' => {
          '2000' => [
            [ '547', '548', 'STA IRIA RIBEIRA SANTARM' ] ] },
        'Casal Mata Quatro' => {
          '2005' => [
            [ '085', '088', 'PVOA DA ISENTA' ] ] },
        'Alcobacinha' => {
          '2005' => [
            [ '089', '089', 'VRZEA STR' ] ] },
        'Quinta dos Cardeais de Baixo' => {
          '2005' => [
            [ '092', '096', 'MOARRIA' ] ] },
        'Quinta de Vale de Lobos' => {
          '2005' => [
            [ '097', '100', 'AZOIA DE BAIXO' ] ] },
        'Casais da Cabaa' => {
          '2005' => [
            [ '513', '513', 'ROMEIRA' ] ] },
        'Espinhal' => {
          '2000' => [
            [ '699', '702', 'SO VICENTE DO PAUL' ] ] },
        'Ponte do Celeiro' => {
          '2005' => [
            [ '260', '260', 'ALMOSTER STR' ] ] },
        'Aldeia da Ribeira' => {
          '2025' => [
            [ '071', '096', 'ALCANEDE' ] ] },
        'Alcanede' => {
          '2025' => [
            [ '030', '049', 'ALCANEDE' ] ] },
        'Vrzea' => {
          '2005' => [
            [ '001', '028', 'VRZEA STR' ] ],
          '2025' => [
            [ '250', '252', 'ALCANEDE' ] ] },
        'Fonte Boa' => {
          '2005' => [
            [ '048', '051', 'VALE DE SANTARM' ] ] },
        'Romeira' => {
          '2005' => [
            [ '076', '076', 'ROMEIRA' ] ] },
        'Vidigo Poente' => {
          '2005' => [
            [ '115', '130', 'ABITUREIRAS' ] ] },
        'Quinta dos Soidos' => {
          '2005' => [
            [ '371', '371', 'ABITUREIRAS' ] ] },
        'Quinta dos Reverendos' => {
          '2005' => [
            [ '078', '080', 'ROMEIRA' ] ] },
        'Aldeia de Alm' => {
          '2025' => [
            [ '101', '138', 'ALCANEDE' ] ] }
          },
      'Rio Maior' => {
        'Lobo Morto' => {
          '2040' => [
            [ '132', '134', 'RIO MAIOR' ] ] },
        'Casal Longra' => {
          '2040' => [
            [ '017', '017', 'RIO MAIOR' ] ] },
        'Asseiceira' => {
          '2040' => [
            [ '481', '483', 'ASSEICEIRA RMR' ] ] },
        'Rio Maior' => {
          '2040' => [
            [ '130', '130', 'RIO MAIOR' ],
            [ '201', '241', 'RIO MAIOR' ],
            [ '243', '347', 'RIO MAIOR' ],
            [ '349', '358', 'RIO MAIOR' ],
            [ '413', '414', 'RIO MAIOR' ] ] },
        'Ribeira de So Joo' => {
          '2040' => [
            [ '511', '512', 'RIBEIRA DE SO JOO' ] ] },
        'Lagoa da M' => {
          '2040' => [
            [ '131', '131', 'RIBEIRA DE SO JOO' ] ] },
        'Casalinho' => {
          '2040' => [
            [ '086', '094', 'RIO MAIOR' ] ] },
        'Alfouvez' => {
          '2040' => [
            [ '051', '053', 'AZAMBUJEIRA' ] ] },
        'Alcobertas' => {
          '2040' => [
            [ '011', '016', 'ALCOBERTAS' ] ] },
        'Casais da Cheira' => {
          '2040' => [
            [ '152', '155', 'FRGUAS RMR' ] ] },
        'Carvalhais' => {
          '2040' => [
            [ '150', '150', 'FRGUAS RMR' ],
            [ '491', '495', 'SO SEBASTIO' ] ] },
        'Freiria' => {
          '2040' => [
            [ '101', '123', 'RIO MAIOR' ] ] },
        'Vila da Marmeleira' => {
          '2040' => [
            [ '158', '158', 'MARMELEIRA RMR' ] ] },
        'Chos' => {
          '2040' => [
            [ '018', '025', 'ALCOBERTAS' ] ] },
        'Casal do Alto' => {
          '2040' => [
            [ '171', '176', 'OUTEIRO DA CORTIADA' ] ] },
        'Assentiz' => {
          '2040' => [
            [ '536', '536', 'ASSENTIZ RMR' ] ] },
        'Santo Andr' => {
          '2040' => [
            [ '401', '411', 'RIO MAIOR' ] ] },
        'Arruda dos Pises' => {
          '2040' => [
            [ '041', '041', 'ARRUDA DOS PISES' ] ] },
        'Arrouquelas' => {
          '2040' => [
            [ '031', '032', 'ARROUQUELAS' ] ] },
        'Casais da Arroteia' => {
          '2040' => [
            [ '533', '535', 'MALAQUEIJO' ] ] },
        'P da Serra' => {
          '2040' => [
            [ '141', '143', 'RIO MAIOR' ] ] },
        'Atgueda' => {
          '2040' => [
            [ '451', '462', 'SO JOO DA RIBEIRA' ] ] },
        'Aberta' => {
          '2040' => [
            [ '061', '084', 'RIO MAIOR' ] ] },
        'Vale Brejo' => {
          '2040' => [
            [ '157', '157', 'FRGUAS RMR' ] ] },
        'Alfouves' => {
          '2040' => [
            [ '531', '531', 'MALAQUEIJO' ] ] }
           },
      'Cartaxo' => {
        'Alqueido' => {
          '2070' => [
            [ '501', '529', 'VALADA' ] ] },
        'Casais da Lapa' => {
          '2070' => [
            [ '351', '352', 'LAPA' ] ] },
        'Alto do Padro' => {
          '2070' => [
            [ '001', '003', 'CARTAXO' ] ] },
        'Casal Lambel' => {
          '2070' => [
            [ '301', '329', 'EREIRA CTX' ] ] },
        'Casal de Solposto' => {
          '2070' => [
            [ '551', '600', 'VALE DA PINTA' ] ] },
        'Cartaxo' => {
          '2070' => [
            [ '010', '153', 'CARTAXO' ],
            [ '159', '159', 'CARTAXO' ] ] },
        'Vale da Pedra' => {
          '2070' => [
            [ '710', '738', 'VALE DA PEDRA' ] ] },
        'Bairro do Setil' => {
          '2070' => [
            [ '701', '703', 'VALE DA PEDRA' ] ] },
        'Casais da Amendoeira' => {
          '2070' => [
            [ '361', '462', 'PONTVEL' ] ] },
        'Vila Ch de Ourique' => {
          '2070' => [
            [ '611', '668', 'VILA CH DE OURIQUE' ],
            [ '670', '681', 'VILA CH DE OURIQUE' ] ] },
        'Casais Penedos' => {
          '2070' => [
            [ '201', '235', 'CARTAXO' ] ] }
               },
      'Mao' => {
        'Arganil' => {
          '6120' => [
            [ '211', '233', 'CARDIGOS' ] ] },
        'Penhascoso' => {
          '6120' => [
            [ '621', '640', 'PENHASCOSO' ] ] },
        'Alpalho' => {
          '6120' => [
            [ '011', '038', 'ENVENDOS' ] ] },
        'Mantela' => {
          '6120' => [
            [ '771', '784', 'MAO' ] ] },
        'Alto Pereiro' => {
          '6120' => [
            [ '701', '707', 'MAO' ] ] },
        'Casal da Barba Pouca' => {
          '6120' => [
            [ '611', '613', 'PENHASCOSO' ] ] },
        'Balancho' => {
          '6120' => [
            [ '311', '328', 'CARVOEIRO MAC' ] ] },
        'Levada' => {
          '6120' => [
            [ '709', '709', 'MAO' ] ] },
        'Mao' => {
          '6120' => [
            [ '720', '753', 'MAO' ],
            [ '755', '760', 'MAO' ] ] },
        'Estao Caminho de Ferro Ortiga' => {
          '6120' => [
            [ '708', '708', 'ORTIGA' ] ] },
        'Aboboreira' => {
          '6120' => [
            [ '111', '117', 'ABOBOREIRA' ] ] },
        'Barragem de Belver' => {
          '6120' => [
            [ '511', '562', 'ORTIGA' ] ] },
        'So Jos das Matas' => {
          '6120' => [
            [ '069', '069', 'ENVENDOS' ] ] },
        'Aldeia de Eiras' => {
          '6120' => [
            [ '151', '169', 'AMNDOA' ] ] },
        'Portela de Abrantes' => {
          '6120' => [
            [ '661', '667', 'PENHASCOSO' ] ] }
             },
      'Torres Novas' => {
        'Borreco' => {
          '2350' => [
            [ '211', '215', 'PARCEIROS DE IGREJA' ] ] },
        'Lapas' => {
          '2350' => [
            [ '101', '137', 'LAPAS' ] ] },
        'Casal Novo' => {
          '2350' => [
            [ '714', '715', 'TORRES NOVAS' ] ] },
        'Pimenteis' => {
          '2350' => [
            [ '139', '143', 'LAPAS' ] ] },
        'Quinta de Entre guas' => {
          '2350' => [
            [ '734', '735', 'TORRES NOVAS' ] ] },
        'Carvalhal do Pombo' => {
          '2350' => [
            [ '016', '017', 'ASSENTIZ TNV' ] ] },
        'Casal da Fonte' => {
          '2350' => [
            [ '019', '020', 'ASSENTIZ TNV' ] ] },
        'Riachos' => {
          '2350' => [
            [ '291', '303', 'RIACHOS' ],
            [ '321', '321', 'RIACHOS' ],
            [ '323', '374', 'RIACHOS' ] ] },
        'Gateiras de Santo Antnio' => {
          '2350' => [
            [ '624', '624', 'TORRES NOVAS' ] ] },
        'Charruada' => {
          '2350' => [
            [ '022', '030', 'ASSENTIZ TNV' ] ] },
        'Silv' => {
          '2350' => [
            [ '743', '748', 'TORRES NOVAS' ] ] },
        'Rio Frio' => {
          '2350' => [
            [ '737', '738', 'TORRES NOVAS' ] ] },
        'Alcorriol' => {
          '2350' => [
            [ '661', '679', 'TORRES NOVAS' ] ] },
        'Alto Senhora da Vitria' => {
          '2350' => [
            [ '391', '398', 'RIBEIRA BRANCA' ] ] },
        'Pedrgo' => {
          '2350' => [
            [ '225', '271', 'PEDRGO TNV' ] ] },
        'Boquilobo' => {
          '2350' => [
            [ '051', '059', 'BROGUEIRA' ] ] },
        'Atouguia' => {
          '2350' => [
            [ '471', '508', 'TORRES NOVAS' ] ] },
        'Alcorochel' => {
          '2350' => [
            [ '001', '001', 'ALCOROCHEL' ] ] },
        'Almonda' => {
          '2350' => [
            [ '817', '817', 'ZIBREIRA' ] ] },
        'Cabeo de Soudo' => {
          '2350' => [
            [ '071', '078', 'CHANCELARIA TNV' ] ] },
        'Ribeiro da Boa gua' => {
          '2350' => [
            [ '520', '622', 'TORRES NOVAS' ] ] },
        'Charneca da Meia Via' => {
          '2350' => [
            [ '623', '623', 'MEIA VIA' ] ] },
        'Pintanhos' => {
          '2350' => [
            [ '647', '649', 'TORRES NOVAS' ] ] },
        'Renova' => {
          '2354' => [
            [ '001', '001', 'ZIBREIRA' ] ] },
        'Mezies' => {
          '2350' => [
            [ '405', '405', 'TORRES NOVAS' ] ] },
        'Charneca de Alcorochel' => {
          '2350' => [
            [ '003', '003', 'ALCOROCHEL' ] ] },
        'Alm da Ribeira' => {
          '2350' => [
            [ '010', '014', 'ASSENTIZ TNV' ] ] },
        'Torres Novas' => {
          '2350' => [
            [ '409', '424', 'TORRES NOVAS' ],
            [ '426', '464', 'TORRES NOVAS' ],
            [ '681', '682', 'TORRES NOVAS' ],
            [ '686', '688', 'TORRES NOVAS' ],
            [ '690', '690', 'TORRES NOVAS' ],
            [ '750', '816', 'TORRES NOVAS' ] ] },
        'Quebradas' => {
          '2350' => [
            [ '727', '728', 'TORRES NOVAS' ] ] },
        'rgea' => {
          '2350' => [
            [ '151', '154', 'OLAIA' ] ] },
        'Soudos' => {
          '2350' => [
            [ '201', '203', 'VILA DO PAO' ] ] },
        'Meia Via' => {
          '2350' => [
            [ '625', '646', 'MEIA VIA' ] ] },
        'Carrascosa' => {
          '2350' => [
            [ '692', '712', 'TORRES NOVAS' ] ] },
        'Zibreira' => {
          '2350' => [
            [ '826', '833', 'ZIBREIRA' ],
            [ '838', '860', 'ZIBREIRA' ] ] },
        'A-do-Freire' => {
          '2350' => [
            [ '221', '223', 'PEDRGO TNV' ] ] },
        'Chicharo' => {
          '2350' => [
            [ '156', '181', 'OLAIA' ] ] },
        'Nicho do Rodrigo' => {
          '2350' => [
            [ '717', '725', 'TORRES NOVAS' ] ] }
              },
      'Salvaterra de Magos' => {
        'Marinhais' => {
          '2125' => [
            [ '101', '143', 'MARINHAIS' ],
            [ '145', '199', 'MARINHAIS' ],
            [ '402', '402', 'MARINHAIS' ] ] },
        'Foros de Salvaterra' => {
          '2120' => [
            [ '181', '193', 'FOROS DE SALVATERRA' ] ] },
        'Casal Xofral' => {
          '2125' => [
            [ '311', '365', 'MUGE' ] ] },
        'Raret' => {
          '2125' => [
            [ '201', '227', 'MARINHAIS' ] ] },
        'Alto Cilho' => {
          '2120' => [
            [ '201', '211', 'FOROS DE SALVATERRA' ] ] },
        'Arneiro da Preta' => {
          '2120' => [
            [ '011', '022', 'SALVATERRA DE MAGOS' ] ] },
        'Glria do Ribatejo' => {
          '2125' => [
            [ '021', '069', 'GLRIA DO RIBATEJO' ] ] },
        'Granho' => {
          '2125' => [
            [ '401', '401', 'GRANHO' ] ] },
        'Salvaterra de Magos' => {
          '2120' => [
            [ '051', '118', 'SALVATERRA DE MAGOS' ] ] },
        'Foros Novos' => {
          '2120' => [
            [ '213', '228', 'FOROS DE SALVATERRA' ] ] },
        'Cabeo Nogueira' => {
          '2125' => [
            [ '011', '012', 'GLRIA DO RIBATEJO' ] ] }
               },
      'Ferreira do Zzere' => {
        'Alqueido' => {
          '2240' => [
            [ '551', '560', 'PIAS FZZ' ] ] },
        'Quinta das Paredes' => {
          '2240' => [
            [ '228', '242', 'BECO' ] ] },
        'Quinta do Boim' => {
          '2240' => [
            [ '572', '588', 'PIAS FZZ' ] ] },
        'Outeiro da Frasoeira' => {
          '2240' => [
            [ '599', '628', 'DORNES' ] ] },
        'Queimadas do Toco' => {
          '2240' => [
            [ '570', '570', 'PIAS FZZ' ] ] },
        'Avecasta' => {
          '2240' => [
            [ '104', '141', 'AREIAS FZZ' ] ] },
        'Fontainhas' => {
          '2240' => [
            [ '410', '426', 'IGREJA NOVA DO SOBRAL' ] ] },
        'Courelas' => {
          '2240' => [
            [ '507', '515', 'PAIO MENDES' ] ] },
        'Quinta da Mata' => {
          '2240' => [
            [ '517', '520', 'PAIO MENDES' ] ] },
        'Gravulha' => {
          '2240' => [
            [ '037', '042', 'GUAS BELAS FZZ' ] ] },
        'Ferreira do Zzere' => {
          '2240' => [
            [ '346', '354', 'FERREIRA DO ZZERE' ],
            [ '356', '382', 'FERREIRA DO ZZERE' ] ] },
        'Chos' => {
          '2240' => [
            [ '300', '300', 'CHOS' ] ] },
        'Casal Farroeiro' => {
          '2240' => [
            [ '100', '102', 'AREIAS FZZ' ] ] },
        'Cardal' => {
          '2240' => [
            [ '324', '344', 'FERREIRA DO ZZERE' ] ] },
        'Levada' => {
          '2240' => [
            [ '501', '505', 'PAIO MENDES' ] ] },
        'Horta da Fonte' => {
          '2240' => [
            [ '015', '017', 'GUAS BELAS FZZ' ] ] },
        'Bairrada' => {
          '2240' => [
            [ '321', '322', 'FERREIRA DO ZZERE' ] ] },
        'Louriceira' => {
          '2240' => [
            [ '562', '568', 'PIAS FZZ' ] ] },
        'Casal do Varela' => {
          '2240' => [
            [ '008', '012', 'GUAS BELAS FZZ' ] ] },
        'Alqueido de Santo Amaro' => {
          '2240' => [
            [ '201', '226', 'BECO' ] ] },
        'Barcai' => {
          '2240' => [
            [ '001', '006', 'GUAS BELAS FZZ' ] ] },
        'Sobreiras' => {
          '2240' => [
            [ '023', '035', 'GUAS BELAS FZZ' ] ] },
        'Matos' => {
          '2240' => [
            [ '429', '442', 'IGREJA NOVA DO SOBRAL' ] ] },
        'Palheiros' => {
          '2240' => [
            [ '019', '021', 'GUAS BELAS FZZ' ] ] }
              },
      'Sardoal' => {
        'Amieira' => {
          '2230' => [
            [ '051', '063', 'SANTIAGO DE MONTALEGRE' ] ] },
        'Tapada da Torre' => {
          '2230' => [
            [ '161', '164', 'SARDOAL' ] ] },
        'Sardoal' => {
          '2230' => [
            [ '121', '145', 'SARDOAL' ],
            [ '200', '236', 'SARDOAL' ] ] },
        'Casal Pedro Maia' => {
          '2230' => [
            [ '001', '015', 'ALCARAVELA' ] ] },
        'Valhascos' => {
          '2230' => [
            [ '180', '180', 'VALHASCOS' ] ] },
        'Andreus' => {
          '2230' => [
            [ '101', '108', 'SARDOAL' ] ] },
        'Venda' => {
          '2230' => [
            [ '017', '017', 'ALCARAVELA' ] ] }
               },
      'Almeirim' => {
        'Raposa' => {
          '2080' => [
            [ '701', '721', 'RAPOSA' ] ] },
        'Casalinho Velho' => {
          '2080' => [
            [ '368', '368', 'BENFICA DO RIBATEJO' ] ] },
        'Casal Monteiro' => {
          '2080' => [
            [ '201', '202', 'ALMEIRIM' ] ] },
        'Benfica do Ribatejo' => {
          '2080' => [
            [ '321', '332', 'BENFICA DO RIBATEJO' ],
            [ '334', '349', 'BENFICA DO RIBATEJO' ] ] },
        'Casal Adrio' => {
          '2080' => [
            [ '360', '360', 'BENFICA DO RIBATEJO' ] ] },
        'Fazendas de Almeirim' => {
          '2080' => [
            [ '500', '503', 'FAZENDAS DE ALMEIRIM' ],
            [ '511', '528', 'FAZENDAS DE ALMEIRIM' ],
            [ '530', '533', 'FAZENDAS DE ALMEIRIM' ],
            [ '537', '542', 'FAZENDAS DE ALMEIRIM' ],
            [ '544', '547', 'FAZENDAS DE ALMEIRIM' ],
            [ '549', '551', 'FAZENDAS DE ALMEIRIM' ],
            [ '555', '555', 'FAZENDAS DE ALMEIRIM' ],
            [ '557', '569', 'FAZENDAS DE ALMEIRIM' ],
            [ '571', '576', 'FAZENDAS DE ALMEIRIM' ],
            [ '578', '579', 'FAZENDAS DE ALMEIRIM' ],
            [ '581', '593', 'FAZENDAS DE ALMEIRIM' ],
            [ '595', '598', 'FAZENDAS DE ALMEIRIM' ],
            [ '600', '602', 'FAZENDAS DE ALMEIRIM' ],
            [ '604', '604', 'FAZENDAS DE ALMEIRIM' ],
            [ '609', '619', 'FAZENDAS DE ALMEIRIM' ],
            [ '621', '646', 'FAZENDAS DE ALMEIRIM' ] ] },
        'Foros de Benfica' => {
          '2080' => [
            [ '400', '407', 'BENFICA DO RIBATEJO' ] ] },
        'Feteira' => {
          '2080' => [
            [ '620', '620', 'ALMEIRIM' ] ] },
        'Quinta da Gafaria' => {
          '2080' => [
            [ '204', '205', 'ALMEIRIM' ] ] },
        'Casal Branco' => {
          '2080' => [
            [ '362', '362', 'BENFICA DO RIBATEJO' ] ] },
        'Tapada' => {
          '2080' => [
            [ '220', '222', 'ALMEIRIM' ] ] },
        'Casal M Charana' => {
          '2080' => [
            [ '366', '366', 'BENFICA DO RIBATEJO' ] ] },
        'Vale Fornos' => {
          '2080' => [
            [ '653', '653', 'FAZENDAS DE ALMEIRIM' ] ] },
        'Almeirim' => {
          '2080' => [
            [ '001', '001', 'ALMEIRIM' ],
            [ '003', '014', 'ALMEIRIM' ],
            [ '018', '018', 'ALMEIRIM' ],
            [ '020', '022', 'ALMEIRIM' ],
            [ '026', '026', 'ALMEIRIM' ],
            [ '028', '028', 'ALMEIRIM' ],
            [ '030', '031', 'ALMEIRIM' ],
            [ '033', '105', 'ALMEIRIM' ],
            [ '107', '112', 'ALMEIRIM' ],
            [ '114', '127', 'ALMEIRIM' ],
            [ '129', '133', 'ALMEIRIM' ],
            [ '135', '142', 'ALMEIRIM' ],
            [ '144', '145', 'ALMEIRIM' ],
            [ '147', '165', 'ALMEIRIM' ],
            [ '169', '176', 'ALMEIRIM' ],
            [ '178', '196', 'ALMEIRIM' ] ] },
        'Casal Fris' => {
          '2080' => [
            [ '364', '364', 'BENFICA DO RIBATEJO' ] ] },
        'Cortiis' => {
          '2080' => [
            [ '380', '389', 'BENFICA DO RIBATEJO' ],
            [ '391', '398', 'BENFICA DO RIBATEJO' ] ] },
        'Arneiro da Azeitada' => {
          '2080' => [
            [ '302', '303', 'BENFICA DO RIBATEJO' ] ] }
          },
      'Alpiara' => {
        'Zona Industrial' => {
          '2090' => [
            [ '242', '242', 'ALPIARA' ] ] },
        'Olival de Atela' => {
          '2090' => [
            [ '224', '226', 'ALPIARA' ] ] },
        'Atela' => {
          '2090' => [
            [ '201', '202', 'ALPIARA' ] ] },
        'Alpiara' => {
          '2090' => [
            [ '020', '168', 'ALPIARA' ] ] },
        'Charneca de Frade' => {
          '2090' => [
            [ '211', '214', 'ALPIARA' ] ] },
        'Torrinha' => {
          '2090' => [
            [ '236', '237', 'ALPIARA' ] ] },
        'Casalinho' => {
          '2090' => [
            [ '209', '209', 'ALPIARA' ] ] },
        'Lagualva de Baixo' => {
          '2090' => [
            [ '221', '222', 'ALPIARA' ] ] },
        'Frade de Baixo' => {
          '2090' => [
            [ '216', '219', 'ALPIARA' ] ] }
          },
      'Constncia' => {
        'Enxertal' => {
          '2250' => [
            [ '320', '335', 'STA MARGARIDA DA COUTADA' ] ] },
        'Constncia' => {
          '2250' => [
            [ '020', '032', 'CONSTNCIA' ],
            [ '034', '035', 'CONSTNCIA' ],
            [ '037', '055', 'CONSTNCIA' ],
            [ '058', '094', 'CONSTNCIA' ] ] },
        'So Bento' => {
          '2250' => [
            [ '380', '385', 'STA MARGARIDA DA COUTADA' ] ] },
        'Montalvo' => {
          '2250' => [
            [ '220', '220', 'MONTALVO' ],
            [ '222', '250', 'MONTALVO' ] ] },
        'Almegue' => {
          '2250' => [
            [ '011', '014', 'CONSTNCIA' ] ] },
        'Quinta da Gorda' => {
          '2250' => [
            [ '271', '273', 'MONTALVO' ] ] },
        'Santa Margarida da Coutada' => {
          '2250' => [
            [ '350', '373', 'STA MARGARIDA DA COUTADA' ] ] },
        'Pereira' => {
          '2250' => [
            [ '340', '343', 'STA MARGARIDA DA COUTADA' ] ] },
        'Cabeo' => {
          '2250' => [
            [ '211', '216', 'MONTALVO' ] ] },
        'Bairro' => {
          '2250' => [
            [ '311', '315', 'STA MARGARIDA DA COUTADA' ] ] }
            },
      'Alcanena' => {
        'Zona Industrial' => {
          '2380' => [
            [ '633', '666', 'VILA MOREIRA' ] ] },
        'Malhou' => {
          '2380' => [
            [ '534', '547', 'MALHOU' ] ] },
        'Carvalheiro' => {
          '2380' => [
            [ '401', '436', 'LOURICEIRA' ] ] },
        'Cho de Cima' => {
          '2380' => [
            [ '501', '532', 'MALHOU' ] ] },
        'Quinta do Alviela' => {
          '2380' => [
            [ '450', '450', 'LOURICEIRA' ] ] },
        'Caneiros' => {
          '2380' => [
            [ '561', '563', 'MOITAS VENDA' ] ] },
        'Cabeo' => {
          '2380' => [
            [ '570', '575', 'MONSANTO ACN' ] ] },
        'Ches' => {
          '2380' => [
            [ '153', '173', 'ALCANENA' ] ] },
        'Alcanena' => {
          '2380' => [
            [ '011', '091', 'ALCANENA' ] ],
          '2384' => [
            [ '003', '003', 'ALCANENA' ] ] },
        'Espinheiro' => {
          '2380' => [
            [ '301', '337', 'ESPINHEIRO' ] ] },
        'Peral' => {
          '2380' => [
            [ '180', '189', 'ALCANENA' ] ] },
        'Bugalhos' => {
          '2380' => [
            [ '220', '226', 'BUGALHOS' ] ] },
        'Vale Alto' => {
          '2395' => [
            [ '301', '344', 'MINDE' ] ] },
        'Vale Calado' => {
          '2380' => [
            [ '631', '631', 'VILA MOREIRA' ] ] },
        'Minde' => {
          '2395' => [
            [ '101', '110', 'MINDE' ],
            [ '112', '160', 'MINDE' ],
            [ '162', '217', 'MINDE' ] ] },
        'Casal Penhores' => {
          '2380' => [
            [ '151', '151', 'ALCANENA' ] ] },
        'Moita' => {
          '2380' => [
            [ '550', '550', 'MALHOU' ] ] },
        'Covo do Coelho' => {
          '2395' => [
            [ '001', '041', 'MINDE' ] ] },
        'Bajouco' => {
          '2380' => [
            [ '601', '608', 'SERRA DE SANTO ANTNIO' ] ] }
          },
      'Vila Nova da Barquinha' => {
        'Cardal' => {
          '2260' => [
            [ '501', '522', 'VILA NOVA DA BARQUINHA' ] ] },
        'Limeiras' => {
          '2260' => [
            [ '110', '124', 'PRAIA DO RIBATEJO' ] ] },
        'Casal da Cre' => {
          '2260' => [
            [ '061', '066', 'ATALAIA VNB' ] ] },
        'Azenha' => {
          '2260' => [
            [ '301', '308', 'TANCOS' ] ] },
        'Atalaia' => {
          '2260' => [
            [ '011', '050', 'ATALAIA VNB' ] ] },
        'Vila Nova da Barquinha' => {
          '2260' => [
            [ '368', '381', 'VILA NOVA DA BARQUINHA' ],
            [ '387', '387', 'VILA NOVA DA BARQUINHA' ],
            [ '389', '452', 'VILA NOVA DA BARQUINHA' ] ] },
        'Tancos' => {
          '2260' => [
            [ '310', '331', 'TANCOS' ] ] },
        'Madeiras' => {
          '2260' => [
            [ '094', '106', 'PRAIA DO RIBATEJO' ],
            [ '141', '158', 'PRAIA DO RIBATEJO' ] ] },
        'Quatro Estradas' => {
          '2260' => [
            [ '260', '262', 'PRAIA DO RIBATEJO' ] ] },
        'Praia do Ribatejo' => {
          '2260' => [
            [ '201', '246', 'PRAIA DO RIBATEJO' ] ] },
        'Quinta da Ponte da Pedra' => {
          '2260' => [
            [ '598', '599', 'VILA NOVA DA BARQUINHA' ] ] },
        'Moita do Norte' => {
          '2260' => [
            [ '541', '584', 'VILA NOVA DA BARQUINHA' ] ] }
                  },
      'Chamusca' => {
        'Caneira' => {
          '2140' => [
            [ '501', '537', 'PARREIRA' ] ] },
        'Areolas' => {
          '2140' => [
            [ '301', '316', 'PINHEIRO GRANDE' ] ] },
        'Chamusca' => {
          '2140' => [
            [ '051', '089', 'CHAMUSCA' ],
            [ '091', '167', 'CHAMUSCA' ] ] },
        'Carregueira' => {
          '2140' => [
            [ '651', '680', 'CARREGUEIRA' ] ] },
        'Arrepiado' => {
          '2140' => [
            [ '601', '618', 'CARREGUEIRA' ] ] },
        'Casal da Pocaria' => {
          '2140' => [
            [ '351', '386', 'ULME' ] ] },
        'Convento de Santo Antnio' => {
          '2140' => [
            [ '180', '184', 'CHAMUSCA' ] ] },
        'Arneiro de Cima' => {
          '2140' => [
            [ '011', '015', 'CHAMUSCA' ] ] },
        'Quinta das Sesmarias' => {
          '2140' => [
            [ '401', '434', 'VALE DE CAVALOS' ] ] },
        'Casais do Bom Nome' => {
          '2140' => [
            [ '201', '225', 'CHOUTO' ] ] }
          },
      'Benavente' => {
        'Vau' => {
          '2890' => [
            [ '405', '405', 'SAMORA CORREIA' ] ] },
        'Barrosa' => {
          '2130' => [
            [ '162', '178', 'BARROSA' ] ] },
        'Pancas' => {
          '2135' => [
            [ '012', '038', 'SAMORA CORREIA' ] ] },
        'Bate Orelhas' => {
          '2135' => [
            [ '004', '005', 'SAMORA CORREIA' ] ] },
        'Benavente' => {
          '2130' => [
            [ '010', '017', 'BENAVENTE' ],
            [ '019', '027', 'BENAVENTE' ],
            [ '029', '075', 'BENAVENTE' ],
            [ '181', '270', 'BENAVENTE' ] ] },
        'Bilrete' => {
          '2120' => [
            [ '001', '001', 'BENAVENTE' ] ] },
        'Cerrado das guas' => {
          '2130' => [
            [ '101', '111', 'BENAVENTE' ] ] },
        'Foros de Almada' => {
          '2130' => [
            [ '121', '121', 'SANTO ESTEVO BNV' ] ] },
        'Santo Estevo' => {
          '2130' => [
            [ '123', '154', 'SANTO ESTEVO BNV' ] ] },
        'Bela Vista' => {
          '2890' => [
            [ '401', '403', 'SAMORA CORREIA' ] ] },
        'Adema' => {
          '2135' => [
            [ '001', '002', 'SAMORA CORREIA' ] ] },
        'Tapada dos Curralinhos' => {
          '2135' => [
            [ '406', '407', 'SAMORA CORREIA' ] ] },
        'Catapereiro' => {
          '2135' => [
            [ '007', '010', 'SAMORA CORREIA' ] ] },
        'Porto Alto' => {
          '2135' => [
            [ '104', '108', 'SAMORA CORREIA' ],
            [ '111', '116', 'SAMORA CORREIA' ],
            [ '118', '121', 'SAMORA CORREIA' ],
            [ '123', '124', 'SAMORA CORREIA' ],
            [ '126', '126', 'SAMORA CORREIA' ],
            [ '128', '128', 'SAMORA CORREIA' ],
            [ '130', '133', 'SAMORA CORREIA' ],
            [ '135', '137', 'SAMORA CORREIA' ],
            [ '139', '155', 'SAMORA CORREIA' ] ] },
        'Samora Correia' => {
          '2135' => [
            [ '040', '041', 'SAMORA CORREIA' ],
            [ '043', '054', 'SAMORA CORREIA' ],
            [ '056', '068', 'SAMORA CORREIA' ],
            [ '070', '075', 'SAMORA CORREIA' ],
            [ '077', '077', 'SAMORA CORREIA' ],
            [ '079', '100', 'SAMORA CORREIA' ],
            [ '201', '232', 'SAMORA CORREIA' ],
            [ '235', '238', 'SAMORA CORREIA' ],
            [ '240', '242', 'SAMORA CORREIA' ],
            [ '244', '254', 'SAMORA CORREIA' ],
            [ '256', '259', 'SAMORA CORREIA' ],
            [ '261', '286', 'SAMORA CORREIA' ],
            [ '288', '293', 'SAMORA CORREIA' ],
            [ '295', '298', 'SAMORA CORREIA' ],
            [ '301', '301', 'SAMORA CORREIA' ],
            [ '303', '309', 'SAMORA CORREIA' ],
            [ '311', '312', 'SAMORA CORREIA' ],
            [ '316', '323', 'SAMORA CORREIA' ],
            [ '401', '402', 'SAMORA CORREIA' ] ] }
           },
      'Entroncamento' => {
        'Casal do Bonito' => {
          '2330' => [
            [ '506', '506', 'ENTRONCAMENTO' ] ] },
        'Barreira da Fita' => {
          '2330' => [
            [ '501', '503', 'ENTRONCAMENTO' ] ] },
        'Casal do Conde' => {
          '2330' => [
            [ '521', '530', 'ENTRONCAMENTO' ] ] },
        'Coves de Baixo' => {
          '2330' => [
            [ '509', '513', 'ENTRONCAMENTO' ] ] },
        'Foros da Lameira' => {
          '2330' => [
            [ '515', '519', 'ENTRONCAMENTO' ] ] },
        'Entroncamento' => {
          '2330' => [
            [ '001', '141', 'ENTRONCAMENTO' ],
            [ '143', '163', 'ENTRONCAMENTO' ],
            [ '165', '224', 'ENTRONCAMENTO' ],
            [ '227', '267', 'ENTRONCAMENTO' ] ] }
               },
      'Tomar' => {
        'Alviobeira' => {
          '2305' => [
            [ '061', '075', 'ALVIOBEIRA' ] ] },
        'Cadaval' => {
          '2305' => [
            [ '551', '551', 'PEDREIRA TMR' ] ] },
        'Casalinho' => {
          '2305' => [
            [ '610', '632', 'SABACHEIRA' ] ] },
        'Casal das Freiras' => {
          '2305' => [
            [ '409', '414', 'MADALENA TMR' ] ] },
        'Bexiga' => {
          '2305' => [
            [ '501', '520', 'PAIALVO' ] ] },
        'Quinta da Gndara' => {
          '2300' => [
            [ '413', '431', 'TOMAR' ] ] },
        'Boca da Mata' => {
          '2300' => [
            [ '158', '164', 'SO PEDRO DE TOMAR' ] ] },
        'Granja das Fontes' => {
          '2305' => [
            [ '228', '242', 'CARREGUEIROS' ] ] },
        'Casal da Ferssura' => {
          '2300' => [
            [ '363', '363', 'TOMAR' ] ] },
        'Abadia' => {
          '2300' => [
            [ '211', '266', 'SERRA' ] ] },
        'Castelo do Bode' => {
          '2300' => [
            [ '196', '196', 'SO PEDRO DE TOMAR' ] ] },
        'Alto da Barca Nova' => {
          '2300' => [
            [ '151', '154', 'SO PEDRO DE TOMAR' ] ] },
        'Barroqueiros' => {
          '2305' => [
            [ '202', '226', 'CARREGUEIROS' ] ] },
        'Mataco' => {
          '2300' => [
            [ '029', '036', 'JUNCEIRA' ] ] },
        'Casal Marmelo' => {
          '2305' => [
            [ '400', '407', 'MADALENA TMR' ] ] },
        'Quinta do Troviscal' => {
          '2300' => [
            [ '186', '194', 'SO PEDRO DE TOMAR' ] ] },
        'Hortinha' => {
          '2300' => [
            [ '023', '024', 'JUNCEIRA' ] ] },
        'gua Boa' => {
          '2305' => [
            [ '601', '604', 'SABACHEIRA' ] ] },
        'Casal dos Tomazes' => {
          '2300' => [
            [ '166', '184', 'SO PEDRO DE TOMAR' ] ] },
        'Tomar' => {
          '2300' => [
            [ '320', '335', 'TOMAR' ],
            [ '337', '361', 'TOMAR' ],
            [ '408', '411', 'TOMAR' ],
            [ '434', '434', 'TOMAR' ],
            [ '436', '565', 'TOMAR' ],
            [ '567', '609', 'TOMAR' ],
            [ '611', '617', 'TOMAR' ],
            [ '619', '620', 'TOMAR' ],
            [ '622', '635', 'TOMAR' ],
            [ '638', '641', 'TOMAR' ] ] },
        'Juncais de Cima' => {
          '2300' => [
            [ '397', '406', 'TOMAR' ] ] },
        'Casal Mouzinho' => {
          '2300' => [
            [ '642', '642', 'SO PEDRO DE TOMAR' ] ] },
        'Alto do Pintado' => {
          '2305' => [
            [ '300', '336', 'CASAIS TMR' ] ] },
        'Asseiceira' => {
          '2305' => [
            [ '101', '118', 'ASSEICEIRA TMR' ] ] },
        'Beberriqueira' => {
          '2300' => [
            [ '156', '156', 'SO PEDRO DE TOMAR' ] ] },
        'Algarvias' => {
          '2300' => [
            [ '302', '305', 'TOMAR' ] ] },
        'Alto de So Loureno' => {
          '2300' => [
            [ '307', '308', 'TOMAR' ] ] },
        'Casal Joo Dias' => {
          '2305' => [
            [ '415', '415', 'PAIALVO' ] ] },
        'Madalena' => {
          '2305' => [
            [ '425', '433', 'MADALENA TMR' ] ] },
        'Quinta de Cima' => {
          '2305' => [
            [ '119', '119', 'BESELGA TMR' ],
            [ '435', '436', 'MADALENA TMR' ] ] },
        'Santa Margarida' => {
          '2305' => [
            [ '438', '446', 'MADALENA TMR' ] ] },
        'Balancho' => {
          '2300' => [
            [ '011', '017', 'JUNCEIRA' ] ] },
        'Casal da Brava' => {
          '2305' => [
            [ '606', '608', 'SABACHEIRA' ] ] },
        'Casal Negro' => {
          '2305' => [
            [ '416', '423', 'MADALENA TMR' ] ] },
        'Casal da Pombinha' => {
          '2300' => [
            [ '365', '395', 'TOMAR' ] ] },
        'Aboboreira Cimeira' => {
          '2300' => [
            [ '061', '110', 'OLALHAS' ] ] },
        'Fonte Quente' => {
          '2305' => [
            [ '553', '558', 'PEDREIRA TMR' ] ] },
        'Alvito' => {
          '2300' => [
            [ '310', '313', 'TOMAR' ] ] },
        'Quinta de Santa Ana' => {
          '2305' => [
            [ '120', '131', 'ASSEICEIRA TMR' ] ] },
        'Assamassa' => {
          '2305' => [
            [ '161', '180', 'BESELGA TMR' ] ] },
        'Bemposta' => {
          '2300' => [
            [ '637', '637', 'SO PEDRO DE TOMAR' ] ] },
        'Barreiro' => {
          '2300' => [
            [ '315', '318', 'TOMAR' ] ] },
        'Carvalhal da Pvoa' => {
          '2305' => [
            [ '011', '026', 'ALM DA RIBEIRA' ] ] }
             },
      'Coruche' => {
        'Terrafeiro' => {
          '2100' => [
            [ '670', '675', 'SANTANA DO MATO' ] ] },
        'Medronheira' => {
          '2100' => [
            [ '400', '408', 'SO JOS DA LAMAROSA' ] ] },
        'Deserto' => {
          '2100' => [
            [ '370', '379', 'COUO' ] ] },
        'Fazenda da Torre' => {
          '2100' => [
            [ '600', '602', 'BRANCA CCH' ] ] },
        'Couo' => {
          '2100' => [
            [ '353', '353', 'COUO' ] ] },
        'Palminha' => {
          '2100' => [
            [ '300', '350', 'COUO' ] ] },
        'Bairro da Areia' => {
          '2100' => [
            [ '018', '022', 'CORUCHE' ] ] },
        'Mata Lobos' => {
          '2100' => [
            [ '650', '653', 'BISCANHO' ] ] },
        'Fazenda dos Pelados' => {
          '2100' => [
            [ '604', '608', 'BRANCA CCH' ] ] },
        'Montes Emdios' => {
          '2100' => [
            [ '540', '541', 'FAJARDA' ] ] },
        'Fazenda da Ribeira' => {
          '2100' => [
            [ '620', '627', 'ERRA' ] ] },
        'Bogalhes' => {
          '2100' => [
            [ '024', '057', 'CORUCHE' ] ] },
        'Agolada' => {
          '2100' => [
            [ '011', '016', 'CORUCHE' ] ] },
        'Coruche' => {
          '2100' => [
            [ '100', '108', 'CORUCHE' ],
            [ '110', '131', 'CORUCHE' ],
            [ '133', '155', 'CORUCHE' ],
            [ '157', '198', 'CORUCHE' ],
            [ '200', '228', 'CORUCHE' ] ] },
        'Monte das Figueiras' => {
          '2100' => [
            [ '500', '537', 'FAJARDA' ] ] }
               },
      'Ourm' => {
        'Piso do Oleiro' => {
          '2435' => [
            [ '102', '133', 'CAXARIAS' ] ] },
        'Boleiros' => {
          '2495' => [
            [ '310', '340', 'FTIMA' ] ] },
        'Castelo' => {
          '2435' => [
            [ '036', '100', 'CAXARIAS' ] ] },
        'Alvega' => {
          '2490' => [
            [ '051', '098', 'ATOUGUIA' ] ] },
        'Aljustrel' => {
          '2495' => [
            [ '301', '303', 'FTIMA' ] ] },
        'Vale do Freixo' => {
          '2435' => [
            [ '178', '179', 'ESPITE' ] ] },
        'Olival' => {
          '2435' => [
            [ '451', '464', 'OLIVAL VNO' ] ] },
        'Aldeia de Santa Teresa' => {
          '2435' => [
            [ '250', '265', 'FREIXIANDA' ] ] },
        'Abadia' => {
          '2435' => [
            [ '021', '033', 'CAXARIAS' ] ] },
        'Berimbela' => {
          '2435' => [
            [ '501', '512', 'RIBEIRA DO FRRIO' ] ] },
        'Vale das Antas' => {
          '2435' => [
            [ '730', '731', 'URQUEIRA' ] ] },
        'Pinhais Novos' => {
          '2435' => [
            [ '165', '166', 'ESPITE' ] ] },
        'Ladeira' => {
          '2435' => [
            [ '155', '155', 'ESPITE' ] ] },
        'Salgueiral' => {
          '2435' => [
            [ '170', '170', 'ESPITE' ] ] },
        'Aldeia Nova' => {
          '2435' => [
            [ '421', '441', 'OLIVAL VNO' ] ] },
        'Cercal' => {
          '2490' => [
            [ '101', '123', 'CERCAL VNO' ] ] },
        'Outeiro das Gameiras' => {
          '2435' => [
            [ '405', '405', 'MATAS' ] ] },
        'Agroal' => {
          '2435' => [
            [ '201', '212', 'FORMIGAIS' ] ] },
        'Outeiro da Bica' => {
          '2490' => [
            [ '161', '163', 'GONDEMARIA' ] ] },
        'Cova da Iria' => {
          '2495' => [
            [ '401', '454', 'FTIMA' ],
            [ '456', '457', 'FTIMA' ] ] },
        'Eira da Pedra' => {
          '2495' => [
            [ '501', '530', 'FTIMA' ] ] },
        'Areias' => {
          '2490' => [
            [ '124', '132', 'GONDEMARIA' ] ] },
        'Cristvos' => {
          '2435' => [
            [ '539', '569', 'SEIA' ] ] },
        'Abades' => {
          '2435' => [
            [ '221', '238', 'FREIXIANDA' ] ] },
        'Lameirinha' => {
          '2435' => [
            [ '575', '579', 'SEIA' ] ] },
        'Sesmarias' => {
          '2435' => [
            [ '173', '173', 'ESPITE' ] ] },
        'Freixianda' => {
          '2435' => [
            [ '271', '287', 'FREIXIANDA' ] ] },
        'Moimento' => {
          '2495' => [
            [ '650', '661', 'FTIMA' ] ] },
        'Toucinhos' => {
          '2490' => [
            [ '040', '040', 'ALBURITEL' ] ] },
        'Casais Galegos' => {
          '2435' => [
            [ '034', '035', 'CASAL DOS BERNARDOS' ] ] },
        'Arieiro' => {
          '2435' => [
            [ '141', '153', 'ESPITE' ] ] },
        'Maia' => {
          '2435' => [
            [ '160', '160', 'ESPITE' ] ] },
        'Matas' => {
          '2435' => [
            [ '340', '347', 'MATAS' ],
            [ '391', '392', 'MATAS' ],
            [ '397', '399', 'MATAS' ] ] },
        'Giesteira' => {
          '2495' => [
            [ '590', '620', 'FTIMA' ] ] },
        'Gondemaria' => {
          '2490' => [
            [ '135', '152', 'GONDEMARIA' ] ] },
        'Vale do Juncal' => {
          '2435' => [
            [ '320', '321', 'FREIXIANDA' ] ] },
        'Casal de Baixo' => {
          '2435' => [
            [ '521', '533', 'RIO DE COUROS' ] ] },
        'Maxieira' => {
          '2495' => [
            [ '641', '641', 'FTIMA' ] ] },
        'Ftima' => {
          '2495' => [
            [ '551', '564', 'FTIMA' ] ] },
        'Olheiros' => {
          '2435' => [
            [ '101', '101', 'CASAL DOS BERNARDOS' ] ] },
        'Poas' => {
          '2435' => [
            [ '407', '407', 'MATAS' ] ] },
        'Peras Ruivas' => {
          '2435' => [
            [ '581', '661', 'SEIA' ] ] },
        'Lomba de gua' => {
          '2495' => [
            [ '630', '630', 'FTIMA' ] ] },
        'Galinhas' => {
          '2435' => [
            [ '301', '317', 'FREIXIANDA' ] ] },
        'Casa Velha' => {
          '2495' => [
            [ '350', '354', 'FTIMA' ] ] },
        'Vale da M' => {
          '2435' => [
            [ '175', '175', 'ESPITE' ] ] },
        'Amieira' => {
          '2435' => [
            [ '681', '718', 'URQUEIRA' ] ] },
        'Cacinheira' => {
          '2435' => [
            [ '001', '013', 'CASAL DOS BERNARDOS' ] ] },
        'Alburitel' => {
          '2490' => [
            [ '001', '033', 'ALBURITEL' ] ] },
        'Gaiola' => {
          '2495' => [
            [ '580', '580', 'FTIMA' ] ] },
        'Lavradio' => {
          '2435' => [
            [ '351', '351', 'MATAS' ],
            [ '356', '356', 'MATAS' ],
            [ '359', '360', 'MATAS' ],
            [ '362', '364', 'MATAS' ],
            [ '367', '367', 'MATAS' ],
            [ '369', '369', 'MATAS' ],
            [ '371', '375', 'MATAS' ],
            [ '377', '378', 'MATAS' ],
            [ '800', '814', 'MATAS' ] ] },
        'Ourm' => {
          '2490' => [
            [ '201', '305', 'OURM' ],
            [ '307', '346', 'OURM' ],
            [ '348', '377', 'OURM' ],
            [ '379', '773', 'OURM' ],
            [ '776', '780', 'OURM' ] ] },
        'Pairia' => {
          '2435' => [
            [ '471', '495', 'OLIVAL VNO' ] ] }
             },
      'Abrantes' => {
        'Atalaia' => {
          '2230' => [
            [ '801', '807', 'SOUTO ABT' ] ] },
        'Pego' => {
          '2205' => [
            [ '380', '380', 'PEGO' ] ] },
        'Aldeinha' => {
          '2200' => [
            [ '751', '763', 'RIO DE MOINHOS ABT' ] ] },
        'Alvega' => {
          '2205' => [
            [ '101', '120', 'ALVEGA' ] ] },
        'So Miguel do Rio Torto' => {
          '2205' => [
            [ '551', '580', 'SO MIGUEL DO RIO TORTO' ] ] },
        'Bicas' => {
          '2205' => [
            [ '491', '511', 'SO MIGUEL DO RIO TORTO' ] ] },
        'Vale da Horta' => {
          '2205' => [
            [ '410', '411', 'SO FACUNDO' ] ] },
        'Cho de Lucas' => {
          '2205' => [
            [ '610', '721', 'TRAMAGAL' ] ] },
        'Vale de Aor' => {
          '2205' => [
            [ '409', '409', 'BEMPOSTA ABT' ] ] },
        'Cabeo M Jacob' => {
          '2205' => [
            [ '531', '537', 'SO MIGUEL DO RIO TORTO' ] ] },
        'Casal de Cortido' => {
          '2205' => [
            [ '251', '251', 'CONCAVADA' ] ] },
        'Alagoa' => {
          '2200' => [
            [ '621', '650', 'MARTINCHEL' ] ] },
        'Vale Vinha' => {
          '2200' => [
            [ '601', '601', 'ALDEIA DO MATO' ] ] },
        'gua Travessa' => {
          '2205' => [
            [ '161', '162', 'BEMPOSTA ABT' ] ] },
        'Barrada' => {
          '2205' => [
            [ '401', '408', 'SO FACUNDO' ] ] },
        'Rio de Moinhos' => {
          '2200' => [
            [ '781', '804', 'RIO DE MOINHOS ABT' ] ] },
        'Vale Cortias Baixo' => {
          '2205' => [
            [ '595', '596', 'SO MIGUEL DO RIO TORTO' ] ] },
        'Aldeias' => {
          '2200' => [
            [ '671', '733', 'MOURISCAS' ] ] },
        'Brunheirinho' => {
          '2205' => [
            [ '201', '217', 'BEMPOSTA ABT' ] ] },
        'Monte Novo' => {
          '2205' => [
            [ '290', '292', 'CONCAVADA' ] ] },
        'Casal de Santo Isidro' => {
          '2205' => [
            [ '301', '377', 'PEGO' ] ] },
        'Tramagal' => {
          '2205' => [
            [ '387', '387', 'TRAMAGAL' ],
            [ '430', '430', 'TRAMAGAL' ] ] },
        'Tapada do Chafariz' => {
          '2200' => [
            [ '512', '514', 'ABRANTES' ] ] },
        'Bemposta' => {
          '2205' => [
            [ '171', '183', 'BEMPOSTA ABT' ],
            [ '194', '194', 'BEMPOSTA ABT' ] ] },
        'gua das Casas' => {
          '2230' => [
            [ '831', '840', 'FONTES ABT' ] ] },
        'Concavada' => {
          '2205' => [
            [ '255', '284', 'CONCAVADA' ] ] },
        'Carril' => {
          '2230' => [
            [ '861', '866', 'CARVALHAL ABT' ] ] },
        'Areia de Baixo' => {
          '2205' => [
            [ '131', '141', 'ALVEGA' ] ] },
        'Casal da Ferrugenta' => {
          '2205' => [
            [ '801', '831', 'VALE DAS MS' ] ] },
        'Areias' => {
          '2205' => [
            [ '451', '471', 'SO MIGUEL DO RIO TORTO' ] ] },
        'Abrantes' => {
          '2205' => [
            [ '001', '076', 'ABRANTES' ] ],
          '2200' => [
            [ '001', '082', 'ABRANTES' ],
            [ '084', '131', 'ABRANTES' ],
            [ '133', '217', 'ABRANTES' ],
            [ '219', '223', 'ABRANTES' ],
            [ '225', '258', 'ABRANTES' ],
            [ '260', '384', 'ABRANTES' ],
            [ '386', '386', 'ABRANTES' ],
            [ '388', '429', 'ABRANTES' ],
            [ '431', '445', 'ABRANTES' ],
            [ '447', '510', 'ABRANTES' ] ] }
          }
      },
    'Lisboa' => {
      'Lisboa' => {
        'Lisboa' => {
          '1700' => [
            [ '001', '029', 'LISBOA' ],
            [ '031', '033', 'LISBOA' ],
            [ '035', '039', 'LISBOA' ],
            [ '041', '052', 'LISBOA' ],
            [ '056', '056', 'LISBOA' ],
            [ '059', '063', 'LISBOA' ],
            [ '066', '067', 'LISBOA' ],
            [ '069', '079', 'LISBOA' ],
            [ '081', '082', 'LISBOA' ],
            [ '084', '109', 'LISBOA' ],
            [ '111', '112', 'LISBOA' ],
            [ '114', '129', 'LISBOA' ],
            [ '131', '138', 'LISBOA' ],
            [ '140', '173', 'LISBOA' ],
            [ '175', '207', 'LISBOA' ],
            [ '209', '240', 'LISBOA' ],
            [ '243', '268', 'LISBOA' ],
            [ '271', '275', 'LISBOA' ],
            [ '277', '277', 'LISBOA' ],
            [ '284', '284', 'LISBOA' ],
            [ '288', '288', 'LISBOA' ],
            [ '291', '291', 'LISBOA' ],
            [ '293', '300', 'LISBOA' ],
            [ '302', '309', 'LISBOA' ],
            [ '313', '327', 'LISBOA' ],
            [ '329', '332', 'LISBOA' ],
            [ '334', '360', 'LISBOA' ],
            [ '363', '371', 'LISBOA' ],
            [ '401', '401', 'LISBOA' ],
            [ '411', '414', 'LISBOA' ],
            [ '416', '419', 'LISBOA' ],
            [ '921', '921', 'LISBOA' ] ],
          '1169' => [
            [ '006', '010', 'LISBOA' ],
            [ '012', '012', 'LISBOA' ],
            [ '015', '019', 'LISBOA' ],
            [ '021', '024', 'LISBOA' ],
            [ '027', '030', 'LISBOA' ],
            [ '032', '033', 'LISBOA' ],
            [ '035', '037', 'LISBOA' ],
            [ '039', '043', 'LISBOA' ],
            [ '045', '056', 'LISBOA' ],
            [ '058', '068', 'LISBOA' ],
            [ '070', '072', 'LISBOA' ],
            [ '074', '077', 'LISBOA' ],
            [ '079', '096', 'LISBOA' ],
            [ '099', '116', 'LISBOA' ],
            [ '118', '120', 'LISBOA' ],
            [ '122', '125', 'LISBOA' ],
            [ '127', '129', 'LISBOA' ],
            [ '131', '133', 'LISBOA' ],
            [ '135', '141', 'LISBOA' ],
            [ '144', '159', 'LISBOA' ],
            [ '162', '163', 'LISBOA' ],
            [ '165', '165', 'LISBOA' ],
            [ '167', '169', 'LISBOA' ],
            [ '173', '174', 'LISBOA' ],
            [ '176', '180', 'LISBOA' ],
            [ '182', '182', 'LISBOA' ],
            [ '184', '184', 'LISBOA' ],
            [ '186', '186', 'LISBOA' ],
            [ '188', '189', 'LISBOA' ],
            [ '192', '214', 'LISBOA' ],
            [ '216', '216', 'LISBOA' ],
            [ '218', '220', 'LISBOA' ],
            [ '252', '252', 'LISBOA' ],
            [ '300', '300', 'LISBOA' ] ],
          '1769' => [
            [ '001', '009', 'LISBOA' ],
            [ '011', '015', 'LISBOA' ],
            [ '018', '023', 'LISBOA' ] ],
          '1092' => [
            [ '001', '005', 'LISBOA' ] ],
          '1049' => [
            [ '001', '005', 'LISBOA' ],
            [ '008', '012', 'LISBOA' ],
            [ '014', '015', 'LISBOA' ],
            [ '017', '018', 'LISBOA' ],
            [ '021', '023', 'LISBOA' ],
            [ '025', '025', 'LISBOA' ],
            [ '027', '027', 'LISBOA' ],
            [ '029', '029', 'LISBOA' ],
            [ '032', '032', 'LISBOA' ],
            [ '035', '035', 'LISBOA' ],
            [ '037', '037', 'LISBOA' ],
            [ '039', '039', 'LISBOA' ],
            [ '041', '050', 'LISBOA' ],
            [ '052', '058', 'LISBOA' ],
            [ '061', '068', 'LISBOA' ],
            [ '071', '071', 'LISBOA' ],
            [ '075', '078', 'LISBOA' ] ],
          '1950' => [
            [ '001', '006', 'LISBOA' ],
            [ '008', '034', 'LISBOA' ],
            [ '037', '037', 'LISBOA' ],
            [ '039', '039', 'LISBOA' ],
            [ '041', '057', 'LISBOA' ],
            [ '059', '064', 'LISBOA' ],
            [ '066', '067', 'LISBOA' ],
            [ '069', '069', 'LISBOA' ],
            [ '071', '074', 'LISBOA' ],
            [ '079', '080', 'LISBOA' ],
            [ '082', '082', 'LISBOA' ],
            [ '087', '091', 'LISBOA' ],
            [ '094', '094', 'LISBOA' ],
            [ '096', '112', 'LISBOA' ],
            [ '116', '116', 'LISBOA' ],
            [ '119', '119', 'LISBOA' ],
            [ '122', '123', 'LISBOA' ],
            [ '126', '162', 'LISBOA' ],
            [ '164', '164', 'LISBOA' ],
            [ '166', '172', 'LISBOA' ],
            [ '174', '214', 'LISBOA' ],
            [ '216', '216', 'LISBOA' ],
            [ '219', '219', 'LISBOA' ],
            [ '223', '225', 'LISBOA' ],
            [ '229', '230', 'LISBOA' ],
            [ '233', '234', 'LISBOA' ],
            [ '237', '245', 'LISBOA' ],
            [ '247', '247', 'LISBOA' ],
            [ '249', '250', 'LISBOA' ],
            [ '253', '253', 'LISBOA' ],
            [ '256', '259', 'LISBOA' ],
            [ '261', '261', 'LISBOA' ],
            [ '263', '269', 'LISBOA' ],
            [ '271', '285', 'LISBOA' ],
            [ '287', '294', 'LISBOA' ],
            [ '297', '307', 'LISBOA' ],
            [ '317', '318', 'LISBOA' ],
            [ '321', '321', 'LISBOA' ],
            [ '323', '323', 'LISBOA' ],
            [ '326', '328', 'LISBOA' ],
            [ '330', '331', 'LISBOA' ],
            [ '333', '334', 'LISBOA' ],
            [ '336', '348', 'LISBOA' ],
            [ '351', '351', 'LISBOA' ],
            [ '353', '354', 'LISBOA' ],
            [ '356', '375', 'LISBOA' ],
            [ '377', '380', 'LISBOA' ],
            [ '406', '406', 'LISBOA' ],
            [ '408', '408', 'LISBOA' ],
            [ '413', '413', 'LISBOA' ],
            [ '420', '421', 'LISBOA' ],
            [ '426', '442', 'LISBOA' ] ],
          '1990' => [
            [ '001', '001', 'LISBOA' ],
            [ '005', '007', 'LISBOA' ],
            [ '009', '009', 'LISBOA' ],
            [ '011', '011', 'LISBOA' ],
            [ '023', '027', 'LISBOA' ],
            [ '036', '036', 'LISBOA' ],
            [ '038', '038', 'LISBOA' ],
            [ '040', '040', 'LISBOA' ],
            [ '044', '045', 'LISBOA' ],
            [ '047', '047', 'LISBOA' ],
            [ '050', '051', 'LISBOA' ],
            [ '057', '058', 'LISBOA' ],
            [ '061', '061', 'LISBOA' ],
            [ '067', '067', 'LISBOA' ],
            [ '069', '069', 'LISBOA' ],
            [ '072', '075', 'LISBOA' ],
            [ '079', '079', 'LISBOA' ],
            [ '083', '084', 'LISBOA' ],
            [ '087', '088', 'LISBOA' ],
            [ '090', '091', 'LISBOA' ],
            [ '093', '096', 'LISBOA' ],
            [ '101', '101', 'LISBOA' ],
            [ '108', '108', 'LISBOA' ],
            [ '110', '113', 'LISBOA' ],
            [ '118', '122', 'LISBOA' ],
            [ '124', '124', 'LISBOA' ],
            [ '135', '136', 'LISBOA' ],
            [ '138', '138', 'LISBOA' ],
            [ '156', '156', 'LISBOA' ],
            [ '160', '161', 'LISBOA' ],
            [ '163', '163', 'LISBOA' ],
            [ '165', '169', 'LISBOA' ],
            [ '173', '173', 'LISBOA' ],
            [ '175', '176', 'LISBOA' ],
            [ '180', '180', 'LISBOA' ],
            [ '182', '182', 'LISBOA' ],
            [ '184', '186', 'LISBOA' ],
            [ '193', '193', 'LISBOA' ],
            [ '197', '197', 'LISBOA' ],
            [ '203', '205', 'LISBOA' ],
            [ '207', '207', 'LISBOA' ],
            [ '212', '212', 'LISBOA' ],
            [ '215', '217', 'LISBOA' ],
            [ '221', '223', 'LISBOA' ],
            [ '225', '225', 'LISBOA' ],
            [ '229', '231', 'LISBOA' ],
            [ '233', '233', 'LISBOA' ],
            [ '235', '235', 'LISBOA' ],
            [ '237', '239', 'LISBOA' ],
            [ '243', '245', 'LISBOA' ],
            [ '247', '247', 'LISBOA' ],
            [ '249', '249', 'LISBOA' ],
            [ '254', '254', 'LISBOA' ],
            [ '273', '273', 'LISBOA' ],
            [ '277', '278', 'LISBOA' ],
            [ '280', '281', 'LISBOA' ],
            [ '286', '286', 'LISBOA' ],
            [ '288', '288', 'LISBOA' ],
            [ '291', '291', 'LISBOA' ],
            [ '294', '296', 'LISBOA' ],
            [ '299', '302', 'LISBOA' ],
            [ '600', '614', 'LISBOA' ],
            [ '616', '616', 'LISBOA' ] ],
          '1050' => [
            [ '001', '145', 'LISBOA' ],
            [ '148', '162', 'LISBOA' ],
            [ '164', '197', 'LISBOA' ],
            [ '199', '212', 'LISBOA' ],
            [ '214', '235', 'LISBOA' ],
            [ '237', '243', 'LISBOA' ],
            [ '250', '250', 'LISBOA' ],
            [ '908', '908', 'LISBOA' ] ],
          '1399' => [
            [ '002', '004', 'LISBOA' ],
            [ '006', '009', 'LISBOA' ],
            [ '011', '011', 'LISBOA' ],
            [ '013', '026', 'LISBOA' ],
            [ '028', '035', 'LISBOA' ],
            [ '037', '037', 'LISBOA' ],
            [ '039', '044', 'LISBOA' ],
            [ '046', '055', 'LISBOA' ] ],
          '1249' => [
            [ '001', '004', 'LISBOA' ],
            [ '006', '007', 'LISBOA' ],
            [ '010', '015', 'LISBOA' ],
            [ '017', '018', 'LISBOA' ],
            [ '020', '021', 'LISBOA' ],
            [ '023', '024', 'LISBOA' ],
            [ '027', '033', 'LISBOA' ],
            [ '035', '035', 'LISBOA' ],
            [ '037', '038', 'LISBOA' ],
            [ '040', '051', 'LISBOA' ],
            [ '054', '055', 'LISBOA' ],
            [ '057', '063', 'LISBOA' ],
            [ '065', '075', 'LISBOA' ],
            [ '077', '080', 'LISBOA' ],
            [ '082', '084', 'LISBOA' ],
            [ '086', '090', 'LISBOA' ],
            [ '092', '094', 'LISBOA' ],
            [ '097', '110', 'LISBOA' ],
            [ '113', '117', 'LISBOA' ],
            [ '119', '120', 'LISBOA' ],
            [ '122', '122', 'LISBOA' ],
            [ '125', '125', 'LISBOA' ],
            [ '130', '130', 'LISBOA' ],
            [ '140', '140', 'LISBOA' ],
            [ '150', '150', 'LISBOA' ],
            [ '155', '155', 'LISBOA' ],
            [ '165', '165', 'LISBOA' ],
            [ '180', '180', 'LISBOA' ],
            [ '185', '185', 'LISBOA' ],
            [ '190', '190', 'LISBOA' ],
            [ '195', '195', 'LISBOA' ],
            [ '200', '200', 'LISBOA' ],
            [ '205', '205', 'LISBOA' ],
            [ '210', '210', 'LISBOA' ],
            [ '220', '220', 'LISBOA' ],
            [ '225', '225', 'LISBOA' ],
            [ '230', '230', 'LISBOA' ],
            [ '235', '235', 'LISBOA' ],
            [ '240', '240', 'LISBOA' ],
            [ '245', '245', 'LISBOA' ],
            [ '250', '250', 'LISBOA' ],
            [ '255', '255', 'LISBOA' ],
            [ '260', '260', 'LISBOA' ],
            [ '265', '265', 'LISBOA' ],
            [ '270', '270', 'LISBOA' ],
            [ '280', '280', 'LISBOA' ],
            [ '286', '287', 'LISBOA' ] ],
          '1200' => [
            [ '001', '447', 'LISBOA' ],
            [ '450', '450', 'LISBOA' ],
            [ '452', '476', 'LISBOA' ],
            [ '478', '484', 'LISBOA' ],
            [ '601', '751', 'LISBOA' ],
            [ '753', '769', 'LISBOA' ],
            [ '771', '872', 'LISBOA' ],
            [ '888', '888', 'LISBOA' ] ],
          '1069' => [
            [ '001', '003', 'LISBOA' ],
            [ '005', '014', 'LISBOA' ],
            [ '016', '025', 'LISBOA' ],
            [ '027', '033', 'LISBOA' ],
            [ '035', '057', 'LISBOA' ],
            [ '059', '062', 'LISBOA' ],
            [ '065', '070', 'LISBOA' ],
            [ '072', '072', 'LISBOA' ],
            [ '074', '076', 'LISBOA' ],
            [ '079', '082', 'LISBOA' ],
            [ '084', '086', 'LISBOA' ],
            [ '089', '089', 'LISBOA' ],
            [ '092', '093', 'LISBOA' ],
            [ '095', '098', 'LISBOA' ],
            [ '102', '104', 'LISBOA' ],
            [ '106', '108', 'LISBOA' ],
            [ '110', '119', 'LISBOA' ],
            [ '121', '135', 'LISBOA' ],
            [ '138', '138', 'LISBOA' ],
            [ '140', '154', 'LISBOA' ],
            [ '157', '157', 'LISBOA' ],
            [ '160', '161', 'LISBOA' ],
            [ '163', '163', 'LISBOA' ],
            [ '165', '175', 'LISBOA' ],
            [ '178', '178', 'LISBOA' ],
            [ '180', '181', 'LISBOA' ],
            [ '184', '184', 'LISBOA' ],
            [ '186', '186', 'LISBOA' ],
            [ '188', '188', 'LISBOA' ],
            [ '190', '190', 'LISBOA' ],
            [ '192', '205', 'LISBOA' ],
            [ '208', '217', 'LISBOA' ],
            [ '300', '300', 'LISBOA' ],
            [ '304', '304', 'LISBOA' ],
            [ '306', '307', 'LISBOA' ],
            [ '400', '400', 'LISBOA' ],
            [ '402', '402', 'LISBOA' ],
            [ '413', '413', 'LISBOA' ] ],
          '1750' => [
            [ '001', '004', 'LISBOA' ],
            [ '006', '014', 'LISBOA' ],
            [ '017', '018', 'LISBOA' ],
            [ '020', '029', 'LISBOA' ],
            [ '031', '043', 'LISBOA' ],
            [ '045', '047', 'LISBOA' ],
            [ '051', '054', 'LISBOA' ],
            [ '056', '057', 'LISBOA' ],
            [ '063', '067', 'LISBOA' ],
            [ '069', '073', 'LISBOA' ],
            [ '075', '075', 'LISBOA' ],
            [ '077', '077', 'LISBOA' ],
            [ '079', '081', 'LISBOA' ],
            [ '083', '088', 'LISBOA' ],
            [ '094', '095', 'LISBOA' ],
            [ '097', '100', 'LISBOA' ],
            [ '103', '103', 'LISBOA' ],
            [ '107', '121', 'LISBOA' ],
            [ '123', '132', 'LISBOA' ],
            [ '134', '175', 'LISBOA' ],
            [ '177', '180', 'LISBOA' ],
            [ '182', '182', 'LISBOA' ],
            [ '184', '196', 'LISBOA' ],
            [ '200', '204', 'LISBOA' ],
            [ '206', '208', 'LISBOA' ],
            [ '210', '215', 'LISBOA' ],
            [ '217', '219', 'LISBOA' ],
            [ '221', '224', 'LISBOA' ],
            [ '226', '239', 'LISBOA' ],
            [ '241', '245', 'LISBOA' ],
            [ '247', '247', 'LISBOA' ],
            [ '249', '256', 'LISBOA' ],
            [ '261', '261', 'LISBOA' ],
            [ '263', '263', 'LISBOA' ],
            [ '265', '265', 'LISBOA' ],
            [ '268', '268', 'LISBOA' ],
            [ '271', '273', 'LISBOA' ],
            [ '276', '284', 'LISBOA' ],
            [ '286', '294', 'LISBOA' ],
            [ '296', '301', 'LISBOA' ],
            [ '304', '308', 'LISBOA' ],
            [ '310', '310', 'LISBOA' ],
            [ '312', '318', 'LISBOA' ],
            [ '321', '323', 'LISBOA' ],
            [ '328', '329', 'LISBOA' ],
            [ '331', '332', 'LISBOA' ],
            [ '335', '344', 'LISBOA' ],
            [ '350', '352', 'LISBOA' ],
            [ '356', '361', 'LISBOA' ],
            [ '372', '380', 'LISBOA' ],
            [ '391', '418', 'LISBOA' ],
            [ '423', '425', 'LISBOA' ],
            [ '428', '437', 'LISBOA' ],
            [ '926', '928', 'LISBOA' ] ],
          '1349' => [
            [ '001', '017', 'LISBOA' ],
            [ '019', '022', 'LISBOA' ],
            [ '024', '029', 'LISBOA' ],
            [ '032', '056', 'LISBOA' ],
            [ '058', '058', 'LISBOA' ] ],
          '1250' => [
            [ '001', '083', 'LISBOA' ],
            [ '085', '205', 'LISBOA' ],
            [ '208', '238', 'LISBOA' ],
            [ '240', '264', 'LISBOA' ],
            [ '266', '266', 'LISBOA' ],
            [ '268', '272', 'LISBOA' ],
            [ '301', '301', 'LISBOA' ] ],
          '1998' => [
            [ '010', '024', 'LISBOA' ] ],
          '1748' => [
            [ '001', '010', 'LISBOA' ] ],
          '1849' => [
            [ '001', '006', 'LISBOA' ],
            [ '008', '014', 'LISBOA' ],
            [ '017', '017', 'LISBOA' ],
            [ '019', '029', 'LISBOA' ] ],
          '1900' => [
            [ '002', '032', 'LISBOA' ],
            [ '034', '053', 'LISBOA' ],
            [ '055', '062', 'LISBOA' ],
            [ '064', '080', 'LISBOA' ],
            [ '082', '098', 'LISBOA' ],
            [ '100', '104', 'LISBOA' ],
            [ '107', '122', 'LISBOA' ],
            [ '124', '129', 'LISBOA' ],
            [ '131', '138', 'LISBOA' ],
            [ '140', '145', 'LISBOA' ],
            [ '148', '151', 'LISBOA' ],
            [ '153', '169', 'LISBOA' ],
            [ '172', '174', 'LISBOA' ],
            [ '176', '218', 'LISBOA' ],
            [ '220', '232', 'LISBOA' ],
            [ '234', '238', 'LISBOA' ],
            [ '240', '250', 'LISBOA' ],
            [ '253', '269', 'LISBOA' ],
            [ '271', '282', 'LISBOA' ],
            [ '284', '325', 'LISBOA' ],
            [ '327', '332', 'LISBOA' ],
            [ '334', '336', 'LISBOA' ],
            [ '339', '339', 'LISBOA' ],
            [ '341', '351', 'LISBOA' ],
            [ '354', '354', 'LISBOA' ],
            [ '356', '356', 'LISBOA' ],
            [ '358', '358', 'LISBOA' ],
            [ '360', '375', 'LISBOA' ],
            [ '377', '383', 'LISBOA' ],
            [ '385', '388', 'LISBOA' ],
            [ '391', '411', 'LISBOA' ],
            [ '414', '440', 'LISBOA' ],
            [ '442', '442', 'LISBOA' ],
            [ '453', '453', 'LISBOA' ],
            [ '455', '455', 'LISBOA' ],
            [ '598', '598', 'LISBOA' ],
            [ '650', '650', 'LISBOA' ],
            [ '696', '696', 'LISBOA' ],
            [ '789', '789', 'LISBOA' ],
            [ '793', '793', 'LISBOA' ],
            [ '803', '804', 'LISBOA' ],
            [ '844', '844', 'LISBOA' ],
            [ '863', '863', 'LISBOA' ],
            [ '865', '865', 'LISBOA' ],
            [ '881', '882', 'LISBOA' ],
            [ '884', '888', 'LISBOA' ],
            [ '915', '918', 'LISBOA' ] ],
          '1209' => [
            [ '500', '505', 'LISBOA' ] ],
          '1269' => [
            [ '001', '003', 'LISBOA' ],
            [ '005', '005', 'LISBOA' ],
            [ '007', '011', 'LISBOA' ],
            [ '013', '013', 'LISBOA' ],
            [ '015', '015', 'LISBOA' ],
            [ '017', '022', 'LISBOA' ],
            [ '024', '024', 'LISBOA' ],
            [ '026', '026', 'LISBOA' ],
            [ '028', '036', 'LISBOA' ],
            [ '038', '042', 'LISBOA' ],
            [ '045', '047', 'LISBOA' ],
            [ '050', '060', 'LISBOA' ],
            [ '063', '078', 'LISBOA' ],
            [ '080', '081', 'LISBOA' ],
            [ '083', '101', 'LISBOA' ],
            [ '103', '107', 'LISBOA' ],
            [ '109', '109', 'LISBOA' ],
            [ '111', '113', 'LISBOA' ],
            [ '115', '115', 'LISBOA' ],
            [ '118', '124', 'LISBOA' ],
            [ '126', '128', 'LISBOA' ],
            [ '130', '130', 'LISBOA' ],
            [ '132', '146', 'LISBOA' ],
            [ '149', '149', 'LISBOA' ],
            [ '152', '164', 'LISBOA' ],
            [ '200', '200', 'LISBOA' ],
            [ '250', '250', 'LISBOA' ],
            [ '269', '269', 'LISBOA' ] ],
          '1949' => [
            [ '002', '003', 'LISBOA' ],
            [ '010', '010', 'LISBOA' ],
            [ '017', '017', 'LISBOA' ] ],
          '1350' => [
            [ '001', '001', 'LISBOA' ],
            [ '003', '027', 'LISBOA' ],
            [ '030', '047', 'LISBOA' ],
            [ '049', '059', 'LISBOA' ],
            [ '062', '086', 'LISBOA' ],
            [ '089', '089', 'LISBOA' ],
            [ '091', '113', 'LISBOA' ],
            [ '115', '119', 'LISBOA' ],
            [ '122', '122', 'LISBOA' ],
            [ '126', '136', 'LISBOA' ],
            [ '138', '157', 'LISBOA' ],
            [ '159', '159', 'LISBOA' ],
            [ '161', '175', 'LISBOA' ],
            [ '177', '192', 'LISBOA' ],
            [ '197', '211', 'LISBOA' ],
            [ '213', '216', 'LISBOA' ],
            [ '218', '221', 'LISBOA' ],
            [ '223', '223', 'LISBOA' ],
            [ '225', '227', 'LISBOA' ],
            [ '229', '266', 'LISBOA' ],
            [ '268', '312', 'LISBOA' ],
            [ '314', '328', 'LISBOA' ],
            [ '330', '333', 'LISBOA' ],
            [ '335', '338', 'LISBOA' ],
            [ '340', '340', 'LISBOA' ],
            [ '342', '349', 'LISBOA' ],
            [ '351', '355', 'LISBOA' ],
            [ '401', '404', 'LISBOA' ],
            [ '410', '411', 'LISBOA' ] ],
          '1070' => [
            [ '001', '006', 'LISBOA' ],
            [ '008', '037', 'LISBOA' ],
            [ '039', '040', 'LISBOA' ],
            [ '042', '051', 'LISBOA' ],
            [ '057', '096', 'LISBOA' ],
            [ '098', '103', 'LISBOA' ],
            [ '110', '113', 'LISBOA' ],
            [ '116', '118', 'LISBOA' ],
            [ '120', '120', 'LISBOA' ],
            [ '123', '134', 'LISBOA' ],
            [ '136', '138', 'LISBOA' ],
            [ '140', '159', 'LISBOA' ],
            [ '161', '170', 'LISBOA' ],
            [ '172', '187', 'LISBOA' ],
            [ '189', '230', 'LISBOA' ],
            [ '232', '232', 'LISBOA' ],
            [ '234', '234', 'LISBOA' ],
            [ '237', '254', 'LISBOA' ],
            [ '257', '257', 'LISBOA' ],
            [ '264', '287', 'LISBOA' ],
            [ '289', '289', 'LISBOA' ],
            [ '292', '301', 'LISBOA' ],
            [ '306', '306', 'LISBOA' ],
            [ '310', '312', 'LISBOA' ],
            [ '321', '326', 'LISBOA' ],
            [ '350', '350', 'LISBOA' ],
            [ '361', '362', 'LISBOA' ],
            [ '371', '373', 'LISBOA' ],
            [ '500', '500', 'LISBOA' ],
            [ '907', '907', 'LISBOA' ] ],
          '1150' => [
            [ '001', '132', 'LISBOA' ],
            [ '134', '197', 'LISBOA' ],
            [ '199', '203', 'LISBOA' ],
            [ '205', '258', 'LISBOA' ],
            [ '260', '347', 'LISBOA' ] ],
          '1959' => [
            [ '001', '009', 'LISBOA' ] ],
          '1170' => [
            [ '001', '009', 'LISBOA' ],
            [ '011', '023', 'LISBOA' ],
            [ '025', '101', 'LISBOA' ],
            [ '103', '126', 'LISBOA' ],
            [ '128', '180', 'LISBOA' ],
            [ '182', '245', 'LISBOA' ],
            [ '247', '260', 'LISBOA' ],
            [ '263', '265', 'LISBOA' ],
            [ '268', '268', 'LISBOA' ],
            [ '270', '304', 'LISBOA' ],
            [ '306', '309', 'LISBOA' ],
            [ '311', '326', 'LISBOA' ],
            [ '328', '401', 'LISBOA' ],
            [ '403', '403', 'LISBOA' ] ],
          '1549' => [
            [ '001', '021', 'LISBOA' ] ],
          '1068' => [
            [ '003', '003', 'LISBOA' ] ],
          '1100' => [
            [ '001', '091', 'LISBOA' ],
            [ '093', '105', 'LISBOA' ],
            [ '107', '254', 'LISBOA' ],
            [ '256', '279', 'LISBOA' ],
            [ '281', '282', 'LISBOA' ],
            [ '284', '351', 'LISBOA' ],
            [ '353', '353', 'LISBOA' ],
            [ '355', '378', 'LISBOA' ],
            [ '380', '391', 'LISBOA' ],
            [ '393', '475', 'LISBOA' ],
            [ '477', '589', 'LISBOA' ],
            [ '591', '625', 'LISBOA' ],
            [ '651', '651', 'LISBOA' ] ],
          '1149' => [
            [ '001', '002', 'LISBOA' ],
            [ '004', '006', 'LISBOA' ],
            [ '008', '012', 'LISBOA' ],
            [ '014', '025', 'LISBOA' ],
            [ '027', '035', 'LISBOA' ],
            [ '038', '039', 'LISBOA' ],
            [ '041', '045', 'LISBOA' ],
            [ '048', '057', 'LISBOA' ],
            [ '059', '066', 'LISBOA' ],
            [ '068', '074', 'LISBOA' ],
            [ '076', '080', 'LISBOA' ],
            [ '082', '083', 'LISBOA' ],
            [ '085', '085', 'LISBOA' ],
            [ '087', '093', 'LISBOA' ] ],
          '1166' => [
            [ '001', '004', 'LISBOA' ] ],
          '1208' => [
            [ '148', '148', 'LISBOA' ] ],
          '1099' => [
            [ '007', '015', 'LISBOA' ],
            [ '017', '019', 'LISBOA' ],
            [ '021', '025', 'LISBOA' ],
            [ '029', '029', 'LISBOA' ],
            [ '031', '042', 'LISBOA' ],
            [ '044', '046', 'LISBOA' ],
            [ '048', '054', 'LISBOA' ],
            [ '056', '059', 'LISBOA' ],
            [ '061', '065', 'LISBOA' ],
            [ '067', '074', 'LISBOA' ],
            [ '076', '081', 'LISBOA' ],
            [ '085', '088', 'LISBOA' ] ],
          '1800' => [
            [ '001', '012', 'LISBOA' ],
            [ '014', '016', 'LISBOA' ],
            [ '018', '032', 'LISBOA' ],
            [ '038', '039', 'LISBOA' ],
            [ '042', '044', 'LISBOA' ],
            [ '046', '065', 'LISBOA' ],
            [ '068', '090', 'LISBOA' ],
            [ '092', '093', 'LISBOA' ],
            [ '096', '108', 'LISBOA' ],
            [ '110', '114', 'LISBOA' ],
            [ '117', '122', 'LISBOA' ],
            [ '124', '126', 'LISBOA' ],
            [ '128', '129', 'LISBOA' ],
            [ '131', '136', 'LISBOA' ],
            [ '138', '138', 'LISBOA' ],
            [ '142', '142', 'LISBOA' ],
            [ '144', '144', 'LISBOA' ],
            [ '146', '147', 'LISBOA' ],
            [ '149', '149', 'LISBOA' ],
            [ '151', '168', 'LISBOA' ],
            [ '172', '172', 'LISBOA' ],
            [ '175', '188', 'LISBOA' ],
            [ '190', '197', 'LISBOA' ],
            [ '200', '205', 'LISBOA' ],
            [ '207', '208', 'LISBOA' ],
            [ '210', '210', 'LISBOA' ],
            [ '212', '212', 'LISBOA' ],
            [ '214', '218', 'LISBOA' ],
            [ '220', '220', 'LISBOA' ],
            [ '223', '224', 'LISBOA' ],
            [ '227', '227', 'LISBOA' ],
            [ '230', '230', 'LISBOA' ],
            [ '232', '233', 'LISBOA' ],
            [ '236', '237', 'LISBOA' ],
            [ '239', '247', 'LISBOA' ],
            [ '249', '255', 'LISBOA' ],
            [ '261', '262', 'LISBOA' ],
            [ '268', '281', 'LISBOA' ],
            [ '283', '292', 'LISBOA' ],
            [ '294', '294', 'LISBOA' ],
            [ '297', '297', 'LISBOA' ],
            [ '300', '303', 'LISBOA' ],
            [ '306', '308', 'LISBOA' ],
            [ '310', '310', 'LISBOA' ],
            [ '314', '314', 'LISBOA' ],
            [ '316', '321', 'LISBOA' ],
            [ '323', '323', 'LISBOA' ],
            [ '325', '333', 'LISBOA' ],
            [ '335', '350', 'LISBOA' ],
            [ '352', '352', 'LISBOA' ],
            [ '354', '379', 'LISBOA' ],
            [ '391', '396', 'LISBOA' ],
            [ '399', '400', 'LISBOA' ],
            [ '411', '413', 'LISBOA' ],
            [ '421', '423', 'LISBOA' ],
            [ '910', '910', 'LISBOA' ],
            [ '919', '920', 'LISBOA' ],
            [ '925', '925', 'LISBOA' ],
            [ '929', '929', 'LISBOA' ] ],
          '1749' => [
            [ '012', '012', 'LISBOA' ],
            [ '014', '022', 'LISBOA' ],
            [ '024', '024', 'LISBOA' ],
            [ '026', '027', 'LISBOA' ],
            [ '029', '036', 'LISBOA' ],
            [ '038', '039', 'LISBOA' ],
            [ '041', '046', 'LISBOA' ],
            [ '049', '049', 'LISBOA' ],
            [ '051', '055', 'LISBOA' ],
            [ '057', '067', 'LISBOA' ],
            [ '069', '086', 'LISBOA' ],
            [ '088', '093', 'LISBOA' ],
            [ '096', '096', 'LISBOA' ],
            [ '098', '099', 'LISBOA' ],
            [ '101', '101', 'LISBOA' ],
            [ '103', '103', 'LISBOA' ],
            [ '106', '106', 'LISBOA' ],
            [ '108', '109', 'LISBOA' ] ],
          '1500' => [
            [ '001', '050', 'LISBOA' ],
            [ '052', '052', 'LISBOA' ],
            [ '054', '081', 'LISBOA' ],
            [ '083', '090', 'LISBOA' ],
            [ '092', '119', 'LISBOA' ],
            [ '121', '122', 'LISBOA' ],
            [ '124', '124', 'LISBOA' ],
            [ '128', '128', 'LISBOA' ],
            [ '130', '140', 'LISBOA' ],
            [ '144', '145', 'LISBOA' ],
            [ '147', '150', 'LISBOA' ],
            [ '152', '159', 'LISBOA' ],
            [ '161', '169', 'LISBOA' ],
            [ '171', '171', 'LISBOA' ],
            [ '173', '180', 'LISBOA' ],
            [ '183', '185', 'LISBOA' ],
            [ '187', '192', 'LISBOA' ],
            [ '194', '201', 'LISBOA' ],
            [ '203', '208', 'LISBOA' ],
            [ '210', '233', 'LISBOA' ],
            [ '235', '252', 'LISBOA' ],
            [ '254', '261', 'LISBOA' ],
            [ '263', '271', 'LISBOA' ],
            [ '273', '275', 'LISBOA' ],
            [ '277', '279', 'LISBOA' ],
            [ '281', '281', 'LISBOA' ],
            [ '284', '290', 'LISBOA' ],
            [ '292', '296', 'LISBOA' ],
            [ '298', '313', 'LISBOA' ],
            [ '326', '351', 'LISBOA' ],
            [ '353', '354', 'LISBOA' ],
            [ '356', '387', 'LISBOA' ],
            [ '389', '389', 'LISBOA' ],
            [ '392', '392', 'LISBOA' ],
            [ '394', '402', 'LISBOA' ],
            [ '405', '425', 'LISBOA' ],
            [ '428', '428', 'LISBOA' ],
            [ '430', '431', 'LISBOA' ],
            [ '433', '433', 'LISBOA' ],
            [ '435', '447', 'LISBOA' ],
            [ '449', '455', 'LISBOA' ],
            [ '459', '459', 'LISBOA' ],
            [ '462', '466', 'LISBOA' ],
            [ '468', '472', 'LISBOA' ],
            [ '475', '492', 'LISBOA' ],
            [ '495', '501', 'LISBOA' ],
            [ '504', '519', 'LISBOA' ],
            [ '521', '525', 'LISBOA' ],
            [ '527', '530', 'LISBOA' ],
            [ '532', '535', 'LISBOA' ],
            [ '537', '548', 'LISBOA' ],
            [ '550', '556', 'LISBOA' ],
            [ '558', '559', 'LISBOA' ],
            [ '561', '561', 'LISBOA' ],
            [ '564', '564', 'LISBOA' ],
            [ '567', '568', 'LISBOA' ],
            [ '570', '570', 'LISBOA' ],
            [ '573', '573', 'LISBOA' ],
            [ '576', '582', 'LISBOA' ],
            [ '584', '586', 'LISBOA' ],
            [ '588', '597', 'LISBOA' ],
            [ '599', '603', 'LISBOA' ],
            [ '605', '606', 'LISBOA' ],
            [ '608', '629', 'LISBOA' ],
            [ '637', '640', 'LISBOA' ],
            [ '651', '651', 'LISBOA' ],
            [ '661', '661', 'LISBOA' ],
            [ '700', '703', 'LISBOA' ],
            [ '711', '715', 'LISBOA' ],
            [ '851', '851', 'LISBOA' ],
            [ '890', '890', 'LISBOA' ],
            [ '895', '897', 'LISBOA' ],
            [ '899', '901', 'LISBOA' ],
            [ '909', '909', 'LISBOA' ] ],
          '1600' => [
            [ '001', '009', 'LISBOA' ],
            [ '011', '011', 'LISBOA' ],
            [ '013', '013', 'LISBOA' ],
            [ '015', '036', 'LISBOA' ],
            [ '038', '038', 'LISBOA' ],
            [ '042', '042', 'LISBOA' ],
            [ '048', '060', 'LISBOA' ],
            [ '062', '074', 'LISBOA' ],
            [ '076', '079', 'LISBOA' ],
            [ '081', '083', 'LISBOA' ],
            [ '087', '098', 'LISBOA' ],
            [ '100', '102', 'LISBOA' ],
            [ '104', '104', 'LISBOA' ],
            [ '114', '114', 'LISBOA' ],
            [ '118', '128', 'LISBOA' ],
            [ '130', '131', 'LISBOA' ],
            [ '134', '143', 'LISBOA' ],
            [ '145', '146', 'LISBOA' ],
            [ '148', '149', 'LISBOA' ],
            [ '151', '180', 'LISBOA' ],
            [ '182', '191', 'LISBOA' ],
            [ '193', '193', 'LISBOA' ],
            [ '196', '211', 'LISBOA' ],
            [ '214', '233', 'LISBOA' ],
            [ '235', '236', 'LISBOA' ],
            [ '238', '238', 'LISBOA' ],
            [ '251', '256', 'LISBOA' ],
            [ '258', '260', 'LISBOA' ],
            [ '296', '302', 'LISBOA' ],
            [ '305', '306', 'LISBOA' ],
            [ '371', '372', 'LISBOA' ],
            [ '401', '401', 'LISBOA' ],
            [ '404', '404', 'LISBOA' ],
            [ '406', '417', 'LISBOA' ],
            [ '419', '420', 'LISBOA' ],
            [ '423', '423', 'LISBOA' ],
            [ '425', '437', 'LISBOA' ],
            [ '439', '439', 'LISBOA' ],
            [ '441', '451', 'LISBOA' ],
            [ '453', '464', 'LISBOA' ],
            [ '466', '466', 'LISBOA' ],
            [ '468', '480', 'LISBOA' ],
            [ '482', '485', 'LISBOA' ],
            [ '487', '491', 'LISBOA' ],
            [ '493', '493', 'LISBOA' ],
            [ '495', '509', 'LISBOA' ],
            [ '525', '525', 'LISBOA' ],
            [ '528', '528', 'LISBOA' ],
            [ '530', '540', 'LISBOA' ],
            [ '542', '546', 'LISBOA' ],
            [ '548', '549', 'LISBOA' ],
            [ '552', '552', 'LISBOA' ],
            [ '555', '555', 'LISBOA' ],
            [ '559', '561', 'LISBOA' ],
            [ '563', '563', 'LISBOA' ],
            [ '565', '572', 'LISBOA' ],
            [ '575', '581', 'LISBOA' ],
            [ '583', '588', 'LISBOA' ],
            [ '590', '595', 'LISBOA' ],
            [ '597', '598', 'LISBOA' ],
            [ '600', '600', 'LISBOA' ],
            [ '605', '606', 'LISBOA' ],
            [ '608', '610', 'LISBOA' ],
            [ '612', '618', 'LISBOA' ],
            [ '620', '622', 'LISBOA' ],
            [ '624', '631', 'LISBOA' ],
            [ '634', '649', 'LISBOA' ],
            [ '651', '651', 'LISBOA' ],
            [ '653', '664', 'LISBOA' ],
            [ '667', '668', 'LISBOA' ],
            [ '670', '712', 'LISBOA' ],
            [ '716', '743', 'LISBOA' ],
            [ '746', '746', 'LISBOA' ],
            [ '748', '757', 'LISBOA' ],
            [ '759', '766', 'LISBOA' ],
            [ '768', '778', 'LISBOA' ],
            [ '780', '785', 'LISBOA' ],
            [ '791', '792', 'LISBOA' ],
            [ '794', '799', 'LISBOA' ],
            [ '801', '815', 'LISBOA' ],
            [ '817', '821', 'LISBOA' ],
            [ '824', '831', 'LISBOA' ],
            [ '864', '865', 'LISBOA' ],
            [ '867', '872', 'LISBOA' ],
            [ '874', '880', 'LISBOA' ],
            [ '891', '894', 'LISBOA' ],
            [ '896', '896', 'LISBOA' ],
            [ '898', '898', 'LISBOA' ],
            [ '901', '902', 'LISBOA' ],
            [ '922', '922', 'LISBOA' ] ],
          '1449' => [
            [ '001', '001', 'LISBOA' ],
            [ '003', '006', 'LISBOA' ],
            [ '008', '017', 'LISBOA' ],
            [ '019', '032', 'LISBOA' ],
            [ '034', '034', 'LISBOA' ] ],
          '1000' => [
            [ '001', '005', 'LISBOA' ],
            [ '007', '021', 'LISBOA' ],
            [ '023', '030', 'LISBOA' ],
            [ '032', '040', 'LISBOA' ],
            [ '042', '049', 'LISBOA' ],
            [ '051', '117', 'LISBOA' ],
            [ '119', '198', 'LISBOA' ],
            [ '201', '201', 'LISBOA' ],
            [ '203', '203', 'LISBOA' ],
            [ '205', '234', 'LISBOA' ],
            [ '236', '268', 'LISBOA' ],
            [ '270', '280', 'LISBOA' ],
            [ '282', '287', 'LISBOA' ],
            [ '289', '304', 'LISBOA' ] ],
          '1067' => [
            [ '001', '001', 'LISBOA' ] ],
          '1400' => [
            [ '001', '006', 'LISBOA' ],
            [ '008', '032', 'LISBOA' ],
            [ '034', '036', 'LISBOA' ],
            [ '038', '038', 'LISBOA' ],
            [ '041', '043', 'LISBOA' ],
            [ '045', '047', 'LISBOA' ],
            [ '049', '052', 'LISBOA' ],
            [ '054', '054', 'LISBOA' ],
            [ '056', '061', 'LISBOA' ],
            [ '066', '069', 'LISBOA' ],
            [ '071', '074', 'LISBOA' ],
            [ '077', '081', 'LISBOA' ],
            [ '083', '107', 'LISBOA' ],
            [ '109', '134', 'LISBOA' ],
            [ '136', '136', 'LISBOA' ],
            [ '138', '141', 'LISBOA' ],
            [ '143', '150', 'LISBOA' ],
            [ '152', '181', 'LISBOA' ],
            [ '183', '185', 'LISBOA' ],
            [ '187', '195', 'LISBOA' ],
            [ '197', '244', 'LISBOA' ],
            [ '246', '251', 'LISBOA' ],
            [ '253', '260', 'LISBOA' ],
            [ '262', '265', 'LISBOA' ],
            [ '267', '316', 'LISBOA' ],
            [ '318', '333', 'LISBOA' ],
            [ '335', '345', 'LISBOA' ],
            [ '347', '366', 'LISBOA' ],
            [ '370', '383', 'LISBOA' ],
            [ '391', '392', 'LISBOA' ],
            [ '399', '402', 'LISBOA' ],
            [ '405', '409', 'LISBOA' ],
            [ '422', '422', 'LISBOA' ],
            [ '424', '424', 'LISBOA' ],
            [ '902', '905', 'LISBOA' ] ],
          '1199' => [
            [ '001', '002', 'LISBOA' ],
            [ '004', '013', 'LISBOA' ],
            [ '015', '018', 'LISBOA' ] ],
          '1300' => [
            [ '001', '001', 'LISBOA' ],
            [ '003', '012', 'LISBOA' ],
            [ '014', '019', 'LISBOA' ],
            [ '021', '021', 'LISBOA' ],
            [ '023', '038', 'LISBOA' ],
            [ '040', '054', 'LISBOA' ],
            [ '057', '070', 'LISBOA' ],
            [ '073', '080', 'LISBOA' ],
            [ '082', '087', 'LISBOA' ],
            [ '091', '125', 'LISBOA' ],
            [ '128', '206', 'LISBOA' ],
            [ '210', '212', 'LISBOA' ],
            [ '215', '218', 'LISBOA' ],
            [ '221', '247', 'LISBOA' ],
            [ '249', '261', 'LISBOA' ],
            [ '263', '268', 'LISBOA' ],
            [ '270', '289', 'LISBOA' ],
            [ '291', '316', 'LISBOA' ],
            [ '318', '330', 'LISBOA' ],
            [ '332', '386', 'LISBOA' ],
            [ '388', '418', 'LISBOA' ],
            [ '420', '422', 'LISBOA' ],
            [ '425', '440', 'LISBOA' ],
            [ '442', '444', 'LISBOA' ],
            [ '446', '466', 'LISBOA' ],
            [ '468', '477', 'LISBOA' ],
            [ '480', '484', 'LISBOA' ],
            [ '486', '496', 'LISBOA' ],
            [ '498', '501', 'LISBOA' ],
            [ '504', '504', 'LISBOA' ],
            [ '507', '521', 'LISBOA' ],
            [ '524', '556', 'LISBOA' ],
            [ '559', '569', 'LISBOA' ],
            [ '571', '571', 'LISBOA' ],
            [ '573', '578', 'LISBOA' ],
            [ '580', '587', 'LISBOA' ],
            [ '589', '590', 'LISBOA' ],
            [ '596', '602', 'LISBOA' ],
            [ '604', '607', 'LISBOA' ],
            [ '609', '612', 'LISBOA' ],
            [ '621', '621', 'LISBOA' ],
            [ '651', '651', 'LISBOA' ],
            [ '653', '653', 'LISBOA' ],
            [ '661', '661', 'LISBOA' ],
            [ '906', '906', 'LISBOA' ] ],
          '1649' => [
            [ '002', '014', 'LISBOA' ],
            [ '016', '016', 'LISBOA' ],
            [ '018', '026', 'LISBOA' ],
            [ '028', '031', 'LISBOA' ],
            [ '033', '035', 'LISBOA' ],
            [ '038', '041', 'LISBOA' ],
            [ '043', '043', 'LISBOA' ] ] }
            },
      'Odivelas' => {
        'Porto da Pai' => {
          '1679' => [
            [ '013', '013', 'PONTINHA' ],
            [ '016', '017', 'PONTINHA' ] ],
          '1675' => [
            [ '197', '210', 'PONTINHA' ] ] },
        'Odivelas' => {
          '2675' => [
            [ '078', '080', 'ODIVELAS' ],
            [ '209', '240', 'ODIVELAS' ],
            [ '281', '290', 'ODIVELAS' ],
            [ '292', '292', 'ODIVELAS' ],
            [ '294', '341', 'ODIVELAS' ],
            [ '343', '349', 'ODIVELAS' ],
            [ '351', '358', 'ODIVELAS' ],
            [ '360', '442', 'ODIVELAS' ],
            [ '446', '479', 'ODIVELAS' ],
            [ '481', '484', 'ODIVELAS' ],
            [ '486', '518', 'ODIVELAS' ],
            [ '520', '599', 'ODIVELAS' ],
            [ '601', '605', 'ODIVELAS' ],
            [ '608', '628', 'ODIVELAS' ],
            [ '630', '677', 'ODIVELAS' ],
            [ '933', '933', 'ODIVELAS' ] ] },
        'Olival Basto' => {
          '2620' => [
            [ '032', '068', 'OLIVAL BASTO' ],
            [ '491', '493', 'OLIVAL BASTO' ] ] },
        'Pontinha' => {
          '1679' => [
            [ '001', '002', 'PONTINHA' ],
            [ '004', '004', 'PONTINHA' ],
            [ '020', '021', 'PONTINHA' ] ],
          '1675' => [
            [ '003', '060', 'PONTINHA' ],
            [ '069', '069', 'PONTINHA' ],
            [ '071', '071', 'PONTINHA' ],
            [ '106', '140', 'PONTINHA' ],
            [ '144', '144', 'PONTINHA' ],
            [ '147', '151', 'PONTINHA' ],
            [ '153', '153', 'PONTINHA' ],
            [ '155', '158', 'PONTINHA' ],
            [ '160', '178', 'PONTINHA' ],
            [ '180', '195', 'PONTINHA' ],
            [ '324', '350', 'PONTINHA' ] ] },
        'Vale Nogueira' => {
          '1685' => [
            [ '542', '544', 'CANEAS' ],
            [ '546', '550', 'CANEAS' ],
            [ '552', '552', 'CANEAS' ],
            [ '555', '556', 'CANEAS' ],
            [ '559', '559', 'CANEAS' ] ] },
        'Piarras' => {
          '1685' => [
            [ '513', '517', 'CANEAS' ] ] },
        'Caneas' => {
          '1685' => [
            [ '001', '002', 'CANEAS' ],
            [ '004', '008', 'CANEAS' ],
            [ '010', '035', 'CANEAS' ],
            [ '037', '062', 'CANEAS' ],
            [ '065', '071', 'CANEAS' ],
            [ '073', '076', 'CANEAS' ],
            [ '078', '087', 'CANEAS' ],
            [ '089', '097', 'CANEAS' ],
            [ '099', '104', 'CANEAS' ],
            [ '236', '238', 'CANEAS' ],
            [ '243', '243', 'CANEAS' ],
            [ '248', '248', 'CANEAS' ],
            [ '254', '254', 'CANEAS' ],
            [ '350', '350', 'CANEAS' ],
            [ '352', '356', 'CANEAS' ],
            [ '358', '371', 'CANEAS' ],
            [ '375', '378', 'CANEAS' ],
            [ '380', '386', 'CANEAS' ],
            [ '389', '392', 'CANEAS' ],
            [ '394', '408', 'CANEAS' ],
            [ '410', '410', 'CANEAS' ],
            [ '413', '416', 'CANEAS' ],
            [ '419', '420', 'CANEAS' ],
            [ '422', '435', 'CANEAS' ],
            [ '438', '438', 'CANEAS' ],
            [ '441', '450', 'CANEAS' ],
            [ '454', '458', 'CANEAS' ],
            [ '460', '472', 'CANEAS' ],
            [ '474', '486', 'CANEAS' ],
            [ '488', '490', 'CANEAS' ],
            [ '519', '520', 'CANEAS' ],
            [ '525', '525', 'CANEAS' ],
            [ '529', '531', 'CANEAS' ],
            [ '534', '539', 'CANEAS' ],
            [ '561', '595', 'CANEAS' ],
            [ '597', '612', 'CANEAS' ],
            [ '856', '856', 'CANEAS' ] ] },
        'Serra da Luz' => {
          '1675' => [
            [ '063', '067', 'PONTINHA' ],
            [ '212', '216', 'PONTINHA' ],
            [ '223', '226', 'PONTINHA' ],
            [ '229', '236', 'PONTINHA' ],
            [ '239', '321', 'PONTINHA' ],
            [ '608', '633', 'PONTINHA' ] ] },
        'Granjas Novas' => {
          '2620' => [
            [ '309', '311', 'RAMADA' ] ] },
        'Bairro Stio da Vrzea' => {
          '2620' => [
            [ '250', '254', 'RAMADA' ],
            [ '256', '289', 'RAMADA' ] ] },
        'Pvoa de Santo Adrio' => {
          '2621' => [
            [ '001', '001', 'PVOA DE SANTO ADRIO' ] ],
          '2620' => [
            [ '069', '122', 'PVOA DE SANTO ADRIO' ],
            [ '124', '142', 'PVOA DE SANTO ADRIO' ],
            [ '144', '166', 'PVOA DE SANTO ADRIO' ],
            [ '168', '183', 'PVOA DE SANTO ADRIO' ],
            [ '401', '412', 'PVOA DE SANTO ADRIO' ],
            [ '494', '495', 'PVOA DE SANTO ADRIO' ] ] },
        'Lugar D\'Alm' => {
          '1685' => [
            [ '115', '136', 'CANEAS' ],
            [ '492', '501', 'CANEAS' ],
            [ '503', '503', 'CANEAS' ],
            [ '505', '511', 'CANEAS' ] ] },
        'Bairro Cassapia' => {
          '2620' => [
            [ '001', '030', 'OLIVAL BASTO' ] ] },
        'Jardim da Radial' => {
          '2620' => [
            [ '184', '247', 'RAMADA' ],
            [ '313', '400', 'RAMADA' ],
            [ '413', '490', 'RAMADA' ] ] },
        'Bons Dias' => {
          '2620' => [
            [ '291', '291', 'RAMADA' ],
            [ '293', '307', 'RAMADA' ] ] },
        'Ramada' => {
          '2620' => [
            [ '503', '503', 'RAMADA' ] ] },
        'Bairro Casal dos Aprstimos' => {
          '2620' => [
            [ '496', '501', 'RAMADA' ] ] },
        'Bairro Vale do Forno' => {
          '2675' => [
            [ '242', '279', 'ODIVELAS' ] ] },
        'Pai' => {
          '1675' => [
            [ '074', '074', 'PONTINHA' ],
            [ '076', '078', 'PONTINHA' ],
            [ '084', '093', 'PONTINHA' ],
            [ '096', '104', 'PONTINHA' ] ] },
        'Trigache' => {
          '1685' => [
            [ '205', '233', 'FAMES' ],
            [ '744', '759', 'FAMES' ],
            [ '761', '782', 'FAMES' ] ] },
        'Senhor Roubado' => {
          '2620' => [
            [ '502', '502', 'OLIVAL BASTO' ] ] },
        'Bairro dos Carrascais' => {
          '1685' => [
            [ '107', '113', 'CANEAS' ] ] },
        'Casal da Silveira' => {
          '1685' => [
            [ '799', '820', 'FAMES' ],
            [ '822', '822', 'FAMES' ],
            [ '824', '824', 'FAMES' ],
            [ '826', '855', 'FAMES' ] ] },
        'Fames' => {
          '1689' => [
            [ '018', '019', 'FAMES' ] ],
          '1685' => [
            [ '137', '152', 'FAMES' ],
            [ '155', '190', 'FAMES' ],
            [ '192', '203', 'FAMES' ],
            [ '235', '235', 'FAMES' ],
            [ '239', '242', 'FAMES' ],
            [ '249', '253', 'FAMES' ],
            [ '637', '647', 'FAMES' ],
            [ '649', '654', 'FAMES' ],
            [ '656', '661', 'FAMES' ],
            [ '664', '665', 'FAMES' ],
            [ '669', '679', 'FAMES' ],
            [ '689', '692', 'FAMES' ],
            [ '695', '698', 'FAMES' ],
            [ '700', '702', 'FAMES' ],
            [ '704', '705', 'FAMES' ],
            [ '707', '707', 'FAMES' ],
            [ '709', '710', 'FAMES' ],
            [ '713', '714', 'FAMES' ],
            [ '718', '719', 'FAMES' ],
            [ '721', '722', 'FAMES' ],
            [ '725', '736', 'FAMES' ],
            [ '738', '738', 'FAMES' ],
            [ '740', '741', 'FAMES' ],
            [ '784', '797', 'FAMES' ],
            [ '857', '862', 'FAMES' ],
            [ '865', '873', 'FAMES' ],
            [ '877', '881', 'FAMES' ],
            [ '883', '888', 'FAMES' ],
            [ '890', '891', 'FAMES' ],
            [ '895', '902', 'FAMES' ],
            [ '904', '911', 'FAMES' ],
            [ '913', '924', 'FAMES' ] ] }
              },
      'Oeiras' => {
        'Valejas' => {
          '2730' => [
            [ '024', '112', 'BARCARENA' ] ],
          '2790' => [
            [ '183', '202', 'CARNAXIDE' ] ] },
        'Vila Fria' => {
          '2740' => [
            [ '001', '142', 'PORTO SALVO' ],
            [ '280', '291', 'PORTO SALVO' ] ] },
        'Miraflores' => {
          '1499' => [
            [ '003', '006', 'ALGS' ],
            [ '008', '011', 'ALGS' ],
            [ '013', '021', 'ALGS' ],
            [ '025', '026', 'ALGS' ],
            [ '029', '030', 'ALGS' ],
            [ '033', '033', 'ALGS' ],
            [ '035', '036', 'ALGS' ] ],
          '1495' => [
            [ '027', '047', 'ALGS' ],
            [ '135', '145', 'ALGS' ],
            [ '147', '159', 'ALGS' ],
            [ '161', '163', 'ALGS' ],
            [ '190', '191', 'ALGS' ],
            [ '208', '209', 'ALGS' ],
            [ '236', '236', 'ALGS' ] ] },
        'Leio' => {
          '2740' => [
            [ '274', '278', 'PORTO SALVO' ] ] },
        'Barcarena' => {
          '2744' => [
            [ '009', '009', 'BARCARENA' ] ] },
        'Queluz de Baixo' => {
          '2730' => [
            [ '008', '022', 'BARCARENA' ] ],
          '2734' => [
            [ '501', '504', 'BARCARENA' ] ] },
        'Cruz Quebrada' => {
          '1499' => [
            [ '002', '002', 'CRUZ QUEBRADA-DAFUNDO' ],
            [ '024', '024', 'CRUZ QUEBRADA-DAFUNDO' ],
            [ '047', '049', 'CRUZ QUEBRADA-DAFUNDO' ] ],
          '1495' => [
            [ '201', '202', 'CRUZ QUEBRADA-DAFUNDO' ],
            [ '207', '207', 'CRUZ QUEBRADA-DAFUNDO' ],
            [ '561', '561', 'CRUZ QUEBRADA-DAFUNDO' ],
            [ '681', '692', 'CRUZ QUEBRADA-DAFUNDO' ],
            [ '694', '696', 'CRUZ QUEBRADA-DAFUNDO' ],
            [ '698', '728', 'CRUZ QUEBRADA-DAFUNDO' ],
            [ '762', '768', 'CRUZ QUEBRADA-DAFUNDO' ] ] },
        'Laveiras' => {
          '2760' => [
            [ '190', '192', 'CAXIAS' ] ] },
        'Tercena' => {
          '2730' => [
            [ '001', '004', 'BARCARENA' ] ] },
        'Pao de Arcos' => {
          '2774' => [
            [ '001', '001', 'PAO DE ARCOS' ],
            [ '503', '503', 'PAO DE ARCOS' ],
            [ '506', '506', 'PAO DE ARCOS' ],
            [ '515', '529', 'PAO DE ARCOS' ] ],
          '2770' => [
            [ '001', '110', 'PAO DE ARCOS' ],
            [ '112', '183', 'PAO DE ARCOS' ],
            [ '185', '227', 'PAO DE ARCOS' ],
            [ '530', '532', 'PAO DE ARCOS' ] ] },
        'Oeiras' => {
          '2784' => [
            [ '501', '502', 'OEIRAS' ],
            [ '505', '505', 'OEIRAS' ],
            [ '508', '510', 'OEIRAS' ],
            [ '514', '514', 'OEIRAS' ],
            [ '529', '529', 'OEIRAS' ],
            [ '540', '542', 'OEIRAS' ],
            [ '544', '544', 'OEIRAS' ],
            [ '546', '546', 'OEIRAS' ],
            [ '554', '554', 'OEIRAS' ] ],
          '2780' => [
            [ '001', '039', 'OEIRAS' ],
            [ '041', '070', 'OEIRAS' ],
            [ '072', '160', 'OEIRAS' ],
            [ '162', '164', 'OEIRAS' ],
            [ '166', '262', 'OEIRAS' ],
            [ '264', '283', 'OEIRAS' ],
            [ '285', '346', 'OEIRAS' ],
            [ '348', '348', 'OEIRAS' ],
            [ '352', '360', 'OEIRAS' ],
            [ '362', '376', 'OEIRAS' ],
            [ '416', '417', 'OEIRAS' ],
            [ '494', '494', 'OEIRAS' ],
            [ '594', '594', 'OEIRAS' ],
            [ '769', '770', 'OEIRAS' ],
            [ '776', '779', 'OEIRAS' ],
            [ '875', '875', 'OEIRAS' ],
            [ '967', '968', 'OEIRAS' ] ] },
        'Carnaxide' => {
          '2794' => [
            [ '002', '087', 'CARNAXIDE' ],
            [ '100', '100', 'CARNAXIDE' ],
            [ '112', '112', 'CARNAXIDE' ] ],
          '2790' => [
            [ '005', '024', 'CARNAXIDE' ],
            [ '026', '037', 'CARNAXIDE' ],
            [ '039', '065', 'CARNAXIDE' ],
            [ '067', '181', 'CARNAXIDE' ],
            [ '204', '218', 'CARNAXIDE' ],
            [ '221', '223', 'CARNAXIDE' ],
            [ '225', '226', 'CARNAXIDE' ],
            [ '228', '237', 'CARNAXIDE' ],
            [ '459', '461', 'CARNAXIDE' ],
            [ '463', '464', 'CARNAXIDE' ],
            [ '471', '489', 'CARNAXIDE' ],
            [ '496', '502', 'CARNAXIDE' ] ] },
        'Linda a Velha' => {
          '2799' => [
            [ '505', '508', 'LINDA A VELHA' ],
            [ '530', '561', 'LINDA A VELHA' ] ],
          '2795' => [
            [ '002', '038', 'LINDA A VELHA' ],
            [ '042', '061', 'LINDA A VELHA' ],
            [ '063', '099', 'LINDA A VELHA' ],
            [ '101', '111', 'LINDA A VELHA' ],
            [ '113', '113', 'LINDA A VELHA' ],
            [ '115', '119', 'LINDA A VELHA' ],
            [ '121', '162', 'LINDA A VELHA' ],
            [ '164', '167', 'LINDA A VELHA' ],
            [ '171', '171', 'LINDA A VELHA' ],
            [ '174', '176', 'LINDA A VELHA' ],
            [ '178', '184', 'LINDA A VELHA' ],
            [ '186', '188', 'LINDA A VELHA' ],
            [ '190', '198', 'LINDA A VELHA' ],
            [ '200', '200', 'LINDA A VELHA' ],
            [ '202', '208', 'LINDA A VELHA' ],
            [ '211', '218', 'LINDA A VELHA' ],
            [ '221', '242', 'LINDA A VELHA' ],
            [ '839', '839', 'LINDA A VELHA' ] ] },
        'Queijas' => {
          '2794' => [
            [ '001', '001', 'QUEIJAS' ] ],
          '2790' => [
            [ '001', '003', 'QUEIJAS' ],
            [ '301', '370', 'QUEIJAS' ],
            [ '372', '385', 'QUEIJAS' ],
            [ '387', '422', 'QUEIJAS' ],
            [ '462', '462', 'QUEIJAS' ],
            [ '466', '469', 'QUEIJAS' ],
            [ '494', '495', 'QUEIJAS' ],
            [ '503', '503', 'QUEIJAS' ] ] },
        'Dafundo' => {
          '1499' => [
            [ '007', '007', 'CRUZ QUEBRADA-DAFUNDO' ],
            [ '038', '041', 'CRUZ QUEBRADA-DAFUNDO' ] ],
          '1495' => [
            [ '166', '166', 'CRUZ QUEBRADA-DAFUNDO' ],
            [ '734', '745', 'CRUZ QUEBRADA-DAFUNDO' ],
            [ '747', '751', 'CRUZ QUEBRADA-DAFUNDO' ],
            [ '753', '754', 'CRUZ QUEBRADA-DAFUNDO' ] ] },
        'Linda-a-Pastora' => {
          '2790' => [
            [ '425', '458', 'QUEIJAS' ],
            [ '490', '490', 'QUEIJAS' ],
            [ '492', '492', 'QUEIJAS' ] ] },
        'Porto Salvo' => {
          '2744' => [
            [ '001', '008', 'PORTO SALVO' ] ],
          '2740' => [
            [ '181', '187', 'PORTO SALVO' ],
            [ '189', '192', 'PORTO SALVO' ],
            [ '194', '203', 'PORTO SALVO' ],
            [ '205', '248', 'PORTO SALVO' ],
            [ '251', '254', 'PORTO SALVO' ],
            [ '256', '257', 'PORTO SALVO' ],
            [ '264', '272', 'PORTO SALVO' ] ] },
        'Laje' => {
          '2740' => [
            [ '144', '179', 'PORTO SALVO' ] ] },
        'Leceia' => {
          '2730' => [
            [ '114', '273', 'BARCARENA' ] ] },
        'Algs' => {
          '1499' => [
            [ '046', '046', 'ALGS' ] ],
          '1495' => [
            [ '001', '003', 'ALGS' ],
            [ '005', '007', 'ALGS' ],
            [ '009', '012', 'ALGS' ],
            [ '014', '014', 'ALGS' ],
            [ '016', '017', 'ALGS' ],
            [ '019', '025', 'ALGS' ],
            [ '050', '062', 'ALGS' ],
            [ '064', '064', 'ALGS' ],
            [ '067', '072', 'ALGS' ],
            [ '074', '079', 'ALGS' ],
            [ '081', '092', 'ALGS' ],
            [ '096', '113', 'ALGS' ],
            [ '115', '118', 'ALGS' ],
            [ '120', '133', 'ALGS' ],
            [ '165', '165', 'ALGS' ],
            [ '167', '174', 'ALGS' ],
            [ '203', '206', 'ALGS' ],
            [ '231', '234', 'ALGS' ],
            [ '769', '771', 'ALGS' ] ] },
        'Caxias' => {
          '2761' => [
            [ '001', '001', 'CAXIAS' ] ],
          '2760' => [
            [ '001', '022', 'CAXIAS' ],
            [ '024', '130', 'CAXIAS' ],
            [ '133', '133', 'CAXIAS' ],
            [ '135', '145', 'CAXIAS' ],
            [ '151', '157', 'CAXIAS' ],
            [ '184', '188', 'CAXIAS' ] ] }
            },
      'Cadaval' => {
        'Olival Fechado' => {
          '2550' => [
            [ '428', '429', 'PAINHO' ] ] },
        'Casal Coqueira' => {
          '2550' => [
            [ '302', '302', 'FIGUEIROS' ] ] },
        'Casal do Bagucho' => {
          '2550' => [
            [ '414', '423', 'PAINHO' ] ] },
        'Cadaval' => {
          '2550' => [
            [ '005', '010', 'CADAVAL' ],
            [ '127', '142', 'CADAVAL' ],
            [ '148', '160', 'CADAVAL' ],
            [ '162', '180', 'CADAVAL' ] ] },
        'Casal da Boavista' => {
          '2550' => [
            [ '520', '572', 'VERMELHA' ] ] },
        'Casal Galinhas' => {
          '2550' => [
            [ '425', '426', 'PAINHO' ] ] },
        'Casal Correias' => {
          '2550' => [
            [ '407', '412', 'PAINHO' ] ] },
        'Casal do Pinheiro' => {
          '2550' => [
            [ '482', '482', 'VERMELHA' ] ] },
        'Arrabalde' => {
          '2550' => [
            [ '060', '064', 'VILAR CDV' ] ] },
        'Casal Alto do Nascente' => {
          '2550' => [
            [ '201', '203', 'CERCAL CDV' ] ] },
        'Aboboreira' => {
          '2550' => [
            [ '401', '405', 'PAINHO' ] ] },
        'Arneiros' => {
          '2550' => [
            [ '471', '481', 'PRO MONIZ' ] ] },
        'Figueiros' => {
          '2550' => [
            [ '305', '307', 'FIGUEIROS' ] ] },
        'Dagorda' => {
          '2550' => [
            [ '500', '505', 'VERMELHA' ] ] },
        'Barreiras' => {
          '2550' => [
            [ '441', '459', 'PERAL CDV' ] ] },
        'Cercal' => {
          '2550' => [
            [ '210', '251', 'CERCAL CDV' ] ] },
        'Boia' => {
          '2550' => [
            [ '351', '351', 'LAMAS CDV' ] ] },
        'Ado Lobo' => {
          '2550' => [
            [ '101', '125', 'CADAVAL' ] ] },
        'Casais da Olaria' => {
          '2550' => [
            [ '353', '380', 'LAMAS CDV' ] ] },
        'Casal do Moinho' => {
          '2550' => [
            [ '066', '077', 'VILAR CDV' ] ] },
        'Adro' => {
          '2550' => [
            [ '011', '020', 'ALGUBER' ] ] }
             },
      'Lourinh' => {
        'Campelos' => {
          '2530' => [
            [ '397', '446', 'MIRAGAIA' ] ] },
        'Ribamar' => {
          '2530' => [
            [ '848', '849', 'RIBAMAR' ],
            [ '851', '871', 'RIBAMAR' ] ] },
        'Atalaia' => {
          '2530' => [
            [ '001', '058', 'ATALAIA LNH' ] ] },
        'Casal Arrifes' => {
          '2530' => [
            [ '563', '625', 'REGUENGO GRANDE' ] ] },
        'Lourinh' => {
          '2530' => [
            [ '116', '117', 'LOURINH' ],
            [ '119', '328', 'LOURINH' ],
            [ '850', '850', 'LOURINH' ] ] },
        'Casal Carrapial' => {
          '2530' => [
            [ '447', '512', 'MOITA DOS FERREIROS' ] ] },
        'Cabea Gorda' => {
          '2530' => [
            [ '329', '396', 'MARTELEIRA' ] ] },
        'Abelheira' => {
          '2530' => [
            [ '059', '114', 'LOURINH' ] ] },
        'Casais das Barrocas' => {
          '2530' => [
            [ '727', '767', 'SANTA BRBARA LNH' ] ] },
        'Carqueja' => {
          '2530' => [
            [ '683', '726', 'S BARTOLOMEU DOS GALEGOS' ] ] },
        'Cesaredas' => {
          '2530' => [
            [ '513', '562', 'MOLEDO LNH' ] ] },
        'Casais da Falda' => {
          '2530' => [
            [ '768', '847', 'VIMEIRO LNH' ] ] },
        'Praia de Porto Dinheiro' => {
          '2530' => [
            [ '626', '682', 'RIBAMAR' ] ] }
              },
      'Mafra' => {
        'Casal Malforno' => {
          '2665' => [
            [ '049', '055', 'ENXARA DO BISPO' ] ] },
        'Chana' => {
          '2640' => [
            [ '601', '605', 'SOBRAL DA ABELHEIRA' ] ] },
        'Venda do Pinheiro' => {
          '2665' => [
            [ '499', '595', 'VENDA DO PINHEIRO' ] ] },
        'Mafra' => {
          '2644' => [
            [ '001', '001', 'MAFRA' ] ],
          '2640' => [
            [ '389', '401', 'MAFRA' ],
            [ '403', '404', 'MAFRA' ],
            [ '409', '421', 'MAFRA' ],
            [ '437', '445', 'MAFRA' ],
            [ '450', '452', 'MAFRA' ],
            [ '454', '539', 'MAFRA' ],
            [ '773', '807', 'MAFRA' ],
            [ '809', '824', 'MAFRA' ] ] },
        'Malveira' => {
          '2665' => [
            [ '185', '256', 'MALVEIRA' ],
            [ '258', '281', 'MALVEIRA' ] ] },
        'Casal da Torre' => {
          '2640' => [
            [ '423', '428', 'MAFRA' ] ] },
        'Pobral' => {
          '2640' => [
            [ '190', '191', 'CHELEIROS' ] ] },
        'Fonte Boa da Brincosa' => {
          '2655' => [
            [ '400', '400', 'ERICEIRA' ] ] },
        'Arrifana' => {
          '2640' => [
            [ '770', '772', 'SO MIGUEL DE ALCAINA' ] ] },
        'Aboboreira' => {
          '2665' => [
            [ '001', '020', 'AZUEIRA' ] ] },
        'Seixal' => {
          '2655' => [
            [ '430', '471', 'ERICEIRA' ] ] },
        'Fontanhas' => {
          '2644' => [
            [ '002', '002', 'SO MIGUEL DE ALCAINA' ] ],
          '2640' => [
            [ '700', '750', 'SO MIGUEL DE ALCAINA' ] ] },
        'Santo Isidoro' => {
          '2640' => [
            [ '080', '098', 'SANTO ISIDORO MFR' ] ] },
        'Valbom' => {
          '2655' => [
            [ '150', '150', 'CARVOEIRA MFR' ] ] },
        'Venda do Valador' => {
          '2665' => [
            [ '290', '296', 'MALVEIRA' ],
            [ '599', '604', 'VENDA DO PINHEIRO' ] ] },
        'Sobral da Abelheira' => {
          '2640' => [
            [ '610', '645', 'SOBRAL DA ABELHEIRA' ] ] },
        'Presinheira' => {
          '2665' => [
            [ '370', '383', 'MILHARADO' ] ] },
        'Barril' => {
          '2655' => [
            [ '015', '015', 'CARVOEIRA MFR' ],
            [ '020', '020', 'CARVOEIRA MFR' ] ] },
        'Pao D\'Ilhas' => {
          '2640' => [
            [ '825', '825', 'SANTO ISIDORO MFR' ] ] },
        'Igreja Nova' => {
          '2640' => [
            [ '300', '315', 'IGREJA NOVA MFR' ],
            [ '320', '344', 'IGREJA NOVA MFR' ] ] },
        'Carvoeira' => {
          '2655' => [
            [ '030', '056', 'CARVOEIRA MFR' ] ] },
        'Lapa da Serra' => {
          '2655' => [
            [ '140', '140', 'CARVOEIRA MFR' ],
            [ '410', '410', 'ERICEIRA' ] ] },
        'Azenhas dos Tanoeiros' => {
          '2640' => [
            [ '201', '223', 'ENCARNAO' ] ] },
        'Juncal' => {
          '2640' => [
            [ '360', '377', 'IGREJA NOVA MFR' ] ] },
        'Cheleiros' => {
          '2640' => [
            [ '120', '174', 'CHELEIROS' ] ] },
        'Vila Franca do Rosrio' => {
          '2669' => [
            [ '001', '001', 'VILA FRANCA DO ROSRIO' ] ] },
        'Pico' => {
          '2665' => [
            [ '150', '152', 'GRADIL' ] ] },
        'Gonalvinhos' => {
          '2640' => [
            [ '430', '435', 'MAFRA' ] ] },
        'Encarnao' => {
          '2640' => [
            [ '230', '232', 'ENCARNAO' ] ] },
        'Outeirinho' => {
          '2655' => [
            [ '415', '415', 'ERICEIRA' ] ] },
        'Ericeira' => {
          '2655' => [
            [ '001', '004', 'ERICEIRA' ],
            [ '006', '006', 'ERICEIRA' ],
            [ '139', '139', 'ERICEIRA' ],
            [ '200', '204', 'ERICEIRA' ],
            [ '210', '219', 'ERICEIRA' ],
            [ '221', '372', 'ERICEIRA' ] ] },
        'Pvoa da Galega' => {
          '2665' => [
            [ '300', '317', 'MILHARADO' ],
            [ '320', '338', 'MILHARADO' ],
            [ '340', '365', 'MILHARADO' ] ] },
        'Ribamar' => {
          '2640' => [
            [ '001', '008', 'SANTO ISIDORO MFR' ],
            [ '010', '040', 'SANTO ISIDORO MFR' ],
            [ '050', '076', 'SANTO ISIDORO MFR' ] ] },
        'Ameiras' => {
          '2665' => [
            [ '401', '415', 'SANTO ESTEVO DAS GALS' ] ] },
        'Venda das Pulgas' => {
          '2665' => [
            [ '062', '063', 'ENXARA DO BISPO' ] ] },
        'Mato Moinho' => {
          '2640' => [
            [ '560', '567', 'MAFRA' ] ] },
        'Casalinho das Oliveiras' => {
          '2655' => [
            [ '070', '138', 'CARVOEIRA MFR' ] ] },
        'Galiza' => {
          '2640' => [
            [ '250', '266', 'ENCARNAO' ] ] },
        'Casa Nova' => {
          '2655' => [
            [ '420', '420', 'ERICEIRA' ],
            [ '425', '425', 'ERICEIRA' ] ] },
        'Alto do Seixal' => {
          '2665' => [
            [ '416', '421', 'VILA FRANCA DO ROSRIO' ] ] },
        'Carapiteira' => {
          '2665' => [
            [ '101', '134', 'GRADIL' ] ] },
        'Carvalhal' => {
          '2640' => [
            [ '111', '111', 'CHELEIROS' ] ] },
        'Baleia' => {
          '2655' => [
            [ '010', '010', 'CARVOEIRA MFR' ] ] },
        'Fonte Boa dos Nabos' => {
          '2655' => [
            [ '405', '405', 'ERICEIRA' ],
            [ '474', '478', 'ERICEIRA' ] ] },
        'Vils' => {
          '2640' => [
            [ '580', '582', 'MAFRA' ] ] },
        'Murgeira' => {
          '2640' => [
            [ '569', '578', 'MAFRA' ] ] },
        'Quint' => {
          '2665' => [
            [ '057', '060', 'ENXARA DO BISPO' ] ] }
           },
      'Sintra' => {
        'Vrzea de Sintra' => {
          '2710' => [
            [ '653', '678', 'SINTRA' ] ] },
        'Vila Verde' => {
          '2705' => [
            [ '830', '904', 'TERRUGEM SNT' ] ] },
        'Janas' => {
          '2710' => [
            [ '265', '265', 'SINTRA' ],
            [ '267', '281', 'SINTRA' ],
            [ '283', '283', 'SINTRA' ],
            [ '285', '287', 'SINTRA' ],
            [ '289', '302', 'SINTRA' ] ] },
        'Ranholas' => {
          '2710' => [
            [ '304', '325', 'SINTRA' ] ] },
        'Carrascal' => {
          '2710' => [
            [ '163', '166', 'SINTRA' ],
            [ '176', '227', 'SINTRA' ] ] },
        'Fontanelas' => {
          '2705' => [
            [ '795', '795', 'SO JOO DAS LAMPAS' ] ] },
        'Vale Mouro' => {
          '2635' => [
            [ '624', '629', 'RIO DE MOURO' ] ],
          '2735' => [
            [ '650', '660', 'AGUALVA-CACM' ] ] },
        'Praia das Mas' => {
          '2705' => [
            [ '122', '127', 'COLARES' ],
            [ '130', '169', 'COLARES' ],
            [ '295', '304', 'COLARES' ],
            [ '306', '306', 'COLARES' ],
            [ '308', '337', 'COLARES' ] ] },
        'Olival Santssimo' => {
          '2605' => [
            [ '693', '795', 'CASAL DE CAMBRA' ] ] },
        'Almoageme' => {
          '2705' => [
            [ '011', '060', 'COLARES' ],
            [ '063', '063', 'COLARES' ],
            [ '065', '065', 'COLARES' ],
            [ '067', '085', 'COLARES' ] ] },
        'Cho de Meninos' => {
          '2710' => [
            [ '003', '031', 'SINTRA' ] ] },
        'Rodzio' => {
          '2705' => [
            [ '339', '356', 'COLARES' ],
            [ '363', '363', 'COLARES' ] ] },
        'Pero Pinheiro' => {
          '2715' => [
            [ '001', '005', 'PERO PINHEIRO' ],
            [ '852', '852', 'PERO PINHEIRO' ],
            [ '857', '862', 'PERO PINHEIRO' ] ] },
        'Mucifal' => {
          '2705' => [
            [ '237', '237', 'COLARES' ],
            [ '239', '242', 'COLARES' ],
            [ '244', '266', 'COLARES' ],
            [ '282', '293', 'COLARES' ] ] },
        'Lameiras' => {
          '2715' => [
            [ '776', '778', 'TERRUGEM SNT' ] ] },
        'So Joo das Lampas' => {
          '2705' => [
            [ '793', '793', 'SO JOO DAS LAMPAS' ] ] },
        'Queluz' => {
          '2745' => [
            [ '001', '006', 'QUELUZ' ],
            [ '011', '028', 'QUELUZ' ],
            [ '030', '036', 'QUELUZ' ],
            [ '039', '044', 'QUELUZ' ],
            [ '046', '049', 'QUELUZ' ],
            [ '051', '150', 'QUELUZ' ],
            [ '152', '192', 'QUELUZ' ],
            [ '194', '219', 'QUELUZ' ],
            [ '221', '246', 'QUELUZ' ],
            [ '248', '250', 'QUELUZ' ],
            [ '252', '272', 'QUELUZ' ],
            [ '274', '305', 'QUELUZ' ],
            [ '307', '313', 'QUELUZ' ],
            [ '315', '327', 'QUELUZ' ],
            [ '330', '342', 'QUELUZ' ],
            [ '344', '349', 'QUELUZ' ],
            [ '367', '367', 'QUELUZ' ],
            [ '369', '370', 'QUELUZ' ],
            [ '372', '379', 'QUELUZ' ],
            [ '694', '700', 'QUELUZ' ],
            [ '702', '709', 'QUELUZ' ],
            [ '711', '712', 'QUELUZ' ],
            [ '714', '722', 'QUELUZ' ],
            [ '724', '725', 'QUELUZ' ],
            [ '727', '751', 'QUELUZ' ],
            [ '753', '764', 'QUELUZ' ],
            [ '766', '822', 'QUELUZ' ],
            [ '825', '839', 'QUELUZ' ],
            [ '841', '860', 'QUELUZ' ],
            [ '862', '862', 'QUELUZ' ],
            [ '864', '865', 'QUELUZ' ],
            [ '867', '898', 'QUELUZ' ] ],
          '2749' => [
            [ '505', '506', 'QUELUZ' ] ] },
        'Cruz da Moa' => {
          '2715' => [
            [ '905', '905', 'TERRUGEM SNT' ] ] },
        'Nafarros' => {
          '2710' => [
            [ '445', '447', 'SINTRA' ] ] },
        'Mem Martins' => {
          '2729' => [
            [ '008', '014', 'MEM MARTINS' ] ],
          '2725' => [
            [ '001', '087', 'MEM MARTINS' ],
            [ '168', '180', 'MEM MARTINS' ],
            [ '206', '256', 'MEM MARTINS' ],
            [ '258', '294', 'MEM MARTINS' ],
            [ '297', '456', 'MEM MARTINS' ],
            [ '458', '482', 'MEM MARTINS' ],
            [ '617', '624', 'MEM MARTINS' ] ] },
        'Belas' => {
          '2605' => [
            [ '001', '001', 'BELAS' ],
            [ '035', '035', 'BELAS' ],
            [ '037', '062', 'BELAS' ],
            [ '064', '111', 'BELAS' ],
            [ '166', '166', 'BELAS' ],
            [ '168', '171', 'BELAS' ],
            [ '173', '181', 'BELAS' ],
            [ '184', '189', 'BELAS' ],
            [ '191', '219', 'BELAS' ],
            [ '267', '272', 'BELAS' ],
            [ '288', '292', 'BELAS' ] ] },
        'Eugaria' => {
          '2705' => [
            [ '209', '235', 'COLARES' ] ] },
        'Penedo' => {
          '2705' => [
            [ '269', '278', 'COLARES' ] ] },
        'Tapada das Mercs' => {
          '2729' => [
            [ '001', '006', 'MEM MARTINS' ] ],
          '2725' => [
            [ '530', '531', 'MEM MARTINS' ],
            [ '533', '533', 'MEM MARTINS' ],
            [ '535', '543', 'MEM MARTINS' ],
            [ '545', '546', 'MEM MARTINS' ],
            [ '548', '550', 'MEM MARTINS' ],
            [ '552', '561', 'MEM MARTINS' ],
            [ '563', '571', 'MEM MARTINS' ],
            [ '573', '586', 'MEM MARTINS' ],
            [ '645', '661', 'MEM MARTINS' ] ] },
        'Morelena' => {
          '2715' => [
            [ '011', '131', 'PERO PINHEIRO' ] ] },
        'Serra das Minas' => {
          '2635' => [
            [ '001', '012', 'RIO DE MOURO' ],
            [ '405', '407', 'RIO DE MOURO' ],
            [ '409', '431', 'RIO DE MOURO' ],
            [ '433', '453', 'RIO DE MOURO' ],
            [ '635', '652', 'RIO DE MOURO' ] ] },
        'Talade' => {
          '2639' => [
            [ '502', '502', 'RIO DE MOURO' ] ],
          '2635' => [
            [ '631', '631', 'RIO DE MOURO' ] ] },
        'Arms' => {
          '2715' => [
            [ '771', '772', 'TERRUGEM SNT' ] ] },
        'Agualva-Cacm' => {
          '2739' => [
            [ '501', '502', 'AGUALVA-CACM' ] ],
          '2735' => [
            [ '047', '071', 'AGUALVA-CACM' ],
            [ '073', '133', 'AGUALVA-CACM' ],
            [ '135', '170', 'AGUALVA-CACM' ],
            [ '172', '172', 'AGUALVA-CACM' ],
            [ '174', '174', 'AGUALVA-CACM' ],
            [ '176', '181', 'AGUALVA-CACM' ],
            [ '183', '189', 'AGUALVA-CACM' ],
            [ '192', '192', 'AGUALVA-CACM' ],
            [ '204', '204', 'AGUALVA-CACM' ],
            [ '206', '259', 'AGUALVA-CACM' ],
            [ '261', '316', 'AGUALVA-CACM' ],
            [ '318', '333', 'AGUALVA-CACM' ],
            [ '335', '337', 'AGUALVA-CACM' ],
            [ '340', '382', 'AGUALVA-CACM' ],
            [ '453', '470', 'AGUALVA-CACM' ],
            [ '472', '522', 'AGUALVA-CACM' ],
            [ '548', '557', 'AGUALVA-CACM' ],
            [ '559', '599', 'AGUALVA-CACM' ],
            [ '628', '635', 'AGUALVA-CACM' ],
            [ '682', '682', 'AGUALVA-CACM' ] ] },
        'Azenhas do Mar' => {
          '2705' => [
            [ '088', '093', 'COLARES' ],
            [ '095', '120', 'COLARES' ] ] },
        'So Marcos' => {
          '2735' => [
            [ '001', '045', 'AGUALVA-CACM' ],
            [ '524', '526', 'AGUALVA-CACM' ],
            [ '528', '530', 'AGUALVA-CACM' ],
            [ '532', '545', 'AGUALVA-CACM' ],
            [ '601', '626', 'AGUALVA-CACM' ],
            [ '662', '663', 'AGUALVA-CACM' ],
            [ '668', '668', 'AGUALVA-CACM' ],
            [ '670', '680', 'AGUALVA-CACM' ] ] },
        'Fao' => {
          '2715' => [
            [ '864', '870', 'PERO PINHEIRO' ] ] },
        'Pernigem' => {
          '2705' => [
            [ '726', '791', 'SO JOO DAS LAMPAS' ] ] },
        'Negrais' => {
          '2715' => [
            [ '853', '854', 'ALMARGEM DO BISPO' ] ] },
        'Abrunheira' => {
          '2710' => [
            [ '033', '034', 'SINTRA' ],
            [ '038', '038', 'SINTRA' ],
            [ '042', '052', 'SINTRA' ],
            [ '054', '115', 'SINTRA' ] ] },
        'Idanha' => {
          '2605' => [
            [ '113', '115', 'BELAS' ],
            [ '117', '117', 'BELAS' ],
            [ '119', '122', 'BELAS' ],
            [ '222', '260', 'BELAS' ],
            [ '263', '264', 'BELAS' ],
            [ '274', '274', 'BELAS' ],
            [ '277', '286', 'BELAS' ] ] },
        'Azoia' => {
          '2705' => [
            [ '001', '001', 'COLARES' ] ] },
        'Galamares' => {
          '2710' => [
            [ '229', '229', 'SINTRA' ],
            [ '231', '262', 'SINTRA' ] ] },
        'Rio de Mouro' => {
          '2635' => [
            [ '095', '118', 'RIO DE MOURO' ],
            [ '373', '403', 'RIO DE MOURO' ] ] },
        'Serra da Silveira' => {
          '2605' => [
            [ '125', '149', 'BELAS' ],
            [ '151', '151', 'BELAS' ],
            [ '153', '164', 'BELAS' ] ] },
        'Campo Raso' => {
          '2710' => [
            [ '142', '160', 'SINTRA' ] ] },
        'Bairro do Totobola' => {
          '2710' => [
            [ '416', '418', 'SINTRA' ] ] },
        'Lourel' => {
          '2710' => [
            [ '363', '363', 'SINTRA' ],
            [ '365', '376', 'SINTRA' ],
            [ '378', '388', 'SINTRA' ],
            [ '390', '398', 'SINTRA' ],
            [ '400', '414', 'SINTRA' ] ],
          '2725' => [
            [ '204', '204', 'MEM MARTINS' ] ] },
        'Fervena' => {
          '2705' => [
            [ '906', '914', 'TERRUGEM SNT' ] ] },
        'Casal do Sequeiro' => {
          '2715' => [
            [ '774', '774', 'TERRUGEM SNT' ] ] },
        'Cabriz' => {
          '2710' => [
            [ '117', '140', 'SINTRA' ] ] },
        'Colares' => {
          '2705' => [
            [ '182', '182', 'COLARES' ],
            [ '186', '186', 'COLARES' ],
            [ '188', '207', 'COLARES' ],
            [ '361', '361', 'COLARES' ] ] },
        'Ribeira de Sintra' => {
          '2710' => [
            [ '461', '553', 'SINTRA' ] ] },
        'Pego Longo' => {
          '2745' => [
            [ '381', '385', 'QUELUZ' ] ] },
        'Capa Rota' => {
          '2714' => [
            [ '538', '550', 'SINTRA' ],
            [ '555', '555', 'SINTRA' ] ] },
        'Terrugem' => {
          '2705' => [
            [ '916', '918', 'TERRUGEM SNT' ] ] },
        'Algueiro' => {
          '2725' => [
            [ '089', '108', 'MEM MARTINS' ],
            [ '110', '162', 'MEM MARTINS' ],
            [ '504', '528', 'MEM MARTINS' ],
            [ '626', '640', 'MEM MARTINS' ],
            [ '643', '643', 'MEM MARTINS' ],
            [ '671', '689', 'MEM MARTINS' ],
            [ '691', '695', 'MEM MARTINS' ],
            [ '697', '703', 'MEM MARTINS' ] ] },
        'Anos' => {
          '2715' => [
            [ '611', '685', 'MONTELAVAR' ] ] },
        'Vale Flores' => {
          '2710' => [
            [ '680', '721', 'SINTRA' ],
            [ '723', '736', 'SINTRA' ] ] },
        'Telhal' => {
          '2725' => [
            [ '588', '615', 'MEM MARTINS' ] ] },
        'Barrunchal' => {
          '2710' => [
            [ '168', '174', 'SINTRA' ] ] },
        'Mercs' => {
          '2725' => [
            [ '484', '493', 'MEM MARTINS' ],
            [ '495', '502', 'MEM MARTINS' ] ] },
        'Linh' => {
          '2710' => [
            [ '001', '001', 'SINTRA' ],
            [ '328', '361', 'SINTRA' ] ] },
        'Casal de Cambra' => {
          '2605' => [
            [ '002', '009', 'CASAL DE CAMBRA' ],
            [ '797', '849', 'CASAL DE CAMBRA' ],
            [ '851', '863', 'CASAL DE CAMBRA' ],
            [ '867', '903', 'CASAL DE CAMBRA' ] ] },
        'Varge Mondar' => {
          '2635' => [
            [ '480', '492', 'RIO DE MOURO' ],
            [ '494', '495', 'RIO DE MOURO' ],
            [ '498', '512', 'RIO DE MOURO' ],
            [ '521', '521', 'RIO DE MOURO' ],
            [ '523', '596', 'RIO DE MOURO' ] ] },
        'Coutinho Afonso' => {
          '2725' => [
            [ '164', '164', 'MEM MARTINS' ] ] },
        'Albarraque' => {
          '2635' => [
            [ '014', '052', 'RIO DE MOURO' ],
            [ '054', '079', 'RIO DE MOURO' ],
            [ '081', '093', 'RIO DE MOURO' ],
            [ '214', '215', 'RIO DE MOURO' ],
            [ '518', '518', 'RIO DE MOURO' ] ] },
        'Cabra Figa' => {
          '2635' => [
            [ '120', '191', 'RIO DE MOURO' ] ] },
        'Ral' => {
          '2710' => [
            [ '449', '459', 'SINTRA' ] ] },
        'Sintra' => {
          '2710' => [
            [ '555', '651', 'SINTRA' ] ],
          '2714' => [
            [ '501', '536', 'SINTRA' ] ] },
        'Lopas' => {
          '2735' => [
            [ '384', '451', 'AGUALVA-CACM' ] ] },
        'Covas' => {
          '2635' => [
            [ '217', '249', 'RIO DE MOURO' ],
            [ '598', '605', 'RIO DE MOURO' ] ] },
        'Ligeiras' => {
          '2635' => [
            [ '253', '263', 'RIO DE MOURO' ] ] },
        'Montelavar' => {
          '2715' => [
            [ '687', '719', 'MONTELAVAR' ],
            [ '856', '856', 'MONTELAVAR' ],
            [ '863', '863', 'MONTELAVAR' ] ] },
        'Casal da Barota' => {
          '2605' => [
            [ '010', '032', 'BELAS' ],
            [ '651', '666', 'BELAS' ] ] },
        'Casais de Mem Martins' => {
          '2639' => [
            [ '001', '001', 'RIO DE MOURO' ] ],
          '2635' => [
            [ '193', '209', 'RIO DE MOURO' ] ] },
        'Almargem do Bispo' => {
          '2715' => [
            [ '210', '246', 'ALMARGEM DO BISPO' ],
            [ '851', '851', 'ALMARGEM DO BISPO' ] ] },
        'Gouveia' => {
          '2705' => [
            [ '409', '724', 'SO JOO DAS LAMPAS' ] ] },
        'Pexiligais' => {
          '2725' => [
            [ '182', '202', 'MEM MARTINS' ],
            [ '663', '664', 'MEM MARTINS' ],
            [ '666', '669', 'MEM MARTINS' ] ] },
        'Banzo' => {
          '2705' => [
            [ '171', '180', 'COLARES' ] ] },
        'Francos' => {
          '2635' => [
            [ '251', '251', 'RIO DE MOURO' ] ] },
        'Aruil' => {
          '2715' => [
            [ '249', '431', 'ALMARGEM DO BISPO' ] ] },
        'Rinchoa' => {
          '2635' => [
            [ '267', '270', 'RIO DE MOURO' ],
            [ '273', '276', 'RIO DE MOURO' ],
            [ '278', '279', 'RIO DE MOURO' ],
            [ '281', '283', 'RIO DE MOURO' ],
            [ '285', '290', 'RIO DE MOURO' ],
            [ '292', '293', 'RIO DE MOURO' ],
            [ '297', '318', 'RIO DE MOURO' ],
            [ '323', '323', 'RIO DE MOURO' ],
            [ '325', '371', 'RIO DE MOURO' ],
            [ '455', '457', 'RIO DE MOURO' ],
            [ '460', '463', 'RIO DE MOURO' ],
            [ '465', '477', 'RIO DE MOURO' ],
            [ '607', '622', 'RIO DE MOURO' ] ] },
        'Portela' => {
          '2710' => [
            [ '420', '443', 'SINTRA' ] ] }
            },
      'Amadora' => {
        'Amadora' => {
          '2704' => [
            [ '501', '507', 'AMADORA' ],
            [ '509', '512', 'AMADORA' ] ],
          '2700' => [
            [ '001', '001', 'AMADORA' ],
            [ '006', '009', 'AMADORA' ],
            [ '013', '014', 'AMADORA' ],
            [ '016', '020', 'AMADORA' ],
            [ '024', '026', 'AMADORA' ],
            [ '029', '036', 'AMADORA' ],
            [ '038', '040', 'AMADORA' ],
            [ '042', '043', 'AMADORA' ],
            [ '046', '046', 'AMADORA' ],
            [ '049', '049', 'AMADORA' ],
            [ '052', '053', 'AMADORA' ],
            [ '055', '055', 'AMADORA' ],
            [ '057', '061', 'AMADORA' ],
            [ '063', '065', 'AMADORA' ],
            [ '067', '073', 'AMADORA' ],
            [ '076', '076', 'AMADORA' ],
            [ '080', '080', 'AMADORA' ],
            [ '082', '087', 'AMADORA' ],
            [ '089', '091', 'AMADORA' ],
            [ '094', '095', 'AMADORA' ],
            [ '101', '102', 'AMADORA' ],
            [ '104', '107', 'AMADORA' ],
            [ '109', '109', 'AMADORA' ],
            [ '111', '111', 'AMADORA' ],
            [ '115', '115', 'AMADORA' ],
            [ '119', '123', 'AMADORA' ],
            [ '125', '130', 'AMADORA' ],
            [ '132', '135', 'AMADORA' ],
            [ '140', '140', 'AMADORA' ],
            [ '142', '146', 'AMADORA' ],
            [ '149', '149', 'AMADORA' ],
            [ '153', '153', 'AMADORA' ],
            [ '156', '156', 'AMADORA' ],
            [ '159', '163', 'AMADORA' ],
            [ '165', '166', 'AMADORA' ],
            [ '168', '173', 'AMADORA' ],
            [ '182', '183', 'AMADORA' ],
            [ '187', '195', 'AMADORA' ],
            [ '197', '197', 'AMADORA' ],
            [ '200', '201', 'AMADORA' ],
            [ '203', '203', 'AMADORA' ],
            [ '205', '213', 'AMADORA' ],
            [ '217', '221', 'AMADORA' ],
            [ '228', '233', 'AMADORA' ],
            [ '235', '235', 'AMADORA' ],
            [ '237', '238', 'AMADORA' ],
            [ '240', '241', 'AMADORA' ],
            [ '245', '245', 'AMADORA' ],
            [ '247', '248', 'AMADORA' ],
            [ '251', '259', 'AMADORA' ],
            [ '261', '266', 'AMADORA' ],
            [ '268', '269', 'AMADORA' ],
            [ '276', '280', 'AMADORA' ],
            [ '289', '290', 'AMADORA' ],
            [ '293', '301', 'AMADORA' ],
            [ '305', '349', 'AMADORA' ],
            [ '351', '353', 'AMADORA' ],
            [ '356', '356', 'AMADORA' ],
            [ '358', '360', 'AMADORA' ],
            [ '362', '365', 'AMADORA' ],
            [ '368', '370', 'AMADORA' ],
            [ '377', '377', 'AMADORA' ],
            [ '380', '384', 'AMADORA' ],
            [ '386', '387', 'AMADORA' ],
            [ '391', '391', 'AMADORA' ],
            [ '393', '402', 'AMADORA' ],
            [ '405', '405', 'AMADORA' ],
            [ '409', '414', 'AMADORA' ],
            [ '416', '426', 'AMADORA' ],
            [ '428', '428', 'AMADORA' ],
            [ '433', '434', 'AMADORA' ],
            [ '437', '441', 'AMADORA' ],
            [ '445', '453', 'AMADORA' ],
            [ '456', '456', 'AMADORA' ],
            [ '458', '460', 'AMADORA' ],
            [ '463', '467', 'AMADORA' ],
            [ '473', '473', 'AMADORA' ],
            [ '475', '478', 'AMADORA' ],
            [ '481', '482', 'AMADORA' ],
            [ '486', '489', 'AMADORA' ],
            [ '491', '491', 'AMADORA' ],
            [ '502', '503', 'AMADORA' ],
            [ '511', '517', 'AMADORA' ],
            [ '526', '532', 'AMADORA' ],
            [ '535', '535', 'AMADORA' ],
            [ '538', '540', 'AMADORA' ],
            [ '542', '545', 'AMADORA' ],
            [ '547', '548', 'AMADORA' ],
            [ '551', '553', 'AMADORA' ],
            [ '555', '556', 'AMADORA' ],
            [ '559', '559', 'AMADORA' ],
            [ '568', '573', 'AMADORA' ],
            [ '575', '592', 'AMADORA' ],
            [ '595', '596', 'AMADORA' ],
            [ '598', '598', 'AMADORA' ],
            [ '600', '600', 'AMADORA' ],
            [ '602', '602', 'AMADORA' ],
            [ '605', '606', 'AMADORA' ],
            [ '610', '613', 'AMADORA' ],
            [ '615', '620', 'AMADORA' ],
            [ '622', '622', 'AMADORA' ],
            [ '624', '633', 'AMADORA' ],
            [ '638', '638', 'AMADORA' ],
            [ '640', '642', 'AMADORA' ],
            [ '644', '654', 'AMADORA' ],
            [ '657', '657', 'AMADORA' ],
            [ '660', '660', 'AMADORA' ],
            [ '663', '663', 'AMADORA' ],
            [ '665', '666', 'AMADORA' ],
            [ '668', '672', 'AMADORA' ],
            [ '676', '679', 'AMADORA' ],
            [ '682', '684', 'AMADORA' ],
            [ '686', '686', 'AMADORA' ],
            [ '688', '692', 'AMADORA' ],
            [ '694', '700', 'AMADORA' ],
            [ '702', '703', 'AMADORA' ],
            [ '705', '710', 'AMADORA' ],
            [ '712', '713', 'AMADORA' ],
            [ '715', '715', 'AMADORA' ],
            [ '724', '725', 'AMADORA' ],
            [ '727', '727', 'AMADORA' ],
            [ '732', '733', 'AMADORA' ],
            [ '737', '739', 'AMADORA' ],
            [ '743', '744', 'AMADORA' ],
            [ '747', '748', 'AMADORA' ],
            [ '752', '753', 'AMADORA' ],
            [ '757', '757', 'AMADORA' ],
            [ '759', '760', 'AMADORA' ],
            [ '763', '763', 'AMADORA' ],
            [ '765', '765', 'AMADORA' ],
            [ '768', '772', 'AMADORA' ],
            [ '774', '774', 'AMADORA' ],
            [ '776', '779', 'AMADORA' ],
            [ '786', '789', 'AMADORA' ],
            [ '794', '794', 'AMADORA' ],
            [ '798', '801', 'AMADORA' ],
            [ '803', '806', 'AMADORA' ],
            [ '808', '810', 'AMADORA' ],
            [ '812', '812', 'AMADORA' ],
            [ '814', '816', 'AMADORA' ],
            [ '821', '822', 'AMADORA' ],
            [ '824', '824', 'AMADORA' ],
            [ '826', '827', 'AMADORA' ],
            [ '829', '829', 'AMADORA' ],
            [ '832', '834', 'AMADORA' ],
            [ '838', '838', 'AMADORA' ],
            [ '840', '841', 'AMADORA' ],
            [ '843', '846', 'AMADORA' ],
            [ '848', '848', 'AMADORA' ],
            [ '850', '850', 'AMADORA' ],
            [ '856', '856', 'AMADORA' ],
            [ '859', '859', 'AMADORA' ],
            [ '863', '863', 'AMADORA' ],
            [ '866', '866', 'AMADORA' ],
            [ '870', '871', 'AMADORA' ],
            [ '875', '875', 'AMADORA' ],
            [ '890', '890', 'AMADORA' ],
            [ '892', '895', 'AMADORA' ],
            [ '897', '897', 'AMADORA' ],
            [ '899', '899', 'AMADORA' ],
            [ '918', '918', 'AMADORA' ],
            [ '930', '935', 'AMADORA' ],
            [ '937', '938', 'AMADORA' ],
            [ '951', '951', 'AMADORA' ],
            [ '960', '960', 'AMADORA' ],
            [ '964', '964', 'AMADORA' ],
            [ '968', '969', 'AMADORA' ],
            [ '971', '971', 'AMADORA' ] ],
          '2614' => [
            [ '502', '504', 'AMADORA' ],
            [ '506', '509', 'AMADORA' ],
            [ '511', '511', 'AMADORA' ],
            [ '513', '513', 'AMADORA' ],
            [ '517', '519', 'AMADORA' ] ],
          '2720' => [
            [ '002', '005', 'AMADORA' ],
            [ '007', '008', 'AMADORA' ],
            [ '010', '012', 'AMADORA' ],
            [ '029', '030', 'AMADORA' ],
            [ '034', '038', 'AMADORA' ],
            [ '041', '042', 'AMADORA' ],
            [ '044', '050', 'AMADORA' ],
            [ '052', '054', 'AMADORA' ],
            [ '056', '057', 'AMADORA' ],
            [ '059', '060', 'AMADORA' ],
            [ '062', '067', 'AMADORA' ],
            [ '071', '071', 'AMADORA' ],
            [ '077', '078', 'AMADORA' ],
            [ '080', '083', 'AMADORA' ],
            [ '085', '090', 'AMADORA' ],
            [ '092', '094', 'AMADORA' ],
            [ '099', '099', 'AMADORA' ],
            [ '101', '101', 'AMADORA' ],
            [ '104', '104', 'AMADORA' ],
            [ '109', '110', 'AMADORA' ],
            [ '113', '113', 'AMADORA' ],
            [ '119', '119', 'AMADORA' ],
            [ '121', '122', 'AMADORA' ],
            [ '124', '127', 'AMADORA' ],
            [ '129', '132', 'AMADORA' ],
            [ '134', '136', 'AMADORA' ],
            [ '138', '138', 'AMADORA' ],
            [ '140', '141', 'AMADORA' ],
            [ '143', '143', 'AMADORA' ],
            [ '154', '155', 'AMADORA' ],
            [ '158', '159', 'AMADORA' ],
            [ '161', '180', 'AMADORA' ],
            [ '190', '198', 'AMADORA' ],
            [ '201', '206', 'AMADORA' ],
            [ '208', '208', 'AMADORA' ],
            [ '210', '215', 'AMADORA' ],
            [ '218', '220', 'AMADORA' ],
            [ '222', '223', 'AMADORA' ],
            [ '225', '226', 'AMADORA' ],
            [ '228', '228', 'AMADORA' ],
            [ '233', '233', 'AMADORA' ],
            [ '238', '240', 'AMADORA' ],
            [ '249', '257', 'AMADORA' ],
            [ '259', '259', 'AMADORA' ],
            [ '261', '264', 'AMADORA' ],
            [ '266', '276', 'AMADORA' ],
            [ '295', '297', 'AMADORA' ],
            [ '299', '299', 'AMADORA' ],
            [ '304', '304', 'AMADORA' ],
            [ '306', '306', 'AMADORA' ],
            [ '310', '310', 'AMADORA' ],
            [ '312', '313', 'AMADORA' ],
            [ '316', '316', 'AMADORA' ],
            [ '318', '322', 'AMADORA' ],
            [ '324', '326', 'AMADORA' ],
            [ '328', '330', 'AMADORA' ],
            [ '335', '335', 'AMADORA' ],
            [ '338', '338', 'AMADORA' ],
            [ '341', '342', 'AMADORA' ],
            [ '348', '348', 'AMADORA' ],
            [ '350', '357', 'AMADORA' ],
            [ '360', '360', 'AMADORA' ],
            [ '363', '373', 'AMADORA' ],
            [ '380', '380', 'AMADORA' ],
            [ '388', '390', 'AMADORA' ],
            [ '400', '403', 'AMADORA' ],
            [ '413', '416', 'AMADORA' ],
            [ '418', '419', 'AMADORA' ],
            [ '425', '430', 'AMADORA' ],
            [ '435', '441', 'AMADORA' ],
            [ '443', '447', 'AMADORA' ],
            [ '449', '451', 'AMADORA' ],
            [ '454', '454', 'AMADORA' ],
            [ '457', '458', 'AMADORA' ],
            [ '463', '465', 'AMADORA' ],
            [ '467', '469', 'AMADORA' ],
            [ '477', '477', 'AMADORA' ],
            [ '491', '491', 'AMADORA' ],
            [ '493', '494', 'AMADORA' ],
            [ '497', '497', 'AMADORA' ],
            [ '499', '500', 'AMADORA' ],
            [ '505', '505', 'AMADORA' ],
            [ '507', '509', 'AMADORA' ],
            [ '511', '512', 'AMADORA' ],
            [ '514', '516', 'AMADORA' ],
            [ '519', '521', 'AMADORA' ],
            [ '524', '528', 'AMADORA' ],
            [ '532', '533', 'AMADORA' ],
            [ '538', '539', 'AMADORA' ],
            [ '698', '700', 'AMADORA' ],
            [ '775', '775', 'AMADORA' ],
            [ '796', '797', 'AMADORA' ] ],
          '2610' => [
            [ '001', '003', 'AMADORA' ],
            [ '005', '006', 'AMADORA' ],
            [ '008', '009', 'AMADORA' ],
            [ '011', '011', 'AMADORA' ],
            [ '014', '048', 'AMADORA' ],
            [ '053', '074', 'AMADORA' ],
            [ '078', '095', 'AMADORA' ],
            [ '097', '105', 'AMADORA' ],
            [ '107', '124', 'AMADORA' ],
            [ '126', '140', 'AMADORA' ],
            [ '142', '162', 'AMADORA' ],
            [ '164', '184', 'AMADORA' ],
            [ '186', '186', 'AMADORA' ],
            [ '189', '190', 'AMADORA' ],
            [ '192', '290', 'AMADORA' ],
            [ '292', '304', 'AMADORA' ] ],
          '2724' => [
            [ '510', '510', 'AMADORA' ],
            [ '522', '522', 'AMADORA' ] ],
          '2650' => [
            [ '001', '001', 'AMADORA' ],
            [ '007', '023', 'AMADORA' ],
            [ '036', '037', 'AMADORA' ],
            [ '039', '052', 'AMADORA' ],
            [ '054', '064', 'AMADORA' ],
            [ '066', '074', 'AMADORA' ],
            [ '077', '078', 'AMADORA' ],
            [ '080', '082', 'AMADORA' ],
            [ '087', '092', 'AMADORA' ],
            [ '097', '097', 'AMADORA' ],
            [ '103', '127', 'AMADORA' ],
            [ '129', '134', 'AMADORA' ],
            [ '137', '138', 'AMADORA' ],
            [ '140', '151', 'AMADORA' ],
            [ '153', '157', 'AMADORA' ],
            [ '161', '161', 'AMADORA' ],
            [ '167', '169', 'AMADORA' ],
            [ '171', '177', 'AMADORA' ],
            [ '180', '204', 'AMADORA' ],
            [ '206', '212', 'AMADORA' ],
            [ '215', '250', 'AMADORA' ],
            [ '253', '265', 'AMADORA' ],
            [ '277', '278', 'AMADORA' ],
            [ '280', '280', 'AMADORA' ],
            [ '283', '288', 'AMADORA' ],
            [ '290', '295', 'AMADORA' ],
            [ '298', '314', 'AMADORA' ],
            [ '320', '328', 'AMADORA' ],
            [ '331', '333', 'AMADORA' ],
            [ '338', '338', 'AMADORA' ],
            [ '340', '346', 'AMADORA' ],
            [ '348', '358', 'AMADORA' ],
            [ '363', '379', 'AMADORA' ],
            [ '381', '383', 'AMADORA' ],
            [ '385', '395', 'AMADORA' ],
            [ '398', '401', 'AMADORA' ],
            [ '403', '432', 'AMADORA' ],
            [ '434', '443', 'AMADORA' ] ] }
             },
      'Vila Franca de Xira' => {
        'Cabo de Vialonga' => {
          '2625' => [
            [ '251', '258', 'VIALONGA' ],
            [ '580', '590', 'VIALONGA' ],
            [ '592', '592', 'VIALONGA' ],
            [ '595', '595', 'VIALONGA' ],
            [ '601', '602', 'VIALONGA' ],
            [ '714', '724', 'VIALONGA' ],
            [ '731', '734', 'VIALONGA' ] ] },
        'Abrulhinho' => {
          '2615' => [
            [ '621', '636', 'CALHANDRIZ' ] ] },
        'Sobralinho' => {
          '2615' => [
            [ '660', '670', 'SOBRALINHO' ],
            [ '675', '715', 'SOBRALINHO' ],
            [ '719', '722', 'SOBRALINHO' ],
            [ '724', '724', 'SOBRALINHO' ],
            [ '727', '731', 'SOBRALINHO' ],
            [ '733', '742', 'SOBRALINHO' ],
            [ '744', '756', 'SOBRALINHO' ] ] },
        'A-dos-Potes' => {
          '2615' => [
            [ '012', '035', 'ALVERCA DO RIBATEJO' ] ] },
        'Quinta de Baixo' => {
          '2600' => [
            [ '077', '137', 'VILA FRANCA DE XIRA' ] ] },
        'Granja' => {
          '2626' => [
            [ '501', '501', 'VIALONGA' ] ] },
        'Verdelha de Baixo' => {
          '2625' => [
            [ '472', '489', 'FORTE DA CASA' ] ] },
        'Arcena Pequena' => {
          '2615' => [
            [ '265', '273', 'ALVERCA DO RIBATEJO' ],
            [ '372', '381', 'ALVERCA DO RIBATEJO' ],
            [ '383', '397', 'ALVERCA DO RIBATEJO' ] ] },
        'A-de-Freire' => {
          '2600' => [
            [ '773', '779', 'SO JOO DOS MONTES' ],
            [ '781', '847', 'SO JOO DOS MONTES' ] ] },
        'Vala do Carregado' => {
          '2600' => [
            [ '733', '736', 'CASTANHEIRA DO RIBATEJO' ] ] },
        'So Joo dos Montes' => {
          '2600' => [
            [ '767', '770', 'SO JOO DOS MONTES' ] ] },
        'Boca da Lapa' => {
          '2625' => [
            [ '573', '575', 'VIALONGA' ] ] },
        'Bom Sucesso' => {
          '2615' => [
            [ '327', '344', 'ALVERCA DO RIBATEJO' ] ] },
        'Forte da Casa' => {
          '2625' => [
            [ '380', '403', 'FORTE DA CASA' ],
            [ '406', '436', 'FORTE DA CASA' ],
            [ '443', '443', 'FORTE DA CASA' ],
            [ '445', '445', 'FORTE DA CASA' ],
            [ '448', '459', 'FORTE DA CASA' ],
            [ '461', '462', 'FORTE DA CASA' ],
            [ '465', '469', 'FORTE DA CASA' ] ],
          '2626' => [
            [ '503', '503', 'FORTE DA CASA' ] ] },
        'Alpriate' => {
          '2625' => [
            [ '571', '571', 'VIALONGA' ] ] },
        'Castanheira do Ribatejo' => {
          '2600' => [
            [ '600', '629', 'CASTANHEIRA DO RIBATEJO' ],
            [ '634', '653', 'CASTANHEIRA DO RIBATEJO' ],
            [ '656', '661', 'CASTANHEIRA DO RIBATEJO' ],
            [ '665', '672', 'CASTANHEIRA DO RIBATEJO' ],
            [ '675', '681', 'CASTANHEIRA DO RIBATEJO' ],
            [ '683', '690', 'CASTANHEIRA DO RIBATEJO' ],
            [ '695', '729', 'CASTANHEIRA DO RIBATEJO' ] ],
          '2601' => [
            [ '504', '504', 'CASTANHEIRA DO RIBATEJO' ] ] },
        'Urbanizao Quinta Dona Cndida' => {
          '2625' => [
            [ '631', '644', 'VIALONGA' ] ] },
        'Vila Franca de Xira' => {
          '2600' => [
            [ '001', '002', 'VILA FRANCA DE XIRA' ],
            [ '140', '154', 'VILA FRANCA DE XIRA' ],
            [ '158', '160', 'VILA FRANCA DE XIRA' ],
            [ '162', '201', 'VILA FRANCA DE XIRA' ],
            [ '203', '203', 'VILA FRANCA DE XIRA' ],
            [ '206', '206', 'VILA FRANCA DE XIRA' ],
            [ '209', '210', 'VILA FRANCA DE XIRA' ],
            [ '212', '214', 'VILA FRANCA DE XIRA' ],
            [ '218', '235', 'VILA FRANCA DE XIRA' ],
            [ '237', '247', 'VILA FRANCA DE XIRA' ],
            [ '251', '285', 'VILA FRANCA DE XIRA' ] ],
          '2601' => [
            [ '502', '502', 'VILA FRANCA DE XIRA' ] ] },
        'Pvoa de Santa Iria' => {
          '2625' => [
            [ '002', '005', 'PVOA DE SANTA IRIA' ],
            [ '007', '047', 'PVOA DE SANTA IRIA' ],
            [ '051', '064', 'PVOA DE SANTA IRIA' ],
            [ '066', '069', 'PVOA DE SANTA IRIA' ],
            [ '071', '072', 'PVOA DE SANTA IRIA' ],
            [ '074', '074', 'PVOA DE SANTA IRIA' ],
            [ '076', '077', 'PVOA DE SANTA IRIA' ],
            [ '079', '083', 'PVOA DE SANTA IRIA' ],
            [ '085', '087', 'PVOA DE SANTA IRIA' ],
            [ '092', '092', 'PVOA DE SANTA IRIA' ],
            [ '095', '095', 'PVOA DE SANTA IRIA' ],
            [ '097', '097', 'PVOA DE SANTA IRIA' ],
            [ '099', '110', 'PVOA DE SANTA IRIA' ],
            [ '112', '118', 'PVOA DE SANTA IRIA' ],
            [ '121', '123', 'PVOA DE SANTA IRIA' ],
            [ '125', '125', 'PVOA DE SANTA IRIA' ],
            [ '130', '130', 'PVOA DE SANTA IRIA' ],
            [ '133', '147', 'PVOA DE SANTA IRIA' ],
            [ '152', '152', 'PVOA DE SANTA IRIA' ],
            [ '156', '171', 'PVOA DE SANTA IRIA' ],
            [ '173', '181', 'PVOA DE SANTA IRIA' ],
            [ '187', '188', 'PVOA DE SANTA IRIA' ],
            [ '193', '193', 'PVOA DE SANTA IRIA' ],
            [ '195', '197', 'PVOA DE SANTA IRIA' ],
            [ '200', '204', 'PVOA DE SANTA IRIA' ],
            [ '207', '207', 'PVOA DE SANTA IRIA' ],
            [ '209', '209', 'PVOA DE SANTA IRIA' ],
            [ '211', '212', 'PVOA DE SANTA IRIA' ],
            [ '214', '221', 'PVOA DE SANTA IRIA' ],
            [ '223', '231', 'PVOA DE SANTA IRIA' ],
            [ '233', '240', 'PVOA DE SANTA IRIA' ],
            [ '242', '243', 'PVOA DE SANTA IRIA' ],
            [ '245', '245', 'PVOA DE SANTA IRIA' ],
            [ '247', '250', 'PVOA DE SANTA IRIA' ],
            [ '259', '269', 'PVOA DE SANTA IRIA' ],
            [ '271', '271', 'PVOA DE SANTA IRIA' ],
            [ '490', '490', 'PVOA DE SANTA IRIA' ] ],
          '2626' => [
            [ '502', '502', 'PVOA DE SANTA IRIA' ],
            [ '504', '507', 'PVOA DE SANTA IRIA' ] ] },
        'Quintanilho' => {
          '2625' => [
            [ '627', '629', 'VIALONGA' ] ] },
        'Fonte Santa' => {
          '2625' => [
            [ '606', '625', 'VIALONGA' ] ] },
        'Vialonga' => {
          '2625' => [
            [ '084', '084', 'VIALONGA' ],
            [ '244', '244', 'VIALONGA' ],
            [ '604', '604', 'VIALONGA' ],
            [ '647', '674', 'VIALONGA' ],
            [ '676', '677', 'VIALONGA' ],
            [ '679', '687', 'VIALONGA' ],
            [ '689', '693', 'VIALONGA' ],
            [ '695', '700', 'VIALONGA' ],
            [ '702', '712', 'VIALONGA' ],
            [ '726', '729', 'VIALONGA' ] ],
          '2626' => [
            [ '508', '508', 'VIALONGA' ] ] },
        'Alverca do Ribatejo' => {
          '2615' => [
            [ '001', '005', 'ALVERCA DO RIBATEJO' ],
            [ '037', '042', 'ALVERCA DO RIBATEJO' ],
            [ '046', '070', 'ALVERCA DO RIBATEJO' ],
            [ '072', '130', 'ALVERCA DO RIBATEJO' ],
            [ '132', '134', 'ALVERCA DO RIBATEJO' ],
            [ '137', '137', 'ALVERCA DO RIBATEJO' ],
            [ '139', '161', 'ALVERCA DO RIBATEJO' ],
            [ '166', '167', 'ALVERCA DO RIBATEJO' ],
            [ '169', '170', 'ALVERCA DO RIBATEJO' ],
            [ '173', '174', 'ALVERCA DO RIBATEJO' ],
            [ '180', '180', 'ALVERCA DO RIBATEJO' ],
            [ '184', '185', 'ALVERCA DO RIBATEJO' ],
            [ '187', '189', 'ALVERCA DO RIBATEJO' ],
            [ '194', '197', 'ALVERCA DO RIBATEJO' ],
            [ '199', '199', 'ALVERCA DO RIBATEJO' ],
            [ '204', '263', 'ALVERCA DO RIBATEJO' ],
            [ '275', '276', 'ALVERCA DO RIBATEJO' ],
            [ '278', '324', 'ALVERCA DO RIBATEJO' ],
            [ '346', '369', 'ALVERCA DO RIBATEJO' ] ] },
        'Loureiro' => {
          '2615' => [
            [ '638', '648', 'CALHANDRIZ' ] ] },
        'Cachoeiras' => {
          '2600' => [
            [ '581', '585', 'CACHOEIRAS' ] ] },
        'Alhandra' => {
          '2600' => [
            [ '400', '401', 'ALHANDRA' ],
            [ '403', '406', 'ALHANDRA' ],
            [ '410', '413', 'ALHANDRA' ],
            [ '415', '464', 'ALHANDRA' ],
            [ '470', '495', 'ALHANDRA' ],
            [ '497', '500', 'ALHANDRA' ],
            [ '503', '504', 'ALHANDRA' ],
            [ '506', '543', 'ALHANDRA' ] ],
          '2601' => [
            [ '501', '501', 'ALHANDRA' ],
            [ '503', '503', 'ALHANDRA' ] ] },
        'Moucho da Pvoa' => {
          '2600' => [
            [ '004', '075', 'VILA FRANCA DE XIRA' ] ] },
        'Areias' => {
          '2615' => [
            [ '007', '010', 'ALVERCA DO RIBATEJO' ] ] }
             },
      'Cascais' => {
        'Zambujeiro' => {
          '2755' => [
            [ '311', '316', 'ALCABIDECHE' ] ] },
        'Carrascal de Manique' => {
          '2645' => [
            [ '443', '444', 'ALCABIDECHE' ],
            [ '446', '478', 'ALCABIDECHE' ] ] },
        'Janes' => {
          '2755' => [
            [ '139', '181', 'ALCABIDECHE' ] ] },
        'Madorna' => {
          '2785' => [
            [ '001', '035', 'SO DOMINGOS DE RANA' ],
            [ '412', '413', 'SO DOMINGOS DE RANA' ],
            [ '415', '418', 'SO DOMINGOS DE RANA' ],
            [ '740', '740', 'SO DOMINGOS DE RANA' ],
            [ '742', '786', 'SO DOMINGOS DE RANA' ] ],
          '2775' => [
            [ '001', '005', 'PAREDE' ],
            [ '044', '044', 'PAREDE' ],
            [ '051', '052', 'PAREDE' ],
            [ '054', '061', 'PAREDE' ],
            [ '066', '066', 'PAREDE' ],
            [ '068', '068', 'PAREDE' ] ] },
        'Trajouce' => {
          '2785' => [
            [ '643', '649', 'SO DOMINGOS DE RANA' ],
            [ '651', '653', 'SO DOMINGOS DE RANA' ],
            [ '655', '665', 'SO DOMINGOS DE RANA' ],
            [ '672', '675', 'SO DOMINGOS DE RANA' ] ] },
        'Zambujal' => {
          '2785' => [
            [ '423', '425', 'SO DOMINGOS DE RANA' ],
            [ '706', '708', 'SO DOMINGOS DE RANA' ] ] },
        'Cabeo de Mouro' => {
          '2785' => [
            [ '122', '135', 'SO DOMINGOS DE RANA' ] ] },
        'Pai do Vento' => {
          '2755' => [
            [ '297', '309', 'ALCABIDECHE' ],
            [ '522', '523', 'ALCABIDECHE' ],
            [ '534', '534', 'ALCABIDECHE' ] ] },
        'Murches' => {
          '2755' => [
            [ '255', '263', 'ALCABIDECHE' ],
            [ '266', '266', 'ALCABIDECHE' ],
            [ '271', '293', 'ALCABIDECHE' ],
            [ '331', '335', 'ALCABIDECHE' ] ] },
        'Malveira da Serra' => {
          '2755' => [
            [ '001', '017', 'ALCABIDECHE' ],
            [ '183', '253', 'ALCABIDECHE' ],
            [ '669', '669', 'ALCABIDECHE' ] ] },
        'Bicesse' => {
          '2645' => [
            [ '358', '361', 'ALCABIDECHE' ],
            [ '363', '364', 'ALCABIDECHE' ],
            [ '368', '372', 'ALCABIDECHE' ],
            [ '374', '375', 'ALCABIDECHE' ],
            [ '378', '380', 'ALCABIDECHE' ],
            [ '382', '382', 'ALCABIDECHE' ],
            [ '384', '385', 'ALCABIDECHE' ],
            [ '387', '401', 'ALCABIDECHE' ],
            [ '411', '441', 'ALCABIDECHE' ],
            [ '535', '539', 'ALCABIDECHE' ],
            [ '548', '551', 'ALCABIDECHE' ],
            [ '586', '620', 'ALCABIDECHE' ] ] },
        'Bairro de Santo Antnio' => {
          '2765' => [
            [ '131', '184', 'ESTORIL' ],
            [ '596', '597', 'ESTORIL' ],
            [ '600', '605', 'ESTORIL' ] ] },
        'Lombos' => {
          '2775' => [
            [ '400', '401', 'CARCAVELOS' ],
            [ '403', '403', 'CARCAVELOS' ],
            [ '501', '530', 'CARCAVELOS' ],
            [ '645', '650', 'CARCAVELOS' ] ] },
        'Estoril' => {
          '2765' => [
            [ '187', '211', 'ESTORIL' ],
            [ '213', '225', 'ESTORIL' ],
            [ '227', '245', 'ESTORIL' ],
            [ '571', '587', 'ESTORIL' ],
            [ '607', '616', 'ESTORIL' ] ],
          '2769' => [
            [ '510', '515', 'ESTORIL' ] ] },
        'Alto dos Campitos' => {
          '2765' => [
            [ '060', '102', 'ESTORIL' ] ] },
        'Abboda' => {
          '2785' => [
            [ '045', '046', 'SO DOMINGOS DE RANA' ],
            [ '181', '186', 'SO DOMINGOS DE RANA' ],
            [ '188', '215', 'SO DOMINGOS DE RANA' ],
            [ '668', '670', 'SO DOMINGOS DE RANA' ] ] },
        'Alvide' => {
          '2756' => [
            [ '501', '502', 'ALCABIDECHE' ] ],
          '2755' => [
            [ '048', '055', 'ALCABIDECHE' ],
            [ '057', '058', 'ALCABIDECHE' ],
            [ '060', '073', 'ALCABIDECHE' ],
            [ '076', '079', 'ALCABIDECHE' ],
            [ '082', '089', 'ALCABIDECHE' ],
            [ '114', '114', 'ALCABIDECHE' ],
            [ '390', '390', 'ALCABIDECHE' ],
            [ '546', '547', 'ALCABIDECHE' ],
            [ '574', '574', 'ALCABIDECHE' ] ] },
        'Mato Cheirinhos' => {
          '2789' => [
            [ '514', '518', 'SO DOMINGOS DE RANA' ] ],
          '2785' => [
            [ '275', '290', 'SO DOMINGOS DE RANA' ],
            [ '308', '349', 'SO DOMINGOS DE RANA' ],
            [ '493', '526', 'SO DOMINGOS DE RANA' ],
            [ '801', '806', 'SO DOMINGOS DE RANA' ] ] },
        'Sassoeiros' => {
          '2775' => [
            [ '368', '371', 'CARCAVELOS' ] ] },
        'So Pedro do Estoril' => {
          '2765' => [
            [ '560', '563', 'ESTORIL' ],
            [ '565', '566', 'ESTORIL' ] ] },
        'Outeiro de Polima' => {
          '2789' => [
            [ '503', '509', 'SO DOMINGOS DE RANA' ] ],
          '2785' => [
            [ '292', '305', 'SO DOMINGOS DE RANA' ] ] },
        'Areia' => {
          '2750' => [
            [ '052', '055', 'CASCAIS' ],
            [ '057', '061', 'CASCAIS' ],
            [ '063', '083', 'CASCAIS' ],
            [ '087', '091', 'CASCAIS' ],
            [ '094', '104', 'CASCAIS' ] ] },
        'Murtal' => {
          '2775' => [
            [ '070', '094', 'PAREDE' ],
            [ '096', '114', 'PAREDE' ],
            [ '116', '116', 'PAREDE' ],
            [ '118', '138', 'PAREDE' ],
            [ '140', '165', 'PAREDE' ] ] },
        'Abuxarda' => {
          '2755' => [
            [ '020', '020', 'ALCABIDECHE' ],
            [ '022', '046', 'ALCABIDECHE' ],
            [ '091', '110', 'ALCABIDECHE' ],
            [ '318', '318', 'ALCABIDECHE' ],
            [ '321', '326', 'ALCABIDECHE' ] ] },
        'Monte Estoril' => {
          '2765' => [
            [ '249', '342', 'ESTORIL' ],
            [ '451', '492', 'ESTORIL' ],
            [ '589', '594', 'ESTORIL' ] ] },
        'Atib' => {
          '2765' => [
            [ '107', '108', 'ESTORIL' ],
            [ '111', '115', 'ESTORIL' ] ] },
        'So Joo do Estoril' => {
          '2765' => [
            [ '494', '504', 'ESTORIL' ],
            [ '506', '509', 'ESTORIL' ],
            [ '511', '558', 'ESTORIL' ] ],
          '2769' => [
            [ '501', '504', 'ESTORIL' ] ] },
        'Parede' => {
          '2779' => [
            [ '501', '501', 'PAREDE' ],
            [ '513', '513', 'PAREDE' ] ],
          '2775' => [
            [ '007', '008', 'PAREDE' ],
            [ '047', '047', 'PAREDE' ],
            [ '167', '169', 'PAREDE' ],
            [ '171', '175', 'PAREDE' ],
            [ '178', '266', 'PAREDE' ],
            [ '268', '287', 'PAREDE' ],
            [ '289', '304', 'PAREDE' ],
            [ '308', '311', 'PAREDE' ],
            [ '363', '367', 'PAREDE' ],
            [ '372', '373', 'PAREDE' ],
            [ '377', '378', 'PAREDE' ],
            [ '390', '394', 'PAREDE' ] ] },
        'Alapraia' => {
          '2765' => [
            [ '001', '017', 'ESTORIL' ],
            [ '020', '034', 'ESTORIL' ],
            [ '036', '036', 'ESTORIL' ],
            [ '038', '041', 'ESTORIL' ],
            [ '043', '058', 'ESTORIL' ] ] },
        'Cascais' => {
          '2750' => [
            [ '001', '015', 'CASCAIS' ],
            [ '193', '194', 'CASCAIS' ],
            [ '209', '218', 'CASCAIS' ],
            [ '277', '289', 'CASCAIS' ],
            [ '291', '327', 'CASCAIS' ],
            [ '329', '362', 'CASCAIS' ],
            [ '364', '384', 'CASCAIS' ],
            [ '387', '431', 'CASCAIS' ],
            [ '433', '483', 'CASCAIS' ],
            [ '485', '485', 'CASCAIS' ],
            [ '488', '489', 'CASCAIS' ],
            [ '496', '496', 'CASCAIS' ],
            [ '498', '515', 'CASCAIS' ],
            [ '517', '544', 'CASCAIS' ],
            [ '620', '622', 'CASCAIS' ],
            [ '624', '636', 'CASCAIS' ],
            [ '640', '648', 'CASCAIS' ],
            [ '651', '676', 'CASCAIS' ],
            [ '678', '693', 'CASCAIS' ] ],
          '2754' => [
            [ '501', '505', 'CASCAIS' ],
            [ '507', '514', 'CASCAIS' ],
            [ '517', '517', 'CASCAIS' ],
            [ '519', '525', 'CASCAIS' ],
            [ '527', '535', 'CASCAIS' ] ] },
        'Penedo' => {
          '2785' => [
            [ '352', '397', 'SO DOMINGOS DE RANA' ],
            [ '400', '408', 'SO DOMINGOS DE RANA' ],
            [ '528', '545', 'SO DOMINGOS DE RANA' ] ],
          '2775' => [
            [ '010', '021', 'PAREDE' ],
            [ '031', '042', 'PAREDE' ],
            [ '306', '306', 'PAREDE' ] ] },
        'Alcoito' => {
          '2645' => [
            [ '104', '117', 'ALCABIDECHE' ],
            [ '119', '129', 'ALCABIDECHE' ],
            [ '131', '146', 'ALCABIDECHE' ],
            [ '149', '149', 'ALCABIDECHE' ],
            [ '152', '153', 'ALCABIDECHE' ],
            [ '155', '156', 'ALCABIDECHE' ],
            [ '158', '164', 'ALCABIDECHE' ],
            [ '264', '271', 'ALCABIDECHE' ],
            [ '273', '284', 'ALCABIDECHE' ],
            [ '286', '291', 'ALCABIDECHE' ],
            [ '295', '297', 'ALCABIDECHE' ],
            [ '299', '305', 'ALCABIDECHE' ],
            [ '542', '545', 'ALCABIDECHE' ] ] },
        'Carrascal de Alvide' => {
          '2755' => [
            [ '116', '137', 'ALCABIDECHE' ],
            [ '406', '406', 'ALCABIDECHE' ] ] },
        'Areias' => {
          '2765' => [
            [ '104', '105', 'ESTORIL' ],
            [ '568', '569', 'ESTORIL' ] ] },
        'Manique' => {
          '2645' => [
            [ '001', '007', 'ALCABIDECHE' ],
            [ '480', '491', 'ALCABIDECHE' ],
            [ '494', '495', 'ALCABIDECHE' ],
            [ '503', '511', 'ALCABIDECHE' ],
            [ '518', '521', 'ALCABIDECHE' ],
            [ '553', '553', 'ALCABIDECHE' ],
            [ '555', '584', 'ALCABIDECHE' ] ] },
        'Alcabideche' => {
          '2645' => [
            [ '023', '069', 'ALCABIDECHE' ],
            [ '071', '087', 'ALCABIDECHE' ],
            [ '090', '102', 'ALCABIDECHE' ],
            [ '309', '356', 'ALCABIDECHE' ] ],
          '2649' => [
            [ '501', '517', 'ALCABIDECHE' ] ] },
        'Aldeia de Juzo' => {
          '2750' => [
            [ '017', '043', 'CASCAIS' ],
            [ '046', '047', 'CASCAIS' ],
            [ '049', '049', 'CASCAIS' ],
            [ '814', '842', 'CASCAIS' ] ] },
        'Bairro da Martinha' => {
          '2765' => [
            [ '119', '125', 'ESTORIL' ],
            [ '127', '129', 'ESTORIL' ] ] },
        'Talade' => {
          '2785' => [
            [ '733', '738', 'SO DOMINGOS DE RANA' ],
            [ '796', '799', 'SO DOMINGOS DE RANA' ] ] },
        'Carcavelos' => {
          '2779' => [
            [ '510', '512', 'CARCAVELOS' ] ],
          '2775' => [
            [ '374', '376', 'CARCAVELOS' ],
            [ '549', '551', 'CARCAVELOS' ],
            [ '553', '554', 'CARCAVELOS' ],
            [ '557', '573', 'CARCAVELOS' ],
            [ '575', '643', 'CARCAVELOS' ],
            [ '710', '711', 'CARCAVELOS' ],
            [ '713', '718', 'CARCAVELOS' ],
            [ '734', '810', 'CARCAVELOS' ] ] },
        'Atrozela' => {
          '2645' => [
            [ '260', '262', 'ALCABIDECHE' ] ] },
        'Rebelva' => {
          '2775' => [
            [ '317', '361', 'PAREDE' ],
            [ '388', '388', 'PAREDE' ],
            [ '700', '708', 'CARCAVELOS' ],
            [ '720', '721', 'CARCAVELOS' ],
            [ '722', '722', 'PAREDE' ],
            [ '723', '729', 'CARCAVELOS' ],
            [ '731', '731', 'CARCAVELOS' ] ] },
        'Pau Gordo' => {
          '2645' => [
            [ '524', '533', 'ALCABIDECHE' ] ] },
        'Torre' => {
          '2750' => [
            [ '196', '200', 'CASCAIS' ],
            [ '202', '204', 'CASCAIS' ],
            [ '206', '207', 'CASCAIS' ],
            [ '606', '618', 'CASCAIS' ],
            [ '695', '710', 'CASCAIS' ],
            [ '712', '714', 'CASCAIS' ],
            [ '738', '763', 'CASCAIS' ],
            [ '768', '788', 'CASCAIS' ],
            [ '790', '812', 'CASCAIS' ] ] },
        'Tires' => {
          '2789' => [
            [ '521', '523', 'SO DOMINGOS DE RANA' ] ],
          '2785' => [
            [ '137', '164', 'SO DOMINGOS DE RANA' ],
            [ '217', '246', 'SO DOMINGOS DE RANA' ],
            [ '248', '272', 'SO DOMINGOS DE RANA' ],
            [ '603', '603', 'SO DOMINGOS DE RANA' ],
            [ '605', '614', 'SO DOMINGOS DE RANA' ],
            [ '616', '622', 'SO DOMINGOS DE RANA' ],
            [ '624', '630', 'SO DOMINGOS DE RANA' ],
            [ '632', '641', 'SO DOMINGOS DE RANA' ],
            [ '677', '703', 'SO DOMINGOS DE RANA' ],
            [ '808', '808', 'SO DOMINGOS DE RANA' ] ] },
        'Galiza' => {
          '2765' => [
            [ '117', '117', 'ESTORIL' ],
            [ '344', '354', 'ESTORIL' ],
            [ '358', '447', 'ESTORIL' ] ] },
        'Alto dos Lombos' => {
          '2775' => [
            [ '652', '659', 'CARCAVELOS' ],
            [ '661', '698', 'CARCAVELOS' ] ] },
        'Cabreiro' => {
          '2645' => [
            [ '307', '307', 'ALCABIDECHE' ],
            [ '403', '405', 'ALCABIDECHE' ],
            [ '407', '409', 'ALCABIDECHE' ] ] },
        'So Domingos de Rana' => {
          '2785' => [
            [ '581', '596', 'SO DOMINGOS DE RANA' ],
            [ '599', '600', 'SO DOMINGOS DE RANA' ] ] },
        'Cobre' => {
          '2750' => [
            [ '106', '191', 'CASCAIS' ],
            [ '491', '494', 'CASCAIS' ],
            [ '560', '577', 'CASCAIS' ],
            [ '579', '581', 'CASCAIS' ],
            [ '583', '585', 'CASCAIS' ],
            [ '587', '589', 'CASCAIS' ],
            [ '591', '592', 'CASCAIS' ],
            [ '594', '603', 'CASCAIS' ] ] },
        'Caparide' => {
          '2785' => [
            [ '420', '421', 'SO DOMINGOS DE RANA' ],
            [ '548', '567', 'SO DOMINGOS DE RANA' ],
            [ '792', '794', 'SO DOMINGOS DE RANA' ] ] },
        'Arneiro' => {
          '2785' => [
            [ '048', '048', 'SO DOMINGOS DE RANA' ],
            [ '050', '117', 'SO DOMINGOS DE RANA' ],
            [ '166', '179', 'SO DOMINGOS DE RANA' ],
            [ '811', '820', 'SO DOMINGOS DE RANA' ] ],
          '2775' => [
            [ '379', '387', 'CARCAVELOS' ],
            [ '533', '547', 'CARCAVELOS' ] ] },
        'Polima' => {
          '2785' => [
            [ '037', '039', 'SO DOMINGOS DE RANA' ] ] },
        'Charneca' => {
          '2750' => [
            [ '546', '558', 'CASCAIS' ] ] },
        'Junqueiro' => {
          '2779' => [
            [ '502', '502', 'CARCAVELOS' ] ] },
        'Rana' => {
          '2785' => [
            [ '569', '573', 'SO DOMINGOS DE RANA' ],
            [ '575', '579', 'SO DOMINGOS DE RANA' ],
            [ '716', '730', 'SO DOMINGOS DE RANA' ] ],
          '2775' => [
            [ '024', '027', 'PAREDE' ] ] },
        'Birre' => {
          '2750' => [
            [ '220', '229', 'CASCAIS' ],
            [ '231', '250', 'CASCAIS' ],
            [ '253', '265', 'CASCAIS' ],
            [ '267', '275', 'CASCAIS' ] ] },
        'Amoreira' => {
          '2645' => [
            [ '010', '021', 'ALCABIDECHE' ],
            [ '167', '218', 'ALCABIDECHE' ],
            [ '220', '225', 'ALCABIDECHE' ],
            [ '228', '228', 'ALCABIDECHE' ],
            [ '231', '231', 'ALCABIDECHE' ],
            [ '233', '254', 'ALCABIDECHE' ],
            [ '624', '625', 'ALCABIDECHE' ] ] },
        'Matarraque' => {
          '2785' => [
            [ '428', '491', 'SO DOMINGOS DE RANA' ] ] }
             },
      'Alenquer' => {
        'Casal Tereza Moa' => {
          '2580' => [
            [ '352', '395', 'ALENQUER' ] ] },
        'Atalaia' => {
          '2580' => [
            [ '401', '409', 'VENTOSA ALQ' ] ] },
        'Casalinho' => {
          '2580' => [
            [ '611', '611', 'CADAFAIS' ] ] },
        'Abrigada' => {
          '2580' => [
            [ '001', '011', 'ABRIGADA' ],
            [ '013', '018', 'ABRIGADA' ],
            [ '020', '020', 'ABRIGADA' ],
            [ '022', '022', 'ABRIGADA' ],
            [ '024', '037', 'ABRIGADA' ],
            [ '039', '059', 'ABRIGADA' ],
            [ '091', '100', 'ABRIGADA' ],
            [ '109', '109', 'ABRIGADA' ],
            [ '111', '120', 'ABRIGADA' ],
            [ '123', '130', 'ABRIGADA' ],
            [ '410', '410', 'ABRIGADA' ],
            [ '623', '623', 'ABRIGADA' ] ] },
        'Carregado' => {
          '2580' => [
            [ '464', '501', 'CARREGADO' ],
            [ '503', '512', 'CARREGADO' ],
            [ '586', '587', 'CARREGADO' ],
            [ '591', '596', 'CARREGADO' ],
            [ '598', '598', 'CARREGADO' ],
            [ '600', '606', 'CARREGADO' ],
            [ '649', '649', 'CARREGADO' ] ] },
        'Cabanas do Cho' => {
          '2580' => [
            [ '062', '064', 'ABRIGADA' ],
            [ '245', '250', 'ABRIGADA' ],
            [ '262', '282', 'ABRIGADA' ],
            [ '414', '430', 'ABRIGADA' ],
            [ '521', '521', 'ABRIGADA' ] ] },
        'Carambacha de Baixo' => {
          '2580' => [
            [ '461', '462', 'CARREGADO' ] ] },
        'Estribeiro' => {
          '2580' => [
            [ '066', '080', 'ABRIGADA' ],
            [ '443', '460', 'ABRIGADA' ] ] },
        'Casal Cabeo' => {
          '2580' => [
            [ '583', '585', 'CARNOTA' ] ] },
        'Casal do Amaral' => {
          '2580' => [
            [ '349', '350', 'ALENQUER' ] ] },
        'Corujeira' => {
          '2580' => [
            [ '612', '613', 'PEREIRO DE PALHACANA' ] ] },
        'Pousoa' => {
          '2580' => [
            [ '411', '413', 'VENTOSA ALQ' ] ] },
        'Bonvizinho' => {
          '2580' => [
            [ '252', '256', 'PEREIRO DE PALHACANA' ] ] },
        'Cabanas de Torres' => {
          '2580' => [
            [ '121', '122', 'CABANAS DE TORRES' ] ] },
        'Casais Furuana' => {
          '2580' => [
            [ '614', '616', 'VILA VERDE DOS FRANCOS' ] ] },
        'Pereiro Palhacana' => {
          '2580' => [
            [ '258', '261', 'PEREIRO DE PALHACANA' ] ] },
        'Atouguia' => {
          '2580' => [
            [ '135', '140', 'ABRIGADA' ],
            [ '143', '143', 'ABRIGADA' ],
            [ '159', '170', 'ABRIGADA' ] ] },
        'Calada' => {
          '2580' => [
            [ '144', '158', 'CARNOTA' ] ] },
        'Casais da Almnia' => {
          '2580' => [
            [ '431', '442', 'VILA VERDE DOS FRANCOS' ] ] },
        'Vale das Pedras' => {
          '2580' => [
            [ '251', '251', 'OTA' ] ] },
        'Bairro' => {
          '2580' => [
            [ '186', '190', 'ABRIGADA' ],
            [ '196', '200', 'ABRIGADA' ],
            [ '225', '229', 'ABRIGADA' ],
            [ '234', '240', 'ABRIGADA' ] ] },
        'Casal Queimada' => {
          '2580' => [
            [ '513', '513', 'ALDEIA GAVINHA' ] ] },
        'Casais Novos' => {
          '2581' => [
            [ '001', '001', 'ALENQUER' ] ] },
        'Remolha' => {
          '2580' => [
            [ '515', '516', 'ALDEIA GAVINHA' ] ] },
        'Casais da Bemposta' => {
          '2580' => [
            [ '607', '609', 'MECA' ] ] },
        'Meca' => {
          '2580' => [
            [ '599', '599', 'MECA' ] ] },
        'Paredes' => {
          '2580' => [
            [ '397', '400', 'ALENQUER' ],
            [ '560', '561', 'ALENQUER' ],
            [ '563', '581', 'ALENQUER' ],
            [ '637', '637', 'ALENQUER' ] ] },
        'Aldeia Gavinha' => {
          '2580' => [
            [ '101', '108', 'ALDEIA GAVINHA' ] ] },
        'Vale de Flores' => {
          '2580' => [
            [ '514', '514', 'CARREGADO' ] ] },
        'Aldeia Galega' => {
          '2580' => [
            [ '081', '090', 'ALDEIA GALEGA DA MERCEANA' ] ] },
        'Alenquer' => {
          '2580' => [
            [ '283', '347', 'ALENQUER' ],
            [ '517', '520', 'ALENQUER' ],
            [ '522', '538', 'ALENQUER' ],
            [ '541', '545', 'ALENQUER' ],
            [ '617', '619', 'ALENQUER' ],
            [ '648', '648', 'ALENQUER' ] ] },
        'Casal de Santo Antnio' => {
          '2580' => [
            [ '547', '550', 'ALENQUER' ] ] },
        '-dos-Carneiros' => {
          '2580' => [
            [ '551', '559', 'RIBAFRIA' ] ] },
        'Antas' => {
          '2580' => [
            [ '141', '142', 'CARNOTA' ] ] },
        'Bogarreus' => {
          '2580' => [
            [ '171', '185', 'MECA' ] ] },
        'Aldeia' => {
          '2580' => [
            [ '241', '244', 'OTA' ] ] },
        'Penafirme da Mata' => {
          '2580' => [
            [ '230', '233', 'OLHALVO' ] ] },
        'Olhalvo' => {
          '2580' => [
            [ '191', '195', 'OLHALVO' ],
            [ '201', '224', 'OLHALVO' ],
            [ '610', '610', 'OLHALVO' ],
            [ '620', '622', 'OLHALVO' ],
            [ '647', '647', 'OLHALVO' ] ] },
        'Casal da Cabreira' => {
          '2580' => [
            [ '012', '012', 'ALENQUER' ] ] },
        'Pobrais' => {
          '2580' => [
            [ '639', '639', 'PEREIRO DE PALHACANA' ] ] },
        'Tojal' => {
          '2580' => [
            [ '110', '110', 'ALDEIA GAVINHA' ] ] },
        'Quinta de So Cristvo' => {
          '2580' => [
            [ '640', '646', 'ALDEIA GALEGA DA MERCEANA' ] ] },
        'Refugidos' => {
          '2580' => [
            [ '131', '134', 'CADAFAIS' ] ] },
        'Bemposta' => {
          '2580' => [
            [ '624', '634', 'ALENQUER' ] ] }
              },
      'Torres Vedras' => {
        'Alfeiria' => {
          '2565' => [
            [ '114', '126', 'CARMES' ] ] },
        'Casal Novo' => {
          '2560' => [
            [ '191', '195', 'SO PEDRO DA CADEIRA' ] ] },
        'Ramalhal' => {
          '2565' => [
            [ '646', '659', 'RAMALHAL' ],
            [ '661', '676', 'RAMALHAL' ],
            [ '678', '695', 'RAMALHAL' ],
            [ '697', '700', 'RAMALHAL' ] ] },
        'Casalinhos de Alfaiata' => {
          '2560' => [
            [ '391', '445', 'SILVEIRA' ] ] },
        'Folgorosa' => {
          '2565' => [
            [ '171', '229', 'DOIS PORTOS' ] ] },
        'Ribaldeira' => {
          '2565' => [
            [ '243', '268', 'DOIS PORTOS' ] ] },
        'Foz' => {
          '2560' => [
            [ '178', '185', 'SO PEDRO DA CADEIRA' ] ] },
        'So Pedro da Cadeira' => {
          '2560' => [
            [ '200', '229', 'SO PEDRO DA CADEIRA' ] ] },
        'Santa Cruz' => {
          '2560' => [
            [ '450', '510', 'SILVEIRA' ] ] },
        'Silveira' => {
          '2560' => [
            [ '521', '560', 'SILVEIRA' ] ] },
        'Casal da Murteira' => {
          '2565' => [
            [ '827', '845', 'VENTOSA TVD' ] ] },
        'Aldeia Nova' => {
          '2560' => [
            [ '580', '592', 'TORRES VEDRAS' ] ] },
        'Casal do Grilo' => {
          '2565' => [
            [ '853', '853', 'FREIRIA' ] ] },
        'Torres Vedras' => {
          '2560' => [
            [ '230', '258', 'TORRES VEDRAS' ],
            [ '270', '286', 'TORRES VEDRAS' ],
            [ '288', '303', 'TORRES VEDRAS' ],
            [ '305', '375', 'TORRES VEDRAS' ],
            [ '613', '678', 'TORRES VEDRAS' ],
            [ '680', '681', 'TORRES VEDRAS' ],
            [ '683', '688', 'TORRES VEDRAS' ],
            [ '690', '692', 'TORRES VEDRAS' ],
            [ '695', '710', 'TORRES VEDRAS' ] ] },
        'Casais da Sestaria' => {
          '2565' => [
            [ '427', '445', 'MAXIAL' ] ] },
        'Ameal' => {
          '2565' => [
            [ '641', '643', 'RAMALHAL' ] ] },
        'Aldeia de Nossa Senhora da Glria' => {
          '2565' => [
            [ '131', '162', 'CARVOEIRA TVD' ] ] },
        'Maceira' => {
          '2560' => [
            [ '070', '103', 'MACEIRA TVD' ] ] },
        'Runa' => {
          '2565' => [
            [ '730', '757', 'RUNA' ] ] },
        'Pombal' => {
          '2565' => [
            [ '771', '785', 'TURCIFAL' ] ] },
        'Ribeira de Abadia' => {
          '2565' => [
            [ '358', '371', 'MATACES' ] ] },
        'Quinta Fonte da Portela' => {
          '2565' => [
            [ '711', '716', 'RUNA' ] ] },
        'Freiria' => {
          '2565' => [
            [ '279', '327', 'FREIRIA' ] ] },
        'Casal Novo da Amieira' => {
          '2565' => [
            [ '001', '063', 'CAMPELOS' ] ] },
        'Campelos' => {
          '2565' => [
            [ '065', '077', 'CAMPELOS' ],
            [ '079', '101', 'CAMPELOS' ] ] },
        'Ponte do Rol' => {
          '2560' => [
            [ '108', '168', 'PONTE DO ROL' ],
            [ '170', '177', 'PONTE DO ROL' ],
            [ '712', '712', 'PONTE DO ROL' ] ] },
        'Alto do Zzinho' => {
          '2565' => [
            [ '513', '557', 'MONTE REDONDO TVD' ] ] },
        'Mataces' => {
          '2565' => [
            [ '352', '356', 'MATACES' ],
            [ '373', '389', 'MATACES' ],
            [ '391', '405', 'MATACES' ] ] },
        'Turcifal' => {
          '2565' => [
            [ '791', '818', 'TURCIFAL' ] ] },
        'Quinta da Bela Vista' => {
          '2565' => [
            [ '586', '614', 'OUTEIRO DA CABEA' ] ] },
        'Formigal' => {
          '2560' => [
            [ '713', '716', 'SO PEDRO DA CADEIRA' ] ] },
        'Maxial' => {
          '2565' => [
            [ '453', '489', 'MAXIAL' ] ] },
        'A dos Cunhados' => {
          '2560' => [
            [ '001', '063', 'A DOS CUNHADOS' ] ] }
             },
      'Arruda dos Vinhos' => {
        'Espogeiro' => {
          '2630' => [
            [ '357', '359', 'ARRUDA DOS VINHOS' ] ] },
        'Casal Novo' => {
          '2630' => [
            [ '352', '352', 'ARRUDA DOS VINHOS' ] ] },
        'Casal do Rodo' => {
          '2630' => [
            [ '343', '349', 'ARRUDA DOS VINHOS' ] ] },
        'Arranh' => {
          '2630' => [
            [ '021', '023', 'ARRANH' ],
            [ '025', '026', 'ARRANH' ],
            [ '028', '028', 'ARRANH' ],
            [ '031', '033', 'ARRANH' ],
            [ '035', '035', 'ARRANH' ],
            [ '037', '039', 'ARRANH' ],
            [ '041', '079', 'ARRANH' ] ] },
        'Santiago dos Velhos' => {
          '2630' => [
            [ '527', '549', 'SANTIAGO DOS VELHOS' ],
            [ '560', '591', 'SANTIAGO DOS VELHOS' ] ] },
        'Casal da Capela' => {
          '2630' => [
            [ '306', '316', 'ARRUDA DOS VINHOS' ] ] },
        'Arruda dos Vinhos' => {
          '2630' => [
            [ '110', '116', 'ARRUDA DOS VINHOS' ],
            [ '118', '180', 'ARRUDA DOS VINHOS' ],
            [ '182', '189', 'ARRUDA DOS VINHOS' ],
            [ '192', '192', 'ARRUDA DOS VINHOS' ],
            [ '195', '243', 'ARRUDA DOS VINHOS' ],
            [ '245', '248', 'ARRUDA DOS VINHOS' ],
            [ '251', '259', 'ARRUDA DOS VINHOS' ],
            [ '262', '263', 'ARRUDA DOS VINHOS' ],
            [ '267', '272', 'ARRUDA DOS VINHOS' ],
            [ '274', '276', 'ARRUDA DOS VINHOS' ],
            [ '278', '279', 'ARRUDA DOS VINHOS' ],
            [ '282', '285', 'ARRUDA DOS VINHOS' ],
            [ '288', '304', 'ARRUDA DOS VINHOS' ] ] },
        'Corredouras' => {
          '2630' => [
            [ '355', '355', 'ARRUDA DOS VINHOS' ] ] },
        'Casal da Machada' => {
          '2630' => [
            [ '318', '321', 'ARRUDA DOS VINHOS' ] ] },
        'Cardosas' => {
          '2630' => [
            [ '412', '412', 'CARDOSAS' ],
            [ '414', '415', 'CARDOSAS' ],
            [ '417', '419', 'CARDOSAS' ],
            [ '421', '426', 'CARDOSAS' ],
            [ '428', '433', 'CARDOSAS' ] ] },
        'Alcobelas' => {
          '2630' => [
            [ '014', '014', 'ARRANH' ] ] },
        'Arranh de Cima' => {
          '2630' => [
            [ '091', '098', 'ARRANH' ] ] },
        'Alm' => {
          '2630' => [
            [ '016', '017', 'ARRANH' ] ] },
        'A-de-Mouro' => {
          '2630' => [
            [ '501', '525', 'SANTIAGO DOS VELHOS' ] ] },
        'Casal do No H' => {
          '2630' => [
            [ '435', '444', 'CARDOSAS' ] ] },
        'Casal das Antas' => {
          '2630' => [
            [ '323', '341', 'ARRUDA DOS VINHOS' ] ] },
        'Bogalho' => {
          '2630' => [
            [ '401', '410', 'CARDOSAS' ] ] },
        'A-do-Bao' => {
          '2630' => [
            [ '011', '012', 'ARRANH' ] ] }
                 },
      'Azambuja' => {
        'Vila Nova da Rainha' => {
          '2050' => [
            [ '501', '526', 'VILA NOVA DA RAINHA' ] ] },
        'Pvoa de Manique' => {
          '2065' => [
            [ '381', '381', 'MANIQUE DO INTENDENTE' ] ] },
        'Casal Vale da Rainha' => {
          '2065' => [
            [ '106', '107', 'ALCOENTRE' ] ] },
        'Aveiras de Cima' => {
          '2050' => [
            [ '061', '185', 'AVEIRAS DE CIMA' ] ] },
        'Bizantina' => {
          '2065' => [
            [ '100', '104', 'ALCOENTRE' ] ] },
        'Casais das Amarelas' => {
          '2050' => [
            [ '190', '199', 'AVEIRAS DE CIMA' ] ] },
        'Vale Camarinha' => {
          '2065' => [
            [ '280', '280', 'ALCOENTRE' ] ] },
        'Vale do Paraiso' => {
          '2050' => [
            [ '401', '422', 'VALE DO PARASO' ],
            [ '424', '446', 'VALE DO PARASO' ] ] },
        'Maussa' => {
          '2065' => [
            [ '601', '620', 'MAUSSA' ],
            [ '622', '637', 'MAUSSA' ] ] },
        'Carrascal' => {
          '2065' => [
            [ '551', '557', 'VILA NOVA DE SO PEDRO' ] ] },
        'Manique do Intendente' => {
          '2065' => [
            [ '315', '345', 'MANIQUE DO INTENDENTE' ],
            [ '347', '364', 'MANIQUE DO INTENDENTE' ] ] },
        'Arrifana' => {
          '2065' => [
            [ '311', '312', 'MANIQUE DO INTENDENTE' ] ] },
        'Vale de guas' => {
          '2065' => [
            [ '282', '285', 'ALCOENTRE' ] ] },
        'Alcoentre' => {
          '2065' => [
            [ '009', '050', 'ALCOENTRE' ] ] },
        'Quebradas' => {
          '2065' => [
            [ '110', '153', 'ALCOENTRE' ] ] },
        'Aveiras de Baixo' => {
          '2050' => [
            [ '011', '041', 'AVEIRAS DE BAIXO' ] ] },
        'Azambuja' => {
          '2050' => [
            [ '001', '005', 'AZAMBUJA' ],
            [ '186', '186', 'AZAMBUJA' ],
            [ '249', '280', 'AZAMBUJA' ],
            [ '301', '371', 'AZAMBUJA' ] ] },
        'Tagarro' => {
          '2065' => [
            [ '201', '252', 'ALCOENTRE' ] ] }
              },
      'Sobral Monte Agrao' => {
        'So Martinho' => {
          '2590' => [
            [ '429', '432', 'SOBRAL DE MONTE AGRAO' ] ] },
        'Casal da Azenha' => {
          '2590' => [
            [ '216', '220', 'SOBRAL DE MONTE AGRAO' ] ] },
        'Cachimbos' => {
          '2590' => [
            [ '211', '214', 'SOBRAL DE MONTE AGRAO' ] ] },
        'Ch' => {
          '2590' => [
            [ '230', '230', 'SOBRAL DE MONTE AGRAO' ] ] },
        'Sobral de Monte Agrao' => {
          '2590' => [
            [ '001', '007', 'SOBRAL DE MONTE AGRAO' ],
            [ '011', '031', 'SOBRAL DE MONTE AGRAO' ],
            [ '034', '061', 'SOBRAL DE MONTE AGRAO' ],
            [ '063', '064', 'SOBRAL DE MONTE AGRAO' ],
            [ '066', '067', 'SOBRAL DE MONTE AGRAO' ],
            [ '069', '072', 'SOBRAL DE MONTE AGRAO' ],
            [ '074', '083', 'SOBRAL DE MONTE AGRAO' ] ] },
        'Barqueira' => {
          '2590' => [
            [ '501', '504', 'SOBRAL DE MONTE AGRAO' ] ] },
        'Fetais de Nossa Senhora' => {
          '2590' => [
            [ '270', '294', 'SOBRAL DE MONTE AGRAO' ] ] },
        'Abadia' => {
          '2590' => [
            [ '202', '209', 'SOBRAL DE MONTE AGRAO' ] ] },
        'Chos' => {
          '2590' => [
            [ '506', '512', 'SOBRAL DE MONTE AGRAO' ] ] },
        'Chancos' => {
          '2590' => [
            [ '265', '265', 'SOBRAL DE MONTE AGRAO' ] ] },
        'Serreira' => {
          '2590' => [
            [ '401', '425', 'SOBRAL DE MONTE AGRAO' ] ] }
             },
      'Loures' => {
        'Bairro Nossa Senhora da Sade' => {
          '2680' => [
            [ '465', '466', 'UNHOS' ],
            [ '506', '507', 'UNHOS' ],
            [ '569', '569', 'UNHOS' ],
            [ '585', '585', 'UNHOS' ],
            [ '601', '601', 'UNHOS' ] ] },
        'Bairro de Santo Antnio' => {
          '2680' => [
            [ '516', '516', 'CAMARATE' ] ] },
        'Fetais' => {
          '2680' => [
            [ '134', '140', 'CAMARATE' ],
            [ '142', '142', 'CAMARATE' ],
            [ '145', '145', 'CAMARATE' ],
            [ '147', '149', 'CAMARATE' ],
            [ '151', '166', 'CAMARATE' ],
            [ '168', '179', 'CAMARATE' ] ] },
        'Bairro do Tentilho' => {
          '2680' => [
            [ '496', '496', 'UNHOS' ],
            [ '498', '501', 'UNHOS' ],
            [ '520', '520', 'UNHOS' ],
            [ '546', '546', 'UNHOS' ],
            [ '551', '552', 'UNHOS' ],
            [ '558', '559', 'UNHOS' ],
            [ '563', '563', 'UNHOS' ] ] },
        'So Julio do Tojal' => {
          '2660' => [
            [ '456', '456', 'SO JULIO DO TOJAL' ] ] },
        'Bairro da Boavista' => {
          '2680' => [
            [ '001', '029', 'CAMARATE' ] ] },
        'Bairro dos Coves' => {
          '2695' => [
            [ '006', '062', 'BOBADELA LRS' ] ] },
        'Vila de Rei' => {
          '2674' => [
            [ '505', '505', 'BUCELAS' ] ] },
        'Unhos' => {
          '2680' => [
            [ '434', '437', 'UNHOS' ],
            [ '441', '447', 'UNHOS' ],
            [ '449', '451', 'UNHOS' ],
            [ '517', '518', 'UNHOS' ] ] },
        'Bairro da Figueira' => {
          '2695' => [
            [ '791', '791', 'BOBADELA LRS' ],
            [ '800', '801', 'BOBADELA LRS' ],
            [ '828', '828', 'BOBADELA LRS' ],
            [ '830', '830', 'BOBADELA LRS' ] ] },
        'Catujal Velho' => {
          '2680' => [
            [ '416', '424', 'UNHOS' ],
            [ '426', '426', 'UNHOS' ],
            [ '428', '432', 'UNHOS' ],
            [ '482', '483', 'UNHOS' ] ] },
        'Bairro de Angola' => {
          '2680' => [
            [ '034', '037', 'CAMARATE' ],
            [ '041', '073', 'CAMARATE' ] ] },
        'Frielas' => {
          '2660' => [
            [ '001', '055', 'FRIELAS' ],
            [ '059', '059', 'FRIELAS' ],
            [ '063', '098', 'FRIELAS' ] ] },
        'Bairro das Queimadas' => {
          '2680' => [
            [ '355', '356', 'UNHOS' ],
            [ '358', '414', 'UNHOS' ] ] },
        'Quinta da Vitria' => {
          '2685' => [
            [ '181', '190', 'PORTELA LRS' ] ] },
        'Loures' => {
          '2670' => [
            [ '337', '379', 'LOURES' ],
            [ '410', '456', 'LOURES' ],
            [ '458', '461', 'LOURES' ],
            [ '463', '540', 'LOURES' ] ],
          '2674' => [
            [ '501', '504', 'LOURES' ] ] },
        'Prior Velho' => {
          '2689' => [
            [ '504', '504', 'PRIOR VELHO' ],
            [ '507', '507', 'PRIOR VELHO' ],
            [ '509', '511', 'PRIOR VELHO' ],
            [ '513', '514', 'PRIOR VELHO' ],
            [ '516', '520', 'PRIOR VELHO' ],
            [ '522', '524', 'PRIOR VELHO' ],
            [ '530', '531', 'PRIOR VELHO' ],
            [ '537', '537', 'PRIOR VELHO' ] ],
          '2685' => [
            [ '312', '314', 'PRIOR VELHO' ],
            [ '316', '316', 'PRIOR VELHO' ],
            [ '318', '319', 'PRIOR VELHO' ],
            [ '323', '327', 'PRIOR VELHO' ],
            [ '330', '340', 'PRIOR VELHO' ],
            [ '343', '352', 'PRIOR VELHO' ],
            [ '355', '356', 'PRIOR VELHO' ],
            [ '363', '368', 'PRIOR VELHO' ],
            [ '370', '376', 'PRIOR VELHO' ],
            [ '378', '380', 'PRIOR VELHO' ],
            [ '382', '385', 'PRIOR VELHO' ],
            [ '387', '390', 'PRIOR VELHO' ],
            [ '392', '405', 'PRIOR VELHO' ],
            [ '407', '410', 'PRIOR VELHO' ],
            [ '869', '869', 'PRIOR VELHO' ],
            [ '882', '882', 'PRIOR VELHO' ] ] },
          '7 Casas' => {
          '2674' => [
            [ '506', '506', 'LOURES' ] ] },
        'Bairro dos Moinhos Velhos' => {
          '2695' => [
            [ '672', '687', 'SO JOO DA TALHA' ] ] },
        'Urbanizao Quinta do Galeo' => {
          '2680' => [
            [ '464', '464', 'CAMARATE' ],
            [ '480', '481', 'CAMARATE' ],
            [ '502', '503', 'CAMARATE' ],
            [ '521', '523', 'CAMARATE' ],
            [ '545', '545', 'CAMARATE' ] ] },
        'Quinta da Boca' => {
          '2660' => [
            [ '366', '437', 'SO JULIO DO TOJAL' ] ] },
        'Bairro do Espinhal' => {
          '2680' => [
            [ '493', '494', 'UNHOS' ],
            [ '504', '504', 'UNHOS' ],
            [ '511', '513', 'UNHOS' ],
            [ '544', '544', 'UNHOS' ],
            [ '575', '575', 'UNHOS' ],
            [ '578', '578', 'UNHOS' ],
            [ '580', '581', 'UNHOS' ] ] },
        'Apelao' => {
          '2680' => [
            [ '261', '281', 'APELAO' ],
            [ '283', '284', 'APELAO' ],
            [ '286', '292', 'APELAO' ],
            [ '294', '304', 'APELAO' ],
            [ '308', '311', 'APELAO' ],
            [ '313', '322', 'APELAO' ],
            [ '325', '329', 'APELAO' ],
            [ '603', '603', 'APELAO' ] ] },
        'Moscavide' => {
          '1885' => [
            [ '001', '027', 'MOSCAVIDE' ],
            [ '029', '074', 'MOSCAVIDE' ],
            [ '076', '076', 'MOSCAVIDE' ],
            [ '078', '086', 'MOSCAVIDE' ] ],
          '1886' => [
            [ '501', '501', 'MOSCAVIDE' ] ],
          '1990' => [
            [ '354', '354', 'MOSCAVIDE' ],
            [ '374', '378', 'MOSCAVIDE' ],
            [ '382', '382', 'MOSCAVIDE' ],
            [ '384', '389', 'MOSCAVIDE' ],
            [ '391', '392', 'MOSCAVIDE' ],
            [ '394', '395', 'MOSCAVIDE' ],
            [ '398', '399', 'MOSCAVIDE' ],
            [ '401', '403', 'MOSCAVIDE' ],
            [ '413', '413', 'MOSCAVIDE' ],
            [ '420', '422', 'MOSCAVIDE' ],
            [ '424', '427', 'MOSCAVIDE' ] ] },
        'Vale de Figueira' => {
          '2695' => [
            [ '129', '136', 'SO JOO DA TALHA' ],
            [ '504', '505', 'SO JOO DA TALHA' ],
            [ '638', '643', 'SO JOO DA TALHA' ] ] },
        'Bobadela' => {
          '2699' => [
            [ '535', '535', 'BOBADELA LRS' ] ],
          '2695' => [
            [ '066', '080', 'BOBADELA LRS' ],
            [ '082', '083', 'BOBADELA LRS' ],
            [ '137', '138', 'BOBADELA LRS' ],
            [ '444', '445', 'BOBADELA LRS' ],
            [ '467', '469', 'BOBADELA LRS' ],
            [ '809', '810', 'BOBADELA LRS' ] ] },
        'Bairro de So Jos' => {
          '2680' => [
            [ '467', '474', 'CAMARATE' ],
            [ '547', '550', 'CAMARATE' ],
            [ '553', '553', 'CAMARATE' ],
            [ '570', '572', 'CAMARATE' ],
            [ '591', '595', 'CAMARATE' ] ] },
        'Manjoeira' => {
          '2660' => [
            [ '455', '455', 'SANTO ANTO DO TOJAL' ],
            [ '464', '465', 'SANTO ANTO DO TOJAL' ] ] },
        'Infantado' => {
          '2670' => [
            [ '383', '408', 'LOURES' ] ] },
        'Sacavm' => {
          '2689' => [
            [ '501', '503', 'SACAVM' ],
            [ '508', '508', 'SACAVM' ],
            [ '521', '521', 'SACAVM' ] ],
          '2685' => [
            [ '002', '003', 'SACAVM' ],
            [ '005', '005', 'SACAVM' ],
            [ '010', '039', 'SACAVM' ],
            [ '041', '048', 'SACAVM' ],
            [ '051', '114', 'SACAVM' ],
            [ '116', '116', 'SACAVM' ],
            [ '119', '120', 'SACAVM' ],
            [ '124', '151', 'SACAVM' ],
            [ '870', '873', 'SACAVM' ],
            [ '877', '877', 'SACAVM' ],
            [ '879', '881', 'SACAVM' ],
            [ '883', '883', 'SACAVM' ] ],
          '1990' => [
            [ '242', '242', 'SACAVM' ],
            [ '362', '362', 'SACAVM' ],
            [ '503', '505', 'SACAVM' ],
            [ '511', '511', 'SACAVM' ],
            [ '513', '514', 'SACAVM' ],
            [ '519', '519', 'SACAVM' ],
            [ '521', '522', 'SACAVM' ] ] },
        'Bucelas' => {
          '2670' => [
            [ '628', '628', 'BUCELAS' ],
            [ '685', '685', 'BUCELAS' ] ] },
        'Parque das Naes' => {
          '1990' => [
            [ '356', '356', 'MOSCAVIDE' ],
            [ '368', '372', 'MOSCAVIDE' ],
            [ '408', '408', 'MOSCAVIDE' ] ] },
        'Bairro de So Francisco' => {
          '2680' => [
            [ '477', '478', 'CAMARATE' ],
            [ '489', '492', 'CAMARATE' ],
            [ '495', '495', 'CAMARATE' ],
            [ '574', '574', 'CAMARATE' ],
            [ '582', '584', 'CAMARATE' ],
            [ '589', '589', 'CAMARATE' ] ],
          '2681' => [
            [ '501', '502', 'CAMARATE' ] ] },
        'Bairro Belo Horizonte' => {
          '2695' => [
            [ '088', '088', 'SO JOO DA TALHA' ],
            [ '107', '108', 'SO JOO DA TALHA' ],
            [ '117', '117', 'SO JOO DA TALHA' ],
            [ '446', '446', 'SO JOO DA TALHA' ],
            [ '867', '868', 'SO JOO DA TALHA' ] ] },
        'Bairro da Bela Vista' => {
          '2695' => [
            [ '001', '002', 'BOBADELA LRS' ],
            [ '109', '109', 'BOBADELA LRS' ],
            [ '447', '447', 'BOBADELA LRS' ],
            [ '465', '465', 'BOBADELA LRS' ],
            [ '509', '510', 'BOBADELA LRS' ],
            [ '856', '857', 'BOBADELA LRS' ] ] },
        'Bairro dos Troviscais' => {
          '2695' => [
            [ '110', '110', 'SO JOO DA TALHA' ],
            [ '112', '114', 'SO JOO DA TALHA' ],
            [ '120', '120', 'SO JOO DA TALHA' ],
            [ '453', '454', 'SO JOO DA TALHA' ],
            [ '499', '500', 'SO JOO DA TALHA' ],
            [ '855', '855', 'SO JOO DA TALHA' ],
            [ '871', '873', 'SO JOO DA TALHA' ] ] },
        'Santa Iria de Azia' => {
          '2691' => [
            [ '001', '003', 'SANTA IRIA DE AZOIA' ] ],
          '2690' => [
            [ '141', '188', 'SANTA IRIA DE AZOIA' ],
            [ '190', '212', 'SANTA IRIA DE AZOIA' ],
            [ '214', '233', 'SANTA IRIA DE AZOIA' ],
            [ '235', '243', 'SANTA IRIA DE AZOIA' ],
            [ '383', '395', 'SANTA IRIA DE AZOIA' ],
            [ '397', '435', 'SANTA IRIA DE AZOIA' ],
            [ '437', '441', 'SANTA IRIA DE AZOIA' ],
            [ '443', '489', 'SANTA IRIA DE AZOIA' ],
            [ '491', '502', 'SANTA IRIA DE AZOIA' ],
            [ '504', '523', 'SANTA IRIA DE AZOIA' ],
            [ '526', '596', 'SANTA IRIA DE AZOIA' ] ] },
        'Bairro do Cabeo da Aguieira' => {
          '2680' => [
            [ '475', '476', 'UNHOS' ],
            [ '524', '526', 'UNHOS' ],
            [ '587', '588', 'UNHOS' ],
            [ '590', '590', 'UNHOS' ],
            [ '596', '599', 'UNHOS' ] ] },
        'Quinta de Santa Rosa' => {
          '2680' => [
            [ '586', '586', 'CAMARATE' ] ] },
        'Bairro das Sousas' => {
          '2680' => [
            [ '452', '461', 'CAMARATE' ],
            [ '484', '485', 'CAMARATE' ],
            [ '568', '568', 'CAMARATE' ] ] },
        'Catujal' => {
          '2680' => [
            [ '462', '463', 'UNHOS' ],
            [ '565', '567', 'UNHOS' ],
            [ '605', '605', 'UNHOS' ] ] },
        'Urbanizao da Portela' => {
          '2689' => [
            [ '505', '505', 'PORTELA LRS' ] ],
          '2685' => [
            [ '192', '192', 'PORTELA LRS' ],
            [ '194', '194', 'PORTELA LRS' ],
            [ '197', '200', 'PORTELA LRS' ],
            [ '202', '204', 'PORTELA LRS' ],
            [ '206', '229', 'PORTELA LRS' ],
            [ '231', '240', 'PORTELA LRS' ],
            [ '243', '247', 'PORTELA LRS' ],
            [ '874', '876', 'PORTELA LRS' ],
            [ '878', '878', 'PORTELA LRS' ] ] },
        'Bairro das Palmeiras' => {
          '2695' => [
            [ '794', '794', 'SO JOO DA TALHA' ],
            [ '829', '829', 'SO JOO DA TALHA' ] ] },
        'So Joo da Talha' => {
          '2699' => [
            [ '536', '536', 'SO JOO DA TALHA' ] ],
          '2695' => [
            [ '096', '096', 'SO JOO DA TALHA' ],
            [ '098', '101', 'SO JOO DA TALHA' ],
            [ '448', '448', 'SO JOO DA TALHA' ],
            [ '451', '451', 'SO JOO DA TALHA' ],
            [ '461', '461', 'SO JOO DA TALHA' ],
            [ '466', '466', 'SO JOO DA TALHA' ],
            [ '476', '478', 'SO JOO DA TALHA' ],
            [ '480', '483', 'SO JOO DA TALHA' ],
            [ '487', '490', 'SO JOO DA TALHA' ],
            [ '511', '555', 'SO JOO DA TALHA' ],
            [ '645', '657', 'SO JOO DA TALHA' ],
            [ '659', '670', 'SO JOO DA TALHA' ],
            [ '689', '715', 'SO JOO DA TALHA' ],
            [ '718', '719', 'SO JOO DA TALHA' ],
            [ '721', '723', 'SO JOO DA TALHA' ],
            [ '725', '728', 'SO JOO DA TALHA' ],
            [ '730', '786', 'SO JOO DA TALHA' ],
            [ '790', '790', 'SO JOO DA TALHA' ],
            [ '814', '814', 'SO JOO DA TALHA' ],
            [ '853', '853', 'SO JOO DA TALHA' ] ] },
        'Quinta So Joo das Areias' => {
          '2680' => [
            [ '181', '181', 'CAMARATE' ] ] },
        'Bairro Nossa Senhora da Nazar' => {
          '2680' => [
            [ '515', '515', 'UNHOS' ] ] },
        'Bairro do Moinho de Vento' => {
          '2680' => [
            [ '330', '353', 'UNHOS' ] ] },
        'Casal da Pedra' => {
          '2670' => [
            [ '740', '773', 'LOUSA LRS' ] ] },
        'Bairro de Santiago' => {
          '2680' => [
            [ '075', '075', 'CAMARATE' ],
            [ '077', '092', 'CAMARATE' ] ] },
        'Bairro Boca' => {
          '2660' => [
            [ '355', '364', 'SO JULIO DO TOJAL' ] ] },
        'Bairro do Grilo' => {
          '2680' => [
            [ '095', '105', 'CAMARATE' ] ] },
        'Junqueira' => {
          '2660' => [
            [ '466', '468', 'SO JULIO DO TOJAL' ] ] },
        'Camarate' => {
          '2680' => [
            [ '031', '031', 'CAMARATE' ],
            [ '110', '118', 'CAMARATE' ],
            [ '120', '120', 'CAMARATE' ],
            [ '124', '132', 'CAMARATE' ],
            [ '505', '505', 'CAMARATE' ],
            [ '508', '510', 'CAMARATE' ],
            [ '514', '514', 'CAMARATE' ],
            [ '534', '534', 'CAMARATE' ],
            [ '557', '557', 'CAMARATE' ],
            [ '560', '562', 'CAMARATE' ],
            [ '576', '577', 'CAMARATE' ],
            [ '602', '602', 'CAMARATE' ],
            [ '604', '604', 'CAMARATE' ] ] },
        'Bairro Olival das Bairradas' => {
          '2680' => [
            [ '486', '488', 'UNHOS' ],
            [ '535', '538', 'UNHOS' ],
            [ '554', '556', 'UNHOS' ] ] },
        'Fanhes' => {
          '2670' => [
            [ '686', '738', 'FANHES' ] ] },
        'Santo Antnio dos Cavaleiros' => {
          '2660' => [
            [ '204', '237', 'SANTO ANTNIO CAVALEIROS' ],
            [ '240', '266', 'SANTO ANTNIO CAVALEIROS' ],
            [ '268', '268', 'SANTO ANTNIO CAVALEIROS' ],
            [ '270', '317', 'SANTO ANTNIO CAVALEIROS' ],
            [ '319', '337', 'SANTO ANTNIO CAVALEIROS' ],
            [ '340', '340', 'SANTO ANTNIO CAVALEIROS' ],
            [ '346', '354', 'SANTO ANTNIO CAVALEIROS' ],
            [ '438', '439', 'SANTO ANTNIO CAVALEIROS' ],
            [ '441', '454', 'SANTO ANTNIO CAVALEIROS' ],
            [ '457', '463', 'SANTO ANTNIO CAVALEIROS' ] ] },
        'Bairro Terra de Frades' => {
          '2690' => [
            [ '245', '253', 'SANTA IRIA DE AZOIA' ] ] },
        'Bairro de So Vicente' => {
          '2695' => [
            [ '103', '104', 'SO JOO DA TALHA' ],
            [ '127', '127', 'SO JOO DA TALHA' ],
            [ '139', '140', 'SO JOO DA TALHA' ] ] },
        'Quinta da Chamorra' => {
          '2670' => [
            [ '630', '680', 'BUCELAS' ] ] },
        'Bairro Ch' => {
          '2660' => [
            [ '099', '203', 'SANTO ANTO DO TOJAL' ] ] },
        'Bairro Quinta da Fbrica' => {
          '2680' => [
            [ '573', '573', 'UNHOS' ] ] },
        'Bairro da Esperana' => {
          '2695' => [
            [ '799', '799', 'SO JOO DA TALHA' ],
            [ '817', '817', 'SO JOO DA TALHA' ] ] },
        'Bairro dos Telefones' => {
          '2695' => [
            [ '452', '452', 'BOBADELA LRS' ],
            [ '854', '854', 'BOBADELA LRS' ],
            [ '874', '875', 'BOBADELA LRS' ] ] },
        'Bairro da Fraternidade' => {
          '2695' => [
            [ '090', '090', 'SO JOO DA TALHA' ],
            [ '092', '093', 'SO JOO DA TALHA' ],
            [ '122', '123', 'SO JOO DA TALHA' ],
            [ '464', '464', 'SO JOO DA TALHA' ],
            [ '470', '471', 'SO JOO DA TALHA' ],
            [ '508', '508', 'SO JOO DA TALHA' ],
            [ '630', '636', 'SO JOO DA TALHA' ],
            [ '802', '808', 'SO JOO DA TALHA' ],
            [ '820', '827', 'SO JOO DA TALHA' ],
            [ '831', '831', 'SO JOO DA TALHA' ],
            [ '833', '846', 'SO JOO DA TALHA' ] ] },
        'Bairro da Petrogal' => {
          '2695' => [
            [ '004', '004', 'BOBADELA LRS' ],
            [ '085', '087', 'BOBADELA LRS' ],
            [ '089', '089', 'BOBADELA LRS' ],
            [ '091', '091', 'BOBADELA LRS' ],
            [ '094', '095', 'BOBADELA LRS' ],
            [ '097', '097', 'BOBADELA LRS' ],
            [ '102', '102', 'BOBADELA LRS' ],
            [ '105', '106', 'BOBADELA LRS' ],
            [ '111', '111', 'BOBADELA LRS' ],
            [ '115', '116', 'BOBADELA LRS' ],
            [ '118', '119', 'BOBADELA LRS' ],
            [ '121', '121', 'BOBADELA LRS' ],
            [ '124', '126', 'BOBADELA LRS' ],
            [ '128', '128', 'BOBADELA LRS' ],
            [ '449', '450', 'BOBADELA LRS' ],
            [ '455', '460', 'BOBADELA LRS' ],
            [ '462', '463', 'BOBADELA LRS' ],
            [ '472', '475', 'BOBADELA LRS' ],
            [ '479', '479', 'BOBADELA LRS' ],
            [ '484', '486', 'BOBADELA LRS' ],
            [ '491', '493', 'BOBADELA LRS' ],
            [ '495', '498', 'BOBADELA LRS' ],
            [ '501', '503', 'BOBADELA LRS' ],
            [ '506', '507', 'BOBADELA LRS' ],
            [ '793', '793', 'BOBADELA LRS' ],
            [ '795', '796', 'BOBADELA LRS' ],
            [ '798', '798', 'BOBADELA LRS' ],
            [ '812', '813', 'BOBADELA LRS' ],
            [ '815', '816', 'BOBADELA LRS' ],
            [ '818', '819', 'BOBADELA LRS' ],
            [ '832', '832', 'BOBADELA LRS' ],
            [ '847', '852', 'BOBADELA LRS' ],
            [ '860', '866', 'BOBADELA LRS' ],
            [ '869', '870', 'BOBADELA LRS' ] ] },
        'Bairro da Vinha Grande' => {
          '2695' => [
            [ '494', '494', 'SO JOO DA TALHA' ],
            [ '811', '811', 'SO JOO DA TALHA' ],
            [ '858', '859', 'SO JOO DA TALHA' ] ] },
        'Quintanilho' => {
          '2664' => [
            [ '500', '502', 'SO JULIO DO TOJAL' ] ] },
        'Bairro da Castelhana' => {
          '2695' => [
            [ '081', '081', 'SO JOO DA TALHA' ],
            [ '557', '564', 'SO JOO DA TALHA' ],
            [ '566', '628', 'SO JOO DA TALHA' ],
            [ '792', '792', 'SO JOO DA TALHA' ],
            [ '797', '797', 'SO JOO DA TALHA' ] ] },
        'Bairro Casal dos Machados' => {
          '2680' => [
            [ '479', '479', 'UNHOS' ],
            [ '533', '533', 'UNHOS' ],
            [ '540', '541', 'UNHOS' ] ] },
        'Portela da Azia' => {
          '2690' => [
            [ '255', '281', 'SANTA IRIA DE AZOIA' ],
            [ '283', '317', 'SANTA IRIA DE AZOIA' ],
            [ '319', '379', 'SANTA IRIA DE AZOIA' ] ] },
        'Bairro da Bogalheira' => {
          '2680' => [
            [ '183', '186', 'CAMARATE' ],
            [ '497', '497', 'CAMARATE' ],
            [ '519', '519', 'CAMARATE' ],
            [ '527', '532', 'CAMARATE' ],
            [ '542', '543', 'CAMARATE' ],
            [ '564', '564', 'CAMARATE' ],
            [ '579', '579', 'CAMARATE' ],
            [ '600', '600', 'CAMARATE' ] ] }
            }
          },
        'vora' => {
      'Arraiolos' => {
        'Monte Cotovia' => {
          '7040' => [
            [ '301', '303', 'SANTA JUSTA' ] ] },
        'Arraiolos' => {
          '7040' => [
            [ '010', '051', 'ARRAIOLOS' ],
            [ '053', '071', 'ARRAIOLOS' ],
            [ '073', '092', 'ARRAIOLOS' ] ] },
        'Aldeia da Serra' => {
          '7040' => [
            [ '401', '419', 'SO GREGRIO ARL' ] ] },
        'Monte a Rabisco' => {
          '7040' => [
            [ '260', '273', 'IGREJINHA' ] ] },
        'Herdade do Almargem' => {
          '7040' => [
            [ '501', '516', 'SO PEDRO GAFANHOEIRA' ] ] },
        'Quinta Horta do Negrinho' => {
          '7040' => [
            [ '601', '602', 'VIMIEIRO ARL' ] ] },
        'So Pedro Gafanhoeira' => {
          '7040' => [
            [ '520', '546', 'SO PEDRO GAFANHOEIRA' ] ] },
        'Casas Novas' => {
          '7040' => [
            [ '110', '130', 'ARRAIOLOS' ] ] },
        'Herdade Comenda Grande' => {
          '7040' => [
            [ '201', '235', 'IGREJINHA' ] ] },
        'Sabugueiro' => {
          '7040' => [
            [ '701', '723', 'SABUGUEIRO ARL' ] ] },
        'Vimieiro' => {
          '7040' => [
            [ '610', '668', 'VIMIEIRO ARL' ] ] }
              },
        'vora' => {
        'Azaruja' => {
          '7005' => [
            [ '100', '132', 'AZARUJA' ] ] },
        'vora' => {
          '7000' => [
            [ '025', '026', 'VORA' ],
            [ '049', '049', 'VORA' ],
            [ '056', '057', 'VORA' ],
            [ '061', '074', 'VORA' ],
            [ '076', '077', 'VORA' ],
            [ '079', '080', 'VORA' ],
            [ '131', '138', 'VORA' ],
            [ '141', '141', 'VORA' ],
            [ '153', '153', 'VORA' ],
            [ '162', '163', 'VORA' ],
            [ '180', '180', 'VORA' ],
            [ '187', '188', 'VORA' ],
            [ '250', '253', 'VORA' ],
            [ '283', '283', 'VORA' ],
            [ '292', '313', 'VORA' ],
            [ '334', '338', 'VORA' ],
            [ '363', '412', 'VORA' ],
            [ '452', '466', 'VORA' ],
            [ '472', '472', 'VORA' ],
            [ '501', '503', 'VORA' ],
            [ '505', '570', 'VORA' ],
            [ '572', '643', 'VORA' ],
            [ '645', '688', 'VORA' ],
            [ '691', '692', 'VORA' ],
            [ '705', '708', 'VORA' ],
            [ '712', '713', 'VORA' ],
            [ '719', '719', 'VORA' ],
            [ '721', '721', 'VORA' ],
            [ '723', '723', 'VORA' ],
            [ '726', '726', 'VORA' ],
            [ '732', '733', 'VORA' ],
            [ '736', '737', 'VORA' ],
            [ '739', '741', 'VORA' ],
            [ '744', '745', 'VORA' ],
            [ '748', '749', 'VORA' ],
            [ '752', '752', 'VORA' ],
            [ '755', '755', 'VORA' ],
            [ '757', '759', 'VORA' ],
            [ '766', '768', 'VORA' ],
            [ '772', '773', 'VORA' ],
            [ '781', '781', 'VORA' ],
            [ '783', '786', 'VORA' ],
            [ '792', '795', 'VORA' ],
            [ '797', '797', 'VORA' ],
            [ '799', '804', 'VORA' ],
            [ '806', '806', 'VORA' ],
            [ '809', '813', 'VORA' ],
            [ '815', '816', 'VORA' ],
            [ '818', '820', 'VORA' ],
            [ '822', '824', 'VORA' ],
            [ '830', '830', 'VORA' ],
            [ '833', '836', 'VORA' ],
            [ '839', '842', 'VORA' ],
            [ '845', '845', 'VORA' ],
            [ '847', '849', 'VORA' ],
            [ '854', '854', 'VORA' ],
            [ '856', '856', 'VORA' ],
            [ '859', '859', 'VORA' ],
            [ '862', '866', 'VORA' ],
            [ '868', '868', 'VORA' ],
            [ '870', '870', 'VORA' ],
            [ '872', '878', 'VORA' ],
            [ '880', '881', 'VORA' ],
            [ '883', '883', 'VORA' ],
            [ '888', '888', 'VORA' ],
            [ '890', '893', 'VORA' ],
            [ '896', '899', 'VORA' ],
            [ '902', '902', 'VORA' ],
            [ '906', '907', 'VORA' ],
            [ '909', '909', 'VORA' ],
            [ '912', '912', 'VORA' ],
            [ '918', '920', 'VORA' ],
            [ '922', '925', 'VORA' ],
            [ '928', '930', 'VORA' ],
            [ '932', '932', 'VORA' ],
            [ '935', '935', 'VORA' ],
            [ '937', '937', 'VORA' ],
            [ '940', '947', 'VORA' ],
            [ '949', '961', 'VORA' ],
            [ '963', '964', 'VORA' ],
            [ '966', '967', 'VORA' ],
            [ '969', '969', 'VORA' ],
            [ '971', '971', 'VORA' ] ],
          '7005' => [
            [ '001', '003', 'VORA' ],
            [ '133', '135', 'VORA' ],
            [ '137', '138', 'VORA' ],
            [ '140', '140', 'VORA' ],
            [ '144', '144', 'VORA' ],
            [ '150', '150', 'VORA' ],
            [ '152', '152', 'VORA' ],
            [ '158', '158', 'VORA' ],
            [ '160', '161', 'VORA' ],
            [ '169', '169', 'VORA' ],
            [ '175', '175', 'VORA' ],
            [ '177', '177', 'VORA' ],
            [ '181', '181', 'VORA' ],
            [ '187', '198', 'VORA' ],
            [ '204', '204', 'VORA' ],
            [ '206', '206', 'VORA' ],
            [ '208', '213', 'VORA' ],
            [ '218', '270', 'VORA' ],
            [ '272', '279', 'VORA' ],
            [ '283', '288', 'VORA' ],
            [ '290', '297', 'VORA' ],
            [ '299', '320', 'VORA' ],
            [ '322', '326', 'VORA' ],
            [ '328', '331', 'VORA' ],
            [ '335', '341', 'VORA' ],
            [ '343', '343', 'VORA' ],
            [ '345', '346', 'VORA' ],
            [ '348', '349', 'VORA' ],
            [ '351', '354', 'VORA' ],
            [ '356', '356', 'VORA' ],
            [ '358', '359', 'VORA' ],
            [ '361', '367', 'VORA' ],
            [ '369', '374', 'VORA' ],
            [ '376', '383', 'VORA' ],
            [ '385', '385', 'VORA' ],
            [ '387', '393', 'VORA' ],
            [ '395', '395', 'VORA' ],
            [ '397', '397', 'VORA' ],
            [ '399', '403', 'VORA' ],
            [ '405', '413', 'VORA' ],
            [ '415', '430', 'VORA' ],
            [ '432', '451', 'VORA' ],
            [ '453', '456', 'VORA' ],
            [ '458', '459', 'VORA' ],
            [ '461', '463', 'VORA' ],
            [ '465', '475', 'VORA' ],
            [ '477', '485', 'VORA' ],
            [ '487', '500', 'VORA' ],
            [ '502', '552', 'VORA' ],
            [ '554', '563', 'VORA' ],
            [ '565', '599', 'VORA' ],
            [ '601', '604', 'VORA' ],
            [ '606', '621', 'VORA' ],
            [ '623', '632', 'VORA' ],
            [ '634', '644', 'VORA' ],
            [ '654', '670', 'VORA' ],
            [ '794', '828', 'VORA' ],
            [ '851', '873', 'VORA' ] ],
          '7004' => [
            [ '501', '508', 'VORA' ],
            [ '511', '511', 'VORA' ],
            [ '514', '514', 'VORA' ],
            [ '516', '516', 'VORA' ],
            [ '550', '550', 'VORA' ] ] },
        'Herdade Ximenes' => {
          '7000' => [
            [ '014', '021', 'N SENHORA GRAA DO DIVOR' ] ] },
        'Casa da Boa Esperana' => {
          '7005' => [
            [ '671', '715', 'N SENHORA DE MACHEDE' ] ] },
        'Monte da Bragada' => {
          '7200' => [
            [ '041', '042', 'SO VICENTE DO PIGEIRO' ] ] },
        'Canada do Bairro' => {
          '7005' => [
            [ '716', '750', 'SO MANOS' ] ] },
        'Foros da Amada' => {
          '7000' => [
            [ '221', '223', 'N SENHORA DE GUADALUPE' ] ] },
        'Foros da Carvalhas' => {
          '7000' => [
            [ '011', '013', 'NOSSA SENHORA DA BOA F' ] ] },
        'Bairro da Tapada' => {
          '7005' => [
            [ '751', '770', 'SO MIGUEL DE MACHEDE' ] ] },
        'Castelos' => {
          '7000' => [
            [ '201', '202', 'S SEBASTIO DA GIESTEIRA' ] ] },
        'Quinta Verde' => {
          '7005' => [
            [ '830', '831', 'VORA' ] ] },
        'Quinta Padre Pina' => {
          '7000' => [
            [ '171', '173', 'VORA' ] ] },
        'MARE - Mercado Abastecedor da Regio vora' => {
          '7009' => [
            [ '503', '505', 'VORA' ] ] },
        'So Matias' => {
          '7000' => [
            [ '177', '178', 'VORA' ] ] },
        'Quinta da Clareja' => {
          '7000' => [
            [ '175', '175', 'VORA' ] ] },
        'Herdade da Serra da Alpedrinha' => {
          '7005' => [
            [ '771', '793', 'TORRE DE COELHEIROS' ] ] },
        'Monte da Igreja' => {
          '7200' => [
            [ '031', '031', 'N SENHORA DE MACHEDE' ] ] },
        'Quinta do Sezes' => {
          '7005' => [
            [ '833', '849', 'VORA' ] ] },
        'Foros da Cheira' => {
          '7000' => [
            [ '081', '093', 'NOSSA SENHORA DA TOREGA' ] ] }
          },
      'Vila Viosa' => {
        'Vila Viosa' => {
          '7160' => [
            [ '367', '367', 'VILA VIOSA' ],
            [ '369', '369', 'VILA VIOSA' ] ] },
        'Estao Cp' => {
          '7160' => [
            [ '201', '204', 'VILA VIOSA' ] ] },
        'Monte da Boavista' => {
          '7160' => [
            [ '361', '366', 'PARDAIS' ] ] },
        'Bencatel' => {
          '7160' => [
            [ '050', '079', 'BENCATEL' ] ] },
        'So Romo' => {
          '7160' => [
            [ '120', '135', 'CILADAS' ] ] },
        'Urbanizao do Palcio' => {
          '7160' => [
            [ '206', '290', 'VILA VIOSA' ] ] },
        'Serra das Correias' => {
          '7160' => [
            [ '101', '101', 'CILADAS' ] ] },
        'Aldeia da Freira' => {
          '7160' => [
            [ '011', '024', 'BENCATEL' ] ] },
        'Almagreira' => {
          '7160' => [
            [ '351', '359', 'PARDAIS' ] ] }
          },
      'Borba' => {
        'So Gregrio' => {
          '7150' => [
            [ '390', '390', 'RIO DE MOINHOS BRB' ] ] },
        'Aldeia dos Grilos' => {
          '7150' => [
            [ '201', '249', 'BORBA' ] ] },
        'Borba' => {
          '7150' => [
            [ '101', '175', 'BORBA' ] ] },
        'Rio de Moinhos' => {
          '7150' => [
            [ '361', '381', 'RIO DE MOINHOS BRB' ] ] },
        'Aldeia de Fidalgo' => {
          '7150' => [
            [ '318', '351', 'RIO DE MOINHOS BRB' ] ] },
        'Monte da Boavista Alegre' => {
          '7150' => [
            [ '279', '317', 'ORADA' ] ] },
        'Aldeia das Gorduras' => {
          '7150' => [
            [ '261', '277', 'ORADA' ] ] }
          },
      'Viana do Alentejo' => {
        'Horta da Boavista' => {
          '7090' => [
            [ '201', '217', 'VIANA DO ALENTEJO' ] ] },
        'Alcovas' => {
          '7090' => [
            [ '010', '098', 'ALCOVAS' ] ] },
        'Aguiar' => {
          '7090' => [
            [ '401', '432', 'AGUIAR VNT' ] ] },
        'Viana do Alentejo' => {
          '7090' => [
            [ '220', '299', 'VIANA DO ALENTEJO' ] ] }
                },
      'Alandroal' => {
        'Monte Pipeira' => {
          '7250' => [
            [ '201', '203', 'ALANDROAL' ] ] },
        'Azinhal Redondo' => {
          '7250' => [
            [ '261', '289', 'CAPELINS' ] ] },
        'Montes das Ferrarias' => {
          '7250' => [
            [ '051', '053', 'SO BRS DOS MATOS' ] ] },
        'Vicentes' => {
          '7250' => [
            [ '065', '069', 'TERENA' ] ] },
        'Alandroal' => {
          '7250' => [
            [ '101', '154', 'ALANDROAL' ] ] },
        'Arrabalde de So Lzaro' => {
          '7250' => [
            [ '241', '243', 'JUROMENHA' ] ] },
        'Aldeia da Venda' => {
          '7200' => [
            [ '011', '021', 'SANTIAGO MAIOR' ] ] }
              },
      'Mouro' => {
        'Monte do Carvoeiro' => {
          '7240' => [
            [ '019', '020', 'GRANJA MOU' ] ] },
        'Luz' => {
          '7240' => [
            [ '100', '103', 'LUZ MOU' ] ] },
        'Monte da Abegoaria' => {
          '7240' => [
            [ '201', '204', 'MOURO' ] ] },
        'Mouro' => {
          '7240' => [
            [ '220', '243', 'MOURO' ],
            [ '245', '272', 'MOURO' ] ] },
        'Baldio da Granja' => {
          '7240' => [
            [ '011', '016', 'GRANJA MOU' ] ] }
           },
      'Montemor-o-Novo' => {
        'Alhos Vedros' => {
          '7050' => [
            [ '631', '646', 'CORTIADAS DE LAVRE' ] ] },
        'Monte do Barraco de Cima' => {
          '7050' => [
            [ '437', '444', 'CABRELA' ] ] },
        'Patalim' => {
          '7050' => [
            [ '321', '355', 'MONTEMOR-O-NOVO' ] ] },
        'Monte dos Cimarros' => {
          '7050' => [
            [ '486', '497', 'LAVRE' ] ] },
        'Casal de So Jos' => {
          '7050' => [
            [ '461', '463', 'LAVRE' ] ] },
        'Colnias' => {
          '7050' => [
            [ '664', '671', 'SILVEIRAS' ] ] },
        'Rangina' => {
          '7050' => [
            [ '446', '446', 'CABRELA' ] ] },
        'Monte da Ladeira de Cima' => {
          '7050' => [
            [ '475', '484', 'LAVRE' ] ] },
        'Vale de Carvalhos' => {
          '7050' => [
            [ '448', '448', 'CABRELA' ] ] },
        'Quinta Caida' => {
          '7050' => [
            [ '673', '678', 'SILVEIRAS' ] ] },
        'Herdade do Carrascal' => {
          '7050' => [
            [ '465', '465', 'LAVRE' ] ] },
        'Lavre' => {
          '7050' => [
            [ '467', '473', 'LAVRE' ] ] },
        'Aduraia' => {
          '7050' => [
            [ '661', '661', 'SILVEIRAS' ] ] },
        'Adua Corticeira' => {
          '7050' => [
            [ '001', '116', 'MONTEMOR-O-NOVO' ] ] },
        'Alsilor' => {
          '7050' => [
            [ '581', '602', 'SO CRISTVO' ] ] },
        'Montemor-o-Novo' => {
          '7050' => [
            [ '118', '118', 'MONTEMOR-O-NOVO' ],
            [ '120', '155', 'MONTEMOR-O-NOVO' ],
            [ '157', '269', 'MONTEMOR-O-NOVO' ],
            [ '271', '307', 'MONTEMOR-O-NOVO' ] ] },
        'Baldios' => {
          '7050' => [
            [ '401', '435', 'CABRELA' ] ] },
        'Courela da Freixeirinha' => {
          '7050' => [
            [ '701', '705', 'FOROS DE VALE DE FIGUEIRA' ] ] },
        'Amieira' => {
          '7050' => [
            [ '511', '563', 'SANTIAGO DO ESCOURAL' ] ] },
        'Ciborro' => {
          '7050' => [
            [ '611', '618', 'CIBORRO' ] ] }
              },
      'Mora' => {
        'Vale de Pegas' => {
          '7490' => [
            [ '120', '120', 'CABEO' ] ] },
        'Cabeceiras do Vale da Bica' => {
          '7490' => [
            [ '301', '315', 'MORA' ] ] },
        'Cabeo' => {
          '7490' => [
            [ '051', '090', 'CABEO' ],
            [ '092', '112', 'CABEO' ] ] },
        'Brotas' => {
          '7490' => [
            [ '011', '032', 'BROTAS' ] ] },
        'Bairro Coelho Lopes' => {
          '7490' => [
            [ '401', '409', 'PAVIA' ] ] },
        'Pao de Baixo' => {
          '7490' => [
            [ '319', '324', 'MORA' ] ] },
        'Pavia' => {
          '7490' => [
            [ '420', '450', 'PAVIA' ] ] },
        'Mora' => {
          '7490' => [
            [ '201', '263', 'MORA' ] ] }
         },
      'Portel' => {
        'Horta do Farumbo' => {
          '7220' => [
            [ '351', '442', 'PORTEL' ] ] },
        'Santana' => {
          '7220' => [
            [ '481', '504', 'SANTANA PRL' ],
            [ '506', '509', 'SANTANA PRL' ] ] },
        'Alqueva' => {
          '7220' => [
            [ '011', '046', 'ALQUEVA' ] ] },
        'So Bartolomeu do Outeiro' => {
          '7220' => [
            [ '521', '540', 'SO BARTOLOMEU DO OUTEIRO' ] ] },
        'Vera Cruz' => {
          '7220' => [
            [ '561', '582', 'VERA CRUZ' ] ] },
        'Quinta Santo Antnio dos Capuchos' => {
          '7220' => [
            [ '461', '461', 'PORTEL' ] ] },
        'Monte de Trigo' => {
          '7220' => [
            [ '201', '248', 'MONTE DO TRIGO' ] ] },
        'Amieira' => {
          '7220' => [
            [ '101', '133', 'AMIEIRA PRL' ] ] },
        'Oriola' => {
          '7220' => [
            [ '301', '301', 'ORIOLA' ] ] }
           },
      'Estremoz' => {
        'Amores' => {
          '7100' => [
            [ '650', '669', 'SO LOURENO DE MAMPORCO' ] ] },
        'Monte Corpo de Deus' => {
          '7100' => [
            [ '620', '620', 'SO BENTO DE ANA LOURA' ] ] },
        'Arcos' => {
          '7100' => [
            [ '011', '015', 'ARCOS ETZ' ],
            [ '017', '020', 'ARCOS ETZ' ],
            [ '024', '027', 'ARCOS ETZ' ] ] },
        'Casa dos Cantoneiros' => {
          '7100' => [
            [ '400', '402', 'SANTA VITRIA DO AMEIXIAL' ] ] },
        'Herdade das Carvalhas' => {
          '7100' => [
            [ '040', '042', 'GLRIA' ] ] },
        'Monte da Horta Nova' => {
          '7100' => [
            [ '600', '600', 'SO BENTO DO AMEIXIAL' ] ] },
        'Veiros' => {
          '7100' => [
            [ '680', '710', 'VEIROS ETZ' ] ] },
        'Santa Vitria do Ameixial' => {
          '7100' => [
            [ '405', '406', 'SANTA VITRIA DO AMEIXIAL' ] ] },
        'Eira da Cerina' => {
          '7100' => [
            [ '030', '036', 'ARCOS ETZ' ] ] },
        'Horta do Bota Fora' => {
          '7100' => [
            [ '144', '150', 'ESTREMOZ' ] ] },
        'Bairro das Correias' => {
          '7100' => [
            [ '200', '266', 'VORA MONTE' ] ] },
        'vora Monte' => {
          '7100' => [
            [ '300', '318', 'VORA MONTE' ] ] },
        'Espinheiro' => {
          '7100' => [
            [ '640', '641', 'SO DOMINGOS DE ANA LOURA' ] ] },
        'Azenha do Porto' => {
          '7100' => [
            [ '050', '078', 'ESTREMOZ' ] ] },
        'Monte Branco' => {
          '7100' => [
            [ '580', '580', 'SANTO ESTEVO ETZ' ] ] },
        'So Bento do Ameixial' => {
          '7100' => [
            [ '610', '610', 'SO BENTO DO AMEIXIAL' ] ] },
        'So Bento do Cortio' => {
          '7100' => [
            [ '630', '630', 'SO BENTO DO CORTIO' ] ] },
        'Estremoz' => {
          '7100' => [
            [ '100', '138', 'ESTREMOZ' ],
            [ '451', '464', 'ESTREMOZ' ],
            [ '500', '570', 'ESTREMOZ' ] ] },
        'Herdade da Cavaleira' => {
          '7100' => [
            [ '080', '080', 'ESTREMOZ' ] ] }
             },
      'Vendas Novas' => {
        'Monte dos Nicolaus' => {
          '2965' => [
            [ '441', '442', 'LANDEIRA' ] ] },
        'Landeira' => {
          '2965' => [
            [ '401', '421', 'LANDEIRA' ] ] },
        'Vendas Novas' => {
          '7080' => [
            [ '011', '103', 'VENDAS NOVAS' ],
            [ '105', '183', 'VENDAS NOVAS' ] ] },
        'Afeiteira' => {
          '7080' => [
            [ '301', '346', 'VENDAS NOVAS' ] ] },
        'Monte da Moinhola' => {
          '2965' => [
            [ '431', '431', 'LANDEIRA' ] ] }
           },
      'Redondo' => {
        'Foros da Fonte Seca' => {
          '7170' => [
            [ '102', '126', 'REDONDO' ] ] },
        'Redondo' => {
          '7170' => [
            [ '001', '088', 'REDONDO' ] ] },
        'Aldeias de Montoito' => {
          '7200' => [
            [ '051', '053', 'MONTOITO' ] ] }
            },
      'Reguengos de Monsaraz' => {
        'Caridade' => {
          '7200' => [
            [ '220', '231', 'REGUENGOS DE MONSARAZ' ] ] },
        'Barrada' => {
          '7200' => [
            [ '171', '182', 'MONSARAZ' ] ] },
        'So Pedro' => {
          '7200' => [
            [ '101', '150', 'CORVAL' ] ] },
        'Monte da Defezinha' => {
          '7200' => [
            [ '569', '569', 'CAMPINHO' ] ] },
        'Campinho' => {
          '7200' => [
            [ '501', '537', 'CAMPINHO' ] ] },
        'Perolivas' => {
          '7200' => [
            [ '450', '478', 'REGUENGOS DE MONSARAZ' ] ] },
        'Monte da Alerta' => {
          '7200' => [
            [ '185', '185', 'MONSARAZ' ] ] },
        'Horta do Paias' => {
          '7200' => [
            [ '071', '073', 'CAMPO RMZ' ] ] },
        'Monte da Azeimota' => {
          '7200' => [
            [ '081', '085', 'CORVAL' ] ] },
        'Reguengos de Monsaraz' => {
          '7200' => [
            [ '250', '413', 'REGUENGOS DE MONSARAZ' ] ] },
        'Monte da Ribeira' => {
          '7200' => [
            [ '201', '208', 'REGUENGOS DE MONSARAZ' ] ] }
              }
         },
    'Ilha das Flores' => {
      'Lajes das Flores' => {
        'Terra Ch' => {
          '9960' => [
            [ '580', '580', 'LOMBA LGF' ] ] },
        'Cima da Fonte' => {
          '9960' => [
            [ '310', '310', 'LAJEDO' ] ] },
        'Rocas' => {
          '9960' => [
            [ '140', '140', 'FAJZINHA' ] ] },
        'Tabuleiro' => {
          '9960' => [
            [ '250', '250', 'FAZENDA' ] ] },
        'Fazenda' => {
          '9960' => [
            [ '220', '220', 'FAZENDA' ] ] },
        'Barreira Vermelha' => {
          '9960' => [
            [ '200', '200', 'FAZENDA' ] ] },
        'Campanrio' => {
          '9960' => [
            [ '300', '300', 'LAJEDO' ] ] },
        'Cabo Baixo das Casas' => {
          '9960' => [
            [ '600', '600', 'MOSTEIRO LGF' ] ] },
        'Cruz' => {
          '9960' => [
            [ '410', '410', 'LAJES DAS FLORES' ],
            [ '500', '500', 'LOMBA LGF' ] ] },
        'Ramal da Cruz' => {
          '9960' => [
            [ '560', '560', 'LOMBA LGF' ] ] },
        'Fornalha' => {
          '9960' => [
            [ '340', '340', 'LAJEDO' ] ] },
        'Rossio' => {
          '9960' => [
            [ '150', '150', 'FAJZINHA' ] ] },
        'Entre Cancelas' => {
          '9960' => [
            [ '420', '420', 'LAJES DAS FLORES' ] ] },
        'Ponta da Faja Grande' => {
          '9960' => [
            [ '060', '060', 'FAJ GRANDE' ] ] },
        'Ribeira Seca' => {
          '9960' => [
            [ '484', '484', 'LAJES DAS FLORES' ] ] },
        'Marujo' => {
          '9960' => [
            [ '530', '530', 'LOMBA LGF' ] ] },
        'Ladeira' => {
          '9960' => [
            [ '120', '120', 'FAJZINHA' ] ] },
        'Grota' => {
          '9960' => [
            [ '510', '510', 'LOMBA LGF' ] ] },
        'Eiras' => {
          '9960' => [
            [ '210', '210', 'FAZENDA' ] ] },
        'Costa' => {
          '9960' => [
            [ '330', '330', 'LAJEDO' ] ] },
        'Ribeirinha' => {
          '9960' => [
            [ '230', '230', 'FAZENDA' ],
            [ '570', '570', 'LOMBA LGF' ] ] },
        'Fontinha' => {
          '9960' => [
            [ '040', '040', 'FAJ GRANDE' ] ] },
        'Caldeira' => {
          '9960' => [
            [ '620', '620', 'MOSTEIRO LGF' ] ] },
        'Lajes das Flores' => {
          '9960' => [
            [ '430', '446', 'LAJES DAS FLORES' ] ] },
        'Mosteiro' => {
          '9960' => [
            [ '630', '630', 'MOSTEIRO LGF' ] ] },
        'Quada' => {
          '9960' => [
            [ '070', '070', 'FAJ GRANDE' ] ] },
        'Courelas' => {
          '9960' => [
            [ '020', '020', 'FAJ GRANDE' ] ] },
        'Espelhao' => {
          '9960' => [
            [ '100', '100', 'FAJZINHA' ] ] },
        'Achada' => {
          '9960' => [
            [ '400', '400', 'LAJES DAS FLORES' ] ] },
        'Monte de Cima' => {
          '9960' => [
            [ '474', '474', 'LAJES DAS FLORES' ] ] },
        'Passagem' => {
          '9960' => [
            [ '482', '482', 'LAJES DAS FLORES' ] ] },
        'Quarteiro' => {
          '9960' => [
            [ '550', '550', 'LOMBA LGF' ] ] },
        'Lomba' => {
          '9960' => [
            [ '520', '520', 'LOMBA LGF' ] ] },
        'Faj Grande' => {
          '9960' => [
            [ '030', '030', 'FAJ GRANDE' ] ] },
        'Fajzinha' => {
          '9960' => [
            [ '110', '110', 'FAJZINHA' ] ] },
        'Outeiro' => {
          '9960' => [
            [ '050', '050', 'FAJ GRANDE' ],
            [ '130', '130', 'FAJZINHA' ],
            [ '480', '480', 'LAJES DAS FLORES' ],
            [ '540', '540', 'LOMBA LGF' ] ] },
        'Lajedo' => {
          '9960' => [
            [ '360', '360', 'LAJEDO' ] ] },
        'Lamaceiro' => {
          '9960' => [
            [ '470', '470', 'LAJES DAS FLORES' ] ] },
        'Morres de Baixo' => {
          '9960' => [
            [ '476', '476', 'LAJES DAS FLORES' ] ] },
        'Combastio' => {
          '9960' => [
            [ '320', '320', 'LAJEDO' ] ] },
        'Assumada' => {
          '9960' => [
            [ '010', '010', 'FAJ GRANDE' ] ] },
        'Ribeirinha de Baixo' => {
          '9960' => [
            [ '240', '240', 'FAZENDA' ] ] },
        'Morres de Cima' => {
          '9960' => [
            [ '478', '478', 'LAJES DAS FLORES' ] ] },
        'Cabo Cima das Casas' => {
          '9960' => [
            [ '610', '610', 'MOSTEIRO LGF' ] ] },
        'Lombo' => {
          '9960' => [
            [ '472', '472', 'LAJES DAS FLORES' ] ] },
        'Groto' => {
          '9960' => [
            [ '350', '350', 'LAJEDO' ] ] }
                   },
      'Santa Cruz das Flores' => {
        'Zona Industrial' => {
          '9970' => [
            [ '390', '390', 'SANTA CRUZ DAS FLORES' ] ] },
        'Ribeira dos Barqueiros' => {
          '9970' => [
            [ '275', '275', 'SANTA CRUZ DAS FLORES' ] ] },
        'Ponta Delgada' => {
          '9970' => [
            [ '070', '097', 'PONTA DELGADA SCF' ] ] },
        'Santa Cruz das Flores' => {
          '9970' => [
            [ '301', '342', 'SANTA CRUZ DAS FLORES' ] ] },
        'Rua do Santo Espirito Santo' => {
          '9970' => [
            [ '285', '285', 'SANTA CRUZ DAS FLORES' ] ] },
        'Cedros' => {
          '9970' => [
            [ '031', '033', 'CEDROS SCF' ] ] },
        'Fazenda de Santa Cruz' => {
          '9970' => [
            [ '240', '251', 'SANTA CRUZ DAS FLORES' ] ] },
        'Monte' => {
          '9970' => [
            [ '265', '265', 'SANTA CRUZ DAS FLORES' ] ] },
        'Moinho' => {
          '9970' => [
            [ '260', '260', 'SANTA CRUZ DAS FLORES' ] ] },
        'Boavista' => {
          '9970' => [
            [ '210', '210', 'SANTA CRUZ DAS FLORES' ],
            [ '215', '215', 'SANTA CRUZ DAS FLORES' ] ] },
        'Cais de So Pedro' => {
          '9970' => [
            [ '220', '220', 'SANTA CRUZ DAS FLORES' ] ] },
        'Ribeira do Pomar' => {
          '9970' => [
            [ '270', '270', 'SANTA CRUZ DAS FLORES' ] ] },
        'Casas do Concelho' => {
          '9970' => [
            [ '061', '064', 'PONTA DELGADA SCF' ] ] },
        'Cais do Boqueiro' => {
          '9970' => [
            [ '225', '225', 'SANTA CRUZ DAS FLORES' ] ] },
        'Alto da Rua' => {
          '9970' => [
            [ '205', '205', 'SANTA CRUZ DAS FLORES' ] ] },
        'Caveira' => {
          '9970' => [
            [ '010', '010', 'CAVEIRA' ] ] },
        'Vales' => {
          '9970' => [
            [ '380', '380', 'SANTA CRUZ DAS FLORES' ] ] },
        'Casas da Ponte' => {
          '9970' => [
            [ '230', '230', 'SANTA CRUZ DAS FLORES' ] ] },
        'Ribeirinha do Monte' => {
          '9970' => [
            [ '280', '280', 'SANTA CRUZ DAS FLORES' ] ] }
                  }
             },
    'Leiria' => {
      'Leiria' => {
        'Marinha' => {
          '2425' => [
            [ '480', '527', 'SOUTO DA CARPALHOSA' ] ] },
        'Souto de Baixo' => {
          '2420' => [
            [ '426', '426', 'CARANGUEJEIRA' ] ] },
        'Barradas' => {
          '2400' => [
            [ '762', '801', 'AMOR' ] ] },
        'Cortes' => {
          '2410' => [
            [ '501', '534', 'LEIRIA' ] ] },
        'Alcogulhe de Baixo' => {
          '2400' => [
            [ '821', '828', 'LEIRIA' ] ] },
        'Leiria' => {
          '2400' => [
            [ '013', '016', 'LEIRIA' ],
            [ '076', '077', 'LEIRIA' ],
            [ '079', '084', 'LEIRIA' ],
            [ '086', '089', 'LEIRIA' ],
            [ '093', '099', 'LEIRIA' ],
            [ '101', '115', 'LEIRIA' ],
            [ '117', '130', 'LEIRIA' ],
            [ '132', '132', 'LEIRIA' ],
            [ '134', '144', 'LEIRIA' ],
            [ '146', '161', 'LEIRIA' ],
            [ '164', '166', 'LEIRIA' ],
            [ '168', '168', 'LEIRIA' ],
            [ '170', '181', 'LEIRIA' ],
            [ '186', '188', 'LEIRIA' ],
            [ '190', '197', 'LEIRIA' ],
            [ '199', '211', 'LEIRIA' ],
            [ '213', '233', 'LEIRIA' ],
            [ '235', '241', 'LEIRIA' ],
            [ '250', '252', 'LEIRIA' ],
            [ '254', '254', 'LEIRIA' ],
            [ '256', '263', 'LEIRIA' ],
            [ '265', '265', 'LEIRIA' ],
            [ '268', '268', 'LEIRIA' ],
            [ '473', '473', 'LEIRIA' ],
            [ '489', '489', 'LEIRIA' ],
            [ '504', '520', 'LEIRIA' ],
            [ '522', '532', 'LEIRIA' ],
            [ '560', '560', 'LEIRIA' ],
            [ '702', '703', 'LEIRIA' ],
            [ '706', '713', 'LEIRIA' ] ],
          '2414' => [
            [ '001', '019', 'LEIRIA' ],
            [ '022', '023', 'LEIRIA' ] ],
          '2415' => [
            [ '301', '605', 'LEIRIA' ],
            [ '607', '614', 'LEIRIA' ],
            [ '618', '793', 'LEIRIA' ] ],
          '2404' => [
            [ '001', '001', 'LEIRIA' ] ],
          '2403' => [
            [ '002', '007', 'LEIRIA' ] ],
          '2410' => [
            [ '031', '049', 'LEIRIA' ],
            [ '051', '078', 'LEIRIA' ],
            [ '081', '091', 'LEIRIA' ],
            [ '093', '132', 'LEIRIA' ],
            [ '134', '140', 'LEIRIA' ],
            [ '143', '147', 'LEIRIA' ],
            [ '149', '149', 'LEIRIA' ],
            [ '152', '152', 'LEIRIA' ],
            [ '158', '168', 'LEIRIA' ],
            [ '170', '184', 'LEIRIA' ],
            [ '186', '273', 'LEIRIA' ],
            [ '275', '364', 'LEIRIA' ],
            [ '366', '392', 'LEIRIA' ],
            [ '394', '469', 'LEIRIA' ],
            [ '471', '478', 'LEIRIA' ] ],
          '2419' => [
            [ '001', '004', 'LEIRIA' ],
            [ '009', '011', 'LEIRIA' ],
            [ '013', '013', 'LEIRIA' ] ] },
        'Coues' => {
          '2420' => [
            [ '226', '243', 'MEMRIA' ] ] },
        'Ponte da Pedra' => {
          '2419' => [
            [ '005', '008', 'REGUEIRA DE PONTES' ] ] },
        'Vale Juncal' => {
          '2400' => [
            [ '497', '497', 'LEIRIA' ] ] },
        'Boco' => {
          '2425' => [
            [ '327', '328', 'CARVIDE' ],
            [ '404', '450', 'CARVIDE' ] ] },
        'Bidoeira Cima' => {
          '2416' => [
            [ '012', '012', 'BIDOEIRA DE CIMA' ] ] },
        'Caldelas' => {
          '2420' => [
            [ '051', '168', 'CARANGUEJEIRA' ] ] },
        'Parceiros' => {
          '2400' => [
            [ '441', '442', 'LEIRIA' ] ] },
        'Gndara de Aqum' => {
          '2425' => [
            [ '459', '461', 'CARVIDE' ] ] },
        'Meia Lgua' => {
          '2400' => [
            [ '430', '432', 'LEIRIA' ] ] },
        'Monte Agudo' => {
          '2425' => [
            [ '724', '750', 'ORTIGOSA' ] ] },
        'Curvachia' => {
          '2410' => [
            [ '848', '858', 'LEIRIA' ] ] },
        'Abadia' => {
          '2410' => [
            [ '841', '846', 'LEIRIA' ] ] },
        'Caxieira' => {
          '2420' => [
            [ '427', '434', 'LEIRIA' ] ] },
        'Andreus' => {
          '2410' => [
            [ '001', '029', 'LEIRIA' ] ] },
        'Barreiria' => {
          '2495' => [
            [ '101', '110', 'SANTA CATARINA DA SERRA' ] ] },
        'Machados' => {
          '2420' => [
            [ '415', '418', 'LEIRIA' ] ] },
        'Arrabal' => {
          '2420' => [
            [ '001', '050', 'ARRABAL' ] ] },
        'Mangas' => {
          '2405' => [
            [ '028', '036', 'MACEIRA LRA' ] ] },
        'Moures' => {
          '2420' => [
            [ '425', '425', 'ARRABAL' ] ] },
        'Covinhas' => {
          '2415' => [
            [ '795', '803', 'LEIRIA' ] ] },
        'Picheleiro' => {
          '2400' => [
            [ '802', '802', 'LEIRIA' ] ] },
        'Casal do Ralha' => {
          '2400' => [
            [ '018', '020', 'LEIRIA' ] ] },
        'Agodim' => {
          '2424' => [
            [ '002', '004', 'COLMEIAS' ] ],
          '2420' => [
            [ '169', '225', 'COLMEIAS' ] ] },
        'Bidoeira Baixo' => {
          '2415' => [
            [ '001', '010', 'BIDOEIRA DE CIMA' ] ] },
        'Aroeira' => {
          '2425' => [
            [ '601', '625', 'MONTE REDONDO LRA' ] ] },
        'Vale da Bajouca' => {
          '2425' => [
            [ '205', '212', 'BAJOUCA' ] ] },
        'Chs' => {
          '2415' => [
            [ '130', '213', 'REGUEIRA DE PONTES' ] ] },
        'Ch Laranjeira' => {
          '2425' => [
            [ '816', '900', 'SOUTO DA CARPALHOSA' ] ] },
        'Vale Frade' => {
          '2400' => [
            [ '493', '493', 'LEIRIA' ] ] },
        'Barreiros' => {
          '2403' => [
            [ '008', '009', 'AMOR' ] ] },
        'Loureira' => {
          '2495' => [
            [ '121', '171', 'SANTA CATARINA DA SERRA' ] ] },
        'Vale Arieira' => {
          '2400' => [
            [ '491', '491', 'LEIRIA' ] ] },
        'Riba D\'Aves' => {
          '2425' => [
            [ '674', '722', 'ORTIGOSA' ],
            [ '752', '800', 'ORTIGOSA' ] ] },
        'Soutocico' => {
          '2424' => [
            [ '001', '001', 'ARRABAL' ] ] },
        'Junqueiros' => {
          '2425' => [
            [ '355', '362', 'CARVIDE' ] ] },
        'Arroteia' => {
          '2425' => [
            [ '801', '814', 'SOUTO DA CARPALHOSA' ] ] },
        'Casal dos Secos' => {
          '2425' => [
            [ '451', '458', 'COIMBRO' ] ] },
        'Amor' => {
          '2400' => [
            [ '760', '760', 'AMOR' ] ] },
        'Alcaidaria' => {
          '2415' => [
            [ '011', '024', 'LEIRIA' ] ] },
        'Magueigia' => {
          '2495' => [
            [ '181', '194', 'SANTA CATARINA DA SERRA' ] ] },
        'Moinhos de Carvide' => {
          '2425' => [
            [ '288', '325', 'CARVIDE' ],
            [ '364', '374', 'CARVIDE' ],
            [ '376', '402', 'CARVIDE' ] ] },
        'Casal de Baixo' => {
          '2425' => [
            [ '350', '353', 'CARVIDE' ] ] },
        'Carreira' => {
          '2425' => [
            [ '251', '284', 'CARREIRA LRA' ] ] },
        'Salgueiros' => {
          '2425' => [
            [ '201', '203', 'BAJOUCA' ] ] },
        'Ponte Cabreira' => {
          '2400' => [
            [ '446', '448', 'LEIRIA' ] ] },
        'Carvide' => {
          '2425' => [
            [ '343', '346', 'CARVIDE' ] ] },
        'Andrezes' => {
          '2425' => [
            [ '161', '185', 'BAJOUCA' ] ] },
        'Monte Real' => {
          '2425' => [
            [ '035', '119', 'MONTE REAL' ] ] },
        'Brejo' => {
          '2425' => [
            [ '001', '033', 'MONTE REAL' ] ] },
        'Aparios' => {
          '2420' => [
            [ '244', '411', 'LEIRIA' ] ] },
        'Carpalho' => {
          '2425' => [
            [ '330', '341', 'CARVIDE' ] ] },
        'Amieira' => {
          '2415' => [
            [ '025', '030', 'REGUEIRA DE PONTES' ] ] },
        'A-do-Barbas' => {
          '2405' => [
            [ '001', '026', 'MACEIRA LRA' ] ] },
        'Chana' => {
          '2495' => [
            [ '201', '250', 'CHAINA' ] ] },
        'Gaspara' => {
          '2425' => [
            [ '187', '198', 'BAJOUCA' ] ] },
        'Brasil' => {
          '2420' => [
            [ '419', '424', 'COLMEIAS' ] ] }
            },
      'Alvaizere' => {
        'Loureira' => {
          '3250' => [
            [ '034', '034', 'PUSSOS' ] ] },
        'Aldeia da Serra' => {
          '3250' => [
            [ '310', '336', 'PELM' ] ] },
        'Cabeas' => {
          '3260' => [
            [ '890', '890', 'MAS DE DONA MARIA' ] ] },
        'Alqueido de Pussos' => {
          '3250' => [
            [ '365', '395', 'PUSSOS' ] ] },
        'Vila Mas Dona Maria' => {
          '3250' => [
            [ '249', '293', 'MAS DE DONA MARIA' ] ] },
        'Cabaos' => {
          '3250' => [
            [ '350', '361', 'PUSSOS' ] ] },
        'Venda do Henrique' => {
          '3260' => [
            [ '880', '880', 'PUSSOS' ] ] },
        'Alvaizere' => {
          '3250' => [
            [ '100', '119', 'ALVAIZERE' ] ] },
        'Aldeia Nova' => {
          '3250' => [
            [ '020', '033', 'ALMOSTER AVZ' ] ] },
        'Alto do Passal' => {
          '3250' => [
            [ '400', '430', 'REGO DA MURTA' ] ] },
        'Macieira' => {
          '3250' => [
            [ '035', '043', 'ALMOSTER AVZ' ] ] },
        'Vinha Grande' => {
          '3250' => [
            [ '295', '296', 'MAS DE DONA MARIA' ] ] },
        'Almeida' => {
          '3250' => [
            [ '140', '173', 'ALVAIZERE' ] ] },
        'Amarela' => {
          '3250' => [
            [ '200', '216', 'MAS DE CAMINHO' ] ] }
          },
      'Pombal' => {
        'Alqueido' => {
          '3100' => [
            [ '631', '639', 'SANTIAGO DE LITM' ] ] },
        'gua Formosa' => {
          '3105' => [
            [ '104', '147', 'ILHA PBL' ] ] },
        'Baltaria' => {
          '3100' => [
            [ '803', '809', 'VILA C' ] ] },
        'Casal da Rosa' => {
          '3100' => [
            [ '009', '010', 'SANTIAGO DE LITM' ] ] },
        'Cadavais' => {
          '3100' => [
            [ '732', '734', 'SO SIMO DE LITM' ] ] },
        'Casal das Freiras' => {
          '3100' => [
            [ '645', '649', 'SANTIAGO DE LITM' ] ] },
        'Agudas' => {
          '3105' => [
            [ '299', '350', 'REDINHA' ] ] },
        'Mata de Ruge gua' => {
          '3100' => [
            [ '898', '898', 'ALBERGARIA DOS DOZE' ] ] },
        'Cruta' => {
          '3100' => [
            [ '650', '650', 'POMBAL' ] ] },
        'Estrada' => {
          '3100' => [
            [ '338', '362', 'POMBAL' ] ] },
        'Antes' => {
          '3105' => [
            [ '148', '175', 'LOURIAL' ],
            [ '176', '217', 'MATA MOURISCA' ] ] },
        'Adugete' => {
          '3105' => [
            [ '278', '298', 'PELARIGA' ] ] },
        'Barros da Paz' => {
          '3105' => [
            [ '001', '024', 'ALMAGREIRA PBL' ] ] },
        'Aroeiras' => {
          '3100' => [
            [ '801', '801', 'VILA C' ] ] },
        'Aldeia de Baixo' => {
          '3100' => [
            [ '701', '730', 'SO SIMO DE LITM' ] ] },
        'Souto' => {
          '3100' => [
            [ '370', '394', 'POMBAL' ] ] },
        'Souro' => {
          '3100' => [
            [ '690', '692', 'SANTIAGO DE LITM' ] ] },
        'Espinheiras' => {
          '3105' => [
            [ '451', '453', 'ALMAGREIRA PBL' ] ] },
        'Fontinha' => {
          '3100' => [
            [ '811', '837', 'VILA C' ] ] },
        'Meirinhas' => {
          '3105' => [
            [ '238', '277', 'MEIRINHAS' ],
            [ '450', '450', 'MEIRINHAS' ] ] },
        'Murzeleira' => {
          '3100' => [
            [ '092', '098', 'ALBERGARIA DOS DOZE' ] ] },
        'Abilheira' => {
          '3100' => [
            [ '011', '075', 'ABIUL' ] ] },
        'Farroubal' => {
          '3100' => [
            [ '651', '688', 'SANTIAGO DE LITM' ] ] },
        'Arroteia' => {
          '3100' => [
            [ '929', '929', 'VILA C' ] ] },
        'Arieiro' => {
          '3105' => [
            [ '218', '236', 'MEIRINHAS' ] ] },
        'Pombal' => {
          '3100' => [
            [ '400', '571', 'POMBAL' ] ] },
        'Moinho das Freiras' => {
          '3100' => [
            [ '851', '851', 'SANTIAGO DE LITM' ] ] },
        'Roussa' => {
          '3100' => [
            [ '364', '368', 'POMBAL' ] ] },
        'Casal da Mouca' => {
          '3100' => [
            [ '643', '643', 'SANTIAGO DE LITM' ] ] },
        'Afonsos' => {
          '3100' => [
            [ '311', '336', 'POMBAL' ] ] },
        'Ch' => {
          '3105' => [
            [ '351', '449', 'VERMOIL' ] ] },
        'Alto da Granja' => {
          '3100' => [
            [ '899', '927', 'POMBAL' ] ] },
        'Canaria' => {
          '3100' => [
            [ '641', '641', 'SANTIAGO DE LITM' ] ] },
        'Albergaria dos Doze' => {
          '3100' => [
            [ '081', '090', 'ALBERGARIA DOS DOZE' ] ] },
        'Lagoeiros' => {
          '3105' => [
            [ '070', '103', 'GUIA PBL' ] ] },
        'Alhais' => {
          '3105' => [
            [ '053', '069', 'CARRIO' ] ] },
        'Cubo' => {
          '3100' => [
            [ '695', '700', 'SANTIAGO DE LITM' ] ] },
        'Alto dos Mendes' => {
          '3105' => [
            [ '025', '052', 'CARNIDE' ] ] },
        'Carvalheiras' => {
          '3105' => [
            [ '454', '454', 'ILHA PBL' ] ] }
            },
      'Porto de Ms' => {
        'Mira de Aire' => {
          '2485' => [
            [ '001', '009', 'MIRA DE AIRE' ],
            [ '011', '024', 'MIRA DE AIRE' ],
            [ '031', '034', 'MIRA DE AIRE' ],
            [ '040', '041', 'MIRA DE AIRE' ],
            [ '043', '044', 'MIRA DE AIRE' ],
            [ '050', '072', 'MIRA DE AIRE' ],
            [ '075', '079', 'MIRA DE AIRE' ],
            [ '083', '084', 'MIRA DE AIRE' ],
            [ '086', '088', 'MIRA DE AIRE' ],
            [ '092', '093', 'MIRA DE AIRE' ],
            [ '095', '097', 'MIRA DE AIRE' ],
            [ '099', '107', 'MIRA DE AIRE' ],
            [ '109', '111', 'MIRA DE AIRE' ],
            [ '113', '119', 'MIRA DE AIRE' ],
            [ '121', '139', 'MIRA DE AIRE' ],
            [ '141', '143', 'MIRA DE AIRE' ],
            [ '145', '163', 'MIRA DE AIRE' ],
            [ '165', '184', 'MIRA DE AIRE' ] ] },
        'Marinha de Baixo' => {
          '2480' => [
            [ '213', '214', 'SERRO VENTOSO' ] ] },
        'Alcaria' => {
          '2480' => [
            [ '011', '012', 'ALCARIA PMS' ] ] },
        'Alto do Pessegueiro' => {
          '2480' => [
            [ '091', '101', 'PEDREIRAS' ] ] },
        'Porto de Ms' => {
          '2480' => [
            [ '006', '008', 'PORTO DE MS' ],
            [ '300', '347', 'PORTO DE MS' ],
            [ '851', '851', 'PORTO DE MS' ],
            [ '853', '853', 'PORTO DE MS' ] ] },
        'Cruz da Lgua' => {
          '2480' => [
            [ '854', '854', 'PEDREIRAS' ] ] },
        'Alqueido da Serra' => {
          '2480' => [
            [ '013', '013', 'ALQUEIDO DA SERRA' ] ] },
        'Juncal' => {
          '2480' => [
            [ '065', '078', 'JUNCAL' ],
            [ '348', '388', 'JUNCAL' ],
            [ '852', '852', 'JUNCAL' ] ] },
        'Portela Vale Espinho' => {
          '2480' => [
            [ '216', '216', 'ARRIMAL' ] ] },
        'Esparrela' => {
          '2480' => [
            [ '102', '102', 'PORTO DE MS' ] ] },
        'Mendiga' => {
          '2480' => [
            [ '215', '215', 'MENDIGA' ] ] },
        'Bouceiros' => {
          '2480' => [
            [ '021', '027', 'ALQUEIDO DA SERRA' ] ] },
        'Marinha da Mendiga' => {
          '2480' => [
            [ '212', '212', 'MENDIGA' ] ] },
        'Barreira Junqueira' => {
          '2480' => [
            [ '121', '123', 'SO BENTO PMS' ] ] },
        'Corredoura' => {
          '2480' => [
            [ '855', '860', 'PORTO DE MS' ] ] },
        'Andainho' => {
          '2480' => [
            [ '052', '052', 'JUNCAL' ] ] },
        'Alto de Alvados' => {
          '2480' => [
            [ '031', '034', 'ALVADOS' ] ] },
        'Covo da Carvalha' => {
          '2485' => [
            [ '200', '237', 'MIRA DE AIRE' ] ] },
        'Casal Santo Antnio' => {
          '2480' => [
            [ '125', '143', 'SO BENTO PMS' ] ] },
        'Azoio Dinez' => {
          '2480' => [
            [ '009', '010', 'PEDREIRAS' ] ] },
        'Feteira' => {
          '2480' => [
            [ '103', '114', 'PEDREIRAS' ] ] },
        'Picamilho' => {
          '2480' => [
            [ '080', '090', 'JUNCAL' ] ] },
        'Bom Sucesso' => {
          '2480' => [
            [ '180', '189', 'PORTO DE MS' ] ] },
        'Casais dos Vales' => {
          '2480' => [
            [ '124', '124', 'ALQUEIDO DA SERRA' ] ] },
        'Cabea Veada' => {
          '2480' => [
            [ '203', '203', 'MENDIGA' ] ] },
        'Cabeceiras' => {
          '2480' => [
            [ '054', '063', 'CALVARIA DE CIMA' ] ] },
        'Serro Ventoso' => {
          '2480' => [
            [ '217', '218', 'SERRO VENTOSO' ] ] },
        'Alqueido do Arrimal' => {
          '2480' => [
            [ '041', '045', 'ARRIMAL' ] ] },
        'Bemposta' => {
          '2480' => [
            [ '201', '201', 'MENDIGA' ] ] },
        'Alto da Vala' => {
          '2480' => [
            [ '151', '173', 'PORTO DE MS' ] ] },
        'Bezerra' => {
          '2480' => [
            [ '202', '202', 'SERRO VENTOSO' ] ] },
        'Amarela' => {
          '2480' => [
            [ '051', '051', 'CALVARIA DE CIMA' ] ] },
        'Casais do Cho' => {
          '2480' => [
            [ '204', '211', 'SERRO VENTOSO' ] ] }
            },
      'Pedrgo Grande' => {
        'Alto Louriceira' => {
          '3270' => [
            [ '053', '066', 'PEDRGO GRANDE' ] ] },
        'Pedrgo Grande' => {
          '3270' => [
            [ '090', '163', 'PEDRGO GRANDE' ] ] },
        'Alagoa' => {
          '3270' => [
            [ '052', '052', 'VILA FACAIA' ] ] },
        'Agria' => {
          '3270' => [
            [ '051', '051', 'PEDRGO GRANDE' ] ] },
        'Aldeia das Freiras' => {
          '3270' => [
            [ '201', '225', 'VILA FACAIA' ] ] },
        'Fontainhas' => {
          '3270' => [
            [ '068', '079', 'PEDRGO GRANDE' ] ] },
        'Adega' => {
          '3270' => [
            [ '011', '031', 'GRAA' ] ] }
               },
      'Nazar' => {
        'Nazar' => {
          '2450' => [
            [ '100', '127', 'NAZAR' ],
            [ '129', '257', 'NAZAR' ],
            [ '259', '285', 'NAZAR' ] ] },
        'Valado dos Frades' => {
          '2450' => [
            [ '301', '396', 'VALADO DOS FRADES' ] ] },
        'Stio da Nazar' => {
          '2450' => [
            [ '065', '065', 'NAZAR' ] ] },
        'Pederneira' => {
          '2450' => [
            [ '060', '060', 'NAZAR' ] ] },
        'Fanhais' => {
          '2450' => [
            [ '051', '051', 'NAZAR' ] ] },
        'Casais Baixo' => {
          '2450' => [
            [ '011', '027', 'FAMALICO NZR' ] ] },
        'Bairro do Salvador' => {
          '2450' => [
            [ '071', '077', 'NAZAR' ] ] }
            },
        'bidos' => {
        'bidos' => {
          '2510' => [
            [ '001', '001', 'BIDOS' ],
            [ '034', '034', 'BIDOS' ],
            [ '036', '037', 'BIDOS' ],
            [ '039', '039', 'BIDOS' ],
            [ '042', '043', 'BIDOS' ],
            [ '045', '055', 'BIDOS' ],
            [ '057', '057', 'BIDOS' ],
            [ '060', '060', 'BIDOS' ],
            [ '062', '065', 'BIDOS' ],
            [ '068', '071', 'BIDOS' ],
            [ '073', '084', 'BIDOS' ],
            [ '086', '086', 'BIDOS' ],
            [ '089', '089', 'BIDOS' ],
            [ '094', '096', 'BIDOS' ],
            [ '099', '104', 'BIDOS' ],
            [ '106', '106', 'BIDOS' ],
            [ '113', '113', 'BIDOS' ],
            [ '115', '115', 'BIDOS' ],
            [ '118', '118', 'BIDOS' ],
            [ '121', '126', 'BIDOS' ],
            [ '128', '128', 'BIDOS' ],
            [ '130', '130', 'BIDOS' ],
            [ '132', '132', 'BIDOS' ],
            [ '134', '135', 'BIDOS' ],
            [ '137', '141', 'BIDOS' ],
            [ '143', '147', 'BIDOS' ],
            [ '170', '171', 'BIDOS' ] ] },
        'Casais da Capeleira' => {
          '2510' => [
            [ '018', '018', 'BIDOS' ] ] },
        'Gaeiras' => {
          '2510' => [
            [ '700', '702', 'GAEIRAS' ],
            [ '705', '725', 'GAEIRAS' ],
            [ '727', '751', 'GAEIRAS' ],
            [ '761', '761', 'GAEIRAS' ] ] },
        'Casais Brancos' => {
          '2510' => [
            [ '212', '212', 'BIDOS' ] ] },
        'Arelho' => {
          '2510' => [
            [ '191', '194', 'BIDOS' ] ] },
        'Quinta de Santana' => {
          '2510' => [
            [ '771', '771', 'GAEIRAS' ] ] },
        'Casais da Navalha' => {
          '2510' => [
            [ '020', '020', 'BIDOS' ] ] },
        'Aldeia dos Pescadores' => {
          '2510' => [
            [ '661', '665', 'VAU' ] ] },
        'Olho Marinho' => {
          '2510' => [
            [ '511', '519', 'OLHO MARINHO' ],
            [ '521', '526', 'OLHO MARINHO' ],
            [ '528', '529', 'OLHO MARINHO' ],
            [ '531', '546', 'OLHO MARINHO' ],
            [ '548', '558', 'OLHO MARINHO' ],
            [ '560', '566', 'OLHO MARINHO' ],
            [ '568', '575', 'OLHO MARINHO' ] ] },
        'Casais da Arruda' => {
          '2510' => [
            [ '501', '502', 'OLHO MARINHO' ] ] },
        'A-da-Gorda' => {
          '2510' => [
            [ '011', '016', 'BIDOS' ] ] },
        'Usseira' => {
          '2510' => [
            [ '772', '772', 'USSEIRA' ] ] },
        'Vale Benfeito' => {
          '2510' => [
            [ '472', '472', 'AMOREIRA OBD' ] ] },
        'Amoreira' => {
          '2510' => [
            [ '401', '420', 'AMOREIRA OBD' ],
            [ '422', '425', 'AMOREIRA OBD' ],
            [ '427', '427', 'AMOREIRA OBD' ],
            [ '429', '430', 'AMOREIRA OBD' ],
            [ '432', '432', 'AMOREIRA OBD' ],
            [ '434', '434', 'AMOREIRA OBD' ],
            [ '436', '446', 'AMOREIRA OBD' ],
            [ '448', '452', 'AMOREIRA OBD' ],
            [ '454', '456', 'AMOREIRA OBD' ],
            [ '458', '459', 'AMOREIRA OBD' ] ] },
        'Sobral da Lagoa' => {
          '2510' => [
            [ '651', '651', 'SOBRAL DE LAGOA' ] ] },
        'A dos Negros' => {
          '2510' => [
            [ '321', '343', 'A DOS NEGROS' ] ] },
        'Perna de Pau' => {
          '2510' => [
            [ '581', '582', 'OLHO MARINHO' ] ] },
        'Casal da Avarela' => {
          '2510' => [
            [ '214', '217', 'BIDOS' ] ] }
            },
      'Castanheira de Pra' => {
        'Valongo' => {
          '3280' => [
            [ '006', '053', 'CASTANHEIRA DE PRA' ] ] },
        'Sarzedas de So Pedro' => {
          '3280' => [
            [ '100', '124', 'CASTANHEIRA DE PRA' ] ] },
        'Carrial' => {
          '3280' => [
            [ '200', '201', 'COENTRAL' ] ] }
             },
      'Bombarral' => {
        'Columbeira' => {
          '2540' => [
            [ '588', '590', 'ROLIA' ],
            [ '592', '610', 'ROLIA' ],
            [ '612', '615', 'ROLIA' ],
            [ '663', '665', 'ROLIA' ] ] },
        'Quinta do Sanguinhal' => {
          '2540' => [
            [ '216', '218', 'BOMBARRAL' ] ] },
        'Cantarola' => {
          '2540' => [
            [ '687', '694', 'VALE COVO' ] ] },
        'P' => {
          '2540' => [
            [ '479', '484', 'P' ] ] },
        'Casal do Frade' => {
          '2540' => [
            [ '409', '413', 'CARVALHAL BBR' ] ] },
        'Azambujeira dos Carros' => {
          '2540' => [
            [ '534', '562', 'ROLIA' ] ] },
        'Sobral do Parelho' => {
          '2540' => [
            [ '467', '467', 'CARVALHAL BBR' ] ] },
        'Vale Covo' => {
          '2540' => [
            [ '157', '157', 'VALE COVO' ],
            [ '165', '165', 'VALE COVO' ],
            [ '700', '713', 'VALE COVO' ],
            [ '715', '724', 'VALE COVO' ],
            [ '726', '751', 'VALE COVO' ] ] },
        'Rossio do Carvalhal' => {
          '2540' => [
            [ '422', '454', 'CARVALHAL BBR' ] ] },
        'Baraais' => {
          '2540' => [
            [ '564', '578', 'ROLIA' ],
            [ '580', '581', 'ROLIA' ] ] },
        'Moita Boa' => {
          '2540' => [
            [ '201', '202', 'BOMBARRAL' ] ] },
        'Rolia' => {
          '2540' => [
            [ '668', '668', 'ROLIA' ],
            [ '670', '678', 'ROLIA' ] ] },
        'Casal do Brejo' => {
          '2540' => [
            [ '167', '172', 'BOMBARRAL' ] ] },
        'Carvalhal' => {
          '2540' => [
            [ '326', '337', 'CARVALHAL BBR' ],
            [ '355', '369', 'CARVALHAL BBR' ],
            [ '371', '371', 'CARVALHAL BBR' ],
            [ '373', '374', 'CARVALHAL BBR' ],
            [ '376', '381', 'CARVALHAL BBR' ],
            [ '383', '384', 'CARVALHAL BBR' ] ] },
        'Casais de Tourinha' => {
          '2540' => [
            [ '387', '394', 'CARVALHAL BBR' ] ] },
        'Bombarral' => {
          '2540' => [
            [ '007', '007', 'BOMBARRAL' ],
            [ '009', '012', 'BOMBARRAL' ],
            [ '015', '021', 'BOMBARRAL' ],
            [ '023', '026', 'BOMBARRAL' ],
            [ '029', '033', 'BOMBARRAL' ],
            [ '035', '037', 'BOMBARRAL' ],
            [ '041', '041', 'BOMBARRAL' ],
            [ '044', '044', 'BOMBARRAL' ],
            [ '046', '046', 'BOMBARRAL' ],
            [ '049', '050', 'BOMBARRAL' ],
            [ '052', '065', 'BOMBARRAL' ],
            [ '067', '067', 'BOMBARRAL' ],
            [ '069', '071', 'BOMBARRAL' ],
            [ '073', '076', 'BOMBARRAL' ],
            [ '078', '085', 'BOMBARRAL' ],
            [ '087', '089', 'BOMBARRAL' ],
            [ '091', '098', 'BOMBARRAL' ],
            [ '100', '101', 'BOMBARRAL' ],
            [ '103', '107', 'BOMBARRAL' ],
            [ '109', '127', 'BOMBARRAL' ],
            [ '129', '133', 'BOMBARRAL' ],
            [ '135', '141', 'BOMBARRAL' ],
            [ '143', '143', 'BOMBARRAL' ],
            [ '145', '153', 'BOMBARRAL' ],
            [ '155', '156', 'BOMBARRAL' ],
            [ '159', '160', 'BOMBARRAL' ] ] },
        'Casal da Lavandeira' => {
          '2540' => [
            [ '396', '398', 'CARVALHAL BBR' ] ] },
        'Barreiras' => {
          '2540' => [
            [ '001', '005', 'BOMBARRAL' ] ] },
        'Casal Vale D\'Arcos' => {
          '2540' => [
            [ '415', '420', 'CARVALHAL BBR' ] ] },
        'Barrolobo' => {
          '2540' => [
            [ '348', '348', 'CARVALHAL BBR' ] ] },
        'Estorninho' => {
          '2540' => [
            [ '185', '185', 'BOMBARRAL' ] ] },
        'Bom Vento' => {
          '2540' => [
            [ '352', '352', 'CARVALHAL BBR' ] ] },
        'Camaro' => {
          '2540' => [
            [ '163', '164', 'BOMBARRAL' ] ] },
        'Delgada' => {
          '2540' => [
            [ '617', '621', 'ROLIA' ],
            [ '623', '637', 'ROLIA' ],
            [ '639', '641', 'ROLIA' ],
            [ '643', '643', 'ROLIA' ],
            [ '645', '661', 'ROLIA' ] ] },
        'Casal do Fialho' => {
          '2540' => [
            [ '584', '586', 'ROLIA' ] ] },
        'Fames' => {
          '2540' => [
            [ '189', '189', 'BOMBARRAL' ] ] },
        'Casal do Aniceto' => {
          '2540' => [
            [ '402', '404', 'CARVALHAL BBR' ] ] },
        'Casal da Lamarosa' => {
          '2540' => [
            [ '487', '533', 'P' ] ] },
        'Barrocalvo' => {
          '2540' => [
            [ '158', '158', 'CARVALHAL BBR' ],
            [ '342', '342', 'CARVALHAL BBR' ],
            [ '344', '345', 'CARVALHAL BBR' ] ] }
               },
      'Alcobaa' => {
        'Alqueido' => {
          '2460' => [
            [ '391', '443', 'COZ' ] ] },
        'Casalinho' => {
          '2460' => [
            [ '853', '853', 'COZ' ] ] },
        'Covo do Milho' => {
          '2460' => [
            [ '815', '815', 'TURQUEL' ] ] },
        'Arieira' => {
          '2460' => [
            [ '341', '362', 'CELA ACB' ] ] },
        'Azenha de Cima' => {
          '2460' => [
            [ '801', '801', 'TURQUEL' ] ] },
        'Paredes da Vitria' => {
          '2445' => [
            [ '130', '130', 'PATAIAS' ] ] },
        'gua de Madeiros' => {
          '2445' => [
            [ '011', '057', 'PATAIAS' ] ] },
        'Lgua' => {
          '2445' => [
            [ '112', '112', 'PATAIAS' ] ] },
        'Acipreste' => {
          '2460' => [
            [ '471', '481', 'ALCOBAA' ] ] },
        'Barba Torta' => {
          '2460' => [
            [ '191', '209', 'ALFEIZERO' ] ] },
        'Pises' => {
          '2445' => [
            [ '435', '473', 'PATAIAS' ] ] },
        'Melvoa' => {
          '2445' => [
            [ '115', '115', 'PATAIAS' ] ] },
        'Alm do Porto' => {
          '2460' => [
            [ '301', '317', 'BRRIO ACB' ] ] },
        'Pataias Gare' => {
          '2445' => [
            [ '401', '420', 'PATAIAS' ] ] },
        'Benedita' => {
          '2475' => [
            [ '100', '152', 'BENEDITA' ] ] },
        'Quinta da Azenha' => {
          '2460' => [
            [ '451', '453', 'COZ' ] ] },
        'Alcobaa' => {
          '2461' => [
            [ '501', '506', 'ALCOBAA' ] ],
          '2460' => [
            [ '001', '009', 'ALCOBAA' ],
            [ '011', '048', 'ALCOBAA' ],
            [ '050', '082', 'ALCOBAA' ],
            [ '085', '089', 'ALCOBAA' ],
            [ '913', '915', 'ALCOBAA' ] ] },
        'Vale Furado' => {
          '2445' => [
            [ '511', '523', 'PATAIAS' ],
            [ '525', '527', 'PATAIAS' ] ] },
        'Aldeia Nova' => {
          '2460' => [
            [ '761', '781', 'VIMEIRO ACB' ] ] },
        'Montes' => {
          '2460' => [
            [ '821', '852', 'MONTES' ],
            [ '854', '869', 'MONTES' ],
            [ '904', '912', 'MONTES' ],
            [ '916', '916', 'MONTES' ] ] },
        'Azenha de Baixo' => {
          '2460' => [
            [ '785', '799', 'TURQUEL' ] ] },
        'Pedra do Ouro' => {
          '2445' => [
            [ '431', '431', 'PATAIAS' ] ] },
        'Mina do Azeiche' => {
          '2445' => [
            [ '121', '125', 'PATAIAS' ] ] },
        'Figueiras do Entrudo' => {
          '2460' => [
            [ '804', '805', 'ALCOBAA' ] ] },
        'Aljubarrota' => {
          '2460' => [
            [ '601', '619', 'ALJUBARROTA (PRAZERES)' ],
            [ '711', '721', 'ALJUBARROTA (SO VICENTE)' ] ] },
        'Martingana' => {
          '2445' => [
            [ '701', '737', 'MARTINGANA' ] ] },
        'Ardido' => {
          '2460' => [
            [ '817', '820', 'TURQUEL' ] ] },
        'Aliceira' => {
          '2460' => [
            [ '741', '744', 'VESTIARIA' ] ] },
        'Alfeizero' => {
          '2460' => [
            [ '101', '175', 'ALFEIZERO' ] ] },
        'Valeiro' => {
          '2445' => [
            [ '540', '540', 'PATAIAS' ] ] },
        'Polvoeira' => {
          '2445' => [
            [ '491', '497', 'PATAIAS' ] ] },
        'Alpedriz' => {
          '2460' => [
            [ '231', '263', 'ALPEDRIZ' ] ] },
        'Algaro' => {
          '2475' => [
            [ '011', '043', 'BENEDITA' ] ] },
        'Casal da Costa' => {
          '2460' => [
            [ '176', '176', 'ALCOBAA' ] ] },
        'Mijaretes' => {
          '2445' => [
            [ '117', '117', 'PATAIAS' ] ] },
        'Pataias' => {
          '2445' => [
            [ '201', '379', 'PATAIAS' ] ] },
        'So Martinho do Porto' => {
          '2460' => [
            [ '083', '084', 'SO MARTINHO DO PORTO' ],
            [ '090', '091', 'SO MARTINHO DO PORTO' ],
            [ '093', '093', 'SO MARTINHO DO PORTO' ],
            [ '095', '099', 'SO MARTINHO DO PORTO' ],
            [ '631', '694', 'SO MARTINHO DO PORTO' ],
            [ '870', '873', 'SO MARTINHO DO PORTO' ] ] },
        'Pvoa' => {
          '2460' => [
            [ '455', '455', 'COZ' ] ] },
        'Moita do Poo' => {
          '2460' => [
            [ '100', '100', 'TURQUEL' ] ] },
        'Ferraria' => {
          '2445' => [
            [ '075', '105', 'PATAIAS' ] ],
          '2460' => [
            [ '281', '283', 'ALPEDRIZ' ] ] },
        'Casal da Ortiga' => {
          '2460' => [
            [ '483', '517', 'ALCOBAA' ] ] },
        'Serra dos Mangues' => {
          '2460' => [
            [ '697', '699', 'SO MARTINHO DO PORTO' ] ] },
        'Martingana-Gare' => {
          '2445' => [
            [ '761', '771', 'MARTINGANA' ] ] },
        'Turquel' => {
          '2460' => [
            [ '806', '813', 'TURQUEL' ],
            [ '874', '903', 'TURQUEL' ] ] },
        'Bemposta' => {
          '2460' => [
            [ '521', '526', 'MAIORGA' ] ] },
        'Cabrela' => {
          '2445' => [
            [ '071', '072', 'PATAIAS' ] ] },
        'Feitosa' => {
          '2460' => [
            [ '803', '803', 'TURQUEL' ] ] },
        'Maiorga' => {
          '2460' => [
            [ '531', '583', 'MAIORGA' ] ] }
              },
      'Peniche' => {
        'Alto Verssimo' => {
          '2525' => [
            [ '005', '005', 'ATOUGUIA DA BALEIA' ] ] },
        'Alto Foz' => {
          '2525' => [
            [ '001', '002', 'ATOUGUIA DA BALEIA' ] ] },
        'Ferrel' => {
          '2520' => [
            [ '101', '192', 'FERREL' ] ] },
        'So Bernardino' => {
          '2525' => [
            [ '751', '780', 'ATOUGUIA DA BALEIA' ] ] },
        'Baleal' => {
          '2520' => [
            [ '001', '016', 'FERREL' ] ] },
        'Herdade Salgados' => {
          '2525' => [
            [ '551', '559', 'ATOUGUIA DA BALEIA' ] ] },
        'Casais Baleal' => {
          '2520' => [
            [ '051', '061', 'FERREL' ] ] },
        'Carnide' => {
          '2525' => [
            [ '220', '220', 'ATOUGUIA DA BALEIA' ] ] },
        'Casal Brejoeiras' => {
          '2525' => [
            [ '792', '793', 'SERRA D\'EL REI' ] ] },
        'Cruz' => {
          '2525' => [
            [ '460', '460', 'ATOUGUIA DA BALEIA' ] ] },
        'Fetais' => {
          '2525' => [
            [ '500', '500', 'ATOUGUIA DA BALEIA' ] ] },
        'Atouguia da Baleia' => {
          '2525' => [
            [ '021', '090', 'ATOUGUIA DA BALEIA' ] ] },
        'Coimbr' => {
          '2525' => [
            [ '401', '418', 'ATOUGUIA DA BALEIA' ] ] },
        'Estrada' => {
          '2525' => [
            [ '471', '487', 'ATOUGUIA DA BALEIA' ] ] },
        'Reinaldes' => {
          '2525' => [
            [ '671', '692', 'ATOUGUIA DA BALEIA' ] ] },
        'Casal Estalagem' => {
          '2525' => [
            [ '361', '379', 'ATOUGUIA DA BALEIA' ] ] },
        'Ribafria' => {
          '2525' => [
            [ '711', '737', 'ATOUGUIA DA BALEIA' ] ] },
        'Casais Brancos' => {
          '2525' => [
            [ '231', '251', 'ATOUGUIA DA BALEIA' ] ] },
        'Bolhos' => {
          '2525' => [
            [ '101', '122', 'ATOUGUIA DA BALEIA' ] ] },
        'Casais do Jlio' => {
          '2525' => [
            [ '281', '293', 'ATOUGUIA DA BALEIA' ] ] },
        'Peniche' => {
          '2520' => [
            [ '201', '363', 'PENICHE' ],
            [ '367', '444', 'PENICHE' ],
            [ '446', '552', 'PENICHE' ],
            [ '601', '632', 'PENICHE' ] ] },
        'Casais Mestre Mendo' => {
          '2525' => [
            [ '311', '330', 'ATOUGUIA DA BALEIA' ] ] },
        'Vale Grou' => {
          '2525' => [
            [ '791', '791', 'ATOUGUIA DA BALEIA' ] ] },
        'Consolao' => {
          '2525' => [
            [ '431', '454', 'ATOUGUIA DA BALEIA' ] ] },
        'Casal Albeoico' => {
          '2525' => [
            [ '341', '351', 'ATOUGUIA DA BALEIA' ] ] },
        'Serra D\'El Rei' => {
          '2525' => [
            [ '801', '882', 'SERRA D\'EL REI' ] ] },
        'Geraldes' => {
          '2525' => [
            [ '511', '534', 'ATOUGUIA DA BALEIA' ] ] },
        'Botado' => {
          '2525' => [
            [ '150', '150', 'ATOUGUIA DA BALEIA' ] ] },
        'Bufarda' => {
          '2525' => [
            [ '161', '192', 'ATOUGUIA DA BALEIA' ] ] }
             },
      'Marinha Grande' => {
        'Amieirinha' => {
          '2430' => [
            [ '021', '045', 'MARINHA GRANDE' ] ] },
        'Trutas' => {
          '2430' => [
            [ '520', '526', 'MARINHA GRANDE' ] ] },
        'Lameira da Embra' => {
          '2430' => [
            [ '170', '174', 'MARINHA GRANDE' ] ] },
        'Embra' => {
          '2430' => [
            [ '091', '134', 'MARINHA GRANDE' ] ] },
        'Picassinos' => {
          '2430' => [
            [ '415', '446', 'MARINHA GRANDE' ],
            [ '448', '463', 'MARINHA GRANDE' ] ] },
        'Sitias' => {
          '2430' => [
            [ '515', '515', 'MARINHA GRANDE' ] ] },
        'Albergaria' => {
          '2430' => [
            [ '011', '013', 'MARINHA GRANDE' ] ] },
        'Forno da Telha' => {
          '2430' => [
            [ '136', '168', 'MARINHA GRANDE' ] ] },
        'Tojeira de Picassinos' => {
          '2430' => [
            [ '359', '403', 'MARINHA GRANDE' ] ] },
        'Boavista' => {
          '2430' => [
            [ '061', '072', 'MARINHA GRANDE' ] ] },
        'Casal da Anja' => {
          '2430' => [
            [ '823', '830', 'VIEIRA DE LEIRIA' ] ] },
        'Benta' => {
          '2430' => [
            [ '048', '059', 'MARINHA GRANDE' ] ] },
        'Vieira de Leiria' => {
          '2430' => [
            [ '592', '761', 'VIEIRA DE LEIRIA' ],
            [ '763', '821', 'VIEIRA DE LEIRIA' ] ] },
        'Marinha Grande' => {
          '2430' => [
            [ '200', '201', 'MARINHA GRANDE' ],
            [ '203', '316', 'MARINHA GRANDE' ],
            [ '822', '822', 'MARINHA GRANDE' ] ] },
        'Moita' => {
          '2445' => [
            [ '571', '596', 'MOITA MGR' ],
            [ '599', '606', 'MOITA MGR' ],
            [ '608', '609', 'MOITA MGR' ],
            [ '611', '622', 'MOITA MGR' ],
            [ '624', '624', 'MOITA MGR' ] ] },
        'So Pedro de Muel' => {
          '2430' => [
            [ '480', '511', 'MARINHA GRANDE' ] ] },
        'Marinha Pequena' => {
          '2430' => [
            [ '350', '350', 'MARINHA GRANDE' ] ] },
        'Vale' => {
          '2445' => [
            [ '630', '637', 'MOITA MGR' ] ] },
        'Almoinha Velha' => {
          '2445' => [
            [ '561', '562', 'MOITA MGR' ] ] },
        'Pilado' => {
          '2430' => [
            [ '471', '475', 'MARINHA GRANDE' ] ] }
              },
      'Batalha' => {
        'Cela' => {
          '2440' => [
            [ '151', '197', 'BATALHA' ] ] },
        'Brancas' => {
          '2440' => [
            [ '145', '149', 'BATALHA' ] ] },
        'Golfeiros' => {
          '2440' => [
            [ '038', '039', 'BATALHA' ] ] },
        'Santo Anto' => {
          '2440' => [
            [ '053', '053', 'BATALHA' ] ] },
        'Forneiros' => {
          '2440' => [
            [ '035', '036', 'BATALHA' ] ] },
        'Casal Novo' => {
          '2440' => [
            [ '024', '026', 'BATALHA' ] ] },
        'Bico Sachos' => {
          '2440' => [
            [ '230', '237', 'GOLPILHEIRA' ] ] },
        'Paredes' => {
          '2440' => [
            [ '044', '044', 'BATALHA' ] ] },
        'Alcaidaria' => {
          '2440' => [
            [ '201', '214', 'REGUENGO DO FETAL' ] ] },
        'Corga' => {
          '2440' => [
            [ '033', '033', 'BATALHA' ] ] },
        'Casal das Carvalhas' => {
          '2440' => [
            [ '001', '001', 'BATALHA' ] ] },
        'Casal de Santa Joana' => {
          '2440' => [
            [ '028', '028', 'BATALHA' ] ] },
        'Celeiro' => {
          '2440' => [
            [ '030', '030', 'REGUENGO DO FETAL' ] ] },
        'Casal do Marra' => {
          '2440' => [
            [ '300', '470', 'BATALHA' ] ] },
        'Arneiro' => {
          '2440' => [
            [ '012', '012', 'BATALHA' ] ] },
        'Casal do Arqueiro' => {
          '2440' => [
            [ '019', '022', 'BATALHA' ] ] },
        'Vila Facaia' => {
          '2440' => [
            [ '200', '200', 'BATALHA' ] ] },
        'Monho de Vento' => {
          '2440' => [
            [ '041', '042', 'BATALHA' ] ] },
        'Barreirinho Velho' => {
          '2495' => [
            [ '011', '037', 'SO MAMEDE' ] ] },
        'Alto da Fonte' => {
          '2440' => [
            [ '010', '010', 'BATALHA' ] ] },
        'Colipo' => {
          '2440' => [
            [ '032', '032', 'GOLPILHEIRA' ] ] },
        'Quinta Nova' => {
          '2440' => [
            [ '050', '050', 'BATALHA' ] ] },
        'Batalha' => {
          '2440' => [
            [ '055', '143', 'BATALHA' ] ] },
        'Cancelas' => {
          '2440' => [
            [ '015', '017', 'BATALHA' ] ] },
        'Centas' => {
          '2440' => [
            [ '031', '031', 'BATALHA' ] ] },
        'Casal da Ponte Nova' => {
          '2440' => [
            [ '046', '048', 'BATALHA' ] ] },
        'Garruchas' => {
          '2440' => [
            [ '037', '037', 'REGUENGO DO FETAL' ] ] }
             },
      'Ansio' => {
        'Casal de Santo Antnio' => {
          '3240' => [
            [ '351', '356', 'AVELAR' ] ] },
        'guas Frreas' => {
          '3240' => [
            [ '201', '204', 'ANSIO' ] ] },
        'Alqueido' => {
          '3240' => [
            [ '651', '701', 'SANTIAGO DA GUARDA' ] ] },
        'Alqueido So Brs' => {
          '3240' => [
            [ '451', '508', 'CHO DE COUCE' ] ] },
        'Aljazede' => {
          '3240' => [
            [ '401', '422', 'ALVORGE' ] ] },
        'Barreiras' => {
          '3240' => [
            [ '551', '562', 'LAGARTEIRA' ] ] },
        'Bate gua' => {
          '3240' => [
            [ '206', '236', 'ANSIO' ] ] },
        'Casal So Joo' => {
          '3240' => [
            [ '702', '710', 'TORRE DE VALE DE TODOS' ] ] },
        'Avelar' => {
          '3240' => [
            [ '301', '334', 'AVELAR' ] ] },
        'Ansio' => {
          '3240' => [
            [ '101', '102', 'ANSIO' ],
            [ '123', '157', 'ANSIO' ] ] },
        'Adegas' => {
          '3240' => [
            [ '581', '617', 'POUSAFLORES' ] ] }
            },
      'Figueir dos Vinhos' => {
        'Aldeia Cimeira' => {
          '3260' => [
            [ '500', '512', 'BAIRRADAS' ] ] },
        'Caldeireiro' => {
          '3260' => [
            [ '306', '361', 'FIGUEIR DOS VINHOS' ] ] },
        'Agria Grande' => {
          '3260' => [
            [ '300', '304', 'FIGUEIR DOS VINHOS' ] ] },
        'Abrunheira' => {
          '3260' => [
            [ '020', '057', 'AGUDA' ] ] },
        'Arega' => {
          '3260' => [
            [ '070', '101', 'AREGA' ] ] },
        'Figueir dos Vinhos' => {
          '3260' => [
            [ '400', '430', 'FIGUEIR DOS VINHOS' ] ] },
        'Aldeia Fundeira' => {
          '3260' => [
            [ '200', '224', 'CAMPELO' ] ] }
             },
      'Caldas da Rainha' => {
        'Casal Novo' => {
          '2500' => [
            [ '356', '389', 'ALVORNINHA' ] ] },
        'Granja Nova' => {
          '2500' => [
            [ '793', '797', 'SANTA CATARINA CLD' ] ] },
        'Landal' => {
          '2500' => [
            [ '530', '541', 'LANDAL' ] ] },
        'Santa Catarina' => {
          '2500' => [
            [ '864', '879', 'SANTA CATARINA CLD' ] ] },
        'Foz do Arelho' => {
          '2500' => [
            [ '450', '517', 'FOZ DO ARELHO' ],
            [ '606', '606', 'FOZ DO ARELHO' ],
            [ '889', '896', 'FOZ DO ARELHO' ] ] },
        'Nadadouro' => {
          '2500' => [
            [ '551', '565', 'NADADOURO' ],
            [ '567', '605', 'NADADOURO' ],
            [ '607', '611', 'NADADOURO' ],
            [ '912', '912', 'NADADOURO' ] ] },
        'Alvorninha' => {
          '2500' => [
            [ '330', '354', 'ALVORNINHA' ] ] },
        'Carrasqueira' => {
          '2500' => [
            [ '731', '749', 'VIDAIS' ] ] },
        'Barrantes' => {
          '2500' => [
            [ '621', '643', 'SALIR DE MATOS' ] ] },
        'Bouro' => {
          '2500' => [
            [ '711', '719', 'SERRA DO BOURO' ] ] },
        'Salir do Porto' => {
          '2500' => [
            [ '651', '697', 'SALIR DO PORTO' ],
            [ '897', '911', 'SALIR DO PORTO' ] ] },
        'Fanadia' => {
          '2500' => [
            [ '064', '065', 'SO GREGRIO CLD' ] ] },
        'A dos Francos' => {
          '2500' => [
            [ '001', '053', 'A DOS FRANCOS' ] ] },
        'Antas de Cima' => {
          '2500' => [
            [ '401', '415', 'CARVALHAL BENFEITO' ] ] },
        'Casais da Ponte' => {
          '2500' => [
            [ '431', '434', 'COTO' ] ] },
        'Abrunheira' => {
          '2500' => [
            [ '761', '791', 'SANTA CATARINA CLD' ] ] },
        'Casal das Penas' => {
          '2500' => [
            [ '799', '799', 'SANTA CATARINA CLD' ] ] },
        'Caldas da Rainha' => {
          '2500' => [
            [ '068', '076', 'CALDAS DA RAINHA' ],
            [ '078', '274', 'CALDAS DA RAINHA' ],
            [ '276', '278', 'CALDAS DA RAINHA' ],
            [ '280', '280', 'CALDAS DA RAINHA' ],
            [ '282', '282', 'CALDAS DA RAINHA' ],
            [ '285', '290', 'CALDAS DA RAINHA' ],
            [ '292', '329', 'CALDAS DA RAINHA' ],
            [ '750', '760', 'CALDAS DA RAINHA' ],
            [ '798', '798', 'CALDAS DA RAINHA' ],
            [ '801', '863', 'CALDAS DA RAINHA' ],
            [ '880', '888', 'CALDAS DA RAINHA' ],
            [ '913', '914', 'CALDAS DA RAINHA' ] ] }
                }
          },
    'Ilha do Pico' => {
      'Lajes do Pico' => {
        'Arrife' => {
          '9930' => [
            [ '301', '310', 'RIBEIRAS' ] ] },
        'Boca das Canadas' => {
          '9930' => [
            [ '011', '011', 'CALHETA DE NESQUIM' ] ] },
        'Almagreira de Baixo' => {
          '9930' => [
            [ '101', '108', 'LAJES DO PICO' ] ] },
        'Canadas' => {
          '9930' => [
            [ '051', '057', 'CALHETA DE NESQUIM' ] ] },
        'Ramal' => {
          '9930' => [
            [ '171', '179', 'LAJES DO PICO' ] ] },
        'Ermida' => {
          '9930' => [
            [ '471', '473', 'SO JOO LGP' ] ] },
        'Altamora' => {
          '9930' => [
            [ '201', '206', 'PIEDADE' ] ] },
        'Aguada' => {
          '9930' => [
            [ '401', '401', 'SO JOO LGP' ] ] },
        'Piedade' => {
          '9930' => [
            [ '221', '229', 'PIEDADE' ] ] },
        'Ponta da Ilha' => {
          '9930' => [
            [ '251', '254', 'PIEDADE' ] ] },
        'Calheta de Nesquim' => {
          '9930' => [
            [ '021', '031', 'CALHETA DE NESQUIM' ] ] },
        'Ribeira Funda' => {
          '9930' => [
            [ '320', '343', 'RIBEIRINHA LGP' ] ] },
        'Lajes do Pico' => {
          '9930' => [
            [ '121', '154', 'LAJES DO PICO' ] ] },
        'Faias' => {
          '9930' => [
            [ '208', '215', 'PIEDADE' ] ] },
        'Companhia de Cima' => {
          '9930' => [
            [ '451', '457', 'SO JOO LGP' ] ] },
        'Companhia de Baixo' => {
          '9930' => [
            [ '421', '433', 'SO JOO LGP' ] ] }
             },
      'So Roque do Pico' => {
        'Prainha de Cima' => {
          '9940' => [
            [ '060', '069', 'PRAINHA' ] ] },
        'Arrife' => {
          '9940' => [
            [ '011', '024', 'PRAINHA' ] ] },
        'Santo Antnio' => {
          '9940' => [
            [ '221', '238', 'SANTO ANTNIO SRP' ] ] },
        'Terras Limpas' => {
          '9940' => [
            [ '190', '191', 'SANTO AMARO SRP' ] ] },
        'So Roque do Pico' => {
          '9940' => [
            [ '321', '367', 'SO ROQUE DO PICO' ] ] },
        'So Vicente' => {
          '9940' => [
            [ '251', '252', 'SANTO ANTNIO SRP' ] ] },
        'Santo Amaro' => {
          '9940' => [
            [ '171', '185', 'SANTO AMARO SRP' ] ] },
        'Cabecinho' => {
          '9940' => [
            [ '151', '157', 'SANTO AMARO SRP' ] ] },
        'Santa Luzia' => {
          '9940' => [
            [ '121', '131', 'SANTA LUZIA SRP' ] ] },
        'Baleeiras' => {
          '9940' => [
            [ '301', '312', 'SO ROQUE DO PICO' ] ] },
        'Almas' => {
          '9940' => [
            [ '201', '213', 'SANTO ANTNIO SRP' ] ] },
        'Alto da Bonana' => {
          '9940' => [
            [ '101', '112', 'SANTA LUZIA SRP' ] ] },
        'Prainha de Baixo' => {
          '9940' => [
            [ '031', '043', 'PRAINHA' ] ] }
                 },
      'Madalena' => {
        'Outeiro' => {
          '9950' => [
            [ '361', '365', 'MADALENA (PICO)' ] ] },
        'Terra do Po' => {
          '9950' => [
            [ '451', '451', 'SO CAETANO MAD' ] ] },
        'So Caetano' => {
          '9950' => [
            [ '421', '428', 'SO CAETANO MAD' ] ] },
        'Alto da Cruz' => {
          '9950' => [
            [ '101', '105', 'CANDELRIA MAD' ] ] },
        'Arieiro' => {
          '9950' => [
            [ '501', '508', 'SO MATEUS MAD' ] ] },
        'Candelria' => {
          '9950' => [
            [ '121', '130', 'CANDELRIA MAD' ] ] },
        'Pranha' => {
          '9950' => [
            [ '430', '430', 'SO CAETANO MAD' ] ] },
        'Areia Funda' => {
          '9950' => [
            [ '301', '305', 'MADALENA (PICO)' ] ] },
        'So Mateus' => {
          '9950' => [
            [ '521', '541', 'SO MATEUS MAD' ] ] },
        'Criao Velha' => {
          '9950' => [
            [ '221', '236', 'CRIAO VELHA' ] ] },
        'Cabeo' => {
          '9950' => [
            [ '051', '056', 'BANDEIRAS' ] ] },
        'Aeroporto' => {
          '9950' => [
            [ '011', '011', 'BANDEIRAS' ] ] },
        'Canto da Candelria' => {
          '9950' => [
            [ '151', '156', 'CANDELRIA MAD' ] ] },
        'Madalena' => {
          '9950' => [
            [ '321', '341', 'MADALENA (PICO)' ] ] },
        'Bandeiras' => {
          '9950' => [
            [ '021', '030', 'BANDEIRAS' ] ] }
              }
          },
    'Bragana' => {
      'Vila Flor' => {
        'Zona Industrial' => {
          '5360' => [
            [ '450', '450', 'VILA FLOR' ] ] },
        'Vila Flor' => {
          '5360' => [
            [ '307', '316', 'VILA FLOR' ],
            [ '318', '383', 'VILA FLOR' ],
            [ '385', '398', 'VILA FLOR' ],
            [ '400', '415', 'VILA FLOR' ] ] },
        'Seixo de Manhoses' => {
          '5360' => [
            [ '181', '189', 'SEIXO DE MANHOSES' ] ] },
        'Candoso' => {
          '5360' => [
            [ '031', '038', 'CANDOSO' ] ] },
        'Assares' => {
          '5360' => [
            [ '011', '011', 'ASSARES' ] ] },
        'Sampaio' => {
          '5360' => [
            [ '160', '160', 'SAMPAIO' ] ] },
        'Arco' => {
          '5360' => [
            [ '300', '305', 'VILA FLOR' ] ] },
        'Vale Frechoso' => {
          '5360' => [
            [ '220', '220', 'VALE FRECHOSO' ] ] },
        'Mouro' => {
          '5360' => [
            [ '090', '090', 'MOURO VFL' ] ] },
        'Folgares' => {
          '5360' => [
            [ '061', '063', 'FREIXIEL' ] ] },
        'Sames' => {
          '5360' => [
            [ '150', '150', 'SAMES' ] ] },
        'Nabo' => {
          '5360' => [
            [ '101', '109', 'NABO' ] ] },
        'Macedinho' => {
          '5360' => [
            [ '201', '203', 'TRINDADE VFL' ] ] },
        'Alagoa' => {
          '5360' => [
            [ '231', '237', 'VALTORNO' ] ] },
        'Benlhevai' => {
          '5360' => [
            [ '021', '021', 'BENLHEVAI' ] ] },
        'Lodes' => {
          '5360' => [
            [ '080', '080', 'LODES' ] ] },
        'Santa Comba da Vilaria' => {
          '5360' => [
            [ '170', '170', 'SANTA COMBA DE VILARIA' ] ] },
        'Meireles' => {
          '5360' => [
            [ '491', '493', 'VILAS BOAS VFL' ] ] },
        'Roios' => {
          '5360' => [
            [ '130', '130', 'ROIOS' ] ] },
        'Carvalho de Egas' => {
          '5360' => [
            [ '050', '050', 'CARVALHO DE EGAS' ] ] },
        'Vilarinho das Azenhas' => {
          '5360' => [
            [ '470', '470', 'VILARINHO DAS AZENHAS' ] ] }
           },
      'Carrazeda de Ansies' => {
        'Brunheda' => {
          '5140' => [
            [ '203', '206', 'PINHAL DO NORTE' ] ] },
        'Zedes' => {
          '5140' => [
            [ '300', '302', 'ZEDES' ] ] },
        'Castanheiro Norte' => {
          '5140' => [
            [ '131', '136', 'CASTANHEIRO' ] ] },
        'Pinhal do Douro' => {
          '5140' => [
            [ '270', '271', 'VILARINHO DA CASTANHEIRA' ] ] },
        'Mogo de Malta' => {
          '5140' => [
            [ '171', '171', 'MOGO DE MALTA' ] ] },
        'Besteiros' => {
          '5140' => [
            [ '141', '143', 'FONTE LONGA CRZ' ] ] },
        'Samorinha' => {
          '5140' => [
            [ '120', '120', 'CARRAZEDA DE ANSIES' ] ] },
        'Luzelos' => {
          '5140' => [
            [ '161', '162', 'MARZAGO' ] ] },
        'Alganhafres' => {
          '5140' => [
            [ '251', '252', 'SELORES' ] ] },
        'Vilarinho da Castanheira' => {
          '5140' => [
            [ '275', '286', 'VILARINHO DA CASTANHEIRA' ] ] },
        'Amedo' => {
          '5140' => [
            [ '011', '012', 'AMEDO' ] ] },
        'Carrazeda de Ansies' => {
          '5140' => [
            [ '055', '106', 'CARRAZEDA DE ANSIES' ],
            [ '108', '109', 'CARRAZEDA DE ANSIES' ] ] },
        'Codeais' => {
          '5140' => [
            [ '201', '202', 'PEREIROS CRZ' ] ] },
        'Beira Grande' => {
          '5140' => [
            [ '021', '022', 'BEIRA GRANDE' ] ] },
        'Bairro Moinho de Vento' => {
          '5140' => [
            [ '051', '052', 'CARRAZEDA DE ANSIES' ] ] },
        'Coleja' => {
          '5140' => [
            [ '231', '233', 'SEIXO DE ANSIES' ] ] },
        'Mogo de Ansies' => {
          '5140' => [
            [ '031', '032', 'BELVER CRZ' ] ] },
        'Lavandeira' => {
          '5140' => [
            [ '144', '144', 'LAVANDEIRA' ] ] },
        'Paradela' => {
          '5140' => [
            [ '221', '223', 'POMBAL CRZ' ] ] },
        'Ribalonga' => {
          '5140' => [
            [ '224', '224', 'RIBALONGA CRZ' ] ] },
        'Arnal' => {
          '5140' => [
            [ '151', '154', 'LINHARES CRZ' ] ] },
        'Misquel' => {
          '5140' => [
            [ '181', '184', 'PARAMBOS' ] ] }
                },
      'Miranda do Douro' => {
        'Zona Industrial' => {
          '5210' => [
            [ '175', '175', 'MIRANDA DO DOURO' ] ] },
        'Constantim' => {
          '5210' => [
            [ '030', '030', 'CONSTANTIM MDR' ] ] },
        'Malhadas' => {
          '5210' => [
            [ '150', '150', 'MALHADAS' ] ] },
        'Ifanes' => {
          '5210' => [
            [ '101', '133', 'IFANES' ] ] },
        'Prado Gato' => {
          '5225' => [
            [ '041', '041', 'PALAOULO' ] ] },
        'Atenor' => {
          '5225' => [
            [ '011', '012', 'ATENOR' ] ] },
        'So Martinho da Angueira' => {
          '5210' => [
            [ '300', '300', 'SO MARTINHO DE ANGUEIRA' ] ] },
        'Vila Ch de Braciosa' => {
          '5210' => [
            [ '335', '349', 'VILA CH DE BRACIOSA' ] ] },
        'Sendim' => {
          '5225' => [
            [ '100', '153', 'SENDIM' ] ] },
        'guas Vivas' => {
          '5225' => [
            [ '031', '031', 'GUAS VIVAS' ] ] },
        'Barrocal do Douro' => {
          '5225' => [
            [ '071', '072', 'PICOTE' ] ] },
        'Fonte Ladro' => {
          '5225' => [
            [ '161', '163', 'SILVA MDR' ] ] },
        'Pvoa' => {
          '5210' => [
            [ '281', '295', 'PVOA' ] ] },
        'Miranda do Douro' => {
          '5210' => [
            [ '001', '003', 'MIRANDA DO DOURO' ],
            [ '180', '237', 'MIRANDA DO DOURO' ] ] },
        'Especiosa' => {
          '5210' => [
            [ '080', '080', 'GENSIO' ] ] },
        'Aldeia Nova' => {
          '5210' => [
            [ '170', '173', 'MIRANDA DO DOURO' ] ] },
        'Gensio' => {
          '5210' => [
            [ '090', '090', 'GENSIO' ] ] },
        'Fonte da Aldeia' => {
          '5210' => [
            [ '321', '331', 'VILA CH DE BRACIOSA' ] ] },
        'Freixiosa' => {
          '5210' => [
            [ '333', '333', 'VILA CH DE BRACIOSA' ] ] },
        'Palaoulo' => {
          '5225' => [
            [ '032', '032', 'PALAOULO' ] ] },
        'Paradela' => {
          '5210' => [
            [ '261', '267', 'PARADELA MDR' ] ] },
        'Crcio' => {
          '5210' => [
            [ '041', '064', 'DUAS IGREJAS MDR' ] ] },
        'Cicouro' => {
          '5210' => [
            [ '020', '020', 'CICOURO' ] ] }
            },
      'Bragana' => {
        'Alfaio' => {
          '5300' => [
            [ '401', '402', 'ALFAIO' ] ] },
        'Carocedo' => {
          '5300' => [
            [ '531', '532', 'FAILDE' ] ] },
        'Fronteira' => {
          '5300' => [
            [ '771', '775', 'QUINTANILHA' ] ] },
        'So Pedro de Sarracenos' => {
          '5300' => [
            [ '874', '874', 'SO PEDRO DE SERRACENOS' ] ] },
        'Rio de Onor' => {
          '5300' => [
            [ '821', '822', 'RIO DE ONOR' ] ] },
        'So Julio de Palcios' => {
          '5300' => [
            [ '871', '873', 'SO JULIO DE PALCIOS' ] ] },
        'Castrelos' => {
          '5300' => [
            [ '471', '474', 'CASTRELOS' ] ] },
        'Quinta da Aveleira' => {
          '5300' => [
            [ '491', '494', 'COELHOSO' ] ] },
        'Paradinha Nova' => {
          '5300' => [
            [ '731', '732', 'PARADINHA NOVA' ] ] },
        'Chos' => {
          '5300' => [
            [ '841', '846', 'SALSAS' ] ] },
        'Rabal' => {
          '5300' => [
            [ '791', '791', 'RABAL' ] ] },
        'Fontes Transbaceiro' => {
          '5300' => [
            [ '741', '744', 'PARMIO' ] ] },
        'Nogueira' => {
          '5300' => [
            [ '701', '703', 'NOGUEIRA BGC' ] ] },
        'Carragosa' => {
          '5300' => [
            [ '451', '453', 'CARRAGOSA' ] ] },
        'Frana' => {
          '5300' => [
            [ '541', '544', 'FRANA' ] ] },
        'Rio Frio' => {
          '5300' => [
            [ '831', '832', 'RIO FRIO BGC' ] ] },
        'Rebordos' => {
          '5300' => [
            [ '811', '812', 'REBORDOS' ] ] },
        'Pombares' => {
          '5300' => [
            [ '761', '761', 'POMBARES' ] ] },
        'Bairro Baixo' => {
          '5300' => [
            [ '521', '525', 'ESPINHOSELA' ] ] },
        'Serapicos' => {
          '5300' => [
            [ '891', '893', 'SERAPICOS BGC' ] ] },
        'Pa' => {
          '5300' => [
            [ '691', '693', 'MS BGC' ] ] },
        'Carvas' => {
          '5300' => [
            [ '551', '553', 'GIMONDE' ] ] },
        'Calvelhe' => {
          '5300' => [
            [ '441', '441', 'CALVELHE' ] ] },
        'Aveleda' => {
          '5300' => [
            [ '411', '412', 'AVELEDA BGC' ] ] },
        'Castro de Avels' => {
          '5300' => [
            [ '481', '483', 'BRAGANA' ] ] },
        'Deilo' => {
          '5300' => [
            [ '501', '503', 'DEILO' ] ] },
        'Freixedelo' => {
          '5300' => [
            [ '581', '582', 'GRIJ DE PARADA' ] ] },
        'Parada' => {
          '5300' => [
            [ '721', '722', 'PARADA BGC' ] ] },
        'Fermentos' => {
          '5300' => [
            [ '881', '883', 'SENDAS' ] ] },
        'Pereiros' => {
          '5300' => [
            [ '801', '803', 'REBORDAINHOS' ] ] },
        'Babe' => {
          '5300' => [
            [ '421', '422', 'BABE' ] ] },
        'Alimonde' => {
          '5300' => [
            [ '461', '462', 'CARRAZEDO BGC' ] ] },
        'Gondesende' => {
          '5300' => [
            [ '561', '563', 'GONDESENDE' ] ] },
        'Bragana' => {
          '5300' => [
            [ '001', '003', 'BRAGANA' ],
            [ '005', '007', 'BRAGANA' ],
            [ '009', '020', 'BRAGANA' ],
            [ '023', '023', 'BRAGANA' ],
            [ '025', '026', 'BRAGANA' ],
            [ '028', '035', 'BRAGANA' ],
            [ '037', '038', 'BRAGANA' ],
            [ '040', '040', 'BRAGANA' ],
            [ '042', '044', 'BRAGANA' ],
            [ '046', '047', 'BRAGANA' ],
            [ '050', '051', 'BRAGANA' ],
            [ '053', '054', 'BRAGANA' ],
            [ '056', '056', 'BRAGANA' ],
            [ '058', '059', 'BRAGANA' ],
            [ '061', '062', 'BRAGANA' ],
            [ '065', '079', 'BRAGANA' ],
            [ '081', '089', 'BRAGANA' ],
            [ '091', '093', 'BRAGANA' ],
            [ '095', '101', 'BRAGANA' ],
            [ '104', '108', 'BRAGANA' ],
            [ '111', '114', 'BRAGANA' ],
            [ '116', '121', 'BRAGANA' ],
            [ '123', '126', 'BRAGANA' ],
            [ '128', '142', 'BRAGANA' ],
            [ '144', '144', 'BRAGANA' ],
            [ '146', '147', 'BRAGANA' ],
            [ '149', '149', 'BRAGANA' ],
            [ '151', '151', 'BRAGANA' ],
            [ '153', '155', 'BRAGANA' ],
            [ '158', '158', 'BRAGANA' ],
            [ '161', '163', 'BRAGANA' ],
            [ '166', '167', 'BRAGANA' ],
            [ '169', '174', 'BRAGANA' ],
            [ '176', '179', 'BRAGANA' ],
            [ '181', '181', 'BRAGANA' ],
            [ '183', '192', 'BRAGANA' ],
            [ '194', '202', 'BRAGANA' ],
            [ '204', '210', 'BRAGANA' ],
            [ '213', '213', 'BRAGANA' ],
            [ '217', '225', 'BRAGANA' ],
            [ '227', '227', 'BRAGANA' ],
            [ '229', '230', 'BRAGANA' ],
            [ '232', '232', 'BRAGANA' ],
            [ '235', '248', 'BRAGANA' ],
            [ '252', '253', 'BRAGANA' ],
            [ '257', '257', 'BRAGANA' ],
            [ '261', '263', 'BRAGANA' ],
            [ '265', '274', 'BRAGANA' ],
            [ '276', '278', 'BRAGANA' ],
            [ '281', '289', 'BRAGANA' ],
            [ '291', '299', 'BRAGANA' ],
            [ '302', '320', 'BRAGANA' ],
            [ '323', '325', 'BRAGANA' ],
            [ '330', '379', 'BRAGANA' ],
            [ '381', '384', 'BRAGANA' ],
            [ '386', '400', 'BRAGANA' ],
            [ '403', '410', 'BRAGANA' ],
            [ '413', '420', 'BRAGANA' ],
            [ '424', '430', 'BRAGANA' ],
            [ '435', '440', 'BRAGANA' ],
            [ '442', '450', 'BRAGANA' ],
            [ '454', '460', 'BRAGANA' ],
            [ '463', '470', 'BRAGANA' ],
            [ '476', '479', 'BRAGANA' ],
            [ '488', '490', 'BRAGANA' ],
            [ '495', '500', 'BRAGANA' ],
            [ '504', '518', 'BRAGANA' ],
            [ '520', '520', 'BRAGANA' ],
            [ '526', '526', 'BRAGANA' ],
            [ '528', '528', 'BRAGANA' ],
            [ '533', '540', 'BRAGANA' ],
            [ '545', '550', 'BRAGANA' ],
            [ '554', '560', 'BRAGANA' ],
            [ '564', '580', 'BRAGANA' ],
            [ '583', '590', 'BRAGANA' ],
            [ '645', '645', 'BRAGANA' ],
            [ '650', '660', 'BRAGANA' ],
            [ '664', '670', 'BRAGANA' ],
            [ '675', '676', 'BRAGANA' ],
            [ '678', '680', 'BRAGANA' ],
            [ '684', '690', 'BRAGANA' ],
            [ '694', '694', 'BRAGANA' ],
            [ '696', '700', 'BRAGANA' ],
            [ '704', '710', 'BRAGANA' ],
            [ '713', '718', 'BRAGANA' ],
            [ '720', '720', 'BRAGANA' ],
            [ '723', '730', 'BRAGANA' ],
            [ '733', '739', 'BRAGANA' ] ],
          '5301' => [
            [ '851', '857', 'BRAGANA' ],
            [ '859', '868', 'BRAGANA' ] ] },
        'Vilar' => {
          '5300' => [
            [ '681', '683', 'MILHO' ] ] },
        'Pao de Sortes' => {
          '5300' => [
            [ '901', '904', 'SORTES' ] ] },
        'Frieira' => {
          '5300' => [
            [ '661', '663', 'MACEDO DO MATO' ] ] },
        'Pinela' => {
          '5300' => [
            [ '751', '752', 'PINELA' ] ] },
        'Baal' => {
          '5300' => [
            [ '432', '434', 'BAAL' ] ] },
        'Loteamento das Comunidades' => {
          '5300' => [
            [ '849', '856', 'BRAGANA' ] ] },
        'Outeiro' => {
          '5300' => [
            [ '711', '712', 'OUTEIRO BGC' ] ] },
        'Zoio' => {
          '5300' => [
            [ '911', '913', 'ZOIO' ] ] },
        'Veigas' => {
          '5300' => [
            [ '781', '783', 'QUINTELA DE LAMPAAS' ] ] },
        'Santa Comba de Rossas' => {
          '5300' => [
            [ '861', '861', 'SANTA COMBA DE ROSSAS' ] ] },
        'Parque do Campismo' => {
          '5300' => [
            [ '671', '674', 'MEIXEDO BGC' ] ] },
        'Izeda' => {
          '5300' => [
            [ '591', '644', 'IZEDA' ] ] }
          },
      'Macedo de Cavaleiros' => {
        'Vale da Porca' => {
          '5340' => [
            [ '470', '470', 'VALE DA PORCA' ] ] },
        'Morais' => {
          '5340' => [
            [ '351', '353', 'MORAIS' ] ] },
        'Lagoa' => {
          '5340' => [
            [ '160', '160', 'LAGOA MCD' ] ] },
        'Amendoeira' => {
          '5340' => [
            [ '021', '024', 'AMENDOEIRA' ] ] },
        'Argana' => {
          '5340' => [
            [ '171', '174', 'LAMALONGA' ] ] },
        'Arrifana' => {
          '5340' => [
            [ '481', '482', 'VALE DE PRADOS' ] ] },
        'Gralhos' => {
          '5340' => [
            [ '451', '452', 'TALHINHAS' ] ] },
        'Talhas' => {
          '5340' => [
            [ '440', '440', 'TALHAS' ] ] },
        'Bousende' => {
          '5340' => [
            [ '131', '133', 'ESPADANEDO MCD' ] ] },
        'Vale Benfeito' => {
          '5340' => [
            [ '460', '460', 'VALE BENFEITO' ] ] },
        'Lamas' => {
          '5340' => [
            [ '180', '180', 'LAMAS DE PODENCE' ] ] },
        'Salselas' => {
          '5340' => [
            [ '400', '400', 'SALSELAS' ] ] },
        'Comunhas' => {
          '5340' => [
            [ '141', '142', 'FERREIRA MCD' ] ] },
        'Azibeiro' => {
          '5340' => [
            [ '391', '392', 'PODENCE' ] ] },
        'Vinhas' => {
          '5340' => [
            [ '520', '520', 'VINHAS' ] ] },
        'Vilarinho do Monte' => {
          '5340' => [
            [ '510', '510', 'VILARINHO DO MONTE' ] ] },
        'Sesulfe' => {
          '5340' => [
            [ '421', '422', 'SESULFE' ] ] },
        'Quinta da Caida' => {
          '5340' => [
            [ '031', '031', 'ARCAS' ] ] },
        'Bornes' => {
          '5340' => [
            [ '051', '052', 'BORNES' ] ] },
        'Corujas' => {
          '5340' => [
            [ '110', '110', 'CORUJAS' ] ] },
        'Vilarinho de Agrocho' => {
          '5340' => [
            [ '500', '500', 'VILARINHO DE AGROCHO' ] ] },
        'Muros' => {
          '5340' => [
            [ '360', '360', 'MURS' ] ] },
        'Vilar do Monte' => {
          '5340' => [
            [ '490', '490', 'VILAR DO MONTE MCD' ] ] },
        'Macedo de Cavaleiros' => {
          '5340' => [
            [ '193', '233', 'MACEDO DE CAVALEIROS' ],
            [ '235', '293', 'MACEDO DE CAVALEIROS' ] ] },
        'Balsamo' => {
          '5340' => [
            [ '091', '092', 'CHACIM' ] ] },
        'Carrapatas' => {
          '5340' => [
            [ '070', '070', 'CARRAPATAS' ] ] },
        'Santa Combinha' => {
          '5340' => [
            [ '410', '410', 'SANTA COMBINHA' ] ] },
        'Ala' => {
          '5340' => [
            [ '011', '014', 'ALA' ] ] },
        'Travanca' => {
          '5340' => [
            [ '295', '296', 'MACEDO DE CAVALEIROS' ] ] },
        'Grij - Gar' => {
          '5340' => [
            [ '151', '153', 'GRIJ DE VALE BENFEITO' ] ] },
        'Cernadela' => {
          '5340' => [
            [ '101', '102', 'CORTIOS' ] ] },
        'Cabanas' => {
          '5340' => [
            [ '431', '433', 'SOUTELO MOURISCO' ] ] },
        'Edroso' => {
          '5340' => [
            [ '120', '120', 'EDROSO' ] ] },
        'Lombo' => {
          '5340' => [
            [ '190', '190', 'LOMBO' ] ] },
        'Bagueixe' => {
          '5340' => [
            [ '040', '040', 'BAGUEIXE' ] ] },
        'Apeadouro de Castelos' => {
          '5340' => [
            [ '081', '082', 'CASTELOS' ] ] },
        'Malta' => {
          '5340' => [
            [ '371', '372', 'OLMOS' ] ] },
        'Burga' => {
          '5340' => [
            [ '060', '060', 'BURGA' ] ] },
        'Peredo' => {
          '5340' => [
            [ '380', '380', 'PEREDO' ] ] }
                },
      'Mirandela' => {
        'Cedanhos' => {
          '5370' => [
            [ '651', '652', 'VALE DE ASNES' ] ] },
        'Quinta das Cerdeiras' => {
          '5370' => [
            [ '559', '565', 'MIRANDELA' ] ] },
        'Abreiro' => {
          '5370' => [
            [ '021', '023', 'ABREIRO' ] ] },
        'Rego de Vide' => {
          '5370' => [
            [ '110', '110', 'COBRO' ] ] },
        'Vila Verde' => {
          '5370' => [
            [ '690', '690', 'VILA VERDE MDL' ] ] },
        'Ervideira' => {
          '5385' => [
            [ '056', '058', 'SO PEDRO VELHO' ] ] },
        'Torre Dona Chama' => {
          '5385' => [
            [ '080', '118', 'TORRE DE DONA CHAMA' ] ] },
        'Boua' => {
          '5385' => [
            [ '031', '033', 'BOUA' ] ] },
        'Freixeda' => {
          '5370' => [
            [ '140', '140', 'FREIXEDA' ] ] },
        'Carvalhais' => {
          '5370' => [
            [ '081', '088', 'CARVALHAIS MDL' ] ] },
        'Alto dos Avidagos' => {
          '5370' => [
            [ '051', '054', 'AVIDAGOS' ] ] },
        'Vale de Lagoa' => {
          '5370' => [
            [ '030', '030', 'ALVITES' ] ] },
        'Franco' => {
          '5370' => [
            [ '120', '120', 'FRANCO' ] ] },
        'Quinta da Gricha' => {
          '5370' => [
            [ '681', '682', 'VILA BOA MDL' ] ] },
        'Valverde da Gestosa' => {
          '5370' => [
            [ '670', '670', 'VALVERDE MDL' ] ] },
        'Fonte da Urze' => {
          '5370' => [
            [ '151', '152', 'LAMAS DE ORELHO' ] ] },
        'Freixedinha' => {
          '5370' => [
            [ '631', '632', 'SO SALVADOR MDL' ] ] },
        'Guide' => {
          '5385' => [
            [ '071', '073', 'TORRE DE DONA CHAMA' ] ] },
        'Fradizela' => {
          '5385' => [
            [ '041', '042', 'FRADIZELA' ] ] },
        'Azenha do Areal' => {
          '5370' => [
            [ '131', '137', 'FRECHAS' ] ] },
        'Longra' => {
          '5370' => [
            [ '060', '060', 'BARCEL' ] ] },
        'Caravelas' => {
          '5370' => [
            [ '075', '075', 'CARAVELAS' ] ] },
        'Valongo das Meadas' => {
          '5370' => [
            [ '070', '070', 'CABANELAS MDL' ] ] },
        'Vale de Couo' => {
          '5370' => [
            [ '620', '620', 'ROMEU' ] ] },
        'Couos' => {
          '5385' => [
            [ '051', '055', 'MRIAS' ] ] },
        'Mirandela' => {
          '5370' => [
            [ '200', '301', 'MIRANDELA' ],
            [ '303', '306', 'MIRANDELA' ],
            [ '308', '322', 'MIRANDELA' ],
            [ '324', '383', 'MIRANDELA' ],
            [ '385', '452', 'MIRANDELA' ],
            [ '454', '481', 'MIRANDELA' ],
            [ '483', '528', 'MIRANDELA' ] ] },
        'Vale de Juncal' => {
          '5370' => [
            [ '010', '010', 'ABAMBRES' ] ] },
        'Passos' => {
          '5370' => [
            [ '602', '602', 'PASSOS MDL' ] ] },
        'Pereira' => {
          '5370' => [
            [ '610', '610', 'PEREIRA MDL' ] ] },
        'Cedes' => {
          '5370' => [
            [ '101', '103', 'CEDES' ] ] },
        'Vale de Salgueiro' => {
          '5370' => [
            [ '660', '660', 'VALE DE SALGUEIRO' ] ] },
        'Vale de Telhas' => {
          '5385' => [
            [ '140', '140', 'VALE DE TELHAS' ] ] },
        'Aguieiras' => {
          '5385' => [
            [ '011', '018', 'AGUIEIRAS' ] ] },
        'Quintas' => {
          '5385' => [
            [ '131', '134', 'VALE DE GOUVINHAS' ] ] },
        'Eivados' => {
          '5370' => [
            [ '641', '644', 'SUES' ] ] },
        'Pousadas' => {
          '5370' => [
            [ '040', '040', 'AVANTOS' ] ] },
        'Bronceda' => {
          '5370' => [
            [ '551', '557', 'MIRANDELA' ] ] },
        'Canto do Adro' => {
          '5370' => [
            [ '171', '176', 'MASCARENHAS' ] ] },
        'So Pedro de Vale do Conde' => {
          '5370' => [
            [ '160', '160', 'MARMELOS' ] ] },
        'Navalho' => {
          '5370' => [
            [ '601', '601', 'NAVALHO' ] ] }
           },
      'Torre de Moncorvo' => {
        'Urros' => {
          '5160' => [
            [ '401', '401', 'URROS' ] ] },
        'Estevais' => {
          '5160' => [
            [ '021', '022', 'ADEGANHA' ] ] },
        'Castedo' => {
          '5160' => [
            [ '071', '071', 'CASTEDO TMC' ] ] },
        'Adeganha' => {
          '5160' => [
            [ '002', '002', 'ADEGANHA' ] ] },
        'Quinta Branca' => {
          '5160' => [
            [ '111', '114', 'LARINHO' ] ] },
        'Larinho' => {
          '5160' => [
            [ '001', '001', 'LARINHO' ] ] },
        'Torre de Moncorvo' => {
          '5160' => [
            [ '003', '003', 'TORRE DE MONCORVO' ],
            [ '200', '223', 'TORRE DE MONCORVO' ],
            [ '226', '303', 'TORRE DE MONCORVO' ] ] },
        'Souto da Velha' => {
          '5160' => [
            [ '171', '171', 'SOUTO DA VELHA' ] ] },
        'Peredo dos Castelhanos' => {
          '5160' => [
            [ '161', '161', 'PEREDO DOS CASTELHANOS' ] ] },
        'Horta da Vilaria' => {
          '5160' => [
            [ '101', '102', 'HORTA DA VILARIA' ] ] },
        'Cabanas de Baixo' => {
          '5160' => [
            [ '031', '035', 'CABEA BOA' ] ] },
        'Ch' => {
          '5160' => [
            [ '131', '134', 'LOUSA TMC' ] ] },
        'Felgar' => {
          '5160' => [
            [ '074', '089', 'FELGAR' ] ] },
        'Sequeiros' => {
          '5160' => [
            [ '011', '011', 'AOREIRA' ] ] },
        'Quinta das Odreiras' => {
          '5160' => [
            [ '151', '152', 'MS TMC' ] ] },
        'Cardanha' => {
          '5160' => [
            [ '041', '041', 'CARDANHA' ] ] },
        'Felgueiras' => {
          '5160' => [
            [ '091', '091', 'FELGUEIRAS TMC' ] ] },
        'Maores' => {
          '5160' => [
            [ '141', '141', 'MAORES' ] ] },
        'Peladinhos' => {
          '5160' => [
            [ '061', '069', 'CARVIAIS' ] ] }
             },
      'Mogadouro' => {
        'Estevais' => {
          '5200' => [
            [ '140', '140', 'CASTELO BRANCO MGD' ] ] },
        'Ventozelo' => {
          '5200' => [
            [ '530', '530', 'VENTOZELO' ] ] },
        'Placas de T' => {
          '5200' => [
            [ '421', '422', 'T' ] ] },
        'So Pedro' => {
          '5200' => [
            [ '160', '160', 'MEIRINHOS' ] ] },
        'Sanhoane' => {
          '5200' => [
            [ '384', '384', 'SANHOANE MGD' ] ] },
        'Sampaio' => {
          '5200' => [
            [ '010', '010', 'AZINHOSO' ] ] },
        'Castanheira' => {
          '5200' => [
            [ '120', '120', 'CASTANHEIRA MGD' ] ] },
        'Remondes' => {
          '5200' => [
            [ '370', '370', 'REMONDES' ] ] },
        'Bru' => {
          '5200' => [
            [ '090', '090', 'BRU' ] ] },
        'Vale da Madre' => {
          '5200' => [
            [ '500', '500', 'VALE DA MADRE' ] ] },
        'Granja' => {
          '5200' => [
            [ '381', '383', 'SALDANHA' ] ] },
        'Penas Roias' => {
          '5200' => [
            [ '311', '313', 'PENAS ROIAS' ] ] },
        'Mogadouro' => {
          '5200' => [
            [ '200', '260', 'MOGADOURO' ],
            [ '262', '284', 'MOGADOURO' ] ] },
        'Vale de Porco' => {
          '5200' => [
            [ '510', '510', 'VALE DE PORCO' ] ] },
        'Brunhosinho' => {
          '5200' => [
            [ '110', '110', 'BRUNHOSINHO' ] ] },
        'Estao Gare' => {
          '5200' => [
            [ '541', '544', 'VILA DE ALA' ] ] },
        'Santo Andr' => {
          '5200' => [
            [ '521', '523', 'VALVERDE MGD' ] ] },
        'Soutelo' => {
          '5200' => [
            [ '410', '410', 'SOUTELO MGD' ] ] },
        'Brunhoso' => {
          '5200' => [
            [ '100', '100', 'BRUNHOSO' ] ] },
        'Figueirinha' => {
          '5200' => [
            [ '451', '452', 'TRAVANCA MGD' ] ] },
        'Vila dos Sinos' => {
          '5200' => [
            [ '571', '572', 'VILARINHO DOS GALEGOS' ] ] },
        'Urrs' => {
          '5200' => [
            [ '461', '484', 'URRS' ] ] },
        'Castro Vicente' => {
          '5350' => [
            [ '201', '204', 'CASTRO VICENTE' ] ] },
        'Quintas Quebradas' => {
          '5200' => [
            [ '150', '150', 'CASTELO BRANCO MGD' ] ] },
        'Salgueiro' => {
          '5200' => [
            [ '300', '300', 'PARADELA MGD' ] ] },
        'Vilar de Rei' => {
          '5200' => [
            [ '560', '560', 'VILAR DE REI' ] ] },
        'Zava' => {
          '5200' => [
            [ '286', '289', 'MOGADOURO' ] ] },
        'Macedo do Peso' => {
          '5200' => [
            [ '401', '404', 'SO MARTINHO DO PESO' ] ] },
        'Bemposta' => {
          '5200' => [
            [ '021', '080', 'BEMPOSTA MGD' ] ] },
        'Castelo Branco' => {
          '5200' => [
            [ '130', '130', 'CASTELO BRANCO MGD' ] ] },
        'Algosinho' => {
          '5200' => [
            [ '351', '352', 'PEREDO DA BEMPOSTA' ] ] }
           },
      'Freixo Espada  Cinta' => {
        'Lagoaa' => {
          '5180' => [
            [ '201', '216', 'LAGOAA' ],
            [ '218', '231', 'LAGOAA' ] ] },
        'Freixo de Espada  Cinta' => {
          '5180' => [
            [ '171', '179', 'FREIXO DE ESPADA  CINTA' ] ] },
        'Adrinho' => {
          '5180' => [
            [ '101', '169', 'FREIXO DE ESPADA  CINTA' ] ] },
        'Ligares' => {
          '5180' => [
            [ '301', '303', 'LIGARES' ] ] },
        'Poiares' => {
          '5180' => [
            [ '340', '363', 'POIARES FEC' ] ] },
        'Mazouco' => {
          '5180' => [
            [ '320', '320', 'MAZOUCO' ] ] },
        'Fornos' => {
          '5180' => [
            [ '011', '037', 'FORNOS FEC' ] ] },
        'Quinta Simes' => {
          '5180' => [
            [ '250', '250', 'LAGOAA' ] ] }
                 },
      'Vimioso' => {
        'Ponte do Moinho' => {
          '5230' => [
            [ '281', '283', 'VIMIOSO' ] ] },
        'Caarelhos' => {
          '5230' => [
            [ '090', '090', 'CAARELHOS' ] ] },
        'Mora' => {
          '5230' => [
            [ '231', '233', 'UVA' ] ] },
        'Ferradal' => {
          '5230' => [
            [ '070', '070', 'ARGOZELO' ] ] },
        'Campo de Vboras' => {
          '5230' => [
            [ '100', '100', 'CAMPO DE VBORAS' ] ] },
        'Algoso' => {
          '5230' => [
            [ '010', '010', 'ALGOSO' ] ] },
        'Santulho' => {
          '5230' => [
            [ '201', '217', 'SANTULHO' ] ] },
        'Argozelo' => {
          '5230' => [
            [ '031', '066', 'ARGOZELO' ] ] },
        'So Joanico' => {
          '5230' => [
            [ '251', '253', 'VALE DE FRADES' ] ] },
        'Vilar Seco' => {
          '5230' => [
            [ '270', '270', 'VILAR SECO VMS' ] ] },
        'Angueira' => {
          '5230' => [
            [ '020', '020', 'ANGUEIRA' ] ] },
        'Avelanoso' => {
          '5230' => [
            [ '080', '080', 'AVELANOSO' ] ] },
        'Avinh' => {
          '5230' => [
            [ '151', '159', 'MATELA VMS' ] ] },
        'Pinelo' => {
          '5230' => [
            [ '181', '182', 'PINELO' ] ] },
        'Caro' => {
          '5230' => [
            [ '121', '135', 'CARO' ] ] },
        'Vimioso' => {
          '5230' => [
            [ '300', '302', 'VIMIOSO' ],
            [ '304', '327', 'VIMIOSO' ] ] }
               },
      'Alfandega da F' => {
        'Sardo' => {
          '5350' => [
            [ '351', '352', 'SENDIM DA RIBEIRA' ] ] },
        'Alfndega da F' => {
          '5350' => [
            [ '001', '051', 'ALFNDEGA DA F' ] ] },
        'Parada' => {
          '5350' => [
            [ '280', '280', 'PARADA AFE' ] ] },
        'Saldonha' => {
          '5350' => [
            [ '300', '300', 'SALDONHA' ] ] },
        'Eucisia' => {
          '5350' => [
            [ '231', '232', 'EUCISIA' ] ] },
        'Cerejais' => {
          '5350' => [
            [ '220', '220', 'CEREJAIS' ] ] },
        'Colmeais' => {
          '5350' => [
            [ '431', '432', 'VILARES DE VILARIA' ] ] },
        'Vilarelhos' => {
          '5350' => [
            [ '420', '420', 'VILARELHOS' ] ] },
        'Vale Pereiro' => {
          '5350' => [
            [ '370', '370', 'VALE PEREIRO' ] ] },
        'Soeima' => {
          '5350' => [
            [ '360', '360', 'SOEIMA' ] ] },
        'Ferradosa' => {
          '5350' => [
            [ '241', '242', 'FERRADOSA' ] ] },
        'Castelo' => {
          '5350' => [
            [ '055', '056', 'ALFNDEGA DA F' ] ] },
        'Gebelim' => {
          '5350' => [
            [ '250', '250', 'GEBELIM' ] ] },
        'Pombal' => {
          '5350' => [
            [ '290', '290', 'POMBAL AFE' ] ] },
        'Legonha' => {
          '5350' => [
            [ '401', '402', 'VILAR CHO AFE' ] ] },
        'Covelas' => {
          '5350' => [
            [ '311', '313', 'SAMBADE' ] ] },
        'Valtelheiro' => {
          '5350' => [
            [ '060', '075', 'ALFNDEGA DA F' ] ] },
        'Valverde' => {
          '5350' => [
            [ '390', '390', 'VALVERDE AFE' ] ] },
        'Vales' => {
          '5350' => [
            [ '380', '380', 'VALES AFE' ] ] },
        'Sendim da Serra' => {
          '5350' => [
            [ '353', '353', 'SENDIM DA SERRA' ] ] },
        'Cabreira' => {
          '5350' => [
            [ '261', '262', 'GOUVEIA AFE' ] ] },
        'Agrobom' => {
          '5350' => [
            [ '101', '102', 'AGROBOM' ] ] }
                 },
      'Vinhais' => {
        'Brito de Lomba' => {
          '5320' => [
            [ '031', '034', 'EDRAL' ] ] },
        'Celas' => {
          '5320' => [
            [ '021', '024', 'CELAS' ] ] },
        'Ousilho' => {
          '5320' => [
            [ '100', '100', 'OUSILHO' ] ] },
        'Prada' => {
          '5320' => [
            [ '221', '222', 'VILA VERDE VNH' ] ] },
        'Ferreiros de Lomba' => {
          '5320' => [
            [ '231', '232', 'VILAR DE LOMBA' ] ] },
        'Moimenta' => {
          '5320' => [
            [ '070', '070', 'MOIMENTA VNH' ] ] },
        'Edrosa' => {
          '5320' => [
            [ '041', '042', 'EDROSA' ] ] },
        'Soeira' => {
          '5320' => [
            [ '170', '170', 'SOEIRA' ] ] },
        'Cisterna' => {
          '5320' => [
            [ '131', '134', 'QUIRS' ] ] },
        'Alvaredos' => {
          '5320' => [
            [ '010', '010', 'ALVAREDOS' ] ] },
        'Nunes' => {
          '5320' => [
            [ '091', '092', 'NUNES' ] ] },
        'Mofreita' => {
          '5320' => [
            [ '060', '060', 'MOFREITA' ] ] },
        'Brito de Baixo' => {
          '5320' => [
            [ '111', '113', 'PENHAS JUNTAS' ] ] },
        'Pinheiro Novo' => {
          '5320' => [
            [ '121', '123', 'PINHEIRO NOVO' ] ] },
        'Santa Cruz' => {
          '5320' => [
            [ '140', '140', 'SANTA CRUZ VNH' ] ] },
        'Barraco das Latas' => {
          '5320' => [
            [ '241', '244', 'VILAR DE OSSOS' ] ] },
        'Cides' => {
          '5320' => [
            [ '251', '252', 'VILAR DE PEREGRINOS' ] ] },
        'Gestosa' => {
          '5320' => [
            [ '261', '263', 'VILAR SECO DE LOMBA' ] ] },
        'Rebordelo' => {
          '5335' => [
            [ '101', '126', 'REBORDELO' ] ] },
        'Vila Boa de Ousilho' => {
          '5320' => [
            [ '210', '210', 'VILA BOA DE OUSILHO' ] ] },
        'Abo' => {
          '5335' => [
            [ '031', '033', 'CANDEDO VNH' ] ] },
        'Vinhais' => {
          '5320' => [
            [ '301', '337', 'VINHAIS' ] ] },
        'Armoniz' => {
          '5320' => [
            [ '271', '279', 'VINHAIS' ] ] },
        'Dine' => {
          '5320' => [
            [ '051', '052', 'FRESULFE' ] ] },
        'Carvalhas' => {
          '5320' => [
            [ '081', '086', 'MONTOUTO' ] ] },
        'Vale de Armeiro' => {
          '5335' => [
            [ '131', '131', 'REBORDELO' ] ] },
        'Maceira' => {
          '5335' => [
            [ '135', '136', 'VALE DE JANEIRO' ] ] },
        'Quintela' => {
          '5320' => [
            [ '101', '102', 'PA' ] ] },
        'Ervedosa' => {
          '5335' => [
            [ '071', '074', 'ERVEDOSA VNH' ] ] },
        'Castro' => {
          '5320' => [
            [ '161', '165', 'SOBREIRO DE BAIXO' ] ] },
        'Travanca' => {
          '5320' => [
            [ '180', '180', 'TRAVANCA VNH' ] ] },
        'Contim' => {
          '5320' => [
            [ '151', '154', 'SANTALHA' ] ] },
        'Curopos' => {
          '5335' => [
            [ '051', '053', 'CUROPOS' ] ] },
        'Cabea da Igreja' => {
          '5320' => [
            [ '191', '198', 'TUIZELO' ] ] },
        'Minas de Ervedosa' => {
          '5335' => [
            [ '132', '134', 'VALE DAS FONTES' ] ] },
        'So Jomil' => {
          '5320' => [
            [ '160', '160', 'SO JOMIL' ] ] },
        'Agrocho' => {
          '5335' => [
            [ '011', '011', 'AGROCHO' ] ] }
               }
      },
    'Aveiro' => {
      'Mealhada' => {
        'Mealhada' => {
          '3050' => [
            [ '007', '010', 'MEALHADA' ],
            [ '321', '382', 'MEALHADA' ] ] },
        'Casal Comba' => {
          '3050' => [
            [ '151', '177', 'CASAL COMBA' ] ] },
        'Lendiosa' => {
          '3050' => [
            [ '181', '187', 'CASAL COMBA' ] ] },
        'Lograssol' => {
          '3050' => [
            [ '504', '511', 'VACARIA' ] ] },
        'Almas do Buaco' => {
          '3050' => [
            [ '201', '209', 'LUSO' ] ] },
        'Luso' => {
          '3050' => [
            [ '221', '247', 'LUSO' ] ] },
        'Canedo' => {
          '3050' => [
            [ '401', '404', 'PAMPILHOSA' ] ] },
        'Antes' => {
          '3050' => [
            [ '011', '046', 'ANTES' ] ] },
        'Cardal' => {
          '3050' => [
            [ '301', '301', 'MEALHADA' ] ] },
        'Arinhos' => {
          '3050' => [
            [ '551', '583', 'VENTOSA DO BAIRRO' ] ] },
        'Carreira' => {
          '3050' => [
            [ '501', '502', 'VACARIA' ] ] },
        'Mata do Bussaco' => {
          '3050' => [
            [ '261', '267', 'LUSO' ] ] },
        'Pvoa do Loureiro' => {
          '3050' => [
            [ '480', '481', 'PAMPILHOSA' ] ] },
        'Ades' => {
          '3050' => [
            [ '071', '095', 'BARCOUO' ] ] },
        'Pampilhosa' => {
          '3050' => [
            [ '410', '457', 'PAMPILHOSA' ] ] },
        'Cavaleiros' => {
          '3050' => [
            [ '100', '106', 'BARCOUO' ] ] },
        'Carqueijo' => {
          '3050' => [
            [ '131', '131', 'CASAL COMBA' ] ] }
              },
      'Ovar' => {
        'Maceda' => {
          '3885' => [
            [ '701', '849', 'MACEDA' ] ] },
        'Ovar' => {
          '3880' => [
            [ '001', '005', 'OVAR' ],
            [ '007', '007', 'OVAR' ],
            [ '009', '017', 'OVAR' ],
            [ '019', '062', 'OVAR' ],
            [ '064', '065', 'OVAR' ],
            [ '067', '083', 'OVAR' ],
            [ '086', '086', 'OVAR' ],
            [ '094', '094', 'OVAR' ],
            [ '096', '097', 'OVAR' ],
            [ '099', '100', 'OVAR' ],
            [ '102', '113', 'OVAR' ],
            [ '118', '134', 'OVAR' ],
            [ '136', '141', 'OVAR' ],
            [ '143', '144', 'OVAR' ],
            [ '146', '166', 'OVAR' ],
            [ '169', '176', 'OVAR' ],
            [ '178', '184', 'OVAR' ],
            [ '186', '186', 'OVAR' ],
            [ '190', '194', 'OVAR' ],
            [ '196', '207', 'OVAR' ],
            [ '209', '215', 'OVAR' ],
            [ '217', '223', 'OVAR' ],
            [ '225', '230', 'OVAR' ],
            [ '232', '232', 'OVAR' ],
            [ '234', '241', 'OVAR' ],
            [ '243', '244', 'OVAR' ],
            [ '246', '253', 'OVAR' ],
            [ '255', '256', 'OVAR' ],
            [ '258', '272', 'OVAR' ],
            [ '274', '277', 'OVAR' ],
            [ '279', '282', 'OVAR' ],
            [ '285', '286', 'OVAR' ],
            [ '288', '308', 'OVAR' ],
            [ '310', '347', 'OVAR' ],
            [ '350', '393', 'OVAR' ],
            [ '714', '714', 'OVAR' ],
            [ '921', '921', 'OVAR' ] ] },
        'Texugos' => {
          '3885' => [
            [ '051', '059', 'ARADA' ] ] },
        'Jogal' => {
          '3885' => [
            [ '171', '174', 'ARADA' ] ] },
        'Cortegaa' => {
          '3885' => [
            [ '201', '229', 'CORTEGAA OVR' ],
            [ '231', '268', 'CORTEGAA OVR' ],
            [ '270', '276', 'CORTEGAA OVR' ],
            [ '278', '342', 'CORTEGAA OVR' ] ] },
        'Quintas de Baixo' => {
          '3885' => [
            [ '176', '187', 'ARADA' ] ] },
        'Juncalinho' => {
          '3885' => [
            [ '081', '086', 'ARADA' ] ] },
        'So Vicente de Pereira Jus' => {
          '3880' => [
            [ '820', '920', 'S VICENTE DE PEREIRA JUS' ] ] },
        'Ameixoeirinha' => {
          '3885' => [
            [ '011', '014', 'ARADA' ] ] },
        'Corgas' => {
          '3885' => [
            [ '116', '119', 'ARADA' ] ] },
        'Esmoriz' => {
          '3885' => [
            [ '401', '569', 'ESMORIZ' ],
            [ '572', '577', 'ESMORIZ' ],
            [ '579', '679', 'ESMORIZ' ],
            [ '681', '691', 'ESMORIZ' ] ] },
        'Cavadas' => {
          '3885' => [
            [ '041', '044', 'ARADA' ] ] },
        'Pedreira' => {
          '3885' => [
            [ '126', '134', 'ARADA' ] ] },
        'Alcapedrinha' => {
          '3885' => [
            [ '046', '049', 'ARADA' ],
            [ '096', '099', 'ARADA' ],
            [ '101', '114', 'ARADA' ] ] },
        'Pedras de Cima' => {
          '3885' => [
            [ '089', '089', 'ARADA' ] ] },
        'Vlega' => {
          '3880' => [
            [ '463', '466', 'VLEGA' ],
            [ '481', '481', 'VLEGA' ],
            [ '485', '485', 'VLEGA' ],
            [ '495', '495', 'VLEGA' ],
            [ '506', '507', 'VLEGA' ],
            [ '510', '514', 'VLEGA' ],
            [ '521', '522', 'VLEGA' ],
            [ '524', '525', 'VLEGA' ],
            [ '528', '528', 'VLEGA' ],
            [ '531', '646', 'VLEGA' ],
            [ '682', '682', 'VLEGA' ] ] },
        'Preguia' => {
          '3885' => [
            [ '036', '039', 'ARADA' ],
            [ '121', '124', 'ARADA' ],
            [ '166', '166', 'ARADA' ],
            [ '169', '169', 'ARADA' ] ] },
        'Olho Marinho' => {
          '3885' => [
            [ '002', '004', 'ARADA' ],
            [ '016', '034', 'ARADA' ],
            [ '091', '094', 'ARADA' ] ] },
        'Salgueiral de Cima' => {
          '3880' => [
            [ '702', '711', 'SO JOO OVR' ] ] },
        'So Joo' => {
          '3880' => [
            [ '008', '008', 'SO JOO OVR' ],
            [ '085', '085', 'SO JOO OVR' ],
            [ '101', '101', 'SO JOO OVR' ],
            [ '177', '177', 'SO JOO OVR' ],
            [ '187', '188', 'SO JOO OVR' ],
            [ '257', '257', 'SO JOO OVR' ],
            [ '697', '700', 'SO JOO OVR' ],
            [ '713', '713', 'SO JOO OVR' ],
            [ '715', '726', 'SO JOO OVR' ],
            [ '728', '744', 'SO JOO OVR' ],
            [ '746', '771', 'SO JOO OVR' ],
            [ '774', '797', 'SO JOO OVR' ],
            [ '799', '808', 'SO JOO OVR' ] ] },
        'Pedras de Baixo' => {
          '3885' => [
            [ '006', '009', 'ARADA' ],
            [ '141', '164', 'ARADA' ] ] },
        'Murteira' => {
          '3885' => [
            [ '061', '079', 'ARADA' ],
            [ '136', '139', 'ARADA' ] ] }
          },
      'Murtosa' => {
        'Urbanizao Lopes da Cunha' => {
          '3870' => [
            [ '380', '380', 'TORREIRA' ] ] },
        'Tomadia' => {
          '3870' => [
            [ '100', '268', 'MURTOSA' ] ] },
        'Urbanizao Raul dos Santos' => {
          '3870' => [
            [ '390', '405', 'TORREIRA' ] ] },
        'Muranzel' => {
          '3870' => [
            [ '301', '304', 'TORREIRA' ] ] },
        'Torreira' => {
          '3870' => [
            [ '320', '336', 'TORREIRA' ],
            [ '340', '365', 'TORREIRA' ] ] },
        'Monte' => {
          '3870' => [
            [ '271', '295', 'MURTOSA' ] ] },
        'Agro' => {
          '3870' => [
            [ '011', '063', 'BUNHEIRO' ] ] }
             },
      'Oliveira de Azemeis' => {
        'Costa M' => {
          '3720' => [
            [ '683', '757', 'SO ROQUE' ] ] },
        'guas Frreas' => {
          '3700' => [
            [ '701', '723', 'MACIEIRA DE SARNES' ] ] },
        'Vale de Madeiros' => {
          '3720' => [
            [ '428', '430', 'PALMAZ' ] ] },
        'Areal' => {
          '3720' => [
            [ '111', '123', 'MADAIL' ] ] },
        'Casal Marinho' => {
          '3700' => [
            [ '667', '680', 'FAJES' ],
            [ '700', '700', 'FAJES' ] ] },
        'Camum' => {
          '3700' => [
            [ '601', '632', 'CESAR' ] ] },
        'Areosa' => {
          '3720' => [
            [ '461', '487', 'PINHEIRO DA BEMPOSTA' ] ] },
        'Oliveira de Azemis' => {
          '3720' => [
            [ '001', '001', 'OLIVEIRA DE AZEMIS' ],
            [ '090', '090', 'OLIVEIRA DE AZEMIS' ],
            [ '200', '279', 'OLIVEIRA DE AZEMIS' ],
            [ '281', '312', 'OLIVEIRA DE AZEMIS' ],
            [ '314', '319', 'OLIVEIRA DE AZEMIS' ] ] },
        'Abolembra' => {
          '3720' => [
            [ '531', '551', 'SO MARTINHO DA GNDARA' ] ] },
        'Vila Ch' => {
          '3720' => [
            [ '495', '500', 'SO ROQUE' ] ] },
        'Casal' => {
          '3720' => [
            [ '414', '416', 'PALMAZ' ] ] },
        'Cucujes' => {
          '3720' => [
            [ '350', '410', 'VILA DE CUCUJES' ],
            [ '760', '905', 'VILA DE CUCUJES' ] ] },
        'Bustelo' => {
          '3720' => [
            [ '607', '620', 'SO ROQUE' ],
            [ '623', '629', 'SO ROQUE' ],
            [ '631', '631', 'SO ROQUE' ],
            [ '633', '633', 'SO ROQUE' ],
            [ '635', '635', 'SO ROQUE' ],
            [ '638', '650', 'SO ROQUE' ] ] },
        'Cruz' => {
          '3700' => [
            [ '801', '807', 'FAJES' ] ] },
        'Igreja' => {
          '3720' => [
            [ '579', '602', 'UL' ] ] },
        'Gndara' => {
          '3700' => [
            [ '724', '729', 'CESAR' ] ] },
        'Monte' => {
          '3700' => [
            [ '634', '647', 'CESAR' ] ] },
        'Retorta' => {
          '3700' => [
            [ '378', '387', 'FAJES' ],
            [ '580', '580', 'FAJES' ] ] },
        'Alumieira' => {
          '3720' => [
            [ '051', '076', 'LOUREIRO OAZ' ] ] },
        'Souto' => {
          '3720' => [
            [ '604', '606', 'UL' ] ] },
        'Alvies' => {
          '3720' => [
            [ '411', '412', 'PALMAZ' ] ] },
        'Moinho do Meio' => {
          '3720' => [
            [ '088', '088', 'OLIVEIRA DE AZEMIS' ] ] },
        'Aguincheira' => {
          '3720' => [
            [ '501', '510', 'SANTIAGO DA RIBA-UL' ] ] },
        'Cimo de Vila' => {
          '3700' => [
            [ '750', '759', 'NOGUEIRA DO CRAVO OAZ' ],
            [ '774', '794', 'NOGUEIRA DO CRAVO OAZ' ] ] },
        'Aido de Alm' => {
          '3720' => [
            [ '441', '456', 'PINDELO' ] ] },
        'Ribeira Verde' => {
          '3700' => [
            [ '796', '800', 'NOGUEIRA DO CRAVO OAZ' ] ] },
        'Arroteia' => {
          '3700' => [
            [ '761', '771', 'NOGUEIRA DO CRAVO OAZ' ] ] },
        'Zona Industrial de Fajes' => {
          '3700' => [
            [ '650', '662', 'FAJES' ] ] },
        'Mouro' => {
          '3720' => [
            [ '089', '089', 'MACINHATA DA SEIXA' ] ] },
        'Alvelhe' => {
          '3720' => [
            [ '171', '192', 'OSSELA' ] ] },
        'Torre' => {
          '3700' => [
            [ '682', '691', 'FAJES' ] ] },
        'Alto do Monte' => {
          '3720' => [
            [ '081', '087', 'MACINHATA DA SEIXA' ] ] },
        'Cavadas' => {
          '3700' => [
            [ '351', '376', 'FAJES' ] ] },
        'Outeiro' => {
          '3720' => [
            [ '514', '526', 'SANTIAGO DA RIBA-UL' ] ] },
        'Carreira' => {
          '3720' => [
            [ '002', '005', 'OSSELA' ] ] },
        'Piso' => {
          '3700' => [
            [ '906', '907', 'FAJES' ] ] },
        'Abelheira' => {
          '3720' => [
            [ '141', '153', 'OLIVEIRA DE AZEMIS' ] ] },
        'Silvares' => {
          '3720' => [
            [ '091', '102', 'MACINHATA DA SEIXA' ] ] },
        'Cabo da Aldeia' => {
          '3700' => [
            [ '693', '698', 'FAJES' ] ] },
        'Arrifaninha' => {
          '3720' => [
            [ '011', '035', 'CARREGOSA' ] ] },
        'Souto da Costa' => {
          '3700' => [
            [ '595', '597', 'FAJES' ],
            [ '599', '600', 'FAJES' ] ] },
        'Barbeito' => {
          '3700' => [
            [ '664', '665', 'FAJES' ] ] },
        'So Mamede' => {
          '3700' => [
            [ '582', '585', 'FAJES' ],
            [ '587', '590', 'FAJES' ],
            [ '592', '593', 'FAJES' ] ] },
        'Quint' => {
          '3720' => [
            [ '573', '575', 'TRAVANCA OAZ' ] ] },
        'Avelo' => {
          '3720' => [
            [ '554', '571', 'TRAVANCA OAZ' ] ] },
        'M' => {
          '3720' => [
            [ '420', '426', 'PALMAZ' ] ] }
             },
      'Aveiro' => {
        'Horta' => {
          '3800' => [
            [ '713', '722', 'EIXO' ] ] },
        'Eirol' => {
          '3800' => [
            [ '681', '703', 'EIROL' ],
            [ '705', '709', 'EIROL' ],
            [ '712', '712', 'EIROL' ] ] },
        'Costa do Valado' => {
          '3810' => [
            [ '791', '794', 'OLIVEIRINHA' ],
            [ '796', '830', 'OLIVEIRINHA' ] ] },
        'Vessada' => {
          '3810' => [
            [ '605', '612', 'NARIZ' ] ] },
        'So Jacinto' => {
          '3800' => [
            [ '901', '905', 'SO JACINTO' ] ] },
        'Carregal' => {
          '3800' => [
            [ '851', '886', 'REQUEIXO' ] ] },
        'Azurva' => {
          '3800' => [
            [ '749', '752', 'EIXO' ],
            [ '754', '759', 'EIXO' ],
            [ '761', '808', 'EIXO' ] ] },
        'Quintas' => {
          '3810' => [
            [ '833', '889', 'OLIVEIRINHA' ] ] },
        'Aveiro' => {
          '3800' => [
            [ '001', '058', 'AVEIRO' ],
            [ '060', '184', 'AVEIRO' ],
            [ '186', '227', 'AVEIRO' ],
            [ '232', '263', 'AVEIRO' ],
            [ '265', '315', 'AVEIRO' ],
            [ '317', '384', 'AVEIRO' ],
            [ '386', '386', 'AVEIRO' ],
            [ '388', '393', 'AVEIRO' ],
            [ '395', '395', 'AVEIRO' ],
            [ '400', '409', 'AVEIRO' ],
            [ '411', '423', 'AVEIRO' ],
            [ '520', '520', 'AVEIRO' ],
            [ '555', '555', 'AVEIRO' ] ],
          '3814' => [
            [ '501', '505', 'AVEIRO' ] ],
          '3804' => [
            [ '501', '507', 'AVEIRO' ] ],
          '3810' => [
            [ '001', '161', 'AVEIRO' ],
            [ '163', '329', 'AVEIRO' ],
            [ '331', '365', 'AVEIRO' ],
            [ '367', '402', 'AVEIRO' ],
            [ '404', '494', 'AVEIRO' ],
            [ '496', '504', 'AVEIRO' ],
            [ '508', '508', 'AVEIRO' ],
            [ '510', '512', 'AVEIRO' ],
            [ '514', '528', 'AVEIRO' ],
            [ '530', '541', 'AVEIRO' ],
            [ '543', '545', 'AVEIRO' ],
            [ '547', '551', 'AVEIRO' ],
            [ '604', '604', 'AVEIRO' ],
            [ '831', '832', 'AVEIRO' ] ] },
        'Pvoa do Pao' => {
          '3800' => [
            [ '556', '624', 'CACIA' ] ] },
        'Montes de Azurva' => {
          '3800' => [
            [ '731', '747', 'EIXO' ] ] },
        'Almiar' => {
          '3800' => [
            [ '724', '724', 'EIROL' ] ] },
        'Nariz' => {
          '3810' => [
            [ '559', '603', 'NARIZ' ] ] },
        'Cabeo' => {
          '3800' => [
            [ '521', '554', 'CACIA' ] ] },
        'Mamodeiro' => {
          '3810' => [
            [ '731', '783', 'NOSSA SENHORA DE FTIMA' ] ] },
        'Eixo' => {
          '3800' => [
            [ '810', '814', 'EIXO' ] ] }
            },
      'Albergaria-a-Velha' => {
        'Ameal' => {
          '3850' => [
            [ '360', '362', 'ALQUERUBIM' ] ] },
        'Angeja' => {
          '3850' => [
            [ '401', '418', 'ANGEJA' ],
            [ '420', '421', 'ANGEJA' ],
            [ '423', '433', 'ANGEJA' ],
            [ '435', '441', 'ANGEJA' ],
            [ '443', '452', 'ANGEJA' ],
            [ '454', '455', 'ANGEJA' ],
            [ '458', '458', 'ANGEJA' ],
            [ '460', '460', 'ANGEJA' ],
            [ '463', '464', 'ANGEJA' ] ] },
        'Vale da Cana' => {
          '3850' => [
            [ '660', '663', 'FROSSOS' ] ] },
        'Albergaria-a-Nova' => {
          '3850' => [
            [ '501', '501', 'BRANCA ALB' ] ] },
        'Telhadela' => {
          '3850' => [
            [ '714', '718', 'RIBEIRA DE FRGUAS' ] ] },
        'Valmaior' => {
          '3850' => [
            [ '835', '836', 'VALMAIOR' ] ] },
        'Frias' => {
          '3850' => [
            [ '233', '235', 'ALBERGARIA-A-VELHA' ] ] },
        'So Joo de Loure' => {
          '3850' => [
            [ '771', '791', 'SO JOO DE LOURE' ],
            [ '793', '805', 'SO JOO DE LOURE' ] ] },
        'Carvalhais' => {
          '3850' => [
            [ '562', '562', 'BRANCA ALB' ] ] },
        'Albergaria-a-Velha' => {
          '3850' => [
            [ '001', '012', 'ALBERGARIA-A-VELHA' ],
            [ '014', '017', 'ALBERGARIA-A-VELHA' ],
            [ '019', '022', 'ALBERGARIA-A-VELHA' ],
            [ '024', '025', 'ALBERGARIA-A-VELHA' ],
            [ '027', '027', 'ALBERGARIA-A-VELHA' ],
            [ '030', '031', 'ALBERGARIA-A-VELHA' ],
            [ '034', '044', 'ALBERGARIA-A-VELHA' ],
            [ '048', '054', 'ALBERGARIA-A-VELHA' ],
            [ '057', '057', 'ALBERGARIA-A-VELHA' ],
            [ '061', '071', 'ALBERGARIA-A-VELHA' ],
            [ '073', '081', 'ALBERGARIA-A-VELHA' ],
            [ '084', '085', 'ALBERGARIA-A-VELHA' ],
            [ '089', '096', 'ALBERGARIA-A-VELHA' ],
            [ '101', '101', 'ALBERGARIA-A-VELHA' ],
            [ '103', '104', 'ALBERGARIA-A-VELHA' ],
            [ '109', '109', 'ALBERGARIA-A-VELHA' ],
            [ '111', '111', 'ALBERGARIA-A-VELHA' ],
            [ '113', '113', 'ALBERGARIA-A-VELHA' ],
            [ '115', '116', 'ALBERGARIA-A-VELHA' ],
            [ '118', '118', 'ALBERGARIA-A-VELHA' ],
            [ '120', '121', 'ALBERGARIA-A-VELHA' ],
            [ '124', '127', 'ALBERGARIA-A-VELHA' ],
            [ '129', '135', 'ALBERGARIA-A-VELHA' ],
            [ '137', '138', 'ALBERGARIA-A-VELHA' ],
            [ '140', '143', 'ALBERGARIA-A-VELHA' ],
            [ '145', '145', 'ALBERGARIA-A-VELHA' ],
            [ '147', '149', 'ALBERGARIA-A-VELHA' ],
            [ '151', '155', 'ALBERGARIA-A-VELHA' ],
            [ '157', '159', 'ALBERGARIA-A-VELHA' ],
            [ '161', '166', 'ALBERGARIA-A-VELHA' ],
            [ '170', '178', 'ALBERGARIA-A-VELHA' ],
            [ '182', '182', 'ALBERGARIA-A-VELHA' ],
            [ '184', '184', 'ALBERGARIA-A-VELHA' ],
            [ '200', '200', 'ALBERGARIA-A-VELHA' ],
            [ '202', '203', 'ALBERGARIA-A-VELHA' ],
            [ '205', '205', 'ALBERGARIA-A-VELHA' ],
            [ '207', '209', 'ALBERGARIA-A-VELHA' ],
            [ '211', '211', 'ALBERGARIA-A-VELHA' ],
            [ '213', '215', 'ALBERGARIA-A-VELHA' ],
            [ '217', '217', 'ALBERGARIA-A-VELHA' ],
            [ '219', '219', 'ALBERGARIA-A-VELHA' ],
            [ '221', '221', 'ALBERGARIA-A-VELHA' ],
            [ '223', '223', 'ALBERGARIA-A-VELHA' ],
            [ '225', '226', 'ALBERGARIA-A-VELHA' ],
            [ '230', '231', 'ALBERGARIA-A-VELHA' ],
            [ '237', '237', 'ALBERGARIA-A-VELHA' ],
            [ '240', '242', 'ALBERGARIA-A-VELHA' ],
            [ '837', '837', 'ALBERGARIA-A-VELHA' ] ] },
        'Frossos' => {
          '3850' => [
            [ '610', '612', 'FROSSOS' ],
            [ '614', '617', 'FROSSOS' ],
            [ '620', '626', 'FROSSOS' ],
            [ '628', '633', 'FROSSOS' ],
            [ '635', '647', 'FROSSOS' ] ] },
        'Pardos' => {
          '3850' => [
            [ '367', '368', 'ALQUERUBIM' ] ] },
        'Azenhas' => {
          '3850' => [
            [ '751', '753', 'SO JOO DE LOURE' ] ] },
        'Boavista' => {
          '3850' => [
            [ '480', '485', 'ANGEJA' ] ] },
        'Cancela' => {
          '3850' => [
            [ '560', '560', 'BRANCA ALB' ] ] },
        'Cruzinha' => {
          '3850' => [
            [ '824', '833', 'VALMAIOR' ] ] },
        'Branca' => {
          '3850' => [
            [ '511', '511', 'BRANCA ALB' ],
            [ '513', '519', 'BRANCA ALB' ],
            [ '522', '523', 'BRANCA ALB' ],
            [ '525', '526', 'BRANCA ALB' ],
            [ '529', '531', 'BRANCA ALB' ],
            [ '533', '535', 'BRANCA ALB' ],
            [ '537', '538', 'BRANCA ALB' ],
            [ '540', '540', 'BRANCA ALB' ],
            [ '544', '544', 'BRANCA ALB' ],
            [ '547', '547', 'BRANCA ALB' ] ] },
        'Casaldima' => {
          '3850' => [
            [ '564', '567', 'BRANCA ALB' ] ] },
        'Barroca' => {
          '3850' => [
            [ '503', '503', 'BRANCA ALB' ] ] },
        'Eiras' => {
          '3850' => [
            [ '569', '589', 'BRANCA ALB' ] ] },
        'Sobreiro' => {
          '3850' => [
            [ '251', '253', 'ALBERGARIA-A-VELHA' ],
            [ '255', '270', 'ALBERGARIA-A-VELHA' ],
            [ '272', '273', 'ALBERGARIA-A-VELHA' ],
            [ '275', '275', 'ALBERGARIA-A-VELHA' ] ] },
        'Aores' => {
          '3850' => [
            [ '821', '822', 'VALMAIOR' ] ] },
        'Loure' => {
          '3850' => [
            [ '755', '758', 'SO JOO DE LOURE' ] ] },
        'Fial' => {
          '3850' => [
            [ '364', '365', 'ALQUERUBIM' ] ] },
        'Alto dos Barreiros' => {
          '3850' => [
            [ '701', '712', 'RIBEIRA DE FRGUAS' ] ] },
        'Alquerubim' => {
          '3850' => [
            [ '301', '303', 'ALQUERUBIM' ],
            [ '309', '309', 'ALQUERUBIM' ],
            [ '312', '312', 'ALQUERUBIM' ],
            [ '316', '316', 'ALQUERUBIM' ],
            [ '324', '324', 'ALQUERUBIM' ],
            [ '331', '331', 'ALQUERUBIM' ],
            [ '335', '336', 'ALQUERUBIM' ],
            [ '339', '339', 'ALQUERUBIM' ],
            [ '344', '344', 'ALQUERUBIM' ],
            [ '347', '349', 'ALQUERUBIM' ],
            [ '351', '351', 'ALQUERUBIM' ],
            [ '354', '356', 'ALQUERUBIM' ] ] },
        'Urgueiras' => {
          '3850' => [
            [ '280', '280', 'ALBERGARIA-A-VELHA' ] ] }
            },
      'Oliveira do Bairro' => {
        'Recamonde' => {
          '3770' => [
            [ '301', '305', 'OLIVEIRA DO BAIRRO' ] ] },
        'Pedreira' => {
          '3770' => [
            [ '061', '067', 'OI' ] ] },
        'Caneira' => {
          '3770' => [
            [ '031', '038', 'MAMARROSA' ] ] },
        'Agras de Cima' => {
          '3770' => [
            [ '051', '059', 'OI' ] ] },
        'Oi' => {
          '3770' => [
            [ '865', '865', 'OI' ] ] },
        'Carvalho' => {
          '3770' => [
            [ '401', '411', 'TROVISCAL OBR' ] ] },
        'Albergue' => {
          '3770' => [
            [ '351', '357', 'PALHAA' ] ] },
        'Azurveira' => {
          '3770' => [
            [ '011', '018', 'BUSTOS' ] ] },
        'Oliveira do Bairro' => {
          '3770' => [
            [ '201', '234', 'OLIVEIRA DO BAIRRO' ],
            [ '851', '853', 'OLIVEIRA DO BAIRRO' ],
            [ '855', '864', 'OLIVEIRA DO BAIRRO' ] ] },
        'guas Empoas' => {
          '3770' => [
            [ '101', '108', 'OLIVEIRA DO BAIRRO' ] ] }
            },
      'Estarreja' => {
        'Estarreja' => {
          '3864' => [
            [ '001', '004', 'ESTARREJA' ] ],
          '3860' => [
            [ '201', '273', 'ESTARREJA' ],
            [ '275', '281', 'ESTARREJA' ],
            [ '283', '341', 'ESTARREJA' ],
            [ '344', '345', 'ESTARREJA' ],
            [ '351', '392', 'ESTARREJA' ],
            [ '395', '404', 'ESTARREJA' ],
            [ '529', '529', 'ESTARREJA' ],
            [ '674', '674', 'ESTARREJA' ],
            [ '676', '680', 'ESTARREJA' ] ] },
        'Salreu' => {
          '3865' => [
            [ '188', '294', 'SALREU' ],
            [ '302', '302', 'SALREU' ],
            [ '304', '309', 'SALREU' ],
            [ '311', '332', 'SALREU' ],
            [ '335', '335', 'SALREU' ],
            [ '337', '340', 'SALREU' ] ] },
        'Fermel' => {
          '3865' => [
            [ '101', '146', 'FERMEL' ],
            [ '295', '300', 'FERMEL' ] ] },
        'Veiros' => {
          '3860' => [
            [ '274', '274', 'VEIROS ETR' ],
            [ '282', '282', 'VEIROS ETR' ],
            [ '342', '343', 'VEIROS ETR' ],
            [ '346', '350', 'VEIROS ETR' ],
            [ '394', '394', 'VEIROS ETR' ],
            [ '585', '595', 'VEIROS ETR' ],
            [ '601', '668', 'VEIROS ETR' ],
            [ '675', '675', 'VEIROS ETR' ],
            [ '750', '750', 'VEIROS ETR' ] ] },
        'Canelas' => {
          '3865' => [
            [ '001', '042', 'CANELAS ETR' ],
            [ '301', '301', 'CANELAS ETR' ] ] },
        'Pardilh' => {
          '3860' => [
            [ '421', '528', 'PARDILH' ],
            [ '530', '544', 'PARDILH' ],
            [ '681', '727', 'PARDILH' ],
            [ '729', '738', 'PARDILH' ],
            [ '740', '740', 'PARDILH' ] ] },
        'Avanca' => {
          '3860' => [
            [ '001', '148', 'AVANCA' ],
            [ '670', '673', 'AVANCA' ] ] }
               },
      'So Joo da Madeira' => {
        'So Joo da Madeira' => {
          '3700' => [
            [ '010', '149', 'SO JOO DA MADEIRA' ],
            [ '151', '179', 'SO JOO DA MADEIRA' ],
            [ '181', '215', 'SO JOO DA MADEIRA' ],
            [ '217', '238', 'SO JOO DA MADEIRA' ],
            [ '240', '243', 'SO JOO DA MADEIRA' ],
            [ '246', '293', 'SO JOO DA MADEIRA' ],
            [ '295', '310', 'SO JOO DA MADEIRA' ],
            [ '312', '312', 'SO JOO DA MADEIRA' ],
            [ '314', '346', 'SO JOO DA MADEIRA' ] ],
          '3701' => [
            [ '501', '501', 'SO JOO DA MADEIRA' ] ] }
             },
      'Espinho' => {
        'Espinho' => {
          '4504' => [
            [ '851', '856', 'ESPINHO' ] ],
          '4500' => [
            [ '001', '011', 'ESPINHO' ],
            [ '014', '014', 'ESPINHO' ],
            [ '019', '019', 'ESPINHO' ],
            [ '021', '082', 'ESPINHO' ],
            [ '084', '100', 'ESPINHO' ],
            [ '103', '105', 'ESPINHO' ],
            [ '108', '130', 'ESPINHO' ],
            [ '132', '156', 'ESPINHO' ],
            [ '158', '169', 'ESPINHO' ],
            [ '171', '172', 'ESPINHO' ],
            [ '174', '178', 'ESPINHO' ],
            [ '180', '182', 'ESPINHO' ],
            [ '184', '186', 'ESPINHO' ],
            [ '188', '201', 'ESPINHO' ],
            [ '203', '203', 'ESPINHO' ],
            [ '205', '208', 'ESPINHO' ],
            [ '211', '211', 'ESPINHO' ],
            [ '214', '214', 'ESPINHO' ],
            [ '216', '259', 'ESPINHO' ],
            [ '261', '261', 'ESPINHO' ],
            [ '263', '293', 'ESPINHO' ],
            [ '296', '306', 'ESPINHO' ],
            [ '308', '309', 'ESPINHO' ],
            [ '311', '311', 'ESPINHO' ],
            [ '314', '323', 'ESPINHO' ],
            [ '326', '329', 'ESPINHO' ],
            [ '331', '333', 'ESPINHO' ],
            [ '335', '335', 'ESPINHO' ],
            [ '337', '339', 'ESPINHO' ],
            [ '341', '346', 'ESPINHO' ],
            [ '348', '350', 'ESPINHO' ],
            [ '352', '353', 'ESPINHO' ],
            [ '355', '370', 'ESPINHO' ],
            [ '372', '372', 'ESPINHO' ],
            [ '374', '376', 'ESPINHO' ],
            [ '378', '381', 'ESPINHO' ],
            [ '383', '387', 'ESPINHO' ],
            [ '389', '389', 'ESPINHO' ],
            [ '391', '391', 'ESPINHO' ],
            [ '395', '434', 'ESPINHO' ],
            [ '436', '452', 'ESPINHO' ],
            [ '454', '502', 'ESPINHO' ],
            [ '504', '513', 'ESPINHO' ],
            [ '515', '574', 'ESPINHO' ],
            [ '576', '600', 'ESPINHO' ],
            [ '604', '606', 'ESPINHO' ],
            [ '608', '608', 'ESPINHO' ],
            [ '613', '620', 'ESPINHO' ],
            [ '622', '657', 'ESPINHO' ],
            [ '659', '690', 'ESPINHO' ],
            [ '801', '808', 'ESPINHO' ],
            [ '813', '813', 'ESPINHO' ],
            [ '815', '815', 'ESPINHO' ],
            [ '817', '817', 'ESPINHO' ] ] }
             },
      'Vale de Cambra' => {
        'Agros' => {
          '3730' => [
            [ '160', '178', 'JUNQUEIRA VLC' ] ] },
        'Batalha' => {
          '3730' => [
            [ '100', '110', 'CEPELOS' ] ] },
        'Ares' => {
          '3730' => [
            [ '001', '015', 'ARES' ] ] },
        'Campo Grande' => {
          '3730' => [
            [ '400', '415', 'VALE DE CAMBRA' ] ] },
        'Vale de Cambra' => {
          '3730' => [
            [ '200', '205', 'VALE DE CAMBRA' ],
            [ '220', '226', 'VALE DE CAMBRA' ],
            [ '240', '323', 'VALE DE CAMBRA' ] ] },
        'Calada' => {
          '3730' => [
            [ '430', '437', 'VILA COVA DE PERRINHO' ] ] },
        'Agras' => {
          '3730' => [
            [ '130', '148', 'CODAL' ] ] },
        'Cancelo' => {
          '3730' => [
            [ '371', '387', 'ROGE' ] ] },
        'Relva' => {
          '3730' => [
            [ '029', '094', 'VALE DE CAMBRA' ] ] }
              },
      'Anadia' => {
        'Avels de Cima' => {
          '3780' => [
            [ '401', '415', 'AVELS DE CIMA' ] ] },
        'Azenha' => {
          '3780' => [
            [ '581', '599', 'VILARINHO DO BAIRRO' ] ] },
        'Curia' => {
          '3780' => [
            [ '541', '544', 'TAMENGOS' ] ] },
        'Bicarenho' => {
          '3780' => [
            [ '521', '529', 'SANGALHOS' ] ] },
        'Cabana' => {
          '3780' => [
            [ '171', '181', 'SO LOURENO DO BAIRRO' ] ] },
        'Alfolas' => {
          '3780' => [
            [ '290', '295', 'ANADIA' ] ] },
        'Vidoeiro' => {
          '3780' => [
            [ '150', '150', 'SANGALHOS' ] ] },
        'Algeriz' => {
          '3780' => [
            [ '561', '566', 'VILA NOVA DE MONSARROS' ] ] },
        'Avels de Caminho' => {
          '3780' => [
            [ '351', '351', 'AVELS DE CAMINHO' ] ] },
        'Amieiro' => {
          '3780' => [
            [ '471', '482', 'MOITA AND' ] ] },
        'S' => {
          '3780' => [
            [ '101', '102', 'SANGALHOS' ] ] },
        'Canha' => {
          '3780' => [
            [ '451', '454', 'MOGOFORES' ] ] },
        'So Joo da Azenha' => {
          '3780' => [
            [ '140', '140', 'SANGALHOS' ] ] },
        'Aldeamento de So Paulo' => {
          '3780' => [
            [ '501', '503', 'IS DO BAIRRO' ] ] },
        'Ancas' => {
          '3780' => [
            [ '051', '051', 'ANCAS' ] ] },
        'Anadia' => {
          '3780' => [
            [ '201', '243', 'ANADIA' ] ] },
        'Sangalhos' => {
          '3780' => [
            [ '111', '135', 'SANGALHOS' ] ] },
        'Aguim' => {
          '3780' => [
            [ '621', '623', 'AGUIM' ] ] },
        'Paredes do Bairro' => {
          '3780' => [
            [ '611', '612', 'PAREDES DO BAIRRO' ] ] },
        'Amoreira da Gndara' => {
          '3780' => [
            [ '011', '018', 'AMOREIRA DA GNDARA' ] ] }
            },
        'lhavo' => {
        'Gafanha da Nazar' => {
          '3830' => [
            [ '551', '609', 'GAFANHA DA NAZAR' ],
            [ '611', '611', 'GAFANHA DA NAZAR' ],
            [ '616', '785', 'GAFANHA DA NAZAR' ] ] },
        'Gafanha do Carmo' => {
          '3830' => [
            [ '401', '425', 'GAFANHA DO CARMO' ] ] },
        'lhavo' => {
          '3830' => [
            [ '167', '203', 'LHAVO' ],
            [ '205', '303', 'LHAVO' ] ] },
        'Ervosas' => {
          '3830' => [
            [ '001', '165', 'LHAVO' ] ] },
        'Costa Nova do Prado' => {
          '3830' => [
            [ '451', '526', 'GAFANHA DA ENCARNAO' ] ] }
            },
      'Vagos' => {
        'Sosa' => {
          '3840' => [
            [ '340', '342', 'SOSA' ] ] },
        'Canto' => {
          '3840' => [
            [ '061', '074', 'CALVO VGS' ] ] },
        'Ervedal' => {
          '3840' => [
            [ '551', '551', 'SANTO ANDR DE VAGOS' ] ] },
        'Praia da Vagueira' => {
          '3840' => [
            [ '271', '286', 'GAFANHA DA BOA HORA' ] ] },
        'Canto de Baixo' => {
          '3840' => [
            [ '321', '322', 'PONTE DE VAGOS' ] ] },
        'Cabeos' => {
          '3840' => [
            [ '121', '133', 'COVO DO LOBO' ] ] },
        'Calvo' => {
          '3840' => [
            [ '041', '053', 'CALVO VGS' ] ] },
        'Cabeo do Barro' => {
          '3840' => [
            [ '501', '507', 'SANTO ANTNIO DE VAGOS' ] ] },
        'Ponte de Vagos' => {
          '3840' => [
            [ '324', '326', 'PONTE DE VAGOS' ] ] },
        'Gndara' => {
          '3840' => [
            [ '182', '187', 'FONTE DE ANGEO' ] ] },
        'Ladeira do Mano' => {
          '3840' => [
            [ '381', '386', 'VAGOS' ] ] },
        'Parada de Cima' => {
          '3840' => [
            [ '191', '212', 'FONTE DE ANGEO' ] ] },
        'Fonte de Angeo' => {
          '3840' => [
            [ '161', '172', 'FONTE DE ANGEO' ] ] },
        'Santo Andr' => {
          '3840' => [
            [ '553', '560', 'SANTO ANDR DE VAGOS' ] ] },
        'Choca do Mar' => {
          '3840' => [
            [ '081', '089', 'CALVO VGS' ] ] },
        'Mesas' => {
          '3840' => [
            [ '571', '572', 'SANTA CATARINA VGS' ] ] },
        'Gafanha do Areo' => {
          '3840' => [
            [ '265', '265', 'GAFANHA DA BOA HORA' ] ] },
        'Cais do Areo' => {
          '3840' => [
            [ '251', '262', 'GAFANHA DA BOA HORA' ] ] },
        'Rines' => {
          '3840' => [
            [ '220', '221', 'FONTE DE ANGEO' ] ] },
        'Junco do Bico' => {
          '3840' => [
            [ '100', '101', 'CALVO VGS' ] ] },
        'Vagos' => {
          '3840' => [
            [ '001', '001', 'VAGOS' ],
            [ '401', '480', 'VAGOS' ] ] },
        'Lavandeira' => {
          '3840' => [
            [ '344', '379', 'SOSA' ] ] },
        'Carregosa' => {
          '3840' => [
            [ '301', '304', 'OUCA' ] ] },
        'Vale' => {
          '3840' => [
            [ '328', '328', 'PONTE DE VAGOS' ] ] },
        'Cabecinhas' => {
          '3840' => [
            [ '011', '032', 'CALVO VGS' ] ] },
        'Gafanha da Vagueira' => {
          '3840' => [
            [ '288', '288', 'GAFANHA DA BOA HORA' ] ] }
           },
      'Castelo de Paiva' => {
        'Santa Maria de Sardoura' => {
          '4550' => [
            [ '235', '235', 'SANTA MARIA DE SARDOURA' ],
            [ '731', '772', 'SANTA MARIA DE SARDOURA' ],
            [ '808', '808', 'SANTA MARIA DE SARDOURA' ],
            [ '832', '832', 'SANTA MARIA DE SARDOURA' ],
            [ '852', '852', 'SANTA MARIA DE SARDOURA' ],
            [ '854', '855', 'SANTA MARIA DE SARDOURA' ],
            [ '858', '858', 'SANTA MARIA DE SARDOURA' ] ] },
        'Castelo de Paiva' => {
          '4550' => [
            [ '100', '105', 'CASTELO DE PAIVA' ],
            [ '107', '112', 'CASTELO DE PAIVA' ],
            [ '116', '129', 'CASTELO DE PAIVA' ],
            [ '131', '132', 'CASTELO DE PAIVA' ],
            [ '134', '137', 'CASTELO DE PAIVA' ],
            [ '139', '143', 'CASTELO DE PAIVA' ],
            [ '146', '162', 'CASTELO DE PAIVA' ],
            [ '203', '203', 'CASTELO DE PAIVA' ],
            [ '205', '206', 'CASTELO DE PAIVA' ],
            [ '211', '213', 'CASTELO DE PAIVA' ],
            [ '215', '215', 'CASTELO DE PAIVA' ],
            [ '218', '222', 'CASTELO DE PAIVA' ],
            [ '225', '226', 'CASTELO DE PAIVA' ],
            [ '228', '228', 'CASTELO DE PAIVA' ],
            [ '230', '230', 'CASTELO DE PAIVA' ],
            [ '234', '234', 'CASTELO DE PAIVA' ],
            [ '236', '236', 'CASTELO DE PAIVA' ],
            [ '238', '240', 'CASTELO DE PAIVA' ],
            [ '242', '245', 'CASTELO DE PAIVA' ],
            [ '248', '248', 'CASTELO DE PAIVA' ],
            [ '251', '252', 'CASTELO DE PAIVA' ],
            [ '257', '257', 'CASTELO DE PAIVA' ],
            [ '261', '262', 'CASTELO DE PAIVA' ],
            [ '264', '268', 'CASTELO DE PAIVA' ],
            [ '271', '271', 'CASTELO DE PAIVA' ],
            [ '274', '274', 'CASTELO DE PAIVA' ],
            [ '276', '280', 'CASTELO DE PAIVA' ],
            [ '282', '282', 'CASTELO DE PAIVA' ],
            [ '285', '285', 'CASTELO DE PAIVA' ] ] },
        'Bairros' => {
          '4550' => [
            [ '001', '002', 'BAIRROS' ],
            [ '004', '069', 'BAIRROS' ],
            [ '071', '096', 'BAIRROS' ] ] },
        'Pedorido' => {
          '4550' => [
            [ '501', '528', 'PEDORIDO' ],
            [ '530', '541', 'PEDORIDO' ],
            [ '543', '544', 'PEDORIDO' ],
            [ '546', '555', 'PEDORIDO' ] ] },
        'Paraso' => {
          '4550' => [
            [ '451', '460', 'PARASO' ],
            [ '462', '462', 'PARASO' ],
            [ '464', '471', 'PARASO' ] ] },
        'Fornos' => {
          '4550' => [
            [ '352', '403', 'FORNOS CPV' ],
            [ '405', '416', 'FORNOS CPV' ] ] },
        'So Martinho de Sardoura' => {
          '4550' => [
            [ '233', '233', 'SO MARTINHO DE SARDOURA' ],
            [ '790', '802', 'SO MARTINHO DE SARDOURA' ],
            [ '804', '805', 'SO MARTINHO DE SARDOURA' ],
            [ '807', '807', 'SO MARTINHO DE SARDOURA' ],
            [ '810', '810', 'SO MARTINHO DE SARDOURA' ],
            [ '813', '815', 'SO MARTINHO DE SARDOURA' ],
            [ '819', '819', 'SO MARTINHO DE SARDOURA' ],
            [ '822', '829', 'SO MARTINHO DE SARDOURA' ],
            [ '831', '831', 'SO MARTINHO DE SARDOURA' ],
            [ '834', '834', 'SO MARTINHO DE SARDOURA' ],
            [ '836', '838', 'SO MARTINHO DE SARDOURA' ],
            [ '840', '840', 'SO MARTINHO DE SARDOURA' ],
            [ '842', '844', 'SO MARTINHO DE SARDOURA' ],
            [ '846', '846', 'SO MARTINHO DE SARDOURA' ],
            [ '849', '849', 'SO MARTINHO DE SARDOURA' ],
            [ '851', '851', 'SO MARTINHO DE SARDOURA' ],
            [ '853', '853', 'SO MARTINHO DE SARDOURA' ],
            [ '856', '857', 'SO MARTINHO DE SARDOURA' ],
            [ '859', '863', 'SO MARTINHO DE SARDOURA' ],
            [ '865', '902', 'SO MARTINHO DE SARDOURA' ] ] },
        'Raiva' => {
          '4550' => [
            [ '247', '247', 'RAIVA' ],
            [ '571', '581', 'RAIVA' ],
            [ '583', '598', 'RAIVA' ],
            [ '600', '604', 'RAIVA' ],
            [ '606', '613', 'RAIVA' ],
            [ '615', '616', 'RAIVA' ],
            [ '618', '631', 'RAIVA' ] ] },
        'Real' => {
          '4550' => [
            [ '250', '250', 'REAL CPV' ],
            [ '286', '298', 'REAL CPV' ],
            [ '300', '313', 'REAL CPV' ],
            [ '315', '317', 'REAL CPV' ],
            [ '319', '332', 'REAL CPV' ],
            [ '463', '463', 'REAL CPV' ],
            [ '472', '472', 'REAL CPV' ],
            [ '652', '678', 'REAL CPV' ],
            [ '680', '680', 'REAL CPV' ],
            [ '682', '694', 'REAL CPV' ],
            [ '697', '699', 'REAL CPV' ],
            [ '701', '707', 'REAL CPV' ] ] }
                },
        'gueda' => {
        'Barrosa' => {
          '3750' => [
            [ '671', '690', 'PRSTIMO' ] ] },
        'Aguieira' => {
          '3750' => [
            [ '802', '838', 'VALONGO DO VOUGA' ] ] },
        'Avelal Baixo' => {
          '3750' => [
            [ '371', '380', 'CASTANHEIRA DO VOUGA' ] ] },
        'Casainho de Baixo' => {
          '3750' => [
            [ '401', '406', 'ESPINHEL' ] ] },
        'Alombada' => {
          '3750' => [
            [ '581', '604', 'MACINHATA DO VOUGA' ] ] },
        'Alvarim' => {
          '3750' => [
            [ '361', '364', 'BELAZAIMA DO CHO' ] ] },
        'Lamas Vouga' => {
          '3750' => [
            [ '551', '552', 'LAMAS DO VOUGA' ] ] },
        'Alagoa' => {
          '3750' => [
            [ '301', '327', 'GUEDA' ] ] },
        'Aguada de Baixo' => {
          '3750' => [
            [ '031', '035', 'AGUADA DE BAIXO' ] ] },
        'Ois da Ribeira' => {
          '3750' => [
            [ '650', '650', 'IS DA RIBEIRA' ] ] },
        'Barro' => {
          '3750' => [
            [ '351', '353', 'BARR AGD' ] ] },
        'Almear' => {
          '3750' => [
            [ '751', '755', 'TRAVASS' ] ] },
        'Aguada de Cima' => {
          '3750' => [
            [ '041', '070', 'AGUADA DE CIMA' ] ] },
        'Catraia da Borralha' => {
          '3750' => [
            [ '861', '865', 'BORRALHA' ] ] },
        'Macieira de Alcoba' => {
          '3750' => [
            [ '561', '562', 'MACIEIRA DE ALCOBA' ] ] },
        'Alcafaz' => {
          '3750' => [
            [ '011', '019', 'AGADO' ] ] },
        'Brejo' => {
          '3750' => [
            [ '711', '727', 'RECARDES' ] ] },
        'A-dos-Ferreiros' => {
          '3750' => [
            [ '801', '801', 'PRSTIMO' ] ] },
        'gueda' => {
          '3750' => [
            [ '101', '148', 'GUEDA' ],
            [ '150', '181', 'GUEDA' ] ] },
        'Ameinho' => {
          '3750' => [
            [ '851', '859', 'BORRALHA' ] ] },
        'Alto Gestal' => {
          '3750' => [
            [ '771', '794', 'TROFA AGD' ] ] },
        'Fontinha' => {
          '3750' => [
            [ '741', '743', 'SEGADES' ] ] },
        'Fermentelos' => {
          '3750' => [
            [ '421', '497', 'FERMENTELOS' ] ] }
            },
      'Sever do Vouga' => {
        'Arrta' => {
          '3740' => [
            [ '201', '234', 'SEVER DO VOUGA' ] ] },
        'Abobareira' => {
          '3740' => [
            [ '171', '186', 'ROCAS DO VOUGA' ] ] },
        'Alto de Santa Lzia' => {
          '3740' => [
            [ '101', '136', 'PESSEGUEIRO DO VOUGA' ] ] },
        'Arcela' => {
          '3740' => [
            [ '061', '082', 'PARADELA SVV' ] ] },
        'Arestal' => {
          '3740' => [
            [ '416', '426', 'DORNELAS SVV' ] ] },
        'Angios' => {
          '3740' => [
            [ '311', '344', 'SILVA ESCURA SVV' ] ] },
        'Vale da Grama' => {
          '3740' => [
            [ '290', '295', 'SEVER DO VOUGA' ] ] },
        'Boua' => {
          '3740' => [
            [ '011', '023', 'CEDRIM' ] ] },
        'Sever do Vouga' => {
          '3740' => [
            [ '251', '273', 'SEVER DO VOUGA' ] ] },
        'Alinhada' => {
          '3740' => [
            [ '031', '043', 'COUTO DE ESTEVES' ] ] },
        'Arcas' => {
          '3740' => [
            [ '401', '415', 'TALHADAS' ] ] }
              },
      'Arouca' => {
        'Adaufe' => {
          '4540' => [
            [ '431', '457', 'MOLDES' ] ] },
        'Abrunhal' => {
          '4540' => [
            [ '361', '389', 'FERMEDO' ] ] },
        'Adros' => {
          '4540' => [
            [ '511', '559', 'SANTA EULLIA ARC' ] ] },
        'Cortegaa' => {
          '4540' => [
            [ '401', '405', 'JANARDE' ] ] },
        'Albergaria da Serra' => {
          '4540' => [
            [ '011', '014', 'ALBERGARIA DA SERRA' ] ] },
        'Arreais' => {
          '4540' => [
            [ '341', '349', 'ESPIUNCA' ] ] },
        'Cabreira' => {
          '4540' => [
            [ '170', '186', 'AROUCA' ] ] },
        'Alagoa' => {
          '4540' => [
            [ '471', '499', 'ROSSAS ARC' ] ] },
        'Baa' => {
          '4540' => [
            [ '571', '586', 'SO MIGUEL DO MATO ARC' ] ] },
        'Cancelos' => {
          '4540' => [
            [ '671', '687', 'VRZEA ARC' ] ] },
        'Alhavaite' => {
          '4540' => [
            [ '201', '229', 'BURGO' ] ] },
        'Covelo de Paiv' => {
          '4540' => [
            [ '281', '284', 'COVELO DE PAIV' ] ] },
        'Aldeia' => {
          '4540' => [
            [ '601', '627', 'TROPEO' ] ] },
        'Cabreiros' => {
          '4540' => [
            [ '241', '244', 'CABREIROS ARC' ] ] },
        'Abelheira' => {
          '4540' => [
            [ '291', '323', 'ESCARIZ' ] ] },
        'Forno Telheiro' => {
          '4540' => [
            [ '647', '659', 'URR ARC' ] ] },
        'Barbeito' => {
          '4540' => [
            [ '261', '274', 'CHAVE' ] ] },
        'Canelas de Baixo' => {
          '4540' => [
            [ '251', '256', 'CANELAS ARC' ] ] },
        'Albisqueiros' => {
          '4540' => [
            [ '021', '058', 'ALVARENGA ARC' ] ] },
        'Arouca' => {
          '4540' => [
            [ '098', '148', 'AROUCA' ] ] },
        'Agras' => {
          '4540' => [
            [ '411', '423', 'MANSORES' ] ] },
        'Bugiganga' => {
          '4540' => [
            [ '641', '645', 'URR ARC' ] ] }
            },
      'Santa Maria da Feira' => {
        'Areal' => {
          '4520' => [
            [ '605', '624', 'SO JOO DE VER' ] ] },
        'Airas' => {
          '4505' => [
            [ '684', '691', 'CALDAS DE SO JORGE' ] ] },
        'Monte Calvo' => {
          '3700' => [
            [ '898', '905', 'ROMARIZ' ] ] },
        'Escapes' => {
          '4520' => [
            [ '011', '015', 'ESCAPES' ],
            [ '018', '030', 'ESCAPES' ],
            [ '035', '037', 'ESCAPES' ] ] },
        'Fornos' => {
          '4520' => [
            [ '312', '320', 'FORNOS VFR' ] ] },
        'Alm do Rio' => {
          '4520' => [
            [ '101', '123', 'ESPARGO' ] ] },
        'Romariz' => {
          '3700' => [
            [ '808', '896', 'ROMARIZ' ] ] },
        'Vila Boa' => {
          '4520' => [
            [ '283', '285', 'SANTA MARIA DA FEIRA' ] ] },
        'Paos de Brando' => {
          '4535' => [
            [ '264', '307', 'PAOS DE BRANDO' ],
            [ '309', '339', 'PAOS DE BRANDO' ],
            [ '512', '525', 'PAOS DE BRANDO' ] ] },
        'Lourosa' => {
          '4535' => [
            [ '001', '006', 'LOUROSA' ],
            [ '008', '061', 'LOUROSA' ],
            [ '063', '072', 'LOUROSA' ],
            [ '074', '090', 'LOUROSA' ],
            [ '092', '115', 'LOUROSA' ],
            [ '117', '155', 'LOUROSA' ],
            [ '526', '531', 'LOUROSA' ] ] },
        'Azevedo' => {
          '4525' => [
            [ '301', '309', 'GIO VFR' ] ] },
        'Canedo' => {
          '4525' => [
            [ '009', '198', 'CANEDO VFR' ],
            [ '200', '293', 'CANEDO VFR' ] ] },
        'Arrifana' => {
          '3700' => [
            [ '401', '403', 'ARRIFANA VFR' ],
            [ '408', '409', 'ARRIFANA VFR' ],
            [ '411', '574', 'ARRIFANA VFR' ] ] },
        'Vila Maior' => {
          '4525' => [
            [ '480', '482', 'VILA MAIOR VFR' ],
            [ '484', '507', 'VILA MAIOR VFR' ],
            [ '509', '539', 'VILA MAIOR VFR' ] ] },
        'Bairro Santo Andr' => {
          '4520' => [
            [ '151', '151', 'SANTA MARIA DA FEIRA' ] ] },
        'Casais' => {
          '3700' => [
            [ '731', '749', 'MILHEIRS DE POIARES' ] ] },
        'Nogueira da Regedoura' => {
          '4500' => [
            [ '695', '769', 'NOGUEIRA DA REGEDOURA' ],
            [ '771', '800', 'NOGUEIRA DA REGEDOURA' ] ] },
        'Santa Maria de Lamas' => {
          '4535' => [
            [ '340', '345', 'SANTA MARIA DE LAMAS' ],
            [ '347', '366', 'SANTA MARIA DE LAMAS' ],
            [ '368', '372', 'SANTA MARIA DE LAMAS' ],
            [ '374', '378', 'SANTA MARIA DE LAMAS' ],
            [ '380', '388', 'SANTA MARIA DE LAMAS' ],
            [ '390', '406', 'SANTA MARIA DE LAMAS' ],
            [ '408', '427', 'SANTA MARIA DE LAMAS' ],
            [ '508', '508', 'SANTA MARIA DE LAMAS' ] ] },
        'Guisande' => {
          '4525' => [
            [ '310', '350', 'GUISANDE VFR' ],
            [ '551', '555', 'GUISANDE VFR' ] ] },
        'Mozelos' => {
          '4535' => [
            [ '156', '195', 'MOZELOS VFR' ],
            [ '197', '221', 'MOZELOS VFR' ],
            [ '223', '263', 'MOZELOS VFR' ],
            [ '367', '367', 'MOZELOS VFR' ],
            [ '407', '407', 'MOZELOS VFR' ],
            [ '509', '511', 'MOZELOS VFR' ],
            [ '532', '541', 'MOZELOS VFR' ],
            [ '568', '568', 'MOZELOS VFR' ] ] },
        'Louredo' => {
          '4525' => [
            [ '351', '402', 'LOUREDO VFR' ],
            [ '556', '565', 'LOUREDO VFR' ] ] },
        'Oleiros' => {
          '4535' => [
            [ '428', '431', 'SO PAIO DE OLEIROS' ],
            [ '437', '507', 'SO PAIO DE OLEIROS' ],
            [ '542', '566', 'SO PAIO DE OLEIROS' ] ] },
        'Areeiro' => {
          '4520' => [
            [ '301', '310', 'FORNOS VFR' ] ] },
        'Aldeia' => {
          '4520' => [
            [ '501', '511', 'SANFINS VFR' ],
            [ '701', '719', 'SOUTO VFR' ] ],
          '4505' => [
            [ '565', '577', 'PIGEIROS' ] ] },
        'Agoncida' => {
          '4520' => [
            [ '401', '413', 'MOSTEIR' ] ] },
        'Argoncilhe' => {
          '4505' => [
            [ '001', '005', 'ARGONCILHE' ],
            [ '007', '010', 'ARGONCILHE' ],
            [ '013', '018', 'ARGONCILHE' ],
            [ '020', '039', 'ARGONCILHE' ],
            [ '041', '050', 'ARGONCILHE' ],
            [ '052', '056', 'ARGONCILHE' ],
            [ '058', '076', 'ARGONCILHE' ],
            [ '078', '078', 'ARGONCILHE' ],
            [ '080', '088', 'ARGONCILHE' ],
            [ '090', '120', 'ARGONCILHE' ],
            [ '122', '130', 'ARGONCILHE' ],
            [ '132', '153', 'ARGONCILHE' ],
            [ '155', '157', 'ARGONCILHE' ],
            [ '159', '177', 'ARGONCILHE' ],
            [ '179', '182', 'ARGONCILHE' ],
            [ '184', '198', 'ARGONCILHE' ],
            [ '200', '215', 'ARGONCILHE' ],
            [ '218', '218', 'ARGONCILHE' ],
            [ '692', '712', 'ARGONCILHE' ],
            [ '714', '718', 'ARGONCILHE' ],
            [ '726', '734', 'ARGONCILHE' ],
            [ '834', '834', 'ARGONCILHE' ] ] },
        'Lavoura' => {
          '4505' => [
            [ '800', '800', 'PIGEIROS' ] ] },
        'Aldo' => {
          '4520' => [
            [ '801', '824', 'TRAVANCA VFR' ] ] },
        'Sanfins' => {
          '4520' => [
            [ '521', '553', 'SANFINS VFR' ] ] },
        'Prpria' => {
          '4520' => [
            [ '600', '603', 'SO JOO DE VER' ] ] },
        'Lobo' => {
          '4505' => [
            [ '422', '422', 'LOBO' ],
            [ '424', '529', 'LOBO' ],
            [ '531', '564', 'LOBO' ],
            [ '828', '833', 'LOBO' ],
            [ '836', '836', 'LOBO' ] ] },
        'Alpossos' => {
          '4520' => [
            [ '451', '488', 'RIO MEO' ] ] },
        'Santa Maria da Feira' => {
          '4520' => [
            [ '259', '262', 'SANTA MARIA DA FEIRA' ],
            [ '264', '280', 'SANTA MARIA DA FEIRA' ],
            [ '287', '289', 'SANTA MARIA DA FEIRA' ] ] },
        'Vale' => {
          '4525' => [
            [ '403', '479', 'VALE VFR' ],
            [ '550', '550', 'VALE VFR' ] ] },
        'Sanguedo' => {
          '4505' => [
            [ '578', '660', 'SANGUEDO' ],
            [ '772', '799', 'SANGUEDO' ],
            [ '801', '817', 'SANGUEDO' ] ] },
        'Europarque' => {
          '4520' => [
            [ '153', '257', 'SANTA MARIA DA FEIRA' ] ] },
        'Vinhais' => {
          '4520' => [
            [ '322', '326', 'FORNOS VFR' ] ] },
        'Fies' => {
          '4505' => [
            [ '220', '269', 'FIES VFR' ],
            [ '271', '315', 'FIES VFR' ],
            [ '317', '325', 'FIES VFR' ],
            [ '327', '354', 'FIES VFR' ],
            [ '356', '369', 'FIES VFR' ],
            [ '371', '381', 'FIES VFR' ],
            [ '383', '385', 'FIES VFR' ],
            [ '387', '388', 'FIES VFR' ],
            [ '390', '400', 'FIES VFR' ],
            [ '402', '417', 'FIES VFR' ],
            [ '419', '421', 'FIES VFR' ],
            [ '423', '423', 'FIES VFR' ],
            [ '719', '725', 'FIES VFR' ],
            [ '818', '827', 'FIES VFR' ],
            [ '835', '835', 'FIES VFR' ] ] }
              }
          },
    'Ilha de So Miguel' => {
      'Ribeira Grande' => {
        'Ribeira Grande' => {
          '9600' => [
            [ '500', '505', 'RIBEIRA GRANDE' ],
            [ '507', '586', 'RIBEIRA GRANDE' ] ] },
        'Pico da Pedra' => {
          '9600' => [
            [ '051', '081', 'PICO DA PEDRA' ] ] },
        'So Pedro' => {
          '9625' => [
            [ '380', '380', 'MAIA (SO MIGUEL)' ] ] },
        'Lomba de Santa Brbara' => {
          '9600' => [
            [ '401', '401', 'SANTA BRBARA RGR' ] ] },
        'Calo de Baixo' => {
          '9625' => [
            [ '201', '206', 'LOMBA DE SO PEDRO' ] ] },
        'Calhetas' => {
          '9600' => [
            [ '011', '022', 'CALHETAS' ] ] },
        'Maia' => {
          '9625' => [
            [ '321', '337', 'MAIA (SO MIGUEL)' ],
            [ '339', '345', 'MAIA (SO MIGUEL)' ],
            [ '347', '352', 'MAIA (SO MIGUEL)' ] ] },
        'Santa Brbara' => {
          '9600' => [
            [ '410', '425', 'SANTA BRBARA RGR' ] ] },
        'Fenais da Ajuda' => {
          '9625' => [
            [ '021', '035', 'FENAIS DA AJUDA' ] ] },
        'Areal de Santa Barbara' => {
          '9600' => [
            [ '201', '201', 'RIBEIRA SECA RGR' ] ] },
        'Porto Formoso' => {
          '9625' => [
            [ '410', '433', 'PORTO FORMOSO' ] ] },
        'Burguete' => {
          '9625' => [
            [ '101', '101', 'LOMBA DA MAIA' ] ] },
        'Ribeira Seca' => {
          '9600' => [
            [ '211', '236', 'RIBEIRA SECA RGR' ] ] },
        'Gramas de Baixo' => {
          '9600' => [
            [ '301', '302', 'RIBEIRINHA RGR' ] ] },
        'Calos da Maia' => {
          '9625' => [
            [ '301', '305', 'MAIA (SO MIGUEL)' ] ] },
        'Casas Novas' => {
          '9625' => [
            [ '501', '502', 'SO BRS RGR' ] ] },
        'So Brs' => {
          '9625' => [
            [ '510', '510', 'SO BRS RGR' ] ] },
        'Ribeirinha' => {
          '9600' => [
            [ '310', '332', 'RIBEIRINHA RGR' ] ] },
        'Rabo de Peixe' => {
          '9600' => [
            [ '082', '125', 'RABO DE PEIXE' ],
            [ '127', '145', 'RABO DE PEIXE' ] ] },
        'Casa do Monte' => {
          '9625' => [
            [ '011', '014', 'FENAIS DA AJUDA' ] ] },
        'Lomba da Maia' => {
          '9625' => [
            [ '111', '127', 'LOMBA DA MAIA' ] ] },
        'Ribeira Funda' => {
          '9625' => [
            [ '060', '060', 'FENAIS DA AJUDA' ] ] },
        'So Sebastio' => {
          '9625' => [
            [ '391', '391', 'MAIA (SO MIGUEL)' ] ] },
        'Ribeira dos Lagos' => {
          '9625' => [
            [ '050', '050', 'FENAIS DA AJUDA' ] ] },
        'Casas da Ponte' => {
          '9625' => [
            [ '400', '401', 'PORTO FORMOSO' ] ] }
              },
      'Vila Franca do Campo' => {
        'Ponte Nova' => {
          '9680' => [
            [ '355', '355', 'GUA DE ALTO' ] ] },
        'Vila Franca do Campo' => {
          '9680' => [
            [ '101', '196', 'VILA FRANCA DO CAMPO' ] ] },
        'Santana' => {
          '9680' => [
            [ '060', '060', 'VILA FRANCA DO CAMPO' ] ] },
        'Ribeira Seca' => {
          '9680' => [
            [ '015', '029', 'VILA FRANCA DO CAMPO' ] ] },
        'Porto Tagarete' => {
          '9680' => [
            [ '011', '012', 'VILA FRANCA DO CAMPO' ] ] },
        'gua de Alto' => {
          '9680' => [
            [ '301', '324', 'GUA DE ALTO' ] ] },
        'Ponta da Gara' => {
          '9680' => [
            [ '420', '469', 'PONTA GARA' ] ] },
        'Ribeira das Tanhas' => {
          '9680' => [
            [ '501', '520', 'RIBEIRA DAS TAINHAS' ] ] },
        'Praia' => {
          '9680' => [
            [ '360', '360', 'GUA DE ALTO' ] ] },
        'Ponte da Praia' => {
          '9680' => [
            [ '350', '350', 'GUA DE ALTO' ] ] },
        'Bairro Meio Moio' => {
          '9680' => [
            [ '410', '410', 'PONTA GARA' ] ] },
        'Praia gua D\'Alto' => {
          '9680' => [
            [ '365', '365', 'GUA DE ALTO' ] ] },
        'Ribeira da Praia' => {
          '9680' => [
            [ '370', '370', 'GUA DE ALTO' ] ] }
                    },
      'Povoao' => {
        'Lomba do Boto' => {
          '9650' => [
            [ '310', '310', 'POVOAO' ] ] },
        'Lomba do Carro' => {
          '9650' => [
            [ '320', '320', 'POVOAO' ] ] },
        'Faj do Calhau' => {
          '9650' => [
            [ '060', '060', 'GUA RETORTA' ] ] },
        'Ribeira Quente' => {
          '9675' => [
            [ '161', '176', 'RIBEIRA QUENTE' ] ] },
        'Lomba do Louo' => {
          '9650' => [
            [ '241', '251', 'N SENHORA DOS REMDIOS' ] ] },
        'Povoao' => {
          '9650' => [
            [ '401', '430', 'POVOAO' ] ] },
        'gua Retorta' => {
          '9650' => [
            [ '011', '025', 'GUA RETORTA' ] ] },
        'Lomba do Cavaleiro' => {
          '9650' => [
            [ '330', '330', 'POVOAO' ] ] },
        'Fagundas' => {
          '9650' => [
            [ '050', '050', 'GUA RETORTA' ] ] },
        'Fogo' => {
          '9675' => [
            [ '130', '142', 'RIBEIRA QUENTE' ] ] },
        'Burguete' => {
          '9650' => [
            [ '101', '118', 'FAIAL DA TERRA' ] ] },
        'Lomba do Alcaide' => {
          '9650' => [
            [ '211', '215', 'N SENHORA DOS REMDIOS' ],
            [ '217', '219', 'N SENHORA DOS REMDIOS' ] ] },
        'Bairro Alto' => {
          '9675' => [
            [ '110', '110', 'RIBEIRA QUENTE' ] ] },
        'Furnas' => {
          '9675' => [
            [ '021', '053', 'FURNAS' ],
            [ '055', '067', 'FURNAS' ] ] },
        'Lomba dos Ps' => {
          '9650' => [
            [ '350', '350', 'POVOAO' ] ] },
        'Beira Mar' => {
          '9675' => [
            [ '120', '120', 'RIBEIRA QUENTE' ] ] },
        'Lomba do Pomar' => {
          '9650' => [
            [ '340', '340', 'POVOAO' ] ] },
        'Arrebento das Pimentas' => {
          '9675' => [
            [ '011', '012', 'FURNAS' ] ] },
        'Outeiro do Burguete' => {
          '9650' => [
            [ '150', '150', 'FAIAL DA TERRA' ] ] },
        'Lagoa das Furnas' => {
          '9675' => [
            [ '090', '090', 'FURNAS' ] ] },
        'Sanguinho' => {
          '9650' => [
            [ '160', '160', 'FAIAL DA TERRA' ] ] }
              },
      'Lagoa (So Miguel)' => {
        'Atalhada' => {
          '9560' => [
            [ '401', '420', 'LAGOA (SO MIGUEL)' ] ] },
        'Foral Novo' => {
          '9560' => [
            [ '280', '280', 'GUA DE PAU' ] ] },
        'Pico da Forca' => {
          '9560' => [
            [ '290', '290', 'GUA DE PAU' ] ] },
        'Ribeira Ch' => {
          '9560' => [
            [ '550', '550', 'RIBEIRA CH' ] ] },
        'Cabouco' => {
          '9560' => [
            [ '311', '314', 'CABOUCO' ] ] },
        'Areeiro' => {
          '9560' => [
            [ '301', '309', 'CABOUCO' ] ] },
        'Socas' => {
          '9560' => [
            [ '450', '450', 'LAGOA (SO MIGUEL)' ] ] },
        'Lagoa' => {
          '9560' => [
            [ '010', '091', 'LAGOA (SO MIGUEL)' ],
            [ '093', '140', 'LAGOA (SO MIGUEL)' ] ] },
        'Malaca de Baixo' => {
          '9560' => [
            [ '350', '350', 'CABOUCO' ] ] },
        'Remdios' => {
          '9560' => [
            [ '501', '502', 'LAGOA (SO MIGUEL)' ] ] },
        'gua de Pau' => {
          '9560' => [
            [ '201', '246', 'GUA DE PAU' ] ] },
        'Fischer' => {
          '9560' => [
            [ '340', '340', 'CABOUCO' ] ] },
        'Malaca de Cima' => {
          '9560' => [
            [ '360', '360', 'CABOUCO' ] ] },
        'Amoreirinha' => {
          '9560' => [
            [ '270', '270', 'GUA DE PAU' ] ] }
                  },
      'Ponta Delgada' => {
        'Ajuda' => {
          '9545' => [
            [ '011', '035', 'BRETANHA' ] ] },
        'Mosteiros' => {
          '9555' => [
            [ '141', '161', 'MOSTEIROS PDL' ] ] },
        'Canadas' => {
          '9555' => [
            [ '011', '025', 'CANDELRIA PDL' ] ] },
        'Lombas' => {
          '9555' => [
            [ '130', '132', 'MOSTEIROS PDL' ] ] },
        'Santo Antnio' => {
          '9545' => [
            [ '401', '439', 'SANTO ANTNIO PDL' ] ] },
        'Seara' => {
          '9555' => [
            [ '191', '204', 'SETE CIDADES' ] ] },
        'Ponta Delgada' => {
          '9504' => [
            [ '501', '514', 'PONTA DELGADA' ],
            [ '516', '536', 'PONTA DELGADA' ],
            [ '542', '545', 'PONTA DELGADA' ] ],
          '9500' => [
            [ '010', '117', 'PONTA DELGADA' ],
            [ '119', '119', 'PONTA DELGADA' ],
            [ '150', '155', 'PONTA DELGADA' ],
            [ '157', '162', 'PONTA DELGADA' ],
            [ '165', '199', 'PONTA DELGADA' ],
            [ '202', '209', 'PONTA DELGADA' ],
            [ '211', '213', 'PONTA DELGADA' ],
            [ '216', '217', 'PONTA DELGADA' ],
            [ '221', '226', 'PONTA DELGADA' ],
            [ '229', '231', 'PONTA DELGADA' ],
            [ '233', '234', 'PONTA DELGADA' ],
            [ '236', '253', 'PONTA DELGADA' ],
            [ '291', '350', 'PONTA DELGADA' ],
            [ '767', '787', 'PONTA DELGADA' ],
            [ '789', '793', 'PONTA DELGADA' ] ] },
        'Relva' => {
          '9500' => [
            [ '651', '653', 'PONTA DELGADA' ],
            [ '655', '657', 'PONTA DELGADA' ],
            [ '659', '676', 'PONTA DELGADA' ],
            [ '680', '682', 'PONTA DELGADA' ] ] },
        'Arrifes' => {
          '9500' => [
            [ '361', '384', 'PONTA DELGADA' ] ] },
        'Lombinha de Cima' => {
          '9555' => [
            [ '100', '103', 'GINETES' ] ] },
        'Santa Brbara' => {
          '9545' => [
            [ '311', '333', 'SANTA BRBARA PDL' ],
            [ '335', '341', 'SANTA BRBARA PDL' ] ] },
        'Lombinha de Baixo' => {
          '9555' => [
            [ '091', '093', 'GINETES' ] ] },
        'Socorro' => {
          '9555' => [
            [ '045', '047', 'CANDELRIA PDL' ] ] },
        'Pilar' => {
          '9545' => [
            [ '080', '080', 'BRETANHA' ] ] },
        'Faj de Cima' => {
          '9500' => [
            [ '501', '501', 'PONTA DELGADA' ],
            [ '503', '513', 'PONTA DELGADA' ] ] },
        'Joo Bom' => {
          '9545' => [
            [ '051', '069', 'BRETANHA' ] ] },
        'Feteiras' => {
          '9500' => [
            [ '551', '559', 'PONTA DELGADA' ] ] },
        'Ginetes' => {
          '9555' => [
            [ '060', '076', 'GINETES' ] ] },
        'Aflitos' => {
          '9545' => [
            [ '201', '202', 'FENAIS DA LUZ' ] ] },
        'Serto' => {
          '9545' => [
            [ '450', '450', 'SANTO ANTNIO PDL' ] ] },
        'Lomba dos Gagos' => {
          '9555' => [
            [ '080', '081', 'GINETES' ] ] },
        'Covoada' => {
          '9500' => [
            [ '401', '401', 'PONTA DELGADA' ],
            [ '404', '406', 'PONTA DELGADA' ] ] },
        'So Roque' => {
          '9500' => [
            [ '711', '717', 'PONTA DELGADA' ],
            [ '719', '720', 'PONTA DELGADA' ] ] },
        'Cho das Machadas' => {
          '9555' => [
            [ '121', '123', 'MOSTEIROS PDL' ] ] },
        'Canto Redondo' => {
          '9555' => [
            [ '030', '033', 'CANDELRIA PDL' ] ] },
        'Fenais da Luz' => {
          '9545' => [
            [ '210', '225', 'FENAIS DA LUZ' ],
            [ '227', '249', 'FENAIS DA LUZ' ] ] },
        'Pico de Mafra' => {
          '9555' => [
            [ '170', '171', 'MOSTEIROS PDL' ] ] },
        'Atafona' => {
          '9545' => [
            [ '501', '503', 'SO VICENTE FERREIRA' ] ] },
        'Ribeira do Ferreiro' => {
          '9555' => [
            [ '040', '043', 'CANDELRIA PDL' ] ] },
        'Outeito Alto' => {
          '9545' => [
            [ '260', '262', 'FENAIS DA LUZ' ] ] },
        'Capelas' => {
          '9545' => [
            [ '101', '106', 'CAPELAS' ],
            [ '108', '162', 'CAPELAS' ] ] },
        'Remdios' => {
          '9545' => [
            [ '301', '301', 'REMDIOS' ] ] },
        'Livramento' => {
          '9500' => [
            [ '601', '607', 'PONTA DELGADA' ],
            [ '609', '609', 'PONTA DELGADA' ],
            [ '611', '619', 'PONTA DELGADA' ],
            [ '725', '764', 'PONTA DELGADA' ] ] },
        'Faj de Baixo' => {
          '9500' => [
            [ '431', '433', 'PONTA DELGADA' ],
            [ '435', '439', 'PONTA DELGADA' ],
            [ '441', '443', 'PONTA DELGADA' ],
            [ '445', '466', 'PONTA DELGADA' ],
            [ '515', '518', 'PONTA DELGADA' ],
            [ '701', '709', 'PONTA DELGADA' ] ] },
        'So Vicente de Ferreira' => {
          '9545' => [
            [ '520', '548', 'SO VICENTE FERREIRA' ] ] }
                   },
      'Nordeste' => {
        'Lomba da Cruz' => {
          '9630' => [
            [ '081', '091', 'LOMBA DA FAZENDA' ] ] },
        'Ramal' => {
          '9630' => [
            [ '270', '285', 'SALGA' ] ] },
        'Algarvia' => {
          '9630' => [
            [ '220', '254', 'NORDESTINHO' ] ] },
        'Lomba da Fazenda' => {
          '9630' => [
            [ '100', '125', 'LOMBA DA FAZENDA' ] ] },
        'Achadinha' => {
          '9630' => [
            [ '041', '051', 'ACHADINHA' ] ] },
        'Nordeste' => {
          '9630' => [
            [ '141', '147', 'NORDESTE' ],
            [ '149', '201', 'NORDESTE' ] ] },
        'Assomada' => {
          '9630' => [
            [ '070', '070', 'LOMBA DA FAZENDA' ] ] },
        'Achada' => {
          '9630' => [
            [ '011', '024', 'ACHADA' ] ] },
        'Feteira Grande' => {
          '9630' => [
            [ '301', '314', 'SANTANA NRD' ] ] }
              }
          },
    'Ilha do Corvo' => {
      'Corvo' => {
        'Cancela' => {
          '9980' => [
            [ '010', '010', 'CORVO' ] ] },
        'Outeiro' => {
          '9980' => [
            [ '060', '060', 'CORVO' ] ] },
        'Maranho' => {
          '9980' => [
            [ '050', '050', 'CORVO' ] ] },
        'Corvo' => {
          '9980' => [
            [ '020', '039', 'CORVO' ] ] }
            }
           },
    'Ilha de So Jorge' => {
      'Calheta (So Jorge)' => {
        'Sanguinhal' => {
          '9850' => [
            [ '281', '286', 'RIBEIRA SECA CHT' ] ] },
        'Alcaarias' => {
          '9850' => [
            [ '011', '014', 'CALHETA (SO JORGE)' ] ] },
        'Santo Anto' => {
          '9875' => [
            [ '051', '070', 'SANTO ANTO' ] ] },
        'Ribeira Seca' => {
          '9850' => [
            [ '251', '261', 'RIBEIRA SECA CHT' ] ] },
        'Boeiro' => {
          '9875' => [
            [ '011', '018', 'SANTO ANTO' ] ] },
        'Norte Pequeno' => {
          '9850' => [
            [ '121', '131', 'NORTE PEQUENO' ] ] },
        'Alqueives' => {
          '9875' => [
            [ '101', '118', 'TOPO' ] ] },
        'So Joo' => {
          '9875' => [
            [ '081', '087', 'SANTO ANTO' ] ] },
        'Aveiro' => {
          '9850' => [
            [ '201', '233', 'RIBEIRA SECA CHT' ] ] },
        'Cruzal' => {
          '9875' => [
            [ '020', '020', 'SANTO ANTO' ] ] },
        'Topo' => {
          '9875' => [
            [ '151', '168', 'TOPO' ] ] },
        'Calheta' => {
          '9850' => [
            [ '021', '042', 'CALHETA (SO JORGE)' ],
            [ '044', '045', 'CALHETA (SO JORGE)' ] ] },
        'Calrinhas' => {
          '9850' => [
            [ '061', '080', 'CALHETA (SO JORGE)' ] ] },
        'Faj da Fundura' => {
          '9850' => [
            [ '101', '106', 'NORTE PEQUENO' ] ] },
        'Porto' => {
          '9850' => [
            [ '161', '162', 'NORTE PEQUENO' ] ] },
        'Entre Ribeiras' => {
          '9875' => [
            [ '025', '042', 'SANTO ANTO' ] ] }
                  },
      'Velas' => {
        'Urzelina' => {
          '9800' => [
            [ '421', '435', 'URZELINA' ] ] },
        'Velas' => {
          '9800' => [
            [ '521', '561', 'VELAS' ] ] },
        'Faj da Isabel Pereira' => {
          '9800' => [
            [ '101', '105', 'NORTE GRANDE (NEVES)' ] ] },
        'Cancela Grande' => {
          '9800' => [
            [ '201', '210', 'ROSAIS' ] ] },
        'Presa das Cruzes' => {
          '9800' => [
            [ '403', '405', 'URZELINA' ] ] },
        'Cabo das Manadas' => {
          '9800' => [
            [ '011', '011', 'MANADAS' ] ] },
        'Rosais' => {
          '9800' => [
            [ '221', '223', 'ROSAIS' ],
            [ '225', '232', 'ROSAIS' ] ] },
        'Areeiro' => {
          '9800' => [
            [ '401', '401', 'URZELINA' ] ] },
        'Santo Amaro' => {
          '9800' => [
            [ '321', '348', 'SANTO AMARO VLS' ] ] },
        'Norte Grande' => {
          '9800' => [
            [ '121', '133', 'NORTE GRANDE (NEVES)' ] ] },
        'Aeroporto de So Jorge' => {
          '9800' => [
            [ '301', '309', 'SANTO AMARO VLS' ] ] },
        'Toledo' => {
          '9800' => [
            [ '370', '370', 'SANTO AMARO VLS' ] ] },
        'Beira' => {
          '9800' => [
            [ '501', '506', 'VELAS' ] ] },
        'Outeiro' => {
          '9800' => [
            [ '151', '153', 'NORTE GRANDE (NEVES)' ] ] },
        'Faj das Almas' => {
          '9800' => [
            [ '021', '041', 'MANADAS' ] ] },
        'Porto' => {
          '9800' => [
            [ '051', '052', 'MANADAS' ] ] }
                }
         },
    'Portalegre' => {
      'Arronches' => {
        'Arronches' => {
          '7340' => [
            [ '001', '041', 'ARRONCHES' ] ] },
        'Mosteiros' => {
          '7340' => [
            [ '210', '217', 'MOSTEIROS ARR' ] ] },
        'Casas Novas' => {
          '7340' => [
            [ '050', '056', 'ARRONCHES' ] ] },
        'Feiteira' => {
          '7340' => [
            [ '140', '150', 'ESPERANA ARR' ] ] },
        'Esperana' => {
          '7340' => [
            [ '120', '129', 'ESPERANA ARR' ] ] },
        'Aldeia Velha' => {
          '7340' => [
            [ '200', '205', 'MOSTEIROS ARR' ] ] },
        'gua de Raiz' => {
          '7340' => [
            [ '100', '110', 'ESPERANA ARR' ] ] },
        'Zuzarte' => {
          '7340' => [
            [ '220', '227', 'MOSTEIROS ARR' ] ] },
        'Hortas de Baixo' => {
          '7340' => [
            [ '111', '117', 'ARRONCHES' ] ] }
             },
      'Castelo de Vide' => {
        'Canto das Nogueiras' => {
          '7320' => [
            [ '301', '305', 'CASTELO DE VIDE' ] ] },
        'Castelo de Vide' => {
          '7320' => [
            [ '101', '209', 'CASTELO DE VIDE' ] ] },
        'Atalaia' => {
          '7320' => [
            [ '401', '419', 'CASTELO DE VIDE' ] ] },
        'Quinta Serra' => {
          '7320' => [
            [ '421', '422', 'CASTELO DE VIDE' ] ] },
        'Pvoa e Meadas' => {
          '7320' => [
            [ '011', '015', 'N SRA GRAA PVOA MEADAS' ] ] }
             },
      'Monforte' => {
        'Quinta de Santo Antnio' => {
          '7450' => [
            [ '156', '156', 'MONFORTE' ] ] },
        'Monte da Babugem' => {
          '7450' => [
            [ '151', '154', 'MONFORTE' ] ] },
        'Monte do Casco' => {
          '7450' => [
            [ '200', '200', 'SANTO ALEIXO' ] ] },
        'Prazeres' => {
          '7450' => [
            [ '160', '160', 'MONFORTE' ] ] },
        'Monte do Azinhal' => {
          '7450' => [
            [ '250', '250', 'VAIAMONTE' ] ] },
        'Monforte' => {
          '7450' => [
            [ '103', '145', 'MONFORTE' ] ] },
        'Monte das Ameixoeiras' => {
          '7450' => [
            [ '010', '010', 'ASSUMAR' ] ] },
        'Vaiamonte' => {
          '7450' => [
            [ '260', '261', 'VAIAMONTE' ],
            [ '263', '286', 'VAIAMONTE' ] ] },
        'Assumar' => {
          '7450' => [
            [ '013', '034', 'ASSUMAR' ] ] },
        'Monte do Paiol' => {
          '7450' => [
            [ '100', '100', 'MONFORTE' ] ] },
        'Monte da Coutada Real' => {
          '7450' => [
            [ '051', '051', 'ASSUMAR' ] ] },
        'Santo Aleixo' => {
          '7450' => [
            [ '221', '222', 'SANTO ALEIXO' ],
            [ '225', '243', 'SANTO ALEIXO' ] ] }
            },
      'Elvas' => {
        'Carrascal' => {
          '7350' => [
            [ '468', '479', 'ELVAS' ] ] },
        'Barbacena' => {
          '7350' => [
            [ '431', '431', 'BARBACENA' ] ] },
        'Brejo' => {
          '7350' => [
            [ '421', '423', 'ELVAS' ] ] },
        'Vila Fernando' => {
          '7350' => [
            [ '511', '511', 'VILA FERNANDO ELV' ] ] },
        'Terrugem' => {
          '7350' => [
            [ '491', '491', 'TERRUGEM ELV' ] ] },
        'Elvas' => {
          '7350' => [
            [ '001', '006', 'ELVAS' ],
            [ '008', '008', 'ELVAS' ],
            [ '011', '014', 'ELVAS' ],
            [ '016', '207', 'ELVAS' ],
            [ '214', '234', 'ELVAS' ],
            [ '236', '288', 'ELVAS' ],
            [ '291', '306', 'ELVAS' ],
            [ '308', '324', 'ELVAS' ],
            [ '425', '425', 'ELVAS' ] ] },
        'So Vicente' => {
          '7350' => [
            [ '481', '481', 'SO VICENTE E VENTOSA' ] ] },
        'Santa Eullia' => {
          '7350' => [
            [ '451', '451', 'SANTA EULLIA ELV' ] ] },
        'Horta da Taa de Prata' => {
          '7350' => [
            [ '401', '410', 'ELVAS' ] ] },
        'Vila Boim' => {
          '7350' => [
            [ '501', '501', 'VILA BOIM' ] ] },
        'Aldeia da Cruz' => {
          '7350' => [
            [ '461', '466', 'ELVAS' ] ] },
        'Alpedreira' => {
          '7350' => [
            [ '441', '445', 'ELVAS' ] ] }
               },
      'Sousel' => {
        'Monte da Seixa' => {
          '7470' => [
            [ '151', '154', 'CASA BRANCA' ] ] },
        'Herdade da Macabra' => {
          '7470' => [
            [ '060', '083', 'CANO' ] ] },
        'Sousel' => {
          '7470' => [
            [ '201', '206', 'SOUSEL' ],
            [ '208', '211', 'SOUSEL' ],
            [ '213', '241', 'SOUSEL' ],
            [ '243', '266', 'SOUSEL' ] ] },
        'Santo Amaro' => {
          '7470' => [
            [ '155', '155', 'SANTO AMARO SSL' ] ] },
        'Monte das Bazoas' => {
          '7470' => [
            [ '181', '182', 'SOUSEL' ] ] },
        'Casa Branca' => {
          '7470' => [
            [ '110', '124', 'CASA BRANCA' ],
            [ '126', '142', 'CASA BRANCA' ] ] },
        'Almadafe' => {
          '7470' => [
            [ '101', '101', 'CASA BRANCA' ] ] },
        'Cano' => {
          '7470' => [
            [ '011', '049', 'CANO' ] ] }
          },
      'Crato' => {
        'Vale do Peso' => {
          '7430' => [
            [ '351', '363', 'VALE DO PESO' ] ] },
        'Flor da Rosa' => {
          '7430' => [
            [ '211', '225', 'FLOR DA ROSA' ] ] },
        'Aldeia da Mata' => {
          '7430' => [
            [ '011', '033', 'ALDEIA DA MATA' ] ] },
        'Monte da Pedra' => {
          '7430' => [
            [ '311', '336', 'MONTE DA PEDRA' ] ] },
        'Crato' => {
          '7430' => [
            [ '111', '178', 'CRATO' ] ] },
        'Gfete' => {
          '7430' => [
            [ '251', '295', 'GFETE' ] ] }
               },
      'Nisa' => {
        'Nisa' => {
          '6050' => [
            [ '382', '388', 'NISA' ] ] },
        'Herdade do Azinhal' => {
          '6050' => [
            [ '491', '493', 'SO SIMO NIS' ] ] },
        'Montalvo' => {
          '6050' => [
            [ '431', '450', 'MONTALVO' ] ] },
        'Albarrol' => {
          '6050' => [
            [ '101', '146', 'AMIEIRA DO TEJO' ] ] },
        'Tolosa' => {
          '6050' => [
            [ '501', '545', 'TOLOSA' ] ] },
        'Quinta do Retiro' => {
          '6050' => [
            [ '409', '412', 'NISA' ] ] },
        'Arez' => {
          '6050' => [
            [ '201', '204', 'AREZ' ] ] },
        'Cacheiro' => {
          '6050' => [
            [ '471', '476', 'SO MATIAS NIS' ] ] },
        'Alpalho' => {
          '6050' => [
            [ '011', '053', 'ALPALHO' ] ] },
        'Carvalhal' => {
          '6050' => [
            [ '081', '082', 'ALPALHO' ] ] },
        'Quinta da Fonte da Cereja' => {
          '6050' => [
            [ '460', '461', 'MONTALVO' ] ] },
        'Salavessa' => {
          '6050' => [
            [ '465', '467', 'MONTALVO' ] ] },
        'Quinta do Bodego' => {
          '6050' => [
            [ '401', '407', 'NISA' ] ] },
        'Barroca do Neto' => {
          '6050' => [
            [ '451', '454', 'SANTANA NIS' ] ] },
        'Cho da Carreira' => {
          '6050' => [
            [ '170', '170', 'AMIEIRA DO TEJO' ] ] },
        'Monte do Casal Ventoso' => {
          '6050' => [
            [ '301', '380', 'NISA' ] ] }
              },
      'Fronteira' => {
        'Vale Maceiras' => {
          '7460' => [
            [ '200', '208', 'SO SATURNINO' ] ] },
        'Fronteira' => {
          '7460' => [
            [ '100', '154', 'FRONTEIRA' ],
            [ '209', '209', 'FRONTEIRA' ] ] },
        'Sulfurea' => {
          '7460' => [
            [ '050', '050', 'CABEO DE VIDE' ] ] },
        'Vale da Seda' => {
          '7460' => [
            [ '160', '160', 'FRONTEIRA' ] ] },
        'Cabeo de Vide' => {
          '7460' => [
            [ '001', '043', 'CABEO DE VIDE' ] ] }
             },
      'Gavio' => {
        'Alvisquer' => {
          '6040' => [
            [ '021', '034', 'BELVER GAV' ] ] },
        'Comenda' => {
          '6040' => [
            [ '041', '043', 'COMENDA' ] ] },
        'Atalaia' => {
          '6040' => [
            [ '011', '011', 'ATALAIA GAV' ] ] },
        'Amieira Cova' => {
          '6040' => [
            [ '051', '060', 'GAVIO' ] ] },
        'Gavio' => {
          '6040' => [
            [ '100', '130', 'GAVIO' ] ] },
        'Moinho do Torro' => {
          '6040' => [
            [ '071', '079', 'MARGEM' ] ] }
          },
        'Ponte de Sor' => {
        'gua Todo o Ano' => {
          '7400' => [
            [ '601', '604', 'TRAMAGA' ] ] },
        'Vale de Boi' => {
          '7400' => [
            [ '580', '580', 'VALE DE AOR' ] ] },
        'Arneiro' => {
          '7400' => [
            [ '111', '136', 'PONTE DE SOR' ] ] },
        'Foros de Arro' => {
          '7425' => [
            [ '201', '243', 'FOROS DE ARRO' ] ] },
        'Ribeira das Vinhas' => {
          '7400' => [
            [ '070', '070', 'GALVEIAS' ] ] },
        'Ponte de Sor' => {
          '7400' => [
            [ '201', '225', 'PONTE DE SOR' ],
            [ '227', '230', 'PONTE DE SOR' ],
            [ '232', '289', 'PONTE DE SOR' ],
            [ '291', '308', 'PONTE DE SOR' ] ] },
        'Abertas de Baixo' => {
          '7425' => [
            [ '011', '020', 'MONTARGIL' ] ] },
        'Galveias' => {
          '7400' => [
            [ '001', '001', 'GALVEIAS' ],
            [ '011', '037', 'GALVEIAS' ],
            [ '039', '048', 'GALVEIAS' ] ] },
        'Salgueirinha' => {
          '7400' => [
            [ '401', '407', 'PONTE DE SOR' ] ] },
        'Alto de So Loureno' => {
          '7400' => [
            [ '451', '460', 'LONGOMEL' ] ] },
        'Hospcios de Baixo' => {
          '7400' => [
            [ '501', '507', 'VALE DE AOR' ] ] },
        'Vale de Aor' => {
          '7400' => [
            [ '521', '558', 'VALE DE AOR' ] ] },
        'Montargil' => {
          '7425' => [
            [ '100', '146', 'MONTARGIL' ] ] }
                },
      'Campo Maior' => {
        'Quinta Paraiso' => {
          '7370' => [
            [ '150', '150', 'CAMPO MAIOR' ] ] },
        'Nossa Selhora da Graa Degolados' => {
          '7370' => [
            [ '191', '191', 'N SENHORA GRAA DEGOLADOS' ] ] },
        'Hortas de So Pedro' => {
          '7370' => [
            [ '170', '171', 'CAMPO MAIOR' ] ] },
        'Ouguela' => {
          '7370' => [
            [ '200', '203', 'CAMPO MAIOR' ] ] },
        'Campo Maior' => {
          '7370' => [
            [ '010', '029', 'CAMPO MAIOR' ],
            [ '031', '110', 'CAMPO MAIOR' ],
            [ '112', '123', 'CAMPO MAIOR' ] ] }
               },
      'Avis' => {
        'Figueira e Barros' => {
          '7480' => [
            [ '351', '352', 'FIGUEIRA E BARROS' ] ] },
        'Monte da Cortesia' => {
          '7480' => [
            [ '181', '182', 'AVIS' ] ] },
        'Monte Sobral' => {
          '7480' => [
            [ '320', '320', 'ERVEDAL AVS' ] ] },
        'Ervedal' => {
          '7480' => [
            [ '282', '310', 'ERVEDAL AVS' ] ] },
        'Monte do Santo Anto' => {
          '7480' => [
            [ '250', '250', 'BENAVILA' ] ] },
        'Benavila' => {
          '7480' => [
            [ '201', '233', 'BENAVILA' ] ] },
        'Aldeia Velha' => {
          '7480' => [
            [ '051', '051', 'ALDEIA VELHA AVS' ] ] },
        'Avis' => {
          '7480' => [
            [ '101', '163', 'AVIS' ] ] },
        'Casas Altas' => {
          '7480' => [
            [ '055', '057', 'ALDEIA VELHA AVS' ] ] },
        'Calio' => {
          '7480' => [
            [ '011', '028', 'ALCRREGO' ] ] },
        'Monte Vale de Marcos' => {
          '7480' => [
            [ '391', '392', 'VALONGO AVS' ] ] },
        'Margem' => {
          '7480' => [
            [ '371', '373', 'MARANHO' ] ] }
              },
      'Portalegre' => {
        'Urra' => {
          '7300' => [
            [ '575', '586', 'URRA' ],
            [ '588', '613', 'URRA' ] ] },
        'Alegrete' => {
          '7300' => [
            [ '311', '329', 'ALEGRETE' ] ] },
        'Areeiro' => {
          '7300' => [
            [ '400', '400', 'RIBEIRA DE NISA' ],
            [ '443', '443', 'RIBEIRA DE NISA' ] ] },
        'Carreiras' => {
          '7300' => [
            [ '355', '357', 'CARREIRAS' ] ] },
        'Freguesia' => {
          '7300' => [
            [ '451', '471', 'SO JULIO PTG' ] ] },
        'Caia' => {
          '7300' => [
            [ '561', '571', 'URRA' ] ] },
        'Buraco' => {
          '7300' => [
            [ '351', '352', 'CARREIRAS' ] ] },
        'Fortios' => {
          '7300' => [
            [ '651', '690', 'FORTIOS' ] ] },
        'Biquinha' => {
          '7300' => [
            [ '401', '409', 'REGUENGO' ] ] },
        'Portalegre' => {
          '7301' => [
            [ '851', '858', 'PORTALEGRE' ] ],
          '7300' => [
            [ '002', '158', 'PORTALEGRE' ],
            [ '160', '229', 'PORTALEGRE' ],
            [ '231', '248', 'PORTALEGRE' ],
            [ '250', '264', 'PORTALEGRE' ],
            [ '266', '269', 'PORTALEGRE' ],
            [ '271', '300', 'PORTALEGRE' ],
            [ '302', '304', 'PORTALEGRE' ],
            [ '493', '499', 'PORTALEGRE' ],
            [ '501', '501', 'PORTALEGRE' ],
            [ '521', '532', 'PORTALEGRE' ],
            [ '541', '551', 'PORTALEGRE' ],
            [ '800', '870', 'PORTALEGRE' ] ] },
        'Assumadas' => {
          '7300' => [
            [ '422', '441', 'RIBEIRA DE NISA' ] ] },
        'Alagoa' => {
          '7300' => [
            [ '301', '301', 'ALAGOA' ] ] },
        'Abreu' => {
          '7300' => [
            [ '371', '371', 'FORTIOS' ] ] },
        'Frangoneiro' => {
          '7300' => [
            [ '373', '383', 'FORTIOS' ] ] },
        'Monte Frei Alvaro' => {
          '7300' => [
            [ '001', '001', 'RIBEIRA DE NISA' ] ] },
        'Vale de Cavalos' => {
          '7300' => [
            [ '335', '337', 'ALEGRETE' ] ] }
              },
      'Marvo' => {
        'Santo Antnio das Areias' => {
          '7330' => [
            [ '250', '273', 'SANTO ANTNIO DAS AREIAS' ] ] },
        'Abegoa' => {
          '7330' => [
            [ '201', '224', 'SANTO ANTNIO DAS AREIAS' ] ] },
        'Torre' => {
          '7330' => [
            [ '150', '155', 'MARVO' ] ] },
        'Alvarres' => {
          '7330' => [
            [ '301', '347', 'SO SALVADOR DA ARAMENHA' ] ] },
        'Marvo' => {
          '7330' => [
            [ '101', '135', 'MARVO' ] ] },
        'Barretos' => {
          '7330' => [
            [ '011', '020', 'BEIR' ] ] },
        'Barros Cardos' => {
          '7330' => [
            [ '051', '075', 'MARVO' ] ] }
          },
      'Alter do Cho' => {
        'Cunheira' => {
          '7440' => [
            [ '251', '253', 'CUNHEIRA' ] ] },
        'Chana' => {
          '7440' => [
            [ '201', '203', 'CHANCELARIA ALT' ] ] },
        'Herdade do No Vs L a Cima' => {
          '7440' => [
            [ '221', '226', 'SEDA' ] ] },
        'Alter Pedroso' => {
          '7440' => [
            [ '151', '158', 'ALTER DO CHO' ] ] },
        'Alter do Cho' => {
          '7440' => [
            [ '011', '084', 'ALTER DO CHO' ],
            [ '086', '101', 'ALTER DO CHO' ] ] }
                 }
        },
    'Viana do Castelo' => {
      'Valena' => {
        'Cruz da Gandra' => {
          '4930' => [
            [ '191', '211', 'CRISTELO COVO' ] ] },
        'Aguilho' => {
          '4930' => [
            [ '301', '328', 'GANDRA VLN' ] ] },
        'Martim' => {
          '4930' => [
            [ '440', '440', 'SANFINS VLN' ] ] },
        'Igreja' => {
          '4930' => [
            [ '070', '070', 'BOIVO' ] ] },
        'Alm da Ponte' => {
          '4930' => [
            [ '501', '516', 'SO PEDRO DA TORRE' ] ] },
        'Aro' => {
          '4930' => [
            [ '001', '028', 'ARO' ] ] },
        'Altamira' => {
          '4930' => [
            [ '131', '140', 'CRISTELO COVO' ] ] },
        'Barracas' => {
          '4930' => [
            [ '801', '821', 'VERDOEJO' ] ] },
        'Valena' => {
          '4930' => [
            [ '587', '619', 'VALENA' ],
            [ '621', '648', 'VALENA' ],
            [ '650', '767', 'VALENA' ] ] },
        'Azenhas' => {
          '4930' => [
            [ '341', '376', 'GANFEI' ] ] },
        'Arraial' => {
          '4930' => [
            [ '521', '536', 'SILVA VLN' ] ] },
        'Abedim' => {
          '4930' => [
            [ '401', '430', 'GONDOMIL' ] ] },
        'Barreiras' => {
          '4930' => [
            [ '271', '290', 'FRIESTAS' ] ] },
        'Brrio' => {
          '4930' => [
            [ '241', '259', 'FONTOURA' ] ] },
        'Cristelo Covo' => {
          '4930' => [
            [ '151', '178', 'CRISTELO COVO' ] ] },
        'Agras' => {
          '4930' => [
            [ '081', '114', 'CERDAL' ] ] },
        'Baixo' => {
          '4930' => [
            [ '551', '557', 'TAIO' ] ] },
        'Argolento' => {
          '4930' => [
            [ '041', '058', 'ARO' ] ] },
        'Azevinhas' => {
          '4930' => [
            [ '451', '483', 'SO JULIO VLN' ] ] }
                 },
      'Ponte de Lima' => {
        'Barranca' => {
          '4990' => [
            [ '735', '735', 'SANDIES' ] ] },
        'Boalhosa' => {
          '4990' => [
            [ '555', '555', 'BOALHOSA' ] ] },
        'Beiro' => {
          '4990' => [
            [ '645', '645', 'GEMIEIRA' ] ] },
        'Cortes' => {
          '4990' => [
            [ '580', '580', 'CALVELO' ] ] },
        'Alm-do-Rio' => {
          '4990' => [
            [ '795', '795', 'VILAR DO MONTE PTL' ] ] },
        'Alminhas' => {
          '4990' => [
            [ '635', '635', 'GAIFAR' ] ] },
        'Agra - Correlh' => {
          '4990' => [
            [ '281', '314', 'PONTE DE LIMA' ] ] },
        'Bertiandos' => {
          '4990' => [
            [ '550', '550', 'BERTIANDOS' ] ] },
        'Barros' => {
          '4990' => [
            [ '770', '781', 'SOUTO REBORDES' ] ] },
        'Barbudos' => {
          '4990' => [
            [ '745', '746', 'SANTA CRUZ DO LIMA' ] ] },
        'Alto' => {
          '4990' => [
            [ '431', '463', 'FREIXO PTL' ] ] },
        'Agro - Ribeira' => {
          '4990' => [
            [ '371', '419', 'PONTE DE LIMA' ] ] },
        'Ardego' => {
          '4990' => [
            [ '535', '535', 'ARDEGO PTL' ] ] },
        'Baganheiro' => {
          '4990' => [
            [ '685', '685', 'QUEIJADA' ] ] },
        'Bacelos' => {
          '4990' => [
            [ '655', '655', 'LABRUJA' ] ] },
        'Amieiro' => {
          '4990' => [
            [ '750', '753', 'SANTA MARIA REBORDES' ] ] },
        'Aveleira' => {
          '4990' => [
            [ '585', '585', 'CEPES PTL' ] ] },
        'Corujeira' => {
          '4990' => [
            [ '640', '643', 'GANDRA PTL' ] ] },
        'Anho Bom' => {
          '4990' => [
            [ '730', '730', 'S PTL' ] ] },
        'Alm' => {
          '4990' => [
            [ '725', '725', 'RENDUFE PTL' ],
            [ '790', '790', 'VILAR DAS ALMAS' ] ] },
        'Barrala' => {
          '4990' => [
            [ '760', '760', 'SERDEDELO' ] ] },
        'Santa Comba' => {
          '4990' => [
            [ '740', '740', 'SANTA COMBA PTL' ] ] },
        'Agriboa' => {
          '4990' => [
            [ '560', '560', 'BRANDARA' ] ] },
        'Antepao - Arcozelo' => {
          '4990' => [
            [ '231', '262', 'PONTE DE LIMA' ] ] },
        'Bodilho' => {
          '4990' => [
            [ '670', '673', 'MOREIRA DO LIMA' ] ] },
        'Barral' => {
          '4990' => [
            [ '630', '631', 'FRIASTELAS' ] ] },
        'Barrosas' => {
          '4990' => [
            [ '605', '605', 'FOJO LOBAL' ] ] },
        'Aboboreira - Arca' => {
          '4990' => [
            [ '201', '202', 'PONTE DE LIMA' ] ] },
        'Ameal' => {
          '4990' => [
            [ '690', '718', 'REFIOS DO LIMA' ] ] },
        'Aba' => {
          '4990' => [
            [ '600', '603', 'FACHA' ] ] },
        'Arcos' => {
          '4990' => [
            [ '530', '530', 'ARCOS PTL' ] ] },
        'Barrosa' => {
          '4990' => [
            [ '501', '519', 'ANAIS' ] ] },
        'Barreinho' => {
          '4990' => [
            [ '575', '575', 'CALHEIROS' ] ] },
        'Balouca' => {
          '4990' => [
            [ '570', '570', 'CABRAO' ] ] },
        'Cancela - Feitosa' => {
          '4990' => [
            [ '331', '337', 'PONTE DE LIMA' ] ] },
        'Fij - Feitosa' => {
          '4990' => [
            [ '339', '354', 'PONTE DE LIMA' ] ] },
        'Balte' => {
          '4990' => [
            [ '810', '832', 'VITORINO DOS PIES' ] ] },
        'Belmonte' => {
          '4990' => [
            [ '620', '624', 'FORNELOS PTL' ] ] },
        'Airo' => {
          '4990' => [
            [ '680', '683', 'POIARES PTL' ] ] },
        'Ponte de Lima' => {
          '4990' => [
            [ '011', '034', 'PONTE DE LIMA' ],
            [ '036', '093', 'PONTE DE LIMA' ],
            [ '095', '098', 'PONTE DE LIMA' ],
            [ '100', '123', 'PONTE DE LIMA' ],
            [ '125', '128', 'PONTE DE LIMA' ],
            [ '130', '171', 'PONTE DE LIMA' ] ] },
        'Beiral do Lima' => {
          '4990' => [
            [ '545', '546', 'BEIRAL DO LIMA' ] ] },
        'Bostelo - Arca' => {
          '4990' => [
            [ '204', '217', 'PONTE DE LIMA' ] ] },
        'Aldeia' => {
          '4990' => [
            [ '675', '675', 'NAVI' ],
            [ '800', '801', 'VITORINO DAS DONAS' ] ] },
        'Abelheira' => {
          '4990' => [
            [ '755', '756', 'SEARA' ] ] },
        'Brrio' => {
          '4990' => [
            [ '540', '540', 'BRRIO PTL' ] ] },
        'Boua de Abade' => {
          '4990' => [
            [ '590', '590', 'ESTOROS PTL' ] ] },
        'Barreiro' => {
          '4990' => [
            [ '565', '565', 'CABAOS PTL' ],
            [ '610', '611', 'FONTO' ],
            [ '650', '650', 'GONDUFE' ],
            [ '660', '660', 'LABRUJ' ],
            [ '665', '665', 'MATO' ] ] }
                 },
      'Mono' => {
        'Mazedo' => {
          '4950' => [
            [ '275', '299', 'MAZEDO' ] ] },
        'Redemouro' => {
          '4950' => [
            [ '630', '630', 'PARADA MNC' ] ] },
        'Casal' => {
          '4950' => [
            [ '270', '270', 'LUZIO' ] ] },
        'Pias' => {
          '4950' => [
            [ '322', '332', 'MERUFE' ] ] },
        'Cruz' => {
          '4950' => [
            [ '070', '070', 'BARROAS E TAIAS' ],
            [ '760', '762', 'SEGUDE' ] ] },
        'Igreja' => {
          '4950' => [
            [ '150', '150', 'LAPELA' ],
            [ '830', '830', 'TRUTE' ] ] },
        'Gandra' => {
          '4950' => [
            [ '198', '223', 'LONGOS VALES' ] ] },
        'Retorta' => {
          '4950' => [
            [ '640', '647', 'PIAS MNC' ] ] },
        'Bairro de So Mamede' => {
          '4950' => [
            [ '810', '821', 'TROVISCOSO' ] ] },
        'Rio Bom' => {
          '4950' => [
            [ '680', '680', 'PORTELA MNC' ] ] },
        'Bornaria' => {
          '4950' => [
            [ '080', '096', 'BELA' ] ] },
        'Bairro Novo' => {
          '4950' => [
            [ '040', '059', 'BARBEITA' ] ] },
        'Mono' => {
          '4950' => [
            [ '411', '411', 'MONO' ],
            [ '414', '418', 'MONO' ],
            [ '420', '421', 'MONO' ],
            [ '423', '428', 'MONO' ],
            [ '430', '430', 'MONO' ],
            [ '432', '433', 'MONO' ],
            [ '436', '447', 'MONO' ],
            [ '449', '449', 'MONO' ],
            [ '452', '452', 'MONO' ],
            [ '454', '458', 'MONO' ],
            [ '463', '474', 'MONO' ],
            [ '478', '480', 'MONO' ],
            [ '482', '484', 'MONO' ],
            [ '486', '489', 'MONO' ],
            [ '491', '495', 'MONO' ],
            [ '497', '498', 'MONO' ],
            [ '500', '500', 'MONO' ],
            [ '502', '502', 'MONO' ],
            [ '505', '506', 'MONO' ],
            [ '508', '510', 'MONO' ],
            [ '512', '513', 'MONO' ],
            [ '516', '517', 'MONO' ] ] },
        'Costa' => {
          '4950' => [
            [ '660', '660', 'PINHEIROS MNC' ] ] },
        'Talho' => {
          '4950' => [
            [ '010', '010', 'ABEDIM' ] ] },
        'Vinho' => {
          '4950' => [
            [ '310', '318', 'MERUFE' ] ] },
        'Bairro Foras Armadas' => {
          '4950' => [
            [ '401', '408', 'MONO' ] ] },
        'Quinta Senra' => {
          '4950' => [
            [ '170', '170', 'LARA' ] ] },
        'Outeiral' => {
          '4950' => [
            [ '840', '840', 'VALADARES MNC' ] ] },
        'Terranova' => {
          '4950' => [
            [ '100', '106', 'CAMBESES MNC' ] ] },
        'Parada' => {
          '4950' => [
            [ '320', '320', 'MERUFE' ] ] },
        'Amorrrosa' => {
          '4950' => [
            [ '120', '132', 'CEIVES' ] ] },
        'Vila Boa' => {
          '4950' => [
            [ '030', '030', 'BADIM' ] ] },
        'Marrajos' => {
          '4950' => [
            [ '670', '670', 'PODAME' ] ] },
        'Martingo' => {
          '4950' => [
            [ '750', '750', 'SAGO' ] ] },
        'Giesteira' => {
          '4950' => [
            [ '800', '800', 'TROPORIZ' ] ] },
        'Quinta da Torre' => {
          '4950' => [
            [ '600', '614', 'MOREIRA MNC' ] ] },
        'Porcalho' => {
          '4950' => [
            [ '520', '534', 'MONO' ] ] },
        'Fonte' => {
          '4950' => [
            [ '770', '791', 'TANGIL' ] ] },
        'Outeiro' => {
          '4950' => [
            [ '020', '020', 'ANHES' ] ] },
        'Lajes' => {
          '4950' => [
            [ '700', '721', 'RIBA DE MOURO' ] ] },
        'So Miguel' => {
          '4950' => [
            [ '350', '359', 'MESSEGES' ] ] },
        'So Joo' => {
          '4950' => [
            [ '740', '740', 'S MNC' ] ] },
        'Bouas' => {
          '4950' => [
            [ '850', '855', 'CORTES MNC' ] ] },
        'Porto' => {
          '4950' => [
            [ '250', '250', 'LORDELO MNC' ] ] }
                },
      'Ponte da Barca' => {
        'Froufe' => {
          '4980' => [
            [ '311', '315', 'ENTRE AMBOS-OS-RIOS' ] ] },
        'Azenha' => {
          '4980' => [
            [ '231', '260', 'CRASTO' ] ] },
        'Barrio' => {
          '4980' => [
            [ '051', '067', 'BOIVES' ] ] },
        'Caapedro' => {
          '4980' => [
            [ '011', '027', 'AZIAS' ] ] },
        'Castelo' => {
          '4980' => [
            [ '451', '453', 'LINDOSO' ],
            [ '681', '682', 'TOUVEDO (SALVADOR)' ] ] },
        'Alvarinha' => {
          '4980' => [
            [ '351', '355', 'GROVELAS' ] ] },
        'Igreja' => {
          '4980' => [
            [ '201', '203', 'BRITELO' ] ] },
        'Cardelha' => {
          '4980' => [
            [ '711', '725', 'SAMPRIZ' ] ] },
        'Agrelos' => {
          '4980' => [
            [ '601', '639', 'PONTE DA BARCA' ] ] },
        'Barranheiras' => {
          '4980' => [
            [ '811', '838', 'VILA NOVA DE MUIA' ] ] },
        'Poa' => {
          '4980' => [
            [ '557', '563', 'PAO VEDRO DE MAGALHES' ] ] },
        'Agrela' => {
          '4980' => [
            [ '501', '529', 'OLEIROS PTB' ],
            [ '771', '781', 'VADE (SO PEDRO)' ] ] },
        'Ermida' => {
          '4980' => [
            [ '320', '320', 'ERMIDA PTB' ] ] },
        'Alvaredo' => {
          '4980' => [
            [ '541', '555', 'PAO VEDRO DE MAGALHES' ] ] },
        'Barreira' => {
          '4980' => [
            [ '751', '766', 'TOUVEDO (SO LOURENO)' ] ] },
        'Costa' => {
          '4980' => [
            [ '683', '683', 'TOUVEDO (SO LOURENO)' ] ] },
        'Caxada' => {
          '4980' => [
            [ '461', '479', 'NOGUEIRA PTB' ] ] },
        'Chouzelas e Arajos' => {
          '4980' => [
            [ '791', '801', 'VADE (SO TOM)' ] ] },
        'Pena' => {
          '4980' => [
            [ '783', '786', 'VADE (SO PEDRO)' ] ] },
        'Bandeira' => {
          '4980' => [
            [ '401', '422', 'LAVRADAS' ] ] },
        'Pedra' => {
          '4980' => [
            [ '782', '782', 'VADE (SO TOM)' ] ] },
        'Barral' => {
          '4980' => [
            [ '741', '748', 'VILA CH (S J BAPTISTA)' ] ] },
        'Coto da Guarda' => {
          '4980' => [
            [ '684', '689', 'TOUVEDO (SALVADOR)' ] ] },
        'Barreiro' => {
          '4980' => [
            [ '281', '295', 'CUIDE DE VILA VERDE' ] ] },
        'Aldego' => {
          '4980' => [
            [ '651', '665', 'RUIVOS' ] ] },
        'Baixo' => {
          '4980' => [
            [ '731', '736', 'VILA CH (SANTIAGO)' ] ] },
        'Germil' => {
          '4980' => [
            [ '330', '330', 'GERMIL PTB' ] ] },
        'Barbeitos' => {
          '4980' => [
            [ '101', '142', 'BRAVES' ] ] }
                  },
      'Vila Nova de Cerveira' => {
        'Atalaia' => {
          '4920' => [
            [ '210', '220', 'VILA NOVA DE CERVEIRA' ] ] },
        'Areal' => {
          '4920' => [
            [ '201', '201', 'VILA NOVA DE CERVEIRA' ] ] },
        'Santo Antnio' => {
          '4920' => [
            [ '020', '020', 'CANDEMIL VNC' ] ] },
        'Vale da Cal' => {
          '4920' => [
            [ '090', '090', 'MENTRESTIDO' ] ] },
        'Vila Nova de Cerveira' => {
          '4920' => [
            [ '222', '250', 'VILA NOVA DE CERVEIRA' ],
            [ '253', '265', 'VILA NOVA DE CERVEIRA' ],
            [ '267', '289', 'VILA NOVA DE CERVEIRA' ],
            [ '291', '296', 'VILA NOVA DE CERVEIRA' ] ] },
        'Igreja' => {
          '4920' => [
            [ '040', '042', 'COVAS VNC' ] ] },
        'Quinta do Vale' => {
          '4920' => [
            [ '070', '072', 'LOIVO' ] ] },
        'Lovelhe' => {
          '4920' => [
            [ '075', '087', 'LOVELHE' ] ] },
        'Urgal' => {
          '4920' => [
            [ '110', '112', 'REBOREDA' ] ] },
        'Quinta' => {
          '4920' => [
            [ '011', '012', 'CAMPOS VNC' ] ] },
        'Boavista' => {
          '4920' => [
            [ '203', '208', 'VILA NOVA DE CERVEIRA' ] ] },
        'Outeiro' => {
          '4920' => [
            [ '100', '100', 'NOGUEIRA VNC' ],
            [ '120', '120', 'SAPARDOS' ] ] },
        'Montorros' => {
          '4920' => [
            [ '140', '140', 'VILA ME VNC' ] ] },
        'Sobreiro' => {
          '4920' => [
            [ '050', '050', 'GONDAR VNC' ] ] },
        'Cornes' => {
          '4920' => [
            [ '030', '030', 'CORNES' ] ] },
        'Rodetes' => {
          '4920' => [
            [ '130', '130', 'SOPO' ] ] },
        'Lagarelha' => {
          '4920' => [
            [ '060', '062', 'GONDARM' ] ] }
                   },
        'Caminha' => {
        'guas Frreas' => {
          '4910' => [
            [ '011', '020', 'NCORA' ] ] },
        'Barge' => {
          '4910' => [
            [ '350', '361', 'VENADE' ] ] },
        'Vila Praia de ncora' => {
          '4910' => [
            [ '384', '400', 'VILA PRAIA DE NCORA' ],
            [ '429', '543', 'VILA PRAIA DE NCORA' ],
            [ '547', '579', 'VILA PRAIA DE NCORA' ] ] },
        'Caminha' => {
          '4910' => [
            [ '101', '114', 'CAMINHA' ],
            [ '116', '133', 'CAMINHA' ],
            [ '135', '136', 'CAMINHA' ],
            [ '145', '145', 'CAMINHA' ],
            [ '149', '150', 'CAMINHA' ],
            [ '152', '153', 'CAMINHA' ],
            [ '155', '167', 'CAMINHA' ] ] },
        'Lagos' => {
          '4910' => [
            [ '060', '060', 'AZEVEDO' ] ] },
        'Coutada' => {
          '4910' => [
            [ '035', '035', 'ARGA DE BAIXO' ] ] },
        'Carvalhais' => {
          '4910' => [
            [ '301', '306', 'ORBACM' ] ] },
        'Boucinha' => {
          '4910' => [
            [ '201', '203', 'LANHELAS' ] ] },
        'Cabreira' => {
          '4910' => [
            [ '340', '347', 'SEIXAS CMN' ] ] },
        'Pereira' => {
          '4910' => [
            [ '180', '180', 'CRISTELO CMN' ] ] },
        'Gandra' => {
          '4910' => [
            [ '040', '040', 'ARGA DE CIMA' ] ] },
        'Sandia' => {
          '4910' => [
            [ '422', '422', 'VILA PRAIA DE NCORA' ] ] },
        'Paraso' => {
          '4910' => [
            [ '415', '415', 'VILA PRAIA DE NCORA' ] ] },
        'Santa Luzia' => {
          '4910' => [
            [ '026', '028', 'NCORA' ] ] },
        'Abelheira' => {
          '4910' => [
            [ '185', '189', 'DEM' ] ] },
        'Alvaria' => {
          '4910' => [
            [ '055', '055', 'ARGELA' ] ] },
        'Vilarelho' => {
          '4910' => [
            [ '138', '141', 'VILARELHO' ],
            [ '597', '601', 'VILARELHO' ],
            [ '603', '609', 'VILARELHO' ],
            [ '611', '613', 'VILARELHO' ],
            [ '616', '619', 'VILARELHO' ],
            [ '621', '621', 'VILARELHO' ] ] },
        'Carotes' => {
          '4910' => [
            [ '195', '195', 'GONDAR CMN' ] ] },
        'Naia' => {
          '4910' => [
            [ '650', '653', 'VILE' ] ] },
        'Agrelo' => {
          '4910' => [
            [ '582', '585', 'VILAR DE MOUROS' ] ] },
        'Moledo' => {
          '4910' => [
            [ '211', '211', 'MOLEDO CMN' ],
            [ '214', '215', 'MOLEDO CMN' ],
            [ '221', '230', 'MOLEDO CMN' ],
            [ '232', '237', 'MOLEDO CMN' ],
            [ '239', '240', 'MOLEDO CMN' ],
            [ '242', '250', 'MOLEDO CMN' ],
            [ '252', '255', 'MOLEDO CMN' ],
            [ '257', '259', 'MOLEDO CMN' ],
            [ '262', '268', 'MOLEDO CMN' ],
            [ '270', '274', 'MOLEDO CMN' ],
            [ '276', '279', 'MOLEDO CMN' ],
            [ '284', '284', 'MOLEDO CMN' ],
            [ '661', '692', 'MOLEDO CMN' ] ] },
        'Ponte' => {
          '4910' => [
            [ '315', '319', 'RIBA DE NCORA' ] ] },
        'Santo Aginha' => {
          '4910' => [
            [ '045', '045', 'ARGA DE SO JOO' ] ] }
                 },
      'Arcos de Valdevez' => {
        'Vilarinho do Souto' => {
          '4970' => [
            [ '140', '140', 'ERMELO AVV' ] ] },
        'Arco' => {
          '4970' => [
            [ '111', '128', 'CENDUFE' ] ] },
        'Alto Cadorcas' => {
          '4970' => [
            [ '195', '199', 'ARCOS DE VALDEVEZ' ] ] },
        'Cruz' => {
          '4970' => [
            [ '555', '555', 'SO COSME E SO DAMIO' ] ] },
        'Alto do Freitas' => {
          '4970' => [
            [ '231', '254', 'ARCOS DE VALDEVEZ' ] ] },
        'Igreja' => {
          '4970' => [
            [ '096', '098', 'CABANA MAIOR' ],
            [ '105', '105', 'CARRALCOVA' ],
            [ '150', '150', 'GAVIEIRA' ],
            [ '190', '190', 'GRADE' ],
            [ '645', '645', 'SISTELO' ] ] },
        'Igreja - Giela' => {
          '4970' => [
            [ '763', '794', 'ARCOS DE VALDEVEZ' ] ] },
        'Faquelo' => {
          '4970' => [
            [ '614', '622', 'ARCOS DE VALDEVEZ' ] ] },
        'Cancela' => {
          '4970' => [
            [ '611', '612', 'ARCOS DE VALDEVEZ' ] ] },
        'Boimo' => {
          '4970' => [
            [ '091', '094', 'CABANA MAIOR' ] ] },
        'Fij' => {
          '4970' => [
            [ '735', '749', 'ARCOS DE VALDEVEZ' ] ] },
        'Casaldoufes' => {
          '4970' => [
            [ '062', '066', 'ALVORA' ] ] },
        'Aspra' => {
          '4970' => [
            [ '371', '394', 'SABADIM' ] ] },
        'Brunhedo' => {
          '4970' => [
            [ '640', '640', 'SENHAREI' ] ] },
        'Alm' => {
          '4970' => [
            [ '071', '080', 'ZERE AVV' ] ] },
        'Cheda' => {
          '4970' => [
            [ '685', '685', 'TABA' ] ] },
        'Costa' => {
          '4970' => [
            [ '505', '505', 'PADREIRO (SANTA CRISTINA)' ] ] },
        'Pinheiro do Couto' => {
          '4970' => [
            [ '130', '130', 'COUTO AVV' ] ] },
        'Monte Aval' => {
          '4974' => [
            [ '012', '012', 'TVORA (SANTA MARIA)' ] ] },
        'Gndra' => {
          '4970' => [
            [ '550', '550', 'SANTAR AVV' ] ] },
        'Aboim' => {
          '4970' => [
            [ '010', '019', 'ABOIM DAS CHOAS' ] ] },
        'Assim Chamado' => {
          '4970' => [
            [ '161', '183', 'GONDORIZ AVV' ] ] },
        'Silvares - Vila Fonche' => {
          '4970' => [
            [ '760', '761', 'ARCOS DE VALDEVEZ' ] ] },
        'Adro' => {
          '4970' => [
            [ '651', '678', 'SOAJO' ] ] },
        'Buaco' => {
          '4970' => [
            [ '635', '635', 'SO VICENTE TVORA' ] ] },
        'Aveleiras' => {
          '4970' => [
            [ '306', '331', 'RIO FRIO AVV' ] ] },
        'Casares' => {
          '4970' => [
            [ '680', '680', 'SOUTO AVV' ] ] },
        'Feijoal' => {
          '4970' => [
            [ '360', '360', 'S AVV' ] ] },
        'Devesa' => {
          '4970' => [
            [ '210', '210', 'MEI' ] ] },
        'Ponte Velha' => {
          '4970' => [
            [ '270', '275', 'ARCOS DE VALDEVEZ' ] ] },
        'Boua Soeiro' => {
          '4970' => [
            [ '032', '052', 'AGUI' ] ] },
        'Algarvia' => {
          '4970' => [
            [ '205', '205', 'JOLDA (MADALENA)' ] ] },
        'Arroteia' => {
          '4970' => [
            [ '691', '734', 'VALE AVV' ] ] },
        'Falperra' => {
          '4970' => [
            [ '480', '483', 'ARCOS DE VALDEVEZ' ] ] },
        'Cotos' => {
          '4970' => [
            [ '215', '215', 'MIRANDA' ] ] },
        'Formigosa' => {
          '4970' => [
            [ '225', '225', 'OLIVEIRA AVV' ] ] },
        'Atade' => {
          '4974' => [
            [ '013', '013', 'ARCOS DE VALDEVEZ' ] ] },
        'Castanheira' => {
          '4970' => [
            [ '145', '145', 'EXTREMO' ] ] },
        'Mourisca' => {
          '4970' => [
            [ '280', '280', 'PORTELA AVV' ] ] },
        'Codeceira' => {
          '4970' => [
            [ '285', '285', 'PROZELO AVV' ] ] },
        'Casal do Eido' => {
          '4970' => [
            [ '750', '750', 'VILELA AVV' ] ] },
        'Boavista' => {
          '4970' => [
            [ '220', '220', 'MONTE REDONDO AVV' ],
            [ '261', '268', 'ARCOS DE VALDEVEZ' ] ] },
        'Outeiro' => {
          '4970' => [
            [ '500', '500', 'PADREIRO (SALVADOR)' ] ] },
        'Secas de Giela' => {
          '4970' => [
            [ '155', '160', 'ARCOS DE VALDEVEZ' ] ] },
        'Alccer' => {
          '4970' => [
            [ '511', '544', 'TVORA (SANTA MARIA)' ] ] },
        'Aldeia' => {
          '4970' => [
            [ '341', '359', 'RIO DE MOINHOS AVV' ] ] },
        'Alqueires' => {
          '4970' => [
            [ '291', '301', 'RIO CABRO' ] ] },
        'Arcos de Valdevez' => {
          '4970' => [
            [ '431', '438', 'ARCOS DE VALDEVEZ' ],
            [ '440', '472', 'ARCOS DE VALDEVEZ' ],
            [ '591', '599', 'ARCOS DE VALDEVEZ' ],
            [ '601', '606', 'ARCOS DE VALDEVEZ' ] ],
          '4974' => [
            [ '001', '011', 'ARCOS DE VALDEVEZ' ] ] },
        'So Miguel' => {
          '4970' => [
            [ '200', '200', 'LOUREDA' ] ] },
        'Eirado' => {
          '4970' => [
            [ '135', '135', 'EIRAS AVV' ] ] },
        'Quintes' => {
          '4970' => [
            [ '255', '255', 'PADROSO AVV' ] ] },
        'So Sebastio' => {
          '4970' => [
            [ '100', '100', 'CABREIRO' ] ] },
        'Barreiro' => {
          '4970' => [
            [ '561', '588', 'SO JORGE AVV' ] ] },
        'Portela' => {
          '4970' => [
            [ '630', '630', 'SO PAIO JOLDA' ] ] }
               },
      'Paredes de Coura' => {
        'Rande' => {
          '4940' => [
            [ '570', '574', 'PAREDES DE COURA' ] ] },
        'Balsa' => {
          '4940' => [
            [ '321', '325', 'INFESTA PCR' ] ] },
        'Boa Vista' => {
          '4940' => [
            [ '221', '236', 'CUNHA PCR' ] ] },
        'Agualonga' => {
          '4940' => [
            [ '011', '035', 'AGUALONGA' ] ] },
        'Bolncia' => {
          '4940' => [
            [ '131', '136', 'COSSOURADO PCR' ] ] },
        'Cascalha' => {
          '4940' => [
            [ '061', '071', 'BICO PCR' ] ] },
        'Angustias' => {
          '4940' => [
            [ '401', '414', 'PADORNELO' ] ] },
        'Alvarim' => {
          '4940' => [
            [ '431', '442', 'PARADA PCR' ] ] },
        'Portelinha' => {
          '4940' => [
            [ '037', '043', 'AGUALONGA' ] ] },
        'Almas' => {
          '4940' => [
            [ '601', '604', 'PORREIRAS' ] ] },
        'Afe' => {
          '4940' => [
            [ '381', '387', 'MOZELOS PCR' ] ] },
        'Ageito' => {
          '4940' => [
            [ '681', '692', 'RUBIES' ] ] },
        'Batoca' => {
          '4940' => [
            [ '181', '187', 'CRISTELO PCR' ] ] },
        'Agro do Monte' => {
          '4940' => [
            [ '651', '669', 'ROMARIGES' ] ] },
        'Encouradas' => {
          '4940' => [
            [ '290', '304', 'FORMARIZ' ] ] },
        'Corujeira' => {
          '4940' => [
            [ '189', '210', 'CRISTELO PCR' ] ] },
        'Agrela' => {
          '4940' => [
            [ '281', '288', 'FORMARIZ' ],
            [ '341', '356', 'INSALDE' ] ] },
        'Aldeia' => {
          '4940' => [
            [ '701', '715', 'VASCES' ] ] },
        'Cabanela' => {
          '4940' => [
            [ '501', '505', 'PAREDES DE COURA' ] ] },
        'Arestim' => {
          '4940' => [
            [ '371', '375', 'LINHARES PCR' ] ] },
        'Paredes de Coura' => {
          '4940' => [
            [ '520', '543', 'PAREDES DE COURA' ] ] },
        'Amieira' => {
          '4940' => [
            [ '621', '635', 'RESENDE PCR' ] ] },
        'Cachadinha' => {
          '4940' => [
            [ '101', '117', 'CASTANHEIRA PCR' ] ] },
        'Carreiros' => {
          '4940' => [
            [ '251', '265', 'FERREIRA PCR' ] ] },
        'Barreiros' => {
          '4940' => [
            [ '151', '162', 'COURA PCR' ] ] }
              },
      'Viana do Castelo' => {
        'Azenha' => {
          '4925' => [
            [ '605', '616', 'TORRE VCT' ] ] },
        'Vila Fria' => {
          '4935' => [
            [ '800', '893', 'VILA FRIA VCT' ] ] },
        'Barrosa' => {
          '4925' => [
            [ '360', '368', 'FREIXIEIRO DE SOUTELO' ] ] },
        'Alvas' => {
          '4905' => [
            [ '300', '325', 'BARROSELAS' ] ] },
        'Vila Nova de Anha' => {
          '4935' => [
            [ '300', '417', 'VILA NOVA DE ANHA' ] ] },
        'Orfos' => {
          '4910' => [
            [ '300', '300', 'FREIXIEIRO DE SOUTELO' ] ] },
        'Caminho da Breia' => {
          '4925' => [
            [ '500', '527', 'NOGUEIRA VCT' ] ] },
        'Calvrio' => {
          '4905' => [
            [ '201', '215', 'ALVARES' ] ] },
        'Alm do Rio' => {
          '4925' => [
            [ '540', '551', 'OUTEIRO VCT' ] ] },
        'Cortegaa' => {
          '4905' => [
            [ '631', '633', 'SUBPORTELA' ] ] },
        'Carro' => {
          '4900' => [
            [ '278', '281', 'CARREO' ] ] },
        'Vilar' => {
          '4925' => [
            [ '670', '678', 'VILAR DE MURTEDA' ] ],
          '4905' => [
            [ '581', '581', 'GERAZ LIMA (STA LEOCDIA)' ] ] },
        'Ventoso' => {
          '4905' => [
            [ '577', '578', 'GERAZ LIMA (STA LEOCDIA)' ] ] },
        'Mazarefes' => {
          '4935' => [
            [ '430', '495', 'MAZAREFES' ] ] },
        'Breia' => {
          '4925' => [
            [ '340', '349', 'CARDIELOS' ] ] },
        'Viana do Castelo' => {
          '4904' => [
            [ '852', '853', 'VIANA DO CASTELO' ],
            [ '856', '862', 'VIANA DO CASTELO' ],
            [ '864', '864', 'VIANA DO CASTELO' ],
            [ '866', '866', 'VIANA DO CASTELO' ],
            [ '869', '878', 'VIANA DO CASTELO' ],
            [ '882', '882', 'VIANA DO CASTELO' ] ],
          '4925' => [
            [ '001', '188', 'VIANA DO CASTELO' ],
            [ '190', '197', 'VIANA DO CASTELO' ] ],
          '4900' => [
            [ '001', '010', 'VIANA DO CASTELO' ],
            [ '013', '014', 'VIANA DO CASTELO' ],
            [ '016', '020', 'VIANA DO CASTELO' ],
            [ '033', '034', 'VIANA DO CASTELO' ],
            [ '039', '039', 'VIANA DO CASTELO' ],
            [ '041', '044', 'VIANA DO CASTELO' ],
            [ '046', '048', 'VIANA DO CASTELO' ],
            [ '201', '201', 'VIANA DO CASTELO' ],
            [ '203', '204', 'VIANA DO CASTELO' ],
            [ '213', '215', 'VIANA DO CASTELO' ],
            [ '258', '270', 'VIANA DO CASTELO' ],
            [ '282', '433', 'VIANA DO CASTELO' ],
            [ '435', '484', 'VIANA DO CASTELO' ],
            [ '486', '486', 'VIANA DO CASTELO' ],
            [ '488', '492', 'VIANA DO CASTELO' ],
            [ '494', '517', 'VIANA DO CASTELO' ],
            [ '519', '547', 'VIANA DO CASTELO' ],
            [ '549', '567', 'VIANA DO CASTELO' ],
            [ '569', '569', 'VIANA DO CASTELO' ],
            [ '571', '571', 'VIANA DO CASTELO' ],
            [ '573', '592', 'VIANA DO CASTELO' ],
            [ '594', '600', 'VIANA DO CASTELO' ],
            [ '640', '644', 'VIANA DO CASTELO' ],
            [ '646', '665', 'VIANA DO CASTELO' ],
            [ '667', '859', 'VIANA DO CASTELO' ],
            [ '861', '937', 'VIANA DO CASTELO' ],
            [ '959', '962', 'VIANA DO CASTELO' ] ],
          '4935' => [
            [ '055', '083', 'VIANA DO CASTELO' ],
            [ '085', '171', 'VIANA DO CASTELO' ],
            [ '174', '201', 'VIANA DO CASTELO' ],
            [ '203', '230', 'VIANA DO CASTELO' ],
            [ '233', '249', 'VIANA DO CASTELO' ] ] },
        'Arques' => {
          '4905' => [
            [ '641', '662', 'VILA DE PUNHE' ] ] },
        'Agrinha' => {
          '4905' => [
            [ '221', '244', 'CARVOEIRO VCT' ] ] },
        'Madorra' => {
          '4925' => [
            [ '576', '578', 'PERRE' ] ] },
        'Coveiras' => {
          '4925' => [
            [ '590', '598', 'SERRELEIS' ] ] },
        'Abrigueiro' => {
          '4925' => [
            [ '450', '459', 'MEIXEDO VCT' ] ] },
        'Agra de Reiros' => {
          '4905' => [
            [ '271', '286', 'MOREIRA DE GERAZ DO LIMA' ] ] },
        'Vila Franca' => {
          '4935' => [
            [ '620', '716', 'VILA FRANCA' ] ] },
        'Areia' => {
          '4935' => [
            [ '560', '573', 'CASTELO DO NEIVA' ] ] },
        'Aldeia' => {
          '4925' => [
            [ '300', '330', 'AMONDE' ],
            [ '470', '492', 'MONTARIA' ] ],
          '4905' => [
            [ '251', '257', 'DEO' ] ] },
        'Casqueira' => {
          '4905' => [
            [ '511', '518', 'MUJES' ] ] },
        'Feira' => {
          '4905' => [
            [ '327', '505', 'BARROSELAS' ] ] },
        'Neiva' => {
          '4935' => [
            [ '231', '232', 'NEIVA' ],
            [ '500', '555', 'NEIVA' ] ] },
        'Costa' => {
          '4925' => [
            [ '572', '574', 'PERRE' ] ],
          '4905' => [
            [ '261', '261', 'DEOCRISTE' ] ] },
        'Amorosa' => {
          '4935' => [
            [ '580', '594', 'CHAF' ] ] },
        'Agros' => {
          '4905' => [
            [ '551', '575', 'GERAZ LIMA (STA LEOCDIA)' ] ] },
        'Agra' => {
          '4925' => [
            [ '400', '430', 'LANHESES' ],
            [ '630', '657', 'VILA MOU' ] ],
          '4905' => [
            [ '531', '543', 'PORTELA SUS' ],
            [ '591', '618', 'GERAZ LIMA (STA MARIA)' ] ] },
        'Perre' => {
          '4925' => [
            [ '580', '587', 'PERRE' ] ] },
        'Outrelo' => {
          '4905' => [
            [ '665', '667', 'VILA DE PUNHE' ] ] },
        'Afife' => {
          '4900' => [
            [ '011', '012', 'AFIFE' ] ] }
              },
      'Melgao' => {
        'Carvalheira' => {
          '4960' => [
            [ '010', '010', 'ALVAREDO' ] ] },
        'Teso' => {
          '4960' => [
            [ '170', '170', 'LAMAS DE MOURO' ] ] },
        'Alm do Rio' => {
          '4960' => [
            [ '023', '034', 'CASTRO LABOREIRO' ] ] },
        'Martingo' => {
          '4960' => [
            [ '330', '330', 'REMOES' ] ] },
        'Moinhos' => {
          '4960' => [
            [ '609', '619', 'MELGAO' ] ] },
        'Almas' => {
          '4960' => [
            [ '420', '469', 'SO PAIO MLG' ] ] },
        'Adegas' => {
          '4960' => [
            [ '340', '400', 'ROUSSAS' ] ] },
        'Ponte do Buraco' => {
          '4960' => [
            [ '259', '277', 'PADERNE MLG' ] ] },
        'Chos' => {
          '4960' => [
            [ '051', '096', 'CASTRO LABOREIRO' ] ] },
        'Adrofreire' => {
          '4960' => [
            [ '020', '021', 'CASTRO LABOREIRO' ] ] },
        'Cousso' => {
          '4960' => [
            [ '120', '120', 'COUSSO' ] ] },
        'Pedregal' => {
          '4960' => [
            [ '130', '130', 'CRISTOVAL' ] ] },
        'Cortegada' => {
          '4960' => [
            [ '300', '300', 'PARADA DO MONTE' ] ] },
        'Branda da Aveleira' => {
          '4960' => [
            [ '160', '160', 'GAVE' ] ] },
        'Casal Maninho' => {
          '4960' => [
            [ '222', '222', 'ALVAREDO' ] ] },
        'Carvalhal' => {
          '4960' => [
            [ '320', '320', 'PRADO' ] ] },
        'Aldeia' => {
          '4960' => [
            [ '201', '221', 'PADERNE MLG' ] ] },
        'Vila' => {
          '4960' => [
            [ '521', '522', 'MELGAO' ],
            [ '524', '531', 'MELGAO' ],
            [ '534', '537', 'MELGAO' ],
            [ '540', '546', 'MELGAO' ],
            [ '548', '549', 'MELGAO' ],
            [ '551', '558', 'MELGAO' ],
            [ '560', '571', 'MELGAO' ],
            [ '573', '576', 'MELGAO' ] ] },
        'Montarro' => {
          '4960' => [
            [ '110', '110', 'CHAVIES' ] ] },
        'Beleco' => {
          '4960' => [
            [ '180', '180', 'PAOS' ] ] },
        'Assadura' => {
          '4960' => [
            [ '501', '519', 'MELGAO' ] ] },
        'Cividade' => {
          '4960' => [
            [ '223', '255', 'PADERNE MLG' ] ] },
        'Monte Crasto' => {
          '4960' => [
            [ '310', '310', 'PENSO MLG' ] ] },
        'Melgao' => {
          '4960' => [
            [ '578', '582', 'MELGAO' ],
            [ '584', '584', 'MELGAO' ],
            [ '587', '588', 'MELGAO' ],
            [ '592', '592', 'MELGAO' ],
            [ '594', '594', 'MELGAO' ],
            [ '596', '596', 'MELGAO' ],
            [ '600', '601', 'MELGAO' ],
            [ '603', '603', 'MELGAO' ] ] },
        'Baixo' => {
          '4960' => [
            [ '140', '140', 'CUBALHO' ] ] },
        'Portocarreiro' => {
          '4960' => [
            [ '150', '150', 'FIES MLG' ] ] }
                 }
        },
    'Beja' => {
      'Serpa' => {
        'Serpa' => {
          '7830' => [
            [ '320', '330', 'SERPA' ],
            [ '332', '380', 'SERPA' ],
            [ '382', '387', 'SERPA' ],
            [ '389', '389', 'SERPA' ],
            [ '392', '407', 'SERPA' ],
            [ '409', '412', 'SERPA' ],
            [ '414', '437', 'SERPA' ],
            [ '439', '439', 'SERPA' ],
            [ '441', '446', 'SERPA' ],
            [ '448', '460', 'SERPA' ],
            [ '462', '464', 'SERPA' ],
            [ '489', '491', 'SERPA' ],
            [ '493', '500', 'SERPA' ] ] },
        'Monte Fronteira' => {
          '7830' => [
            [ '301', '304', 'SERPA' ] ] },
        'Vila Nova de So Bento' => {
          '7830' => [
            [ '049', '078', 'VILA NOVA DE SO BENTO' ],
            [ '080', '105', 'VILA NOVA DE SO BENTO' ],
            [ '465', '468', 'VILA NOVA DE SO BENTO' ] ] },
        'A do Pinto' => {
          '7830' => [
            [ '011', '023', 'VILA NOVA DE SO BENTO' ] ] },
        'Mina da Orada' => {
          '7830' => [
            [ '470', '474', 'PIAS SRP' ] ] },
        'Vale de Vargo' => {
          '7830' => [
            [ '469', '469', 'VALE DE VARGO' ] ] },
        'Pias' => {
          '7830' => [
            [ '201', '211', 'PIAS SRP' ],
            [ '213', '297', 'PIAS SRP' ] ] },
        'Vila Verde de Ficalho' => {
          '7830' => [
            [ '480', '488', 'VILA VERDE DE FICALHO' ],
            [ '594', '595', 'VILA VERDE DE FICALHO' ],
            [ '598', '604', 'VILA VERDE DE FICALHO' ],
            [ '620', '622', 'VILA VERDE DE FICALHO' ],
            [ '624', '624', 'VILA VERDE DE FICALHO' ],
            [ '626', '626', 'VILA VERDE DE FICALHO' ],
            [ '628', '639', 'VILA VERDE DE FICALHO' ],
            [ '641', '651', 'VILA VERDE DE FICALHO' ],
            [ '653', '655', 'VILA VERDE DE FICALHO' ],
            [ '657', '658', 'VILA VERDE DE FICALHO' ],
            [ '660', '663', 'VILA VERDE DE FICALHO' ],
            [ '665', '665', 'VILA VERDE DE FICALHO' ],
            [ '667', '667', 'VILA VERDE DE FICALHO' ] ] },
        'Monte da M' => {
          '7830' => [
            [ '475', '478', 'SERPA' ] ] },
        'Brinches' => {
          '7830' => [
            [ '111', '153', 'BRINCHES' ] ] },
        'Horta dos Machados' => {
          '7830' => [
            [ '501', '533', 'VALE DE VARGO' ] ] }
         },
      'Almodvar' => {
        'Moinho das Mestras' => {
          '7700' => [
            [ '240', '240', 'SANTA CLARA-A-NOVA' ] ] },
        'Monte da Fornalha' => {
          '7700' => [
            [ '250', '253', 'SANTA CRUZ ADV' ] ] },
        'Almodvar' => {
          '7700' => [
            [ '011', '016', 'ALMODVAR' ],
            [ '018', '038', 'ALMODVAR' ],
            [ '040', '089', 'ALMODVAR' ],
            [ '091', '094', 'ALMODVAR' ] ] },
        'Candenilhas' => {
          '7700' => [
            [ '201', '215', 'ALMODVAR' ] ] },
        'Aldeia dos Fernandes' => {
          '7700' => [
            [ '301', '328', 'ALDEIA DOS FERNANDES' ] ] },
        'Rosario' => {
          '7700' => [
            [ '235', '235', 'ROSRIO' ] ] },
        'Monte do Testa' => {
          '7700' => [
            [ '232', '232', 'ROSRIO' ] ] },
        'Monte Casa Velha da Botelha' => {
          '7700' => [
            [ '270', '273', 'SENHORA GRAA DE PADRES' ] ] },
        'Serro das Covas' => {
          '7700' => [
            [ '260', '263', 'SO BARNAB' ] ] },
        'Antas de Baixo' => {
          '7700' => [
            [ '221', '222', 'GOMES AIRES' ] ] }
             },
      'Ourique' => {
        'Monte Alameda' => {
          '7670' => [
            [ '501', '504', 'SANTA LUZIA ORQ' ] ] },
        'Alcarias' => {
          '7670' => [
            [ '011', '011', 'CONCEIO ORQ' ] ] },
        'Ourique' => {
          '7670' => [
            [ '250', '253', 'OURIQUE' ],
            [ '255', '263', 'OURIQUE' ],
            [ '265', '272', 'OURIQUE' ],
            [ '274', '274', 'OURIQUE' ],
            [ '276', '276', 'OURIQUE' ],
            [ '278', '304', 'OURIQUE' ],
            [ '360', '378', 'OURIQUE' ] ] },
        'Alcaria de Ferno Vaz' => {
          '7670' => [
            [ '201', '217', 'OURIQUE' ] ] },
        'Conceio' => {
          '7670' => [
            [ '021', '021', 'CONCEIO ORQ' ] ] },
        'Monte Major' => {
          '7670' => [
            [ '166', '174', 'GARVO' ] ] },
        'Cerro Guincho' => {
          '7670' => [
            [ '601', '614', 'SANTANA DA SERRA' ] ] },
        'Carvalheiras' => {
          '7670' => [
            [ '111', '112', 'GARVO' ] ] },
        'Picaduras' => {
          '7670' => [
            [ '350', '354', 'OURIQUE' ] ] },
        'Monte Arzil' => {
          '7670' => [
            [ '161', '164', 'GARVO' ] ] },
        'Barragem Rocha' => {
          '7670' => [
            [ '401', '407', 'PANOIAS ORQ' ] ] },
        'Garvo' => {
          '7670' => [
            [ '121', '147', 'GARVO' ] ] }
           },
      'Alvito' => {
        'Vila Nova da Baronia' => {
          '7920' => [
            [ '351', '397', 'VILA NOVA DA BARONIA' ] ] },
        'Quinta do Poo Novo' => {
          '7920' => [
            [ '201', '201', 'ALVITO' ] ] },
        'Herdade do Sobral' => {
          '7920' => [
            [ '301', '301', 'VILA NOVA DA BARONIA' ] ] },
        'Alvito' => {
          '7920' => [
            [ '011', '067', 'ALVITO' ] ] }
          },
      'Mrtola' => {
        'Mina de So Domingos' => {
          '7750' => [
            [ '120', '172', 'CORTE DO PINTO' ] ] },
        'Amendoeira do Campo' => {
          '7750' => [
            [ '017', '061', 'ALCARIA RUIVA' ] ] },
        'lvares' => {
          '7750' => [
            [ '501', '517', 'SO JOO DOS CALDEIREIROS' ] ] },
        'Mrtola' => {
          '7750' => [
            [ '320', '369', 'MRTOLA' ] ] },
        'Belo' => {
          '7750' => [
            [ '801', '802', 'SO SEBASTIO DOS CARROS' ] ] },
        'Milhouro' => {
          '7750' => [
            [ '375', '389', 'MRTOLA' ] ] },
        'Achada Se So Sebastio' => {
          '7750' => [
            [ '295', '308', 'MRTOLA' ] ] },
        'Achada do Gamo' => {
          '7750' => [
            [ '401', '418', 'SANTANA DE CAMBAS' ] ] },
        'Monte Barba' => {
          '7750' => [
            [ '190', '190', 'CORTE DO PINTO' ] ] },
        'Monte do Sossego' => {
          '7750' => [
            [ '195', '195', 'CORTE DO PINTO' ] ] },
        'Roucanito' => {
          '7750' => [
            [ '225', '228', 'ESPRITO SANTO' ] ] },
        'Corte Pequena' => {
          '7750' => [
            [ '310', '314', 'MRTOLA' ] ] },
        'Alcaria Longa' => {
          '7750' => [
            [ '601', '630', 'SO MIGUEL DO PINHEIRO' ] ] },
        'gua Salgada' => {
          '7750' => [
            [ '011', '015', 'ALCARIA RUIVA' ] ] },
        'Corte Azinha' => {
          '7750' => [
            [ '101', '103', 'CORTE DO PINTO' ] ] },
        'Palaqueira' => {
          '7750' => [
            [ '220', '222', 'ESPRITO SANTO' ] ] },
        'Carros' => {
          '7750' => [
            [ '805', '812', 'SO SEBASTIO DOS CARROS' ] ] },
        'lamo' => {
          '7750' => [
            [ '201', '217', 'ESPRITO SANTO' ] ] },
        'Barranco' => {
          '7750' => [
            [ '701', '717', 'SO PEDRO DE SOLIS' ] ] }
           },
      'Vidigueira' => {
        'Horta da Aparia' => {
          '7960' => [
            [ '411', '412', 'VILA DE FRADES' ] ] },
        'Vidigueira' => {
          '7960' => [
            [ '211', '305', 'VIDIGUEIRA' ] ] },
        'Pedrgo do Alentejo' => {
          '7960' => [
            [ '021', '060', 'PEDRGO VDG' ] ] },
        'Marmelar' => {
          '7960' => [
            [ '011', '013', 'PEDRGO VDG' ] ] },
        'Alcaria da Serra' => {
          '7960' => [
            [ '111', '122', 'SELMES' ] ] },
        'Vila de Frades' => {
          '7960' => [
            [ '421', '458', 'VILA DE FRADES' ] ] },
        'Cortes de Baixo' => {
          '7960' => [
            [ '131', '134', 'SELMES' ] ] },
        'Selmes' => {
          '7960' => [
            [ '151', '189', 'SELMES' ] ] }
              },
      'Aljustrel' => {
        'Barragem' => {
          '7600' => [
            [ '201', '252', 'ERVIDEL' ] ] },
        'Carregueiro' => {
          '7600' => [
            [ '160', '163', 'ALJUSTREL' ] ] },
        'Monte Branco' => {
          '7600' => [
            [ '501', '503', 'RIO DE MOINHOS AJT' ] ] },
        'Aldeia Nova' => {
          '7600' => [
            [ '401', '436', 'SO JOO DE NEGRILHOS' ] ] },
        'Aljustrel' => {
          '7600' => [
            [ '010', '047', 'ALJUSTREL' ],
            [ '050', '101', 'ALJUSTREL' ],
            [ '103', '108', 'ALJUSTREL' ],
            [ '110', '139', 'ALJUSTREL' ] ] },
        'Monte Aguentinho' => {
          '7600' => [
            [ '371', '374', 'MESSEJANA' ] ] },
        'Messejana' => {
          '7600' => [
            [ '310', '350', 'MESSEJANA' ] ] },
        'Herdade do Salto' => {
          '7600' => [
            [ '270', '274', 'ERVIDEL' ] ] },
        'Montinho Roxo' => {
          '7600' => [
            [ '450', '451', 'SO JOO DE NEGRILHOS' ] ] },
        'Monte da Manteira' => {
          '7600' => [
            [ '165', '180', 'ALJUSTREL' ] ] },
        'Aldeia dos Elvas' => {
          '7600' => [
            [ '301', '304', 'MESSEJANA' ] ] }
             },
      'Ferreira do Alentejo' => {
        'Abegoaria' => {
          '7900' => [
            [ '111', '117', 'FERREIRA DO ALENTEJO' ] ] },
        'Odivelas' => {
          '7900' => [
            [ '371', '374', 'ODIVELAS FAL' ],
            [ '376', '379', 'ODIVELAS FAL' ],
            [ '381', '390', 'ODIVELAS FAL' ],
            [ '392', '405', 'ODIVELAS FAL' ] ] },
        'Vale Barroso' => {
          '7900' => [
            [ '420', '421', 'ODIVELAS FAL' ] ] },
        'Herdade das Quintinhas' => {
          '7900' => [
            [ '345', '360', 'ODIVELAS FAL' ] ] },
        'Herdade da Cameirinha' => {
          '7900' => [
            [ '280', '314', 'FIGUEIRA DOS CAVALEIROS' ] ] },
        'Figueira dos Cavaleiros' => {
          '7900' => [
            [ '231', '258', 'FIGUEIRA DOS CAVALEIROS' ],
            [ '260', '271', 'FIGUEIRA DOS CAVALEIROS' ] ] },
        'Arneiro dos Tremoos' => {
          '7900' => [
            [ '211', '214', 'FIGUEIRA DOS CAVALEIROS' ] ] },
        'Peroguarda' => {
          '7900' => [
            [ '018', '018', 'PEROGUARDA' ],
            [ '452', '458', 'PEROGUARDA' ],
            [ '461', '473', 'PEROGUARDA' ] ] },
        'Ferreira do Alentejo' => {
          '7900' => [
            [ '551', '556', 'FERREIRA DO ALENTEJO' ],
            [ '558', '558', 'FERREIRA DO ALENTEJO' ],
            [ '560', '562', 'FERREIRA DO ALENTEJO' ],
            [ '564', '587', 'FERREIRA DO ALENTEJO' ],
            [ '589', '599', 'FERREIRA DO ALENTEJO' ],
            [ '602', '602', 'FERREIRA DO ALENTEJO' ],
            [ '604', '606', 'FERREIRA DO ALENTEJO' ],
            [ '608', '611', 'FERREIRA DO ALENTEJO' ],
            [ '613', '619', 'FERREIRA DO ALENTEJO' ],
            [ '621', '654', 'FERREIRA DO ALENTEJO' ],
            [ '656', '669', 'FERREIRA DO ALENTEJO' ],
            [ '671', '679', 'FERREIRA DO ALENTEJO' ],
            [ '681', '684', 'FERREIRA DO ALENTEJO' ] ] },
        'Monte do Outeiro' => {
          '7900' => [
            [ '526', '526', 'CANHESTROS' ] ] },
        'Fortes' => {
          '7900' => [
            [ '131', '131', 'FERREIRA DO ALENTEJO' ] ] },
        'Gaspares' => {
          '7900' => [
            [ '133', '198', 'FERREIRA DO ALENTEJO' ] ] },
        'Monte da Amendoeira' => {
          '7900' => [
            [ '076', '096', 'ALFUNDO' ] ] },
        'Caneiras Grandes' => {
          '7900' => [
            [ '342', '343', 'ODIVELAS FAL' ] ] },
        'Alfundo' => {
          '7900' => [
            [ '011', '017', 'ALFUNDO' ],
            [ '019', '024', 'ALFUNDO' ],
            [ '026', '029', 'ALFUNDO' ],
            [ '031', '036', 'ALFUNDO' ],
            [ '038', '039', 'ALFUNDO' ],
            [ '041', '063', 'ALFUNDO' ] ] },
        'Canhestros' => {
          '7900' => [
            [ '491', '513', 'CANHESTROS' ] ] },
        'Herdade do Pinheiro' => {
          '7900' => [
            [ '441', '443', 'PEROGUARDA' ] ] }
            },
      'Cuba' => {
        'Vila Ruiva' => {
          '7940' => [
            [ '451', '480', 'VILA RUIVA CUB' ] ] },
        'Vila Alva' => {
          '7940' => [
            [ '370', '397', 'VILA ALVA' ] ] },
        'Cuba' => {
          '7940' => [
            [ '001', '003', 'CUBA' ],
            [ '005', '028', 'CUBA' ],
            [ '100', '104', 'CUBA' ],
            [ '106', '192', 'CUBA' ] ] },
        'Albergaria dos Fusos' => {
          '7940' => [
            [ '411', '417', 'VILA RUIVA CUB' ] ] },
        'Faro do Alentejo' => {
          '7940' => [
            [ '311', '314', 'FARO DO ALENTEJO' ] ] },
        'Barragem do Alvito' => {
          '7940' => [
            [ '351', '358', 'VILA ALVA' ] ] },
        'Horta das Manteigas' => {
          '7940' => [
            [ '250', '259', 'CUBA' ] ] }
        },
      'Barrancos' => {
        'Barrancos' => {
          '7230' => [
            [ '001', '053', 'BARRANCOS' ] ] }
             },
      'Moura' => {
        'Vale Formoso' => {
          '7885' => [
            [ '290', '290', 'PVOA DE SO MIGUEL' ] ] },
        'Safara' => {
          '7875' => [
            [ '051', '077', 'SAFARA' ] ] },
        'Amareleja' => {
          '7885' => [
            [ '011', '013', 'AMARELEJA' ],
            [ '015', '016', 'AMARELEJA' ],
            [ '018', '018', 'AMARELEJA' ],
            [ '021', '034', 'AMARELEJA' ],
            [ '037', '046', 'AMARELEJA' ],
            [ '048', '056', 'AMARELEJA' ],
            [ '058', '058', 'AMARELEJA' ],
            [ '060', '074', 'AMARELEJA' ],
            [ '076', '080', 'AMARELEJA' ] ] },
        'Herdade da Negrita' => {
          '7875' => [
            [ '101', '106', 'STO ALEIXO DA RESTAURAO' ] ] },
        'Monte da Fonte dos Arcos' => {
          '7875' => [
            [ '011', '012', 'SAFARA' ] ] },
        'Monte da Botelhinha' => {
          '7875' => [
            [ '201', '205', 'SANTO AMADOR' ] ] },
        'Monte Airoso' => {
          '7885' => [
            [ '201', '230', 'PVOA DE SO MIGUEL' ] ] },
        'Sobral da Adia' => {
          '7875' => [
            [ '350', '391', 'SOBRAL DA ADIA' ] ] },
        'Pvoa de So Miguel' => {
          '7885' => [
            [ '251', '280', 'PVOA DE SO MIGUEL' ] ] },
        'Azeiteira' => {
          '7885' => [
            [ '100', '122', 'AMARELEJA' ] ] },
        'Moura' => {
          '7860' => [
            [ '001', '005', 'MOURA' ],
            [ '007', '013', 'MOURA' ],
            [ '015', '024', 'MOURA' ],
            [ '027', '028', 'MOURA' ],
            [ '031', '045', 'MOURA' ],
            [ '047', '051', 'MOURA' ],
            [ '053', '076', 'MOURA' ],
            [ '101', '104', 'MOURA' ],
            [ '107', '112', 'MOURA' ],
            [ '114', '114', 'MOURA' ],
            [ '116', '121', 'MOURA' ],
            [ '125', '136', 'MOURA' ],
            [ '139', '139', 'MOURA' ],
            [ '141', '160', 'MOURA' ],
            [ '162', '170', 'MOURA' ],
            [ '172', '177', 'MOURA' ],
            [ '179', '193', 'MOURA' ],
            [ '195', '198', 'MOURA' ],
            [ '201', '204', 'MOURA' ],
            [ '207', '208', 'MOURA' ],
            [ '211', '218', 'MOURA' ],
            [ '221', '224', 'MOURA' ],
            [ '227', '228', 'MOURA' ],
            [ '231', '234', 'MOURA' ],
            [ '237', '240', 'MOURA' ],
            [ '243', '245', 'MOURA' ] ] },
        'Quinta de So Sebastio' => {
          '7860' => [
            [ '351', '354', 'MOURA' ] ] },
        'Monte da Chamin' => {
          '7875' => [
            [ '301', '304', 'SOBRAL DA ADIA' ] ] },
        'Pomarinho das Cinco Oliveiras' => {
          '7860' => [
            [ '301', '307', 'MOURA' ] ] },
        'Santo Amador' => {
          '7875' => [
            [ '250', '269', 'SANTO AMADOR' ] ] },
        'Santo Aleixo da Restaurao' => {
          '7875' => [
            [ '150', '185', 'STO ALEIXO DA RESTAURAO' ] ] }
         },
      'Castro Verde' => {
        'Monte Novo do Visconde' => {
          '7780' => [
            [ '040', '040', 'CASVEL CVR' ] ] },
        'A do Corvo' => {
          '7780' => [
            [ '401', '403', 'SANTA BRBARA DE PADRES' ] ] },
        'Monte Sorraias' => {
          '7780' => [
            [ '591', '591', 'SO MARCOS DA ATABOEIRA' ] ] },
        'Monte da Chada' => {
          '7780' => [
            [ '404', '404', 'CASTRO VERDE' ] ] },
        'Aivados' => {
          '7780' => [
            [ '010', '010', 'CASTRO VERDE' ] ] },
        'Viseus' => {
          '7780' => [
            [ '491', '494', 'SANTA BRBARA DE PADRES' ] ] },
        'Almeirim' => {
          '7780' => [
            [ '101', '102', 'CASTRO VERDE' ] ] },
        'So Marcos da Ataboeira' => {
          '7780' => [
            [ '520', '546', 'SO MARCOS DA ATABOEIRA' ] ] },
        'Entradas' => {
          '7780' => [
            [ '321', '330', 'ENTRADAS' ],
            [ '340', '354', 'ENTRADAS' ] ] },
        'Galeguinha' => {
          '7780' => [
            [ '301', '308', 'ENTRADAS' ] ] },
        'Monte Novo' => {
          '7780' => [
            [ '405', '409', 'SANTA BRBARA DE PADRES' ] ] },
        'Estao de Ourique' => {
          '7780' => [
            [ '030', '030', 'CASTRO VERDE' ] ] },
        'Castro Verde' => {
          '7780' => [
            [ '090', '099', 'CASTRO VERDE' ],
            [ '120', '219', 'CASTRO VERDE' ],
            [ '264', '288', 'CASTRO VERDE' ],
            [ '290', '290', 'CASTRO VERDE' ],
            [ '292', '297', 'CASTRO VERDE' ],
            [ '299', '300', 'CASTRO VERDE' ] ] },
        'Santa Brbara de Padres' => {
          '7780' => [
            [ '450', '458', 'SANTA BRBARA DE PADRES' ],
            [ '460', '484', 'SANTA BRBARA DE PADRES' ],
            [ '486', '486', 'SANTA BRBARA DE PADRES' ],
            [ '488', '489', 'SANTA BRBARA DE PADRES' ] ] },
        'Casvel' => {
          '7780' => [
            [ '020', '020', 'CASVEL CVR' ] ] },
        'Cerca do Poo Novo' => {
          '7780' => [
            [ '250', '262', 'CASTRO VERDE' ] ] },
        'Monte Alcaria do Coelho' => {
          '7780' => [
            [ '501', '504', 'SO MARCOS DA ATABOEIRA' ] ] }
          },
      'Beja' => {
        'Monte Faleira' => {
          '7800' => [
            [ '720', '721', 'SANTA CLARA DE LOUREDO' ] ] },
        'Monte Porto Peles' => {
          '7800' => [
            [ '650', '655', 'NOSSA SENHORA DAS NEVES' ] ] },
        'Baleizo' => {
          '7800' => [
            [ '611', '611', 'BALEIZO' ] ] },
        'Areeiro' => {
          '7800' => [
            [ '660', '661', 'QUINTOS' ] ] },
        'Trigaches' => {
          '7800' => [
            [ '771', '771', 'TRIGACHES' ] ] },
        'Monte Pereiras' => {
          '7800' => [
            [ '630', '631', 'CABEA GORDA' ] ] },
        'Beringel' => {
          '7800' => [
            [ '620', '620', 'BERINGEL' ],
            [ '800', '849', 'BERINGEL' ],
            [ '851', '852', 'BERINGEL' ],
            [ '856', '857', 'BERINGEL' ],
            [ '859', '863', 'BERINGEL' ] ] },
        'Monte Cardeais' => {
          '7800' => [
            [ '760', '761', 'TRINDADE BJA' ] ] },
        'Albernoa' => {
          '7800' => [
            [ '601', '601', 'ALBERNOA' ] ] },
        'Monte Novo Figueirinha' => {
          '7800' => [
            [ '740', '741', 'SO BRISSOS' ] ] },
        'Monte Carrios' => {
          '7800' => [
            [ '730', '733', 'SANTA VITRIA' ] ] },
        'Quinta Horta Cima' => {
          '7800' => [
            [ '670', '708', 'SALVADA' ] ] },
        'Horta Coelho' => {
          '7800' => [
            [ '750', '751', 'SO MATIAS BJA' ] ] },
        'Beja' => {
          '7801' => [
            [ '851', '859', 'BEJA' ] ],
          '7800' => [
            [ '001', '007', 'BEJA' ],
            [ '009', '011', 'BEJA' ],
            [ '013', '105', 'BEJA' ],
            [ '107', '108', 'BEJA' ],
            [ '111', '399', 'BEJA' ],
            [ '401', '532', 'BEJA' ],
            [ '535', '557', 'BEJA' ],
            [ '559', '561', 'BEJA' ],
            [ '563', '571', 'BEJA' ],
            [ '574', '574', 'BEJA' ],
            [ '578', '579', 'BEJA' ],
            [ '581', '581', 'BEJA' ],
            [ '589', '591', 'BEJA' ],
            [ '850', '850', 'BEJA' ] ] },
        'Monte Igreja' => {
          '7800' => [
            [ '640', '641', 'MOMBEJA' ] ] }
        },
      'Odemira' => {
        'Sabia Gare' => {
          '7665' => [
            [ '853', '857', 'SABIA' ] ] },
        'A de Mateus' => {
          '7630' => [
            [ '011', '013', 'ODEMIRA' ] ] },
        'Portas do Transval' => {
          '7630' => [
            [ '208', '208', 'ODEMIRA' ] ] },
        'Pereiras' => {
          '7665' => [
            [ '881', '881', 'PEREIRAS-GARE' ] ] },
        'Cruzamento de Almograve' => {
          '7630' => [
            [ '063', '063', 'LONGUEIRA/ALMOGRAVE' ] ] },
        'Palmeira' => {
          '7630' => [
            [ '201', '206', 'ODEMIRA' ] ] },
        'Chaparralinho' => {
          '7630' => [
            [ '058', '062', 'ODEMIRA' ] ] },
        'Cagarro de Almogreve' => {
          '7630' => [
            [ '041', '041', 'LONGUEIRA/ALMOGRAVE' ] ] },
        'Ameixiais' => {
          '7630' => [
            [ '428', '490', 'SO LUS' ] ] },
        'Calvinos' => {
          '7630' => [
            [ '042', '052', 'ODEMIRA' ] ] },
        'Monte Adail' => {
          '7630' => [
            [ '093', '106', 'ODEMIRA' ] ] },
        'Moncosa' => {
          '7630' => [
            [ '092', '092', 'LONGUEIRA/ALMOGRAVE' ] ] },
        'Manto Branco' => {
          '7630' => [
            [ '087', '091', 'ODEMIRA' ] ] },
        'Degradada' => {
          '7630' => [
            [ '064', '081', 'ODEMIRA' ] ] },
        'Portela da Roda' => {
          '7630' => [
            [ '210', '242', 'ODEMIRA' ] ] },
        'Amiais da Rocha' => {
          '7630' => [
            [ '020', '021', 'ODEMIRA' ] ] },
        'gua de Bacias' => {
          '7630' => [
            [ '561', '592', 'SO TEOTNIO' ] ] },
        'Colos' => {
          '7630' => [
            [ '301', '335', 'COLOS' ],
            [ '341', '366', 'COLOS' ] ] },
        'So Teotnio' => {
          '7630' => [
            [ '611', '655', 'SO TEOTNIO' ] ] },
        'Bicos' => {
          '7630' => [
            [ '711', '722', 'BICOS' ] ] },
        'Odemira' => {
          '7630' => [
            [ '121', '136', 'ODEMIRA' ],
            [ '138', '141', 'ODEMIRA' ],
            [ '143', '145', 'ODEMIRA' ],
            [ '147', '159', 'ODEMIRA' ],
            [ '161', '165', 'ODEMIRA' ],
            [ '167', '174', 'ODEMIRA' ],
            [ '176', '177', 'ODEMIRA' ],
            [ '179', '179', 'ODEMIRA' ],
            [ '182', '191', 'ODEMIRA' ] ] },
        'Almograve' => {
          '7630' => [
            [ '017', '017', 'LONGUEIRA/ALMOGRAVE' ] ] },
        'Corte Sevilha' => {
          '7665' => [
            [ '801', '814', 'SABIA' ] ] },
        'Alcaria do Clemente' => {
          '7630' => [
            [ '731', '744', 'ZAMBUJEIRA DO MAR' ] ] },
        'Bemposta de Cima' => {
          '7630' => [
            [ '028', '028', 'ODEMIRA' ] ] },
        'Cortebrique' => {
          '7665' => [
            [ '858', '880', 'SANTA CLARA-A-VELHA' ] ] },
        'guas Ferrenhas' => {
          '7645' => [
            [ '011', '012', 'VILA NOVA DE MILFONTES' ] ] },
        'Longueira' => {
          '7630' => [
            [ '084', '084', 'LONGUEIRA/ALMOGRAVE' ] ] },
        'Seisseiras' => {
          '7630' => [
            [ '671', '676', 'SO TEOTNIO' ] ] },
        'Casa Nova' => {
          '7630' => [
            [ '287', '291', 'COLOS' ] ] },
        'Boavista dos Pinheiros' => {
          '7630' => [
            [ '033', '033', 'BOAVISTA DOS PINHEIROS' ] ] },
        'Romeiras' => {
          '7665' => [
            [ '892', '892', 'SANTA CLARA-A-VELHA' ] ] },
        'Aldeamento Moinho da Asneira' => {
          '7645' => [
            [ '014', '023', 'VILA NOVA DE MILFONTES' ] ] },
        'Barranco de Totenique' => {
          '7630' => [
            [ '371', '407', 'RELQUIAS' ] ] },
        'Abelheira' => {
          '7630' => [
            [ '511', '544', 'S MARTINHO DAS AMOREIRAS' ] ] },
        'Sabia' => {
          '7665' => [
            [ '819', '851', 'SABIA' ] ] },
        'Bufardas' => {
          '7645' => [
            [ '025', '075', 'VILA NOVA DE MILFONTES' ] ] },
        'Barreiras' => {
          '7630' => [
            [ '691', '698', 'VALE DE SANTIAGO' ] ] },
        'Abego' => {
          '7630' => [
            [ '271', '285', 'COLOS' ] ] },
        'Zambujeira do Mar' => {
          '7630' => [
            [ '761', '801', 'ZAMBUJEIRA DO MAR' ] ] },
        'Vales' => {
          '7630' => [
            [ '244', '248', 'ODEMIRA' ] ] },
        'Voltinhas' => {
          '7665' => [
            [ '891', '891', 'LUZIANES-GARE' ] ] },
        'Vila Nova de Milfontes' => {
          '7645' => [
            [ '211', '305', 'VILA NOVA DE MILFONTES' ],
            [ '309', '321', 'VILA NOVA DE MILFONTES' ] ] },
        'Cavaleiro' => {
          '7630' => [
            [ '055', '055', 'ODEMIRA' ] ] },
        'Bouas' => {
          '7630' => [
            [ '036', '040', 'ODEMIRA' ] ] },
        'Ave-sol' => {
          '7630' => [
            [ '022', '022', 'LONGUEIRA/ALMOGRAVE' ] ] },
        'Barranco' => {
          '7630' => [
            [ '023', '025', 'ODEMIRA' ] ] },
        'Bezerra' => {
          '7630' => [
            [ '030', '031', 'ODEMIRA' ] ] }
           }
        },
    'Ilha de Santa Maria' => {
      'Vila do Porto' => {
        'Vila do Porto' => {
          '9580' => [
            [ '501', '539', 'VILA DO PORTO' ] ] },
        'Almagreira' => {
          '9580' => [
            [ '011', '032', 'ALMAGREIRA VPT' ] ] },
        'Aeroporto' => {
          '9580' => [
            [ '401', '434', 'VILA DO PORTO' ] ] },
        'Alm' => {
          '9580' => [
            [ '201', '240', 'SANTO ESPRITO' ] ] },
        'Anjos' => {
          '9580' => [
            [ '470', '492', 'VILA DO PORTO' ] ] },
        'gua D\'Alto' => {
          '9580' => [
            [ '301', '333', 'SO PEDRO' ] ] },
        'Arrebento' => {
          '9580' => [
            [ '101', '116', 'SANTA BRBARA VPT' ] ] }
              }
           },
    'Viseu' => {
      'Penalva do Castelo' => {
        'Lisei' => {
          '3550' => [
            [ '331', '332', 'TRANCOZELOS' ] ] },
        'Barrocas' => {
          '3550' => [
            [ '122', '127', 'PENALVA DO CASTELO' ] ] },
        'Penalva do Castelo' => {
          '3550' => [
            [ '101', '116', 'PENALVA DO CASTELO' ],
            [ '118', '120', 'PENALVA DO CASTELO' ],
            [ '129', '130', 'PENALVA DO CASTELO' ],
            [ '132', '140', 'PENALVA DO CASTELO' ],
            [ '142', '167', 'PENALVA DO CASTELO' ],
            [ '169', '180', 'PENALVA DO CASTELO' ],
            [ '184', '184', 'PENALVA DO CASTELO' ],
            [ '186', '188', 'PENALVA DO CASTELO' ] ] },
        'Matela' => {
          '3550' => [
            [ '221', '223', 'MATELA PCT' ] ] },
        'Paul' => {
          '3550' => [
            [ '247', '262', 'PINDO' ] ] },
        'Correndinha' => {
          '3550' => [
            [ '239', '245', 'PINDO' ] ] },
        'Arvoredo' => {
          '3550' => [
            [ '181', '183', 'LUSINDE' ] ] },
        'Rosela' => {
          '3550' => [
            [ '030', '059', 'CASTELO DE PENALVA' ] ] },
        'Esmolfe' => {
          '3550' => [
            [ '071', '075', 'ESMOLFE' ] ] },
        'Antas' => {
          '3550' => [
            [ '011', '012', 'ANTAS PCT' ] ] },
        'Mareco' => {
          '3550' => [
            [ '200', '200', 'MARECO' ] ] },
        'Quinta da Regada da Pedra' => {
          '3550' => [
            [ '346', '346', 'PINDO' ] ] },
        'Trancoselinhos' => {
          '3550' => [
            [ '334', '337', 'TRANCOZELOS' ] ] },
        'Pindo' => {
          '3550' => [
            [ '168', '168', 'PINDO' ] ] },
        'Vila Cova do Covelo' => {
          '3550' => [
            [ '350', '350', 'VILA COVA DO COVELO' ] ] },
        'Oliveira' => {
          '3550' => [
            [ '128', '128', 'PINDO' ] ] },
        'Rio' => {
          '3550' => [
            [ '089', '099', 'GERMIL PCT' ] ] },
        'Real' => {
          '3550' => [
            [ '271', '272', 'REAL PCT' ] ] },
        'Sezures' => {
          '3550' => [
            [ '282', '297', 'SEZURES PCT' ],
            [ '299', '328', 'SEZURES PCT' ] ] }
                 },
      'Mortgua' => {
        'Mortgua' => {
          '3450' => [
            [ '120', '164', 'MORTGUA' ] ] },
        'Caparrosa' => {
          '3450' => [
            [ '091', '097', 'MARMELEIRA MRT' ] ] },
        'gua Levada' => {
          '3450' => [
            [ '051', '071', 'ESPINHO MRT' ] ] },
        'Carvalhal' => {
          '3450' => [
            [ '301', '317', 'PALA MRT' ] ] },
        'Alcordal' => {
          '3450' => [
            [ '012', '017', 'CERCOSA' ] ] },
        'Breda' => {
          '3450' => [
            [ '331', '348', 'SOBRAL MRT' ] ] },
        'Pvoa' => {
          '3450' => [
            [ '440', '440', 'VALE DE REMGIO' ] ] },
        'Cerdeira' => {
          '3450' => [
            [ '381', '387', 'TREZI' ] ] },
        'Almacinha' => {
          '3450' => [
            [ '111', '116', 'MORTGUA' ] ] },
        'Vale de Aores' => {
          '3450' => [
            [ '201', '232', 'MORTGUA' ] ] },
        'Povoinha' => {
          '3450' => [
            [ '450', '450', 'VALE DE REMGIO' ] ] },
        'Almaa' => {
          '3450' => [
            [ '011', '011', 'ALMAA' ] ] },
        'Gndara' => {
          '3450' => [
            [ '401', '421', 'VALE DE REMGIO' ] ] },
        'Quinta do Vau' => {
          '3450' => [
            [ '190', '190', 'MORTGUA' ] ] },
        'Vale de Remgio' => {
          '3450' => [
            [ '460', '460', 'VALE DE REMGIO' ] ] },
        'Barraco' => {
          '3450' => [
            [ '031', '036', 'CORTEGAA MRT' ] ] }
             },
      'Sernancelhe' => {
        'Granjal' => {
          '3640' => [
            [ '130', '130', 'GRANJAL' ] ] },
        'Fonte Arcada' => {
          '3640' => [
            [ '110', '110', 'FONTE ARCADA SRN' ] ] },
        'Lugar' => {
          '3640' => [
            [ '014', '019', 'ARNAS' ] ] },
        'Sernancelhe' => {
          '3640' => [
            [ '209', '250', 'SERNANCELHE' ] ] },
        'Tabosa da Cunha' => {
          '3640' => [
            [ '050', '050', 'CUNHA SRN' ] ] },
        'Bairro Vale Escuro' => {
          '3640' => [
            [ '041', '045', 'CHOSENDO' ] ] },
        'Escurquela' => {
          '3640' => [
            [ '060', '060', 'ESCURQUELA' ] ] },
        'Bairro Calvrio' => {
          '3640' => [
            [ '301', '307', 'VILA DA PONTE SRN' ] ] },
        'Arnas' => {
          '3640' => [
            [ '011', '012', 'ARNAS' ] ] },
        'Ferreirim' => {
          '3640' => [
            [ '071', '098', 'FERREIRIM SRN' ] ] },
        'Lamosa' => {
          '3640' => [
            [ '140', '140', 'LAMOSA' ] ] },
        'Freixinho' => {
          '3640' => [
            [ '120', '120', 'FREIXINHO' ] ] },
        'Faia' => {
          '3620' => [
            [ '800', '800', 'FAIA SRN' ] ] },
        'Bem Haja' => {
          '3640' => [
            [ '180', '180', 'SARZEDA' ] ] },
        'Veiga' => {
          '3640' => [
            [ '290', '290', 'SERNANCELHE' ] ] },
        'Tabosa do Carregal' => {
          '3640' => [
            [ '030', '030', 'CARREGAL' ] ] },
        'Mosteiro da Ribeira' => {
          '3640' => [
            [ '201', '203', 'SERNANCELHE' ] ] },
        'Lapa' => {
          '3640' => [
            [ '170', '170', 'QUINTELA' ] ] },
        'Macieira' => {
          '3640' => [
            [ '150', '150', 'MACIEIRA SRN' ] ] },
        'Penso' => {
          '3640' => [
            [ '160', '160', 'PENSO SRN' ] ] }
          },
      'Nelas' => {
        'Caldas da Felgueira' => {
          '3525' => [
            [ '201', '205', 'CANAS DE SENHORIM' ],
            [ '207', '211', 'CANAS DE SENHORIM' ] ] },
        'Santar' => {
          '3520' => [
            [ '121', '147', 'SANTAR NLS' ] ] },
        'Casal Sancho' => {
          '3520' => [
            [ '111', '113', 'SANTAR NLS' ] ] },
        'Vale de Madeiros' => {
          '3525' => [
            [ '350', '352', 'CANAS DE SENHORIM' ],
            [ '354', '357', 'CANAS DE SENHORIM' ],
            [ '359', '366', 'CANAS DE SENHORIM' ],
            [ '368', '369', 'CANAS DE SENHORIM' ] ] },
        'Pvoa de Santo Antnio' => {
          '3525' => [
            [ '250', '259', 'CANAS DE SENHORIM' ],
            [ '261', '261', 'CANAS DE SENHORIM' ],
            [ '263', '269', 'CANAS DE SENHORIM' ] ] },
        'Moreira de Baixo' => {
          '3520' => [
            [ '226', '228', 'MOREIRA NLS' ] ] },
        'Aguieira' => {
          '3525' => [
            [ '501', '517', 'AGUIEIRA' ],
            [ '519', '530', 'AGUIEIRA' ],
            [ '532', '533', 'AGUIEIRA' ] ] },
        'Vilar Seco' => {
          '3520' => [
            [ '225', '225', 'VILAR SECO NLS' ] ] },
        'Nelas' => {
          '3520' => [
            [ '001', '001', 'NELAS' ],
            [ '031', '103', 'NELAS' ] ] },
        'Canas de Senhorim' => {
          '3525' => [
            [ '001', '013', 'CANAS DE SENHORIM' ],
            [ '015', '016', 'CANAS DE SENHORIM' ],
            [ '018', '020', 'CANAS DE SENHORIM' ],
            [ '022', '023', 'CANAS DE SENHORIM' ],
            [ '025', '026', 'CANAS DE SENHORIM' ],
            [ '028', '032', 'CANAS DE SENHORIM' ],
            [ '034', '034', 'CANAS DE SENHORIM' ],
            [ '036', '053', 'CANAS DE SENHORIM' ],
            [ '055', '058', 'CANAS DE SENHORIM' ],
            [ '060', '060', 'CANAS DE SENHORIM' ],
            [ '062', '062', 'CANAS DE SENHORIM' ],
            [ '064', '068', 'CANAS DE SENHORIM' ],
            [ '070', '070', 'CANAS DE SENHORIM' ],
            [ '072', '076', 'CANAS DE SENHORIM' ],
            [ '078', '078', 'CANAS DE SENHORIM' ],
            [ '080', '080', 'CANAS DE SENHORIM' ],
            [ '082', '082', 'CANAS DE SENHORIM' ],
            [ '084', '092', 'CANAS DE SENHORIM' ],
            [ '095', '097', 'CANAS DE SENHORIM' ],
            [ '102', '104', 'CANAS DE SENHORIM' ] ] },
        'Carvalhal Redondo' => {
          '3525' => [
            [ '401', '406', 'CARVALHAL REDONDO' ],
            [ '411', '420', 'CARVALHAL REDONDO' ],
            [ '422', '430', 'CARVALHAL REDONDO' ],
            [ '432', '433', 'CARVALHAL REDONDO' ],
            [ '435', '455', 'CARVALHAL REDONDO' ] ] },
        'Lapa do Lobo' => {
          '3525' => [
            [ '601', '603', 'LAPA DO LOBO' ],
            [ '605', '608', 'LAPA DO LOBO' ],
            [ '610', '614', 'LAPA DO LOBO' ],
            [ '616', '619', 'LAPA DO LOBO' ],
            [ '621', '622', 'LAPA DO LOBO' ],
            [ '624', '628', 'LAPA DO LOBO' ],
            [ '631', '637', 'LAPA DO LOBO' ],
            [ '639', '639', 'LAPA DO LOBO' ],
            [ '641', '646', 'LAPA DO LOBO' ] ] },
        'Caguna' => {
          '3520' => [
            [ '211', '224', 'SENHORIM' ] ] },
        'Algeraz' => {
          '3520' => [
            [ '011', '011', 'NELAS' ] ] },
        'Urgeiria' => {
          '3525' => [
            [ '301', '314', 'CANAS DE SENHORIM' ] ] },
        'Folhadal' => {
          '3520' => [
            [ '021', '021', 'NELAS' ] ] }
          },
      'So Joo da Pesqueira' => {
        'Pereiros' => {
          '5130' => [
            [ '281', '281', 'PEREIROS SJP' ] ] },
        'Treves' => {
          '5130' => [
            [ '421', '421', 'TREVES' ] ] },
        'Quinta de So Martinho' => {
          '5130' => [
            [ '250', '250', 'NAGOSELO DO DOURO' ] ] },
        'Quinta da Boavista' => {
          '5130' => [
            [ '401', '406', 'SOUTELO DO DOURO' ] ] },
        'Bairro Cho de Canelha' => {
          '5130' => [
            [ '051', '057', 'ERVEDOSA DO DOURO' ] ] },
        'Valongo dos Azeites' => {
          '5130' => [
            [ '501', '501', 'VALONGO DOS AZEITES' ] ] },
        'Castanheiro do Sul' => {
          '5130' => [
            [ '021', '034', 'CASTANHEIRO DO SUL' ] ] },
        'So Joo da Pesqueira' => {
          '5130' => [
            [ '321', '356', 'SO JOO DA PESQUEIRA' ] ] },
        'Fundo do Povo' => {
          '5130' => [
            [ '101', '138', 'ERVEDOSA DO DOURO' ] ] },
        'Alto da Fonte da Ladeira' => {
          '5130' => [
            [ '451', '493', 'VALE DE FIGUEIRA SJP' ] ] },
        'Moitas' => {
          '5130' => [
            [ '271', '272', 'PAREDES DA BEIRA' ] ] },
        'Quinta do Piso' => {
          '5130' => [
            [ '011', '011', 'CASTANHEIRO DO SUL' ] ] },
        'Quinta do Banco' => {
          '5130' => [
            [ '252', '252', 'NAGOSELO DO DOURO' ] ] },
        'Espinho' => {
          '5130' => [
            [ '301', '311', 'SO JOO DA PESQUEIRA' ] ] },
        'Nagoselo do Douro' => {
          '5130' => [
            [ '221', '241', 'NAGOSELO DO DOURO' ] ] },
        'Quinta da Corga' => {
          '5130' => [
            [ '282', '288', 'RIODADES' ] ] },
        'Varzeas de Treves' => {
          '5130' => [
            [ '520', '520', 'VRZEA DE TREVES' ] ] },
        'Quinta de Santa Barbara' => {
          '5130' => [
            [ '551', '557', 'VILAROUCO' ] ] },
        'Ervedosa do Douro' => {
          '5130' => [
            [ '061', '082', 'ERVEDOSA DO DOURO' ] ] },
        'Espinhosa do Douro' => {
          '5130' => [
            [ '203', '207', 'ESPINHOSA' ] ] },
        'Vale do Bispo' => {
          '5130' => [
            [ '201', '202', 'ERVEDOSA DO DOURO' ] ] },
        'Sarzedinho' => {
          '5130' => [
            [ '141', '141', 'ERVEDOSA DO DOURO' ] ] }
              },
      'Cinfes' => {
        'Cabo de Vila' => {
          '4690' => [
            [ '618', '684', 'SOUSELO' ] ] },
        'Aoreira' => {
          '4690' => [
            [ '867', '906', 'CINFES' ] ] },
        'Alcocreiros' => {
          '4690' => [
            [ '081', '090', 'ESPADANEDO CNF' ] ] },
        'Alto' => {
          '4690' => [
            [ '315', '332', 'NESPEREIRA CNF' ] ] },
        'Cruz' => {
          '4690' => [
            [ '343', '352', 'NESPEREIRA CNF' ] ] },
        'Vila Me' => {
          '4690' => [
            [ '400', '400', 'NESPEREIRA CNF' ] ] },
        'Nespereira' => {
          '4690' => [
            [ '363', '366', 'NESPEREIRA CNF' ] ] },
        'Almas' => {
          '4690' => [
            [ '261', '298', 'FORNELOS CNF' ] ] },
        'Adegas' => {
          '4690' => [
            [ '605', '616', 'SOUSELO' ] ] },
        'Granja' => {
          '4690' => [
            [ '355', '356', 'NESPEREIRA CNF' ] ] },
        'Cabeo da Fonte' => {
          '4690' => [
            [ '435', '435', 'RAMIRES' ] ] },
        'Rosrio' => {
          '4690' => [
            [ '158', '182', 'ESPADANEDO CNF' ] ] },
        'Costa' => {
          '4690' => [
            [ '340', '341', 'NESPEREIRA CNF' ] ] },
        'Aidos' => {
          '4690' => [
            [ '781', '866', 'TRAVANCA CNF' ] ] },
        'Boua Milhaa' => {
          '4690' => [
            [ '092', '117', 'ESPADANEDO CNF' ] ] },
        'Azibreiro' => {
          '4690' => [
            [ '209', '255', 'FERREIROS DE TENDAIS' ] ] },
        'Lageal' => {
          '4690' => [
            [ '358', '361', 'NESPEREIRA CNF' ] ] },
        'Paradela' => {
          '4690' => [
            [ '368', '371', 'NESPEREIRA CNF' ] ] },
        'Portelada' => {
          '4690' => [
            [ '382', '382', 'NESPEREIRA CNF' ] ] },
        'Vale' => {
          '4690' => [
            [ '010', '010', 'BUSTELO CNF' ] ] },
        'So Domingos' => {
          '4690' => [
            [ '436', '438', 'OLIVEIRA DO DOURO' ] ] },
        'Arrabalde de Bouas' => {
          '4690' => [
            [ '015', '052', 'CINFES' ] ] },
        'Cavada' => {
          '4690' => [
            [ '334', '335', 'NESPEREIRA CNF' ] ] },
        'Aguilho' => {
          '4690' => [
            [ '751', '767', 'TENDAIS' ] ] },
        'Achas' => {
          '4690' => [
            [ '691', '734', 'TAROUQUELA' ] ] },
        'Vila Pouca' => {
          '4690' => [
            [ '310', '310', 'MOIMENTA CNF' ] ] },
        'Algereu' => {
          '4690' => [
            [ '511', '600', 'SO CRISTOVO DE NOGUEIRA' ] ] },
        'Lavra' => {
          '4690' => [
            [ '119', '146', 'ESPADANEDO CNF' ] ] },
        'Pereira' => {
          '4690' => [
            [ '373', '379', 'NESPEREIRA CNF' ] ] },
        'Santa Marinha' => {
          '4690' => [
            [ '388', '391', 'NESPEREIRA CNF' ] ] },
        'Abeceira' => {
          '4690' => [
            [ '201', '207', 'FERREIROS DE TENDAIS' ] ] },
        'Valdoninhas' => {
          '4690' => [
            [ '396', '396', 'NESPEREIRA CNF' ] ] },
        'Alhes' => {
          '4690' => [
            [ '005', '005', 'ALHES' ] ] },
        'Fragas' => {
          '4690' => [
            [ '054', '070', 'CINFES' ] ] },
        'Adragoso de Baixo' => {
          '4690' => [
            [ '384', '387', 'OLIVEIRA DO DOURO' ] ] },
        'Santiago de Pies' => {
          '4690' => [
            [ '439', '500', 'SANTIAGO DE PIES' ] ] },
        'Ribeirinho' => {
          '4690' => [
            [ '300', '300', 'GRALHEIRA' ] ] },
        'Vales' => {
          '4690' => [
            [ '398', '398', 'NESPEREIRA CNF' ] ] },
        'Talhadouro' => {
          '4690' => [
            [ '393', '394', 'NESPEREIRA CNF' ] ] },
        'Quint' => {
          '4690' => [
            [ '148', '156', 'ESPADANEDO CNF' ] ] },
        'Casal Dosamparados' => {
          '4690' => [
            [ '401', '434', 'OLIVEIRA DO DOURO' ] ] }
            },
      'Castro Daire' => {
        'Mamouros' => {
          '3600' => [
            [ '394', '394', 'MAMOUROS' ] ] },
        'Lomba da Av' => {
          '3600' => [
            [ '348', '349', 'GAFANHO' ] ] },
        'Carvalhas' => {
          '3600' => [
            [ '471', '476', 'MONTEIRAS' ] ] },
        'So Joaninho' => {
          '3600' => [
            [ '651', '651', 'SO JOANINHO CDR' ] ] },
        'Chos de Paiva' => {
          '3600' => [
            [ '522', '530', 'PEPIM' ] ] },
        'Amial' => {
          '3600' => [
            [ '041', '056', 'CABRIL CDR' ] ] },
        'Cuj' => {
          '3600' => [
            [ '300', '300', 'CUJ' ] ] },
        'Casal' => {
          '3600' => [
            [ '344', '346', 'GAFANHO' ] ] },
        'Adenodeiro' => {
          '3600' => [
            [ '451', '463', 'MOLEDO CDR' ] ] },
        'Custilho' => {
          '3600' => [
            [ '270', '287', 'CASTRO DAIRE' ] ] },
        'Castro Daire' => {
          '3600' => [
            [ '069', '090', 'CASTRO DAIRE' ],
            [ '100', '103', 'CASTRO DAIRE' ],
            [ '121', '122', 'CASTRO DAIRE' ],
            [ '124', '124', 'CASTRO DAIRE' ],
            [ '126', '127', 'CASTRO DAIRE' ],
            [ '129', '129', 'CASTRO DAIRE' ],
            [ '132', '132', 'CASTRO DAIRE' ],
            [ '135', '140', 'CASTRO DAIRE' ],
            [ '144', '144', 'CASTRO DAIRE' ],
            [ '147', '147', 'CASTRO DAIRE' ],
            [ '152', '152', 'CASTRO DAIRE' ],
            [ '164', '164', 'CASTRO DAIRE' ],
            [ '167', '169', 'CASTRO DAIRE' ],
            [ '171', '173', 'CASTRO DAIRE' ],
            [ '175', '176', 'CASTRO DAIRE' ],
            [ '178', '178', 'CASTRO DAIRE' ],
            [ '180', '180', 'CASTRO DAIRE' ],
            [ '182', '182', 'CASTRO DAIRE' ],
            [ '186', '187', 'CASTRO DAIRE' ],
            [ '189', '189', 'CASTRO DAIRE' ],
            [ '191', '194', 'CASTRO DAIRE' ],
            [ '199', '199', 'CASTRO DAIRE' ],
            [ '202', '203', 'CASTRO DAIRE' ],
            [ '205', '205', 'CASTRO DAIRE' ],
            [ '209', '209', 'CASTRO DAIRE' ],
            [ '214', '215', 'CASTRO DAIRE' ],
            [ '220', '220', 'CASTRO DAIRE' ],
            [ '222', '222', 'CASTRO DAIRE' ],
            [ '224', '225', 'CASTRO DAIRE' ],
            [ '231', '232', 'CASTRO DAIRE' ],
            [ '234', '234', 'CASTRO DAIRE' ],
            [ '237', '237', 'CASTRO DAIRE' ],
            [ '239', '241', 'CASTRO DAIRE' ],
            [ '243', '243', 'CASTRO DAIRE' ],
            [ '246', '246', 'CASTRO DAIRE' ] ] },
        'Mides' => {
          '3600' => [
            [ '579', '579', 'RERIZ' ] ] },
        'Vale de Figueira' => {
          '3600' => [
            [ '289', '292', 'CASTRO DAIRE' ] ] },
        'Corgo de gua' => {
          '3600' => [
            [ '501', '509', 'PARADA DE ESTER' ] ] },
        'Arcas' => {
          '3600' => [
            [ '421', '435', 'MES' ] ] },
        'Reriz' => {
          '3600' => [
            [ '571', '577', 'RERIZ' ],
            [ '602', '605', 'RERIZ' ] ] },
        'Cetos' => {
          '3600' => [
            [ '551', '556', 'PINHEIRO CDR' ] ] },
        'Campo Benfeito' => {
          '3600' => [
            [ '371', '377', 'GOSENDE' ] ] },
        'Carvalhal' => {
          '3600' => [
            [ '391', '392', 'MAMOUROS' ] ] },
        'Cimo de Aldeia' => {
          '3600' => [
            [ '401', '405', 'MEZIO' ] ] },
        'Veado' => {
          '3600' => [
            [ '581', '600', 'RERIZ' ] ] },
        'Av' => {
          '3600' => [
            [ '342', '342', 'GAFANHO' ] ] },
        'Rebentina' => {
          '3600' => [
            [ '622', '625', 'RIBOLHOS' ] ] },
        'Braceiro' => {
          '3600' => [
            [ '321', '328', 'ESTER' ] ] },
        'Pvoa de Montemuro' => {
          '3600' => [
            [ '558', '563', 'PINHEIRO CDR' ] ] },
        'Moinho Velho' => {
          '3600' => [
            [ '396', '398', 'MAMOUROS' ] ] },
        'Pico' => {
          '3600' => [
            [ '540', '540', 'PICO' ] ] },
        'Raso' => {
          '3600' => [
            [ '351', '353', 'GAFANHO' ] ] },
        'Moura Morta' => {
          '3600' => [
            [ '480', '480', 'MOURA MORTA CDR' ] ] },
        'Alva' => {
          '3600' => [
            [ '021', '030', 'ALVA' ] ] },
        'Almofala' => {
          '3600' => [
            [ '011', '012', 'ALMOFALA CDR' ] ] },
        'Braos de C' => {
          '3600' => [
            [ '105', '106', 'CASTRO DAIRE' ] ] },
        'Carvalhosa' => {
          '3600' => [
            [ '311', '316', 'ERMIDA CDR' ] ] }
           },
        'Mangualde' => {
        'gua Levada' => {
          '3530' => [
            [ '060', '066', 'ESPINHO MGL' ] ],
          '3534' => [
            [ '008', '008', 'ESPINHO MGL' ] ] },
        'Fagilde' => {
          '3530' => [
            [ '070', '077', 'FORNOS DE MACEIRA DO' ] ] },
        'Bairro Pedra de Aires' => {
          '3530' => [
            [ '030', '037', 'CHS DE TAVARES' ] ] },
        'Casal' => {
          '3530' => [
            [ '370', '370', 'TRAVANCA DE TAVARES' ] ] },
        'Pvoa de Cerves' => {
          '3530' => [
            [ '320', '320', 'PVOA DE CERVES' ] ] },
        'Fojos' => {
          '3530' => [
            [ '110', '121', 'MANGUALDE' ] ] },
        'Bairro da Fontinha' => {
          '3530' => [
            [ '331', '336', 'QUINTELA DE AZURARA' ] ] },
        'Mesquitela' => {
          '3530' => [
            [ '301', '302', 'MESQUITELA MGL' ] ] },
        'Bairro Pedra D\'Aires' => {
          '3530' => [
            [ '380', '380', 'VRZEA DE TAVARES' ] ] },
        'Avinh' => {
          '3530' => [
            [ '360', '360', 'SO JOO DA FRESTA' ] ] },
        'Aldeia Nova' => {
          '3530' => [
            [ '340', '349', 'SANTIAGO DE CASSURRES' ] ] },
        'Abrunhosa a Velha' => {
          '3530' => [
            [ '010', '012', 'ABRUNHOSA-A-VELHA' ] ] },
        'Moimenta de Maceira Do' => {
          '3530' => [
            [ '310', '310', 'MOIMENTA DE MACEIRA DO' ] ],
          '3534' => [
            [ '009', '009', 'MOIMENTA DE MACEIRA DO' ] ] },
        'Mangualde' => {
          '3530' => [
            [ '092', '108', 'MANGUALDE' ],
            [ '123', '123', 'MANGUALDE' ],
            [ '125', '261', 'MANGUALDE' ],
            [ '271', '278', 'MANGUALDE' ],
            [ '281', '281', 'MANGUALDE' ],
            [ '283', '285', 'MANGUALDE' ] ],
          '3534' => [
            [ '001', '007', 'MANGUALDE' ] ] },
        'Cunha Alta' => {
          '3530' => [
            [ '040', '040', 'CUNHA ALTA' ] ] },
        'Freixiosa' => {
          '3530' => [
            [ '080', '080', 'FREIXIOSA' ] ] },
        'Lobelhe do Mato' => {
          '3530' => [
            [ '090', '090', 'LOBELHE DO MATO' ] ] },
        'Aldeia de Carvalho' => {
          '3530' => [
            [ '020', '027', 'ALCAFACHE' ] ] },
        'Abrunhosa do Mato' => {
          '3530' => [
            [ '050', '051', 'CUNHA BAIXA' ] ] }
              },
      'Santa Comba Do' => {
        'Barraca da Gestosa' => {
          '3440' => [
            [ '121', '138', 'COUTO DO MOSTEIRO' ] ] },
        'Casal Novo' => {
          '3440' => [
            [ '603', '613', 'VIMIEIRO SCD' ] ] },
        'Campolinho' => {
          '3440' => [
            [ '451', '471', 'SO JOO DE AREIAS' ] ] },
        'Casal Bom' => {
          '3440' => [
            [ '051', '057', 'SO JOANINHO SCD' ] ] },
        'Lameiras' => {
          '3440' => [
            [ '151', '167', 'PINHEIRO DE ZERE' ] ] },
        'So Joaninho' => {
          '3440' => [
            [ '061', '094', 'SO JOANINHO SCD' ],
            [ '096', '098', 'SO JOANINHO SCD' ] ] },
        'Treixedo' => {
          '3440' => [
            [ '510', '560', 'TREIXEDO' ] ] },
        'So Jorge' => {
          '3440' => [
            [ '110', '111', 'SO JOANINHO SCD' ] ] },
        'Anta' => {
          '3440' => [
            [ '601', '601', 'VIMIEIRO SCD' ] ] },
        'Bairro Atrs do Cemitrio' => {
          '3440' => [
            [ '301', '403', 'SANTA COMBA DO' ] ] },
        'Amainhos' => {
          '3440' => [
            [ '501', '503', 'TREIXEDO' ] ] },
        'Bairro da Catraia' => {
          '3440' => [
            [ '001', '019', 'VOA' ] ] },
        'Pinheiro de zere' => {
          '3440' => [
            [ '181', '226', 'PINHEIRO DE ZERE' ] ] },
        'Nagosela' => {
          '3440' => [
            [ '631', '672', 'NAGOSELA' ] ] }
              },
      'Armamar' => {
        'Alm a Vila' => {
          '5110' => [
            [ '251', '252', 'FONTELO' ],
            [ '401', '401', 'QUEIMADA' ] ] },
        'Viela do Nicho' => {
          '5110' => [
            [ '501', '501', 'QUEIMADELA AMM' ] ] },
        'Fontelo' => {
          '5110' => [
            [ '301', '330', 'FONTELO' ] ] },
        'Comenda' => {
          '5110' => [
            [ '271', '271', 'FONTELO' ] ] },
        'Bafancas' => {
          '5110' => [
            [ '641', '642', 'TES' ] ] },
        'Santo Antnio' => {
          '5110' => [
            [ '560', '561', 'SANTO ADRIO' ] ] },
        'Barandas' => {
          '5110' => [
            [ '521', '528', 'SANTIAGO AMM' ] ] },
        'Aricera' => {
          '5110' => [
            [ '051', '052', 'ARICERA' ] ] },
        'Bairro do Lagar' => {
          '5110' => [
            [ '502', '504', 'SANTA CRUZ AMM' ] ] },
        'Armamar' => {
          '5110' => [
            [ '121', '139', 'ARMAMAR' ] ] },
        'Bulhe' => {
          '5110' => [
            [ '671', '673', 'VILA SECA AMM' ] ] },
        'Queimadela' => {
          '5110' => [
            [ '452', '473', 'QUEIMADELA AMM' ] ] },
        'Cimbres' => {
          '5110' => [
            [ '167', '168', 'CIMBRES' ] ] },
        'Mata' => {
          '5110' => [
            [ '631', '632', 'SO ROMO AMM' ] ] },
        'Arvoredo' => {
          '5110' => [
            [ '611', '620', 'SO MARTINHO DAS CHS' ] ] },
        'Pai Calvo' => {
          '5110' => [
            [ '541', '549', 'SANTO ADRIO' ] ] },
        'Coura' => {
          '5110' => [
            [ '069', '069', 'COURA AMM' ] ] },
        'Almoinhos' => {
          '5110' => [
            [ '101', '101', 'ARMAMAR' ] ] },
        'Barroca' => {
          '5114' => [
            [ '001', '001', 'ARMAMAR' ] ],
          '5110' => [
            [ '009', '009', 'ARMAMAR' ],
            [ '651', '662', 'VACALAR' ] ] },
        'Aldeia de Baixo' => {
          '5110' => [
            [ '011', '024', 'ALDEIAS AMM' ] ] },
        'Bairro da Borralha' => {
          '5110' => [
            [ '151', '166', 'ARMAMAR' ] ] },
        'Balteiro' => {
          '5110' => [
            [ '261', '261', 'FONTELO' ] ] },
        'Amieira' => {
          '5110' => [
            [ '371', '377', 'GOUJOIM' ] ] },
        'Bairro das Pitinhas' => {
          '5110' => [
            [ '571', '593', 'SO COSMADO' ] ] },
        'Carvalho' => {
          '5110' => [
            [ '201', '216', 'FOLGOSA AMM' ] ] },
        'Queimada' => {
          '5110' => [
            [ '411', '438', 'QUEIMADA' ],
            [ '451', '451', 'QUEIMADA' ] ] },
        'Quinta da Rapada de Baixo' => {
          '5110' => [
            [ '351', '357', 'FONTELO' ] ] }
            },
      'Viseu' => {
        'Mundo' => {
          '3505' => [
            [ '558', '568', 'VISEU' ] ] },
        'Pereiras' => {
          '3515' => [
            [ '829', '829', 'BODIOSA' ] ] },
        'Cabril' => {
          '3505' => [
            [ '241', '255', 'POVOLIDE' ] ] },
        'Casal' => {
          '3515' => [
            [ '787', '826', 'RIBAFEITA' ] ] },
        'Bairro da Bela Vista' => {
          '3500' => [
            [ '531', '537', 'SILGUEIROS' ] ] },
        'Avelinha' => {
          '3505' => [
            [ '121', '232', 'CEPES VIS' ] ] },
        'Carvalhais' => {
          '3510' => [
            [ '601', '608', 'COUTO DE CIMA' ] ] },
        'Bertelhe' => {
          '3505' => [
            [ '554', '555', 'CEPES VIS' ] ] },
        'Viseu' => {
          '3505' => [
            [ '363', '553', 'VISEU' ] ],
          '3519' => [
            [ '001', '003', 'VISEU' ] ],
          '3514' => [
            [ '501', '513', 'VISEU' ],
            [ '518', '518', 'VISEU' ] ],
          '3504' => [
            [ '501', '507', 'VISEU' ],
            [ '509', '525', 'VISEU' ] ],
          '3510' => [
            [ '001', '001', 'VISEU' ],
            [ '003', '025', 'VISEU' ],
            [ '027', '028', 'VISEU' ],
            [ '030', '071', 'VISEU' ],
            [ '073', '073', 'VISEU' ],
            [ '075', '091', 'VISEU' ],
            [ '094', '100', 'VISEU' ],
            [ '102', '106', 'VISEU' ],
            [ '108', '113', 'VISEU' ],
            [ '115', '121', 'VISEU' ],
            [ '123', '123', 'VISEU' ],
            [ '147', '151', 'VISEU' ],
            [ '159', '160', 'VISEU' ],
            [ '163', '163', 'VISEU' ],
            [ '204', '204', 'VISEU' ],
            [ '212', '212', 'VISEU' ],
            [ '224', '224', 'VISEU' ],
            [ '238', '238', 'VISEU' ],
            [ '275', '275', 'VISEU' ],
            [ '292', '300', 'VISEU' ],
            [ '418', '430', 'VISEU' ],
            [ '543', '543', 'VISEU' ],
            [ '588', '588', 'VISEU' ],
            [ '590', '590', 'VISEU' ],
            [ '596', '596', 'VISEU' ],
            [ '666', '668', 'VISEU' ],
            [ '671', '671', 'VISEU' ],
            [ '673', '674', 'VISEU' ],
            [ '676', '681', 'VISEU' ],
            [ '683', '683', 'VISEU' ],
            [ '685', '685', 'VISEU' ],
            [ '687', '698', 'VISEU' ],
            [ '700', '708', 'VISEU' ],
            [ '718', '724', 'VISEU' ],
            [ '752', '753', 'VISEU' ],
            [ '756', '799', 'VISEU' ],
            [ '900', '900', 'VISEU' ],
            [ '913', '948', 'VISEU' ],
            [ '950', '950', 'VISEU' ] ],
          '3500' => [
            [ '001', '090', 'VISEU' ],
            [ '092', '122', 'VISEU' ],
            [ '124', '172', 'VISEU' ],
            [ '174', '223', 'VISEU' ],
            [ '225', '229', 'VISEU' ],
            [ '321', '321', 'VISEU' ],
            [ '390', '398', 'VISEU' ],
            [ '421', '422', 'VISEU' ],
            [ '425', '425', 'VISEU' ],
            [ '427', '427', 'VISEU' ],
            [ '441', '444', 'VISEU' ],
            [ '446', '446', 'VISEU' ],
            [ '601', '656', 'VISEU' ],
            [ '681', '733', 'VISEU' ],
            [ '794', '794', 'VISEU' ],
            [ '841', '856', 'VISEU' ] ],
          '3515' => [
            [ '096', '417', 'VISEU' ],
            [ '469', '469', 'VISEU' ],
            [ '827', '828', 'VISEU' ],
            [ '835', '845', 'VISEU' ] ] },
        'Brufe' => {
          '3505' => [
            [ '100', '105', 'BARREIROS VIS' ] ] },
        'Vilela' => {
          '3500' => [
            [ '900', '900', 'VISEU' ] ] },
        'Bigas' => {
          '3515' => [
            [ '772', '786', 'LORDOSA' ] ] },
        'Almargem' => {
          '3515' => [
            [ '694', '771', 'CALDE' ],
            [ '846', '846', 'CALDE' ] ] },
        'Alvelos' => {
          '3505' => [
            [ '106', '120', 'CAVERNES' ] ] },
        'Lustosa' => {
          '3515' => [
            [ '830', '834', 'RIBAFEITA' ] ] },
        'Bica' => {
          '3510' => [
            [ '641', '646', 'FARMINHO' ] ] },
        'Outeirinho' => {
          '3505' => [
            [ '353', '362', 'SO PEDRO DE FRANCE' ] ] },
        'Fonte Arcada' => {
          '3514' => [
            [ '517', '517', 'VIL DE SOUTO' ] ] },
        'Pvoa de Torredeita' => {
          '3514' => [
            [ '516', '516', 'TORREDEITA' ] ],
          '3510' => [
            [ '800', '800', 'TORREDEITA' ] ] },
        'Carcavelos' => {
          '3510' => [
            [ '891', '899', 'VIL DE SOUTO' ] ] },
        'Passos do Silgueiros' => {
          '3500' => [
            [ '541', '548', 'SILGUEIROS' ] ] },
        'Mosteiro' => {
          '3500' => [
            [ '539', '539', 'SILGUEIROS' ] ] },
        'Carqueijal' => {
          '3510' => [
            [ '811', '863', 'TORREDEITA' ] ] },
        'Azival' => {
          '3505' => [
            [ '557', '557', 'CEPES VIS' ] ] },
        'Lameirinhas da Rosa' => {
          '3505' => [
            [ '352', '352', 'VISEU' ] ] },
        'Bairro da Gndara' => {
          '3500' => [
            [ '881', '898', 'VISEU' ] ] },
        'Alto do Pinheiro' => {
          '3505' => [
            [ '256', '331', 'SANTOS EVOS' ] ] },
        'Quinta da Ramalhosa' => {
          '3505' => [
            [ '570', '577', 'VISEU' ] ] },
        'Lamaais' => {
          '3505' => [
            [ '332', '351', 'SO PEDRO DE FRANCE' ] ] },
        'Macieira' => {
          '3505' => [
            [ '233', '240', 'COTA' ] ] },
        'Fail' => {
          '3510' => [
            [ '621', '639', 'FAIL' ] ] },
        'Vila Ch do Monte' => {
          '3510' => [
            [ '865', '885', 'TORREDEITA' ] ] },
        'Travanca da Bodiosa' => {
          '3515' => [
            [ '497', '693', 'BODIOSA' ] ] },
        'Quints' => {
          '3505' => [
            [ '556', '556', 'COTA' ] ] },
        'Boa Aldeia' => {
          '3510' => [
            [ '291', '291', 'BOA ALDEIA' ] ] },
        'Canelas' => {
          '3510' => [
            [ '731', '736', 'SO CIPRIANO VIS' ] ] },
        'Couto de Baixo' => {
          '3510' => [
            [ '582', '587', 'COUTO DE BAIXO' ] ] }
          },
      'Tarouca' => {
        'Vrzea da Serra' => {
          '3610' => [
            [ '187', '208', 'VRZEA DA SERRA' ] ] },
        'Cortegada' => {
          '3610' => [
            [ '070', '074', 'SALZEDAS' ] ] },
        'Mondim Beira' => {
          '3610' => [
            [ '049', '056', 'MONDIM DA BEIRA' ] ] },
        'Boavista' => {
          '3610' => [
            [ '011', '016', 'DALVARES' ] ] },
        'Granja Nova' => {
          '3610' => [
            [ '042', '043', 'GRANJA NOVA' ] ] },
        'Ucanha' => {
          '3610' => [
            [ '175', '176', 'UCANHA' ] ] },
        'Vila Ch da Beira' => {
          '3610' => [
            [ '210', '210', 'VILA CH DA BEIRA' ] ] },
        'Teixelo' => {
          '3610' => [
            [ '165', '167', 'TAROUCA' ] ] },
        'Adro' => {
          '3610' => [
            [ '025', '041', 'GOUVIES' ] ] },
        'Couto' => {
          '3610' => [
            [ '080', '084', 'SO JOO DE TAROUCA' ] ] },
        'Tarouca' => {
          '3610' => [
            [ '001', '010', 'TAROUCA' ],
            [ '017', '024', 'TAROUCA' ],
            [ '094', '119', 'TAROUCA' ],
            [ '128', '148', 'TAROUCA' ] ] }
            },
      'Vouzela' => {
        'Moamedes' => {
          '3670' => [
            [ '193', '196', 'SO MIGUEL DO MATO VZL' ] ] },
        'Caveiros de Cima' => {
          '3670' => [
            [ '035', '035', 'CAMBRA' ] ] },
        'Areal' => {
          '3670' => [
            [ '089', '097', 'FATANOS' ] ] },
        'Queijo' => {
          '3670' => [
            [ '099', '104', 'FIGUEIREDO DAS DONAS' ] ] },
        'Mouta de Baixo' => {
          '3670' => [
            [ '010', '029', 'ALCOFRA' ] ] },
        'Ameixas' => {
          '3670' => [
            [ '150', '152', 'PAOS DE VILHARIGUES' ] ] },
        'Adsamo' => {
          '3670' => [
            [ '210', '225', 'VENTOSA VZL' ] ] },
        'Igreja' => {
          '3670' => [
            [ '046', '052', 'CAMBRA' ] ] },
        'Covas' => {
          '3670' => [
            [ '130', '133', 'FORNELO DO MONTE' ] ] },
        'Porto Salto' => {
          '3670' => [
            [ '269', '278', 'VOUZELA' ] ] },
        'Cambra' => {
          '3670' => [
            [ '037', '044', 'CAMBRA' ] ] },
        'Minas Bejanca' => {
          '3670' => [
            [ '192', '192', 'QUEIR' ] ] },
        'Fojo' => {
          '3670' => [
            [ '169', '177', 'QUEIR' ] ] },
        'Gandara' => {
          '3670' => [
            [ '228', '262', 'VOUZELA' ] ] },
        'Caria' => {
          '3670' => [
            [ '190', '191', 'SO MIGUEL DO MATO VZL' ] ] },
        'Carvalhal Vermilhas' => {
          '3670' => [
            [ '080', '081', 'CARVALHAL DE VERMILHAS' ] ] },
        'Adside' => {
          '3670' => [
            [ '053', '066', 'CAMPIA' ] ] },
        'Campia de Cima' => {
          '3670' => [
            [ '036', '036', 'CAMPIA' ] ] }
            },
      'So Pedro do Sul' => {
        'Pindelo dos Milagres' => {
          '3660' => [
            [ '161', '203', 'PINDELO DOS MILAGRES' ] ] },
        'Beirs' => {
          '3660' => [
            [ '601', '607', 'SERRAZES' ] ] },
        'Berbelo' => {
          '3660' => [
            [ '281', '286', 'SO CRISTOVO DE LAFES' ] ] },
        'Aguaneira' => {
          '3660' => [
            [ '241', '261', 'SANTA CRUZ DA TRAPA' ] ] },
        'Ancies' => {
          '3660' => [
            [ '681', '694', 'VRZEA SPS' ] ] },
        'Arada' => {
          '3660' => [
            [ '041', '044', 'CANDAL' ] ] },
        'Cigana' => {
          '3660' => [
            [ '111', '119', 'FIGUEIREDO DE ALVA' ] ] },
        'Alto Barro' => {
          '3660' => [
            [ '031', '035', 'BORDONHOS' ] ] },
        'Ameixiosa' => {
          '3660' => [
            [ '321', '331', 'SO MARTINHO DAS MOITAS' ] ] },
        'Boavista' => {
          '3660' => [
            [ '661', '674', 'VALADARES SPS' ] ] },
        'Costeiras' => {
          '3660' => [
            [ '011', '021', 'BAIES' ] ] },
        'Antas de Baixo' => {
          '3660' => [
            [ '131', '151', 'MANHOUCE' ] ] },
        'Aores' => {
          '3660' => [
            [ '611', '647', 'SUL' ] ] },
        'Abados' => {
          '3660' => [
            [ '051', '076', 'CARVALHAIS SPS' ] ] },
        'Arcozelo' => {
          '3660' => [
            [ '401', '527', 'SO PEDRO DO SUL' ] ] },
        'Amieiros' => {
          '3660' => [
            [ '221', '232', 'PINHO SPS' ] ] },
        'Borduzedo' => {
          '3660' => [
            [ '091', '097', 'COVAS DO RIO' ] ] },
        'gua Fria' => {
          '3660' => [
            [ '701', '727', 'VILA MAIOR SPS' ] ] },
        'Loureiro' => {
          '3660' => [
            [ '300', '312', 'SO FLIX' ] ] }
               },
      'Sto' => {
        'Afonsim' => {
          '3560' => [
            [ '081', '086', 'MIOMA' ] ] },
        'Monte de L' => {
          '3560' => [
            [ '187', '187', 'MIOMA' ] ] },
        'Abrunhosa' => {
          '3560' => [
            [ '120', '130', 'SO MIGUEL DE VILA BOA' ] ] },
        'guas Boas' => {
          '3560' => [
            [ '010', '010', 'GUAS BOAS' ] ] },
        'Vila Boinha' => {
          '3560' => [
            [ '185', '186', 'SO MIGUEL DE VILA BOA' ] ] },
        'Monte de C' => {
          '3560' => [
            [ '087', '087', 'STO' ] ] },
        'Vila Longa' => {
          '3560' => [
            [ '220', '220', 'VILA LONGA' ] ] },
        'Avelal' => {
          '3560' => [
            [ '020', '021', 'AVELAL' ] ] },
        'Decermilo' => {
          '3560' => [
            [ '030', '030', 'DECERMILO' ] ] },
        'Forles' => {
          '3560' => [
            [ '070', '070', 'FORLES' ] ] },
        'Casal Quinta Chedas' => {
          '3560' => [
            [ '210', '217', 'SILV DE CIMA' ] ] },
        'Carvalhal' => {
          '3560' => [
            [ '111', '118', 'ROMS' ] ] },
        'Foz' => {
          '3560' => [
            [ '188', '188', 'FERREIRA DE AVES' ] ] },
        'Sto' => {
          '3560' => [
            [ '189', '196', 'STO' ],
            [ '198', '209', 'STO' ],
            [ '218', '219', 'STO' ],
            [ '221', '223', 'STO' ],
            [ '226', '229', 'STO' ] ] },
        'Quinta das Marras' => {
          '3650' => [
            [ '801', '801', 'FERREIRA DE AVES' ] ] },
        'Vila Nova Aciprestes' => {
          '3560' => [
            [ '224', '224', 'SO MIGUEL DE VILA BOA' ] ] },
        'Rio de Moinhos' => {
          '3560' => [
            [ '090', '109', 'RIO DE MOINHOS SAT' ] ] },
        'Avelosa' => {
          '3560' => [
            [ '131', '184', 'STO' ] ] },
        'Aldeia Nova' => {
          '3560' => [
            [ '041', '066', 'FERREIRA DE AVES' ] ] },
        'Souralva' => {
          '3560' => [
            [ '088', '089', 'MIOMA' ] ] }
          },
      'Lamego' => {
        'Jogo da Bola' => {
          '5100' => [
            [ '771', '795', 'SANDE LMG' ] ] },
        'Alcouce' => {
          '5100' => [
            [ '871', '879', 'VRZEA DE ABRUNHAIS' ] ] },
        'Meijinhos' => {
          '5100' => [
            [ '630', '630', 'MEIJINHOS' ] ] },
        'Angorez' => {
          '5100' => [
            [ '751', '760', 'SAMODES' ] ] },
        'Arieiro' => {
          '5100' => [
            [ '541', '559', 'LALIM' ] ] },
        'Cairro' => {
          '5100' => [
            [ '530', '530', 'FIGUEIRA LMG' ] ] },
        'Bacelos' => {
          '5100' => [
            [ '711', '730', 'PENUDE' ] ] },
        'Codorneiro' => {
          '5100' => [
            [ '661', '693', 'PENAJIA' ] ] },
        'Boavista' => {
          '5100' => [
            [ '501', '516', 'FERREIROS DE AVES' ] ] },
        'Bigorne' => {
          '5100' => [
            [ '330', '330', 'BIGORNE' ] ] },
        'Barroncal' => {
          '5100' => [
            [ '481', '483', 'FERREIRIM LMG' ] ] },
        'Pretarouca' => {
          '5100' => [
            [ '740', '740', 'PRETAROUCA' ] ] },
        'Lamego' => {
          '5100' => [
            [ '001', '001', 'LAMEGO' ],
            [ '007', '009', 'LAMEGO' ],
            [ '076', '093', 'LAMEGO' ],
            [ '095', '109', 'LAMEGO' ],
            [ '112', '201', 'LAMEGO' ] ] },
        'Curral' => {
          '5100' => [
            [ '581', '588', 'LAZARIM' ] ] },
        'Adega do Cho' => {
          '5100' => [
            [ '381', '428', 'CAMBRES' ] ] },
        'Ms' => {
          '5100' => [
            [ '485', '490', 'FERREIRIM LMG' ] ] },
        'Melces' => {
          '5100' => [
            [ '640', '640', 'MELCES' ] ] },
        'Amoreiras' => {
          '5100' => [
            [ '811', '853', 'VALDIGEM' ] ] },
        'Bairral' => {
          '5100' => [
            [ '341', '365', 'BRITIANDE' ] ] },
        'Agra' => {
          '5100' => [
            [ '011', '074', 'LAMEGO' ] ] },
        'Adro' => {
          '5100' => [
            [ '301', '315', 'AVES' ] ] },
        'Parada do Bispo' => {
          '5100' => [
            [ '650', '650', 'PARADA DO BISPO' ] ] },
        'Arneirs' => {
          '5100' => [
            [ '891', '898', 'VILA NOVA SOUTO D\'EL-REI' ] ] },
        'Aboadela' => {
          '5100' => [
            [ '441', '467', 'CEPES LMG' ] ] },
        'Abrunheiro' => {
          '5100' => [
            [ '601', '611', 'MAGUEIJA' ] ] }
           },
      'Carregal do Sal' => {
        'Azenha' => {
          '3430' => [
            [ '275', '275', 'CARREGAL DO SAL' ] ] },
        'Parada' => {
          '3430' => [
            [ '721', '758', 'PARADA CRS' ] ] },
        'Pardieiros' => {
          '3430' => [
            [ '561', '561', 'BEIJS' ] ] },
        'Sobral de Papzios' => {
          '3430' => [
            [ '781', '781', 'SOBRAL CRS' ] ] },
        'Papzios' => {
          '3430' => [
            [ '701', '703', 'PAPZIOS' ] ] },
        'Alvarelhos' => {
          '3430' => [
            [ '271', '271', 'CARREGAL DO SAL' ] ] },
        'Casal da Torre' => {
          '3430' => [
            [ '201', '201', 'CARREGAL DO SAL' ] ] },
        'Travanca de So Tom' => {
          '3430' => [
            [ '420', '420', 'CARREGAL DO SAL' ] ] },
        'Pvoa das Forcadas' => {
          '3430' => [
            [ '770', '771', 'PARADA CRS' ] ] },
        'Beijs' => {
          '3430' => [
            [ '501', '536', 'BEIJS' ],
            [ '538', '540', 'BEIJS' ] ] },
        'Vila Me' => {
          '3430' => [
            [ '425', '425', 'CARREGAL DO SAL' ] ] },
        'Albergaria' => {
          '3430' => [
            [ '261', '261', 'CARREGAL DO SAL' ] ] },
        'Pvoa da Pegada' => {
          '3430' => [
            [ '565', '565', 'BEIJS' ] ] },
        'Laceiras' => {
          '3430' => [
            [ '690', '690', 'CABANAS DE VIRIATO' ] ] },
        'Oliveirinha' => {
          '3430' => [
            [ '391', '397', 'CARREGAL DO SAL' ],
            [ '399', '414', 'CARREGAL DO SAL' ] ] },
        'Cabanas de Viriato' => {
          '3430' => [
            [ '600', '621', 'CABANAS DE VIRIATO' ],
            [ '623', '627', 'CABANAS DE VIRIATO' ],
            [ '629', '649', 'CABANAS DE VIRIATO' ],
            [ '661', '661', 'CABANAS DE VIRIATO' ],
            [ '663', '668', 'CABANAS DE VIRIATO' ],
            [ '670', '671', 'CABANAS DE VIRIATO' ],
            [ '673', '674', 'CABANAS DE VIRIATO' ],
            [ '677', '679', 'CABANAS DE VIRIATO' ],
            [ '682', '685', 'CABANAS DE VIRIATO' ] ] },
        'Vila da Cal' => {
          '3430' => [
            [ '221', '221', 'CARREGAL DO SAL' ] ] },
        'Carregal do Sal' => {
          '3430' => [
            [ '001', '004', 'CARREGAL DO SAL' ],
            [ '006', '010', 'CARREGAL DO SAL' ],
            [ '012', '012', 'CARREGAL DO SAL' ],
            [ '015', '021', 'CARREGAL DO SAL' ],
            [ '024', '029', 'CARREGAL DO SAL' ],
            [ '031', '033', 'CARREGAL DO SAL' ],
            [ '035', '036', 'CARREGAL DO SAL' ],
            [ '038', '040', 'CARREGAL DO SAL' ],
            [ '043', '048', 'CARREGAL DO SAL' ],
            [ '050', '051', 'CARREGAL DO SAL' ],
            [ '054', '056', 'CARREGAL DO SAL' ],
            [ '058', '059', 'CARREGAL DO SAL' ],
            [ '064', '064', 'CARREGAL DO SAL' ],
            [ '067', '068', 'CARREGAL DO SAL' ],
            [ '071', '074', 'CARREGAL DO SAL' ],
            [ '077', '077', 'CARREGAL DO SAL' ],
            [ '079', '079', 'CARREGAL DO SAL' ],
            [ '083', '083', 'CARREGAL DO SAL' ],
            [ '085', '091', 'CARREGAL DO SAL' ],
            [ '094', '094', 'CARREGAL DO SAL' ],
            [ '096', '096', 'CARREGAL DO SAL' ],
            [ '120', '122', 'CARREGAL DO SAL' ],
            [ '125', '128', 'CARREGAL DO SAL' ],
            [ '131', '132', 'CARREGAL DO SAL' ],
            [ '134', '134', 'CARREGAL DO SAL' ],
            [ '136', '136', 'CARREGAL DO SAL' ],
            [ '139', '139', 'CARREGAL DO SAL' ],
            [ '141', '145', 'CARREGAL DO SAL' ],
            [ '147', '169', 'CARREGAL DO SAL' ] ] },
        'Oliveira do Conde' => {
          '3430' => [
            [ '341', '345', 'CARREGAL DO SAL' ],
            [ '348', '348', 'CARREGAL DO SAL' ],
            [ '350', '352', 'CARREGAL DO SAL' ],
            [ '354', '356', 'CARREGAL DO SAL' ],
            [ '358', '368', 'CARREGAL DO SAL' ],
            [ '370', '374', 'CARREGAL DO SAL' ],
            [ '376', '377', 'CARREGAL DO SAL' ] ] },
        'Fiais da Telha' => {
          '3430' => [
            [ '291', '322', 'CARREGAL DO SAL' ] ] },
        'Casal Mendo' => {
          '3430' => [
            [ '211', '211', 'CARREGAL DO SAL' ] ] },
        'Pvoa de Lisboa' => {
          '3430' => [
            [ '570', '570', 'BEIJS' ] ] }
              },
      'Moimenta da Beira' => {
        'Aldeia de Nacomba' => {
          '3620' => [
            [ '010', '010', 'ALDEIA DE NACOMBA' ] ] },
        'Castelo' => {
          '3620' => [
            [ '131', '156', 'CASTELO MBR' ] ] },
        'Peva' => {
          '3620' => [
            [ '441', '443', 'PEVA MBR' ] ] },
        'Arcas' => {
          '3620' => [
            [ '500', '504', 'SEVER MBR' ] ] },
        'Seges' => {
          '3620' => [
            [ '490', '490', 'SEGES' ] ] },
        'Carapito' => {
          '3620' => [
            [ '431', '432', 'PRA VELHA' ] ] },
        'Sarzedo' => {
          '3620' => [
            [ '480', '480', 'SARZEDO MBR' ] ] },
        'Nagosa' => {
          '3620' => [
            [ '400', '400', 'NAGOSA' ] ] },
        'Granja de Oleiros' => {
          '3620' => [
            [ '461', '464', 'RUA' ] ] },
        'Cabaos' => {
          '3620' => [
            [ '090', '090', 'CABAOS MBR' ] ] },
        'Ariz' => {
          '3620' => [
            [ '080', '080', 'ARIZ MBR' ] ] },
        'Moimenta da Beira' => {
          '3620' => [
            [ '300', '367', 'MOIMENTA DA BEIRA' ] ] },
        'Paradinha' => {
          '3620' => [
            [ '410', '410', 'PARADINHA' ] ] },
        'Pass' => {
          '3620' => [
            [ '421', '422', 'PASS' ] ] },
        'Beira Valente' => {
          '3620' => [
            [ '161', '203', 'LEOMIL MBR' ] ] },
        'Caria' => {
          '3620' => [
            [ '100', '106', 'CARIA MBR' ] ] },
        'Mileu' => {
          '3620' => [
            [ '098', '098', 'CARIA MBR' ] ] },
        'Baldos' => {
          '3620' => [
            [ '085', '085', 'BALDOS' ] ] },
        'Barragem do Vilar' => {
          '3620' => [
            [ '520', '521', 'VILAR MBR' ] ] },
        'Arcozelo da Torre' => {
          '3620' => [
            [ '071', '073', 'ARCOZELOS' ] ] },
        'Alvite' => {
          '3620' => [
            [ '021', '053', 'ALVITE MBR' ] ] }
                },
      'Penedono' => {
        'Pvoa de Penela' => {
          '3630' => [
            [ '350', '375', 'PVOA DE PENELA' ] ] },
        'Minas do Ouro' => {
          '3630' => [
            [ '111', '115', 'GRANJA PND' ] ] },
        'Antas' => {
          '3630' => [
            [ '011', '028', 'ANTAS PND' ] ] },
        'Carvalhal' => {
          '3630' => [
            [ '130', '132', 'OUROZINHO' ] ] },
        'Ourozinho' => {
          '3630' => [
            [ '135', '145', 'OUROZINHO' ] ] },
        'Penela da Beira' => {
          '3630' => [
            [ '260', '313', 'PENELA DA BEIRA' ] ] },
        'Souto' => {
          '3630' => [
            [ '400', '415', 'SOUTO PND' ] ] },
        'Penedono' => {
          '3630' => [
            [ '225', '225', 'PENEDONO' ],
            [ '227', '257', 'PENEDONO' ] ] },
        'Castaino' => {
          '3630' => [
            [ '070', '089', 'CASTAINO' ] ] },
        'Beselga' => {
          '3630' => [
            [ '040', '059', 'BESELGA PND' ] ] },
        'Ponte da Veiga' => {
          '3630' => [
            [ '342', '342', 'PVOA DE PENELA' ] ] },
        'Ferronha' => {
          '3630' => [
            [ '205', '215', 'PENEDONO' ] ] },
        'Granja' => {
          '3630' => [
            [ '091', '102', 'GRANJA PND' ] ] },
        'Arcas' => {
          '3630' => [
            [ '390', '390', 'SOUTO PND' ] ] },
        'Bebeses' => {
          '3630' => [
            [ '340', '340', 'PVOA DE PENELA' ] ] },
        'A-do-Bispo' => {
          '3630' => [
            [ '200', '202', 'PENEDONO' ] ] }
             },
      'Resende' => {
        'Alegrete' => {
          '4660' => [
            [ '091', '095', 'FREIGIL' ] ] },
        'Feiro' => {
          '4660' => [
            [ '070', '070', 'FEIRO' ] ] },
        'Fazames' => {
          '4660' => [
            [ '141', '147', 'PAUS' ] ] },
        'Arco' => {
          '4660' => [
            [ '201', '234', 'RESENDE' ] ] },
        'Panchorra' => {
          '4660' => [
            [ '130', '130', 'PANCHORRA' ] ] },
        'Barroncas' => {
          '4660' => [
            [ '051', '061', 'CRQUERE' ] ] },
        'Brejo' => {
          '4660' => [
            [ '301', '309', 'SO CIPRIANO RSD' ] ] },
        'Aldeia Cima' => {
          '4660' => [
            [ '401', '404', 'SO ROMO DE AREGOS' ] ] },
        'Bernardo' => {
          '4660' => [
            [ '031', '038', 'BARR RSD' ] ] },
        'Felgueiras' => {
          '4660' => [
            [ '080', '083', 'FELGUEIRAS RSD' ] ] },
        'Entre Vinhas' => {
          '4660' => [
            [ '101', '104', 'MIOMES' ] ] },
        'Granja Ovados' => {
          '4660' => [
            [ '121', '126', 'OVADAS' ] ] },
        'Barraco' => {
          '4660' => [
            [ '361', '386', 'SO MARTINHO DE MOUROS' ] ] },
        'Bairro' => {
          '4660' => [
            [ '331', '344', 'SO JOO DE FONTOURA' ] ] },
        'Adega' => {
          '4660' => [
            [ '011', '013', 'ANREADE' ] ] },
        'Palma' => {
          '4660' => [
            [ '016', '020', 'ANREADE' ] ] }
            },
      'Vila Nova de Paiva' => {
        'Borralhais' => {
          '3650' => [
            [ '101', '108', 'VILA COVA  COELHEIRA VNP' ] ] },
        'Vila Nova de Paiva' => {
          '3650' => [
            [ '194', '215', 'VILA NOVA DE PAIVA' ],
            [ '217', '226', 'VILA NOVA DE PAIVA' ] ] },
        'Vila Cova  Coelheira' => {
          '3650' => [
            [ '120', '120', 'VILA COVA  COELHEIRA VNP' ] ] },
        'Frguas' => {
          '3650' => [
            [ '020', '020', 'FRGUAS VNP' ] ] },
        'Alhais' => {
          '3650' => [
            [ '010', '011', 'ALHAIS' ] ] },
        'Escaleira' => {
          '3650' => [
            [ '031', '033', 'PENDILHE' ] ] },
        'Adomingueiros' => {
          '3650' => [
            [ '071', '082', 'TOURO' ] ] },
        'Calvrio' => {
          '3650' => [
            [ '041', '051', 'QUEIRIGA' ] ] },
        'Separadora' => {
          '3650' => [
            [ '055', '055', 'QUEIRIGA' ] ] }
                 },
      'Tondela' => {
        'Pvoa de Sardinha' => {
          '3460' => [
            [ '065', '070', 'TONDELA' ] ] },
        'Devaguinha' => {
          '3460' => [
            [ '431', '433', 'SABUGOSA' ] ] },
        'Adia' => {
          '3460' => [
            [ '321', '330', 'TONDELA' ] ] },
        'Cadrao' => {
          '3475' => [
            [ '030', '043', 'CARAMULO' ] ] },
        'Cunha de Tonda' => {
          '3460' => [
            [ '649', '650', 'TONDELA' ] ] },
        'Castelo' => {
          '3475' => [
            [ '070', '074', 'SO JOO DO MONTE' ] ] },
        'Casal' => {
          '3465' => [
            [ '121', '134', 'CASTELES TND' ] ] },
        'Arrifana' => {
          '3465' => [
            [ '021', '022', 'CAMPO DE BESTEIROS' ] ] },
        'Barr' => {
          '3465' => [
            [ '151', '157', 'SANTIAGO DE BESTEIROS' ] ] },
        'Ferreirs do Do' => {
          '3460' => [
            [ '101', '101', 'FERREIRS DO DO' ] ] },
        'Quinta da Andavia' => {
          '3460' => [
            [ '512', '623', 'TONDELA' ] ] },
        'Alenteira' => {
          '3460' => [
            [ '051', '063', 'TONDELA' ] ] },
        'Caparrosa' => {
          '3465' => [
            [ '101', '105', 'CAPARROSA' ] ] },
        'Bairro da Lage da Cruz' => {
          '3460' => [
            [ '371', '375', 'PARADA DE GONTA' ] ] },
        'Quinta das Calvas' => {
          '3460' => [
            [ '357', '359', 'TONDELA' ] ] },
        'Carregueiro' => {
          '3465' => [
            [ '190', '190', 'VILAR DE BESTEIROS' ] ] },
        'Bairro Cornlio' => {
          '3460' => [
            [ '501', '510', 'TONDELA' ] ] },
        'Gandara' => {
          '3460' => [
            [ '704', '712', 'TONDELA' ] ] },
        'Corujeiro' => {
          '3460' => [
            [ '151', '161', 'LAJEOSA TND' ] ] },
        'Parada de Gonta' => {
          '3460' => [
            [ '377', '414', 'PARADA DE GONTA' ] ] },
        'Frguas' => {
          '3460' => [
            [ '301', '304', 'MOSTEIRO DE FRGUAS' ] ] },
        'Campo de Besteiros' => {
          '3465' => [
            [ '051', '060', 'CAMPO DE BESTEIROS' ] ] },
        'Pego' => {
          '3460' => [
            [ '332', '337', 'TONDELA' ] ] },
        'Cimalha' => {
          '3460' => [
            [ '701', '702', 'TONDELA' ] ] },
        'So Miguel do Outeiro' => {
          '3460' => [
            [ '451', '452', 'SO MIGUEL DO OUTEIRO' ] ] },
        'Casal de Tonda' => {
          '3460' => [
            [ '471', '480', 'TONDELA' ] ] },
        'Bairro da Ponte' => {
          '3460' => [
            [ '351', '355', 'TONDELA' ] ] },
        'Arnosa' => {
          '3465' => [
            [ '011', '020', 'BARREIRO DE BESTEIROS' ] ] },
        'Malhapo de Cima' => {
          '3475' => [
            [ '060', '060', 'MOSTEIRINHO' ] ] },
        'Pousadas' => {
          '3465' => [
            [ '195', '195', 'TOURIGO' ] ] },
        'Carvalhal da Mulher' => {
          '3465' => [
            [ '180', '180', 'SILVARES TND' ] ] },
        'Fial' => {
          '3460' => [
            [ '454', '456', 'SO MIGUEL DO OUTEIRO' ] ] },
        'Molelos' => {
          '3460' => [
            [ '009', '024', 'TONDELA' ],
            [ '200', '268', 'TONDELA' ] ] },
        'Toural' => {
          '3460' => [
            [ '660', '660', 'TONDELA' ] ] }
            },
      'Tabuao' => {
        'Gulhufeira' => {
          '5120' => [
            [ '440', '448', 'TVORA' ] ] },
        'Arcos' => {
          '5120' => [
            [ '021', '048', 'ARCOS TBC' ] ] },
        'Granja do Tedo' => {
          '5120' => [
            [ '161', '183', 'GRANJA DO TEDO' ] ] },
        'Chaves' => {
          '5120' => [
            [ '101', '126', 'CHAVES' ] ] },
        'Aleluias' => {
          '5120' => [
            [ '061', '083', 'BARCOS' ] ] },
        'Balsa do Douro' => {
          '5120' => [
            [ '141', '146', 'DESEJOSA' ] ] },
        'Vale de Figueira' => {
          '5120' => [
            [ '461', '479', 'VALE DE FIGUEIRA TBC' ] ] },
        'Valena do Douro' => {
          '5120' => [
            [ '500', '515', 'VALENA DO DOURO' ] ] },
        'Quinta da Corte' => {
          '5120' => [
            [ '491', '496', 'VALENA DO DOURO' ] ] },
        'Ratinho' => {
          '5120' => [
            [ '371', '373', 'TABUAO' ] ] },
        'Casal do Monte' => {
          '5120' => [
            [ '201', '204', 'GRANJINHA' ] ] },
        'Aldeia' => {
          '5120' => [
            [ '331', '360', 'SENDIM TBC' ] ] },
        'Adorigo' => {
          '5120' => [
            [ '011', '014', 'ADORIGO' ] ] },
        'Santa Leocdia' => {
          '5120' => [
            [ '301', '315', 'SANTA LEOCDIA TBC' ] ] },
        'Pereiro' => {
          '5120' => [
            [ '262', '262', 'PEREIRO TBC' ] ] },
        'Paradela' => {
          '5120' => [
            [ '261', '261', 'PARADELA TBC' ] ] },
        'Longa' => {
          '5120' => [
            [ '221', '256', 'LONGA' ] ] },
        'Tabuao' => {
          '5120' => [
            [ '380', '426', 'TABUAO' ] ] },
        'Barreiro' => {
          '5120' => [
            [ '281', '284', 'PINHEIROS TBC' ] ] }
            },
      'Oliveira de Frades' => {
        'Antelas' => {
          '3680' => [
            [ '171', '182', 'PINHEIRO OFR' ] ] },
        'Areal' => {
          '3475' => [
            [ '010', '010', 'ARCA OFR' ] ] },
        'Casal Sejes' => {
          '3680' => [
            [ '300', '305', 'SEJES' ] ] },
        'Bandonagens' => {
          '3680' => [
            [ '280', '288', 'SO VICENTE DE LAFES' ] ] },
        'Bispeira' => {
          '3680' => [
            [ '260', '264', 'SO JOO DA SERRA' ] ] },
        'Cabeo Souto' => {
          '3680' => [
            [ '320', '324', 'SOUTO DE LAFES' ] ] },
        'Feira' => {
          '3680' => [
            [ '190', '195', 'REIGOSO OFR' ] ] },
        'Aidos' => {
          '3680' => [
            [ '200', '245', 'RIBEIRADIO' ] ] },
        'Travasss' => {
          '3680' => [
            [ '170', '170', 'OLIVEIRA DE FRADES' ] ] },
        'Arcozelo das Maias' => {
          '3680' => [
            [ '011', '026', 'ARCOZELO DAS MAIAS' ] ] },
        'Monte Teso' => {
          '3475' => [
            [ '020', '020', 'VARZIELAS' ] ] },
        'Benfeitas' => {
          '3680' => [
            [ '040', '046', 'DESTRIZ' ] ] },
        'Oliveira de Frades' => {
          '3680' => [
            [ '074', '128', 'OLIVEIRA DE FRADES' ],
            [ '132', '145', 'OLIVEIRA DE FRADES' ] ] }
                 }
         },
    'Faro' => {
      'Albufeira' => {
        'Albufeira' => {
          '8201' => [
            [ '851', '851', 'ALBUFEIRA' ],
            [ '861', '863', 'ALBUFEIRA' ],
            [ '870', '871', 'ALBUFEIRA' ],
            [ '877', '877', 'ALBUFEIRA' ] ],
          '8200' => [
            [ '001', '005', 'ALBUFEIRA' ],
            [ '008', '040', 'ALBUFEIRA' ],
            [ '042', '050', 'ALBUFEIRA' ],
            [ '053', '055', 'ALBUFEIRA' ],
            [ '057', '082', 'ALBUFEIRA' ],
            [ '084', '099', 'ALBUFEIRA' ],
            [ '101', '220', 'ALBUFEIRA' ],
            [ '223', '228', 'ALBUFEIRA' ],
            [ '230', '231', 'ALBUFEIRA' ],
            [ '250', '252', 'ALBUFEIRA' ],
            [ '254', '254', 'ALBUFEIRA' ],
            [ '260', '299', 'ALBUFEIRA' ],
            [ '311', '319', 'ALBUFEIRA' ],
            [ '321', '321', 'ALBUFEIRA' ],
            [ '325', '329', 'ALBUFEIRA' ],
            [ '340', '349', 'ALBUFEIRA' ],
            [ '351', '351', 'ALBUFEIRA' ],
            [ '353', '357', 'ALBUFEIRA' ],
            [ '359', '360', 'ALBUFEIRA' ],
            [ '362', '362', 'ALBUFEIRA' ],
            [ '370', '375', 'ALBUFEIRA' ],
            [ '377', '377', 'ALBUFEIRA' ],
            [ '379', '381', 'ALBUFEIRA' ],
            [ '384', '387', 'ALBUFEIRA' ],
            [ '389', '397', 'ALBUFEIRA' ],
            [ '511', '511', 'ALBUFEIRA' ],
            [ '591', '592', 'ALBUFEIRA' ],
            [ '594', '594', 'ALBUFEIRA' ],
            [ '596', '596', 'ALBUFEIRA' ],
            [ '600', '600', 'ALBUFEIRA' ],
            [ '602', '602', 'ALBUFEIRA' ],
            [ '604', '604', 'ALBUFEIRA' ],
            [ '609', '609', 'ALBUFEIRA' ],
            [ '612', '613', 'ALBUFEIRA' ],
            [ '617', '617', 'ALBUFEIRA' ],
            [ '619', '619', 'ALBUFEIRA' ],
            [ '621', '621', 'ALBUFEIRA' ],
            [ '625', '625', 'ALBUFEIRA' ],
            [ '629', '629', 'ALBUFEIRA' ],
            [ '631', '633', 'ALBUFEIRA' ],
            [ '635', '637', 'ALBUFEIRA' ],
            [ '639', '665', 'ALBUFEIRA' ] ] },
        'Matos de Baixo' => {
          '8200' => [
            [ '486', '489', 'PADERNE ABF' ] ] },
        'Alcaria' => {
          '8200' => [
            [ '450', '450', 'PADERNE ABF' ] ] },
        'P da Cruz' => {
          '8200' => [
            [ '497', '499', 'PADERNE ABF' ] ] },
        'Ribeira de Alte' => {
          '8200' => [
            [ '501', '503', 'PADERNE ABF' ] ] },
        'Vale Verde' => {
          '8201' => [
            [ '856', '856', 'GUIA ABF' ] ] },
        'Paderne' => {
          '8201' => [
            [ '855', '855', 'PADERNE ABF' ] ],
          '8200' => [
            [ '507', '510', 'PADERNE ABF' ],
            [ '512', '514', 'PADERNE ABF' ],
            [ '601', '601', 'PADERNE ABF' ],
            [ '622', '622', 'PADERNE ABF' ],
            [ '624', '624', 'PADERNE ABF' ],
            [ '628', '628', 'PADERNE ABF' ],
            [ '630', '630', 'PADERNE ABF' ],
            [ '638', '638', 'PADERNE ABF' ] ] },
        'Vale de Pegas' => {
          '8200' => [
            [ '505', '505', 'PADERNE ABF' ] ] },
        'Tavagueira' => {
          '8201' => [
            [ '875', '876', 'GUIA ABF' ],
            [ '878', '878', 'GUIA ABF' ] ] },
        'Alfarrobeiras' => {
          '8200' => [
            [ '551', '569', 'FERREIRAS' ] ] },
        'Monte dos Elois' => {
          '8200' => [
            [ '492', '495', 'PADERNE ABF' ] ] },
        'Cerro do Roque' => {
          '8200' => [
            [ '470', '474', 'PADERNE ABF' ] ] },
        'Quinta dos lamos' => {
          '8200' => [
            [ '614', '616', 'GUIA ABF' ] ] },
        'Alpouvar' => {
          '8201' => [
            [ '865', '865', 'FERREIRAS' ],
            [ '868', '868', 'FERREIRAS' ] ] },
        'Fonte' => {
          '8200' => [
            [ '415', '449', 'GUIA ABF' ],
            [ '476', '479', 'PADERNE ABF' ] ] },
        'Ladeira' => {
          '8200' => [
            [ '481', '484', 'PADERNE ABF' ] ] },
        'Almeijoafras' => {
          '8200' => [
            [ '452', '468', 'PADERNE ABF' ] ] },
        'Cortelhas' => {
          '8201' => [
            [ '864', '864', 'GUIA ABF' ] ] },
        'lamos' => {
          '8200' => [
            [ '410', '413', 'GUIA ABF' ] ] },
        'Guia' => {
          '8200' => [
            [ '603', '603', 'GUIA ABF' ],
            [ '605', '608', 'GUIA ABF' ],
            [ '610', '610', 'GUIA ABF' ],
            [ '620', '620', 'GUIA ABF' ] ] }
             },
      'Portimo' => {
        'Pedregais' => {
          '8500' => [
            [ '070', '086', 'ALVOR' ] ] },
        'Portimo' => {
          '8500' => [
            [ '069', '069', 'PORTIMO' ],
            [ '300', '300', 'PORTIMO' ],
            [ '304', '304', 'PORTIMO' ],
            [ '307', '311', 'PORTIMO' ],
            [ '313', '314', 'PORTIMO' ],
            [ '316', '316', 'PORTIMO' ],
            [ '318', '319', 'PORTIMO' ],
            [ '321', '329', 'PORTIMO' ],
            [ '331', '334', 'PORTIMO' ],
            [ '336', '370', 'PORTIMO' ],
            [ '373', '414', 'PORTIMO' ],
            [ '416', '663', 'PORTIMO' ],
            [ '665', '668', 'PORTIMO' ],
            [ '670', '676', 'PORTIMO' ],
            [ '678', '810', 'PORTIMO' ],
            [ '812', '824', 'PORTIMO' ],
            [ '826', '827', 'PORTIMO' ],
            [ '831', '838', 'PORTIMO' ],
            [ '840', '844', 'PORTIMO' ] ],
          '8501' => [
            [ '851', '859', 'PORTIMO' ] ] },
        'Alagoas' => {
          '8500' => [
            [ '001', '032', 'ALVOR' ] ] },
        'Figueira' => {
          '8500' => [
            [ '130', '132', 'MEXILHOEIRA GRANDE' ] ] },
        'Sesmarias de Baixo' => {
          '8500' => [
            [ '828', '829', 'ALVOR' ] ] },
        'Facho' => {
          '8500' => [
            [ '055', '059', 'ALVOR' ] ] },
        'Alcalar' => {
          '8500' => [
            [ '120', '128', 'MEXILHOEIRA GRANDE' ] ] },
        'Amoreira' => {
          '8500' => [
            [ '045', '053', 'ALVOR' ] ] },
        'Moinho da Rocha' => {
          '8500' => [
            [ '140', '164', 'MEXILHOEIRA GRANDE' ] ] }
            },
      'Silves' => {
        'Estevais' => {
          '8365' => [
            [ '300', '307', 'ALCANTARILHA' ] ] },
        'Aldeia Ruiva' => {
          '8375' => [
            [ '009', '071', 'S BARTOLOMEU DE MESSINES' ] ] },
        'Azilheira' => {
          '8375' => [
            [ '200', '229', 'SO MARCOS DA SERRA' ] ] },
        'Taliscas' => {
          '8375' => [
            [ '075', '083', 'S BARTOLOMEU DE MESSINES' ] ] },
        'So Marcos da Serra' => {
          '8375' => [
            [ '250', '275', 'SO MARCOS DA SERRA' ] ] },
        'Calios' => {
          '8365' => [
            [ '181', '182', 'ARMAO DE PRA' ] ] },
        'Silves' => {
          '8300' => [
            [ '100', '123', 'SILVES' ],
            [ '126', '191', 'SILVES' ] ] },
        'Aldeia de Tunes' => {
          '8365' => [
            [ '051', '052', 'ALGOZ' ] ] },
        'Torre' => {
          '8365' => [
            [ '184', '184', 'ARMAO DE PRA' ] ] },
        'Tunes' => {
          '8365' => [
            [ '235', '236', 'TUNES' ] ] },
        'Algoz' => {
          '8365' => [
            [ '055', '074', 'ALGOZ' ] ] },
        'Armao de Pra' => {
          '8365' => [
            [ '101', '140', 'ARMAO DE PRA' ],
            [ '142', '149', 'ARMAO DE PRA' ] ] },
        'Amendoais' => {
          '8365' => [
            [ '231', '233', 'TUNES' ] ] },
        'Areias de Pra' => {
          '8365' => [
            [ '201', '208', 'PRA' ] ] },
        'Bais' => {
          '8365' => [
            [ '080', '092', 'ALGOZ' ] ] },
        'Almarjo' => {
          '8300' => [
            [ '010', '013', 'SILVES' ] ] },
        'So Bartolomeu de Messines' => {
          '8375' => [
            [ '100', '157', 'S BARTOLOMEU DE MESSINES' ] ] },
        'Alcantarilha' => {
          '8365' => [
            [ '010', '039', 'ALCANTARILHA' ],
            [ '049', '049', 'ALCANTARILHA' ],
            [ '183', '183', 'ALCANTARILHA' ] ] },
        'Bairro Vermelho' => {
          '8300' => [
            [ '015', '055', 'SILVES' ] ] }
          },
      'Aljezur' => {
        'Saltador' => {
          '8670' => [
            [ '445', '446', 'ROGIL' ] ] },
        'Bordeira' => {
          '8670' => [
            [ '220', '220', 'BORDEIRA' ] ] },
        'Cabeos de Bordeira' => {
          '8670' => [
            [ '225', '225', 'BORDEIRA' ] ] },
        'Igreja Nova' => {
          '8670' => [
            [ '130', '130', 'ALJEZUR' ] ] },
        'Maria Vinagre' => {
          '8670' => [
            [ '430', '430', 'ROGIL' ] ] },
        'Azia' => {
          '8670' => [
            [ '411', '420', 'ROGIL' ] ] },
        'Rogil' => {
          '8670' => [
            [ '440', '440', 'ROGIL' ] ] },
        'Aljezur' => {
          '8670' => [
            [ '001', '008', 'ALJEZUR' ],
            [ '010', '015', 'ALJEZUR' ],
            [ '017', '026', 'ALJEZUR' ],
            [ '051', '062', 'ALJEZUR' ],
            [ '064', '069', 'ALJEZUR' ],
            [ '071', '092', 'ALJEZUR' ] ] },
        'Praia de Odeceixe' => {
          '8670' => [
            [ '325', '325', 'ODECEIXE' ] ] },
        'Escampadinho' => {
          '8670' => [
            [ '311', '312', 'ODECEIXE' ] ] },
        'Monte Novo' => {
          '8670' => [
            [ '236', '238', 'BORDEIRA' ] ] },
        'Arrifana' => {
          '8670' => [
            [ '111', '122', 'ALJEZUR' ] ] },
        'Alfambras' => {
          '8670' => [
            [ '211', '211', 'BORDEIRA' ] ] },
        'Monho da Lgua' => {
          '8670' => [
            [ '136', '160', 'ALJEZUR' ] ] },
        'Malhadais' => {
          '8670' => [
            [ '320', '320', 'ODECEIXE' ] ] },
        'Carrapateira' => {
          '8670' => [
            [ '230', '230', 'BORDEIRA' ] ] }
           },
      'Lagos' => {
        'Odixere' => {
          '8600' => [
            [ '250', '263', 'ODIXERE' ] ] },
        'Alagoas' => {
          '8600' => [
            [ '101', '188', 'LUZ LGS' ] ] },
        'Lagos' => {
          '8600' => [
            [ '281', '284', 'LAGOS' ],
            [ '297', '336', 'LAGOS' ],
            [ '338', '376', 'LAGOS' ],
            [ '500', '599', 'LAGOS' ],
            [ '601', '776', 'LAGOS' ],
            [ '780', '780', 'LAGOS' ] ],
          '8601' => [
            [ '851', '852', 'LAGOS' ] ] },
        'Aguilhadas' => {
          '8600' => [
            [ '011', '028', 'BARO DE SO JOO' ] ] },
        'Vale de Coitos' => {
          '8600' => [
            [ '001', '001', 'ODIXERE' ] ] },
        'Azoia' => {
          '8600' => [
            [ '068', '094', 'BENSAFRIM' ] ] },
        'Quinta das Palmeiras' => {
          '8600' => [
            [ '200', '211', 'LUZ LGS' ] ] },
        'Alfarrobeira' => {
          '8600' => [
            [ '230', '237', 'ODIXERE' ] ] }
         },
      'Castro Marim' => {
        'Odeleite' => {
          '8950' => [
            [ '351', '371', 'ODELEITE' ] ] },
        'Corte Gago' => {
          '8950' => [
            [ '081', '088', 'AZINHAL CTM' ] ] },
        'Alcaria' => {
          '8950' => [
            [ '311', '334', 'ODELEITE' ] ] },
        'Sapal do Cho' => {
          '8950' => [
            [ '275', '281', 'CASTRO MARIM' ] ] },
        'So Bartolomeu do Sul' => {
          '8950' => [
            [ '270', '270', 'CASTRO MARIM' ] ] },
        'Montes Castelhanos' => {
          '8950' => [
            [ '251', '258', 'CASTRO MARIM' ] ] },
        'Encalhe' => {
          '8950' => [
            [ '181', '192', 'CASTRO MARIM' ] ] },
        'Monte do Barranco' => {
          '8950' => [
            [ '430', '442', 'ALTURA' ] ] },
        'Azinhal' => {
          '8950' => [
            [ '021', '056', 'AZINHAL CTM' ] ] },
        'Monte Francisco' => {
          '8950' => [
            [ '201', '230', 'CASTRO MARIM' ] ] },
        'Azeda' => {
          '8950' => [
            [ '101', '107', 'CASTRO MARIM' ] ] },
        'Castro Marim' => {
          '8950' => [
            [ '121', '150', 'CASTRO MARIM' ] ] },
        'Horta Nova' => {
          '8950' => [
            [ '011', '013', 'AZINHAL CTM' ] ] },
        'Alagoa' => {
          '8950' => [
            [ '411', '428', 'ALTURA' ] ] },
        'Portela Alta de Baixo' => {
          '8950' => [
            [ '391', '395', 'ODELEITE' ] ] }
          },
      'Loul' => {
        'Freixo Verde' => {
          '8100' => [
            [ '360', '367', 'BENAFIM' ] ] },
        'Fonte de Arez' => {
          '8100' => [
            [ '359', '359', 'ALTE' ] ] },
        'Almancil' => {
          '8135' => [
            [ '100', '137', 'ALMANCIL' ],
            [ '139', '144', 'ALMANCIL' ],
            [ '146', '165', 'ALMANCIL' ],
            [ '167', '168', 'ALMANCIL' ],
            [ '170', '172', 'ALMANCIL' ],
            [ '853', '864', 'ALMANCIL' ],
            [ '866', '867', 'ALMANCIL' ] ] },
        'Aldeamento Quinta do Lago' => {
          '8135' => [
            [ '869', '869', 'ALMANCIL' ] ] },
        'Alcaria' => {
          '8100' => [
            [ '151', '211', 'SALIR' ] ] },
        'Amendoeira' => {
          '8100' => [
            [ '110', '133', 'QUERENA' ] ] },
        'Alm' => {
          '8135' => [
            [ '011', '037', 'ALMANCIL' ] ] },
        'Abelheira' => {
          '8100' => [
            [ '060', '099', 'BOLIQUEIME' ] ] },
        'Ameixial' => {
          '8100' => [
            [ '050', '056', 'AMEIXIAL' ] ] },
        'Aldeia de Tor' => {
          '8100' => [
            [ '381', '400', 'TR' ] ] },
        'Vale de guas' => {
          '8135' => [
            [ '872', '872', 'ALMANCIL' ] ] },
        'Loul' => {
          '8100' => [
            [ '221', '223', 'LOUL' ],
            [ '225', '238', 'LOUL' ],
            [ '240', '243', 'LOUL' ],
            [ '245', '257', 'LOUL' ],
            [ '259', '264', 'LOUL' ],
            [ '267', '272', 'LOUL' ],
            [ '280', '338', 'LOUL' ],
            [ '500', '506', 'LOUL' ],
            [ '508', '508', 'LOUL' ],
            [ '512', '555', 'LOUL' ],
            [ '558', '589', 'LOUL' ],
            [ '591', '592', 'LOUL' ],
            [ '596', '620', 'LOUL' ],
            [ '625', '629', 'LOUL' ],
            [ '638', '715', 'LOUL' ],
            [ '717', '800', 'LOUL' ],
            [ '851', '851', 'LOUL' ],
            [ '853', '854', 'LOUL' ] ] },
        'Quarteira' => {
          '8125' => [
            [ '001', '003', 'QUARTEIRA' ],
            [ '011', '024', 'QUARTEIRA' ],
            [ '090', '097', 'QUARTEIRA' ],
            [ '100', '100', 'QUARTEIRA' ],
            [ '105', '106', 'QUARTEIRA' ],
            [ '109', '110', 'QUARTEIRA' ],
            [ '112', '112', 'QUARTEIRA' ],
            [ '114', '114', 'QUARTEIRA' ],
            [ '116', '116', 'QUARTEIRA' ],
            [ '118', '119', 'QUARTEIRA' ],
            [ '122', '122', 'QUARTEIRA' ],
            [ '124', '124', 'QUARTEIRA' ],
            [ '134', '134', 'QUARTEIRA' ],
            [ '139', '139', 'QUARTEIRA' ],
            [ '141', '141', 'QUARTEIRA' ],
            [ '143', '143', 'QUARTEIRA' ],
            [ '146', '148', 'QUARTEIRA' ],
            [ '153', '154', 'QUARTEIRA' ],
            [ '156', '160', 'QUARTEIRA' ],
            [ '164', '225', 'QUARTEIRA' ],
            [ '227', '264', 'QUARTEIRA' ],
            [ '298', '298', 'QUARTEIRA' ],
            [ '301', '307', 'QUARTEIRA' ],
            [ '401', '404', 'QUARTEIRA' ],
            [ '406', '611', 'QUARTEIRA' ],
            [ '613', '613', 'QUARTEIRA' ],
            [ '615', '636', 'QUARTEIRA' ],
            [ '701', '710', 'QUARTEIRA' ] ] },
        'Alto Fica' => {
          '8100' => [
            [ '350', '358', 'BENAFIM' ] ] },
        'guas Frias' => {
          '8100' => [
            [ '010', '040', 'ALTE' ] ] }
         },
      'Lagoa (Algarve)' => {
        'Carvoeiro' => {
          '8400' => [
            [ '500', '537', 'CARVOEIRO LGA' ] ] },
        'Alfanzina' => {
          '8400' => [
            [ '550', '566', 'CARVOEIRO LGA' ] ] },
        'Ponte do Charuto' => {
          '8400' => [
            [ '654', '670', 'PARCHAL' ] ] },
        'Porches' => {
          '8400' => [
            [ '470', '499', 'PORCHES' ] ] },
        'Lagoa' => {
          '8400' => [
            [ '301', '403', 'LAGOA' ],
            [ '447', '449', 'LAGOA' ] ],
          '8401' => [
            [ '851', '854', 'LAGOA' ] ] },
        'Alporcinhos' => {
          '8400' => [
            [ '450', '460', 'PORCHES' ] ] },
        'Urbanizao Rocha Brava' => {
          '8400' => [
            [ '568', '568', 'CARVOEIRO LGA' ] ] },
        'Gramacho' => {
          '8400' => [
            [ '080', '080', 'ESTOMBAR' ] ] },
        'Mexilhoeira da Carregao' => {
          '8400' => [
            [ '090', '112', 'ESTOMBAR' ],
            [ '136', '168', 'ESTOMBAR' ] ] },
        'Poo dos Pardais' => {
          '8400' => [
            [ '120', '121', 'ESTOMBAR' ] ] },
        'Boa Nova' => {
          '8400' => [
            [ '200', '218', 'FERRAGUDO' ] ] },
        'Ferragudo' => {
          '8400' => [
            [ '220', '262', 'FERRAGUDO' ] ] },
        'Pateiro' => {
          '8400' => [
            [ '651', '651', 'PARCHAL' ] ] },
        'Carmo' => {
          '8400' => [
            [ '405', '407', 'LAGOA' ] ] },
        'Lameiras' => {
          '8400' => [
            [ '410', '445', 'LAGOA' ] ] },
        'Estombar' => {
          '8400' => [
            [ '060', '066', 'ESTOMBAR' ] ] },
        'Vale de Deus' => {
          '8400' => [
            [ '123', '134', 'ESTOMBAR' ] ] },
        'Urbanizao Presa de Moura' => {
          '8400' => [
            [ '008', '057', 'ESTOMBAR' ] ] },
        'Doca de Pesca' => {
          '8400' => [
            [ '278', '278', 'PARCHAL' ] ] },
        'Bemposta' => {
          '8400' => [
            [ '701', '701', 'LAGOA' ] ] },
        'Pintadinho' => {
          '8400' => [
            [ '270', '277', 'FERRAGUDO' ] ] },
        'Praia Angrinha' => {
          '8400' => [
            [ '279', '282', 'FERRAGUDO' ] ] },
        'Parchal' => {
          '8400' => [
            [ '287', '287', 'PARCHAL' ],
            [ '600', '621', 'PARCHAL' ],
            [ '623', '625', 'PARCHAL' ] ] }
             },
      'Faro' => {
        'Ilha da Culatra' => {
          '8700' => [
            [ '283', '283', 'FARO' ] ] },
        'Campinas' => {
          '8009' => [
            [ '001', '005', 'FARO' ] ] },
        'Jardina' => {
          '8005' => [
            [ '494', '542', 'FARO' ] ] },
        'Perna de Pau' => {
          '8009' => [
            [ '010', '020', 'FARO' ] ] },
        'Agostos' => {
          '8005' => [
            [ '401', '491', 'FARO' ] ] },
        'Ilha do Farol' => {
          '8700' => [
            [ '899', '899', 'FARO' ] ] },
        'Faro' => {
          '8000' => [
            [ '072', '081', 'FARO' ],
            [ '084', '084', 'FARO' ],
            [ '098', '098', 'FARO' ],
            [ '101', '101', 'FARO' ],
            [ '105', '107', 'FARO' ],
            [ '123', '123', 'FARO' ],
            [ '129', '129', 'FARO' ],
            [ '131', '131', 'FARO' ],
            [ '133', '135', 'FARO' ],
            [ '137', '140', 'FARO' ],
            [ '142', '151', 'FARO' ],
            [ '153', '157', 'FARO' ],
            [ '159', '174', 'FARO' ],
            [ '176', '177', 'FARO' ],
            [ '180', '180', 'FARO' ],
            [ '183', '190', 'FARO' ],
            [ '196', '206', 'FARO' ],
            [ '208', '211', 'FARO' ],
            [ '213', '250', 'FARO' ],
            [ '252', '253', 'FARO' ],
            [ '255', '263', 'FARO' ],
            [ '265', '276', 'FARO' ],
            [ '278', '279', 'FARO' ],
            [ '281', '286', 'FARO' ],
            [ '290', '307', 'FARO' ],
            [ '309', '309', 'FARO' ],
            [ '311', '316', 'FARO' ],
            [ '318', '324', 'FARO' ],
            [ '326', '329', 'FARO' ],
            [ '332', '340', 'FARO' ],
            [ '342', '360', 'FARO' ],
            [ '362', '372', 'FARO' ],
            [ '378', '379', 'FARO' ],
            [ '381', '382', 'FARO' ],
            [ '384', '384', 'FARO' ],
            [ '386', '394', 'FARO' ],
            [ '396', '406', 'FARO' ],
            [ '408', '411', 'FARO' ],
            [ '413', '413', 'FARO' ],
            [ '416', '434', 'FARO' ],
            [ '437', '438', 'FARO' ],
            [ '440', '440', 'FARO' ],
            [ '442', '448', 'FARO' ],
            [ '451', '452', 'FARO' ],
            [ '455', '461', 'FARO' ],
            [ '463', '467', 'FARO' ],
            [ '469', '472', 'FARO' ],
            [ '474', '483', 'FARO' ],
            [ '498', '504', 'FARO' ],
            [ '510', '510', 'FARO' ],
            [ '518', '518', 'FARO' ],
            [ '536', '536', 'FARO' ],
            [ '538', '544', 'FARO' ] ],
          '8004' => [
            [ '001', '003', 'FARO' ],
            [ '005', '015', 'FARO' ],
            [ '017', '018', 'FARO' ],
            [ '020', '031', 'FARO' ],
            [ '033', '040', 'FARO' ],
            [ '045', '047', 'FARO' ],
            [ '049', '050', 'FARO' ],
            [ '060', '061', 'FARO' ],
            [ '063', '064', 'FARO' ],
            [ '069', '071', 'FARO' ] ],
          '8005' => [
            [ '100', '142', 'FARO' ],
            [ '144', '288', 'FARO' ],
            [ '290', '336', 'FARO' ] ] }
        },
      'Olho' => {
        'Alfandanga' => {
          '8700' => [
            [ '061', '106', 'MONCARAPACHO' ] ] },
        'Bias do Norte' => {
          '8700' => [
            [ '851', '851', 'MONCARAPACHO' ] ] },
        'Murtais' => {
          '8700' => [
            [ '120', '131', 'MONCARAPACHO' ] ] },
        'Fuseta' => {
          '8700' => [
            [ '011', '043', 'FUSETA' ],
            [ '151', '151', 'FUSETA' ] ] },
        'Ilha da Fuzeta' => {
          '8700' => [
            [ '898', '898', 'FUSETA' ] ] },
        'Olho' => {
          '8700' => [
            [ '152', '282', 'OLHO' ],
            [ '284', '302', 'OLHO' ],
            [ '304', '304', 'OLHO' ],
            [ '307', '308', 'OLHO' ],
            [ '310', '315', 'OLHO' ],
            [ '317', '450', 'OLHO' ],
            [ '453', '578', 'OLHO' ],
            [ '852', '853', 'OLHO' ] ] }
         },
      'Vila do Bispo' => {
        'Burgau' => {
          '8650' => [
            [ '100', '133', 'BUDENS' ] ] },
        'Barrudo' => {
          '8650' => [
            [ '400', '432', 'VILA DO BISPO' ] ] },
        'Budens' => {
          '8650' => [
            [ '050', '089', 'BUDENS' ] ] },
        'Baro de So Miguel' => {
          '8650' => [
            [ '001', '005', 'BARO DE SO MIGUEL' ],
            [ '007', '027', 'BARO DE SO MIGUEL' ],
            [ '450', '453', 'BARO DE SO MIGUEL' ] ] },
        'Stio das Marias' => {
          '8650' => [
            [ '215', '215', 'BUDENS' ] ] },
        'Figueira' => {
          '8650' => [
            [ '150', '183', 'BUDENS' ] ] },
        'Sagres' => {
          '8650' => [
            [ '317', '389', 'SAGRES' ] ] },
        'Quinta do Forte' => {
          '8650' => [
            [ '210', '210', 'BUDENS' ] ] },
        'Praia da Salema' => {
          '8650' => [
            [ '190', '202', 'BUDENS' ] ] },
        'Almadaninhas' => {
          '8650' => [
            [ '040', '041', 'BUDENS' ] ] },
        'Quinta das Furnas' => {
          '8650' => [
            [ '035', '035', 'BARO DE SO MIGUEL' ] ] },
        'Caminho do Infante' => {
          '8650' => [
            [ '140', '140', 'BUDENS' ] ] },
        'Cerro de Monchique' => {
          '8650' => [
            [ '280', '312', 'RAPOSEIRA' ] ] },
        'Monte Rui Vaz' => {
          '8650' => [
            [ '030', '030', 'BARO DE SO MIGUEL' ] ] },
        'Vale do Boi' => {
          '8650' => [
            [ '220', '231', 'BUDENS' ] ] }
           },
      'Tavira' => {
        'Arroio' => {
          '8800' => [
            [ '101', '102', 'LUZ TVR' ] ] },
        'Alcarias Baixas' => {
          '8800' => [
            [ '011', '030', 'CACHOPO' ] ] },
        'Asseca (santiago)' => {
          '8800' => [
            [ '251', '255', 'TAVIRA' ] ] },
        'Aldeamento Pedras da Rainha' => {
          '8800' => [
            [ '051', '054', 'CONCEIO TVR' ] ] },
        'Urbanizao Quinta do Perogil' => {
          '8800' => [
            [ '069', '077', 'CONCEIO TVR' ] ] },
        'Patarinho' => {
          '8800' => [
            [ '501', '501', 'TAVIRA' ] ] },
        'Canada' => {
          '8800' => [
            [ '057', '064', 'CONCEIO TVR' ] ] },
        'Bernardinheiro' => {
          '8800' => [
            [ '513', '513', 'TAVIRA' ] ] },
        'Alcaria de Cume de Cima' => {
          '8800' => [
            [ '151', '168', 'STA CATARINA FONTE BISPO' ] ] },
        'Asseca (santa Maria)' => {
          '8800' => [
            [ '201', '223', 'TAVIRA' ] ] },
        'Aldeamento das Pedras D\'El Rei' => {
          '8800' => [
            [ '531', '572', 'SANTA LUZIA TVR' ] ] },
        'Cabanas' => {
          '8800' => [
            [ '591', '594', 'CABANAS DE TAVIRA' ] ] },
        'Asseca (Santo Estevo)' => {
          '8800' => [
            [ '514', '514', 'SANTO ESTEVO TVR' ] ] },
        'Barrada' => {
          '8800' => [
            [ '104', '121', 'LUZ TVR' ] ] },
        'Tavira' => {
          '8801' => [
            [ '003', '005', 'TAVIRA' ] ],
          '8800' => [
            [ '301', '301', 'TAVIRA' ],
            [ '303', '303', 'TAVIRA' ],
            [ '305', '316', 'TAVIRA' ],
            [ '318', '357', 'TAVIRA' ],
            [ '359', '383', 'TAVIRA' ],
            [ '385', '386', 'TAVIRA' ],
            [ '388', '399', 'TAVIRA' ],
            [ '401', '402', 'TAVIRA' ],
            [ '404', '408', 'TAVIRA' ],
            [ '410', '435', 'TAVIRA' ],
            [ '437', '443', 'TAVIRA' ],
            [ '445', '448', 'TAVIRA' ],
            [ '450', '462', 'TAVIRA' ],
            [ '464', '464', 'TAVIRA' ],
            [ '466', '466', 'TAVIRA' ],
            [ '602', '606', 'TAVIRA' ],
            [ '608', '612', 'TAVIRA' ],
            [ '616', '621', 'TAVIRA' ],
            [ '623', '639', 'TAVIRA' ],
            [ '641', '699', 'TAVIRA' ],
            [ '701', '707', 'TAVIRA' ],
            [ '709', '743', 'TAVIRA' ] ] },
        'Santa Luzia' => {
          '8800' => [
            [ '574', '574', 'SANTA LUZIA TVR' ] ] },
        'Mato da Ordem' => {
          '8800' => [
            [ '066', '067', 'CONCEIO TVR' ] ] },
        'Conceio' => {
          '8800' => [
            [ '079', '082', 'CONCEIO TVR' ] ] },
        'Urbanizao Vale Carangueijo' => {
          '8800' => [
            [ '225', '230', 'TAVIRA' ] ] },
        'Barranco da Nora' => {
          '8800' => [
            [ '502', '512', 'SANTO ESTEVO TVR' ] ] }
          },
      'So Brs de Alportel' => {
        'So Brs de Alportel' => {
          '8150' => [
            [ '101', '108', 'SO BRS DE ALPORTEL' ],
            [ '111', '121', 'SO BRS DE ALPORTEL' ],
            [ '125', '135', 'SO BRS DE ALPORTEL' ],
            [ '137', '161', 'SO BRS DE ALPORTEL' ],
            [ '164', '166', 'SO BRS DE ALPORTEL' ],
            [ '168', '176', 'SO BRS DE ALPORTEL' ] ] },
        'Outeiro' => {
          '8150' => [
            [ '050', '052', 'SO BRS DE ALPORTEL' ] ] },
        'Alcaria' => {
          '8150' => [
            [ '011', '048', 'SO BRS DE ALPORTEL' ] ] },
        'Poo dos Ferreiros' => {
          '8150' => [
            [ '054', '063', 'SO BRS DE ALPORTEL' ] ] }
            },
      'Vila Real de Santo Antnio' => {
        'Monte Gordo' => {
          '8900' => [
            [ '411', '415', 'MONTE GORDO' ],
            [ '417', '417', 'MONTE GORDO' ],
            [ '419', '441', 'MONTE GORDO' ],
            [ '443', '474', 'MONTE GORDO' ],
            [ '476', '476', 'MONTE GORDO' ] ] },
        'Monte da Rosa' => {
          '8900' => [
            [ '041', '057', 'VILA NOVA DE CACELA' ] ] },
        'Stio das Areias' => {
          '8900' => [
            [ '011', '015', 'VILA NOVA DE CACELA' ] ] },
        'Corujeira' => {
          '8900' => [
            [ '026', '039', 'VILA NOVA DE CACELA' ] ] },
        'Santa Rita' => {
          '8900' => [
            [ '059', '059', 'VILA NOVA DE CACELA' ] ] },
        'Casas Novas' => {
          '8900' => [
            [ '023', '024', 'VILA NOVA DE CACELA' ] ] },
        'Bornacha' => {
          '8900' => [
            [ '017', '021', 'VILA NOVA DE CACELA' ] ] },
        'Ceal' => {
          '8900' => [
            [ '102', '104', 'VILA REAL SANTO ANTNIO' ] ] },
        'Monte Olimpio' => {
          '8900' => [
            [ '106', '122', 'VILA REAL SANTO ANTNIO' ] ] },
        'Vila Real de Santo Antnio' => {
          '8900' => [
            [ '201', '207', 'VILA REAL SANTO ANTNIO' ],
            [ '209', '219', 'VILA REAL SANTO ANTNIO' ],
            [ '221', '259', 'VILA REAL SANTO ANTNIO' ],
            [ '261', '323', 'VILA REAL SANTO ANTNIO' ],
            [ '325', '325', 'VILA REAL SANTO ANTNIO' ],
            [ '327', '339', 'VILA REAL SANTO ANTNIO' ] ] },
        'Aldeia Nova' => {
          '8900' => [
            [ '401', '401', 'MONTE GORDO' ] ] },
        'Hortas' => {
          '8900' => [
            [ '406', '407', 'MONTE GORDO' ] ] },
        'Stio do Alto' => {
          '8900' => [
            [ '061', '067', 'VILA NOVA DE CACELA' ] ] },
        'Casa da Audncia' => {
          '8900' => [
            [ '404', '404', 'MONTE GORDO' ] ] }
                  },
      'Alcoutim' => {
        'Alcaria' => {
          '8970' => [
            [ '321', '348', 'VAQUEIROS ACT' ] ] },
        'Alcaria Alta' => {
          '8970' => [
            [ '101', '106', 'GIES' ] ] },
        'Vrzea' => {
          '8970' => [
            [ '391', '392', 'VAQUEIROS ACT' ] ] },
        'Guerreiros do Rio' => {
          '8970' => [
            [ '025', '034', 'ALCOUTIM' ] ] },
        'Afonso Vicente' => {
          '8970' => [
            [ '011', '023', 'ALCOUTIM' ] ] },
        'Alcoutim' => {
          '8970' => [
            [ '051', '079', 'ALCOUTIM' ] ] },
        'Azinhal' => {
          '8970' => [
            [ '201', '221', 'MARTIM LONGO' ] ] },
        'Martinlongo' => {
          '8970' => [
            [ '251', '284', 'MARTIM LONGO' ] ] },
        'Alcaria Cova de Baixo' => {
          '8970' => [
            [ '301', '311', 'PEREIRO ACT' ] ] },
        'Fincia Rodilha' => {
          '8970' => [
            [ '291', '296', 'MARTIM LONGO' ] ] },
        'Vaqueiros' => {
          '8970' => [
            [ '351', '379', 'VAQUEIROS ACT' ] ] }
            },
      'Monchique' => {
        'Alferce' => {
          '8550' => [
            [ '011', '048', 'ALFERCE' ] ] },
        'Aor' => {
          '8550' => [
            [ '201', '398', 'MONCHIQUE' ] ] },
        'Abitureira' => {
          '8550' => [
            [ '101', '169', 'MARMELETE' ] ] },
        'Monchique' => {
          '8550' => [
            [ '421', '473', 'MONCHIQUE' ] ] }
             }
        },
    'Ilha da Graciosa' => {
      'Santa Cruz da Graciosa' => {
        'Alto da Ribeirinha' => {
          '9880' => [
            [ '301', '316', 'SANTA CRUZ DA GRACIOSA' ] ] },
        'Outeiro' => {
          '9880' => [
            [ '170', '174', 'LUZ SCG' ] ] },
        'Praia da Graciosa' => {
          '9880' => [
            [ '210', '230', 'SO MATEUS SCG' ],
            [ '232', '239', 'SO MATEUS SCG' ],
            [ '241', '241', 'SO MATEUS SCG' ],
            [ '243', '245', 'SO MATEUS SCG' ] ] },
        'Trs dos Pomares' => {
          '9880' => [
            [ '285', '285', 'SO MATEUS SCG' ] ] },
        'Rochela' => {
          '9880' => [
            [ '280', '280', 'SO MATEUS SCG' ] ] },
        'Beira Mar da Rochela' => {
          '9880' => [
            [ '201', '207', 'SO MATEUS SCG' ] ] },
        'Limeira' => {
          '9880' => [
            [ '110', '110', 'LUZ SCG' ] ] },
        'Abaixo do Fragoso' => {
          '9880' => [
            [ '101', '108', 'LUZ SCG' ] ] },
        'Luz' => {
          '9880' => [
            [ '120', '154', 'LUZ SCG' ] ] },
        'Almas' => {
          '9880' => [
            [ '011', '029', 'GUADALUPE' ] ] },
        'Santa Cruz da Graciosa' => {
          '9880' => [
            [ '320', '394', 'SANTA CRUZ DA GRACIOSA' ] ] }
                    }
        },
    'Coimbra' => {
      'Penacova' => {
        'Ameal' => {
          '3360' => [
            [ '011', '034', 'CARVALHO PCV' ] ] },
        'Aveleira' => {
          '3360' => [
            [ '101', '110', 'LORVO' ] ] },
        'gua do Soito' => {
          '3360' => [
            [ '171', '204', 'PENACOVA' ] ] },
        'Aguieira' => {
          '3360' => [
            [ '311', '316', 'TRAVANCA DO MONDEGO' ] ] },
        'Casqueira' => {
          '3360' => [
            [ '283', '283', 'FIGUEIRA DE LORVO' ] ] },
        'Arroteia' => {
          '3360' => [
            [ '241', '268', 'SO PEDRO DE ALVA' ] ] },
        'Carregal' => {
          '3360' => [
            [ '071', '078', 'FRIMES' ] ] },
        'Bairro Martins Soares' => {
          '3360' => [
            [ '131', '137', 'OLIVEIRA DO MONDEGO' ] ] },
        'Contenas' => {
          '3360' => [
            [ '284', '293', 'SAZES DO LORVO' ] ] },
        'Azevinheiro' => {
          '3360' => [
            [ '281', '282', 'SAZES DO LORVO' ] ] },
        'Agrelo' => {
          '3360' => [
            [ '051', '055', 'FIGUEIRA DE LORVO' ] ] },
        'Granja' => {
          '3360' => [
            [ '057', '062', 'FIGUEIRA DE LORVO' ] ] },
        'Cortia' => {
          '3360' => [
            [ '151', '152', 'PARADELA PCV' ] ] },
        'Cabecinha' => {
          '3360' => [
            [ '221', '224', 'SO PAIO DE MONDEGO' ] ] }
               },
      'Figueira da Foz' => {
        'Alqueido' => {
          '3090' => [
            [ '431', '432', 'ALQUEIDO' ] ] },
        'Marinha das Ondas' => {
          '3090' => [
            [ '001', '007', 'MARINHA DAS ONDAS' ] ],
          '3094' => [
            [ '002', '002', 'MARINHA DAS ONDAS' ] ] },
        'Figueira da Foz' => {
          '3080' => [
            [ '032', '211', 'FIGUEIRA DA FOZ' ],
            [ '381', '396', 'FIGUEIRA DA FOZ' ] ],
          '3084' => [
            [ '501', '501', 'FIGUEIRA DA FOZ' ] ] },
        'Fontela' => {
          '3090' => [
            [ '641', '653', 'FIGUEIRA DA FOZ' ] ] },
        'So Pedro' => {
          '3090' => [
            [ '745', '750', 'FIGUEIRA DA FOZ' ],
            [ '868', '881', 'FIGUEIRA DA FOZ' ] ] },
        'Pipelo' => {
          '3090' => [
            [ '434', '434', 'ALQUEIDO' ] ] },
        'Sampaio' => {
          '3090' => [
            [ '762', '769', 'MARINHA DAS ONDAS' ] ] },
        'Casal da Fonte' => {
          '3090' => [
            [ '457', '461', 'LAVOS' ] ] },
        'Porto Carvalho' => {
          '3090' => [
            [ '837', '840', 'FERREIRA-A-NOVA' ] ] },
        'Canosa' => {
          '3090' => [
            [ '805', '807', 'FERREIRA-A-NOVA' ] ] },
        'Carritos' => {
          '3080' => [
            [ '843', '849', 'FIGUEIRA DA FOZ' ] ] },
        'Armazns' => {
          '3090' => [
            [ '451', '455', 'LAVOS' ] ] },
        'Porto Lamas' => {
          '3090' => [
            [ '828', '830', 'FERREIRA-A-NOVA' ] ] },
        'Bizorreiro' => {
          '3090' => [
            [ '491', '495', 'PAIO' ] ] },
        'Casal Verde' => {
          '3090' => [
            [ '433', '433', 'PAIO' ] ] },
        'Tavarede' => {
          '3080' => [
            [ '509', '581', 'FIGUEIRA DA FOZ' ],
            [ '621', '640', 'FIGUEIRA DA FOZ' ],
            [ '831', '831', 'FIGUEIRA DA FOZ' ] ] },
        'Zona Industrial da Figueira da Foz' => {
          '3090' => [
            [ '380', '380', 'FIGUEIRA DA FOZ' ] ] },
        'Fonte Nova' => {
          '3080' => [
            [ '213', '299', 'FIGUEIRA DA FOZ' ] ] },
        'Cabedelo' => {
          '3090' => [
            [ '661', '742', 'FIGUEIRA DA FOZ' ] ] },
        'Telhada' => {
          '3090' => [
            [ '497', '500', 'PAIO' ] ] },
        'Calvino' => {
          '3090' => [
            [ '811', '817', 'BORDA DO CAMPO' ] ] },
        'Canivetes' => {
          '3090' => [
            [ '770', '770', 'FIGUEIRA DA FOZ' ] ] },
        'Casal Oliveiras' => {
          '3090' => [
            [ '771', '804', 'SANTANA FIG' ] ] },
        'Arajos' => {
          '3080' => [
            [ '436', '444', 'FIGUEIRA DA FOZ' ] ] },
        'Vale Cardosos' => {
          '3090' => [
            [ '397', '428', 'FIGUEIRA DA FOZ' ] ] },
        'Gala' => {
          '3090' => [
            [ '898', '898', 'FIGUEIRA DA FOZ' ] ],
          '3094' => [
            [ '001', '001', 'FIGUEIRA DA FOZ' ] ] },
        'Alegria' => {
          '3090' => [
            [ '465', '483', 'MAIORCA' ],
            [ '808', '808', 'MAIORCA' ] ] },
        'Casal da Seia' => {
          '3090' => [
            [ '809', '810', 'MARINHA DAS ONDAS' ] ] },
        'Brejieira' => {
          '3080' => [
            [ '841', '841', 'FIGUEIRA DA FOZ' ] ] },
        'Buarcos' => {
          '3080' => [
            [ '301', '305', 'FIGUEIRA DA FOZ' ],
            [ '307', '378', 'FIGUEIRA DA FOZ' ],
            [ '852', '856', 'FIGUEIRA DA FOZ' ],
            [ '861', '897', 'FIGUEIRA DA FOZ' ] ] },
        'Brenha' => {
          '3080' => [
            [ '743', '744', 'FIGUEIRA DA FOZ' ] ] },
        'Cabo Mondego' => {
          '3080' => [
            [ '008', '030', 'FIGUEIRA DA FOZ' ] ] },
        'Bom Sucesso' => {
          '3080' => [
            [ '751', '763', 'BOM SUCESSO' ] ] },
        'Matioa' => {
          '3080' => [
            [ '601', '619', 'FIGUEIRA DA FOZ' ] ] },
        'Praia da Leirosa' => {
          '3090' => [
            [ '484', '485', 'MARINHA DAS ONDAS' ] ] },
        'Arneiro de Sazes' => {
          '3090' => [
            [ '821', '827', 'MOINHOS DA GNDARA' ] ] },
        'Caceira de Baixo' => {
          '3080' => [
            [ '501', '507', 'FIGUEIRA DA FOZ' ] ] },
        'Amieira' => {
          '3090' => [
            [ '832', '836', 'ALQUEIDO' ] ] },
        'Portela' => {
          '3090' => [
            [ '496', '496', 'ALQUEIDO' ] ] },
        'Praia de Buarcos' => {
          '3080' => [
            [ '899', '900', 'FIGUEIRA DA FOZ' ] ] },
        'Ferreira-A-Nova' => {
          '3090' => [
            [ '446', '448', 'FERREIRA-A-NOVA' ] ] }
                },
      'Cantanhede' => {
        'Bolho' => {
          '3060' => [
            [ '081', '083', 'BOLHO' ] ] },
        'Escapes' => {
          '3060' => [
            [ '531', '548', 'SEPINS' ] ] },
        'Vilamar' => {
          '3060' => [
            [ '761', '791', 'VILAMAR' ] ] },
        'Cabeo Redondo' => {
          '3060' => [
            [ '751', '752', 'CORTICEIRO DE CIMA' ] ] },
        'Escoural' => {
          '3060' => [
            [ '671', '693', 'TOCHA' ] ] },
        'Passinho' => {
          '3060' => [
            [ '470', '472', 'OURENT' ] ] },
        'Cantanhede' => {
          '3060' => [
            [ '121', '208', 'CANTANHEDE' ] ] },
        'Barreira da Malhada' => {
          '3060' => [
            [ '281', '299', 'COVES' ] ] },
        'Aljuria' => {
          '3060' => [
            [ '091', '115', 'CADIMA' ] ] },
        'Areia da Camarneira' => {
          '3060' => [
            [ '819', '826', 'CAMARNEIRA' ] ] },
        'Queixada da Raposa' => {
          '3060' => [
            [ '701', '702', 'TOCHA' ] ] },
        'Caniceira' => {
          '3060' => [
            [ '590', '616', 'TOCHA' ] ] },
        'Arrtas' => {
          '3060' => [
            [ '501', '503', 'POCARIA' ] ] },
        'Arrancada' => {
          '3060' => [
            [ '311', '344', 'FEBRES' ] ] },
        'Sanguinheira' => {
          '3060' => [
            [ '798', '818', 'SANGUINHEIRA' ] ] },
        'Casal do Joo' => {
          '3060' => [
            [ '625', '660', 'TOCHA' ] ] },
        'Outil' => {
          '3060' => [
            [ '491', '492', 'OUTIL' ] ] },
        'An' => {
          '3060' => [
            [ '001', '064', 'AN' ] ] },
        'Andrades' => {
          '3060' => [
            [ '581', '588', 'TOCHA' ] ] },
        'Sepins' => {
          '3060' => [
            [ '550', '566', 'SEPINS' ] ] },
        'Murtede' => {
          '3060' => [
            [ '401', '437', 'MURTEDE' ] ] },
        'Lemede' => {
          '3060' => [
            [ '211', '217', 'CANTANHEDE' ] ] },
        'Forno Branco' => {
          '3060' => [
            [ '350', '355', 'FEBRES' ] ] },
        'Lapa' => {
          '3060' => [
            [ '441', '465', 'OURENT' ] ] },
        'Tocha' => {
          '3060' => [
            [ '704', '719', 'TOCHA' ] ] },
        'Pena' => {
          '3060' => [
            [ '521', '523', 'PORTUNHOS' ] ] },
        'Granja de An' => {
          '3060' => [
            [ '070', '070', 'AN' ] ] },
        'Corgo Cvo' => {
          '3060' => [
            [ '731', '740', 'SO CAETANO CNT' ] ] },
        'Carvalho' => {
          '3060' => [
            [ '371', '397', 'MURTEDE' ] ] },
        'Azenha das 7 Fontes' => {
          '3060' => [
            [ '231', '269', 'CORDINH' ] ] }
           },
      'Pampilhosa da Serra' => {
        'Cabril' => {
          '3320' => [
            [ '011', '021', 'CABRIL PPS' ] ] },
        'Pampilhosa da Serra' => {
          '3320' => [
            [ '200', '239', 'PAMPILHOSA DA SERRA' ] ] },
        'Casal da Silva' => {
          '3320' => [
            [ '306', '310', 'PESSEGUEIRO' ] ] },
        'Amoreira Cimeira' => {
          '3320' => [
            [ '331', '341', 'PORTELA DO FOJO' ] ] },
        'Aduro' => {
          '3320' => [
            [ '051', '056', 'DORNELAS DO ZZERE' ] ] },
        'Armadouro' => {
          '3320' => [
            [ '101', '101', 'CABRIL PPS' ] ] },
        'Vale Mosqueiro' => {
          '3320' => [
            [ '113', '113', 'CABRIL PPS' ] ] },
        'Machio de Baixo' => {
          '3320' => [
            [ '131', '135', 'MACHIO' ] ] },
        'Casal da Lapa' => {
          '3320' => [
            [ '391', '393', 'VIDUAL' ] ] },
        'Brejo de Baixo' => {
          '3320' => [
            [ '102', '106', 'JANEIRO DE BAIXO' ] ] },
        'Pessegueiro de Cima' => {
          '3320' => [
            [ '312', '314', 'PESSEGUEIRO' ] ] },
        'Aradas' => {
          '3320' => [
            [ '361', '368', 'UNHAIS-O-VELHO' ] ] },
        'Aor' => {
          '3320' => [
            [ '151', '183', 'PAMPILHOSA DA SERRA' ] ] },
        'Porto de Vacas' => {
          '3320' => [
            [ '108', '112', 'JANEIRO DE BAIXO' ] ] },
        'Portela Armadouro' => {
          '3320' => [
            [ '107', '107', 'CABRIL PPS' ] ] },
        'Boias' => {
          '3320' => [
            [ '071', '087', 'FAJO' ] ] },
        'Braal' => {
          '3320' => [
            [ '301', '304', 'PESSEGUEIRO' ] ] }
              },
      'Penela' => {
        'Chaina' => {
          '3230' => [
            [ '540', '544', 'RABAAL PNL' ] ] },
        'Lagoa de Podentes' => {
          '3230' => [
            [ '491', '500', 'PODENTES' ] ] },
        'Alfafar' => {
          '3230' => [
            [ '481', '485', 'PODENTES' ] ] },
        'Alm de gua' => {
          '3230' => [
            [ '201', '205', 'PENELA' ] ] },
        'Ribeira' => {
          '3230' => [
            [ '535', '536', 'PODENTES' ] ] },
        'Espinhal' => {
          '3230' => [
            [ '070', '090', 'ESPINHAL' ] ] },
        'Bajancas Cimeiras' => {
          '3230' => [
            [ '051', '057', 'ESPINHAL' ] ] },
        'Penela' => {
          '3230' => [
            [ '275', '347', 'PENELA' ] ] },
        'Podentes' => {
          '3230' => [
            [ '521', '530', 'PODENTES' ] ] },
        'Boua' => {
          '3230' => [
            [ '011', '025', 'CUMEEIRA PNL' ] ] },
        'Carvalhais' => {
          '3230' => [
            [ '207', '273', 'PENELA' ] ] },
        'Fetais Cimeiros' => {
          '3230' => [
            [ '100', '114', 'ESPINHAL' ] ] }
             },
      'Gis' => {
        'Oliveirinhas' => {
          '3330' => [
            [ '401', '437', 'VILA NOVA DO CEIRA' ] ] },
        'Gis' => {
          '3330' => [
            [ '301', '334', 'GIS' ] ] },
        'Vila Nova do Ceira' => {
          '3330' => [
            [ '451', '463', 'VILA NOVA DO CEIRA' ] ] },
        'Alvem' => {
          '3330' => [
            [ '201', '208', 'GIS' ] ] },
        'Obrais' => {
          '3330' => [
            [ '101', '151', 'ALVARES' ] ] },
        'Aor' => {
          '3330' => [
            [ '066', '079', 'COLMEAL GOI' ] ] },
        'Pvoa' => {
          '3330' => [
            [ '210', '251', 'GIS' ] ] },
        'Sandinha' => {
          '3330' => [
            [ '011', '019', 'CADAFAZ GOI' ] ] }
           },
      'Soure' => {
        'Granja do Ulmeiro' => {
          '3130' => [
            [ '080', '082', 'GRANJA DO ULMEIRO' ] ] },
        'Valada' => {
          '3130' => [
            [ '066', '066', 'GESTEIRA' ] ] },
        'Casal do Rego' => {
          '3130' => [
            [ '048', '049', 'FIGUEIR DO CAMPO' ] ] },
        'Cabea da Corte' => {
          '3130' => [
            [ '090', '099', 'POMBALINHO SRE' ] ] },
        'Barreiras' => {
          '3130' => [
            [ '420', '435', 'VINHA DA RAINHA' ] ] },
        'Quatro Lagoas' => {
          '3130' => [
            [ '083', '083', 'POMBALINHO SRE' ] ] },
        'Soure' => {
          '3130' => [
            [ '199', '257', 'SOURE' ] ] },
        'Casais So Jorge' => {
          '3130' => [
            [ '020', '024', 'DEGRACIAS' ] ] },
        'Alfarelos' => {
          '3130' => [
            [ '001', '003', 'ALFARELOS' ] ] },
        'Alencarce Baixo' => {
          '3130' => [
            [ '500', '586', 'SOURE' ] ] },
        'Carregosa' => {
          '3130' => [
            [ '060', '064', 'GESTEIRA' ] ] },
        'Vila Nova de Anos' => {
          '3130' => [
            [ '400', '403', 'VILA NOVA DE ANOS' ] ] },
        'Baixos' => {
          '3130' => [
            [ '380', '389', 'TAPUS' ] ] },
        'Quinta Santa Cruz' => {
          '3130' => [
            [ '125', '133', 'SAMUEL' ] ] },
        'Alagoa' => {
          '3130' => [
            [ '110', '123', 'SAMUEL' ] ] },
        'Figueir do Campo' => {
          '3130' => [
            [ '040', '046', 'FIGUEIR DO CAMPO' ] ] },
        'Brunhs' => {
          '3130' => [
            [ '010', '010', 'BRUNHS' ] ] }
            },
      'Lous' => {
        'Cabril' => {
          '3200' => [
            [ '310', '358', 'SERPINS' ] ] },
        'Alcaide' => {
          '3200' => [
            [ '300', '305', 'SERPINS' ] ] },
        'Favaria' => {
          '3200' => [
            [ '084', '085', 'LOUS' ] ] },
        'Ribeiro Branco' => {
          '3200' => [
            [ '118', '118', 'LOUS' ] ] },
        'Boque' => {
          '3200' => [
            [ '306', '306', 'VILARINHO LSA' ] ] },
        'Fontainhas' => {
          '3200' => [
            [ '086', '086', 'GNDARAS LSA' ] ] },
        'Alaperna' => {
          '3200' => [
            [ '020', '049', 'FOZ DE AROUCE' ] ] },
        'Cume' => {
          '3200' => [
            [ '081', '081', 'GNDARAS LSA' ] ] },
        'Parque Industrial dos Matinhos' => {
          '3200' => [
            [ '100', '115', 'LOUS' ] ] },
        'Lous' => {
          '3200' => [
            [ '001', '002', 'LOUS' ],
            [ '200', '272', 'LOUS' ] ] },
        'Cabanes' => {
          '3200' => [
            [ '309', '309', 'VILARINHO LSA' ] ] },
        'Alfocheira' => {
          '3200' => [
            [ '060', '062', 'LOUS' ] ] },
        'Eira de Calva' => {
          '3200' => [
            [ '082', '082', 'LOUS' ] ] },
        'Espinheiro' => {
          '3200' => [
            [ '083', '083', 'GNDARAS LSA' ] ] },
        'Vinhas de Fiscal' => {
          '3200' => [
            [ '394', '401', 'VILARINHO LSA' ] ] },
        'Santa Luzia' => {
          '3200' => [
            [ '119', '119', 'GNDARAS LSA' ] ] },
        'Talasnal' => {
          '3200' => [
            [ '120', '163', 'LOUS' ] ] },
        'Fonte dos Mouros' => {
          '3200' => [
            [ '087', '095', 'LOUS' ] ] },
        'Bairro do Emigrante' => {
          '3200' => [
            [ '370', '392', 'VILARINHO LSA' ] ] },
        'Casal do Ermio' => {
          '3200' => [
            [ '010', '010', 'CASAL DE ERMIO' ] ] },
        'Moita' => {
          '3200' => [
            [ '096', '097', 'GNDARAS LSA' ] ] },
        'Papanata' => {
          '3200' => [
            [ '099', '099', 'GNDARAS LSA' ] ] },
        'Braal' => {
          '3200' => [
            [ '307', '308', 'SERPINS' ] ] },
        'Cabeo do Moiro' => {
          '3200' => [
            [ '064', '080', 'LOUS' ] ] },
        'Padro' => {
          '3200' => [
            [ '098', '098', 'LOUS' ] ] },
        'Reguengo' => {
          '3200' => [
            [ '116', '117', 'GNDARAS LSA' ] ] }
            },
      'Coimbra' => {
        'Ameal' => {
          '3045' => [
            [ '282', '322', 'AMEAL' ] ] },
        'Taveiro' => {
          '3049' => [
            [ '001', '001', 'TAVEIRO' ] ] },
        'Sandelgas' => {
          '3025' => [
            [ '474', '506', 'SO MARTINHO DE RVORE' ] ] },
        'Reveles' => {
          '3045' => [
            [ '444', '504', 'TAVEIRO' ] ] },
        'Pereiros' => {
          '3040' => [
            [ '721', '735', 'CASTELO VIEGAS' ] ] },
        'Boto' => {
          '3020' => [
            [ '521', '527', 'BOTO' ] ] },
        'Castanheira do Campo' => {
          '3025' => [
            [ '507', '586', 'SO SILVESTRE' ] ] },
        'Vila Pouca do Campo' => {
          '3045' => [
            [ '324', '352', 'AMEAL' ] ] },
        'Marmeleira' => {
          '3020' => [
            [ '742', '806', 'SOUSELAS' ] ] },
        'Ceira' => {
          '3030' => [
            [ '848', '881', 'CEIRA' ] ] },
        'Albergaria' => {
          '3040' => [
            [ '551', '589', 'ANTANHOL' ] ] },
        'So Martinho do Pinheiro' => {
          '3020' => [
            [ '808', '911', 'SOUSELAS' ] ] },
        'Andorinha' => {
          '3025' => [
            [ '329', '414', 'LAMAROSA' ] ] },
        'Casa Telhada' => {
          '3040' => [
            [ '755', '830', 'CERNACHE' ] ] },
        'Coimbra' => {
          '3045' => [
            [ '001', '281', 'COIMBRA' ],
            [ '381', '412', 'COIMBRA' ],
            [ '414', '416', 'COIMBRA' ] ],
          '3025' => [
            [ '001', '001', 'COIMBRA' ],
            [ '003', '174', 'COIMBRA' ],
            [ '176', '328', 'COIMBRA' ],
            [ '633', '633', 'COIMBRA' ] ],
          '3020' => [
            [ '080', '089', 'COIMBRA' ],
            [ '093', '093', 'COIMBRA' ],
            [ '095', '099', 'COIMBRA' ],
            [ '101', '102', 'COIMBRA' ],
            [ '104', '108', 'COIMBRA' ],
            [ '111', '115', 'COIMBRA' ],
            [ '119', '122', 'COIMBRA' ],
            [ '124', '136', 'COIMBRA' ],
            [ '154', '154', 'COIMBRA' ],
            [ '163', '165', 'COIMBRA' ],
            [ '167', '188', 'COIMBRA' ],
            [ '190', '194', 'COIMBRA' ],
            [ '196', '199', 'COIMBRA' ],
            [ '205', '208', 'COIMBRA' ],
            [ '210', '210', 'COIMBRA' ],
            [ '212', '221', 'COIMBRA' ],
            [ '223', '231', 'COIMBRA' ],
            [ '233', '233', 'COIMBRA' ],
            [ '235', '240', 'COIMBRA' ],
            [ '242', '244', 'COIMBRA' ],
            [ '246', '246', 'COIMBRA' ],
            [ '248', '251', 'COIMBRA' ],
            [ '253', '255', 'COIMBRA' ],
            [ '257', '257', 'COIMBRA' ],
            [ '261', '261', 'COIMBRA' ],
            [ '263', '267', 'COIMBRA' ],
            [ '269', '278', 'COIMBRA' ],
            [ '280', '285', 'COIMBRA' ],
            [ '306', '309', 'COIMBRA' ],
            [ '322', '324', 'COIMBRA' ],
            [ '368', '369', 'COIMBRA' ],
            [ '371', '371', 'COIMBRA' ],
            [ '384', '385', 'COIMBRA' ],
            [ '422', '424', 'COIMBRA' ],
            [ '428', '430', 'COIMBRA' ],
            [ '438', '438', 'COIMBRA' ],
            [ '458', '458', 'COIMBRA' ],
            [ '461', '462', 'COIMBRA' ],
            [ '476', '496', 'COIMBRA' ] ],
          '3004' => [
            [ '501', '502', 'COIMBRA' ],
            [ '504', '513', 'COIMBRA' ],
            [ '515', '519', 'COIMBRA' ],
            [ '522', '536', 'COIMBRA' ],
            [ '538', '539', 'COIMBRA' ],
            [ '544', '544', 'COIMBRA' ],
            [ '546', '548', 'COIMBRA' ],
            [ '550', '550', 'COIMBRA' ],
            [ '558', '558', 'COIMBRA' ] ],
          '3040' => [
            [ '001', '008', 'COIMBRA' ],
            [ '011', '014', 'COIMBRA' ],
            [ '016', '021', 'COIMBRA' ],
            [ '023', '023', 'COIMBRA' ],
            [ '025', '026', 'COIMBRA' ],
            [ '028', '030', 'COIMBRA' ],
            [ '032', '034', 'COIMBRA' ],
            [ '039', '052', 'COIMBRA' ],
            [ '054', '054', 'COIMBRA' ],
            [ '087', '087', 'COIMBRA' ],
            [ '089', '093', 'COIMBRA' ],
            [ '104', '112', 'COIMBRA' ],
            [ '114', '125', 'COIMBRA' ],
            [ '193', '195', 'COIMBRA' ],
            [ '197', '197', 'COIMBRA' ],
            [ '199', '202', 'COIMBRA' ],
            [ '206', '207', 'COIMBRA' ],
            [ '209', '209', 'COIMBRA' ],
            [ '211', '211', 'COIMBRA' ],
            [ '215', '215', 'COIMBRA' ],
            [ '222', '223', 'COIMBRA' ],
            [ '226', '230', 'COIMBRA' ],
            [ '241', '257', 'COIMBRA' ],
            [ '259', '276', 'COIMBRA' ],
            [ '320', '326', 'COIMBRA' ],
            [ '328', '330', 'COIMBRA' ],
            [ '335', '335', 'COIMBRA' ],
            [ '339', '340', 'COIMBRA' ],
            [ '349', '350', 'COIMBRA' ],
            [ '353', '356', 'COIMBRA' ],
            [ '358', '358', 'COIMBRA' ],
            [ '370', '370', 'COIMBRA' ],
            [ '372', '372', 'COIMBRA' ],
            [ '678', '678', 'COIMBRA' ] ],
          '3000' => [
            [ '001', '011', 'COIMBRA' ],
            [ '013', '038', 'COIMBRA' ],
            [ '040', '046', 'COIMBRA' ],
            [ '048', '048', 'COIMBRA' ],
            [ '050', '080', 'COIMBRA' ],
            [ '082', '088', 'COIMBRA' ],
            [ '090', '090', 'COIMBRA' ],
            [ '092', '092', 'COIMBRA' ],
            [ '094', '095', 'COIMBRA' ],
            [ '097', '104', 'COIMBRA' ],
            [ '106', '125', 'COIMBRA' ],
            [ '128', '130', 'COIMBRA' ],
            [ '132', '143', 'COIMBRA' ],
            [ '145', '145', 'COIMBRA' ],
            [ '147', '148', 'COIMBRA' ],
            [ '150', '151', 'COIMBRA' ],
            [ '153', '153', 'COIMBRA' ],
            [ '155', '155', 'COIMBRA' ],
            [ '157', '180', 'COIMBRA' ],
            [ '182', '187', 'COIMBRA' ],
            [ '189', '192', 'COIMBRA' ],
            [ '194', '197', 'COIMBRA' ],
            [ '199', '210', 'COIMBRA' ],
            [ '213', '215', 'COIMBRA' ],
            [ '217', '235', 'COIMBRA' ],
            [ '238', '254', 'COIMBRA' ],
            [ '256', '272', 'COIMBRA' ],
            [ '274', '278', 'COIMBRA' ],
            [ '280', '313', 'COIMBRA' ],
            [ '315', '320', 'COIMBRA' ],
            [ '322', '327', 'COIMBRA' ],
            [ '329', '331', 'COIMBRA' ],
            [ '333', '333', 'COIMBRA' ],
            [ '335', '341', 'COIMBRA' ],
            [ '343', '343', 'COIMBRA' ],
            [ '345', '348', 'COIMBRA' ],
            [ '350', '351', 'COIMBRA' ],
            [ '353', '353', 'COIMBRA' ],
            [ '355', '363', 'COIMBRA' ],
            [ '365', '368', 'COIMBRA' ],
            [ '370', '370', 'COIMBRA' ],
            [ '372', '409', 'COIMBRA' ],
            [ '411', '416', 'COIMBRA' ],
            [ '420', '423', 'COIMBRA' ],
            [ '426', '426', 'COIMBRA' ],
            [ '428', '436', 'COIMBRA' ],
            [ '438', '438', 'COIMBRA' ],
            [ '442', '442', 'COIMBRA' ],
            [ '445', '449', 'COIMBRA' ],
            [ '451', '452', 'COIMBRA' ],
            [ '456', '460', 'COIMBRA' ],
            [ '464', '470', 'COIMBRA' ],
            [ '472', '473', 'COIMBRA' ],
            [ '475', '477', 'COIMBRA' ],
            [ '481', '481', 'COIMBRA' ],
            [ '484', '484', 'COIMBRA' ],
            [ '486', '487', 'COIMBRA' ],
            [ '489', '492', 'COIMBRA' ],
            [ '494', '494', 'COIMBRA' ],
            [ '503', '503', 'COIMBRA' ],
            [ '507', '509', 'COIMBRA' ],
            [ '511', '511', 'COIMBRA' ],
            [ '515', '516', 'COIMBRA' ],
            [ '520', '520', 'COIMBRA' ],
            [ '522', '525', 'COIMBRA' ],
            [ '538', '538', 'COIMBRA' ],
            [ '540', '540', 'COIMBRA' ],
            [ '542', '543', 'COIMBRA' ],
            [ '546', '550', 'COIMBRA' ],
            [ '552', '552', 'COIMBRA' ],
            [ '554', '555', 'COIMBRA' ],
            [ '559', '559', 'COIMBRA' ],
            [ '561', '561', 'COIMBRA' ],
            [ '563', '564', 'COIMBRA' ],
            [ '573', '575', 'COIMBRA' ],
            [ '578', '578', 'COIMBRA' ],
            [ '589', '589', 'COIMBRA' ],
            [ '593', '594', 'COIMBRA' ],
            [ '597', '597', 'COIMBRA' ],
            [ '601', '604', 'COIMBRA' ],
            [ '606', '606', 'COIMBRA' ] ],
          '3030' => [
            [ '001', '025', 'COIMBRA' ],
            [ '027', '076', 'COIMBRA' ],
            [ '078', '103', 'COIMBRA' ],
            [ '105', '112', 'COIMBRA' ],
            [ '116', '116', 'COIMBRA' ],
            [ '118', '137', 'COIMBRA' ],
            [ '139', '145', 'COIMBRA' ],
            [ '148', '153', 'COIMBRA' ],
            [ '155', '158', 'COIMBRA' ],
            [ '161', '170', 'COIMBRA' ],
            [ '172', '173', 'COIMBRA' ],
            [ '175', '175', 'COIMBRA' ],
            [ '177', '190', 'COIMBRA' ],
            [ '192', '207', 'COIMBRA' ],
            [ '212', '215', 'COIMBRA' ],
            [ '217', '218', 'COIMBRA' ],
            [ '220', '253', 'COIMBRA' ],
            [ '258', '265', 'COIMBRA' ],
            [ '268', '280', 'COIMBRA' ],
            [ '282', '283', 'COIMBRA' ],
            [ '285', '290', 'COIMBRA' ],
            [ '292', '293', 'COIMBRA' ],
            [ '295', '302', 'COIMBRA' ],
            [ '304', '304', 'COIMBRA' ],
            [ '306', '350', 'COIMBRA' ],
            [ '352', '362', 'COIMBRA' ],
            [ '364', '403', 'COIMBRA' ],
            [ '410', '410', 'COIMBRA' ],
            [ '413', '417', 'COIMBRA' ],
            [ '419', '419', 'COIMBRA' ],
            [ '422', '422', 'COIMBRA' ],
            [ '424', '433', 'COIMBRA' ],
            [ '436', '436', 'COIMBRA' ],
            [ '438', '446', 'COIMBRA' ],
            [ '450', '454', 'COIMBRA' ],
            [ '457', '457', 'COIMBRA' ],
            [ '460', '462', 'COIMBRA' ],
            [ '464', '464', 'COIMBRA' ],
            [ '468', '468', 'COIMBRA' ],
            [ '471', '489', 'COIMBRA' ],
            [ '775', '775', 'COIMBRA' ] ] },
        'Carvalhais de Baixo' => {
          '3040' => [
            [ '679', '693', 'ASSAFARGE' ] ] },
        'Abelheira' => {
          '3040' => [
            [ '421', '491', 'ALMALAGUS' ] ] },
        'Casal de So Joo' => {
          '3040' => [
            [ '701', '719', 'CASTELO VIEGAS' ] ] },
        'Moinho do Calhau' => {
          '3045' => [
            [ '418', '443', 'RIBEIRA DE FRADES' ] ] },
        'Casal da Murteira' => {
          '3025' => [
            [ '587', '631', 'VIL DE MATOS' ] ] },
        'Boia' => {
          '3030' => [
            [ '883', '898', 'CEIRA' ] ] },
        'Arzila' => {
          '3045' => [
            [ '353', '380', 'ARZILA' ] ] },
        'Aerdromo' => {
          '3040' => [
            [ '751', '753', 'CERNACHE' ] ] },
        'Abrunheira' => {
          '3040' => [
            [ '641', '677', 'ASSAFARGE' ] ] },
        'Sobral de Ceira' => {
          '3030' => [
            [ '900', '910', 'CEIRA' ] ] },
        'Bostelim' => {
          '3020' => [
            [ '531', '578', 'BRASFEMES' ] ] },
        'So Joo do Campo' => {
          '3025' => [
            [ '415', '473', 'SO JOO DO CAMPO' ],
            [ '632', '632', 'SO JOO DO CAMPO' ] ] },
        'Caselho' => {
          '3020' => [
            [ '921', '929', 'TORRE DE VILELA' ] ] }
              },
      'Arganil' => {
        'Casal da Eira' => {
          '3305' => [
            [ '221', '228', 'MOURA DA SERRA' ] ] },
        'Casal Novo' => {
          '3300' => [
            [ '221', '222', 'CEPOS' ] ] },
        'Alagoas' => {
          '3300' => [
            [ '301', '330', 'POMBEIRO DA BEIRA' ] ] },
        'Vale Peitalva' => {
          '3300' => [
            [ '450', '450', 'SECARIAS' ] ] },
        'Anceriz' => {
          '3305' => [
            [ '010', '010', 'ANCERIZ' ] ] },
        'Alqueve' => {
          '3300' => [
            [ '252', '252', 'FOLQUES' ] ] },
        'Vale do Carro' => {
          '3305' => [
            [ '193', '193', 'COJA' ] ] },
        'Quinta da Fonte de Pipa' => {
          '3305' => [
            [ '050', '050', 'CERDEIRA AGN' ] ] },
        'Mancelavisa' => {
          '3300' => [
            [ '281', '286', 'FOLQUES' ] ] },
        'gua D\'Alte' => {
          '3300' => [
            [ '461', '466', 'TEIXEIRA AGN' ] ] },
        'Alagoa' => {
          '3300' => [
            [ '101', '129', 'ARGANIL' ] ] },
        'Carato' => {
          '3305' => [
            [ '278', '286', 'VILA COVA DE ALVA' ] ] },
        'Covo' => {
          '3305' => [
            [ '250', '267', 'POMARES AGN' ] ] },
        'Adcasal' => {
          '3300' => [
            [ '201', '202', 'CELAVISA' ] ] },
        'Sarzedo' => {
          '3300' => [
            [ '401', '422', 'SARZEDO AGN' ] ] },
        'Arganil' => {
          '3300' => [
            [ '011', '068', 'ARGANIL' ] ] },
        'Quinta do Pau' => {
          '3305' => [
            [ '030', '039', 'BENFEITA' ] ] },
        'Berreiros' => {
          '6285' => [
            [ '011', '019', 'PIDO' ] ] },
        'Coja' => {
          '3305' => [
            [ '090', '104', 'COJA' ],
            [ '110', '153', 'COJA' ] ] },
        'Casal do Sarzedo' => {
          '3300' => [
            [ '431', '433', 'SARZEDO AGN' ] ] },
        'Celavisa' => {
          '3300' => [
            [ '204', '208', 'CELAVISA' ] ] },
        'Folques' => {
          '3300' => [
            [ '261', '276', 'FOLQUES' ] ] },
        'Abrunheira' => {
          '3300' => [
            [ '351', '374', 'SO MARTINHO DA CORTIA' ] ] },
        'Frnea' => {
          '3305' => [
            [ '240', '240', 'PIDO' ] ] },
        'Barril de Alva' => {
          '3305' => [
            [ '020', '020', 'BARRIL DE ALVA' ] ] },
        'Esculca' => {
          '3305' => [
            [ '181', '191', 'COJA' ] ] }
              },
      'Montemor-o-Velho' => {
        'Santo Varo' => {
          '3140' => [
            [ '401', '401', 'SANTO VARO' ] ] },
        'Alhastro' => {
          '3140' => [
            [ '071', '105', 'CARAPINHEIRA' ] ] },
        'Areal' => {
          '3140' => [
            [ '201', '206', 'MONTEMOR-O-VELHO' ] ] },
        'Ereira' => {
          '3140' => [
            [ '672', '672', 'EREIRA MMV' ] ] },
        'Bebedouro' => {
          '3140' => [
            [ '026', '046', 'ARAZEDE' ] ] },
        'Verride' => {
          '3140' => [
            [ '601', '617', 'VERRIDE' ] ] },
        'Casal Penas' => {
          '3140' => [
            [ '437', '438', 'TENTGAL' ] ] },
        'Formoselha' => {
          '3140' => [
            [ '351', '354', 'SANTO VARO' ],
            [ '356', '364', 'SANTO VARO' ] ] },
        'Pereira' => {
          '3140' => [
            [ '331', '346', 'PEREIRA MMV' ] ] },
        'Casal Jagaz' => {
          '3140' => [
            [ '436', '436', 'SEIXO DE GATES' ] ] },
        'Cabea Alta' => {
          '3140' => [
            [ '431', '431', 'SEIXO DE GATES' ] ] },
        'Amieiro' => {
          '3140' => [
            [ '021', '024', 'ARAZEDE' ] ] },
        'Casal Silvas' => {
          '3140' => [
            [ '121', '122', 'GATES' ] ] },
        'Casais Velhos' => {
          '3140' => [
            [ '301', '329', 'PEREIRA MMV' ] ] },
        'Vila Nova da Barca' => {
          '3140' => [
            [ '651', '653', 'VILA NOVA DA BARCA' ] ] },
        'So Jorge' => {
          '3140' => [
            [ '141', '141', 'GATES' ] ] },
        'Montemor-o-Velho' => {
          '3140' => [
            [ '249', '275', 'MONTEMOR-O-VELHO' ],
            [ '851', '853', 'MONTEMOR-O-VELHO' ] ] },
        'Montes de Formoselha' => {
          '3140' => [
            [ '390', '391', 'SANTO VARO' ] ] },
        'Fonterma' => {
          '3140' => [
            [ '439', '449', 'SEIXO DE GATES' ] ] },
        'Casal dos Craveiros' => {
          '3140' => [
            [ '432', '435', 'TENTGAL' ] ] },
        'Calaotas' => {
          '3140' => [
            [ '161', '169', 'MES DO CAMPO' ] ] },
        'Tentgal' => {
          '3140' => [
            [ '551', '581', 'TENTGAL' ] ] },
        'Arroia' => {
          '3140' => [
            [ '142', '149', 'LICEIA' ] ] },
        'Abrunheira' => {
          '3140' => [
            [ '011', '011', 'ABRUNHEIRA' ] ] },
        'Porto Expinheiro' => {
          '3140' => [
            [ '583', '586', 'TENTGAL' ] ] },
        'Portela' => {
          '3140' => [
            [ '481', '500', 'TENTGAL' ] ] },
        'Carril' => {
          '3140' => [
            [ '014', '016', 'ABRUNHEIRA' ] ] }
                 },
      'Condeixa-a-Nova' => {
        'Avenal' => {
          '3150' => [
            [ '275', '291', 'SEBAL' ] ] },
        'Fonte dos Amores' => {
          '3150' => [
            [ '191', '194', 'CONDEIXA-A-NOVA' ] ] },
        'Bem da F' => {
          '3150' => [
            [ '040', '040', 'BEM DA F' ] ] },
        'Alcouce' => {
          '3150' => [
            [ '311', '318', 'VILA SECA CDN' ] ] },
        'Ameixeira' => {
          '3150' => [
            [ '251', '261', 'EGA' ] ] },
        'Alvogadas' => {
          '3150' => [
            [ '011', '025', 'ANOBRA' ] ] },
        'Cadaval Grande' => {
          '3150' => [
            [ '271', '274', 'FURADOURO' ] ] },
        'Barreira' => {
          '3150' => [
            [ '110', '111', 'CONDEIXA-A-NOVA' ] ] },
        'Casal de Santo Amaro' => {
          '3150' => [
            [ '030', '030', 'BELIDE' ] ] },
        'Fonte Coberta' => {
          '3150' => [
            [ '331', '334', 'ZAMBUJAL' ] ] },
        'Alcabideque' => {
          '3150' => [
            [ '211', '231', 'CONDEIXA-A-VELHA' ] ] },
        'Condeixa-a-Nova' => {
          '3150' => [
            [ '120', '155', 'CONDEIXA-A-NOVA' ] ] }
                },
      'Oliveira do Hospital' => {
        'Vila Franca da Beira' => {
          '3405' => [
            [ '601', '645', 'VILA FRANCA DA BEIRA' ] ] },
        'Vila Pouca da Beira' => {
          '3400' => [
            [ '755', '780', 'VILA POUCA DA BEIRA' ] ] },
        'Vale Torto' => {
          '3405' => [
            [ '386', '463', 'SEIXO DA BEIRA' ] ] },
        'Quinta da Ferraria' => {
          '3400' => [
            [ '335', '336', 'ALVOCO DAS VRZEAS' ] ] },
        'Pvoa das Quartas' => {
          '3405' => [
            [ '280', '281', 'LAGOS DA BEIRA' ] ] },
        'Secolinho' => {
          '3400' => [
            [ '269', '270', 'ALDEIA DAS DEZ' ] ] },
        'Malhadinha' => {
          '3400' => [
            [ '331', '332', 'ALVOCO DAS VRZEAS' ] ] },
        'Bobadela' => {
          '3405' => [
            [ '001', '002', 'BOBADELA OHP' ],
            [ '004', '028', 'BOBADELA OHP' ],
            [ '142', '144', 'BOBADELA OHP' ] ] },
        'Santa Ovaia' => {
          '3400' => [
            [ '591', '591', 'SANTA OVAIA' ],
            [ '593', '631', 'SANTA OVAIA' ] ] },
        'Caldas de So Paulo' => {
          '3400' => [
            [ '521', '576', 'PENALVA DE ALVA' ] ] },
        'Andorinha' => {
          '3405' => [
            [ '477', '531', 'TRAVANCA DE LAGOS' ] ] },
        'Meruge' => {
          '3405' => [
            [ '350', '378', 'MERUGE' ] ] },
        'Tapada' => {
          '3400' => [
            [ '339', '340', 'ALVOCO DAS VRZEAS' ] ] },
        'Alentejo' => {
          '3400' => [
            [ '641', '673', 'SO GIO' ] ] },
        'Aldeia das Dez' => {
          '3400' => [
            [ '201', '264', 'ALDEIA DAS DEZ' ],
            [ '592', '592', 'ALDEIA DAS DEZ' ] ] },
        'Vendas de Gavinhos' => {
          '3400' => [
            [ '169', '184', 'OLIVEIRA DO HOSPITAL' ] ] },
        'Biqueiro' => {
          '3400' => [
            [ '001', '166', 'OLIVEIRA DO HOSPITAL' ] ] },
        'Nogueira do Cravo' => {
          '3400' => [
            [ '427', '457', 'NOGUEIRA DO CRAVO OHP' ],
            [ '461', '508', 'NOGUEIRA DO CRAVO OHP' ] ] },
        'Av' => {
          '3400' => [
            [ '357', '384', 'AV' ] ] },
        'Chamusca' => {
          '3405' => [
            [ '251', '278', 'LAGOS DA BEIRA' ] ] },
        'Fiais da Beira' => {
          '3405' => [
            [ '029', '139', 'ERVEDAL OHP' ] ] },
        'Catraia de So Paio' => {
          '3400' => [
            [ '691', '724', 'SO PAIO DE GRAMAOS' ] ] },
        'Lajeosa' => {
          '3405' => [
            [ '301', '337', 'LAJEOSA OHP' ] ] },
        'Lagares da Beira' => {
          '3405' => [
            [ '148', '205', 'LAGARES OHP' ] ] },
        'So Sebastio da Feira' => {
          '3400' => [
            [ '731', '733', 'SO SEBASTIO DA FEIRA' ] ] },
        'Quinta da Madalena' => {
          '3400' => [
            [ '266', '266', 'ALDEIA DAS DEZ' ] ] },
        'Alvoco das Vrzeas' => {
          '3400' => [
            [ '301', '329', 'ALVOCO DAS VRZEAS' ] ] },
        'Digueifel' => {
          '3400' => [
            [ '400', '417', 'LOUROSA OHP' ],
            [ '751', '752', 'VILA POUCA DA BEIRA' ] ] }
               },
      'Tbua' => {
        'Pvoa de Mides' => {
          '3420' => [
            [ '201', '203', 'PVOA DE MIDES' ] ] },
        'Cadoio' => {
          '3420' => [
            [ '131', '149', 'MIDES TBU' ] ] },
        'Carragosela' => {
          '3420' => [
            [ '101', '106', 'ESPARIZ' ] ] },
        'Moita da Serra' => {
          '3420' => [
            [ '034', '035', 'CARAPINHA' ] ] },
        'Lameiras' => {
          '3420' => [
            [ '221', '229', 'SO JOO DA BOA VISTA' ] ] },
        'Arroteia' => {
          '3420' => [
            [ '251', '257', 'SINDE' ] ] },
        'Balocas' => {
          '3420' => [
            [ '051', '070', 'COVAS TBU' ] ] },
        'Azere' => {
          '3420' => [
            [ '011', '014', 'ZERE TBU' ] ] },
        'Olivais' => {
          '3420' => [
            [ '259', '265', 'SINDE' ] ] },
        'Candosa' => {
          '3420' => [
            [ '021', '022', 'CANDOSA' ] ] },
        'Avelar' => {
          '3420' => [
            [ '031', '032', 'CARAPINHA' ] ] },
        'Alvoeira' => {
          '3420' => [
            [ '161', '178', 'MOURONHO' ] ] },
        'Pedras Ferreiras' => {
          '3420' => [
            [ '451', '457', 'VILA NOVA DE OLIVEIRINHA' ] ] },
        'Babau' => {
          '3420' => [
            [ '401', '423', 'TBUA' ] ] },
        'Covelo de Baixo' => {
          '3420' => [
            [ '081', '083', 'COVELO TBU' ] ] },
        'Tbua' => {
          '3420' => [
            [ '301', '342', 'TBUA' ],
            [ '344', '348', 'TBUA' ] ] },
        'Meda de Mouros' => {
          '3420' => [
            [ '121', '121', 'MEDA DE MOUROS' ] ] },
        'Bogalhas' => {
          '3420' => [
            [ '191', '193', 'PINHEIRO DE COJA' ] ] },
        'Nogueira' => {
          '3420' => [
            [ '258', '258', 'ESPARIZ' ] ] }
            },
      'Miranda do Corvo' => {
        'Vale da Proa' => {
          '3220' => [
            [ '441', '444', 'SEMIDE' ] ] },
        'Azenha' => {
          '3220' => [
            [ '011', '019', 'LAMAS MCV' ] ] },
        'Vidual' => {
          '3220' => [
            [ '346', '355', 'RIO VIDE' ] ] },
        'Rio de Vide' => {
          '3220' => [
            [ '336', '336', 'RIO VIDE' ] ] },
        'Senhor da Serra' => {
          '3220' => [
            [ '431', '439', 'SEMIDE' ] ] },
        'Amial' => {
          '3220' => [
            [ '401', '423', 'SEMIDE' ] ] },
        'Vale da Silva' => {
          '3220' => [
            [ '341', '341', 'RIO VIDE' ] ] },
        'Barbens' => {
          '3220' => [
            [ '501', '518', 'VILA NOVA MCV' ] ] },
        'Zorro' => {
          '3220' => [
            [ '531', '537', 'VILA NOVA MCV' ] ] },
        'Pedreira' => {
          '3220' => [
            [ '311', '324', 'RIO VIDE' ] ] },
        'Bairro Novo' => {
          '3220' => [
            [ '031', '072', 'MIRANDA DO CORVO' ] ] },
        'Casal das Cortes' => {
          '3220' => [
            [ '302', '307', 'RIO VIDE' ] ] },
        'Vila Nova' => {
          '3220' => [
            [ '521', '525', 'VILA NOVA MCV' ] ] },
        'Piso' => {
          '3220' => [
            [ '331', '332', 'RIO VIDE' ] ] },
        'Cruz dos Areais' => {
          '3220' => [
            [ '074', '119', 'MIRANDA DO CORVO' ] ] },
        'Moita' => {
          '3220' => [
            [ '231', '245', 'MIRANDA DO CORVO' ] ] },
        'Miranda do Corvo' => {
          '3220' => [
            [ '131', '218', 'MIRANDA DO CORVO' ] ] }
                 },
      'Vila Nova de Poiares' => {
        'Algaa' => {
          '3350' => [
            [ '071', '117', 'VILA NOVA DE POIARES' ] ] },
        'Arrifana' => {
          '3350' => [
            [ '011', '021', 'ARRIFANA PRS' ] ] },
        'Vimieiro' => {
          '3350' => [
            [ '190', '190', 'VILA NOVA DE POIARES' ] ] },
        'Alveite Grande' => {
          '3350' => [
            [ '201', '214', 'SO MIGUEL DE POIARES' ] ] },
        'Vila Nova de Poiares' => {
          '3350' => [
            [ '151', '164', 'VILA NOVA DE POIARES' ] ] },
        'Barreiro' => {
          '3350' => [
            [ '051', '057', 'LAVEGADAS' ] ] },
        'Santa Maria' => {
          '3350' => [
            [ '023', '030', 'ARRIFANA PRS' ] ] }
               },
      'Mira' => {
        'Lagoa' => {
          '3070' => [
            [ '141', '191', 'MIRA' ] ] },
        'Lentisqueira' => {
          '3070' => [
            [ '231', '253', 'MIRA' ] ] },
        'Carromeu' => {
          '3070' => [
            [ '020', '033', 'MIRA' ],
            [ '548', '586', 'MIRA' ] ] },
        'Casal de So Tom' => {
          '3070' => [
            [ '041', '056', 'MIRA' ] ] },
        'Cabeo' => {
          '3070' => [
            [ '451', '466', 'MIRA' ] ] },
        'Seixo' => {
          '3070' => [
            [ '521', '547', 'SEIXO' ] ] },
        'Portomar' => {
          '3070' => [
            [ '351', '376', 'MIRA' ] ] },
        'Cavadas' => {
          '3070' => [
            [ '061', '073', 'MIRA' ] ] },
        'Ramalheiro' => {
          '3070' => [
            [ '401', '417', 'MIRA' ] ] },
        'Arneiro' => {
          '3070' => [
            [ '011', '012', 'MIRA' ] ] },
        'Barra' => {
          '3070' => [
            [ '701', '702', 'PRAIA DE MIRA' ] ] },
        'Carapelhos' => {
          '3070' => [
            [ '601', '617', 'CARAPELHOS' ] ] },
        'Cential' => {
          '3070' => [
            [ '081', '087', 'MIRA' ] ] },
        'Ermida' => {
          '3070' => [
            [ '121', '138', 'MIRA' ] ] },
        'Valeirinha' => {
          '3070' => [
            [ '431', '437', 'MIRA' ] ] },
        'Praia de Mira' => {
          '3070' => [
            [ '721', '731', 'PRAIA DE MIRA' ],
            [ '741', '746', 'PRAIA DE MIRA' ],
            [ '751', '753', 'PRAIA DE MIRA' ],
            [ '761', '766', 'PRAIA DE MIRA' ],
            [ '791', '792', 'PRAIA DE MIRA' ],
            [ '794', '825', 'PRAIA DE MIRA' ] ] },
        'Cabeas Verdes' => {
          '3070' => [
            [ '501', '513', 'SEIXO' ] ] },
        'Presa' => {
          '3070' => [
            [ '381', '394', 'MIRA' ] ] },
        'Colmeal' => {
          '3070' => [
            [ '091', '096', 'MIRA' ],
            [ '101', '113', 'MIRA' ] ] },
        'Corticeiro de Baixo' => {
          '3070' => [
            [ '631', '638', 'CARAPELHOS' ] ] },
        'Leites' => {
          '3070' => [
            [ '201', '217', 'MIRA' ] ] },
        'Mira' => {
          '3070' => [
            [ '301', '318', 'MIRA' ],
            [ '320', '337', 'MIRA' ],
            [ '399', '399', 'MIRA' ] ] }
           }
           },
    'Braga' => {
      'Terras de Bouro' => {
        'Pedrogo' => {
          '4845' => [
            [ '075', '078', 'GERS' ] ] },
        'So Pedro' => {
          '4845' => [
            [ '027', '028', 'RIO CALDO' ] ] },
        'Campo' => {
          '4840' => [
            [ '120', '120', 'RIBEIRA TBR' ] ] },
        'Lagoa' => {
          '4840' => [
            [ '050', '050', 'CHAMOIM' ] ] },
        'Caldas do Gers' => {
          '4845' => [
            [ '067', '067', 'GERS' ] ] },
        'Pardieiro' => {
          '4840' => [
            [ '130', '130', 'SOUTO TBR' ] ] },
        'Casal' => {
          '4840' => [
            [ '060', '060', 'CHORENSE' ] ] },
        'Lij' => {
          '4845' => [
            [ '010', '025', 'RIO CALDO' ] ] },
        'Campos Abades' => {
          '4840' => [
            [ '110', '110', 'MONTE TBR' ] ] },
        'Admeus' => {
          '4845' => [
            [ '061', '065', 'GERS' ] ] },
        'Lama' => {
          '4840' => [
            [ '070', '070', 'CIBES' ] ] },
        'Vidoeiro' => {
          '4845' => [
            [ '081', '081', 'GERS' ] ] },
        'Cruzes' => {
          '4840' => [
            [ '140', '140', 'VILAR TBR' ] ] },
        'Cabaninhas' => {
          '4840' => [
            [ '090', '090', 'GONDORIZ TBR' ] ] },
        'So Pantaleo' => {
          '4840' => [
            [ '010', '010', 'BALANA' ] ] },
        'Brufe' => {
          '4840' => [
            [ '020', '020', 'BRUFE TBR' ] ] },
        'Carvalhal' => {
          '4840' => [
            [ '030', '030', 'CAMPO DO GERS' ] ] },
        'Cruzeiro' => {
          '4840' => [
            [ '080', '080', 'COVIDE' ] ] },
        'Ermida' => {
          '4845' => [
            [ '072', '072', 'GERS' ] ] },
        'Ervedeiros' => {
          '4840' => [
            [ '040', '040', 'CARVALHEIRA' ] ] },
        'Ch-Ermida' => {
          '4845' => [
            [ '070', '070', 'GERS' ] ] },
        'Paradela' => {
          '4845' => [
            [ '043', '045', 'VALDOSENDE' ] ] },
        'Corredoura' => {
          '4840' => [
            [ '100', '100', 'TERRAS DE BOURO' ] ] },
        'Assento' => {
          '4845' => [
            [ '040', '041', 'VALDOSENDE' ] ] }
              },
      'Braga' => {
        'Fulo' => {
          '4705' => [
            [ '627', '627', 'SO PEDRO OLIVEIRA' ] ] },
        'Palmeira' => {
          '4700' => [
            [ '475', '481', 'PALMEIRA BRG' ],
            [ '675', '752', 'PALMEIRA BRG' ] ] },
        'Celeirs' => {
          '4705' => [
            [ '391', '442', 'CELEIRS BRG' ],
            [ '444', '471', 'CELEIRS BRG' ] ],
          '4709' => [
            [ '001', '001', 'CELEIRS BRG' ] ] },
        'Morreira' => {
          '4705' => [
            [ '481', '553', 'MORREIRA' ] ] },
        'Boucinha' => {
          '4700' => [
            [ '760', '760', 'PANOIAS BRG' ] ] },
        'Monte' => {
          '4700' => [
            [ '670', '670', 'PADIM DA GRAA' ] ] },
        'Estrada' => {
          '4710' => [
            [ '670', '670', 'NAVARRA' ] ] },
        'Agrelos' => {
          '4705' => [
            [ '474', '476', 'ESPORES' ] ] },
        'Lage' => {
          '4705' => [
            [ '479', '479', 'GUISANDE BRG' ] ] },
        'Regadas' => {
          '4710' => [
            [ '750', '750', 'SANTA LUCRCIA DE ALGERIZ' ] ] },
        'Aveleira' => {
          '4710' => [
            [ '611', '627', 'CRESPOS' ] ] },
        'Vau' => {
          '4705' => [
            [ '629', '629', 'SEQUEIRA' ] ] },
        'Trandeiras' => {
          '4705' => [
            [ '631', '633', 'TRANDEIRAS' ],
            [ '635', '650', 'TRANDEIRAS' ] ] },
        'Capela' => {
          '4705' => [
            [ '368', '390', 'CABREIROS BRG' ] ] },
        'Pinheiro' => {
          '4705' => [
            [ '628', '628', 'SO VICENTE PENSO' ] ] },
        'Melhorado de Baixo' => {
          '4700' => [
            [ '565', '565', 'MIRE DE TIBES' ] ] },
        'Rulhe' => {
          '4705' => [
            [ '579', '588', 'RULHE' ],
            [ '590', '591', 'RULHE' ],
            [ '593', '615', 'RULHE' ],
            [ '617', '626', 'RULHE' ],
            [ '712', '715', 'RULHE' ] ],
          '4709' => [
            [ '007', '008', 'RULHE' ] ] },
        'Arentim' => {
          '4705' => [
            [ '001', '024', 'ARENTIM' ] ] },
        'Regueira' => {
          '4705' => [
            [ '472', '472', 'CUNHA BRG' ] ] },
        'Bacelares' => {
          '4715' => [
            [ '534', '557', 'SOBREPOSTA' ] ] },
        'Barral' => {
          '4710' => [
            [ '629', '647', 'CRESPOS' ] ] },
        'Vimieiro' => {
          '4705' => [
            [ '652', '670', 'VIMIEIRO BRG' ] ] },
        'Aveleda' => {
          '4705' => [
            [ '025', '067', 'AVELEDA BRG' ],
            [ '070', '073', 'AVELEDA BRG' ],
            [ '075', '075', 'AVELEDA BRG' ],
            [ '716', '716', 'AVELEDA BRG' ] ],
          '4709' => [
            [ '005', '006', 'AVELEDA BRG' ] ] },
        'So Paio Merelim' => {
          '4700' => [
            [ '194', '194', 'SO PAIO MERELIM' ],
            [ '833', '834', 'SO PAIO MERELIM' ],
            [ '836', '839', 'SO PAIO MERELIM' ],
            [ '841', '845', 'SO PAIO MERELIM' ],
            [ '876', '877', 'SO PAIO MERELIM' ] ] },
        'Tadim' => {
          '4705' => [
            [ '671', '671', 'TADIM' ] ] },
        'Vilaa' => {
          '4705' => [
            [ '651', '651', 'VILAA' ] ] },
        'Alm do Rio' => {
          '4715' => [
            [ '454', '475', 'PEDRALVA' ] ] },
        'Fradelos' => {
          '4705' => [
            [ '478', '478', 'FRADELOS BRG' ],
            [ '673', '686', 'FRADELOS BRG' ],
            [ '688', '706', 'FRADELOS BRG' ] ] },
        'Recho' => {
          '4710' => [
            [ '730', '730', 'POUSADA BRG' ] ] },
        'Pedreira' => {
          '4700' => [
            [ '770', '770', 'PASSOS (SO JULIO)' ] ] },
        'Cambas' => {
          '4715' => [
            [ '476', '533', 'SO MAMEDE ESTE' ] ] },
        'Tebosa' => {
          '4705' => [
            [ '630', '630', 'TEBOSA' ] ] },
        'Braga' => {
          '4704' => [
            [ '501', '502', 'BRAGA' ],
            [ '504', '506', 'BRAGA' ],
            [ '508', '515', 'BRAGA' ],
            [ '518', '520', 'BRAGA' ],
            [ '523', '527', 'BRAGA' ],
            [ '530', '538', 'BRAGA' ],
            [ '553', '553', 'BRAGA' ],
            [ '575', '575', 'BRAGA' ] ],
          '4719' => [
            [ '001', '004', 'BRAGA' ] ],
          '4710' => [
            [ '002', '002', 'BRAGA' ],
            [ '041', '064', 'BRAGA' ],
            [ '066', '090', 'BRAGA' ],
            [ '092', '093', 'BRAGA' ],
            [ '228', '230', 'BRAGA' ],
            [ '232', '232', 'BRAGA' ],
            [ '234', '235', 'BRAGA' ],
            [ '237', '237', 'BRAGA' ],
            [ '242', '242', 'BRAGA' ],
            [ '244', '245', 'BRAGA' ],
            [ '249', '251', 'BRAGA' ],
            [ '292', '292', 'BRAGA' ],
            [ '294', '312', 'BRAGA' ],
            [ '316', '316', 'BRAGA' ],
            [ '330', '333', 'BRAGA' ],
            [ '336', '341', 'BRAGA' ],
            [ '345', '351', 'BRAGA' ],
            [ '353', '354', 'BRAGA' ],
            [ '356', '356', 'BRAGA' ],
            [ '358', '362', 'BRAGA' ],
            [ '366', '367', 'BRAGA' ],
            [ '369', '378', 'BRAGA' ],
            [ '380', '386', 'BRAGA' ],
            [ '392', '392', 'BRAGA' ],
            [ '394', '394', 'BRAGA' ],
            [ '400', '401', 'BRAGA' ],
            [ '403', '407', 'BRAGA' ],
            [ '409', '417', 'BRAGA' ],
            [ '419', '420', 'BRAGA' ],
            [ '422', '422', 'BRAGA' ],
            [ '425', '429', 'BRAGA' ],
            [ '435', '443', 'BRAGA' ],
            [ '445', '446', 'BRAGA' ],
            [ '448', '450', 'BRAGA' ],
            [ '452', '453', 'BRAGA' ],
            [ '486', '497', 'BRAGA' ],
            [ '499', '500', 'BRAGA' ],
            [ '503', '506', 'BRAGA' ],
            [ '913', '914', 'BRAGA' ],
            [ '922', '926', 'BRAGA' ] ],
          '4714' => [
            [ '501', '502', 'BRAGA' ] ],
          '4715' => [
            [ '001', '057', 'BRAGA' ],
            [ '060', '118', 'BRAGA' ],
            [ '121', '203', 'BRAGA' ],
            [ '206', '310', 'BRAGA' ],
            [ '312', '381', 'BRAGA' ],
            [ '384', '453', 'BRAGA' ],
            [ '558', '561', 'BRAGA' ] ],
          '4700' => [
            [ '001', '001', 'BRAGA' ],
            [ '003', '006', 'BRAGA' ],
            [ '024', '099', 'BRAGA' ],
            [ '112', '112', 'BRAGA' ],
            [ '114', '114', 'BRAGA' ],
            [ '116', '116', 'BRAGA' ],
            [ '120', '121', 'BRAGA' ],
            [ '123', '124', 'BRAGA' ],
            [ '134', '134', 'BRAGA' ],
            [ '137', '137', 'BRAGA' ],
            [ '143', '143', 'BRAGA' ],
            [ '150', '155', 'BRAGA' ],
            [ '183', '183', 'BRAGA' ],
            [ '203', '212', 'BRAGA' ],
            [ '214', '224', 'BRAGA' ],
            [ '226', '228', 'BRAGA' ],
            [ '230', '230', 'BRAGA' ],
            [ '232', '239', 'BRAGA' ],
            [ '241', '241', 'BRAGA' ],
            [ '244', '284', 'BRAGA' ],
            [ '286', '292', 'BRAGA' ],
            [ '294', '298', 'BRAGA' ],
            [ '300', '330', 'BRAGA' ],
            [ '345', '345', 'BRAGA' ],
            [ '352', '352', 'BRAGA' ],
            [ '355', '385', 'BRAGA' ],
            [ '387', '389', 'BRAGA' ],
            [ '391', '404', 'BRAGA' ],
            [ '407', '440', 'BRAGA' ],
            [ '442', '448', 'BRAGA' ],
            [ '482', '487', 'BRAGA' ],
            [ '915', '915', 'BRAGA' ],
            [ '920', '921', 'BRAGA' ],
            [ '924', '924', 'BRAGA' ] ],
          '4705' => [
            [ '076', '109', 'BRAGA' ],
            [ '111', '136', 'BRAGA' ],
            [ '138', '166', 'BRAGA' ],
            [ '169', '223', 'BRAGA' ],
            [ '226', '252', 'BRAGA' ],
            [ '254', '263', 'BRAGA' ],
            [ '265', '367', 'BRAGA' ],
            [ '672', '672', 'BRAGA' ],
            [ '707', '711', 'BRAGA' ] ],
          '4709' => [
            [ '002', '004', 'BRAGA' ],
            [ '009', '009', 'BRAGA' ] ] },
        'So Pedro Merelim' => {
          '4704' => [
            [ '547', '547', 'SO PEDRO MERELIM' ] ],
          '4700' => [
            [ '846', '847', 'SO PEDRO MERELIM' ],
            [ '850', '871', 'SO PEDRO MERELIM' ] ] },
        'Esmontada' => {
          '4705' => [
            [ '554', '554', 'PENSO (SANTO ESTVO)' ] ] },
        'Pedroso' => {
          '4700' => [
            [ '765', '765', 'PARADA DE TIBES' ] ] },
        'Agrinhas' => {
          '4705' => [
            [ '555', '578', 'PRISCOS' ] ] },
        'Tojal' => {
          '4705' => [
            [ '473', '473', 'ESCUDEIROS' ] ] },
        'Almoinha' => {
          '4705' => [
            [ '480', '480', 'LAMAS BRG' ] ] },
        'Casal da Vila' => {
          '4705' => [
            [ '477', '477', 'FIGUEIREDO BRG' ] ] },
        'Assento' => {
          '4710' => [
            [ '551', '585', 'ADAFE' ] ] }
          },
      'Vizela' => {
        'So Joo' => {
          '4815' => [
            [ '428', '455', 'VIZELA' ],
            [ '564', '568', 'VIZELA' ] ] },
        'So Miguel' => {
          '4815' => [
            [ '393', '426', 'VIZELA' ],
            [ '471', '552', 'VIZELA' ],
            [ '645', '693', 'VIZELA' ] ] },
        'Tagilde' => {
          '4815' => [
            [ '589', '601', 'TAGILDE' ],
            [ '603', '620', 'TAGILDE' ],
            [ '622', '630', 'TAGILDE' ],
            [ '632', '632', 'TAGILDE' ] ] },
        'So Paio Vizela' => {
          '4815' => [
            [ '572', '588', 'VIZELA' ],
            [ '633', '640', 'VIZELA' ] ] },
        'nfias' => {
          '4815' => [
            [ '071', '110', 'NFIAS VIZ' ],
            [ '380', '390', 'NFIAS VIZ' ],
            [ '457', '470', 'NFIAS VIZ' ],
            [ '553', '563', 'NFIAS VIZ' ] ] },
        'Santa Eullia' => {
          '4620' => [
            [ '521', '522', 'SANTA EULLIA VIZ' ],
            [ '524', '527', 'SANTA EULLIA VIZ' ],
            [ '529', '529', 'SANTA EULLIA VIZ' ],
            [ '531', '532', 'SANTA EULLIA VIZ' ],
            [ '534', '535', 'SANTA EULLIA VIZ' ],
            [ '537', '537', 'SANTA EULLIA VIZ' ],
            [ '539', '539', 'SANTA EULLIA VIZ' ],
            [ '541', '542', 'SANTA EULLIA VIZ' ],
            [ '544', '545', 'SANTA EULLIA VIZ' ],
            [ '547', '548', 'SANTA EULLIA VIZ' ],
            [ '550', '550', 'SANTA EULLIA VIZ' ],
            [ '552', '553', 'SANTA EULLIA VIZ' ],
            [ '555', '565', 'SANTA EULLIA VIZ' ],
            [ '567', '567', 'SANTA EULLIA VIZ' ],
            [ '569', '570', 'SANTA EULLIA VIZ' ],
            [ '573', '573', 'SANTA EULLIA VIZ' ],
            [ '575', '576', 'SANTA EULLIA VIZ' ],
            [ '578', '582', 'SANTA EULLIA VIZ' ],
            [ '587', '587', 'SANTA EULLIA VIZ' ],
            [ '589', '590', 'SANTA EULLIA VIZ' ],
            [ '592', '592', 'SANTA EULLIA VIZ' ],
            [ '595', '599', 'SANTA EULLIA VIZ' ],
            [ '601', '602', 'SANTA EULLIA VIZ' ] ] }
           },
      'Vila Verde' => {
        'Borrela' => {
          '4730' => [
            [ '400', '400', 'PONTE VVD' ] ] },
        'Escola' => {
          '4730' => [
            [ '620', '620', 'VALES' ] ] },
        'Bairro de Pedome' => {
          '4730' => [
            [ '701', '797', 'VILA VERDE' ] ] },
        'Casal' => {
          '4730' => [
            [ '170', '172', 'FREIRIZ' ] ] },
        'Breia' => {
          '4730' => [
            [ '490', '492', 'SANTIAGO CARREIRAS' ] ] },
        'Cruz' => {
          '4730' => [
            [ '370', '370', 'PENASCAIS' ],
            [ '410', '410', 'PORTELA DAS CABRAS' ] ] },
        'Encontro' => {
          '4730' => [
            [ '650', '650', 'VILARINHO VVD' ] ] },
        'Boucinha' => {
          '4730' => [
            [ '310', '311', 'NEVOGILDE VVD' ] ] },
        'Boua' => {
          '4730' => [
            [ '062', '075', 'BARBUDO' ] ] },
        'Ramalha' => {
          '4730' => [
            [ '475', '479', 'PRADO (SANTA MARIA)' ] ] },
        'Cachadas' => {
          '4730' => [
            [ '120', '121', 'COUCIEIRO' ] ] },
        'Fontes' => {
          '4730' => [
            [ '020', '020', 'ARCOZELO VVD' ] ] },
        'Crasto' => {
          '4730' => [
            [ '040', '041', 'ATIES' ] ] },
        'Estrada' => {
          '4730' => [
            [ '360', '360', 'PEDREGAIS' ],
            [ '459', '473', 'PRADO (SANTA MARIA)' ] ] },
        'Corujeira' => {
          '4730' => [
            [ '150', '156', 'DUAS IGREJAS VVD' ] ] },
        'Forca da Lomba' => {
          '4730' => [
            [ '390', '394', 'PICO DE REGALADOS' ] ] },
        'Cruzeiro' => {
          '4730' => [
            [ '050', '050', 'AZES' ] ] },
        'Agrlo' => {
          '4730' => [
            [ '430', '430', 'SABARIZ' ] ] },
        'Costa' => {
          '4730' => [
            [ '220', '220', 'GONDIES VVD' ] ] },
        'Eiros' => {
          '4730' => [
            [ '110', '110', 'CODECEDA' ] ] },
        'Cereje' => {
          '4730' => [
            [ '520', '522', 'SO MARTINHO VALBOM' ] ] },
        'Fares' => {
          '4730' => [
            [ '380', '382', 'PICO' ] ] },
        'Couto' => {
          '4730' => [
            [ '500', '502', 'SO MAMEDE ESCARIZ' ] ] },
        'Ciso' => {
          '4730' => [
            [ '080', '080', 'BARROS' ] ] },
        'Sande' => {
          '4730' => [
            [ '585', '585', 'TRAVASSS VVD' ] ] },
        'Devesa' => {
          '4730' => [
            [ '210', '210', 'GOMIDE' ] ] },
        'Cernadas' => {
          '4730' => [
            [ '130', '133', 'COVAS VVD' ] ] },
        'Bairro' => {
          '4730' => [
            [ '530', '532', 'SO MIGUEL CARREIRAS' ] ] },
        'Esperigo' => {
          '4730' => [
            [ '140', '143', 'DOSSOS' ] ] },
        'Colheita' => {
          '4730' => [
            [ '550', '555', 'PRADO (SO MIGUEL)' ] ] },
        'Burrelho' => {
          '4730' => [
            [ '200', '203', 'GODINHAOS' ] ] },
        'Bargo' => {
          '4730' => [
            [ '240', '249', 'LAGE' ] ] },
        'Vila Pouca' => {
          '4730' => [
            [ '385', '385', 'PICO' ] ] },
        'Forcadela' => {
          '4730' => [
            [ '420', '428', 'RIO MAU VVD' ] ] },
        'Casais' => {
          '4730' => [
            [ '230', '230', 'GONDOMAR VVD' ] ] },
        'Carval' => {
          '4730' => [
            [ '610', '614', 'VALDREU' ] ] },
        'Cedofeita' => {
          '4730' => [
            [ '190', '193', 'GOES VVD' ] ] },
        'Cruzes' => {
          '4730' => [
            [ '510', '510', 'SO MARTINHO ESCARIZ' ] ] },
        'Bacelo' => {
          '4730' => [
            [ '010', '014', 'ABOIM DA NBREGA' ] ] },
        'Barroso' => {
          '4730' => [
            [ '350', '350', 'PASS' ] ] },
        'Covelo' => {
          '4730' => [
            [ '270', '274', 'LOUREIRA' ] ] },
        'Boavista' => {
          '4730' => [
            [ '540', '540', 'ORIZ (SO MIGUEL)' ] ] },
        'Barreirinho' => {
          '4730' => [
            [ '480', '483', 'ORIZ (SANTA MARINHA)' ] ] },
        'Fonte' => {
          '4730' => [
            [ '320', '329', 'OLEIROS VVD' ],
            [ '570', '584', 'SOUTELO VVD' ] ] },
        'Lajes' => {
          '4730' => [
            [ '262', '262', 'LANHAS' ] ] },
        'Eira Vedra' => {
          '4730' => [
            [ '340', '343', 'PARADA DE GATIM' ] ] },
        'Aldeia' => {
          '4730' => [
            [ '090', '096', 'CABANELAS VVD' ],
            [ '160', '163', 'ESQUEIROS' ],
            [ '180', '181', 'GEME' ],
            [ '590', '598', 'TURIZ' ] ] },
        'Carvalhido' => {
          '4730' => [
            [ '560', '560', 'SO PEDRO VALBOM' ] ] },
        'Costeira' => {
          '4730' => [
            [ '440', '442', 'SANDE VVD' ] ] },
        'Cobrosa' => {
          '4730' => [
            [ '100', '105', 'CERVES' ] ] },
        'Ch' => {
          '4730' => [
            [ '030', '031', 'ATES VVD' ] ] },
        'Vila' => {
          '4730' => [
            [ '450', '457', 'PRADO (SANTA MARIA)' ] ] },
        'Forninho' => {
          '4730' => [
            [ '280', '283', 'MARRANCOS' ] ] },
        'Cristelo' => {
          '4730' => [
            [ '290', '290', 'MS VVD' ] ] },
        'Barreiro' => {
          '4730' => [
            [ '260', '260', 'LANHAS' ] ] },
        'Cmara' => {
          '4730' => [
            [ '300', '306', 'MOURE VVD' ] ] }
               },
      'Cabeceiras de Basto' => {
        'Entrevais' => {
          '4860' => [
            [ '271', '299', 'PEDRAA' ] ] },
        'gua Redonda' => {
          '4860' => [
            [ '311', '338', 'CABECEIRAS DE BASTO' ] ] },
        'Alm do Ribeiro' => {
          '4860' => [
            [ '151', '179', 'CAVEZ' ] ] },
        'Alm do Rio' => {
          '4860' => [
            [ '121', '126', 'BUCOS' ],
            [ '241', '249', 'PAINZELA' ] ] },
        'Alto' => {
          '4860' => [
            [ '101', '111', 'BASTO' ] ] },
        'Arrabalde' => {
          '4860' => [
            [ '061', '083', 'ARCO DE BALHE' ] ] },
        'Abadim' => {
          '4860' => [
            [ '011', '016', 'ABADIM' ] ] },
        'Gondies' => {
          '4860' => [
            [ '220', '222', 'GONDIES CBC' ] ] },
        'Fundd de Vila' => {
          '4860' => [
            [ '261', '264', 'PASSOS CBC' ] ] },
        'Cunhas' => {
          '4860' => [
            [ '481', '483', 'VILAR DE CUNHAS' ] ] },
        'Cabo' => {
          '4860' => [
            [ '021', '031', 'ALVITE CBC' ] ] },
        'Ervideiro' => {
          '4860' => [
            [ '231', '235', 'OUTEIRO CBC' ] ] },
        'Pinheiro' => {
          '4860' => [
            [ '350', '369', 'CABECEIRAS DE BASTO' ] ] },
        'Pinheiro de Cima' => {
          '4860' => [
            [ '401', '414', 'CABECEIRAS DE BASTO' ] ] },
        'Levandeira' => {
          '4860' => [
            [ '200', '212', 'FAIA CBC' ] ] },
        'Asnela' => {
          '4860' => [
            [ '421', '434', 'RIO DOURO' ] ] },
        'Alto Branco' => {
          '4860' => [
            [ '451', '462', 'VILA NUNE' ] ] },
        'Bouas' => {
          '4860' => [
            [ '131', '144', 'CABECEIRAS DE BASTO' ] ] },
        'Arco de Balhe' => {
          '4860' => [
            [ '041', '046', 'ARCO DE BALHE' ] ] }
            },
      'Amares' => {
        'Picoto' => {
          '4720' => [
            [ '628', '628', 'RENDUFE AMR' ] ] },
        'Areal' => {
          '4720' => [
            [ '132', '136', 'BESTEIROS AMR' ] ] },
        'Anjo da Guarda' => {
          '4720' => [
            [ '059', '059', 'PROZELO AMR' ] ] },
        'Caneiro' => {
          '4720' => [
            [ '064', '067', 'GOES AMR' ] ] },
        'Ferreiros' => {
          '4720' => [
            [ '068', '094', 'FERREIROS AMR' ],
            [ '337', '353', 'FERREIROS AMR' ],
            [ '393', '393', 'FERREIROS AMR' ] ] },
        'Amorim' => {
          '4720' => [
            [ '653', '653', 'BOURO (SANTA MARTA)' ] ] },
        'Aguio' => {
          '4720' => [
            [ '581', '590', 'PORTELA AMR' ] ] },
        'Carrascal' => {
          '4720' => [
            [ '098', '099', 'BOURO (SANTA MARIA)' ] ] },
        'Amares' => {
          '4720' => [
            [ '011', '011', 'AMARES' ],
            [ '013', '022', 'AMARES' ],
            [ '058', '058', 'AMARES' ],
            [ '113', '118', 'AMARES' ],
            [ '785', '786', 'AMARES' ],
            [ '788', '795', 'AMARES' ] ] },
        'Cales' => {
          '4720' => [
            [ '096', '096', 'BOURO (SANTA MARIA)' ],
            [ '411', '412', 'FIGUEIREDO AMR' ] ] },
        'Igreja' => {
          '4720' => [
            [ '062', '063', 'PAREDES SECAS' ],
            [ '371', '374', 'FERREIROS AMR' ] ] },
        'Cancela da Cruz' => {
          '4720' => [
            [ '051', '051', 'AMARES' ] ] },
        'Boua' => {
          '4720' => [
            [ '161', '171', 'BICO AMR' ] ] },
        'Granja' => {
          '4720' => [
            [ '054', '054', 'AMARES' ] ] },
        'Chos' => {
          '4720' => [
            [ '414', '429', 'FIGUEIREDO AMR' ] ] },
        'Rio Bom' => {
          '4720' => [
            [ '385', '385', 'FERREIROS AMR' ] ] },
        'Chouselas' => {
          '4720' => [
            [ '001', '002', 'VILELA AMR' ] ] },
        'Bairro de Santa Maria' => {
          '4720' => [
            [ '004', '005', 'TORRE AMR' ] ] },
        'Cruzeiro' => {
          '4720' => [
            [ '604', '613', 'PROZELO AMR' ] ] },
        'Atade' => {
          '4720' => [
            [ '007', '009', 'RENDUFE AMR' ] ] },
        'Vasconcelos' => {
          '4720' => [
            [ '390', '390', 'FERREIROS AMR' ] ] },
        'Cerdeirinhas' => {
          '4720' => [
            [ '095', '095', 'CARRAZEDO AMR' ],
            [ '335', '335', 'FERREIROS AMR' ] ] },
        'Alm' => {
          '4720' => [
            [ '101', '112', 'BARREIROS AMR' ],
            [ '142', '143', 'BESTEIROS AMR' ],
            [ '281', '291', 'CARRAZEDO AMR' ],
            [ '331', '333', 'FERREIROS AMR' ],
            [ '551', '557', 'PARANHOS AMR' ] ] },
        'Souto' => {
          '4720' => [
            [ '140', '140', 'BESTEIROS AMR' ] ] },
        'Aspa' => {
          '4720' => [
            [ '441', '462', 'FISCAL' ] ] },
        'Barral' => {
          '4720' => [
            [ '244', '268', 'CALDELAS' ] ] },
        'Ancede' => {
          '4720' => [
            [ '010', '010', 'PROZELO AMR' ] ] },
        'Felgueiras' => {
          '4720' => [
            [ '663', '663', 'BOURO (SANTA MARTA)' ] ] },
        'Terreiro' => {
          '4720' => [
            [ '633', '639', 'BOURO (SANTA MARIA)' ] ] },
        'Carvalho' => {
          '4720' => [
            [ '571', '574', 'PAREDES SECAS' ] ] },
        'Redondelo' => {
          '4720' => [
            [ '138', '138', 'BESTEIROS AMR' ] ] },
        'Veiga de Pena' => {
          '4720' => [
            [ '196', '200', 'CAIRES' ] ] },
        'Barrio' => {
          '4720' => [
            [ '701', '712', 'SEQUEIROS AMR' ] ] },
        'Penas' => {
          '4720' => [
            [ '214', '221', 'CAIRES' ] ] },
        'Grova' => {
          '4720' => [
            [ '650', '650', 'BOURO (SANTA MARTA)' ] ] },
        'Castanheiro' => {
          '4720' => [
            [ '656', '659', 'BOURO (SANTA MARTA)' ] ] },
        'Pites' => {
          '4720' => [
            [ '006', '006', 'SEQUEIROS AMR' ] ] },
        'Castanheira' => {
          '4720' => [
            [ '003', '003', 'SERAMIL' ] ] },
        'Via Cova' => {
          '4720' => [
            [ '576', '576', 'PAREDES SECAS' ] ] },
        'Barbades' => {
          '4720' => [
            [ '301', '319', 'DORNELAS AMR' ] ] },
        'Torre' => {
          '4720' => [
            [ '691', '692', 'BOURO (SANTA MARTA)' ] ] },
        'Boucinhas' => {
          '4720' => [
            [ '481', '504', 'GOES AMR' ] ] },
        'Pereira' => {
          '4720' => [
            [ '681', '686', 'BOURO (SANTA MARTA)' ] ] },
        'Junsabes' => {
          '4720' => [
            [ '097', '097', 'CALDELAS' ] ] },
        'Monte Cho' => {
          '4720' => [
            [ '673', '675', 'BOURO (SANTA MARTA)' ] ] },
        'Serto' => {
          '4720' => [
            [ '388', '388', 'FERREIROS AMR' ] ] },
        'Carvalh' => {
          '4720' => [
            [ '100', '100', 'BESTEIROS AMR' ] ] },
        'Fonte' => {
          '4720' => [
            [ '665', '670', 'BOURO (SANTA MARTA)' ] ] },
        'Outeiro' => {
          '4720' => [
            [ '376', '377', 'FERREIROS AMR' ],
            [ '678', '679', 'BOURO (SANTA MARTA)' ] ] },
        'Tomada' => {
          '4720' => [
            [ '693', '698', 'BOURO (SANTA MARIA)' ] ] },
        'Olheiros' => {
          '4720' => [
            [ '631', '632', 'RENDUFE AMR' ] ] },
        'Aldeia' => {
          '4720' => [
            [ '601', '602', 'PROZELO AMR' ],
            [ '741', '756', 'TORRE AMR' ] ] },
        'Bico' => {
          '4720' => [
            [ '521', '536', 'LAGO' ] ] },
        'Tojal' => {
          '4720' => [
            [ '714', '715', 'SEQUEIROS AMR' ] ] },
        'Dornas' => {
          '4720' => [
            [ '641', '649', 'BOURO (SANTA MARIA)' ] ] },
        'Bouas' => {
          '4720' => [
            [ '621', '626', 'RENDUFE AMR' ] ] },
        'Assento' => {
          '4720' => [
            [ '721', '730', 'SERAMIL' ],
            [ '771', '783', 'VILELA AMR' ] ] },
        'Cal' => {
          '4720' => [
            [ '202', '212', 'CAIRES' ] ] },
        'Agrinho' => {
          '4720' => [
            [ '241', '242', 'CALDELAS' ] ] }
           },
      'Barcelos' => {
        'Aldeia de Cima' => {
          '4775' => [
            [ '110', '126', 'GRIMANCELOS' ],
            [ '190', '209', 'NEGREIROS' ] ] },
        'Beiro' => {
          '4905' => [
            [ '088', '115', 'FRAGOSO' ] ] },
        'Silgueiros' => {
          '4755' => [
            [ '401', '404', 'PEREIRA BCL' ] ] },
        'Linhares' => {
          '4750' => [
            [ '481', '494', 'GALEGOS (SO MARTINHO)' ] ] },
        'Casal' => {
          '4750' => [
            [ '621', '626', 'PERELHAL' ] ] },
        'Abelheiros' => {
          '4755' => [
            [ '171', '186', 'CRISTELO BCL' ] ] },
        'Boucinha' => {
          '4755' => [
            [ '501', '512', 'SEQUEADE' ] ] },
        'Boua' => {
          '4750' => [
            [ '761', '774', 'UCHA' ] ],
          '4755' => [
            [ '091', '096', 'BASTUO (SO JOO)' ] ] },
        'Gandra' => {
          '4750' => [
            [ '551', '564', 'MANHENTE' ] ] },
        'Rua Nova' => {
          '4775' => [
            [ '269', '272', 'VIATODOS' ] ] },
        'Monte' => {
          '4750' => [
            [ '531', '544', 'LIJ' ] ],
          '4755' => [
            [ '011', '016', 'AIR' ] ] },
        'Carvalhinho' => {
          '4755' => [
            [ '191', '195', 'ENCOURADOS' ] ] },
        'Estrada' => {
          '4750' => [
            [ '008', '009', 'ABADE DE NEIVA' ] ] },
        'Lage' => {
          '4750' => [
            [ '711', '715', 'TAMEL (SO PEDRO FINS)' ] ] },
        'Poa' => {
          '4775' => [
            [ '101', '102', 'FONTE COBERTA' ] ] },
        'Regadas' => {
          '4755' => [
            [ '001', '006', 'ADES' ] ] },
        'Loteamento de Santo Antnio' => {
          '4775' => [
            [ '261', '263', 'VIATODOS' ] ] },
        'Cruzeiro' => {
          '4750' => [
            [ '701', '706', 'TAMEL (SANTA LEOCDIA)' ] ] },
        'Pinheiro' => {
          '4755' => [
            [ '461', '467', 'RIO COVO (SANTA EUGNIA)' ] ] },
        'Aldeia Nova' => {
          '4775' => [
            [ '220', '236', 'SILVEIROS' ],
            [ '250', '259', 'VIATODOS' ] ] },
        'Coturela' => {
          '4750' => [
            [ '071', '073', 'ALVITO (SO MARTINHO)' ] ] },
        'Agra' => {
          '4775' => [
            [ '170', '176', 'MONTE DE FRALES' ] ] },
        'Terreiro' => {
          '4750' => [
            [ '571', '572', 'MARIZ' ] ] },
        'Devesa' => {
          '4775' => [
            [ '059', '068', 'CARREIRA BCL' ] ] },
        'Vale' => {
          '4755' => [
            [ '571', '578', 'VILAR DE FIGOS' ] ] },
        'Molarido' => {
          '4750' => [
            [ '421', '424', 'CREIXOMIL' ] ] },
        'Santo Antnio' => {
          '4755' => [
            [ '300', '312', 'MARTIM' ] ] },
        'Vila Frescainha' => {
          '4750' => [
            [ '833', '857', 'V FRESCAINHA (S PEDRO)' ] ] },
        'Macieira de Rates' => {
          '4755' => [
            [ '261', '279', 'MACIEIRA DE RATES' ] ] },
        'Barbeira' => {
          '4755' => [
            [ '021', '030', 'ALVELOS' ] ] },
        'Portelinha' => {
          '4750' => [
            [ '861', '865', 'VILAR DO MONTE BCL' ] ] },
        'Azevedo' => {
          '4750' => [
            [ '511', '520', 'LAMA BCL' ] ] },
        'Santo Estevo' => {
          '4755' => [
            [ '351', '360', 'MOURE BCL' ] ] },
        'Pateiro' => {
          '4750' => [
            [ '081', '089', 'ALVITO (SO PEDRO)' ] ] },
        'Gramosa' => {
          '4750' => [
            [ '581', '592', 'OLIVEIRA BCL' ] ] },
        'Sebastopol' => {
          '4755' => [
            [ '040', '049', 'AREIAS DE VILAR' ] ] },
        'Cachada' => {
          '4750' => [
            [ '501', '505', 'IGREJA NOVA BCL' ] ],
          '4755' => [
            [ '518', '538', 'VRZEA BCL' ] ] },
        'Aldeia' => {
          '4750' => [
            [ '241', '248', 'AREIAS BCL' ],
            [ '414', '419', 'COUTO BCL' ],
            [ '441', '445', 'FEITOS' ] ],
          '4905' => [
            [ '011', '022', 'ALDREU' ],
            [ '121', '138', 'PALME' ],
            [ '151', '163', 'TREGOSA' ] ],
          '4755' => [
            [ '231', '235', 'GILMONDE' ] ] },
        'Gandarinha' => {
          '4755' => [
            [ '251', '254', 'GUERAL' ] ] },
        'Silva' => {
          '4750' => [
            [ '678', '696', 'SILVA BCL' ] ] },
        'Barcelos' => {
          '4750' => [
            [ '101', '228', 'BARCELOS' ],
            [ '251', '342', 'BARCELOS' ],
            [ '775', '787', 'BARCELOS' ],
            [ '800', '828', 'BARCELOS' ],
            [ '830', '832', 'BARCELOS' ] ],
          '4755' => [
            [ '051', '071', 'BARCELOS' ] ] },
        'Monte Real' => {
          '4755' => [
            [ '481', '489', 'RIO COVO (SANTA EULLIA)' ] ] },
        'Quintes' => {
          '4755' => [
            [ '211', '214', 'FORNELOS BCL' ] ] },
        'Ponte' => {
          '4750' => [
            [ '361', '370', 'CAMPO BCL' ] ] },
        'Bajouco' => {
          '4755' => [
            [ '141', '148', 'COUREL' ] ] },
        'Barreiro' => {
          '4750' => [
            [ '001', '006', 'ABADE DE NEIVA' ] ] },
        'Real' => {
          '4750' => [
            [ '651', '659', 'RORIZ BCL' ] ] },
        'Caneiro' => {
          '4750' => [
            [ '051', '059', 'ALHEIRA' ] ] },
        'Tamel' => {
          '4750' => [
            [ '716', '726', 'TAMEL (SO VERSSIMO)' ],
            [ '729', '760', 'TAMEL (SO VERSSIMO)' ],
            [ '866', '877', 'TAMEL (SO VERSSIMO)' ],
            [ '879', '884', 'TAMEL (SO VERSSIMO)' ] ] },
        'Moutinho' => {
          '4775' => [
            [ '099', '099', 'FONTE COBERTA' ] ] },
        'Igreja' => {
          '4755' => [
            [ '101', '106', 'CARVALHAL BCL' ],
            [ '221', '226', 'GAMIL' ],
            [ '391', '393', 'PEDRA FURADA' ] ] },
        'Apeadeiro' => {
          '4775' => [
            [ '011', '035', 'CAMBESES BCL' ],
            [ '050', '057', 'CARREIRA BCL' ] ],
          '4905' => [
            [ '061', '077', 'DURRES' ] ] },
        'Anar' => {
          '4750' => [
            [ '601', '610', 'PANQUE' ] ] },
        'Amins' => {
          '4755' => [
            [ '121', '131', 'CHORENTE' ] ] },
        'Ufe' => {
          '4750' => [
            [ '381', '396', 'CARAPEOS' ] ] },
        'Capela' => {
          '4755' => [
            [ '111', '116', 'CARVALHOS BCL' ] ] },
        'Lousas' => {
          '4755' => [
            [ '371', '378', 'PARADELA BCL' ] ] },
        'Cachadinha' => {
          '4750' => [
            [ '461', '473', 'GALEGOS (SANTA MARIA)' ] ],
          '4755' => [
            [ '411', '424', 'POUSA' ] ] },
        'Fonte Fria' => {
          '4755' => [
            [ '469', '469', 'RIO COVO (SANTA EUGNIA)' ] ] },
        'Barrio' => {
          '4775' => [
            [ '140', '157', 'MINHOTES' ] ] },
        'Espezes' => {
          '4755' => [
            [ '331', '338', 'MILHAZES' ] ] },
        'Covas' => {
          '4755' => [
            [ '241', '247', 'GOIOS' ] ] },
        'Brenha' => {
          '4755' => [
            [ '201', '205', 'FARIA' ] ] },
        'Agrela' => {
          '4775' => [
            [ '080', '088', 'CHAVO' ] ],
          '4750' => [
            [ '041', '044', 'AGUIAR BCL' ] ],
          '4755' => [
            [ '081', '084', 'BASTUO (SANTO ESTEVO)' ] ] },
        'Outeiro' => {
          '4775' => [
            [ '265', '267', 'VIATODOS' ] ],
          '4755' => [
            [ '321', '326', 'MIDES BCL' ] ] },
        'Agra Maior' => {
          '4775' => [
            [ '090', '097', 'FONTE COBERTA' ] ] },
        'Barreiras' => {
          '4750' => [
            [ '791', '798', 'VILA COVA BCL' ] ] },
        'Aparecida' => {
          '4905' => [
            [ '031', '043', 'BALUGES' ] ] },
        'Carrapatoso' => {
          '4750' => [
            [ '021', '028', 'ABORIM' ] ] },
        'Jouve' => {
          '4740' => [
            [ '670', '690', 'BARQUEIROS BCL' ] ] },
        'Assento' => {
          '4750' => [
            [ '631', '642', 'QUINTIES' ] ],
          '4755' => [
            [ '551', '557', 'VILA SECA BCL' ] ] },
        'Portela' => {
          '4755' => [
            [ '441', '448', 'REMELHE' ] ] },
        'Cal' => {
          '4750' => [
            [ '401', '411', 'COSSOURADO BCL' ] ] }
             },
      'Vieira do Minho' => {
        'Vale de Cuba' => {
          '4850' => [
            [ '380', '385', 'SOENGAS' ] ] },
        'Loteamento do Telhado' => {
          '4850' => [
            [ '143', '158', 'EIRA VEDRA' ] ] },
        'Arco' => {
          '4850' => [
            [ '331', '343', 'RUIVES VRM' ] ] },
        'Cubos' => {
          '4850' => [
            [ '503', '503', 'VIEIRA DO MINHO' ] ] },
        'Cernados' => {
          '4850' => [
            [ '022', '022', 'ANJOS' ] ] },
        'So Francisco' => {
          '4850' => [
            [ '160', '163', 'EIRA VEDRA' ] ] },
        'Vieira do Minho' => {
          '4850' => [
            [ '521', '551', 'VIEIRA DO MINHO' ] ] },
        'Agra' => {
          '4850' => [
            [ '281', '286', 'ROSSAS VRM' ] ] },
        'Pena' => {
          '4850' => [
            [ '238', '246', 'MOSTEIRO VRM' ] ] },
        'Caminho Novo' => {
          '4850' => [
            [ '074', '095', 'CANTELES' ] ] },
        'Figueir' => {
          '4850' => [
            [ '232', '236', 'MOSTEIRO VRM' ] ] },
        'Barrei' => {
          '4850' => [
            [ '199', '202', 'GUILHOFREI' ] ] },
        'Fundevila' => {
          '4850' => [
            [ '024', '034', 'ANJOS' ] ] },
        'Tapado' => {
          '4850' => [
            [ '010', '016', 'ANISS' ] ] },
        'Cortegaa' => {
          '4850' => [
            [ '271', '274', 'PINHEIRO VRM' ] ] },
        'Casas' => {
          '4850' => [
            [ '288', '318', 'ROSSAS VRM' ] ] },
        'Vilar' => {
          '4850' => [
            [ '165', '165', 'EIRA VEDRA' ] ] },
        'Azevedo' => {
          '4850' => [
            [ '501', '501', 'VIEIRA DO MINHO' ] ] },
        'Ermal' => {
          '4850' => [
            [ '185', '186', 'GUILHOFREI' ] ] },
        'Lomba' => {
          '4850' => [
            [ '190', '192', 'GUILHOFREI' ] ] },
        'Penelas' => {
          '4850' => [
            [ '194', '197', 'GUILHOFREI' ] ] },
        'So Roque' => {
          '4850' => [
            [ '505', '509', 'VIEIRA DO MINHO' ] ] },
        'Aldeia' => {
          '4850' => [
            [ '261', '263', 'PARADA DO BOURO' ],
            [ '361', '372', 'SALAMONDE' ] ] },
        'Ches' => {
          '4850' => [
            [ '047', '049', 'CANIADA' ] ] },
        'Abelheira' => {
          '4850' => [
            [ '471', '481', 'VILAR CHO VRM' ] ] },
        'Fraga' => {
          '4850' => [
            [ '401', '407', 'SOUTELO VRM' ] ] },
        'Taboadela' => {
          '4850' => [
            [ '248', '248', 'MOSTEIRO VRM' ] ] },
        'Cando' => {
          '4850' => [
            [ '211', '223', 'LOUREDO VRM' ] ] },
        'Cambedo' => {
          '4850' => [
            [ '041', '043', 'CAMPOS VRM' ] ] },
        'Bouas' => {
          '4850' => [
            [ '451', '461', 'VENTOSA VRM' ] ] },
        'Barbdo' => {
          '4850' => [
            [ '045', '045', 'CANIADA' ] ] },
        'Assento' => {
          '4850' => [
            [ '051', '056', 'CANIADA' ],
            [ '071', '072', 'CANTELES' ],
            [ '121', '129', 'COVA' ],
            [ '181', '182', 'GUILHOFREI' ],
            [ '421', '433', 'TABUAAS' ] ] },
        'Guilhofrei' => {
          '4850' => [
            [ '188', '188', 'GUILHOFREI' ] ] }
              },
      'Esposende' => {
        'Curvos' => {
          '4740' => [
            [ '181', '185', 'CURVOS' ] ] },
        'Capela' => {
          '4740' => [
            [ '611', '613', 'RIO TINTO EPS' ] ] },
        'Criaz' => {
          '4740' => [
            [ '101', '129', 'APLIA' ] ] },
        'Aldeia' => {
          '4740' => [
            [ '431', '452', 'FORJES' ],
            [ '491', '498', 'GEMESES' ],
            [ '631', '638', 'VILA CH EPS' ] ] },
        'Areia' => {
          '4740' => [
            [ '321', '398', 'FO' ] ] },
        'Belinho' => {
          '4740' => [
            [ '161', '166', 'BELINHO' ] ] },
        'Caveiros' => {
          '4740' => [
            [ '417', '422', 'FONTE BOA' ] ] },
        'Agra' => {
          '4740' => [
            [ '415', '415', 'FONTE BOA' ] ] },
        'Azevedo' => {
          '4740' => [
            [ '011', '019', 'ANTAS EPS' ] ] },
        'Esposende' => {
          '4740' => [
            [ '001', '001', 'ESPOSENDE' ],
            [ '010', '010', 'ESPOSENDE' ],
            [ '200', '209', 'ESPOSENDE' ],
            [ '211', '305', 'ESPOSENDE' ],
            [ '471', '476', 'ESPOSENDE' ],
            [ '521', '522', 'ESPOSENDE' ],
            [ '531', '540', 'ESPOSENDE' ],
            [ '542', '558', 'ESPOSENDE' ],
            [ '571', '577', 'ESPOSENDE' ] ] },
        'Igreja' => {
          '4740' => [
            [ '141', '142', 'APLIA' ] ] },
        'Barral' => {
          '4740' => [
            [ '591', '598', 'PALMEIRA DE FARO' ] ] },
        'Ofir' => {
          '4740' => [
            [ '405', '406', 'FO' ] ] },
        'Baixo' => {
          '4740' => [
            [ '511', '512', 'MAR' ] ] },
        'Aplia' => {
          '4740' => [
            [ '031', '075', 'APLIA' ] ] }
              },
      'Vila Nova de Famalico' => {
        'Casteles' => {
          '4770' => [
            [ '830', '898', 'CASTELES VNF' ] ] },
        'Algide' => {
          '4770' => [
            [ '400', '420', 'POUSADA DE SARAMAGOS' ] ] },
        'Forno' => {
          '4770' => [
            [ '427', '467', 'REQUIO' ] ] },
        'Santa Maria Oliveira' => {
          '4765' => [
            [ '149', '149', 'SANTA MARIA OLIVEIRA' ],
            [ '320', '320', 'SANTA MARIA OLIVEIRA' ],
            [ '322', '358', 'SANTA MARIA OLIVEIRA' ],
            [ '360', '361', 'SANTA MARIA OLIVEIRA' ],
            [ '800', '804', 'SANTA MARIA OLIVEIRA' ] ] },
        'So Miguel Seide' => {
          '4770' => [
            [ '621', '656', 'SO MIGUEL SEIDE' ],
            [ '661', '661', 'SO MIGUEL SEIDE' ] ] },
        'Eiriz' => {
          '4765' => [
            [ '144', '148', 'PEDOME' ] ] },
        'Cima' => {
          '4770' => [
            [ '060', '060', 'BENTE' ] ] },
        'Gavio' => {
          '4760' => [
            [ '001', '083', 'VILA NOVA DE FAMALICO' ],
            [ '085', '086', 'VILA NOVA DE FAMALICO' ],
            [ '088', '089', 'VILA NOVA DE FAMALICO' ],
            [ '091', '201', 'VILA NOVA DE FAMALICO' ],
            [ '205', '210', 'VILA NOVA DE FAMALICO' ],
            [ '212', '216', 'VILA NOVA DE FAMALICO' ],
            [ '218', '239', 'VILA NOVA DE FAMALICO' ],
            [ '270', '304', 'VILA NOVA DE FAMALICO' ],
            [ '413', '420', 'VILA NOVA DE FAMALICO' ],
            [ '732', '734', 'VILA NOVA DE FAMALICO' ] ] },
        'Agra do Barreiro' => {
          '4765' => [
            [ '600', '671', 'DELES' ] ] },
        'Fontainhas' => {
          '4770' => [
            [ '160', '160', 'JESUFREI' ] ] },
        'Gondifelos' => {
          '4760' => [
            [ '490', '490', 'GONDIFELOS' ] ] },
        'Cruz' => {
          '4770' => [
            [ '112', '131', 'CRUZ' ],
            [ '144', '159', 'CRUZ' ],
            [ '161', '174', 'CRUZ' ],
            [ '176', '200', 'CRUZ' ],
            [ '228', '235', 'CRUZ' ] ] },
        'Igreja' => {
          '4765' => [
            [ '168', '175', 'NOVAIS' ] ] },
        'Ventoselas' => {
          '4760' => [
            [ '485', '485', 'FRADELOS VNF' ] ] },
        'Almas' => {
          '4765' => [
            [ '121', '130', 'NOVAIS' ] ] },
        'Segade' => {
          '4765' => [
            [ '080', '081', 'CARREIRA VNF' ] ] },
        'Brufe' => {
          '4760' => [
            [ '240', '250', 'BRUFE VNF' ],
            [ '252', '262', 'BRUFE VNF' ],
            [ '829', '833', 'BRUFE VNF' ],
            [ '849', '849', 'BRUFE VNF' ] ] },
        'Avidos' => {
          '4770' => [
            [ '780', '828', 'AVIDOS' ] ] },
        'Combros' => {
          '4765' => [
            [ '094', '094', 'DELES' ] ] },
        'Almorode' => {
          '4760' => [
            [ '495', '514', 'GONDIFELOS' ] ] },
        'Aldeia do Sol' => {
          '4775' => [
            [ '400', '423', 'LEMENHE' ] ] },
        'Coura' => {
          '4775' => [
            [ '444', '444', 'NINE' ] ] },
        'Cabo' => {
          '4770' => [
            [ '281', '290', 'LAGOA VNF' ] ] },
        'Riba de Ave' => {
          '4765' => [
            [ '181', '182', 'RIBA DE AVE' ],
            [ '184', '184', 'RIBA DE AVE' ],
            [ '187', '187', 'RIBA DE AVE' ],
            [ '191', '191', 'RIBA DE AVE' ],
            [ '200', '275', 'RIBA DE AVE' ] ] },
        'Louro' => {
          '4760' => [
            [ '530', '584', 'LOURO' ],
            [ '735', '738', 'LOURO' ],
            [ '783', '787', 'LOURO' ],
            [ '789', '789', 'LOURO' ],
            [ '792', '799', 'LOURO' ],
            [ '825', '828', 'LOURO' ] ] },
        'Vilares' => {
          '4765' => [
            [ '163', '167', 'PEDOME' ] ] },
        'Borralheira' => {
          '4775' => [
            [ '440', '442', 'NINE' ] ] },
        'Presa' => {
          '4770' => [
            [ '370', '370', 'PORTELA VNF' ] ] },
        'Aldeia Nova' => {
          '4770' => [
            [ '481', '507', 'RUIVES VNF' ] ],
          '4760' => [
            [ '863', '867', 'GONDIFELOS' ] ] },
        'Agra' => {
          '4775' => [
            [ '480', '497', 'SANTA EULLIA ARNOSO' ] ],
          '4765' => [
            [ '131', '134', 'PEDOME' ] ],
          '4770' => [
            [ '521', '541', 'SANTA MARIA ARNOSO' ] ] },
        'Vila Nova de Famalico' => {
          '4764' => [
            [ '501', '503', 'VILA NOVA DE FAMALICO' ] ],
          '4760' => [
            [ '251', '251', 'VILA NOVA DE FAMALICO' ],
            [ '848', '848', 'VILA NOVA DE FAMALICO' ] ] },
        'Agras' => {
          '4770' => [
            [ '301', '335', 'LANDIM' ] ] },
        'Agra da Portela' => {
          '4770' => [
            [ '701', '722', 'TELHADO VNF' ] ] },
        'Bairro' => {
          '4765' => [
            [ '006', '006', 'BAIRRO' ],
            [ '011', '033', 'BAIRRO' ],
            [ '036', '037', 'BAIRRO' ],
            [ '039', '051', 'BAIRRO' ],
            [ '053', '053', 'BAIRRO' ],
            [ '055', '058', 'BAIRRO' ],
            [ '060', '060', 'BAIRRO' ],
            [ '062', '069', 'BAIRRO' ],
            [ '084', '084', 'BAIRRO' ],
            [ '672', '690', 'BAIRRO' ] ] },
        'Anariz' => {
          '4770' => [
            [ '352', '369', 'MOUQUIM' ] ] },
        'Santo Antnio' => {
          '4775' => [
            [ '456', '460', 'NINE' ] ] },
        'Cavales' => {
          '4760' => [
            [ '421', '467', 'CAVALES' ],
            [ '834', '835', 'CAVALES' ] ] },
        'Vilarinho das Cambas' => {
          '4760' => [
            [ '739', '741', 'VILARINHO DAS CAMBAS' ],
            [ '743', '782', 'VILARINHO DAS CAMBAS' ],
            [ '790', '790', 'VILARINHO DAS CAMBAS' ],
            [ '836', '847', 'VILARINHO DAS CAMBAS' ] ] },
        'Aldeia do Monte' => {
          '4770' => [
            [ '561', '589', 'SO COSME VALE' ] ] },
        'Meje' => {
          '4770' => [
            [ '690', '690', 'SEZURES VNF' ] ] },
        'Ponte Serves' => {
          '4765' => [
            [ '156', '159', 'PEDOME' ] ] },
        'Oliveira So Mateus' => {
          '4765' => [
            [ '380', '383', 'OLIVEIRA (SO MATEUS)' ],
            [ '387', '389', 'OLIVEIRA (SO MATEUS)' ],
            [ '700', '779', 'OLIVEIRA (SO MATEUS)' ] ] },
        'guas' => {
          '4770' => [
            [ '071', '087', 'CABEUDOS' ] ] },
        'Lousado' => {
          '4760' => [
            [ '601', '603', 'LOUSADO' ],
            [ '605', '606', 'LOUSADO' ],
            [ '608', '665', 'LOUSADO' ],
            [ '667', '672', 'LOUSADO' ],
            [ '674', '677', 'LOUSADO' ],
            [ '682', '683', 'LOUSADO' ],
            [ '800', '824', 'LOUSADO' ],
            [ '851', '851', 'LOUSADO' ] ] },
        'Joane' => {
          '4770' => [
            [ '201', '227', 'JOANE' ],
            [ '243', '243', 'JOANE' ],
            [ '245', '248', 'JOANE' ],
            [ '251', '252', 'JOANE' ],
            [ '254', '254', 'JOANE' ],
            [ '258', '261', 'JOANE' ],
            [ '265', '265', 'JOANE' ],
            [ '267', '269', 'JOANE' ],
            [ '271', '271', 'JOANE' ],
            [ '273', '273', 'JOANE' ],
            [ '275', '277', 'JOANE' ],
            [ '291', '294', 'JOANE' ],
            [ '296', '299', 'JOANE' ] ] },
        'Agrinha' => {
          '4765' => [
            [ '071', '078', 'CARREIRA VNF' ] ],
          '4770' => [
            [ '662', '681', 'SO PAIO SEIDE' ] ] },
        'Sapos' => {
          '4765' => [
            [ '161', '161', 'PEDOME' ] ] },
        'Antas' => {
          '4760' => [
            [ '850', '850', 'VILA NOVA DE FAMALICO' ],
            [ '868', '880', 'VILA NOVA DE FAMALICO' ] ] },
        'Outeiro' => {
          '4765' => [
            [ '150', '152', 'PEDOME' ] ] },
        'Berredo' => {
          '4770' => [
            [ '350', '350', 'MOGEGE' ] ] },
        'Agra Maior' => {
          '4770' => [
            [ '751', '771', 'VERMOIM VNF' ] ] },
        'Alminhas Roubadas' => {
          '4770' => [
            [ '011', '018', 'ABADE DE VERMOIM' ] ] },
        'Ribeiro' => {
          '4760' => [
            [ '701', '731', 'RIBEIRO' ] ] },
        'Quinta da Terra Nova' => {
          '4765' => [
            [ '193', '194', 'RIBA DE AVE' ] ] },
        'So Joo de Perrelos' => {
          '4765' => [
            [ '112', '120', 'DELES' ] ] },
        'Palhares' => {
          '4775' => [
            [ '453', '454', 'NINE' ] ] },
        'Bemposta' => {
          '4765' => [
            [ '137', '142', 'PEDOME' ] ] },
        'Estao' => {
          '4775' => [
            [ '446', '451', 'NINE' ] ] },
        'Bouas' => {
          '4760' => [
            [ '692', '692', 'OUTIZ' ] ] },
        'Calendrio' => {
          '4760' => [
            [ '306', '335', 'VILA NOVA DE FAMALICO' ],
            [ '337', '393', 'VILA NOVA DE FAMALICO' ],
            [ '395', '399', 'VILA NOVA DE FAMALICO' ],
            [ '468', '469', 'VILA NOVA DE FAMALICO' ],
            [ '476', '479', 'VILA NOVA DE FAMALICO' ],
            [ '483', '484', 'VILA NOVA DE FAMALICO' ],
            [ '486', '489', 'VILA NOVA DE FAMALICO' ],
            [ '491', '493', 'VILA NOVA DE FAMALICO' ],
            [ '515', '526', 'VILA NOVA DE FAMALICO' ],
            [ '528', '529', 'VILA NOVA DE FAMALICO' ],
            [ '585', '589', 'VILA NOVA DE FAMALICO' ],
            [ '591', '600', 'VILA NOVA DE FAMALICO' ],
            [ '681', '681', 'VILA NOVA DE FAMALICO' ],
            [ '684', '690', 'VILA NOVA DE FAMALICO' ],
            [ '693', '700', 'VILA NOVA DE FAMALICO' ],
            [ '852', '862', 'VILA NOVA DE FAMALICO' ] ] },
        'Agra de Alm' => {
          '4770' => [
            [ '601', '618', 'SO MARTINHO VALE' ] ] },
        'Esmeriz' => {
          '4760' => [
            [ '480', '482', 'ESMERIZ' ] ] },
        'Loureiro' => {
          '4765' => [
            [ '096', '102', 'DELES' ] ] }
               },
      'Pvoa de Lanhoso' => {
        'Moinho da Porta' => {
          '4830' => [
            [ '189', '190', 'FONTE ARCADA PVL' ] ] },
        'Bustelos' => {
          '4830' => [
            [ '774', '774', 'TRAVASSOS' ] ] },
        'Fonte do Rei' => {
          '4830' => [
            [ '340', '340', 'LANHOSO' ] ] },
        'Caldezes' => {
          '4830' => [
            [ '435', '435', 'MOURE PVL' ] ] },
        'Arrifana' => {
          '4830' => [
            [ '192', '192', 'FONTE ARCADA PVL' ] ] },
        'Boucinha' => {
          '4830' => [
            [ '812', '831', 'VILELA PVL' ] ] },
        'Alagoa' => {
          '4830' => [
            [ '691', '695', 'SERZEDELO PVL' ] ] },
        'Laca' => {
          '4830' => [
            [ '410', '416', 'MONSUL' ] ] },
        'Monte' => {
          '4830' => [
            [ '102', '107', 'CAMPOS PVL' ],
            [ '312', '312', 'GERAZ DO MINHO' ],
            [ '437', '439', 'MOURE PVL' ],
            [ '640', '640', 'SANTO EMILIO' ] ] },
        'Estrada' => {
          '4830' => [
            [ '625', '626', 'RENDUFINHO' ],
            [ '778', '778', 'TRAVASSOS' ] ] },
        'Carvalhinho' => {
          '4830' => [
            [ '274', '275', 'GARFE' ] ] },
        'Lage' => {
          '4830' => [
            [ '750', '750', 'TADE' ],
            [ '780', '780', 'TRAVASSOS' ] ] },
        'Cancela' => {
          '4830' => [
            [ '404', '404', 'MONSUL' ] ] },
        'Torro' => {
          '4830' => [
            [ '077', '077', 'CALVOS PVL' ] ] },
        'Alm' => {
          '4830' => [
            [ '601', '602', 'OLIVEIRA PVL' ] ] },
        'Pinheiro' => {
          '4830' => [
            [ '314', '315', 'GERAZ DO MINHO' ],
            [ '346', '347', 'LANHOSO' ] ] },
        'Souto' => {
          '4830' => [
            [ '110', '118', 'CAMPOS PVL' ],
            [ '128', '132', 'COVELAS PVL' ],
            [ '350', '352', 'LANHOSO' ],
            [ '441', '448', 'MOURE PVL' ] ] },
        'Vila Seca' => {
          '4830' => [
            [ '652', '652', 'SANTO EMILIO' ] ] },
        'S. Pedro' => {
          '4830' => [
            [ '368', '371', 'LOUREDO PVL' ] ] },
        'Paredes Secas' => {
          '4830' => [
            [ '072', '075', 'CALVOS PVL' ] ] },
        'Requeixo' => {
          '4830' => [
            [ '682', '682', 'SO JOO DE REI' ] ] },
        'Devesa' => {
          '4830' => [
            [ '097', '099', 'CAMPOS PVL' ],
            [ '406', '408', 'MONSUL' ] ] },
        'Alferes' => {
          '4830' => [
            [ '061', '070', 'CALVOS PVL' ] ] },
        'Tapadas' => {
          '4830' => [
            [ '763', '767', 'TADE' ] ] },
        'Pardieiros' => {
          '4830' => [
            [ '699', '699', 'SERZEDELO PVL' ] ] },
        'Couo' => {
          '4830' => [
            [ '373', '373', 'LOUREDO PVL' ] ] },
        'Costeirinha' => {
          '4830' => [
            [ '005', '013', 'GUAS SANTAS PVL' ] ] },
        'Quintela' => {
          '4830' => [
            [ '757', '760', 'TADE' ],
            [ '799', '800', 'VERIM' ] ] },
        'Serzedelo' => {
          '4830' => [
            [ '701', '707', 'SERZEDELO PVL' ] ] },
        'Lajes' => {
          '4830' => [
            [ '675', '678', 'SO JOO DE REI' ] ] },
        'Aldeia' => {
          '4830' => [
            [ '332', '333', 'LANHOSO' ],
            [ '621', '623', 'RENDUFINHO' ] ] },
        'Vila' => {
          '4830' => [
            [ '354', '358', 'LANHOSO' ] ] },
        'Arganha' => {
          '4830' => [
            [ '662', '673', 'SO JOO DE REI' ] ] },
        'Quintes' => {
          '4830' => [
            [ '317', '329', 'GERAZ DO MINHO' ],
            [ '647', '648', 'SANTO EMILIO' ] ] },
        'Burgo' => {
          '4830' => [
            [ '425', '426', 'MONSUL' ] ] },
        'Reguengo' => {
          '4830' => [
            [ '030', '030', 'AJUDE' ] ] },
        'Magalhes' => {
          '4830' => [
            [ '181', '186', 'FERREIROS PVL' ] ] },
        'Leiradela' => {
          '4830' => [
            [ '782', '785', 'TRAVASSOS' ] ] },
        'Reboreda' => {
          '4830' => [
            [ '680', '680', 'SO JOO DE REI' ] ] },
        'Cvo' => {
          '4830' => [
            [ '251', '257', 'GALEGOS PVL' ] ] },
        'Patos' => {
          '4830' => [
            [ '018', '020', 'GUAS SANTAS PVL' ] ] },
        'Braceiras' => {
          '4830' => [
            [ '122', '126', 'COVELAS PVL' ] ] },
        'Igreja' => {
          '4830' => [
            [ '150', '155', 'ESPERANA PVL' ],
            [ '342', '342', 'LANHOSO' ],
            [ '376', '379', 'LOUREDO PVL' ],
            [ '644', '645', 'SANTO EMILIO' ] ] },
        'So Bento' => {
          '4830' => [
            [ '650', '650', 'SANTO EMILIO' ] ] },
        'Roda' => {
          '4830' => [
            [ '293', '296', 'GARFE' ] ] },
        'Costa' => {
          '4830' => [
            [ '194', '209', 'FONTE ARCADA PVL' ],
            [ '212', '218', 'FRADES' ] ] },
        'Calva' => {
          '4830' => [
            [ '299', '310', 'GERAZ DO MINHO' ] ] },
        'Aldeia de Baixo' => {
          '4830' => [
            [ '771', '772', 'TRAVASSOS' ] ] },
        'Cz Louredo' => {
          '4830' => [
            [ '094', '095', 'CAMPOS PVL' ] ] },
        'Condes' => {
          '4830' => [
            [ '278', '291', 'GARFE' ] ] },
        'Cimo de Vila' => {
          '4830' => [
            [ '338', '338', 'LANHOSO' ],
            [ '697', '697', 'SERZEDELO PVL' ],
            [ '744', '747', 'TADE' ] ] },
        'Alcouce' => {
          '4830' => [
            [ '711', '731', 'SOBRADELO DA GOMA' ] ] },
        'Castanheira' => {
          '4830' => [
            [ '776', '776', 'TRAVASSOS' ] ] },
        'Arrabalde' => {
          '4830' => [
            [ '431', '433', 'MOURE PVL' ] ] },
        'Pvoa de Lanhoso' => {
          '4830' => [
            [ '191', '191', 'PVOA DE LANHOSO' ],
            [ '343', '343', 'PVOA DE LANHOSO' ],
            [ '501', '531', 'PVOA DE LANHOSO' ],
            [ '533', '554', 'PVOA DE LANHOSO' ],
            [ '557', '561', 'PVOA DE LANHOSO' ],
            [ '563', '593', 'PVOA DE LANHOSO' ] ] },
        'Covas' => {
          '4830' => [
            [ '044', '051', 'BRUNHAIS' ] ] },
        'Pousadela' => {
          '4830' => [
            [ '418', '423', 'MONSUL' ] ] },
        'Valinhas' => {
          '4830' => [
            [ '188', '188', 'FERREIROS PVL' ] ] },
        'Bulhoso' => {
          '4830' => [
            [ '173', '179', 'FERREIROS PVL' ] ] },
        'Boavista' => {
          '4830' => [
            [ '166', '171', 'FERREIROS PVL' ] ] },
        'Carreira' => {
          '4830' => [
            [ '261', '264', 'GARFE' ] ] },
        'Outeiro' => {
          '4830' => [
            [ '228', '240', 'FRIANDE PVL' ],
            [ '604', '610', 'OLIVEIRA PVL' ],
            [ '754', '755', 'TADE' ] ] },
        'Monte Olivete' => {
          '4830' => [
            [ '752', '752', 'TADE' ] ] },
        'Batocas' => {
          '4830' => [
            [ '802', '804', 'VERIM' ] ] },
        'Barreiras' => {
          '4830' => [
            [ '259', '260', 'GALEGOS PVL' ] ] },
        'Brrio' => {
          '4830' => [
            [ '402', '402', 'MONSUL' ],
            [ '791', '797', 'VERIM' ] ] },
        'Bobeiro' => {
          '4830' => [
            [ '741', '741', 'TADE' ] ] },
        'Sobradelo' => {
          '4830' => [
            [ '628', '636', 'RENDUFINHO' ] ] },
        'Assento' => {
          '4830' => [
            [ '092', '092', 'CAMPOS PVL' ],
            [ '271', '272', 'GARFE' ],
            [ '642', '642', 'SANTO EMILIO' ] ] },
        'Portuzelo' => {
          '4830' => [
            [ '022', '025', 'GUAS SANTAS PVL' ] ] }
               },
      'Fafe' => {
        'Ajuda' => {
          '4820' => [
            [ '051', '068', 'ESTOROS FAF' ] ] },
        'Cortinhas' => {
          '4820' => [
            [ '531', '545', 'MOREIRA DO REI' ] ] },
        'Ameixedo' => {
          '4820' => [
            [ '560', '560', 'QUEIMADELA FAF' ] ] },
        'Areal' => {
          '4820' => [
            [ '525', '529', 'MOREIRA DO REI' ] ] },
        'Escola' => {
          '4820' => [
            [ '830', '830', 'VINHS FAF' ] ] },
        'Alminhas' => {
          '4820' => [
            [ '651', '667', 'SO GENS' ] ] },
        'Ch do Fojo' => {
          '4820' => [
            [ '003', '003', 'AGRELA FAF' ] ] },
        'Cacho' => {
          '4820' => [
            [ '630', '630', 'REVELHE' ] ] },
        'Agrades' => {
          '4820' => [
            [ '731', '759', 'ARES (SO ROMO)' ] ] },
        'Bairro da Ponte de So Jos' => {
          '4820' => [
            [ '005', '005', 'ANTIME' ] ] },
        'Igreja' => {
          '4820' => [
            [ '456', '479', 'GOLES' ] ] },
        'Boua' => {
          '4820' => [
            [ '440', '440', 'FREITAS' ],
            [ '496', '512', 'MEDELO' ],
            [ '765', '765', 'SEIDES' ] ] },
        'Crasto' => {
          '4820' => [
            [ '635', '635', 'RIBEIROS' ] ] },
        'Agro' => {
          '4820' => [
            [ '770', '770', 'SERAFO' ] ] },
        'Cancela' => {
          '4820' => [
            [ '400', '400', 'FAREJA' ] ] },
        'Torro' => {
          '4820' => [
            [ '669', '672', 'SO GENS' ] ] },
        'Costa' => {
          '4820' => [
            [ '645', '645', 'SO CLEMENTE SILVARES' ] ] },
        'Casal de Estime' => {
          '4820' => [
            [ '520', '520', 'MONTE FAF' ] ] },
        'Ribeiro' => {
          '4820' => [
            [ '426', '430', 'FORNELOS FAF' ] ] },
        'Arnozela' => {
          '4820' => [
            [ '013', '013', 'ARNOZELA' ] ] },
        'Felgueiras' => {
          '4820' => [
            [ '405', '405', 'FELGUEIRAS FAF' ] ] },
        'Agras' => {
          '4820' => [
            [ '640', '640', 'ARES (SANTA CRISTINA)' ] ] },
        'Couto' => {
          '4820' => [
            [ '007', '007', 'ARDEGO FAF' ] ] },
        'Fornelo' => {
          '4820' => [
            [ '416', '424', 'FORNELOS FAF' ] ] },
        'Bairro' => {
          '4820' => [
            [ '445', '454', 'GOLES' ] ] },
        'Revelhe' => {
          '4824' => [
            [ '502', '502', 'REVELHE' ] ] },
        'Aidro' => {
          '4820' => [
            [ '825', '825', 'VILA COVA FAF' ] ] },
        'Vinhas' => {
          '4820' => [
            [ '432', '432', 'FORNELOS FAF' ] ] },
        'Azenha' => {
          '4820' => [
            [ '018', '050', 'CEPES' ] ] },
        'Alm do Rio' => {
          '4820' => [
            [ '775', '813', 'TRAVASSS FAF' ] ] },
        'Bastelo' => {
          '4820' => [
            [ '820', '820', 'VRZEA COVA' ] ] },
        'Alminha' => {
          '4820' => [
            [ '015', '016', 'CEPES' ] ] },
        'Lages' => {
          '4820' => [
            [ '086', '097', 'CEPES' ] ] },
        'Antas' => {
          '4820' => [
            [ '491', '494', 'MEDELO' ] ] },
        'Boavista' => {
          '4820' => [
            [ '411', '414', 'FORNELOS FAF' ],
            [ '601', '624', 'REGADAS' ] ] },
        'Aldeia' => {
          '4820' => [
            [ '550', '550', 'PASSOS FAF' ] ] },
        'Pedrado' => {
          '4820' => [
            [ '555', '555', 'PEDRADO' ] ] },
        'Leis' => {
          '4820' => [
            [ '070', '085', 'ESTOROS FAF' ] ] },
        'Agrelo' => {
          '4820' => [
            [ '571', '593', 'QUINCHES' ] ] },
        'Fafe' => {
          '4820' => [
            [ '098', '100', 'FAFE' ],
            [ '102', '102', 'FAFE' ],
            [ '115', '138', 'FAFE' ],
            [ '140', '351', 'FAFE' ],
            [ '353', '368', 'FAFE' ],
            [ '387', '399', 'FAFE' ],
            [ '401', '404', 'FAFE' ],
            [ '406', '410', 'FAFE' ],
            [ '433', '439', 'FAFE' ],
            [ '441', '444', 'FAFE' ],
            [ '480', '484', 'FAFE' ] ],
          '4824' => [
            [ '501', '501', 'FAFE' ] ] },
        'Barbeita' => {
          '4820' => [
            [ '001', '001', 'ABOIM FAF' ] ] },
        'Assento' => {
          '4820' => [
            [ '010', '010', 'ARMIL' ],
            [ '701', '715', 'SO MARTINHO SILVARES' ] ] },
        'Gontim' => {
          '4820' => [
            [ '485', '485', 'GONTIM' ] ] }
         },
      'Guimares' => {
        'Sande So Martinho' => {
          '4805' => [
            [ '501', '541', 'SO MARTINHO SANDE' ] ] },
        'Briteiros Santa Leocdia' => {
          '4809' => [
            [ '012', '012', 'SANTA LEOCDIA BRITEIROS' ] ],
          '4805' => [
            [ '454', '474', 'SANTA LEOCDIA BRITEIROS' ],
            [ '631', '631', 'SANTA LEOCDIA BRITEIROS' ] ] },
        'Brito' => {
          '4809' => [
            [ '001', '001', 'BRITO' ],
            [ '003', '003', 'BRITO' ],
            [ '009', '010', 'BRITO' ] ],
          '4805' => [
            [ '019', '076', 'BRITO' ],
            [ '630', '630', 'BRITO' ] ] },
        'So Faustino' => {
          '4815' => [
            [ '001', '010', 'SO FAUSTINO' ],
            [ '062', '070', 'SO FAUSTINO' ],
            [ '372', '379', 'SO FAUSTINO' ] ] },
        'Prazins Santa Eufmia' => {
          '4800' => [
            [ '606', '619', 'SANTA EUFMIA PRAZINS' ] ] },
        'Infantas' => {
          '4810' => [
            [ '574', '600', 'INFANTAS' ],
            [ '619', '630', 'INFANTAS' ],
            [ '646', '646', 'INFANTAS' ],
            [ '690', '690', 'INFANTAS' ] ] },
        'Figueiredo' => {
          '4805' => [
            [ '184', '186', 'FIGUEIREDO GMR' ],
            [ '602', '602', 'FIGUEIREDO GMR' ] ] },
        'So Torcato' => {
          '4800' => [
            [ '294', '295', 'SO TORCATO' ],
            [ '347', '349', 'SO TORCATO' ],
            [ '562', '565', 'SO TORCATO' ],
            [ '600', '605', 'SO TORCATO' ],
            [ '620', '625', 'SO TORCATO' ],
            [ '647', '650', 'SO TORCATO' ],
            [ '669', '673', 'SO TORCATO' ],
            [ '676', '680', 'SO TORCATO' ],
            [ '697', '700', 'SO TORCATO' ],
            [ '705', '709', 'SO TORCATO' ],
            [ '734', '740', 'SO TORCATO' ],
            [ '756', '760', 'SO TORCATO' ],
            [ '784', '789', 'SO TORCATO' ] ],
          '4804' => [
            [ '507', '507', 'SO TORCATO' ] ] },
        'Lordelo' => {
          '4815' => [
            [ '057', '061', 'LORDELO GMR' ],
            [ '111', '250', 'LORDELO GMR' ] ] },
        'Quinta de Dentro' => {
          '4800' => [
            [ '851', '890', 'SO TORCATO' ] ] },
        'Nespereira' => {
          '4839' => [
            [ '001', '001', 'NESPEREIRA GMR' ] ],
          '4835' => [
            [ '468', '536', 'NESPEREIRA GMR' ] ] },
        'Conde' => {
          '4815' => [
            [ '015', '052', 'CONDE' ],
            [ '056', '056', 'CONDE' ] ] },
        'Cabreira' => {
          '4835' => [
            [ '595', '596', 'GUIMARES' ],
            [ '606', '612', 'GUIMARES' ] ] },
        'Balazar' => {
          '4805' => [
            [ '001', '013', 'BALAZAR GMR' ] ] },
        'Caldas das Taipas' => {
          '4805' => [
            [ '077', '147', 'CALDAS DAS TAIPAS' ],
            [ '149', '173', 'CALDAS DAS TAIPAS' ] ] },
        'Guimares' => {
          '4814' => [
            [ '501', '501', 'GUIMARES' ],
            [ '504', '504', 'GUIMARES' ],
            [ '508', '508', 'GUIMARES' ],
            [ '510', '510', 'GUIMARES' ],
            [ '515', '516', 'GUIMARES' ],
            [ '521', '530', 'GUIMARES' ] ],
          '4800' => [
            [ '001', '029', 'GUIMARES' ],
            [ '031', '031', 'GUIMARES' ],
            [ '033', '047', 'GUIMARES' ],
            [ '049', '172', 'GUIMARES' ],
            [ '174', '188', 'GUIMARES' ],
            [ '404', '404', 'GUIMARES' ],
            [ '406', '409', 'GUIMARES' ],
            [ '411', '414', 'GUIMARES' ],
            [ '416', '420', 'GUIMARES' ],
            [ '430', '434', 'GUIMARES' ],
            [ '436', '439', 'GUIMARES' ],
            [ '441', '445', 'GUIMARES' ],
            [ '453', '455', 'GUIMARES' ],
            [ '471', '474', 'GUIMARES' ],
            [ '476', '476', 'GUIMARES' ],
            [ '531', '532', 'GUIMARES' ],
            [ '535', '535', 'GUIMARES' ] ],
          '4839' => [
            [ '002', '002', 'GUIMARES' ],
            [ '004', '006', 'GUIMARES' ],
            [ '008', '009', 'GUIMARES' ],
            [ '011', '015', 'GUIMARES' ],
            [ '017', '020', 'GUIMARES' ] ],
          '4804' => [
            [ '509', '510', 'GUIMARES' ] ],
          '4835' => [
            [ '001', '055', 'GUIMARES' ],
            [ '057', '115', 'GUIMARES' ],
            [ '117', '444', 'GUIMARES' ],
            [ '602', '603', 'GUIMARES' ] ],
          '4810' => [
            [ '006', '033', 'GUIMARES' ],
            [ '035', '052', 'GUIMARES' ],
            [ '079', '079', 'GUIMARES' ],
            [ '084', '084', 'GUIMARES' ],
            [ '214', '240', 'GUIMARES' ],
            [ '242', '242', 'GUIMARES' ],
            [ '245', '247', 'GUIMARES' ],
            [ '250', '250', 'GUIMARES' ],
            [ '253', '253', 'GUIMARES' ],
            [ '255', '255', 'GUIMARES' ],
            [ '257', '261', 'GUIMARES' ],
            [ '264', '264', 'GUIMARES' ],
            [ '266', '266', 'GUIMARES' ],
            [ '268', '268', 'GUIMARES' ],
            [ '274', '275', 'GUIMARES' ],
            [ '279', '279', 'GUIMARES' ],
            [ '282', '282', 'GUIMARES' ],
            [ '284', '284', 'GUIMARES' ],
            [ '286', '286', 'GUIMARES' ],
            [ '289', '290', 'GUIMARES' ],
            [ '294', '294', 'GUIMARES' ],
            [ '420', '420', 'GUIMARES' ],
            [ '422', '422', 'GUIMARES' ],
            [ '425', '446', 'GUIMARES' ],
            [ '448', '453', 'GUIMARES' ],
            [ '480', '487', 'GUIMARES' ],
            [ '489', '517', 'GUIMARES' ],
            [ '520', '525', 'GUIMARES' ],
            [ '527', '529', 'GUIMARES' ],
            [ '531', '535', 'GUIMARES' ],
            [ '537', '537', 'GUIMARES' ],
            [ '545', '560', 'GUIMARES' ],
            [ '562', '562', 'GUIMARES' ],
            [ '564', '566', 'GUIMARES' ],
            [ '895', '896', 'GUIMARES' ] ] },
        'Prazins de Santo Tirso' => {
          '4800' => [
            [ '268', '270', 'SANTO TIRSO PRAZINS' ],
            [ '291', '291', 'SANTO TIRSO PRAZINS' ],
            [ '701', '704', 'SANTO TIRSO PRAZINS' ],
            [ '892', '894', 'SANTO TIRSO PRAZINS' ],
            [ '896', '900', 'SANTO TIRSO PRAZINS' ],
            [ '929', '944', 'SANTO TIRSO PRAZINS' ] ] },
        'Tabuadelo' => {
          '4835' => [
            [ '445', '467', 'TABUADELO' ],
            [ '597', '601', 'TABUADELO' ] ] },
        'Santo Estevo de Briteiros' => {
          '4805' => [
            [ '480', '483', 'SANTO ESTEVO BRITEIROS' ],
            [ '485', '486', 'SANTO ESTEVO BRITEIROS' ],
            [ '603', '603', 'SANTO ESTEVO BRITEIROS' ] ] },
        'Abao de So Tom' => {
          '4810' => [
            [ '655', '669', 'SO TOM ABAO' ],
            [ '671', '689', 'SO TOM ABAO' ],
            [ '691', '716', 'SO TOM ABAO' ] ] },
        'Ronfe' => {
          '4809' => [
            [ '013', '017', 'RONFE' ] ],
          '4805' => [
            [ '354', '445', 'RONFE' ] ] },
        'Gmeos' => {
          '4810' => [
            [ '567', '573', 'GMEOS GMR' ],
            [ '670', '670', 'GMEOS GMR' ],
            [ '776', '780', 'GMEOS GMR' ],
            [ '782', '791', 'GMEOS GMR' ] ] },
        'Airo Santa Maria' => {
          '4805' => [
            [ '475', '479', 'SANTA MARIA AIRO' ] ] },
        'Gominhes' => {
          '4800' => [
            [ '189', '189', 'GOMINHES' ],
            [ '421', '429', 'GOMINHES' ] ] },
        'Moreira de Cnegos' => {
          '4815' => [
            [ '253', '268', 'MOREIRA DE CNEGOS' ],
            [ '270', '350', 'MOREIRA DE CNEGOS' ],
            [ '352', '353', 'MOREIRA DE CNEGOS' ],
            [ '355', '370', 'MOREIRA DE CNEGOS' ],
            [ '641', '644', 'MOREIRA DE CNEGOS' ] ] },
        'Calvos' => {
          '4810' => [
            [ '601', '618', 'CALVOS GMR' ],
            [ '792', '806', 'CALVOS GMR' ] ] },
        'Gondomar' => {
          '4800' => [
            [ '195', '198', 'GONDOMAR GMR' ],
            [ '440', '440', 'GONDOMAR GMR' ] ] },
        'Souto So Salvador' => {
          '4800' => [
            [ '292', '293', 'SO SALVADOR SOUTO' ],
            [ '790', '790', 'SO SALVADOR SOUTO' ],
            [ '831', '850', 'SO SALVADOR SOUTO' ],
            [ '891', '891', 'SO SALVADOR SOUTO' ] ] },
        'Devesa' => {
          '4810' => [
            [ '781', '781', 'PINHEIRO GMR' ] ] },
        'Longos' => {
          '4805' => [
            [ '192', '210', 'LONGOS' ] ] },
        'Agra da Portela' => {
          '4765' => [
            [ '490', '554', 'SERZEDELO GMR' ] ] },
        'Leites' => {
          '4805' => [
            [ '187', '191', 'LEITES' ] ] },
        'Donim' => {
          '4805' => [
            [ '174', '182', 'DONIM' ],
            [ '584', '601', 'DONIM' ] ] },
        'Barco' => {
          '4809' => [
            [ '004', '005', 'BARCO GMR' ] ],
          '4805' => [
            [ '014', '018', 'BARCO GMR' ] ] },
        'Gona' => {
          '4800' => [
            [ '190', '194', 'GONA' ],
            [ '435', '435', 'GONA' ] ] },
        'Arca' => {
          '4810' => [
            [ '647', '654', 'PINHEIRO GMR' ],
            [ '717', '730', 'PINHEIRO GMR' ] ] },
        'Arosa' => {
          '4800' => [
            [ '211', '213', 'AROSA' ],
            [ '215', '222', 'AROSA' ] ] },
        'Serzedelo' => {
          '4765' => [
            [ '556', '587', 'SERZEDELO GMR' ] ] },
        'Briteiros So Salvador' => {
          '4805' => [
            [ '446', '453', 'SALVADOR BRITEIROS' ],
            [ '499', '499', 'SALVADOR BRITEIROS' ] ] },
        'So Clemente de Sande' => {
          '4805' => [
            [ '487', '492', 'SO CLEMENTE SANDE' ],
            [ '640', '640', 'SO CLEMENTE SANDE' ] ] },
        'Sande Vila Nova' => {
          '4805' => [
            [ '552', '582', 'VILA NOVA SANDE' ],
            [ '618', '621', 'VILA NOVA SANDE' ],
            [ '623', '629', 'VILA NOVA SANDE' ] ] },
        'Vermil' => {
          '4805' => [
            [ '546', '551', 'VERMIL' ] ] },
        'Montinho' => {
          '4810' => [
            [ '768', '775', 'PINHEIRO GMR' ] ] },
        'So Vicente de Oleiros' => {
          '4805' => [
            [ '542', '545', 'OLEIROS GMR' ],
            [ '604', '617', 'OLEIROS GMR' ],
            [ '632', '636', 'OLEIROS GMR' ] ] },
        'Rendufe' => {
          '4800' => [
            [ '199', '210', 'RENDUFE GMR' ],
            [ '223', '225', 'RENDUFE GMR' ],
            [ '251', '255', 'RENDUFE GMR' ],
            [ '530', '530', 'RENDUFE GMR' ] ] },
        'Bairro Fundo de Fomento da Habitao' => {
          '4835' => [
            [ '537', '567', 'GUIMARES' ] ] },
        'Guardizela' => {
          '4765' => [
            [ '401', '485', 'GUARDIZELA' ] ] },
        'Gandarela' => {
          '4835' => [
            [ '568', '594', 'GANDARELA' ] ] },
        'Serzedo' => {
          '4814' => [
            [ '520', '520', 'SERZEDO GMR' ] ],
          '4810' => [
            [ '841', '862', 'SERZEDO GMR' ] ] },
        'Ates' => {
          '4800' => [
            [ '226', '250', 'ATES GMR' ] ] },
        'Airo So Joo Batista' => {
          '4805' => [
            [ '493', '495', 'SO JOO BAPTISTA AIRO' ] ] },
        'Ponte' => {
          '4809' => [
            [ '011', '011', 'PONTE GMR' ] ],
          '4805' => [
            [ '211', '211', 'PONTE GMR' ],
            [ '213', '254', 'PONTE GMR' ],
            [ '256', '281', 'PONTE GMR' ],
            [ '283', '341', 'PONTE GMR' ],
            [ '343', '353', 'PONTE GMR' ],
            [ '637', '639', 'PONTE GMR' ] ] },
        'Santa Maria do Souto' => {
          '4800' => [
            [ '256', '262', 'SANTA MARIA SOUTO' ],
            [ '675', '675', 'SANTA MARIA SOUTO' ] ] },
        'Sande So Loureno' => {
          '4809' => [
            [ '002', '002', 'SO LOURENO SANDE' ] ],
          '4805' => [
            [ '496', '498', 'SO LOURENO SANDE' ],
            [ '500', '500', 'SO LOURENO SANDE' ],
            [ '583', '583', 'SO LOURENO SANDE' ] ] },
        'Varzielas' => {
          '4800' => [
            [ '405', '405', 'CASTELES GMR' ] ] }
              },
      'Celorico de Basto' => {
        'Adoufe' => {
          '4890' => [
            [ '361', '361', 'GMEOS CBT' ] ] },
        'Escola' => {
          '4890' => [
            [ '341', '352', 'GAGOS CBT' ] ] },
        'Alminhas' => {
          '4890' => [
            [ '201', '208', 'CELORICO DE BASTO' ] ] },
        'Ribeira' => {
          '4615' => [
            [ '818', '820', 'AGILDE' ] ] },
        'Cerdeira' => {
          '4890' => [
            [ '501', '509', 'RIBAS' ] ] },
        'Alem Parte' => {
          '4890' => [
            [ '411', '421', 'MOLARES' ] ] },
        'Alm do Rio' => {
          '4890' => [
            [ '561', '573', 'VEADE' ] ] },
        'Codeoso' => {
          '4890' => [
            [ '160', '173', 'CODEOSO' ] ] },
        'Boucinha' => {
          '4890' => [
            [ '011', '015', 'ARNOIA' ] ] },
        'Cabreira' => {
          '4890' => [
            [ '150', '158', 'CARVALHO CBT' ] ] },
        'Rua Nova' => {
          '4890' => [
            [ '550', '554', 'VALE DE BOURO' ] ] },
        'Afes' => {
          '4890' => [
            [ '101', '111', 'BORBA DE MONTANHA' ] ] },
        'Barrosinho' => {
          '4890' => [
            [ '540', '549', 'BASTO (SO CLEMENTE)' ] ] },
        'Mosqueiros' => {
          '4890' => [
            [ '274', '290', 'CELORICO DE BASTO' ] ] },
        'Bouca' => {
          '4890' => [
            [ '471', '486', 'OURILHE' ] ] },
        'Carvalhal' => {
          '4890' => [
            [ '441', '456', 'MOREIRA DO CASTELO' ] ] },
        'Outeiro' => {
          '4615' => [
            [ '814', '816', 'AGILDE' ] ] },
        'Alfarela' => {
          '4890' => [
            [ '121', '133', 'CAARILHE' ] ] },
        'So Jorge' => {
          '4890' => [
            [ '036', '056', 'ARNOIA' ] ] },
        'Cerdeirinhas' => {
          '4890' => [
            [ '140', '147', 'CANEDO DE BASTO' ] ] },
        'Fojo' => {
          '4890' => [
            [ '027', '033', 'ARNOIA' ] ] },
        'So Romo' => {
          '4890' => [
            [ '181', '182', 'CORGO' ] ] },
        'Britelo' => {
          '4890' => [
            [ '220', '240', 'CELORICO DE BASTO' ] ] },
        'Aveia' => {
          '4890' => [
            [ '521', '530', 'BASTO (SANTA TECLA)' ] ] },
        'Corredoura' => {
          '4890' => [
            [ '262', '272', 'CELORICO DE BASTO' ] ] },
        'Casal Diogo' => {
          '4890' => [
            [ '018', '025', 'ARNOIA' ] ] },
        'Lobo' => {
          '4820' => [
            [ '840', '840', 'REGO' ] ] },
        'Alijo' => {
          '4615' => [
            [ '801', '812', 'AGILDE' ] ] },
        'Alm Rio' => {
          '4890' => [
            [ '363', '380', 'GMEOS CBT' ] ] },
        'Assento' => {
          '4890' => [
            [ '301', '330', 'FERVENA' ],
            [ '391', '397', 'INFESTA CBT' ] ] },
        'Carril' => {
          '4890' => [
            [ '260', '260', 'CELORICO DE BASTO' ] ] }
                }
         },
    'Vila Real' => {
      'Montalegre' => {
        'Cela' => {
          '5470' => [
            [ '331', '334', 'OUTEIRO MTR' ],
            [ '361', '364', 'PARADELA MTR' ] ] },
        'Antigo (viade)' => {
          '5470' => [
            [ '521', '529', 'VIADE DE BAIXO' ] ] },
        'Castelo' => {
          '5470' => [
            [ '381', '385', 'PONDRAS' ] ] },
        'Tabuadela' => {
          '5470' => [
            [ '441', '444', 'SALTO' ] ] },
        'Bustelo' => {
          '5470' => [
            [ '541', '543', 'VILA DA PONTE MTR' ] ] },
        'Padroso' => {
          '5470' => [
            [ '350', '350', 'PADROSO MTR' ] ] },
        'Bairro (ferral)' => {
          '5470' => [
            [ '121', '129', 'FERRAL' ] ] },
        'Solveira' => {
          '5470' => [
            [ '480', '480', 'SOLVEIRA' ] ] },
        'Carvalhais' => {
          '5470' => [
            [ '301', '302', 'MORGADE' ] ] },
        'Santo Andr' => {
          '5470' => [
            [ '450', '450', 'SANTO ANDR MTR' ] ] },
        'Padornelos' => {
          '5470' => [
            [ '341', '342', 'PADORNELOS' ] ] },
        'Coveles' => {
          '5470' => [
            [ '091', '092', 'COVELES' ] ] },
        'Currais' => {
          '5470' => [
            [ '391', '393', 'REIGOSO MTR' ] ] },
        'Morgade' => {
          '5470' => [
            [ '304', '305', 'MORGADE' ] ] },
        'Lamacha' => {
          '5470' => [
            [ '321', '323', 'NEGRES' ] ] },
        'Ameal' => {
          '5470' => [
            [ '401', '422', 'SALTO' ] ] },
        'Arcos' => {
          '5470' => [
            [ '051', '057', 'CERVOS' ] ] },
        'Gorda' => {
          '5470' => [
            [ '172', '172', 'CH' ] ] },
        'Pites das Junias' => {
          '5470' => [
            [ '370', '370', 'PITES DAS JUNIAS' ] ] },
        'Covelo do Gerez' => {
          '5470' => [
            [ '101', '105', 'COVELO DO GEREZ' ] ] },
        'Seara' => {
          '5470' => [
            [ '430', '430', 'SALTO' ] ] },
        'Cambeses do Rio' => {
          '5470' => [
            [ '041', '042', 'CAMBESES DO RIO' ] ] },
        'Codeoso' => {
          '5470' => [
            [ '171', '171', 'MEIXEDO MTR' ],
            [ '501', '504', 'VENDA NOVA' ] ] },
        'Azevedo' => {
          '5470' => [
            [ '011', '025', 'CABRIL MTR' ] ] },
        'Mourilhe' => {
          '5470' => [
            [ '311', '312', 'MOURILHE' ] ] },
        'Dones' => {
          '5470' => [
            [ '110', '110', 'DONES' ] ] },
        'Fervidelas' => {
          '5470' => [
            [ '141', '142', 'FERVIDELAS' ] ] },
        'Meixedo' => {
          '5470' => [
            [ '173', '173', 'MEIXEDO MTR' ] ] },
        'Fies do Rio' => {
          '5470' => [
            [ '151', '152', 'FIES DO RIO' ] ] },
        'Casais da Veiga' => {
          '5470' => [
            [ '201', '255', 'MONTALEGRE' ] ] },
        'Antigo (serraquinhos)' => {
          '5470' => [
            [ '462', '466', 'SARRAQUINHOS' ] ] },
        'Tourm' => {
          '5470' => [
            [ '490', '490', 'TOURM' ] ] },
        'Contim' => {
          '5470' => [
            [ '081', '083', 'CONTIM' ] ] },
        'So Miguel de Vilar de Perdizes' => {
          '5470' => [
            [ '461', '461', 'VILAR PERDIZES (S MIGUEL)' ] ] },
        'Medeiros' => {
          '5470' => [
            [ '067', '073', 'CH' ] ] },
        'Aldeia Nova do Barroso' => {
          '5470' => [
            [ '062', '065', 'CH' ] ] },
        'Pomar da Rainha' => {
          '5470' => [
            [ '424', '426', 'SALTO' ] ] },
        'Gralhas' => {
          '5470' => [
            [ '160', '160', 'GRALHAS' ] ] },
        'Meixide' => {
          '5470' => [
            [ '180', '180', 'MEIXIDE' ] ] },
        'Sezelhe' => {
          '5470' => [
            [ '471', '472', 'SEZELHE' ] ] }
             },
      'Boticas' => {
        'Boticas' => {
          '5460' => [
            [ '301', '309', 'BOTICAS' ] ] },
        'Atilho' => {
          '5460' => [
            [ '020', '020', 'ALTURAS DO BARROSO' ] ] },
        'Carreira da Lebre' => {
          '5460' => [
            [ '125', '125', 'BEA' ] ] },
        'Sapelos' => {
          '5460' => [
            [ '501', '502', 'SAPIOS' ] ] },
        'Virtelo' => {
          '5460' => [
            [ '350', '350', 'CERDEDO' ] ] },
        'Codessoso' => {
          '5460' => [
            [ '370', '370', 'CODESSOSO' ] ] },
        'Sangunhedo' => {
          '5460' => [
            [ '330', '330', 'BOTICAS' ] ] },
        'Bea' => {
          '5460' => [
            [ '120', '120', 'BEA' ] ] },
        'Lavradas' => {
          '5460' => [
            [ '135', '135', 'BEA' ] ] },
        'Carvalhelhos' => {
          '5460' => [
            [ '130', '130', 'BEA' ] ] },
        'Antigo' => {
          '5460' => [
            [ '411', '418', 'DORNELAS BTC' ] ] },
        'Fies do Tmega' => {
          '5460' => [
            [ '431', '432', 'FIES DO TAMEGA' ] ] },
        'Ardos' => {
          '5460' => [
            [ '100', '100', 'ARDOS' ] ] },
        'Bobadela' => {
          '5460' => [
            [ '210', '210', 'BOBADELA BTC' ] ] },
        'Granja' => {
          '5460' => [
            [ '451', '452', 'GRANJA BTC' ] ] },
        'Agrelos' => {
          '5460' => [
            [ '491', '495', 'SO SALVADOR DE VIVEIRO' ] ] },
        'Covas do Barroso' => {
          '5460' => [
            [ '381', '383', 'COVAS DO BARROSO' ] ] },
        'Nogueira' => {
          '5460' => [
            [ '220', '220', 'BOBADELA BTC' ] ] },
        'Pinho' => {
          '5460' => [
            [ '471', '473', 'PINHO BTC' ] ] },
        'Antigo de Curros' => {
          '5460' => [
            [ '400', '400', 'CURROS BTC' ] ] },
        'Vilarinho Seco' => {
          '5460' => [
            [ '030', '030', 'ALTURAS DO BARROSO' ] ] },
        'Seirro' => {
          '5460' => [
            [ '155', '155', 'BEA' ] ] },
        'Torneiros' => {
          '5460' => [
            [ '160', '160', 'BEA' ] ] },
        'Alturas do Barroso' => {
          '5460' => [
            [ '010', '010', 'ALTURAS DO BARROSO' ] ] },
        'Vilarinho da M' => {
          '5460' => [
            [ '165', '165', 'BEA' ] ] },
        'Quintas' => {
          '5460' => [
            [ '150', '150', 'BEA' ] ] },
        'Pinhal Novo' => {
          '5460' => [
            [ '145', '145', 'BEA' ] ] },
        'Eir' => {
          '5460' => [
            [ '320', '320', 'BOTICAS' ] ] },
        'Carvalho' => {
          '5460' => [
            [ '511', '512', 'VILAR BTC' ] ] },
        'Minas de Bea' => {
          '5460' => [
            [ '140', '140', 'BEA' ] ] }
          },
      'Santa Marta de Penaguio' => {
        'Balado' => {
          '5030' => [
            [ '251', '264', 'LOUREDO SMP' ] ] },
        'Santa Marta de Penaguio' => {
          '5030' => [
            [ '470', '481', 'SANTA MARTA DE PENAGUIO' ] ] },
        'Balota de Baixo' => {
          '5030' => [
            [ '105', '112', 'FONTES SMP' ] ] },
        'Aveleira' => {
          '5030' => [
            [ '101', '103', 'FONTES SMP' ],
            [ '401', '434', 'LOBRIGOS (S J BAPTISTA)' ] ] },
        'Alminhas' => {
          '5030' => [
            [ '351', '381', 'SANHOANE SMP' ] ] },
        'Santa Ana' => {
          '5030' => [
            [ '383', '386', 'SANHOANE SMP' ] ] },
        'Casal' => {
          '5030' => [
            [ '201', '201', 'FORNELOS SMP' ] ] },
        'Alto' => {
          '5030' => [
            [ '451', '462', 'SANTA MARTA DE PENAGUIO' ] ] },
        'So Miguel de Lobrigos' => {
          '5030' => [
            [ '495', '496', 'SANTA MARTA DE PENAGUIO' ] ] },
        'Fornelos' => {
          '5030' => [
            [ '210', '238', 'FORNELOS SMP' ] ] },
        'Fontes' => {
          '5030' => [
            [ '114', '138', 'FONTES SMP' ],
            [ '494', '494', 'FONTES SMP' ] ] },
        'Alvaes do Corgo' => {
          '5030' => [
            [ '011', '026', 'ALVAES DO CORGO' ] ] },
        'Assento' => {
          '5030' => [
            [ '046', '083', 'CUMEEIRA SMP' ] ] },
        'Atalho' => {
          '5030' => [
            [ '551', '573', 'SEVER SMP' ] ] },
        'Adega' => {
          '5030' => [
            [ '301', '328', 'MEDRES' ] ] }
               },
      'Vila Real' => {
        'Paulos' => {
          '5000' => [
            [ '101', '105', 'VILA REAL' ] ] },
        'Dornelas' => {
          '5000' => [
            [ '141', '142', 'LAMAS DE OLO' ] ] },
        'Jorjais' => {
          '5000' => [
            [ '031', '043', 'ANDRES' ] ] },
        'Prado (Lordelo)' => {
          '5000' => [
            [ '227', '233', 'VILA REAL' ] ] },
        'Minas de Vila Cova' => {
          '5000' => [
            [ '761', '761', 'VILA COVA VRL' ] ] },
        'Vilarinho de Tanha' => {
          '5000' => [
            [ '011', '011', 'ABAAS' ] ] },
        'Penada' => {
          '5000' => [
            [ '771', '776', 'VILA REAL' ] ] },
        'Lordelo' => {
          '5000' => [
            [ '161', '196', 'VILA REAL' ] ] },
        'Moinhos' => {
          '5000' => [
            [ '401', '451', 'VILA REAL' ] ] },
        'Avenida' => {
          '5000' => [
            [ '210', '224', 'VILA REAL' ] ] },
        'Gache' => {
          '5000' => [
            [ '131', '132', 'LAMARES' ] ] },
        'Faijo' => {
          '5000' => [
            [ '741', '748', 'TORGUEDA' ] ] },
        'Ranginha' => {
          '5000' => [
            [ '081', '083', 'CONSTANTIM VRL' ] ] },
        'Carpintaria' => {
          '5000' => [
            [ '111', '111', 'GUIES' ] ] },
        'Torneiros' => {
          '5000' => [
            [ '051', '052', 'VILA REAL' ] ] },
        'Carrazeda' => {
          '5000' => [
            [ '091', '092', 'ERMIDA VRL' ] ] },
        'Alvites' => {
          '5000' => [
            [ '351', '356', 'VILA REAL' ] ] },
        'Calada' => {
          '5000' => [
            [ '021', '028', 'ADOUFE' ] ] },
        'Areias' => {
          '5000' => [
            [ '311', '336', 'MONDRES' ] ] },
        'Fraga de Almotelia' => {
          '5000' => [
            [ '061', '063', 'VILA REAL' ] ] },
        'Bairro de So Miguel' => {
          '5000' => [
            [ '481', '483', 'PENA' ] ] },
        'Quinta' => {
          '5000' => [
            [ '491', '491', 'QUINTA' ] ] },
        'Samard' => {
          '5000' => [
            [ '781', '781', 'VILARINHO DA SAMARD' ] ] },
        'Boavista' => {
          '5000' => [
            [ '071', '072', 'CAMPE' ] ] },
        'Vila Nova de Cima' => {
          '5004' => [
            [ '001', '015', 'VILA REAL' ] ] },
        'Vila Real' => {
          '5000' => [
            [ '253', '310', 'VILA REAL' ],
            [ '453', '457', 'VILA REAL' ],
            [ '460', '460', 'VILA REAL' ],
            [ '462', '465', 'VILA REAL' ],
            [ '467', '472', 'VILA REAL' ],
            [ '501', '547', 'VILA REAL' ],
            [ '549', '560', 'VILA REAL' ],
            [ '565', '671', 'VILA REAL' ],
            [ '677', '704', 'VILA REAL' ],
            [ '706', '730', 'VILA REAL' ],
            [ '886', '893', 'VILA REAL' ] ] },
        'Abobeleira' => {
          '5000' => [
            [ '358', '360', 'VILA REAL' ] ] },
        'Justes' => {
          '5000' => [
            [ '121', '121', 'JUSTES' ] ] },
        'So Cosme' => {
          '5000' => [
            [ '731', '732', 'SO TOM DO CASTELO' ] ] },
        'Magalh' => {
          '5000' => [
            [ '013', '014', 'ABAAS' ] ] },
        'Compra' => {
          '5000' => [
            [ '362', '376', 'VILA REAL' ] ] },
        'Carro Queimado' => {
          '5000' => [
            [ '751', '752', 'VALE DE NOGUEIRAS' ] ] },
        'Quinta da Ribeira' => {
          '5000' => [
            [ '391', '391', 'NOGUEIRA VRL' ] ] }
            },
      'Alij' => {
        'Balsa' => {
          '5070' => [
            [ '551', '558', 'VILA VERDE ALJ' ] ] },
        'Mondego' => {
          '5070' => [
            [ '276', '278', 'FAVAIOS' ] ] },
        'Quinta da Carcuda' => {
          '5070' => [
            [ '150', '155', 'ALIJ' ] ] },
        'Eirinhas' => {
          '5070' => [
            [ '240', '242', 'CASTEDO ALJ' ] ] },
        'Campo de Tiro' => {
          '5070' => [
            [ '070', '072', 'ALIJ' ] ] },
        'Alto de Pegarinhos' => {
          '5070' => [
            [ '301', '305', 'PEGARINHOS' ] ] },
        'Castedo' => {
          '5070' => [
            [ '224', '238', 'CASTEDO ALJ' ] ] },
        'Santa Eugnia' => {
          '5070' => [
            [ '411', '411', 'SANTA EUGNIA' ] ] },
        'Alto de Cotas' => {
          '5070' => [
            [ '251', '252', 'COTAS' ] ] },
        'Alminhas' => {
          '5070' => [
            [ '221', '222', 'CASTEDO ALJ' ] ] },
        'Alij' => {
          '5070' => [
            [ '011', '064', 'ALIJ' ] ] },
        'Quinta do Bonfim' => {
          '5085' => [
            [ '060', '060', 'PINHO' ] ] },
        'Safres' => {
          '5070' => [
            [ '460', '460', 'SO MAMEDE DE RIBATUA' ] ] },
        'So Mamede de Ribatua' => {
          '5070' => [
            [ '471', '503', 'SO MAMEDE DE RIBATUA' ] ] },
        'Favaios' => {
          '5070' => [
            [ '261', '274', 'FAVAIOS' ] ] },
        'Carlo' => {
          '5070' => [
            [ '201', '205', 'CARLO' ] ] },
        'Cruz' => {
          '5085' => [
            [ '010', '010', 'CASAL DE LOIVOS' ] ] },
        'Senhora do Couto' => {
          '5085' => [
            [ '120', '120', 'VILARINHO DE COTAS' ] ] },
        'Alto do Carvalho' => {
          '5070' => [
            [ '531', '534', 'VILA CH ALJ' ] ] },
        'Granja' => {
          '5070' => [
            [ '081', '085', 'ALIJ' ],
            [ '087', '100', 'ALIJ' ] ] },
        'Agrelos' => {
          '5070' => [
            [ '341', '343', 'SANFINS DO DOURO' ] ] },
        'Presandes' => {
          '5070' => [
            [ '121', '133', 'ALIJ' ] ] },
        'Amieiro' => {
          '5070' => [
            [ '171', '188', 'AMIEIRO' ] ] },
        'Cal de Bois' => {
          '5070' => [
            [ '311', '314', 'PPULO' ] ] },
        'Rapadoura' => {
          '5070' => [
            [ '321', '323', 'RIBALONGA ALJ' ] ] },
        'Vale de Mendiz' => {
          '5085' => [
            [ '100', '105', 'VALE DE MENDIZ' ] ] },
        'Cabeda' => {
          '5070' => [
            [ '571', '576', 'VILAR DE MAADA' ] ] },
        'Chousas' => {
          '5070' => [
            [ '451', '456', 'SO MAMEDE DE RIBATUA' ] ] },
        'Pinho' => {
          '5085' => [
            [ '031', '044', 'PINHO' ] ] },
        'Sanfins do Douro' => {
          '5070' => [
            [ '351', '391', 'SANFINS DO DOURO' ] ] }
              },
      'Chaves' => {
        'Moure' => {
          '5400' => [
            [ '826', '827', 'VILELA DO TMEGA' ] ] },
        'Cela' => {
          '5400' => [
            [ '611', '613', 'CELA CHV' ] ] },
        'Anelhe' => {
          '5425' => [
            [ '011', '013', 'ANELHE' ] ] },
        'Agostem' => {
          '5400' => [
            [ '756', '764', 'SO PEDRO DE AGOSTEM' ] ] },
        'Lugar do Barroco' => {
          '5400' => [
            [ '574', '582', 'CHAVES' ] ] },
        'Campelado' => {
          '5400' => [
            [ '071', '071', 'SANTA CRUZ/TRINDADE' ],
            [ '328', '329', 'SANTA CRUZ/TRINDADE' ],
            [ '350', '350', 'SANTA CRUZ/TRINDADE' ],
            [ '375', '375', 'SANTA CRUZ/TRINDADE' ],
            [ '562', '562', 'SANTA CRUZ/TRINDADE' ] ] },
        'Fornos' => {
          '5425' => [
            [ '271', '273', 'SELHARIZ' ] ] },
        'Vila Verde' => {
          '5425' => [
            [ '230', '230', 'OURA' ] ] },
        'Argemil' => {
          '5400' => [
            [ '796', '798', 'TRAVANCAS' ] ] },
        'Matosinhos' => {
          '5400' => [
            [ '740', '740', 'SANTA LEOCDIA CHV' ] ] },
        'Bustelo' => {
          '5400' => [
            [ '607', '607', 'BUSTELO CHV' ] ] },
        'Bolideira' => {
          '5400' => [
            [ '606', '606', 'BOBADELA CHV' ] ] },
        'Paradela de Monforte' => {
          '5400' => [
            [ '725', '725', 'PARADELA CHV' ] ] },
        'Calvo' => {
          '5400' => [
            [ '608', '609', 'CALVO CHV' ] ] },
        'guas Frias' => {
          '5400' => [
            [ '601', '605', 'GUAS FRIAS' ] ] },
        'Oucidres' => {
          '5400' => [
            [ '658', '659', 'OUCIDRES' ] ] },
        'Lama de Arcos' => {
          '5400' => [
            [ '636', '637', 'LAMA DE ARCOS' ] ] },
        'Tronco' => {
          '5400' => [
            [ '800', '800', 'TRONCO' ] ] },
        'Mairos' => {
          '5400' => [
            [ '640', '640', 'MAIROS' ] ] },
        'Codeais' => {
          '5400' => [
            [ '631', '633', 'FAIES' ] ] },
        'Valongo' => {
          '5400' => [
            [ '229', '230', 'SANTA CRUZ/TRINDADE' ],
            [ '301', '301', 'SANTA CRUZ/TRINDADE' ],
            [ '339', '339', 'SANTA CRUZ/TRINDADE' ],
            [ '535', '536', 'SANTA CRUZ/TRINDADE' ],
            [ '559', '559', 'SANTA CRUZ/TRINDADE' ] ] },
        'Santa Cruz' => {
          '5400' => [
            [ '410', '410', 'SANTA CRUZ/TRINDADE' ],
            [ '674', '680', 'SANTA CRUZ/TRINDADE' ],
            [ '681', '681', 'CHAVES' ],
            [ '682', '683', 'OUTEIRO SECO' ],
            [ '684', '686', 'SANTA CRUZ/TRINDADE' ],
            [ '687', '687', 'OUTEIRO SECO' ],
            [ '688', '690', 'SANTA CRUZ/TRINDADE' ],
            [ '691', '691', 'OUTEIRO SECO' ],
            [ '692', '697', 'SANTA CRUZ/TRINDADE' ],
            [ '698', '698', 'OUTEIRO SECO' ],
            [ '709', '711', 'OUTEIRO SECO' ],
            [ '712', '714', 'SANTA CRUZ/TRINDADE' ],
            [ '715', '715', 'OUTEIRO SECO' ],
            [ '716', '717', 'SANTA CRUZ/TRINDADE' ],
            [ '718', '718', 'OUTEIRO SECO' ],
            [ '719', '720', 'SANTA CRUZ/TRINDADE' ],
            [ '831', '837', 'SANTA CRUZ/TRINDADE' ] ] },
        'Seara Velha' => {
          '5400' => [
            [ '780', '780', 'SEARA VELHA' ] ] },
        'Trindade' => {
          '5400' => [
            [ '442', '443', 'SANTA CRUZ/TRINDADE' ] ] },
        'Eiras' => {
          '5400' => [
            [ '623', '624', 'EIRAS CHV' ] ] },
        'Loivos' => {
          '5425' => [
            [ '051', '076', 'LOIVOS' ],
            [ '090', '102', 'LOIVOS' ] ] },
        'Casas Novas' => {
          '5400' => [
            [ '727', '729', 'REDONDELO' ] ] },
        'Agraes' => {
          '5425' => [
            [ '251', '254', 'PVOA DE AGRAES' ] ] },
        'Aveleda' => {
          '5400' => [
            [ '771', '774', 'SO VICENTE CHV' ] ] },
        'Moreiras' => {
          '5400' => [
            [ '643', '644', 'MOREIRAS' ] ] },
        'Cocanha' => {
          '5400' => [
            [ '135', '135', 'SANTA CRUZ/TRINDADE' ],
            [ '217', '217', 'SANTA CRUZ/TRINDADE' ],
            [ '299', '299', 'SANTA CRUZ/TRINDADE' ],
            [ '307', '307', 'SANTA CRUZ/TRINDADE' ],
            [ '407', '408', 'SANTA CRUZ/TRINDADE' ],
            [ '538', '538', 'SANTA CRUZ/TRINDADE' ],
            [ '671', '671', 'SANTA CRUZ/TRINDADE' ],
            [ '699', '708', 'SANTA CRUZ/TRINDADE' ] ] },
        'Parada' => {
          '5400' => [
            [ '732', '736', 'SANFINS CHV' ] ] },
        'Vidago' => {
          '5425' => [
            [ '301', '349', 'VIDAGO' ] ] },
        'Pereira de Selo' => {
          '5425' => [
            [ '501', '502', 'VILAS BOAS CHV' ] ] },
        'Roriz' => {
          '5400' => [
            [ '730', '730', 'RORIZ CHV' ] ] },
        'Curalha' => {
          '5400' => [
            [ '620', '620', 'CURALHA' ] ] },
        'Santo Estevo' => {
          '5400' => [
            [ '750', '750', 'SANTO ESTEVO CHV' ] ] },
        'Quintela' => {
          '5400' => [
            [ '042', '042', 'SANTA CRUZ/TRINDADE' ] ] },
        'Alto da Trindade' => {
          '5400' => [
            [ '182', '182', 'SANTA CRUZ/TRINDADE' ],
            [ '196', '196', 'SANTA CRUZ/TRINDADE' ],
            [ '248', '248', 'SANTA CRUZ/TRINDADE' ],
            [ '281', '282', 'SANTA CRUZ/TRINDADE' ],
            [ '316', '316', 'SANTA CRUZ/TRINDADE' ],
            [ '341', '341', 'SANTA CRUZ/TRINDADE' ],
            [ '378', '379', 'SANTA CRUZ/TRINDADE' ],
            [ '493', '494', 'SANTA CRUZ/TRINDADE' ] ] },
        'Oura' => {
          '5425' => [
            [ '201', '222', 'OURA' ] ] },
        'Estrada do Seara' => {
          '5400' => [
            [ '672', '673', 'OUTEIRO SECO' ] ] },
        'Agrela' => {
          '5400' => [
            [ '627', '628', 'ERVEDEDO' ] ] },
        'Fonte  do Leite' => {
          '5400' => [
            [ '258', '259', 'SANTA CRUZ/TRINDADE' ],
            [ '276', '276', 'SANTA CRUZ/TRINDADE' ],
            [ '358', '359', 'SANTA CRUZ/TRINDADE' ],
            [ '429', '431', 'SANTA CRUZ/TRINDADE' ] ] },
        'Chaves' => {
          '5400' => [
            [ '032', '032', 'CHAVES' ],
            [ '033', '033', 'SANTA CRUZ/TRINDADE' ],
            [ '034', '036', 'CHAVES' ],
            [ '037', '037', 'SANTA CRUZ/TRINDADE' ],
            [ '038', '041', 'CHAVES' ],
            [ '043', '044', 'CHAVES' ],
            [ '045', '048', 'SANTA CRUZ/TRINDADE' ],
            [ '049', '050', 'CHAVES' ],
            [ '051', '051', 'SANTA CRUZ/TRINDADE' ],
            [ '052', '058', 'CHAVES' ],
            [ '059', '059', 'SANTA CRUZ/TRINDADE' ],
            [ '060', '060', 'CHAVES' ],
            [ '061', '061', 'SANTA CRUZ/TRINDADE' ],
            [ '062', '070', 'CHAVES' ],
            [ '072', '084', 'CHAVES' ],
            [ '085', '085', 'SANTA CRUZ/TRINDADE' ],
            [ '086', '099', 'CHAVES' ],
            [ '101', '134', 'CHAVES' ],
            [ '136', '150', 'CHAVES' ],
            [ '151', '153', 'SANTA CRUZ/TRINDADE' ],
            [ '154', '169', 'CHAVES' ],
            [ '171', '180', 'CHAVES' ],
            [ '183', '184', 'CHAVES' ],
            [ '186', '195', 'CHAVES' ],
            [ '197', '216', 'CHAVES' ],
            [ '218', '226', 'CHAVES' ],
            [ '228', '228', 'CHAVES' ],
            [ '231', '247', 'CHAVES' ],
            [ '249', '257', 'CHAVES' ],
            [ '260', '275', 'CHAVES' ],
            [ '277', '280', 'CHAVES' ],
            [ '283', '298', 'CHAVES' ],
            [ '300', '300', 'CHAVES' ],
            [ '303', '306', 'CHAVES' ],
            [ '309', '315', 'CHAVES' ],
            [ '317', '327', 'CHAVES' ],
            [ '330', '330', 'CHAVES' ],
            [ '331', '331', 'SANTA CRUZ/TRINDADE' ],
            [ '332', '338', 'CHAVES' ],
            [ '340', '340', 'CHAVES' ],
            [ '342', '349', 'CHAVES' ],
            [ '351', '357', 'CHAVES' ],
            [ '360', '374', 'CHAVES' ],
            [ '376', '377', 'CHAVES' ],
            [ '380', '406', 'CHAVES' ],
            [ '409', '409', 'CHAVES' ],
            [ '411', '428', 'CHAVES' ],
            [ '432', '432', 'CHAVES' ],
            [ '433', '433', 'SANTA CRUZ/TRINDADE' ],
            [ '434', '441', 'CHAVES' ],
            [ '444', '492', 'CHAVES' ],
            [ '495', '529', 'CHAVES' ],
            [ '531', '534', 'CHAVES' ],
            [ '537', '537', 'CHAVES' ],
            [ '539', '558', 'CHAVES' ],
            [ '560', '561', 'CHAVES' ],
            [ '563', '565', 'CHAVES' ],
            [ '567', '572', 'CHAVES' ] ] },
        'Noval' => {
          '5400' => [
            [ '791', '792', 'SOUTELO CHV' ] ] },
        'Limos' => {
          '5400' => [
            [ '752', '754', 'SO JULIO DE MONTENEGRO' ] ] },
        'Cimo de Vila Castanheira' => {
          '5400' => [
            [ '616', '617', 'CIMO VILA DE CASTANHEIRA' ] ] },
        'Abobeleira' => {
          '5400' => [
            [ '001', '030', 'CHAVES' ] ] },
        'Alanhosa' => {
          '5400' => [
            [ '646', '656', 'NOGUEIRA DA MONTANHA' ] ] },
        'Vilela Seca' => {
          '5400' => [
            [ '820', '820', 'VILELA SECA' ] ] },
        'Curral de Vacas' => {
          '5400' => [
            [ '745', '745', 'SANTO ANTNIO DE MONFORTE' ] ] },
        'Vilarinho das Paranheiras' => {
          '5425' => [
            [ '401', '401', 'VILARINHO DAS PARANHEIRAS' ] ] },
        'Cambedo' => {
          '5400' => [
            [ '811', '814', 'VILARELHO DA RAIA' ] ] },
        'Soutelinho da Raia' => {
          '5400' => [
            [ '785', '785', 'SOUTELINHO DA RAIA' ] ] },
        'Vila Verde da Raia' => {
          '5400' => [
            [ '805', '805', 'VILA VERDE DA RAIA' ] ] },
        'Arcoss' => {
          '5425' => [
            [ '021', '028', 'ARCOSS' ] ] }
               },
      'Ribeira de Pena' => {
        'Bragadas' => {
          '4870' => [
            [ '251', '253', 'STO ALEIXO DE ALM-TAMEGA' ] ] },
        'Alij' => {
          '4870' => [
            [ '021', '024', 'CANEDO RPN' ] ] },
        'Ribeira de Pena' => {
          '4870' => [
            [ '150', '163', 'RIBEIRA DE PENA' ] ] },
        'Friume' => {
          '4870' => [
            [ '122', '122', 'RIBEIRA DE PENA' ] ] },
        'Portela da Trofa' => {
          '4870' => [
            [ '128', '131', 'RIBEIRA DE PENA' ] ] },
        'Lamelas' => {
          '4870' => [
            [ '106', '119', 'RIBEIRA DE PENA' ] ] },
        'Mouro' => {
          '4870' => [
            [ '049', '052', 'CERVA' ] ] },
        'Jalongo' => {
          '4870' => [
            [ '077', '082', 'LIMES' ] ] },
        'Regueiro' => {
          '4870' => [
            [ '133', '133', 'RIBEIRA DE PENA' ] ] },
        'Alvadia' => {
          '4870' => [
            [ '011', '013', 'ALVADIA' ] ] },
        'Ruival' => {
          '4870' => [
            [ '180', '180', 'RIBEIRA DE PENA' ] ] },
        'Quintela' => {
          '4870' => [
            [ '054', '059', 'CERVA' ] ] },
        'Ribeira de Cima' => {
          '4870' => [
            [ '135', '135', 'RIBEIRA DE PENA' ] ] },
        'Olaria' => {
          '4870' => [
            [ '124', '126', 'RIBEIRA DE PENA' ] ] },
        'Minas de So Joo' => {
          '4870' => [
            [ '031', '033', 'CERVA' ] ] },
        'Ancies' => {
          '4870' => [
            [ '101', '104', 'RIBEIRA DE PENA' ] ] },
        'Senra' => {
          '4870' => [
            [ '182', '188', 'RIBEIRA DE PENA' ] ] },
        'Abelheira' => {
          '4870' => [
            [ '201', '225', 'SANTA MARINHA RPN' ] ] },
        'Azeveda' => {
          '4870' => [
            [ '071', '075', 'LIMES' ] ] },
        'Asnela' => {
          '4870' => [
            [ '035', '047', 'CERVA' ] ] }
            },
      'Meso Frio' => {
        'Jardim' => {
          '5040' => [
            [ '162', '165', 'CIDADELHE MSF' ] ] },
        'Camatoga' => {
          '5040' => [
            [ '010', '010', 'VILA MARIM MSF' ] ] },
        'Vila Verde' => {
          '5040' => [
            [ '337', '337', 'MESO FRIO' ] ] },
        'Caldas de Moledo' => {
          '5040' => [
            [ '225', '228', 'OLIVEIRA MSF' ] ] },
        'Minhoto' => {
          '5040' => [
            [ '469', '475', 'VILA MARIM MSF' ] ] },
        'Bairrinho' => {
          '5040' => [
            [ '102', '102', 'BARQUEIROS MSF' ] ] },
        'Estrada' => {
          '5040' => [
            [ '460', '462', 'VILA MARIM MSF' ] ] },
        'Porto Rei' => {
          '5040' => [
            [ '117', '118', 'BARQUEIROS MSF' ] ] },
        'Eira' => {
          '5040' => [
            [ '205', '208', 'OLIVEIRA MSF' ] ] },
        'Rabela' => {
          '5040' => [
            [ '217', '223', 'OLIVEIRA MSF' ] ] },
        'Imaginrio' => {
          '5040' => [
            [ '349', '356', 'MESO FRIO' ] ] },
        'Sobreiro' => {
          '5040' => [
            [ '171', '174', 'CIDADELHE MSF' ] ] },
        'Bogalheira' => {
          '5040' => [
            [ '151', '152', 'CIDADELHE MSF' ] ] },
        'Ribeiro' => {
          '5040' => [
            [ '169', '169', 'CIDADELHE MSF' ] ] },
        'Meso Frio' => {
          '5040' => [
            [ '289', '298', 'MESO FRIO' ],
            [ '401', '410', 'MESO FRIO' ],
            [ '412', '412', 'MESO FRIO' ] ] },
        'Lama do Monte' => {
          '5040' => [
            [ '112', '112', 'BARQUEIROS MSF' ] ] },
        'Freixieiro' => {
          '5040' => [
            [ '109', '109', 'BARQUEIROS MSF' ] ] },
        'Corredoura' => {
          '5040' => [
            [ '457', '458', 'VILA MARIM MSF' ] ] },
        'Bamba' => {
          '5040' => [
            [ '201', '203', 'OLIVEIRA MSF' ] ] },
        'So Martinho' => {
          '5040' => [
            [ '421', '422', 'MESO FRIO' ] ] },
        'Lodoeiro' => {
          '5040' => [
            [ '466', '467', 'VILA MARIM MSF' ] ] },
        'Vale do Couto' => {
          '5040' => [
            [ '300', '300', 'MESO FRIO' ] ] },
        'Barronca' => {
          '5040' => [
            [ '452', '454', 'VILA MARIM MSF' ] ] },
        'Bernardo' => {
          '5040' => [
            [ '104', '105', 'BARQUEIROS MSF' ] ] },
        'Banduja' => {
          '5040' => [
            [ '302', '325', 'MESO FRIO' ] ] },
        'Boavista' => {
          '5040' => [
            [ '436', '450', 'VILA MARIM MSF' ] ] },
        'Outeiro' => {
          '5040' => [
            [ '114', '115', 'BARQUEIROS MSF' ] ] },
        'Rede' => {
          '5040' => [
            [ '481', '496', 'VILA MARIM MSF' ] ] },
        'Rio Teixeira' => {
          '5040' => [
            [ '418', '419', 'MESO FRIO' ] ] },
        'Ribeira da Rede' => {
          '5040' => [
            [ '331', '335', 'MESO FRIO' ] ] },
        'Cidadelhe' => {
          '5040' => [
            [ '154', '160', 'CIDADELHE MSF' ] ] },
        'Oliveira' => {
          '5040' => [
            [ '211', '214', 'OLIVEIRA MSF' ] ] },
        'Quints' => {
          '5040' => [
            [ '121', '122', 'BARQUEIROS MSF' ] ] },
        'Portela' => {
          '5040' => [
            [ '414', '414', 'MESO FRIO' ] ] },
        'Vale das Casas' => {
          '5040' => [
            [ '124', '128', 'BARQUEIROS MSF' ] ] }
             },
      'Vila Pouca de Aguiar' => {
        'Cevivas' => {
          '5450' => [
            [ '290', '298', 'TRESMINAS' ] ] },
        'Parada de Monteiros' => {
          '5450' => [
            [ '220', '220', 'PARADA DE MONTEIROS' ] ] },
        'Barrela de Jales' => {
          '5450' => [
            [ '340', '345', 'VREIA DE JALES' ] ] },
        'Paredes do Alvo' => {
          '5450' => [
            [ '261', '267', 'SOUTELO DE AGUIAR' ] ] },
        'Afonsim' => {
          '5450' => [
            [ '100', '101', 'AFONSIM' ] ] },
        'Guilhado' => {
          '5450' => [
            [ '080', '080', 'VILA POUCA DE AGUIAR' ] ] },
        'Bragado' => {
          '5450' => [
            [ '180', '185', 'BRAGADO' ] ] },
        'Cidadelha' => {
          '5450' => [
            [ '050', '071', 'VILA POUCA DE AGUIAR' ] ] },
        'Barbades de Baixo' => {
          '5450' => [
            [ '320', '323', 'VREIA DE BORNES' ] ] },
        'Sampaio' => {
          '5450' => [
            [ '082', '083', 'VILA POUCA DE AGUIAR' ] ] },
        'Valoura' => {
          '5450' => [
            [ '300', '301', 'VALOURA' ] ] },
        'Vila Pouca de Aguiar' => {
          '5450' => [
            [ '001', '036', 'VILA POUCA DE AGUIAR' ],
            [ '038', '043', 'VILA POUCA DE AGUIAR' ] ] },
        'Pedras Salgadas' => {
          '5450' => [
            [ '140', '151', 'BORNES DE AGUIAR' ],
            [ '153', '169', 'BORNES DE AGUIAR' ] ] },
        'Bolho' => {
          '5450' => [
            [ '200', '204', 'CAPELUDOS' ] ] },
        'Gouves da Serra' => {
          '5450' => [
            [ '210', '212', 'GOUVES DA SERRA' ] ] },
        'Trandeiras' => {
          '5450' => [
            [ '103', '103', 'AFONSIM' ] ] },
        'Santa Marta da Montanha' => {
          '5450' => [
            [ '240', '241', 'SANTA MARTA DA MONTANHA' ] ] },
        'Teles' => {
          '5450' => [
            [ '280', '288', 'TELES VPA' ] ] },
        'Cabanas' => {
          '5450' => [
            [ '230', '232', 'PENSALVOS' ] ] },
        'Bornes de Aguiar' => {
          '5450' => [
            [ '130', '138', 'BORNES DE AGUIAR' ] ] },
        'Hortinhas' => {
          '5450' => [
            [ '360', '371', 'SABROSO DE AGUIAR' ] ] },
        'Alfarela de Jales' => {
          '5450' => [
            [ '120', '123', 'ALFARELA DE JALES' ] ] }
                 },
      'Peso da Rgua' => {
        'Capela' => {
          '5050' => [
            [ '011', '012', 'COVELINHAS' ] ] },
        'Casas Luis Antnio' => {
          '5050' => [
            [ '200', '207', 'PESO DA RGUA' ] ] },
        'Godim' => {
          '5050' => [
            [ '059', '080', 'GODIM' ] ] },
        'Quinta Dona Matilde' => {
          '5050' => [
            [ '445', '445', 'CANELAS PRG' ] ] },
        'Pregegueda' => {
          '5050' => [
            [ '361', '370', 'VILARINHO DOS FREIRES' ] ] },
        'Olival' => {
          '5050' => [
            [ '310', '318', 'LOUREIRO PRG' ] ] },
        'Miradouro de Baixo' => {
          '5050' => [
            [ '440', '443', 'CANELAS PRG' ] ] },
        'Igreja' => {
          '5050' => [
            [ '401', '405', 'VINHS PRG' ] ] },
        'Fundo do Povo' => {
          '5050' => [
            [ '341', '347', 'POIARES PRG' ] ] },
        'Mourinho' => {
          '5050' => [
            [ '021', '030', 'FONTELAS' ] ] },
        'Aldarete' => {
          '5040' => [
            [ '511', '537', 'SEDIELOS' ] ] },
        'Peso da Rgua' => {
          '5054' => [
            [ '001', '003', 'PESO DA RGUA' ] ],
          '5050' => [
            [ '220', '295', 'PESO DA RGUA' ] ] },
        'Canelas' => {
          '5050' => [
            [ '421', '437', 'CANELAS PRG' ] ] },
        'Santa' => {
          '5050' => [
            [ '041', '044', 'GALAFURA' ] ] },
        'Trs do Pao' => {
          '5050' => [
            [ '090', '107', 'GODIM' ] ] },
        'Bairro' => {
          '5050' => [
            [ '321', '329', 'MOURA MORTA PRG' ] ] }
                },
      'Valpaos' => {
        'Nozelos' => {
          '5430' => [
            [ '180', '180', 'NOZELOS' ] ] },
        'Adagoi' => {
          '5445' => [
            [ '061', '073', 'SANTIAGO RIBEIRA ALHARIZ' ] ] },
        'Gorgoo' => {
          '5430' => [
            [ '230', '238', 'SANTA VALHA' ] ] },
        'Junqueira' => {
          '5445' => [
            [ '081', '087', 'SO JOO DA CORVEIRA' ] ] },
        'Deimos' => {
          '5430' => [
            [ '251', '252', 'S PEDRO DE VEIGA DE LILA' ] ] },
        'Alfonge' => {
          '5430' => [
            [ '071', '078', 'ERVES' ] ] },
        'Argemil' => {
          '5445' => [
            [ '101', '109', 'CARRAZEDO DE MONTENEGRO' ] ] },
        'Alvarelhos' => {
          '5430' => [
            [ '021', '022', 'ALVARELHOS VLP' ] ] },
        'Frutuoso' => {
          '5445' => [
            [ '041', '046', 'PADRELA E TZEM' ] ] },
        'Fornos do Pinhal' => {
          '5430' => [
            [ '100', '100', 'FORNOS DO PINHAL' ] ] },
        'Taixo' => {
          '5430' => [
            [ '191', '191', 'POSSACOS' ] ] },
        'Rio Torto' => {
          '5430' => [
            [ '201', '201', 'RIO TORTO VLP' ] ] },
        'Rendufe' => {
          '5445' => [
            [ '051', '052', 'SANTA MARIA DE EMERES' ] ] },
        'Cabeo' => {
          '5430' => [
            [ '401', '406', 'VALPAOS' ] ] },
        'Cadouo' => {
          '5430' => [
            [ '051', '053', 'CANAVESES' ] ] },
        'Fonte Merc' => {
          '5430' => [
            [ '008', '008', 'GUA REVS E CASTRO' ] ] },
        'Carrazedo de Montenegro' => {
          '5445' => [
            [ '151', '174', 'CARRAZEDO DE MONTENEGRO' ] ] },
        'Veiga de Lila' => {
          '5430' => [
            [ '620', '620', 'VEIGA DE LILA' ] ] },
        'Alto da Cotovia' => {
          '5430' => [
            [ '601', '603', 'VASSAL' ] ] },
        'Lebuo' => {
          '5430' => [
            [ '150', '169', 'LEBUO' ] ] },
        'Agordela' => {
          '5430' => [
            [ '291', '293', 'TINHELA' ] ] },
        'Cabanas' => {
          '5445' => [
            [ '031', '033', 'CURROS VLP' ] ] },
        'Argeriz' => {
          '5445' => [
            [ '011', '015', 'ARGERIZ' ] ] },
        'Vilarandelo' => {
          '5430' => [
            [ '630', '666', 'VILARANDELO' ] ] },
        'gua Revs' => {
          '5430' => [
            [ '011', '013', 'GUA REVS E CASTRO' ] ] },
        'Sanfins' => {
          '5430' => [
            [ '220', '220', 'SANFINS VLP' ] ] },
        'Vales' => {
          '5430' => [
            [ '301', '302', 'VALES VLP' ] ] },
        'Aveleda' => {
          '5445' => [
            [ '091', '095', 'SERAPICOS VLP' ] ] },
        'Ferreirs' => {
          '5430' => [
            [ '131', '131', 'LEBUO' ] ] },
        'Valpaos' => {
          '5430' => [
            [ '420', '493', 'VALPAOS' ] ] },
        'Sonim' => {
          '5430' => [
            [ '271', '280', 'SONIM' ] ] },
        'Bouas' => {
          '5430' => [
            [ '041', '047', 'BOUOES' ] ] },
        'Fies' => {
          '5430' => [
            [ '090', '090', 'FIES VLP' ] ] },
        'Barreiros' => {
          '5430' => [
            [ '031', '031', 'BARREIROS VLP' ] ] },
        'Barraco' => {
          '5430' => [
            [ '120', '130', 'FRIES' ] ] }
           },
      'Mura' => {
        'Gueirinho' => {
          '5090' => [
            [ '100', '115', 'MURA' ] ] },
        'Candedo' => {
          '5090' => [
            [ '011', '017', 'CANDEDO MUR' ] ] },
        'Cadaval' => {
          '5090' => [
            [ '051', '053', 'FIOLHOSO' ] ] },
        'Carva' => {
          '5090' => [
            [ '031', '032', 'CARVA' ] ] },
        'Aboleira' => {
          '5090' => [
            [ '071', '082', 'JOU' ] ] },
        'Varges' => {
          '5090' => [
            [ '210', '210', 'PALHEIROS' ] ] },
        'Ribeirinha' => {
          '5090' => [
            [ '220', '220', 'VALONGO DE MILHAIS' ] ] },
        'Asnela' => {
          '5090' => [
            [ '230', '232', 'VILARES MUR' ] ] },
        'Mura' => {
          '5090' => [
            [ '117', '129', 'MURA' ],
            [ '131', '144', 'MURA' ] ] },
        'Sobredo' => {
          '5090' => [
            [ '200', '200', 'NOURA' ] ] }
              },
      'Sabrosa' => {
        'So Paio' => {
          '5060' => [
            [ '471', '476', 'SO MARTINHO DE ANTAS' ] ] },
        'Chanceleiros' => {
          '5085' => [
            [ '201', '201', 'COVAS DO DOURO' ] ] },
        'Souto Maior' => {
          '5060' => [
            [ '511', '529', 'SOUTO MAIOR SBR' ] ] },
        'Vilarinho' => {
          '5060' => [
            [ '140', '140', 'PARADA DE PINHO' ] ] },
        'Calvrio' => {
          '5060' => [
            [ '011', '034', 'CELEIRS SBR' ] ] },
        'Paos' => {
          '5060' => [
            [ '201', '203', 'PAOS SBR' ],
            [ '205', '213', 'PAOS SBR' ] ] },
        'Arca' => {
          '5060' => [
            [ '401', '408', 'S LOURENO DE RIBAPINHO' ] ] },
        'Abrecovo' => {
          '5060' => [
            [ '051', '067', 'GOUVINHAS' ] ] },
        'Sobrados' => {
          '5060' => [
            [ '221', '231', 'PAOS SBR' ] ] },
        'Quinta da Cavadinha' => {
          '5060' => [
            [ '281', '290', 'PROVESENDE' ] ] },
        'Bateiras' => {
          '5085' => [
            [ '241', '245', 'GOUVES DO DOURO' ] ] },
        'Vilela' => {
          '5060' => [
            [ '241', '241', 'PAOS SBR' ] ] },
        'Donelo' => {
          '5085' => [
            [ '206', '223', 'COVAS DO DOURO' ] ] },
        'Vilarinho de So Romo' => {
          '5060' => [
            [ '630', '630', 'VILARINHO DE SO ROMO' ] ] },
        'Antas' => {
          '5060' => [
            [ '421', '441', 'SO MARTINHO DE ANTAS' ] ] },
        'Fermentes' => {
          '5060' => [
            [ '181', '192', 'PAOS SBR' ] ] },
        'So Martinho de Anta' => {
          '5060' => [
            [ '443', '452', 'SO MARTINHO DE ANTAS' ] ] },
        'Covas do Douro' => {
          '5085' => [
            [ '203', '204', 'COVAS DO DOURO' ] ] },
        'Bouco' => {
          '5060' => [
            [ '361', '372', 'SABROSA' ] ] },
        'Carrasco' => {
          '5085' => [
            [ '251', '255', 'SO CRISTOVO DO DOURO' ] ] },
        'Arruda' => {
          '5060' => [
            [ '611', '619', 'VILARINHO DE SO ROMO' ] ] },
        'Paradela de Guies' => {
          '5060' => [
            [ '161', '162', 'PARADELA DE GUIES' ] ] },
        'Pinho Cele' => {
          '5060' => [
            [ '561', '561', 'TORRE DO PINHO' ] ] },
        'Provesende' => {
          '5060' => [
            [ '251', '264', 'PROVESENDE' ] ] },
        'Sabrosa' => {
          '5060' => [
            [ '300', '338', 'SABROSA' ] ] },
        'Feitais' => {
          '5060' => [
            [ '501', '502', 'SOUTO MAIOR SBR' ] ] },
        'Torre do Pinho' => {
          '5060' => [
            [ '565', '591', 'TORRE DO PINHO' ] ] },
        'Tapado do Senhor' => {
          '5060' => [
            [ '540', '541', 'SOUTO MAIOR SBR' ] ] },
        'Parada de Pinho' => {
          '5060' => [
            [ '101', '112', 'PARADA DE PINHO' ],
            [ '114', '127', 'PARADA DE PINHO' ] ] }
          },
      'Mondim de Basto' => {
        'Moinho da Igreja' => {
          '4880' => [
            [ '230', '260', 'MONDIM DE BASTO' ] ] },
        'Lage' => {
          '4880' => [
            [ '181', '181', 'ATEI' ] ] },
        'Alm Ribeiro' => {
          '4880' => [
            [ '161', '180', 'MONDIM DE BASTO' ] ] },
        'Bairro Alm' => {
          '4880' => [
            [ '301', '318', 'VILAR DE FERREIROS' ] ] },
        'Vilar de Viando' => {
          '4880' => [
            [ '212', '212', 'MONDIM DE BASTO' ] ] },
        'Vila Pouca' => {
          '4880' => [
            [ '211', '211', 'ATEI' ] ] },
        'Lameira' => {
          '4880' => [
            [ '182', '210', 'MONDIM DE BASTO' ] ] },
        'Aureira' => {
          '4880' => [
            [ '121', '141', 'ERMELO MDB' ] ] },
        'Anta' => {
          '4880' => [
            [ '071', '087', 'BILH' ] ] },
        'Pardelhas' => {
          '4880' => [
            [ '290', '290', 'PARDELHAS' ] ] },
        'Paradana' => {
          '4880' => [
            [ '281', '282', 'PARADANA' ] ] },
        'Campanh' => {
          '4880' => [
            [ '091', '110', 'CAMPANH' ] ] },
        'Alto da Boua' => {
          '4880' => [
            [ '011', '059', 'ATEI' ] ] }
            }
       },
    'Setbal' => {
      'Sesimbra' => {
        'Facho da Azia' => {
          '2970' => [
            [ '335', '335', 'SESIMBRA' ] ] },
        'Zambujal' => {
          '2970' => [
            [ '811', '872', 'SESIMBRA' ] ] },
        'Sesimbra' => {
          '2970' => [
            [ '620', '794', 'SESIMBRA' ] ] },
        'Fornos' => {
          '2970' => [
            [ '360', '368', 'SESIMBRA' ] ] },
        'Sampaio' => {
          '2970' => [
            [ '575', '585', 'SESIMBRA' ] ] },
        'Palames' => {
          '2970' => [
            [ '530', '530', 'SESIMBRA' ] ] },
        'Casalinho' => {
          '2970' => [
            [ '265', '265', 'SESIMBRA' ] ] },
        'Colina do Pinhal' => {
          '2970' => [
            [ '275', '275', 'SESIMBRA' ] ] },
        'Casal do Sapo' => {
          '2975' => [
            [ '001', '403', 'QUINTA DO CONDE' ] ] },
        'Carrasqueira' => {
          '2970' => [
            [ '220', '245', 'SESIMBRA' ] ] },
        'Lagoa de Albufeira' => {
          '2970' => [
            [ '375', '467', 'SESIMBRA' ],
            [ '469', '472', 'SESIMBRA' ],
            [ '474', '481', 'SESIMBRA' ] ] },
        'Pedreiras' => {
          '2970' => [
            [ '535', '555', 'SESIMBRA' ] ] },
        'Quintinha' => {
          '2970' => [
            [ '291', '319', 'SESIMBRA' ] ] },
        'Caminho Branco' => {
          '2970' => [
            [ '215', '215', 'SESIMBRA' ] ] },
        'Ma' => {
          '2970' => [
            [ '501', '522', 'SESIMBRA' ] ] },
        'Coves' => {
          '2970' => [
            [ '331', '333', 'SESIMBRA' ] ] },
        'Boias' => {
          '2970' => [
            [ '185', '185', 'SESIMBRA' ] ] },
        'Corredoura' => {
          '2970' => [
            [ '280', '280', 'SESIMBRA' ] ] },
        'Quinta do Conde' => {
          '2975' => [
            [ '532', '532', 'QUINTA DO CONDE' ],
            [ '666', '666', 'QUINTA DO CONDE' ] ] },
        'Casais da Azia' => {
          '2970' => [
            [ '251', '258', 'SESIMBRA' ] ] },
        'Santana' => {
          '2970' => [
            [ '340', '340', 'SESIMBRA' ],
            [ '590', '593', 'SESIMBRA' ] ] },
        'Venda Nova' => {
          '2970' => [
            [ '805', '805', 'SESIMBRA' ] ] },
        'Fonte de Sesimbra' => {
          '2970' => [
            [ '350', '350', 'SESIMBRA' ] ] },
        'Quintola de Santana' => {
          '2970' => [
            [ '570', '570', 'SESIMBRA' ] ] },
        'Pinhal de Cima' => {
          '2970' => [
            [ '144', '152', 'SESIMBRA' ] ] },
        'Caixas' => {
          '2970' => [
            [ '195', '207', 'SESIMBRA' ] ] },
        'Calhariz' => {
          '2970' => [
            [ '210', '210', 'SESIMBRA' ] ] },
        'Torres' => {
          '2970' => [
            [ '800', '800', 'SESIMBRA' ] ] },
        'Pinhal do Cabedal' => {
          '2970' => [
            [ '190', '191', 'SESIMBRA' ] ] },
        'Casalo' => {
          '2970' => [
            [ '046', '142', 'SESIMBRA' ],
            [ '155', '156', 'SESIMBRA' ] ] },
        'Serra da Azia' => {
          '2970' => [
            [ '595', '601', 'SESIMBRA' ] ] },
        'Azia' => {
          '2970' => [
            [ '161', '180', 'SESIMBRA' ] ] },
        'Pinheirinhos' => {
          '2970' => [
            [ '560', '560', 'SESIMBRA' ] ] },
        'Casal das Figueiras' => {
          '2970' => [
            [ '261', '261', 'SESIMBRA' ] ] },
        'Almoinha' => {
          '2970' => [
            [ '001', '044', 'SESIMBRA' ] ] }
               },
      'Setbal' => {
        'Brejos de Azeito' => {
          '2925' => [
            [ '058', '070', 'AZEITO' ],
            [ '148', '164', 'AZEITO' ],
            [ '555', '558', 'AZEITO' ],
            [ '561', '576', 'AZEITO' ],
            [ '607', '619', 'AZEITO' ],
            [ '802', '812', 'AZEITO' ] ] },
        'Vila Nogueira de Azeito' => {
          '2925' => [
            [ '388', '396', 'AZEITO' ],
            [ '514', '551', 'AZEITO' ],
            [ '578', '581', 'AZEITO' ],
            [ '585', '590', 'AZEITO' ],
            [ '595', '596', 'AZEITO' ],
            [ '598', '599', 'AZEITO' ],
            [ '602', '605', 'AZEITO' ] ] },
        'Setbal' => {
          '2914' => [
            [ '501', '505', 'SETBAL' ],
            [ '507', '522', 'SETBAL' ],
            [ '524', '525', 'SETBAL' ] ],
          '2910' => [
            [ '001', '141', 'SETBAL' ],
            [ '144', '192', 'SETBAL' ],
            [ '196', '196', 'SETBAL' ],
            [ '200', '219', 'SETBAL' ],
            [ '221', '249', 'SETBAL' ],
            [ '251', '255', 'SETBAL' ],
            [ '257', '258', 'SETBAL' ],
            [ '260', '261', 'SETBAL' ],
            [ '264', '264', 'SETBAL' ],
            [ '267', '270', 'SETBAL' ],
            [ '272', '276', 'SETBAL' ],
            [ '278', '281', 'SETBAL' ],
            [ '284', '299', 'SETBAL' ],
            [ '302', '304', 'SETBAL' ],
            [ '307', '323', 'SETBAL' ],
            [ '325', '327', 'SETBAL' ],
            [ '330', '336', 'SETBAL' ],
            [ '338', '367', 'SETBAL' ],
            [ '369', '383', 'SETBAL' ],
            [ '385', '385', 'SETBAL' ],
            [ '387', '393', 'SETBAL' ],
            [ '395', '410', 'SETBAL' ],
            [ '413', '438', 'SETBAL' ],
            [ '441', '452', 'SETBAL' ],
            [ '454', '488', 'SETBAL' ],
            [ '490', '509', 'SETBAL' ],
            [ '511', '520', 'SETBAL' ],
            [ '522', '542', 'SETBAL' ],
            [ '544', '566', 'SETBAL' ],
            [ '569', '573', 'SETBAL' ],
            [ '575', '582', 'SETBAL' ],
            [ '584', '599', 'SETBAL' ],
            [ '602', '610', 'SETBAL' ],
            [ '612', '616', 'SETBAL' ],
            [ '618', '623', 'SETBAL' ],
            [ '625', '630', 'SETBAL' ],
            [ '632', '644', 'SETBAL' ],
            [ '646', '646', 'SETBAL' ],
            [ '648', '649', 'SETBAL' ],
            [ '653', '656', 'SETBAL' ],
            [ '671', '672', 'SETBAL' ],
            [ '674', '675', 'SETBAL' ],
            [ '677', '678', 'SETBAL' ],
            [ '680', '684', 'SETBAL' ],
            [ '686', '688', 'SETBAL' ],
            [ '690', '707', 'SETBAL' ],
            [ '710', '711', 'SETBAL' ],
            [ '713', '722', 'SETBAL' ],
            [ '724', '728', 'SETBAL' ],
            [ '730', '735', 'SETBAL' ],
            [ '737', '742', 'SETBAL' ],
            [ '744', '755', 'SETBAL' ],
            [ '758', '759', 'SETBAL' ],
            [ '761', '786', 'SETBAL' ],
            [ '788', '793', 'SETBAL' ],
            [ '795', '817', 'SETBAL' ],
            [ '819', '825', 'SETBAL' ],
            [ '827', '828', 'SETBAL' ],
            [ '830', '861', 'SETBAL' ],
            [ '863', '869', 'SETBAL' ] ],
          '2900' => [
            [ '001', '008', 'SETBAL' ],
            [ '013', '035', 'SETBAL' ],
            [ '038', '048', 'SETBAL' ],
            [ '050', '056', 'SETBAL' ],
            [ '058', '132', 'SETBAL' ],
            [ '134', '146', 'SETBAL' ],
            [ '148', '154', 'SETBAL' ],
            [ '156', '171', 'SETBAL' ],
            [ '173', '180', 'SETBAL' ],
            [ '182', '188', 'SETBAL' ],
            [ '190', '279', 'SETBAL' ],
            [ '281', '312', 'SETBAL' ],
            [ '314', '317', 'SETBAL' ],
            [ '319', '320', 'SETBAL' ],
            [ '322', '322', 'SETBAL' ],
            [ '324', '336', 'SETBAL' ],
            [ '338', '345', 'SETBAL' ],
            [ '347', '356', 'SETBAL' ],
            [ '358', '392', 'SETBAL' ],
            [ '394', '400', 'SETBAL' ],
            [ '402', '402', 'SETBAL' ],
            [ '404', '438', 'SETBAL' ],
            [ '440', '496', 'SETBAL' ],
            [ '498', '508', 'SETBAL' ],
            [ '510', '510', 'SETBAL' ],
            [ '512', '528', 'SETBAL' ],
            [ '530', '532', 'SETBAL' ],
            [ '534', '551', 'SETBAL' ],
            [ '553', '553', 'SETBAL' ],
            [ '555', '566', 'SETBAL' ],
            [ '568', '570', 'SETBAL' ],
            [ '572', '582', 'SETBAL' ],
            [ '584', '591', 'SETBAL' ],
            [ '593', '595', 'SETBAL' ],
            [ '597', '605', 'SETBAL' ],
            [ '607', '617', 'SETBAL' ],
            [ '619', '624', 'SETBAL' ],
            [ '626', '638', 'SETBAL' ],
            [ '641', '655', 'SETBAL' ],
            [ '657', '670', 'SETBAL' ],
            [ '672', '682', 'SETBAL' ],
            [ '691', '694', 'SETBAL' ],
            [ '700', '715', 'SETBAL' ],
            [ '717', '727', 'SETBAL' ],
            [ '729', '766', 'SETBAL' ],
            [ '836', '836', 'SETBAL' ] ],
          '2904' => [
            [ '501', '531', 'SETBAL' ],
            [ '533', '541', 'SETBAL' ] ] },
        'Azeito' => {
          '2925' => [
            [ '001', '052', 'AZEITO' ],
            [ '055', '055', 'AZEITO' ],
            [ '072', '111', 'AZEITO' ],
            [ '113', '115', 'AZEITO' ],
            [ '117', '119', 'AZEITO' ],
            [ '121', '123', 'AZEITO' ],
            [ '125', '134', 'AZEITO' ],
            [ '136', '145', 'AZEITO' ],
            [ '166', '178', 'AZEITO' ],
            [ '180', '191', 'AZEITO' ],
            [ '193', '194', 'AZEITO' ],
            [ '196', '196', 'AZEITO' ],
            [ '198', '203', 'AZEITO' ],
            [ '206', '206', 'AZEITO' ],
            [ '208', '217', 'AZEITO' ],
            [ '219', '223', 'AZEITO' ],
            [ '226', '230', 'AZEITO' ],
            [ '232', '232', 'AZEITO' ],
            [ '234', '237', 'AZEITO' ],
            [ '239', '247', 'AZEITO' ],
            [ '249', '252', 'AZEITO' ],
            [ '254', '259', 'AZEITO' ],
            [ '261', '266', 'AZEITO' ],
            [ '268', '268', 'AZEITO' ],
            [ '275', '275', 'AZEITO' ],
            [ '277', '277', 'AZEITO' ],
            [ '282', '284', 'AZEITO' ],
            [ '287', '292', 'AZEITO' ],
            [ '295', '313', 'AZEITO' ],
            [ '316', '354', 'AZEITO' ],
            [ '356', '384', 'AZEITO' ],
            [ '399', '415', 'AZEITO' ],
            [ '418', '421', 'AZEITO' ],
            [ '424', '427', 'AZEITO' ],
            [ '430', '433', 'AZEITO' ],
            [ '435', '437', 'AZEITO' ],
            [ '439', '443', 'AZEITO' ],
            [ '445', '449', 'AZEITO' ],
            [ '452', '454', 'AZEITO' ],
            [ '458', '461', 'AZEITO' ],
            [ '466', '469', 'AZEITO' ],
            [ '472', '481', 'AZEITO' ],
            [ '483', '483', 'AZEITO' ],
            [ '485', '486', 'AZEITO' ],
            [ '488', '488', 'AZEITO' ],
            [ '493', '512', 'AZEITO' ],
            [ '583', '583', 'AZEITO' ],
            [ '593', '593', 'AZEITO' ],
            [ '621', '682', 'AZEITO' ],
            [ '684', '694', 'AZEITO' ],
            [ '696', '698', 'AZEITO' ],
            [ '701', '713', 'AZEITO' ],
            [ '716', '742', 'AZEITO' ],
            [ '744', '760', 'AZEITO' ],
            [ '762', '772', 'AZEITO' ],
            [ '774', '798', 'AZEITO' ],
            [ '800', '800', 'AZEITO' ] ] }
              },
      'Alcochete' => {
        'So Francisco' => {
          '2890' => [
            [ '314', '314', 'SO FRANCISCO' ],
            [ '321', '328', 'SO FRANCISCO' ] ] },
        'Samouco' => {
          '2890' => [
            [ '201', '261', 'SAMOUCO' ] ] },
        'Alcochete' => {
          '2890' => [
            [ '001', '128', 'ALCOCHETE' ] ],
          '2894' => [
            [ '001', '001', 'ALCOCHETE' ] ] },
        'Alto do Chafariz' => {
          '2890' => [
            [ '151', '182', 'ALCOCHETE' ] ] },
        'Terra Nova' => {
          '2890' => [
            [ '271', '271', 'SAMOUCO' ] ] },
        'Alto da Pacheca' => {
          '2890' => [
            [ '301', '312', 'SO FRANCISCO' ] ] }
          },
      'Alccer do Sal' => {
        'Comporta' => {
          '7580' => [
            [ '621', '653', 'COMPORTA' ] ] },
        'Brejos da Carregueira de Baixo' => {
          '7580' => [
            [ '611', '613', 'COMPORTA' ] ] },
        'Torro' => {
          '7595' => [
            [ '101', '179', 'TORRO' ] ] },
        'Foz' => {
          '7580' => [
            [ '511', '514', 'ALCCER DO SAL' ] ] },
        'Alccer do Sal' => {
          '7580' => [
            [ '001', '044', 'ALCCER DO SAL' ],
            [ '046', '094', 'ALCCER DO SAL' ],
            [ '097', '098', 'ALCCER DO SAL' ],
            [ '100', '112', 'ALCCER DO SAL' ],
            [ '114', '121', 'ALCCER DO SAL' ],
            [ '123', '147', 'ALCCER DO SAL' ],
            [ '149', '246', 'ALCCER DO SAL' ],
            [ '250', '250', 'ALCCER DO SAL' ],
            [ '301', '328', 'ALCCER DO SAL' ] ] },
        'Monte do Olival' => {
          '7580' => [
            [ '516', '523', 'ALCCER DO SAL' ] ] },
        'Pego do Altar' => {
          '7580' => [
            [ '712', '713', 'SANTA SUSANA' ] ] },
        'Possanco' => {
          '7580' => [
            [ '680', '681', 'COMPORTA' ] ] },
        'Barranco' => {
          '7580' => [
            [ '701', '710', 'SANTA SUSANA' ] ] },
        'guas Pousadas' => {
          '7580' => [
            [ '501', '503', 'ALCCER DO SAL' ] ] },
        'Arcebispa' => {
          '7580' => [
            [ '505', '509', 'ALCCER DO SAL' ] ] },
        'Bato' => {
          '7595' => [
            [ '011', '056', 'TORRO' ] ] },
        'Vale Bom' => {
          '7595' => [
            [ '191', '195', 'TORRO' ] ] },
        'Casebres' => {
          '7580' => [
            [ '551', '559', 'SO MARTINHO ASL' ] ] }
               },
      'Montijo' => {
        'Peges' => {
          '2985' => [
            [ '212', '230', 'PEGES' ] ] },
        'Atalaia' => {
          '2870' => [
            [ '701', '758', 'ATALAIA MTJ' ] ] },
        'Latadas' => {
          '2100' => [
            [ '001', '001', 'CANHA' ] ] },
        'Montijo' => {
          '2870' => [
            [ '001', '048', 'MONTIJO' ],
            [ '050', '121', 'MONTIJO' ],
            [ '123', '265', 'MONTIJO' ],
            [ '267', '302', 'MONTIJO' ],
            [ '304', '481', 'MONTIJO' ],
            [ '850', '850', 'MONTIJO' ] ] },
        'Apeadeiro de Sarilhos' => {
          '2870' => [
            [ '630', '669', 'ALTO ESTANQUEIRO-JARDIA' ] ] },
        'Peges Gare' => {
          '2985' => [
            [ '270', '270', 'PEGES' ] ] },
        'Foros da Boavista' => {
          '2985' => [
            [ '050', '065', 'CANHA' ] ] },
        'Peges Velhos' => {
          '2985' => [
            [ '150', '160', 'SANTO ISIDRO DE PEGES' ] ] },
        'Afonsoeiro' => {
          '2870' => [
            [ '801', '824', 'MONTIJO' ],
            [ '860', '875', 'MONTIJO' ],
            [ '878', '879', 'MONTIJO' ],
            [ '890', '890', 'MONTIJO' ] ] },
        'Vale Porrim' => {
          '2870' => [
            [ '671', '671', 'ALTO ESTANQUEIRO-JARDIA' ] ] },
        'Afonsos' => {
          '2985' => [
            [ '201', '210', 'PEGES' ] ] },
        'Bairro da Mosca' => {
          '2870' => [
            [ '680', '685', 'ALTO ESTANQUEIRO-JARDIA' ] ] },
        'Figueiras' => {
          '2985' => [
            [ '120', '120', 'SANTO ISIDRO DE PEGES' ] ] },
        'Alto do Estanqueiro' => {
          '2870' => [
            [ '599', '627', 'ALTO ESTANQUEIRO-JARDIA' ] ] },
        'Santo Isidro' => {
          '2985' => [
            [ '101', '111', 'SANTO ISIDRO DE PEGES' ],
            [ '113', '114', 'SANTO ISIDRO DE PEGES' ] ] },
        'Pau Queimado' => {
          '2870' => [
            [ '500', '500', 'MONTIJO' ] ] },
        'Canha' => {
          '2985' => [
            [ '001', '030', 'CANHA' ],
            [ '032', '046', 'CANHA' ] ] },
        'Quinta da Judia' => {
          '2985' => [
            [ '275', '275', 'PEGES' ] ] },
        'Arce' => {
          '2870' => [
            [ '501', '546', 'SARILHOS GRANDES' ] ] },
        'Foros do Trapo' => {
          '2985' => [
            [ '126', '139', 'SANTO ISIDRO DE PEGES' ] ] }
              },
      'Grndola' => {
        'Azinheira dos Barros' => {
          '7570' => [
            [ '003', '012', 'AZINH BARROS S MAM SDO' ] ] },
        'Carvalhal' => {
          '7570' => [
            [ '779', '789', 'CARVALHAL GDL' ] ] },
        'Melides' => {
          '7570' => [
            [ '600', '611', 'MELIDES' ],
            [ '613', '775', 'MELIDES' ] ] },
        'Grndola' => {
          '7570' => [
            [ '120', '137', 'GRNDOLA' ],
            [ '139', '139', 'GRNDOLA' ],
            [ '141', '154', 'GRNDOLA' ],
            [ '156', '172', 'GRNDOLA' ],
            [ '174', '175', 'GRNDOLA' ],
            [ '177', '177', 'GRNDOLA' ],
            [ '179', '179', 'GRNDOLA' ],
            [ '181', '181', 'GRNDOLA' ],
            [ '183', '199', 'GRNDOLA' ],
            [ '201', '208', 'GRNDOLA' ],
            [ '210', '228', 'GRNDOLA' ],
            [ '230', '234', 'GRNDOLA' ],
            [ '236', '245', 'GRNDOLA' ],
            [ '247', '251', 'GRNDOLA' ],
            [ '253', '253', 'GRNDOLA' ],
            [ '255', '273', 'GRNDOLA' ],
            [ '275', '286', 'GRNDOLA' ],
            [ '288', '288', 'GRNDOLA' ],
            [ '290', '299', 'GRNDOLA' ],
            [ '301', '303', 'GRNDOLA' ],
            [ '306', '306', 'GRNDOLA' ],
            [ '308', '314', 'GRNDOLA' ],
            [ '317', '317', 'GRNDOLA' ],
            [ '319', '344', 'GRNDOLA' ] ] },
        'Tojeira' => {
          '7570' => [
            [ '017', '017', 'AZINH BARROS S MAM SDO' ] ] },
        'Aderneira' => {
          '7570' => [
            [ '401', '530', 'MELIDES' ] ] },
        'Brunheiras do Viso' => {
          '7570' => [
            [ '776', '778', 'SANTA MARGARIDA DA SERRA' ] ] },
        'gua Derramada' => {
          '7570' => [
            [ '101', '101', 'GRNDOLA' ] ] },
        'Ameiras de Baixo' => {
          '7570' => [
            [ '104', '118', 'GRNDOLA' ] ] }
               },
      'Santiago do Cacm' => {
        'Zona Industrial' => {
          '7540' => [
            [ '235', '237', 'SANTIAGO DO CACM' ] ] },
        'Daroeira' => {
          '7565' => [
            [ '100', '118', 'ALVALADE' ] ] },
        'So Francisco da Serra' => {
          '7540' => [
            [ '555', '555', 'SO FRANCISCO DA SERRA' ] ] },
        'Aldeia de Santo Andr' => {
          '7500' => [
            [ '011', '062', 'VILA NOVA DE SANTO ANDR' ] ] },
        'Vale de gua' => {
          '7540' => [
            [ '610', '610', 'VALE DE GUA' ] ] },
        'Cerrinho do Retiro' => {
          '7555' => [
            [ '201', '219', 'CERCAL DO ALENTEJO' ] ] },
        'Casito' => {
          '7540' => [
            [ '605', '605', 'VALE DE GUA' ] ] },
        'Paiol da Serra' => {
          '7540' => [
            [ '060', '061', 'SANTIAGO DO CACM' ] ] },
        'Monte da Ronca' => {
          '7540' => [
            [ '650', '650', 'VALE DE GUA' ] ] },
        'Boticos' => {
          '7540' => [
            [ '030', '033', 'ABELA' ] ] },
        'Monte das Torturas' => {
          '7540' => [
            [ '600', '600', 'VALE DE GUA' ] ] },
        'Outeiro de Pedra' => {
          '7540' => [
            [ '400', '402', 'SO DOMINGOS' ] ] },
        'Foros de Vale Coelho' => {
          '7555' => [
            [ '221', '281', 'CERCAL DO ALENTEJO' ] ] },
        'Abela' => {
          '7540' => [
            [ '011', '011', 'ABELA' ] ] },
        'Quinta da Venda' => {
          '7540' => [
            [ '553', '553', 'SO FRANCISCO DA SERRA' ] ] },
        'So Bartolomeu da Serra' => {
          '7540' => [
            [ '321', '321', 'SO BARTOLOMEU DA SERRA' ] ] },
        'Alvalade' => {
          '7565' => [
            [ '011', '090', 'ALVALADE' ] ] },
        'Mulinheta' => {
          '7540' => [
            [ '323', '323', 'SO BARTOLOMEU DA SERRA' ] ] },
        'Foros de Corujo' => {
          '7555' => [
            [ '311', '311', 'VALE DE GUA' ] ] },
        'Monte Peral de Baixo' => {
          '7540' => [
            [ '020', '025', 'ABELA' ] ] },
        'Monte Casa Nova Cho das Eiras' => {
          '7540' => [
            [ '055', '055', 'SANTA CRUZ STC' ] ] },
        'Faleiros' => {
          '7565' => [
            [ '250', '258', 'ERMIDAS-SADO' ] ] },
        'Monte da Cr' => {
          '7540' => [
            [ '230', '230', 'SANTIAGO DO CACM' ] ] },
        'Outeiro' => {
          '7540' => [
            [ '051', '051', 'SANTA CRUZ STC' ] ] },
        'Vila Nova de Santo Andr' => {
          '7500' => [
            [ '100', '100', 'VILA NOVA DE SANTO ANDR' ],
            [ '110', '110', 'VILA NOVA DE SANTO ANDR' ],
            [ '120', '120', 'VILA NOVA DE SANTO ANDR' ],
            [ '130', '130', 'VILA NOVA DE SANTO ANDR' ],
            [ '140', '140', 'VILA NOVA DE SANTO ANDR' ],
            [ '150', '150', 'VILA NOVA DE SANTO ANDR' ],
            [ '160', '160', 'VILA NOVA DE SANTO ANDR' ],
            [ '170', '170', 'VILA NOVA DE SANTO ANDR' ],
            [ '180', '180', 'VILA NOVA DE SANTO ANDR' ],
            [ '190', '190', 'VILA NOVA DE SANTO ANDR' ],
            [ '200', '200', 'VILA NOVA DE SANTO ANDR' ],
            [ '210', '210', 'VILA NOVA DE SANTO ANDR' ],
            [ '220', '220', 'VILA NOVA DE SANTO ANDR' ] ] },
        'Monte da Adega' => {
          '7540' => [
            [ '300', '300', 'SO BARTOLOMEU DA SERRA' ] ] },
        'Relvas Verdes' => {
          '7540' => [
            [ '240', '240', 'SANTIAGO DO CACM' ] ] },
        'Abertas' => {
          '7555' => [
            [ '011', '032', 'CERCAL DO ALENTEJO' ] ] },
        'Santiago do Cacm' => {
          '7540' => [
            [ '100', '222', 'SANTIAGO DO CACM' ] ] },
        'Cercal do Alentejo' => {
          '7555' => [
            [ '101', '148', 'CERCAL DO ALENTEJO' ] ] },
        'Ermidas Aldeia' => {
          '7565' => [
            [ '119', '119', 'ERMIDAS-SADO' ] ] },
        'So Domingos da Serra' => {
          '7540' => [
            [ '415', '415', 'SO DOMINGOS' ] ] },
        'Ermidas do Sado' => {
          '7565' => [
            [ '201', '234', 'ERMIDAS-SADO' ] ] },
        'Poo da Orla' => {
          '7540' => [
            [ '065', '065', 'SANTIAGO DO CACM' ] ] },
        'Cerca das Casas Novas' => {
          '7540' => [
            [ '550', '551', 'SO FRANCISCO DA SERRA' ] ] },
        'Cerro das Colmeias' => {
          '7540' => [
            [ '068', '068', 'SANTIAGO DO CACM' ] ] },
        'Quinta Outeiro dos Cravos' => {
          '7540' => [
            [ '500', '523', 'SO FRANCISCO DA SERRA' ] ] }
            },
      'Moita' => {
        'Barra Cheia' => {
          '2860' => [
            [ '219', '222', 'ALHOS VEDROS' ],
            [ '224', '226', 'ALHOS VEDROS' ],
            [ '228', '264', 'ALHOS VEDROS' ],
            [ '670', '674', 'ALHOS VEDROS' ] ] },
        'Alhos Vedros' => {
          '2860' => [
            [ '004', '018', 'ALHOS VEDROS' ],
            [ '020', '024', 'ALHOS VEDROS' ],
            [ '026', '039', 'ALHOS VEDROS' ],
            [ '041', '109', 'ALHOS VEDROS' ],
            [ '111', '132', 'ALHOS VEDROS' ],
            [ '266', '289', 'ALHOS VEDROS' ] ] },
        'Esteiro Furado' => {
          '2860' => [
            [ '639', '669', 'SARILHOS PEQUENOS' ] ] },
        'Penteado' => {
          '2860' => [
            [ '579', '608', 'MOITA' ] ] },
        'Gaio' => {
          '2860' => [
            [ '609', '638', 'GAIO-ROSRIO' ] ] },
        'Arroteias' => {
          '2860' => [
            [ '141', '217', 'ALHOS VEDROS' ] ] },
        'Baixa da Banheira' => {
          '2835' => [
            [ '001', '001', 'BAIXA DA BANHEIRA' ],
            [ '011', '016', 'BAIXA DA BANHEIRA' ],
            [ '019', '020', 'BAIXA DA BANHEIRA' ],
            [ '022', '023', 'BAIXA DA BANHEIRA' ],
            [ '025', '078', 'BAIXA DA BANHEIRA' ],
            [ '080', '093', 'BAIXA DA BANHEIRA' ],
            [ '096', '175', 'BAIXA DA BANHEIRA' ],
            [ '179', '183', 'BAIXA DA BANHEIRA' ] ] },
        'Vale da Amoreira' => {
          '2835' => [
            [ '010', '010', 'VALE DA AMOREIRA' ],
            [ '184', '185', 'VALE DA AMOREIRA' ],
            [ '191', '298', 'VALE DA AMOREIRA' ] ] },
        'Moita' => {
          '2860' => [
            [ '293', '435', 'MOITA' ],
            [ '437', '577', 'MOITA' ],
            [ '700', '714', 'MOITA' ] ],
          '2864' => [
            [ '001', '010', 'MOITA' ] ] },
        'Alto de So Sebastio' => {
          '2860' => [
            [ '716', '719', 'MOITA' ] ] }
            },
      'Palmela' => {
        'Sesmarias do Pato' => {
          '2965' => [
            [ '661', '661', 'GUAS DE MOURA' ] ] },
        'Lagameas' => {
          '2965' => [
            [ '261', '261', 'POCEIRO' ] ] },
        'Quinta do Borges' => {
          '2950' => [
            [ '801', '805', 'QUINTA DO ANJO' ] ] },
        'Pegarias' => {
          '2950' => [
            [ '421', '439', 'PALMELA' ] ] },
        'Foros da Asseisseira' => {
          '2965' => [
            [ '251', '251', 'POCEIRO' ] ] },
        'Tbua de Salvao' => {
          '2965' => [
            [ '671', '671', 'GUAS DE MOURA' ] ] },
        'Fonte Barreira' => {
          '2965' => [
            [ '631', '631', 'GUAS DE MOURA' ] ] },
        'Bairro Alentejano' => {
          '2950' => [
            [ '501', '531', 'QUINTA DO ANJO' ] ] },
        'Faias' => {
          '2965' => [
            [ '231', '231', 'POCEIRO' ] ] },
        'Algeruz' => {
          '2950' => [
            [ '051', '057', 'PALMELA' ] ] },
        'Palmela Gare' => {
          '2950' => [
            [ '401', '403', 'PALMELA' ] ] },
        'Parque Industrial de Carrascas' => {
          '2951' => [
            [ '503', '503', 'PALMELA' ] ] },
        'Aires' => {
          '2950' => [
            [ '011', '020', 'PALMELA' ],
            [ '022', '030', 'PALMELA' ],
            [ '101', '120', 'PALMELA' ],
            [ '337', '353', 'PALMELA' ],
            [ '482', '483', 'PALMELA' ] ] },
        'Olhos de gua' => {
          '2955' => [
            [ '066', '090', 'PINHAL NOVO' ] ] },
        'Bairro Sesmarias do Pato' => {
          '2965' => [
            [ '611', '611', 'GUAS DE MOURA' ] ] },
        'Arraiados' => {
          '2955' => [
            [ '006', '014', 'PINHAL NOVO' ] ] },
        'Marinhos' => {
          '2965' => [
            [ '651', '651', 'GUAS DE MOURA' ] ] },
        'Passarinhas' => {
          '2965' => [
            [ '291', '291', 'POCEIRO' ] ] },
        'Asseiceira' => {
          '2965' => [
            [ '211', '211', 'POCEIRO' ] ] },
        'Brejos do Poo' => {
          '2965' => [
            [ '221', '221', 'POCEIRO' ] ] },
        'Lagoa do Calvo' => {
          '2965' => [
            [ '271', '271', 'POCEIRO' ] ] },
        'Aldeia do Pinheiro' => {
          '2965' => [
            [ '601', '601', 'GUAS DE MOURA' ] ] },
        'Cabeo Velhinho' => {
          '2950' => [
            [ '059', '071', 'PALMELA' ] ] },
        'Monte Alegre' => {
          '2965' => [
            [ '281', '281', 'POCEIRO' ] ] },
        'Quinta do Anjo' => {
          '2950' => [
            [ '037', '037', 'QUINTA DO ANJO' ],
            [ '533', '535', 'QUINTA DO ANJO' ],
            [ '600', '659', 'QUINTA DO ANJO' ],
            [ '669', '673', 'QUINTA DO ANJO' ],
            [ '696', '756', 'QUINTA DO ANJO' ] ],
          '2951' => [
            [ '510', '510', 'QUINTA DO ANJO' ] ] },
        'Agualva de Cima' => {
          '2965' => [
            [ '501', '517', 'GUAS DE MOURA' ] ] },
        'Terrim' => {
          '2955' => [
            [ '016', '048', 'PINHAL NOVO' ] ] },
        'Poceiro' => {
          '2965' => [
            [ '301', '318', 'POCEIRO' ] ] },
        'Fernando P' => {
          '2965' => [
            [ '621', '621', 'GUAS DE MOURA' ] ] },
        'Cabanas' => {
          '2951' => [
            [ '502', '502', 'QUINTA DO ANJO' ] ] },
        'Quinta da Chapeleira' => {
          '2950' => [
            [ '675', '682', 'QUINTA DO ANJO' ] ] },
        'guas de Moura' => {
          '2965' => [
            [ '531', '534', 'GUAS DE MOURA' ],
            [ '536', '540', 'GUAS DE MOURA' ],
            [ '551', '590', 'GUAS DE MOURA' ] ] },
        'Aldeia Nova da Aroeira' => {
          '2965' => [
            [ '201', '201', 'POCEIRO' ] ] },
        'Forninho' => {
          '2965' => [
            [ '241', '241', 'POCEIRO' ] ] },
        'Herdade dos Travassos' => {
          '2965' => [
            [ '681', '681', 'GUAS DE MOURA' ] ] },
        'Foros do Trapo' => {
          '2965' => [
            [ '351', '351', 'POCEIRO' ] ] },
        'Pinhal Novo' => {
          '2955' => [
            [ '001', '004', 'PINHAL NOVO' ],
            [ '050', '052', 'PINHAL NOVO' ],
            [ '054', '064', 'PINHAL NOVO' ],
            [ '101', '137', 'PINHAL NOVO' ],
            [ '148', '148', 'PINHAL NOVO' ],
            [ '150', '225', 'PINHAL NOVO' ],
            [ '249', '249', 'PINHAL NOVO' ] ] },
        'Vale de Cantadores' => {
          '2951' => [
            [ '501', '501', 'PALMELA' ] ] },
        'Palmela' => {
          '2950' => [
            [ '007', '007', 'PALMELA' ],
            [ '201', '302', 'PALMELA' ],
            [ '304', '335', 'PALMELA' ] ],
          '2951' => [
            [ '505', '505', 'PALMELA' ] ] },
        'Vila Amlia' => {
          '2951' => [
            [ '500', '500', 'QUINTA DO ANJO' ],
            [ '504', '504', 'QUINTA DO ANJO' ],
            [ '506', '506', 'QUINTA DO ANJO' ],
            [ '550', '550', 'QUINTA DO ANJO' ] ] },
        'Herdade do Zambujal' => {
          '2965' => [
            [ '691', '691', 'GUAS DE MOURA' ] ] },
        'Bairro Assuno Piedade' => {
          '2950' => [
            [ '551', '566', 'QUINTA DO ANJO' ] ] }
              },
      'Seixal' => {
        'Paivas' => {
          '2845' => [
            [ '342', '563', 'AMORA' ] ] },
        'Casal do Marco' => {
          '2840' => [
            [ '734', '746', 'SEIXAL' ] ] },
        'Quinta de Valadares' => {
          '2855' => [
            [ '375', '552', 'CORROIOS' ],
            [ '561', '601', 'CORROIOS' ],
            [ '682', '715', 'CORROIOS' ] ] },
        'Corroios' => {
          '2855' => [
            [ '001', '126', 'CORROIOS' ],
            [ '128', '373', 'CORROIOS' ],
            [ '554', '559', 'CORROIOS' ] ] },
        'Foros da Catrapona' => {
          '2840' => [
            [ '701', '720', 'SEIXAL' ] ],
          '2865' => [
            [ '156', '161', 'FERNO FERRO' ],
            [ '163', '165', 'FERNO FERRO' ],
            [ '167', '167', 'FERNO FERRO' ],
            [ '169', '169', 'FERNO FERRO' ],
            [ '171', '171', 'FERNO FERRO' ],
            [ '174', '174', 'FERNO FERRO' ],
            [ '179', '182', 'FERNO FERRO' ],
            [ '184', '407', 'FERNO FERRO' ] ] },
        'Foros de Amora' => {
          '2845' => [
            [ '185', '186', 'AMORA' ],
            [ '188', '340', 'AMORA' ] ] },
        'Cavaquinhas' => {
          '2840' => [
            [ '218', '267', 'SEIXAL' ] ] },
        'Vale de Milhaos' => {
          '2855' => [
            [ '677', '680', 'CORROIOS' ] ] },
        'Zemoto' => {
          '2840' => [
            [ '601', '603', 'ALDEIA DE PAIO PIRES' ] ] },
        'Santa Marta do Pinhal' => {
          '2855' => [
            [ '603', '675', 'CORROIOS' ] ] },
        'Seixal' => {
          '2844' => [
            [ '001', '001', 'SEIXAL' ] ],
          '2840' => [
            [ '475', '521', 'SEIXAL' ],
            [ '523', '537', 'SEIXAL' ] ] },
        'Morgados' => {
          '2865' => [
            [ '624', '638', 'FERNO FERRO' ],
            [ '640', '645', 'FERNO FERRO' ],
            [ '648', '657', 'FERNO FERRO' ],
            [ '667', '673', 'FERNO FERRO' ] ] },
        'Fogueteiro' => {
          '2845' => [
            [ '566', '575', 'AMORA' ] ] },
        'Cavadas' => {
          '2840' => [
            [ '561', '561', 'SEIXAL' ] ] },
        'Cruz de Pau' => {
          '2845' => [
            [ '001', '183', 'AMORA' ],
            [ '577', '612', 'AMORA' ] ] },
        'Farinheiras' => {
          '2840' => [
            [ '562', '591', 'ALDEIA DE PAIO PIRES' ] ] },
        'Ferno Ferro' => {
          '2865' => [
            [ '001', '004', 'FERNO FERRO' ],
            [ '006', '041', 'FERNO FERRO' ],
            [ '043', '046', 'FERNO FERRO' ],
            [ '048', '048', 'FERNO FERRO' ],
            [ '051', '053', 'FERNO FERRO' ],
            [ '056', '061', 'FERNO FERRO' ],
            [ '063', '064', 'FERNO FERRO' ],
            [ '066', '073', 'FERNO FERRO' ],
            [ '075', '075', 'FERNO FERRO' ],
            [ '077', '084', 'FERNO FERRO' ],
            [ '086', '086', 'FERNO FERRO' ],
            [ '090', '091', 'FERNO FERRO' ],
            [ '093', '096', 'FERNO FERRO' ],
            [ '098', '111', 'FERNO FERRO' ],
            [ '115', '115', 'FERNO FERRO' ],
            [ '117', '117', 'FERNO FERRO' ],
            [ '119', '122', 'FERNO FERRO' ],
            [ '126', '128', 'FERNO FERRO' ],
            [ '131', '131', 'FERNO FERRO' ],
            [ '133', '153', 'FERNO FERRO' ],
            [ '409', '479', 'FERNO FERRO' ],
            [ '675', '677', 'FERNO FERRO' ],
            [ '680', '684', 'FERNO FERRO' ],
            [ '686', '697', 'FERNO FERRO' ],
            [ '712', '712', 'FERNO FERRO' ],
            [ '717', '717', 'FERNO FERRO' ],
            [ '719', '757', 'FERNO FERRO' ] ] },
        'Quinta das Laranjeiras' => {
          '2865' => [
            [ '481', '482', 'FERNO FERRO' ],
            [ '484', '493', 'FERNO FERRO' ],
            [ '659', '665', 'FERNO FERRO' ] ] },
        'Redondos' => {
          '2865' => [
            [ '495', '596', 'FERNO FERRO' ],
            [ '598', '604', 'FERNO FERRO' ],
            [ '606', '619', 'FERNO FERRO' ] ] },
        'Torre da Marinha' => {
          '2840' => [
            [ '548', '559', 'SEIXAL' ],
            [ '593', '599', 'SEIXAL' ] ] },
        'Arrentela' => {
          '2840' => [
            [ '142', '216', 'SEIXAL' ],
            [ '269', '372', 'SEIXAL' ],
            [ '374', '473', 'SEIXAL' ],
            [ '614', '617', 'SEIXAL' ] ] },
        'Paio Pires' => {
          '2840' => [
            [ '001', '141', 'ALDEIA DE PAIO PIRES' ],
            [ '538', '546', 'ALDEIA DE PAIO PIRES' ],
            [ '609', '613', 'ALDEIA DE PAIO PIRES' ] ] }
             },
      'Sines' => {
        'Vale Clarinho' => {
          '7520' => [
            [ '300', '308', 'SINES' ] ] },
        'Sonega' => {
          '7555' => [
            [ '301', '301', 'SINES' ] ] },
        'gua Branca' => {
          '7520' => [
            [ '400', '447', 'PORTO COVO' ] ] },
        'Sines' => {
          '7520' => [
            [ '001', '001', 'SINES' ],
            [ '100', '269', 'SINES' ] ] },
        'Vale Chiqueirinho' => {
          '7555' => [
            [ '291', '292', 'PORTO COVO' ] ] },
        'Afeiteira' => {
          '7520' => [
            [ '011', '089', 'SINES' ] ] },
        'Zona Industrial Ligeira 3' => {
          '7520' => [
            [ '310', '312', 'SINES' ] ] }
            },
      'Barreiro' => {
        'Santo Andr' => {
          '2830' => [
            [ '508', '511', 'BARREIRO' ] ] },
        'Fonte do Feto' => {
          '2835' => [
            [ '521', '528', 'SANTO ANTNIO DA CHARNECA' ],
            [ '530', '535', 'SANTO ANTNIO DA CHARNECA' ],
            [ '541', '569', 'SANTO ANTNIO DA CHARNECA' ] ] },
        'Vila Ribeiro' => {
          '2835' => [
            [ '791', '798', 'SANTO ANTNIO DA CHARNECA' ] ] },
        'Vila Ch' => {
          '2835' => [
            [ '462', '464', 'SANTO ANTNIO DA CHARNECA' ],
            [ '803', '805', 'SANTO ANTNIO DA CHARNECA' ],
            [ '811', '811', 'SANTO ANTNIO DA CHARNECA' ] ] },
        'Santo Antnio da Charneca' => {
          '2835' => [
            [ '631', '632', 'SANTO ANTNIO DA CHARNECA' ],
            [ '651', '656', 'SANTO ANTNIO DA CHARNECA' ],
            [ '658', '658', 'SANTO ANTNIO DA CHARNECA' ],
            [ '661', '662', 'SANTO ANTNIO DA CHARNECA' ],
            [ '665', '666', 'SANTO ANTNIO DA CHARNECA' ],
            [ '671', '678', 'SANTO ANTNIO DA CHARNECA' ],
            [ '681', '687', 'SANTO ANTNIO DA CHARNECA' ],
            [ '691', '691', 'SANTO ANTNIO DA CHARNECA' ],
            [ '695', '696', 'SANTO ANTNIO DA CHARNECA' ],
            [ '701', '765', 'SANTO ANTNIO DA CHARNECA' ] ] },
        'Coina' => {
          '2830' => [
            [ '406', '444', 'COINA' ],
            [ '512', '520', 'COINA' ] ] },
        'Penalva' => {
          '2835' => [
            [ '581', '582', 'SANTO ANTNIO DA CHARNECA' ],
            [ '591', '625', 'SANTO ANTNIO DA CHARNECA' ] ],
          '2830' => [
            [ '474', '481', 'COINA' ] ] },
        'Cidade Sol' => {
          '2835' => [
            [ '501', '516', 'SANTO ANTNIO DA CHARNECA' ] ] },
        'Palhais' => {
          '2830' => [
            [ '445', '473', 'PALHAIS BRR' ],
            [ '500', '507', 'PALHAIS BRR' ],
            [ '565', '566', 'PALHAIS BRR' ] ] },
        'Barreiro' => {
          '2830' => [
            [ '001', '003', 'BARREIRO' ],
            [ '007', '044', 'BARREIRO' ],
            [ '046', '046', 'BARREIRO' ],
            [ '048', '051', 'BARREIRO' ],
            [ '053', '069', 'BARREIRO' ],
            [ '071', '126', 'BARREIRO' ],
            [ '128', '161', 'BARREIRO' ],
            [ '163', '188', 'BARREIRO' ],
            [ '191', '405', 'BARREIRO' ],
            [ '482', '490', 'BARREIRO' ],
            [ '493', '499', 'BARREIRO' ],
            [ '521', '535', 'BARREIRO' ],
            [ '547', '564', 'BARREIRO' ],
            [ '567', '570', 'BARREIRO' ],
            [ '830', '831', 'BARREIRO' ] ] },
        'Lavradio' => {
          '2835' => [
            [ '301', '339', 'LAVRADIO' ],
            [ '341', '348', 'LAVRADIO' ],
            [ '351', '351', 'LAVRADIO' ],
            [ '371', '376', 'LAVRADIO' ],
            [ '383', '383', 'LAVRADIO' ],
            [ '386', '386', 'LAVRADIO' ],
            [ '389', '429', 'LAVRADIO' ],
            [ '431', '433', 'LAVRADIO' ],
            [ '435', '458', 'LAVRADIO' ],
            [ '461', '461', 'LAVRADIO' ],
            [ '799', '802', 'LAVRADIO' ],
            [ '807', '807', 'LAVRADIO' ],
            [ '809', '809', 'LAVRADIO' ] ] },
        'Cabeo Verde' => {
          '2835' => [
            [ '471', '497', 'SANTO ANTNIO DA CHARNECA' ] ] }
               },
      'Almada' => {
        'Alto do ndio' => {
          '2815' => [
            [ '623', '739', 'SOBREDA' ],
            [ '741', '752', 'SOBREDA' ],
            [ '785', '787', 'SOBREDA' ] ] },
        'So Pedro de Trafaria' => {
          '2825' => [
            [ '831', '834', 'TRAFARIA' ],
            [ '900', '902', 'TRAFARIA' ] ] },
        'Capuchos' => {
          '2825' => [
            [ '487', '487', 'CAPARICA' ] ] },
        'Marco Cabao' => {
          '2821' => [
            [ '001', '002', 'CHARNECA DA CAPARICA' ] ] },
        'Trafaria' => {
          '2825' => [
            [ '852', '879', 'TRAFARIA' ],
            [ '881', '881', 'TRAFARIA' ],
            [ '883', '889', 'TRAFARIA' ],
            [ '891', '898', 'TRAFARIA' ] ],
          '2829' => [
            [ '502', '502', 'TRAFARIA' ],
            [ '507', '507', 'TRAFARIA' ] ] },
        'Foni' => {
          '2825' => [
            [ '488', '492', 'COSTA DE CAPARICA' ] ] },
        'Quintinhas' => {
          '2820' => [
            [ '291', '424', 'CHARNECA DA CAPARICA' ],
            [ '494', '521', 'CHARNECA DA CAPARICA' ] ] },
        'Santo Antnio de Caparica' => {
          '2825' => [
            [ '424', '446', 'COSTA DE CAPARICA' ] ] },
        'Sobreda' => {
          '2815' => [
            [ '754', '767', 'SOBREDA' ],
            [ '770', '771', 'SOBREDA' ],
            [ '774', '783', 'SOBREDA' ],
            [ '789', '791', 'SOBREDA' ],
            [ '793', '801', 'SOBREDA' ],
            [ '804', '808', 'SOBREDA' ],
            [ '810', '818', 'SOBREDA' ],
            [ '878', '887', 'SOBREDA' ] ],
          '2819' => [
            [ '501', '502', 'SOBREDA' ],
            [ '505', '505', 'SOBREDA' ] ] },
        'Vale de Cavala' => {
          '2820' => [
            [ '426', '492', 'CHARNECA DA CAPARICA' ] ] },
        'Alagoa' => {
          '2820' => [
            [ '001', '289', 'CHARNECA DA CAPARICA' ] ] },
        'Almada' => {
          '2805' => [
            [ '007', '014', 'ALMADA' ],
            [ '017', '030', 'ALMADA' ],
            [ '033', '055', 'ALMADA' ],
            [ '059', '116', 'ALMADA' ],
            [ '118', '258', 'ALMADA' ],
            [ '261', '262', 'ALMADA' ],
            [ '265', '265', 'ALMADA' ],
            [ '267', '267', 'ALMADA' ],
            [ '273', '273', 'ALMADA' ],
            [ '278', '278', 'ALMADA' ],
            [ '280', '281', 'ALMADA' ],
            [ '286', '287', 'ALMADA' ],
            [ '290', '291', 'ALMADA' ],
            [ '294', '294', 'ALMADA' ],
            [ '298', '298', 'ALMADA' ],
            [ '304', '304', 'ALMADA' ],
            [ '311', '323', 'ALMADA' ],
            [ '325', '333', 'ALMADA' ],
            [ '335', '357', 'ALMADA' ] ],
          '2814' => [
            [ '501', '513', 'ALMADA' ] ],
          '2810' => [
            [ '001', '006', 'ALMADA' ],
            [ '010', '015', 'ALMADA' ],
            [ '017', '023', 'ALMADA' ],
            [ '025', '027', 'ALMADA' ],
            [ '029', '057', 'ALMADA' ],
            [ '061', '078', 'ALMADA' ],
            [ '080', '115', 'ALMADA' ],
            [ '119', '122', 'ALMADA' ],
            [ '124', '125', 'ALMADA' ],
            [ '127', '136', 'ALMADA' ],
            [ '138', '142', 'ALMADA' ],
            [ '144', '159', 'ALMADA' ],
            [ '161', '179', 'ALMADA' ],
            [ '184', '186', 'ALMADA' ],
            [ '189', '189', 'ALMADA' ],
            [ '193', '207', 'ALMADA' ],
            [ '209', '218', 'ALMADA' ],
            [ '221', '222', 'ALMADA' ],
            [ '224', '227', 'ALMADA' ],
            [ '230', '241', 'ALMADA' ],
            [ '244', '247', 'ALMADA' ],
            [ '249', '260', 'ALMADA' ],
            [ '262', '264', 'ALMADA' ],
            [ '267', '278', 'ALMADA' ],
            [ '280', '281', 'ALMADA' ],
            [ '283', '286', 'ALMADA' ],
            [ '288', '288', 'ALMADA' ],
            [ '290', '298', 'ALMADA' ],
            [ '300', '337', 'ALMADA' ],
            [ '339', '366', 'ALMADA' ],
            [ '368', '369', 'ALMADA' ],
            [ '371', '400', 'ALMADA' ],
            [ '402', '436', 'ALMADA' ],
            [ '500', '500', 'ALMADA' ],
            [ '911', '911', 'ALMADA' ] ],
          '2809' => [
            [ '001', '014', 'ALMADA' ] ],
          '2800' => [
            [ '001', '002', 'ALMADA' ],
            [ '009', '040', 'ALMADA' ],
            [ '042', '045', 'ALMADA' ],
            [ '047', '061', 'ALMADA' ],
            [ '063', '066', 'ALMADA' ],
            [ '068', '089', 'ALMADA' ],
            [ '091', '094', 'ALMADA' ],
            [ '096', '106', 'ALMADA' ],
            [ '108', '138', 'ALMADA' ],
            [ '146', '149', 'ALMADA' ],
            [ '153', '153', 'ALMADA' ],
            [ '155', '158', 'ALMADA' ],
            [ '161', '163', 'ALMADA' ],
            [ '165', '169', 'ALMADA' ],
            [ '172', '194', 'ALMADA' ],
            [ '199', '210', 'ALMADA' ],
            [ '212', '215', 'ALMADA' ],
            [ '233', '233', 'ALMADA' ],
            [ '252', '257', 'ALMADA' ],
            [ '259', '268', 'ALMADA' ],
            [ '270', '270', 'ALMADA' ],
            [ '274', '304', 'ALMADA' ],
            [ '523', '523', 'ALMADA' ],
            [ '597', '603', 'ALMADA' ],
            [ '628', '630', 'ALMADA' ],
            [ '642', '642', 'ALMADA' ],
            [ '648', '648', 'ALMADA' ],
            [ '660', '678', 'ALMADA' ],
            [ '680', '711', 'ALMADA' ] ],
          '2804' => [
            [ '503', '503', 'ALMADA' ],
            [ '506', '506', 'ALMADA' ],
            [ '516', '516', 'ALMADA' ],
            [ '519', '528', 'ALMADA' ],
            [ '532', '532', 'ALMADA' ],
            [ '535', '535', 'ALMADA' ],
            [ '537', '538', 'ALMADA' ] ] },
        'Monte de Caparica' => {
          '2825' => [
            [ '001', '004', 'CAPARICA' ],
            [ '006', '007', 'CAPARICA' ],
            [ '009', '009', 'CAPARICA' ],
            [ '011', '016', 'CAPARICA' ],
            [ '019', '032', 'CAPARICA' ],
            [ '034', '037', 'CAPARICA' ],
            [ '039', '043', 'CAPARICA' ],
            [ '058', '058', 'CAPARICA' ],
            [ '061', '061', 'CAPARICA' ],
            [ '063', '068', 'CAPARICA' ],
            [ '070', '071', 'CAPARICA' ],
            [ '073', '073', 'CAPARICA' ],
            [ '075', '075', 'CAPARICA' ],
            [ '077', '077', 'CAPARICA' ],
            [ '079', '081', 'CAPARICA' ],
            [ '083', '083', 'CAPARICA' ],
            [ '085', '091', 'CAPARICA' ],
            [ '095', '099', 'CAPARICA' ],
            [ '101', '105', 'CAPARICA' ],
            [ '107', '160', 'CAPARICA' ],
            [ '162', '170', 'CAPARICA' ],
            [ '423', '423', 'CAPARICA' ] ],
          '2829' => [
            [ '503', '503', 'CAPARICA' ],
            [ '505', '505', 'CAPARICA' ],
            [ '508', '509', 'CAPARICA' ],
            [ '511', '517', 'CAPARICA' ] ] },
        'Costa de Caparica' => {
          '2825' => [
            [ '278', '281', 'COSTA DE CAPARICA' ],
            [ '283', '287', 'COSTA DE CAPARICA' ],
            [ '289', '297', 'COSTA DE CAPARICA' ],
            [ '299', '307', 'COSTA DE CAPARICA' ],
            [ '310', '310', 'COSTA DE CAPARICA' ],
            [ '312', '323', 'COSTA DE CAPARICA' ],
            [ '325', '337', 'COSTA DE CAPARICA' ],
            [ '339', '370', 'COSTA DE CAPARICA' ],
            [ '372', '372', 'COSTA DE CAPARICA' ],
            [ '374', '377', 'COSTA DE CAPARICA' ],
            [ '379', '389', 'COSTA DE CAPARICA' ],
            [ '391', '405', 'COSTA DE CAPARICA' ],
            [ '407', '410', 'COSTA DE CAPARICA' ],
            [ '412', '417', 'COSTA DE CAPARICA' ],
            [ '420', '422', 'COSTA DE CAPARICA' ] ],
          '2829' => [
            [ '501', '501', 'COSTA DE CAPARICA' ],
            [ '504', '504', 'COSTA DE CAPARICA' ],
            [ '506', '506', 'COSTA DE CAPARICA' ],
            [ '510', '510', 'COSTA DE CAPARICA' ] ] },
        'So Joo de Caparica' => {
          '2825' => [
            [ '448', '486', 'COSTA DE CAPARICA' ] ] },
        'Murfacm' => {
          '2825' => [
            [ '836', '849', 'TRAFARIA' ] ] },
        'Carcereira' => {
          '2815' => [
            [ '856', '876', 'SOBREDA' ] ] },
        'Costas de Co' => {
          '2825' => [
            [ '045', '056', 'CAPARICA' ] ] },
        'Palhais' => {
          '2820' => [
            [ '523', '674', 'CHARNECA DA CAPARICA' ] ] },
        'Vale Figueira' => {
          '2815' => [
            [ '820', '825', 'SOBREDA' ],
            [ '827', '828', 'SOBREDA' ],
            [ '830', '840', 'SOBREDA' ],
            [ '842', '843', 'SOBREDA' ],
            [ '846', '854', 'SOBREDA' ] ],
          '2819' => [
            [ '507', '507', 'SOBREDA' ] ] }
             }
           },
    'Ilha do Faial' => {
      'Horta' => {
        'Salo' => {
          '9900' => [
            [ '501', '501', 'SALO' ] ] },
        'Aeroporto da Horta' => {
          '9900' => [
            [ '321', '333', 'CASTELO BRANCO HRT' ] ] },
        'Entre Casas' => {
          '9900' => [
            [ '491', '491', 'RIBEIRINHA HRT' ] ] },
        'Canto' => {
          '9900' => [
            [ '503', '505', 'SALO' ] ] },
        'Farrobim Norte' => {
          '9900' => [
            [ '366', '371', 'FETEIRA HRT' ] ] },
        'Areeiro' => {
          '9900' => [
            [ '301', '305', 'CAPELO' ] ] },
        'Horta' => {
          '9900' => [
            [ '011', '011', 'HORTA' ],
            [ '013', '021', 'HORTA' ],
            [ '023', '031', 'HORTA' ],
            [ '033', '033', 'HORTA' ],
            [ '037', '040', 'HORTA' ],
            [ '044', '053', 'HORTA' ],
            [ '056', '059', 'HORTA' ],
            [ '061', '067', 'HORTA' ],
            [ '069', '071', 'HORTA' ],
            [ '073', '091', 'HORTA' ],
            [ '093', '114', 'HORTA' ],
            [ '116', '119', 'HORTA' ],
            [ '121', '150', 'HORTA' ],
            [ '152', '158', 'HORTA' ],
            [ '160', '160', 'HORTA' ],
            [ '162', '175', 'HORTA' ] ],
          '9901' => [
            [ '851', '863', 'HORTA' ] ] },
        'Faj' => {
          '9900' => [
            [ '471', '473', 'PRAIA DO NORTE' ] ] },
        'Praia do Almoxarife' => {
          '9900' => [
            [ '451', '453', 'PRAIA DO ALMOXARIFE' ] ] },
        'Vale das Vinhas' => {
          '9900' => [
            [ '373', '375', 'FETEIRA HRT' ] ] },
        'Flamengos' => {
          '9900' => [
            [ '401', '404', 'FLAMENGOS' ] ] },
        'Pedro Miguel' => {
          '9900' => [
            [ '421', '429', 'PEDRO MIGUEL' ],
            [ '431', '433', 'PEDRO MIGUEL' ],
            [ '435', '437', 'PEDRO MIGUEL' ],
            [ '439', '439', 'PEDRO MIGUEL' ] ] },
        'Cedros' => {
          '9900' => [
            [ '341', '341', 'CEDROS HRT' ] ] },
        'Feteira' => {
          '9900' => [
            [ '361', '364', 'FETEIRA HRT' ] ] },
        'Lameiro Grande' => {
          '9900' => [
            [ '407', '408', 'FLAMENGOS' ] ] },
        'Canadinhas' => {
          '9900' => [
            [ '343', '347', 'CEDROS HRT' ] ] }
            }
           },
    'Ilha Terceira' => {
      'Angra do Herosmo' => {
        'Terra Ch' => {
          '9701' => [
            [ '851', '851', 'TERRA CH' ] ],
          '9700' => [
            [ '681', '688', 'TERRA CH' ],
            [ '701', '708', 'TERRA CH' ],
            [ '710', '720', 'TERRA CH' ] ] },
        'So Bartolomeu' => {
          '9700' => [
            [ '501', '526', 'SO BARTOLOMEU DE REGATOS' ],
            [ '579', '579', 'SO BARTOLOMEU DE REGATOS' ] ] },
        'Angra do Herosmo' => {
          '9701' => [
            [ '852', '878', 'ANGRA DO HEROSMO' ],
            [ '880', '880', 'ANGRA DO HEROSMO' ] ],
          '9700' => [
            [ '009', '009', 'ANGRA DO HEROSMO' ],
            [ '011', '017', 'ANGRA DO HEROSMO' ],
            [ '019', '033', 'ANGRA DO HEROSMO' ],
            [ '035', '035', 'ANGRA DO HEROSMO' ],
            [ '037', '047', 'ANGRA DO HEROSMO' ],
            [ '049', '056', 'ANGRA DO HEROSMO' ],
            [ '058', '066', 'ANGRA DO HEROSMO' ],
            [ '068', '073', 'ANGRA DO HEROSMO' ],
            [ '075', '082', 'ANGRA DO HEROSMO' ],
            [ '085', '099', 'ANGRA DO HEROSMO' ],
            [ '101', '127', 'ANGRA DO HEROSMO' ],
            [ '129', '130', 'ANGRA DO HEROSMO' ],
            [ '132', '138', 'ANGRA DO HEROSMO' ],
            [ '140', '152', 'ANGRA DO HEROSMO' ],
            [ '154', '155', 'ANGRA DO HEROSMO' ],
            [ '157', '167', 'ANGRA DO HEROSMO' ],
            [ '169', '169', 'ANGRA DO HEROSMO' ],
            [ '171', '174', 'ANGRA DO HEROSMO' ],
            [ '176', '189', 'ANGRA DO HEROSMO' ],
            [ '191', '194', 'ANGRA DO HEROSMO' ],
            [ '196', '208', 'ANGRA DO HEROSMO' ],
            [ '211', '241', 'ANGRA DO HEROSMO' ] ] },
        'Serreta' => {
          '9700' => [
            [ '661', '667', 'SERRETA' ] ] },
        'Cinco Ribeiras' => {
          '9700' => [
            [ '321', '324', 'CINCO RIBEIRAS' ] ] },
        'Altares' => {
          '9700' => [
            [ '301', '312', 'ALTARES' ] ] },
        'Doze Ribeiras' => {
          '9700' => [
            [ '331', '341', 'DOZE RIBEIRAS' ] ] },
        'Santa Brbara' => {
          '9700' => [
            [ '471', '481', 'SANTA BRBARA AGH' ] ] },
        'Ribeirinha' => {
          '9700' => [
            [ '411', '418', 'RIBEIRINHA AGH' ],
            [ '420', '421', 'RIBEIRINHA AGH' ],
            [ '423', '424', 'RIBEIRINHA AGH' ],
            [ '426', '451', 'RIBEIRINHA AGH' ] ] },
        'Porto Judeu' => {
          '9700' => [
            [ '362', '381', 'PORTO JUDEU' ] ] },
        'Feteira' => {
          '9700' => [
            [ '128', '128', 'FETEIRA AGH' ],
            [ '348', '348', 'FETEIRA AGH' ],
            [ '351', '360', 'FETEIRA AGH' ] ] },
        'Raminho' => {
          '9700' => [
            [ '401', '401', 'RAMINHO' ],
            [ '403', '409', 'RAMINHO' ] ] },
        'So Mateus' => {
          '9700' => [
            [ '551', '559', 'SO MATEUS DA CALHETA' ],
            [ '561', '568', 'SO MATEUS DA CALHETA' ],
            [ '570', '578', 'SO MATEUS DA CALHETA' ],
            [ '580', '592', 'SO MATEUS DA CALHETA' ] ] },
        'So Sebastio' => {
          '9700' => [
            [ '153', '153', 'VILA DE SO SEBASTIO' ],
            [ '170', '170', 'VILA DE SO SEBASTIO' ],
            [ '361', '361', 'VILA DE SO SEBASTIO' ],
            [ '601', '602', 'VILA DE SO SEBASTIO' ],
            [ '604', '607', 'VILA DE SO SEBASTIO' ],
            [ '610', '637', 'VILA DE SO SEBASTIO' ],
            [ '639', '639', 'VILA DE SO SEBASTIO' ],
            [ '641', '648', 'VILA DE SO SEBASTIO' ],
            [ '650', '653', 'VILA DE SO SEBASTIO' ] ] }
                  },
      'Praia da Vitria' => {
        'Praia da Vitria' => {
          '9760' => [
            [ '408', '413', 'PRAIA DA VITRIA' ],
            [ '415', '432', 'PRAIA DA VITRIA' ],
            [ '434', '441', 'PRAIA DA VITRIA' ],
            [ '443', '458', 'PRAIA DA VITRIA' ],
            [ '460', '470', 'PRAIA DA VITRIA' ],
            [ '472', '475', 'PRAIA DA VITRIA' ],
            [ '477', '479', 'PRAIA DA VITRIA' ],
            [ '481', '485', 'PRAIA DA VITRIA' ],
            [ '487', '509', 'PRAIA DA VITRIA' ],
            [ '511', '512', 'PRAIA DA VITRIA' ],
            [ '514', '521', 'PRAIA DA VITRIA' ],
            [ '523', '588', 'PRAIA DA VITRIA' ],
            [ '851', '856', 'PRAIA DA VITRIA' ] ] },
        'Abaixo da Igreja' => {
          '9760' => [
            [ '189', '189', 'FONTE DO BASTARDO' ] ] },
        'Fonte Bastardo' => {
          '9760' => [
            [ '111', '111', 'FONTE DO BASTARDO' ],
            [ '130', '130', 'FONTE DO BASTARDO' ],
            [ '172', '176', 'FONTE DO BASTARDO' ],
            [ '178', '182', 'FONTE DO BASTARDO' ],
            [ '184', '187', 'FONTE DO BASTARDO' ],
            [ '191', '194', 'FONTE DO BASTARDO' ] ] },
        'Biscoitos' => {
          '9760' => [
            [ '051', '056', 'BISCOITOS' ] ] },
        'Porto Martins' => {
          '9760' => [
            [ '095', '099', 'PORTO MARTINS' ],
            [ '109', '109', 'PORTO MARTINS' ],
            [ '123', '123', 'PORTO MARTINS' ],
            [ '129', '129', 'PORTO MARTINS' ],
            [ '132', '134', 'PORTO MARTINS' ],
            [ '136', '137', 'PORTO MARTINS' ],
            [ '140', '142', 'PORTO MARTINS' ] ] },
        'Serra de Santiago' => {
          '9760' => [
            [ '321', '328', 'LAJES VPV' ] ] },
        'Presa dos Ourives' => {
          '9760' => [
            [ '196', '199', 'FONTE DO BASTARDO' ] ] },
        'Agualva' => {
          '9760' => [
            [ '011', '012', 'AGUALVA' ],
            [ '014', '028', 'AGUALVA' ],
            [ '030', '039', 'AGUALVA' ] ] },
        'Ribeira do Chamusco' => {
          '9760' => [
            [ '058', '059', 'BISCOITOS' ] ] },
        'Cabo da Praia' => {
          '9760' => [
            [ '100', '104', 'CABO DA PRAIA' ],
            [ '106', '108', 'CABO DA PRAIA' ],
            [ '110', '110', 'CABO DA PRAIA' ],
            [ '112', '112', 'CABO DA PRAIA' ],
            [ '114', '114', 'CABO DA PRAIA' ],
            [ '116', '122', 'CABO DA PRAIA' ],
            [ '124', '128', 'CABO DA PRAIA' ],
            [ '131', '131', 'CABO DA PRAIA' ],
            [ '135', '135', 'CABO DA PRAIA' ] ] },
        'Vila Nova' => {
          '9760' => [
            [ '701', '734', 'VILA NOVA VPV' ],
            [ '736', '753', 'VILA NOVA VPV' ] ] },
        'Fontinhas' => {
          '9760' => [
            [ '211', '215', 'FONTINHAS' ] ] },
        'Lajes' => {
          '9760' => [
            [ '260', '268', 'LAJES VPV' ],
            [ '270', '290', 'LAJES VPV' ],
            [ '292', '300', 'LAJES VPV' ] ] },
        'Ribeira Seca' => {
          '9760' => [
            [ '357', '358', 'QUATRO RIBEIRAS' ] ] },
        'Casa da Ribeira' => {
          '9760' => [
            [ '401', '406', 'PRAIA DA VITRIA' ] ] },
        'So Brs' => {
          '9760' => [
            [ '029', '029', 'SO BRS VPV' ],
            [ '269', '269', 'SO BRS VPV' ],
            [ '651', '652', 'SO BRS VPV' ],
            [ '654', '671', 'SO BRS VPV' ] ] },
        'Aeroporto das Lajes' => {
          '9760' => [
            [ '251', '258', 'LAJES VPV' ] ] },
        'Remdios' => {
          '9760' => [
            [ '302', '319', 'LAJES VPV' ] ] },
        'Quatro Ribeiras' => {
          '9760' => [
            [ '351', '353', 'QUATRO RIBEIRAS' ] ] }
                 }
           },
    'Ilha de Porto Santo' => {
      'Porto Santo' => {
        'Casinhas do Pico' => {
          '9400' => [
            [ '040', '040', 'PORTO SANTO' ] ] },
        'Lombas' => {
          '9400' => [
            [ '065', '065', 'PORTO SANTO' ] ] },
        'Vila de Porto Santo' => {
          '9400' => [
            [ '001', '001', 'PORTO SANTO' ],
            [ '150', '159', 'PORTO SANTO' ],
            [ '161', '179', 'PORTO SANTO' ] ] },
        'Tanque' => {
          '9400' => [
            [ '110', '110', 'PORTO SANTO' ] ] },
        'Serra de Fora' => {
          '9400' => [
            [ '105', '105', 'PORTO SANTO' ] ] },
        'Tera' => {
          '9400' => [
            [ '115', '115', 'PORTO SANTO' ] ] },
        'Matas' => {
          '9400' => [
            [ '035', '035', 'PORTO SANTO' ] ] },
        'P do Pico' => {
          '9400' => [
            [ '070', '070', 'PORTO SANTO' ] ] },
        'Camacha' => {
          '9400' => [
            [ '010', '010', 'PORTO SANTO' ] ] },
        'Serra de Dentro' => {
          '9400' => [
            [ '100', '100', 'PORTO SANTO' ] ] },
        'Cabeo' => {
          '9400' => [
            [ '030', '030', 'PORTO SANTO' ] ] },
        'Sales' => {
          '9400' => [
            [ '095', '095', 'PORTO SANTO' ] ] },
        'Levada do Canha' => {
          '9400' => [
            [ '060', '060', 'PORTO SANTO' ] ] },
        'Pedras Pretas' => {
          '9400' => [
            [ '075', '075', 'PORTO SANTO' ] ] },
        'Lapeira' => {
          '9400' => [
            [ '055', '055', 'PORTO SANTO' ] ] },
        'Barroca' => {
          '9400' => [
            [ '005', '005', 'PORTO SANTO' ] ] },
        'Farrobo' => {
          '9400' => [
            [ '050', '050', 'PORTO SANTO' ] ] },
        'Dragoal' => {
          '9400' => [
            [ '045', '045', 'PORTO SANTO' ] ] },
        'Campo de Cima' => {
          '9400' => [
            [ '020', '020', 'PORTO SANTO' ] ] },
        'Casinhas' => {
          '9400' => [
            [ '025', '025', 'PORTO SANTO' ] ] },
        'Campo de Baixo' => {
          '9400' => [
            [ '015', '015', 'PORTO SANTO' ] ] },
        'Penedo' => {
          '9400' => [
            [ '080', '080', 'PORTO SANTO' ] ] },
        'Ponta' => {
          '9400' => [
            [ '085', '085', 'PORTO SANTO' ] ] }
                  }
           },
    'Porto' => {
      'Pvoa de Varzim' => {
        'Pvoa de Varzim' => {
          '4490' => [
            [ '002', '022', 'PVOA DE VARZIM' ],
            [ '024', '034', 'PVOA DE VARZIM' ],
            [ '036', '037', 'PVOA DE VARZIM' ],
            [ '039', '039', 'PVOA DE VARZIM' ],
            [ '041', '057', 'PVOA DE VARZIM' ],
            [ '059', '061', 'PVOA DE VARZIM' ],
            [ '063', '069', 'PVOA DE VARZIM' ],
            [ '071', '117', 'PVOA DE VARZIM' ],
            [ '119', '123', 'PVOA DE VARZIM' ],
            [ '125', '132', 'PVOA DE VARZIM' ],
            [ '135', '195', 'PVOA DE VARZIM' ],
            [ '401', '411', 'PVOA DE VARZIM' ],
            [ '417', '428', 'PVOA DE VARZIM' ],
            [ '430', '443', 'PVOA DE VARZIM' ],
            [ '445', '473', 'PVOA DE VARZIM' ],
            [ '475', '480', 'PVOA DE VARZIM' ],
            [ '482', '501', 'PVOA DE VARZIM' ],
            [ '503', '503', 'PVOA DE VARZIM' ],
            [ '505', '527', 'PVOA DE VARZIM' ],
            [ '529', '569', 'PVOA DE VARZIM' ],
            [ '571', '597', 'PVOA DE VARZIM' ],
            [ '599', '628', 'PVOA DE VARZIM' ],
            [ '630', '631', 'PVOA DE VARZIM' ],
            [ '634', '650', 'PVOA DE VARZIM' ],
            [ '653', '674', 'PVOA DE VARZIM' ],
            [ '676', '692', 'PVOA DE VARZIM' ],
            [ '694', '694', 'PVOA DE VARZIM' ] ],
          '4495' => [
            [ '001', '015', 'PVOA DE VARZIM' ],
            [ '017', '072', 'PVOA DE VARZIM' ],
            [ '074', '100', 'PVOA DE VARZIM' ],
            [ '178', '200', 'PVOA DE VARZIM' ],
            [ '252', '259', 'PVOA DE VARZIM' ],
            [ '301', '305', 'PVOA DE VARZIM' ],
            [ '307', '332', 'PVOA DE VARZIM' ],
            [ '334', '335', 'PVOA DE VARZIM' ],
            [ '337', '344', 'PVOA DE VARZIM' ],
            [ '346', '352', 'PVOA DE VARZIM' ],
            [ '354', '355', 'PVOA DE VARZIM' ],
            [ '357', '361', 'PVOA DE VARZIM' ],
            [ '363', '393', 'PVOA DE VARZIM' ],
            [ '395', '401', 'PVOA DE VARZIM' ],
            [ '403', '408', 'PVOA DE VARZIM' ],
            [ '410', '412', 'PVOA DE VARZIM' ],
            [ '414', '414', 'PVOA DE VARZIM' ],
            [ '416', '419', 'PVOA DE VARZIM' ],
            [ '421', '432', 'PVOA DE VARZIM' ] ] },
        'Navais' => {
          '4495' => [
            [ '201', '251', 'NAVAIS' ] ] },
        'Laundos' => {
          '4570' => [
            [ '311', '380', 'LAUNDOS' ] ] },
        'Terroso' => {
          '4495' => [
            [ '501', '594', 'TERROSO' ] ] },
        'Balazar' => {
          '4570' => [
            [ '011', '084', 'BALAZAR PVZ' ],
            [ '086', '086', 'BALAZAR PVZ' ],
            [ '088', '099', 'BALAZAR PVZ' ] ] },
        'Rates' => {
          '4570' => [
            [ '411', '527', 'RATES' ] ] },
        'Amorim' => {
          '4495' => [
            [ '101', '101', 'AMORIM' ],
            [ '103', '103', 'AMORIM' ],
            [ '105', '108', 'AMORIM' ],
            [ '110', '152', 'AMORIM' ],
            [ '154', '154', 'AMORIM' ],
            [ '156', '156', 'AMORIM' ],
            [ '158', '159', 'AMORIM' ],
            [ '161', '161', 'AMORIM' ],
            [ '163', '177', 'AMORIM' ],
            [ '260', '268', 'AMORIM' ],
            [ '270', '271', 'AMORIM' ],
            [ '273', '273', 'AMORIM' ] ] },
        'Argivai' => {
          '4490' => [
            [ '201', '206', 'ARGIVAI' ],
            [ '208', '251', 'ARGIVAI' ] ] },
        'Estela' => {
          '4570' => [
            [ '201', '211', 'ESTELA' ],
            [ '214', '248', 'ESTELA' ],
            [ '250', '279', 'ESTELA' ] ] }
              },
      'Santo Tirso' => {
        'Cela' => {
          '4780' => [
            [ '254', '254', 'SANTO TIRSO' ] ] },
        'Santo Tirso' => {
          '4780' => [
            [ '362', '584', 'SANTO TIRSO' ],
            [ '723', '728', 'SANTO TIRSO' ],
            [ '733', '739', 'SANTO TIRSO' ] ] },
        'Palmeira' => {
          '4780' => [
            [ '323', '329', 'PALMEIRA STS' ],
            [ '331', '347', 'PALMEIRA STS' ],
            [ '350', '351', 'PALMEIRA STS' ],
            [ '353', '354', 'PALMEIRA STS' ],
            [ '356', '356', 'PALMEIRA STS' ],
            [ '358', '358', 'PALMEIRA STS' ],
            [ '361', '361', 'PALMEIRA STS' ],
            [ '652', '680', 'PALMEIRA STS' ],
            [ '730', '730', 'PALMEIRA STS' ] ] },
        'Sequeir' => {
          '4780' => [
            [ '587', '651', 'SEQUEIR' ] ] },
        'brego' => {
          '4795' => [
            [ '782', '789', 'VILARINHO STS' ] ] },
        'Vila das Aves' => {
          '4795' => [
            [ '001', '018', 'AVES' ],
            [ '020', '024', 'AVES' ],
            [ '026', '054', 'AVES' ],
            [ '056', '083', 'AVES' ],
            [ '085', '132', 'AVES' ],
            [ '135', '154', 'AVES' ],
            [ '156', '157', 'AVES' ],
            [ '841', '845', 'AVES' ],
            [ '896', '896', 'AVES' ] ] },
        'So Salvador do Campo' => {
          '4795' => [
            [ '515', '526', 'SO SALVADOR DO CAMPO' ],
            [ '528', '533', 'SO SALVADOR DO CAMPO' ],
            [ '535', '535', 'SO SALVADOR DO CAMPO' ],
            [ '538', '538', 'SO SALVADOR DO CAMPO' ],
            [ '542', '542', 'SO SALVADOR DO CAMPO' ],
            [ '544', '544', 'SO SALVADOR DO CAMPO' ],
            [ '871', '877', 'SO SALVADOR DO CAMPO' ] ] },
        'Adegas' => {
          '4795' => [
            [ '385', '441', 'SO MAMEDE NEGRELOS' ] ] },
        'Crasto' => {
          '4795' => [
            [ '870', '870', 'VILARINHO STS' ] ] },
        'Burges' => {
          '4780' => [
            [ '098', '244', 'SANTO TIRSO' ] ] },
        'Lage' => {
          '4795' => [
            [ '795', '796', 'VILARINHO STS' ] ] },
        'So Miguel Couto' => {
          '4780' => [
            [ '681', '721', 'SANTO TIRSO' ] ] },
        'Aldeia Nova' => {
          '4825' => [
            [ '234', '259', 'LAMELAS' ] ],
          '4780' => [
            [ '246', '247', 'SANTO TIRSO' ] ] },
        'So Mamede de Negrelos' => {
          '4795' => [
            [ '878', '895', 'SO MAMEDE NEGRELOS' ] ] },
        'Falco' => {
          '4795' => [
            [ '791', '793', 'VILARINHO STS' ] ] },
        'So Martinho do Campo' => {
          '4795' => [
            [ '442', '470', 'SO MARTINHO CAMPO' ],
            [ '472', '482', 'SO MARTINHO CAMPO' ],
            [ '485', '490', 'SO MARTINHO CAMPO' ],
            [ '492', '501', 'SO MARTINHO CAMPO' ],
            [ '504', '504', 'SO MARTINHO CAMPO' ],
            [ '506', '511', 'SO MARTINHO CAMPO' ],
            [ '513', '514', 'SO MARTINHO CAMPO' ],
            [ '802', '802', 'SO MARTINHO CAMPO' ],
            [ '846', '846', 'SO MARTINHO CAMPO' ] ] },
        'Almunha' => {
          '4780' => [
            [ '265', '321', 'LAMA STS' ] ] },
        'Agras' => {
          '4795' => [
            [ '803', '840', 'VILARINHO STS' ] ] },
        'Areias' => {
          '4780' => [
            [ '001', '097', 'AREIAS STS' ],
            [ '731', '731', 'AREIAS STS' ] ] },
        'Rebordes' => {
          '4795' => [
            [ '158', '161', 'REBORDES' ],
            [ '163', '167', 'REBORDES' ],
            [ '169', '169', 'REBORDES' ],
            [ '171', '175', 'REBORDES' ],
            [ '177', '179', 'REBORDES' ],
            [ '182', '192', 'REBORDES' ],
            [ '194', '220', 'REBORDES' ],
            [ '222', '246', 'REBORDES' ],
            [ '248', '252', 'REBORDES' ],
            [ '255', '255', 'REBORDES' ],
            [ '847', '869', 'REBORDES' ] ] },
        'So Tom' => {
          '4795' => [
            [ '545', '781', 'SO TOM NEGRELOS' ] ] },
        'Lama' => {
          '4780' => [
            [ '732', '732', 'LAMA STS' ] ] },
        'Roriz' => {
          '4795' => [
            [ '256', '384', 'RORIZ STS' ] ] },
        'Igreja - Couto (So Miguel)' => {
          '4780' => [
            [ '256', '256', 'SANTO TIRSO' ] ] },
        'Refojos de Riba de Ave' => {
          '4825' => [
            [ '292', '359', 'REFOJOS DE RIBA DE AVE' ] ] },
        'Reguenga' => {
          '4825' => [
            [ '360', '425', 'REGUENGA' ],
            [ '428', '436', 'REGUENGA' ] ] },
        'Agrela' => {
          '4825' => [
            [ '001', '062', 'AGRELA STS' ] ] },
        'Carreira' => {
          '4825' => [
            [ '135', '233', 'SANTO TIRSO' ],
            [ '442', '442', 'SANTO TIRSO' ] ] },
        'Salgueirinhos' => {
          '4795' => [
            [ '798', '801', 'VILARINHO STS' ] ] },
        'Bomjardim' => {
          '4780' => [
            [ '251', '251', 'SANTO TIRSO' ] ] },
        'Agrelo' => {
          '4825' => [
            [ '260', '291', 'SANTO TIRSO' ] ] },
        'Sandim' => {
          '4780' => [
            [ '259', '264', 'SANTO TIRSO' ] ] },
        'gua Longa' => {
          '4825' => [
            [ '063', '067', 'GUA LONGA' ],
            [ '069', '105', 'GUA LONGA' ],
            [ '107', '132', 'GUA LONGA' ],
            [ '437', '441', 'GUA LONGA' ] ] }
          },
      'Trofa' => {
        'So Martinho Bougado' => {
          '4785' => [
            [ '340', '341', 'TROFA' ] ] },
        'Aldeia' => {
          '4745' => [
            [ '201', '269', 'GUIDES' ] ] },
        'Finzes' => {
          '4785' => [
            [ '187', '187', 'TROFA' ],
            [ '227', '309', 'TROFA' ],
            [ '358', '363', 'TROFA' ],
            [ '365', '402', 'TROFA' ] ] },
        'Agra Cana' => {
          '4745' => [
            [ '301', '362', 'MURO' ] ] },
        'Grova' => {
          '4745' => [
            [ '010', '120', 'ALVARELHOS TRF' ] ] },
        'Lagoa' => {
          '4785' => [
            [ '344', '356', 'TROFA' ],
            [ '596', '686', 'TROFA' ] ] },
        'Aldeias de Cima' => {
          '4785' => [
            [ '500', '534', 'TROFA' ],
            [ '536', '594', 'TROFA' ],
            [ '688', '703', 'TROFA' ],
            [ '705', '716', 'TROFA' ] ] },
        'Aldeia Nova' => {
          '4785' => [
            [ '011', '071', 'COVELAS TRF' ] ] },
        'Carvoeiras' => {
          '4745' => [
            [ '507', '630', 'SO ROMO CORONADO' ] ] },
        'Gandara' => {
          '4785' => [
            [ '189', '225', 'TROFA' ] ] },
        'Coroa' => {
          '4785' => [
            [ '099', '185', 'TROFA' ] ] },
        'Covelo' => {
          '4745' => [
            [ '381', '506', 'SO MAMEDE CORONADO' ] ] },
        'Paranho' => {
          '4785' => [
            [ '311', '338', 'TROFA' ] ] }
          },
      'Maia' => {
        'Maia' => {
          '4470' => [
            [ '001', '021', 'MAIA' ],
            [ '023', '030', 'MAIA' ],
            [ '032', '032', 'MAIA' ],
            [ '034', '034', 'MAIA' ],
            [ '037', '037', 'MAIA' ],
            [ '039', '047', 'MAIA' ],
            [ '049', '063', 'MAIA' ],
            [ '065', '077', 'MAIA' ],
            [ '079', '103', 'MAIA' ],
            [ '106', '122', 'MAIA' ],
            [ '125', '128', 'MAIA' ],
            [ '130', '149', 'MAIA' ],
            [ '151', '174', 'MAIA' ],
            [ '176', '179', 'MAIA' ],
            [ '181', '181', 'MAIA' ],
            [ '183', '184', 'MAIA' ],
            [ '187', '193', 'MAIA' ],
            [ '195', '199', 'MAIA' ],
            [ '202', '202', 'MAIA' ],
            [ '205', '210', 'MAIA' ],
            [ '212', '216', 'MAIA' ],
            [ '218', '221', 'MAIA' ],
            [ '223', '231', 'MAIA' ],
            [ '233', '235', 'MAIA' ],
            [ '237', '261', 'MAIA' ],
            [ '263', '264', 'MAIA' ],
            [ '267', '270', 'MAIA' ],
            [ '272', '273', 'MAIA' ],
            [ '275', '282', 'MAIA' ],
            [ '284', '284', 'MAIA' ],
            [ '286', '291', 'MAIA' ],
            [ '294', '298', 'MAIA' ],
            [ '300', '300', 'MAIA' ],
            [ '302', '302', 'MAIA' ],
            [ '304', '314', 'MAIA' ],
            [ '316', '328', 'MAIA' ],
            [ '330', '346', 'MAIA' ],
            [ '348', '355', 'MAIA' ],
            [ '357', '358', 'MAIA' ],
            [ '360', '360', 'MAIA' ],
            [ '362', '372', 'MAIA' ],
            [ '374', '374', 'MAIA' ],
            [ '377', '384', 'MAIA' ],
            [ '387', '402', 'MAIA' ],
            [ '404', '430', 'MAIA' ],
            [ '432', '521', 'MAIA' ],
            [ '523', '524', 'MAIA' ],
            [ '526', '529', 'MAIA' ],
            [ '551', '561', 'MAIA' ],
            [ '563', '563', 'MAIA' ],
            [ '566', '570', 'MAIA' ],
            [ '572', '580', 'MAIA' ],
            [ '582', '594', 'MAIA' ],
            [ '596', '596', 'MAIA' ],
            [ '598', '625', 'MAIA' ],
            [ '627', '628', 'MAIA' ],
            [ '630', '635', 'MAIA' ],
            [ '637', '650', 'MAIA' ],
            [ '652', '659', 'MAIA' ],
            [ '701', '719', 'MAIA' ],
            [ '724', '725', 'MAIA' ],
            [ '727', '729', 'MAIA' ],
            [ '732', '733', 'MAIA' ],
            [ '735', '735', 'MAIA' ],
            [ '738', '740', 'MAIA' ],
            [ '743', '747', 'MAIA' ],
            [ '750', '751', 'MAIA' ],
            [ '753', '754', 'MAIA' ],
            [ '756', '758', 'MAIA' ],
            [ '761', '773', 'MAIA' ],
            [ '777', '778', 'MAIA' ],
            [ '782', '782', 'MAIA' ],
            [ '784', '784', 'MAIA' ],
            [ '786', '786', 'MAIA' ],
            [ '790', '791', 'MAIA' ],
            [ '794', '794', 'MAIA' ],
            [ '799', '799', 'MAIA' ],
            [ '804', '804', 'MAIA' ],
            [ '806', '806', 'MAIA' ],
            [ '808', '808', 'MAIA' ],
            [ '812', '812', 'MAIA' ],
            [ '819', '821', 'MAIA' ],
            [ '824', '826', 'MAIA' ],
            [ '828', '828', 'MAIA' ],
            [ '833', '833', 'MAIA' ],
            [ '840', '840', 'MAIA' ] ],
          '4474' => [
            [ '001', '015', 'MAIA' ] ],
          '4479' => [
            [ '001', '003', 'MAIA' ] ],
          '4475' => [
            [ '001', '004', 'MAIA' ],
            [ '006', '006', 'MAIA' ],
            [ '008', '010', 'MAIA' ],
            [ '012', '016', 'MAIA' ],
            [ '018', '027', 'MAIA' ],
            [ '029', '034', 'MAIA' ],
            [ '036', '037', 'MAIA' ],
            [ '039', '040', 'MAIA' ],
            [ '042', '042', 'MAIA' ],
            [ '044', '044', 'MAIA' ],
            [ '047', '054', 'MAIA' ],
            [ '056', '071', 'MAIA' ],
            [ '073', '100', 'MAIA' ],
            [ '102', '117', 'MAIA' ],
            [ '119', '131', 'MAIA' ],
            [ '134', '135', 'MAIA' ],
            [ '138', '142', 'MAIA' ],
            [ '144', '146', 'MAIA' ],
            [ '148', '148', 'MAIA' ],
            [ '150', '213', 'MAIA' ],
            [ '215', '233', 'MAIA' ],
            [ '235', '268', 'MAIA' ],
            [ '270', '274', 'MAIA' ],
            [ '276', '302', 'MAIA' ],
            [ '304', '309', 'MAIA' ],
            [ '311', '314', 'MAIA' ],
            [ '316', '318', 'MAIA' ],
            [ '320', '340', 'MAIA' ],
            [ '342', '343', 'MAIA' ],
            [ '346', '351', 'MAIA' ],
            [ '353', '365', 'MAIA' ],
            [ '367', '371', 'MAIA' ],
            [ '374', '377', 'MAIA' ],
            [ '380', '389', 'MAIA' ],
            [ '391', '404', 'MAIA' ],
            [ '406', '410', 'MAIA' ],
            [ '451', '463', 'MAIA' ],
            [ '465', '466', 'MAIA' ],
            [ '468', '476', 'MAIA' ],
            [ '478', '480', 'MAIA' ],
            [ '482', '483', 'MAIA' ],
            [ '485', '490', 'MAIA' ],
            [ '492', '500', 'MAIA' ],
            [ '503', '503', 'MAIA' ],
            [ '505', '507', 'MAIA' ],
            [ '509', '509', 'MAIA' ],
            [ '511', '511', 'MAIA' ],
            [ '513', '513', 'MAIA' ],
            [ '522', '525', 'MAIA' ],
            [ '601', '601', 'MAIA' ],
            [ '603', '603', 'MAIA' ],
            [ '606', '607', 'MAIA' ],
            [ '610', '611', 'MAIA' ],
            [ '613', '615', 'MAIA' ],
            [ '617', '617', 'MAIA' ],
            [ '620', '622', 'MAIA' ],
            [ '625', '625', 'MAIA' ],
            [ '627', '627', 'MAIA' ],
            [ '629', '629', 'MAIA' ],
            [ '631', '638', 'MAIA' ],
            [ '640', '646', 'MAIA' ],
            [ '651', '654', 'MAIA' ],
            [ '657', '663', 'MAIA' ],
            [ '669', '671', 'MAIA' ],
            [ '673', '676', 'MAIA' ],
            [ '678', '700', 'MAIA' ],
            [ '703', '707', 'MAIA' ],
            [ '709', '720', 'MAIA' ],
            [ '722', '724', 'MAIA' ],
            [ '726', '733', 'MAIA' ],
            [ '801', '802', 'MAIA' ],
            [ '804', '806', 'MAIA' ],
            [ '810', '813', 'MAIA' ],
            [ '815', '817', 'MAIA' ],
            [ '819', '823', 'MAIA' ],
            [ '825', '826', 'MAIA' ],
            [ '828', '831', 'MAIA' ],
            [ '833', '842', 'MAIA' ],
            [ '845', '846', 'MAIA' ],
            [ '850', '850', 'MAIA' ] ],
          '4425' => [
            [ '001', '013', 'MAIA' ],
            [ '015', '020', 'MAIA' ],
            [ '022', '025', 'MAIA' ],
            [ '027', '028', 'MAIA' ],
            [ '030', '033', 'MAIA' ],
            [ '035', '038', 'MAIA' ],
            [ '040', '059', 'MAIA' ],
            [ '061', '065', 'MAIA' ],
            [ '067', '090', 'MAIA' ],
            [ '092', '094', 'MAIA' ],
            [ '096', '135', 'MAIA' ],
            [ '137', '174', 'MAIA' ],
            [ '176', '176', 'MAIA' ],
            [ '178', '179', 'MAIA' ],
            [ '181', '184', 'MAIA' ],
            [ '187', '189', 'MAIA' ],
            [ '191', '192', 'MAIA' ],
            [ '194', '213', 'MAIA' ],
            [ '215', '215', 'MAIA' ],
            [ '217', '278', 'MAIA' ],
            [ '280', '290', 'MAIA' ],
            [ '300', '301', 'MAIA' ],
            [ '303', '304', 'MAIA' ],
            [ '307', '309', 'MAIA' ],
            [ '311', '318', 'MAIA' ],
            [ '321', '322', 'MAIA' ],
            [ '324', '326', 'MAIA' ],
            [ '328', '334', 'MAIA' ],
            [ '338', '340', 'MAIA' ],
            [ '342', '342', 'MAIA' ],
            [ '344', '386', 'MAIA' ],
            [ '388', '388', 'MAIA' ],
            [ '390', '394', 'MAIA' ],
            [ '396', '397', 'MAIA' ],
            [ '399', '436', 'MAIA' ],
            [ '438', '438', 'MAIA' ],
            [ '440', '464', 'MAIA' ],
            [ '500', '500', 'MAIA' ],
            [ '502', '506', 'MAIA' ],
            [ '508', '511', 'MAIA' ],
            [ '515', '516', 'MAIA' ],
            [ '518', '518', 'MAIA' ],
            [ '520', '526', 'MAIA' ],
            [ '528', '534', 'MAIA' ],
            [ '536', '543', 'MAIA' ],
            [ '545', '547', 'MAIA' ],
            [ '601', '609', 'MAIA' ],
            [ '611', '630', 'MAIA' ],
            [ '632', '643', 'MAIA' ],
            [ '645', '654', 'MAIA' ],
            [ '656', '666', 'MAIA' ],
            [ '668', '675', 'MAIA' ],
            [ '677', '677', 'MAIA' ],
            [ '679', '682', 'MAIA' ],
            [ '685', '702', 'MAIA' ] ] }
         },
      'Matosinhos' => {
        'Guifes' => {
          '4460' => [
            [ '001', '015', 'GUIFES' ],
            [ '017', '029', 'GUIFES' ],
            [ '031', '034', 'GUIFES' ],
            [ '037', '044', 'GUIFES' ],
            [ '047', '051', 'GUIFES' ],
            [ '053', '059', 'GUIFES' ],
            [ '060', '060', 'SENHORA DA HORA' ],
            [ '061', '101', 'GUIFES' ],
            [ '104', '108', 'GUIFES' ],
            [ '110', '125', 'GUIFES' ],
            [ '127', '127', 'GUIFES' ],
            [ '141', '141', 'GUIFES' ],
            [ '143', '143', 'GUIFES' ],
            [ '147', '151', 'GUIFES' ],
            [ '835', '835', 'GUIFES' ],
            [ '895', '898', 'GUIFES' ],
            [ '900', '903', 'GUIFES' ] ] },
        'Perafita' => {
          '4455' => [
            [ '263', '263', 'PERAFITA' ],
            [ '265', '266', 'PERAFITA' ],
            [ '271', '271', 'PERAFITA' ],
            [ '287', '290', 'PERAFITA' ],
            [ '400', '401', 'PERAFITA' ],
            [ '407', '408', 'PERAFITA' ],
            [ '410', '491', 'PERAFITA' ],
            [ '493', '498', 'PERAFITA' ],
            [ '500', '524', 'PERAFITA' ],
            [ '526', '531', 'PERAFITA' ],
            [ '533', '560', 'PERAFITA' ],
            [ '562', '596', 'PERAFITA' ],
            [ '598', '598', 'PERAFITA' ],
            [ '609', '609', 'PERAFITA' ],
            [ '804', '804', 'PERAFITA' ] ] },
        'So Mamede de Infesta' => {
          '4465' => [
            [ '001', '006', 'SO MAMEDE DE INFESTA' ],
            [ '011', '024', 'SO MAMEDE DE INFESTA' ],
            [ '026', '041', 'SO MAMEDE DE INFESTA' ],
            [ '043', '069', 'SO MAMEDE DE INFESTA' ],
            [ '071', '082', 'SO MAMEDE DE INFESTA' ],
            [ '084', '156', 'SO MAMEDE DE INFESTA' ],
            [ '158', '175', 'SO MAMEDE DE INFESTA' ],
            [ '177', '191', 'SO MAMEDE DE INFESTA' ],
            [ '196', '227', 'SO MAMEDE DE INFESTA' ],
            [ '229', '234', 'SO MAMEDE DE INFESTA' ],
            [ '236', '237', 'SO MAMEDE DE INFESTA' ],
            [ '239', '249', 'SO MAMEDE DE INFESTA' ],
            [ '251', '257', 'SO MAMEDE DE INFESTA' ],
            [ '261', '278', 'SO MAMEDE DE INFESTA' ],
            [ '280', '280', 'SO MAMEDE DE INFESTA' ],
            [ '283', '283', 'SO MAMEDE DE INFESTA' ],
            [ '288', '288', 'SO MAMEDE DE INFESTA' ],
            [ '291', '292', 'SO MAMEDE DE INFESTA' ],
            [ '295', '296', 'SO MAMEDE DE INFESTA' ],
            [ '298', '298', 'SO MAMEDE DE INFESTA' ],
            [ '300', '301', 'SO MAMEDE DE INFESTA' ],
            [ '307', '307', 'SO MAMEDE DE INFESTA' ],
            [ '309', '309', 'SO MAMEDE DE INFESTA' ],
            [ '317', '317', 'SO MAMEDE DE INFESTA' ],
            [ '328', '328', 'SO MAMEDE DE INFESTA' ],
            [ '351', '356', 'SO MAMEDE DE INFESTA' ],
            [ '358', '360', 'SO MAMEDE DE INFESTA' ],
            [ '686', '686', 'SO MAMEDE DE INFESTA' ],
            [ '689', '691', 'SO MAMEDE DE INFESTA' ] ] },
        'Santa Cruz do Bispo' => {
          '4455' => [
            [ '561', '561', 'SANTA CRUZ DO BISPO' ],
            [ '791', '794', 'SANTA CRUZ DO BISPO' ],
            [ '796', '803', 'SANTA CRUZ DO BISPO' ],
            [ '805', '818', 'SANTA CRUZ DO BISPO' ],
            [ '820', '822', 'SANTA CRUZ DO BISPO' ],
            [ '824', '831', 'SANTA CRUZ DO BISPO' ],
            [ '833', '847', 'SANTA CRUZ DO BISPO' ],
            [ '849', '849', 'SANTA CRUZ DO BISPO' ],
            [ '851', '851', 'SANTA CRUZ DO BISPO' ],
            [ '853', '860', 'SANTA CRUZ DO BISPO' ],
            [ '866', '867', 'SANTA CRUZ DO BISPO' ],
            [ '872', '873', 'SANTA CRUZ DO BISPO' ],
            [ '876', '876', 'SANTA CRUZ DO BISPO' ] ] },
        'Matosinhos' => {
          '4450' => [
            [ '001', '007', 'MATOSINHOS' ],
            [ '009', '019', 'MATOSINHOS' ],
            [ '021', '049', 'MATOSINHOS' ],
            [ '051', '090', 'MATOSINHOS' ],
            [ '092', '099', 'MATOSINHOS' ],
            [ '101', '104', 'MATOSINHOS' ],
            [ '106', '110', 'MATOSINHOS' ],
            [ '113', '139', 'MATOSINHOS' ],
            [ '141', '159', 'MATOSINHOS' ],
            [ '161', '168', 'MATOSINHOS' ],
            [ '170', '195', 'MATOSINHOS' ],
            [ '197', '208', 'MATOSINHOS' ],
            [ '210', '218', 'MATOSINHOS' ],
            [ '220', '235', 'MATOSINHOS' ],
            [ '237', '287', 'MATOSINHOS' ],
            [ '289', '302', 'MATOSINHOS' ],
            [ '304', '310', 'MATOSINHOS' ],
            [ '312', '312', 'MATOSINHOS' ],
            [ '320', '320', 'MATOSINHOS' ],
            [ '324', '325', 'MATOSINHOS' ],
            [ '329', '330', 'MATOSINHOS' ],
            [ '332', '332', 'MATOSINHOS' ],
            [ '337', '337', 'MATOSINHOS' ],
            [ '340', '340', 'MATOSINHOS' ],
            [ '342', '343', 'MATOSINHOS' ],
            [ '346', '347', 'MATOSINHOS' ],
            [ '358', '358', 'MATOSINHOS' ],
            [ '365', '365', 'MATOSINHOS' ],
            [ '370', '370', 'MATOSINHOS' ],
            [ '374', '374', 'MATOSINHOS' ],
            [ '379', '379', 'MATOSINHOS' ],
            [ '381', '381', 'MATOSINHOS' ],
            [ '383', '396', 'MATOSINHOS' ],
            [ '398', '402', 'MATOSINHOS' ],
            [ '404', '405', 'MATOSINHOS' ],
            [ '407', '408', 'MATOSINHOS' ],
            [ '412', '413', 'MATOSINHOS' ],
            [ '459', '459', 'MATOSINHOS' ],
            [ '475', '475', 'MATOSINHOS' ],
            [ '563', '570', 'MATOSINHOS' ],
            [ '573', '574', 'MATOSINHOS' ],
            [ '576', '579', 'MATOSINHOS' ],
            [ '582', '608', 'MATOSINHOS' ],
            [ '610', '610', 'MATOSINHOS' ],
            [ '612', '618', 'MATOSINHOS' ],
            [ '620', '716', 'MATOSINHOS' ],
            [ '718', '727', 'MATOSINHOS' ],
            [ '730', '730', 'MATOSINHOS' ],
            [ '733', '764', 'MATOSINHOS' ],
            [ '766', '773', 'MATOSINHOS' ],
            [ '776', '803', 'MATOSINHOS' ],
            [ '805', '806', 'MATOSINHOS' ],
            [ '808', '808', 'MATOSINHOS' ],
            [ '819', '819', 'MATOSINHOS' ],
            [ '832', '832', 'MATOSINHOS' ] ],
          '4454' => [
            [ '502', '508', 'MATOSINHOS' ],
            [ '510', '514', 'MATOSINHOS' ] ] },
        'Lavra' => {
          '4455' => [
            [ '001', '022', 'LAVRA' ],
            [ '024', '024', 'LAVRA' ],
            [ '026', '072', 'LAVRA' ],
            [ '074', '075', 'LAVRA' ],
            [ '077', '113', 'LAVRA' ],
            [ '115', '120', 'LAVRA' ],
            [ '122', '159', 'LAVRA' ],
            [ '161', '161', 'LAVRA' ],
            [ '163', '182', 'LAVRA' ],
            [ '184', '218', 'LAVRA' ],
            [ '220', '220', 'LAVRA' ],
            [ '224', '224', 'LAVRA' ],
            [ '227', '227', 'LAVRA' ],
            [ '234', '234', 'LAVRA' ],
            [ '236', '237', 'LAVRA' ],
            [ '240', '240', 'LAVRA' ],
            [ '253', '253', 'LAVRA' ],
            [ '301', '301', 'LAVRA' ],
            [ '525', '525', 'LAVRA' ] ] },
        'Senhora da Hora' => {
          '4460' => [
            [ '140', '140', 'SENHORA DA HORA' ],
            [ '169', '169', 'SENHORA DA HORA' ],
            [ '189', '189', 'SENHORA DA HORA' ],
            [ '191', '192', 'SENHORA DA HORA' ],
            [ '199', '202', 'SENHORA DA HORA' ],
            [ '205', '205', 'SENHORA DA HORA' ],
            [ '208', '208', 'SENHORA DA HORA' ],
            [ '210', '217', 'SENHORA DA HORA' ],
            [ '219', '230', 'SENHORA DA HORA' ],
            [ '232', '263', 'SENHORA DA HORA' ],
            [ '265', '312', 'SENHORA DA HORA' ],
            [ '314', '331', 'SENHORA DA HORA' ],
            [ '333', '373', 'SENHORA DA HORA' ],
            [ '375', '379', 'SENHORA DA HORA' ],
            [ '381', '393', 'SENHORA DA HORA' ],
            [ '395', '403', 'SENHORA DA HORA' ],
            [ '406', '406', 'SENHORA DA HORA' ],
            [ '409', '411', 'SENHORA DA HORA' ],
            [ '414', '414', 'SENHORA DA HORA' ],
            [ '416', '433', 'SENHORA DA HORA' ],
            [ '435', '442', 'SENHORA DA HORA' ],
            [ '444', '445', 'SENHORA DA HORA' ],
            [ '451', '452', 'SENHORA DA HORA' ],
            [ '454', '455', 'SENHORA DA HORA' ],
            [ '460', '460', 'SENHORA DA HORA' ],
            [ '466', '466', 'SENHORA DA HORA' ],
            [ '468', '468', 'SENHORA DA HORA' ],
            [ '470', '470', 'SENHORA DA HORA' ],
            [ '472', '472', 'SENHORA DA HORA' ],
            [ '479', '488', 'SENHORA DA HORA' ],
            [ '492', '494', 'SENHORA DA HORA' ],
            [ '812', '813', 'SENHORA DA HORA' ],
            [ '828', '828', 'SENHORA DA HORA' ],
            [ '841', '841', 'SENHORA DA HORA' ] ],
          '4464' => [
            [ '501', '505', 'SENHORA DA HORA' ],
            [ '513', '513', 'SENHORA DA HORA' ] ] },
        'Custias' => {
          '4460' => [
            [ '152', '157', 'CUSTIAS MTS' ],
            [ '190', '190', 'CUSTIAS MTS' ],
            [ '193', '196', 'CUSTIAS MTS' ],
            [ '198', '198', 'CUSTIAS MTS' ],
            [ '203', '204', 'CUSTIAS MTS' ],
            [ '206', '207', 'CUSTIAS MTS' ],
            [ '209', '209', 'CUSTIAS MTS' ],
            [ '415', '415', 'CUSTIAS MTS' ],
            [ '552', '552', 'CUSTIAS MTS' ],
            [ '561', '563', 'CUSTIAS MTS' ],
            [ '633', '634', 'CUSTIAS MTS' ],
            [ '660', '756', 'CUSTIAS MTS' ],
            [ '758', '765', 'CUSTIAS MTS' ],
            [ '767', '774', 'CUSTIAS MTS' ],
            [ '776', '783', 'CUSTIAS MTS' ],
            [ '785', '811', 'CUSTIAS MTS' ],
            [ '814', '821', 'CUSTIAS MTS' ],
            [ '823', '827', 'CUSTIAS MTS' ],
            [ '829', '834', 'CUSTIAS MTS' ],
            [ '836', '840', 'CUSTIAS MTS' ],
            [ '843', '843', 'CUSTIAS MTS' ],
            [ '846', '847', 'CUSTIAS MTS' ],
            [ '849', '849', 'CUSTIAS MTS' ],
            [ '861', '861', 'CUSTIAS MTS' ],
            [ '864', '865', 'CUSTIAS MTS' ],
            [ '871', '871', 'CUSTIAS MTS' ],
            [ '881', '883', 'CUSTIAS MTS' ],
            [ '886', '889', 'CUSTIAS MTS' ],
            [ '891', '891', 'CUSTIAS MTS' ],
            [ '893', '893', 'CUSTIAS MTS' ] ] },
        'Lea do Balio' => {
          '4465' => [
            [ '008', '009', 'LEA DO BALIO' ],
            [ '157', '157', 'LEA DO BALIO' ],
            [ '176', '176', 'LEA DO BALIO' ],
            [ '192', '195', 'LEA DO BALIO' ],
            [ '235', '235', 'LEA DO BALIO' ],
            [ '258', '258', 'LEA DO BALIO' ],
            [ '294', '294', 'LEA DO BALIO' ],
            [ '332', '337', 'LEA DO BALIO' ],
            [ '339', '350', 'LEA DO BALIO' ],
            [ '540', '551', 'LEA DO BALIO' ],
            [ '553', '562', 'LEA DO BALIO' ],
            [ '564', '574', 'LEA DO BALIO' ],
            [ '576', '582', 'LEA DO BALIO' ],
            [ '584', '632', 'LEA DO BALIO' ],
            [ '635', '635', 'LEA DO BALIO' ],
            [ '637', '652', 'LEA DO BALIO' ],
            [ '654', '666', 'LEA DO BALIO' ],
            [ '669', '685', 'LEA DO BALIO' ],
            [ '687', '688', 'LEA DO BALIO' ],
            [ '692', '710', 'LEA DO BALIO' ],
            [ '713', '731', 'LEA DO BALIO' ],
            [ '733', '747', 'LEA DO BALIO' ],
            [ '749', '751', 'LEA DO BALIO' ],
            [ '753', '753', 'LEA DO BALIO' ],
            [ '755', '759', 'LEA DO BALIO' ],
            [ '761', '761', 'LEA DO BALIO' ],
            [ '764', '771', 'LEA DO BALIO' ],
            [ '773', '773', 'LEA DO BALIO' ],
            [ '775', '775', 'LEA DO BALIO' ],
            [ '780', '780', 'LEA DO BALIO' ],
            [ '784', '784', 'LEA DO BALIO' ],
            [ '797', '798', 'LEA DO BALIO' ],
            [ '807', '807', 'LEA DO BALIO' ] ] }
               },
      'Penafiel' => {
        'Taipas' => {
          '4575' => [
            [ '230', '231', 'ENTRE-OS-RIOS' ] ] },
        'So Julio' => {
          '4560' => [
            [ '050', '050', 'CAPELA' ] ] },
        'Quinta Lages' => {
          '4560' => [
            [ '601', '602', 'PENAFIEL' ] ] },
        'Pinheiral' => {
          '4575' => [
            [ '369', '378', 'PINHEIRO PNF' ] ] },
        'Fornos' => {
          '4560' => [
            [ '558', '569', 'PENAFIEL' ] ] },
        'Casal' => {
          '4575' => [
            [ '127', '128', 'CABEA SANTA' ] ] },
        'Salgueirinho' => {
          '4560' => [
            [ '541', '542', 'PENAFIEL' ],
            [ '771', '793', 'SANTIAGO DE SUBARRIFANA' ] ] },
        'Novelas' => {
          '4560' => [
            [ '262', '275', 'NOVELAS' ],
            [ '277', '282', 'NOVELAS' ],
            [ '284', '293', 'NOVELAS' ],
            [ '295', '306', 'NOVELAS' ],
            [ '308', '312', 'NOVELAS' ],
            [ '315', '315', 'NOVELAS' ],
            [ '317', '319', 'NOVELAS' ],
            [ '323', '327', 'NOVELAS' ] ] },
        'Boua' => {
          '4575' => [
            [ '531', '543', 'SEBOLIDO' ] ] },
        'Padrao' => {
          '4560' => [
            [ '041', '042', 'BUSTELO PNF' ] ] },
        'Barral' => {
          '4560' => [
            [ '135', '164', 'GUILHUFE' ] ] },
        'Aveleda' => {
          '4560' => [
            [ '556', '556', 'PENAFIEL' ] ] },
        'Devesa' => {
          '4560' => [
            [ '750', '750', 'PEROZELO' ] ] },
        'Barreiros' => {
          '4575' => [
            [ '561', '580', 'VALPEDRE' ] ] },
        'Bairro' => {
          '4575' => [
            [ '161', '185', 'CANELAS PNF' ] ],
          '4560' => [
            [ '121', '122', 'GALEGOS PNF' ] ] },
        'Reiberto' => {
          '4575' => [
            [ '582', '591', 'VALPEDRE' ] ] },
        'Couo' => {
          '4560' => [
            [ '363', '369', 'PAO DE SOUSA' ] ] },
        'Cabroelo' => {
          '4575' => [
            [ '200', '200', 'CAPELA' ] ] },
        'Ribeira' => {
          '4575' => [
            [ '505', '511', 'RIO DE MOINHOS PNF' ] ] },
        'Torre' => {
          '4575' => [
            [ '513', '517', 'RIO DE MOINHOS PNF' ] ] },
        'Fundo de Vila' => {
          '4560' => [
            [ '111', '112', 'FONTE ARCADA PNF' ] ] },
        'Casal de Ouro' => {
          '4560' => [
            [ '356', '361', 'PAO DE SOUSA' ] ] },
        'Cruzes' => {
          '4560' => [
            [ '871', '872', 'VILA COVA PNF' ] ] },
        'Mosqueiros' => {
          '4560' => [
            [ '755', '755', 'RANS' ] ] },
        'Lages' => {
          '4560' => [
            [ '231', '232', 'MILHUNDOS' ] ] },
        'Zona Envolvente' => {
          '4560' => [
            [ '611', '611', 'PENAFIEL' ] ] },
        'Vez de Aviz' => {
          '4560' => [
            [ '034', '034', 'ABRAGO' ] ] },
        'Quinta do Serrado' => {
          '4560' => [
            [ '514', '535', 'PENAFIEL' ] ] },
        'Aldeia' => {
          '4575' => [
            [ '081', '095', 'BOELHE' ],
            [ '391', '419', 'PORTELA PNF' ] ],
          '4560' => [
            [ '061', '063', 'CROCA' ] ] },
        'Pao de Sousa' => {
          '4560' => [
            [ '378', '394', 'PAO DE SOUSA' ] ],
          '4564' => [
            [ '006', '006', 'PAO DE SOUSA' ] ] },
        'Rio Mau' => {
          '4575' => [
            [ '628', '631', 'RIO MAU PNF' ] ] },
        'So Miguel' => {
          '4575' => [
            [ '226', '228', 'ENTRE-OS-RIOS' ] ] },
        'Cerrado' => {
          '4575' => [
            [ '241', '249', 'FIGUEIRA PNF' ] ] },
        'Irivo' => {
          '4560' => [
            [ '173', '173', 'IRIVO' ] ] },
        'Ribaais' => {
          '4560' => [
            [ '027', '027', 'ABRAGO' ] ] },
        'Galego' => {
          '4560' => [
            [ '800', '800', 'SO MAMEDE RECEZINHOS' ] ] },
        'Miragaia' => {
          '4560' => [
            [ '017', '017', 'ABRAGO' ] ] },
        'Quinta da Aveleda' => {
          '4560' => [
            [ '730', '730', 'PENAFIEL' ] ] },
        'Castelo' => {
          '4560' => [
            [ '044', '044', 'BUSTELO PNF' ] ] },
        'Penafiel' => {
          '4560' => [
            [ '451', '512', 'PENAFIEL' ],
            [ '539', '539', 'PENAFIEL' ],
            [ '546', '551', 'PENAFIEL' ],
            [ '553', '554', 'PENAFIEL' ],
            [ '595', '595', 'PENAFIEL' ],
            [ '597', '599', 'PENAFIEL' ],
            [ '604', '604', 'PENAFIEL' ],
            [ '621', '623', 'PENAFIEL' ],
            [ '627', '627', 'PENAFIEL' ],
            [ '632', '634', 'PENAFIEL' ],
            [ '636', '638', 'PENAFIEL' ],
            [ '641', '641', 'PENAFIEL' ],
            [ '656', '657', 'PENAFIEL' ],
            [ '659', '659', 'PENAFIEL' ],
            [ '679', '679', 'PENAFIEL' ],
            [ '682', '682', 'PENAFIEL' ],
            [ '686', '687', 'PENAFIEL' ],
            [ '694', '694', 'PENAFIEL' ],
            [ '705', '705', 'PENAFIEL' ],
            [ '708', '709', 'PENAFIEL' ],
            [ '719', '719', 'PENAFIEL' ],
            [ '875', '907', 'PENAFIEL' ] ],
          '4564' => [
            [ '001', '005', 'PENAFIEL' ],
            [ '007', '007', 'PENAFIEL' ],
            [ '858', '858', 'PENAFIEL' ] ] },
        'Igreja' => {
          '4560' => [
            [ '015', '015', 'ABRAGO' ] ] },
        'Amargoa' => {
          '4575' => [
            [ '351', '367', 'PINHEIRO PNF' ] ] },
        'Cepo' => {
          '4575' => [
            [ '130', '134', 'CABEA SANTA' ],
            [ '291', '307', 'PAREDES PNF' ] ] },
        'Agrelos' => {
          '4575' => [
            [ '121', '125', 'CABEA SANTA' ] ] },
        'Soutelo' => {
          '4560' => [
            [ '805', '807', 'SO MARTINHO RECEZINHOS' ] ] },
        'Beco' => {
          '4560' => [
            [ '572', '573', 'PENAFIEL' ] ] },
        'Pontes' => {
          '4560' => [
            [ '221', '223', 'MARECOS' ] ] },
        'Curadeiras' => {
          '4560' => [
            [ '590', '590', 'PENAFIEL' ] ] },
        'Aldeia de Baixo' => {
          '4560' => [
            [ '181', '201', 'LAGARES PNF' ] ] },
        'Campos da Agra' => {
          '4560' => [
            [ '071', '073', 'DUAS IGREJAS PNF' ] ] },
        'Agueiros' => {
          '4560' => [
            [ '341', '345', 'PAO DE SOUSA' ] ] },
        'Santa Marta' => {
          '4560' => [
            [ '765', '765', 'SANTA MARTA' ] ] },
        'Alm Ribeiro' => {
          '4575' => [
            [ '611', '626', 'RIO MAU PNF' ] ] },
        'Santiago de Subrrifana' => {
          '4560' => [
            [ '795', '796', 'SANTIAGO DE SUBARRIFANA' ] ] },
        'Pieres' => {
          '4560' => [
            [ '131', '133', 'GUILHUFE' ] ] },
        'Junqueira' => {
          '4560' => [
            [ '371', '376', 'PAO DE SOUSA' ] ] },
        'Quinta de Santa Eullia' => {
          '4560' => [
            [ '396', '419', 'PAO DE SOUSA' ] ] },
        'Outeiro' => {
          '4575' => [
            [ '097', '102', 'BOELHE' ] ] },
        'Agrela' => {
          '4575' => [
            [ '471', '503', 'RIO DE MOINHOS PNF' ] ] },
        'Abol' => {
          '4575' => [
            [ '211', '223', 'ENTRE-OS-RIOS' ] ] },
        'Chamusca' => {
          '4560' => [
            [ '055', '055', 'CASTELES PNF' ] ] },
        'Esplanada da Nobre' => {
          '4560' => [
            [ '577', '578', 'PENAFIEL' ] ] },
        'Fonte Carreira' => {
          '4575' => [
            [ '136', '146', 'CABEA SANTA' ] ] },
        'Miraval' => {
          '4560' => [
            [ '210', '210', 'LUZIM' ] ] },
        'Asa Velha' => {
          '4575' => [
            [ '261', '280', 'OLDRES' ] ] },
        'Urr' => {
          '4560' => [
            [ '822', '822', 'URR PNF' ],
            [ '825', '825', 'URR PNF' ],
            [ '830', '857', 'URR PNF' ],
            [ '860', '869', 'URR PNF' ] ] },
        'Assento' => {
          '4560' => [
            [ '347', '354', 'PAO DE SOUSA' ] ] }
             },
      'Paredes' => {
        'Soutinho' => {
          '4580' => [
            [ '387', '390', 'DUAS IGREJAS PRD' ] ] },
        'Alegrete' => {
          '4585' => [
            [ '581', '585', 'RECAREI' ] ] },
        'Ferreiros' => {
          '4580' => [
            [ '660', '677', 'VILELA PRD' ] ] },
        'Vinha' => {
          '4580' => [
            [ '695', '717', 'VILELA PRD' ] ] },
        'Rochao' => {
          '4585' => [
            [ '638', '638', 'RECAREI' ] ] },
        'Gandra' => {
          '4585' => [
            [ '066', '072', 'GANDRA PRD' ],
            [ '074', '074', 'GANDRA PRD' ],
            [ '076', '109', 'GANDRA PRD' ],
            [ '111', '123', 'GANDRA PRD' ],
            [ '126', '133', 'GANDRA PRD' ],
            [ '135', '175', 'GANDRA PRD' ],
            [ '177', '205', 'GANDRA PRD' ],
            [ '207', '231', 'GANDRA PRD' ],
            [ '233', '238', 'GANDRA PRD' ],
            [ '553', '574', 'GANDRA PRD' ] ] },
        'Mouriz' => {
          '4580' => [
            [ '601', '601', 'MOURIZ' ] ] },
        'Florido' => {
          '4580' => [
            [ '286', '290', 'BESTEIROS PRD' ] ] },
        'Chos' => {
          '4580' => [
            [ '297', '300', 'BITARES' ] ] },
        'Estrada' => {
          '4580' => [
            [ '657', '658', 'VILELA PRD' ] ] },
        'Roseiral' => {
          '4580' => [
            [ '234', '235', 'PAREDES' ] ] },
        'Soutilho' => {
          '4580' => [
            [ '275', '280', 'PAREDES' ] ] },
        'Pinheiro' => {
          '4580' => [
            [ '281', '283', 'BEIRE' ] ] },
        'Aldeia Nova' => {
          '4580' => [
            [ '374', '385', 'DUAS IGREJAS PRD' ] ] },
        'Cete' => {
          '4580' => [
            [ '309', '313', 'CETE' ],
            [ '315', '340', 'CETE' ] ] },
        'Santa Comba' => {
          '4585' => [
            [ '694', '699', 'SOBREIRA' ] ] },
        'Gondomar' => {
          '4585' => [
            [ '748', '769', 'VANDOMA' ] ] },
        'Cancelas' => {
          '4580' => [
            [ '005', '006', 'PAREDES' ] ] },
        'Corredoura' => {
          '4580' => [
            [ '606', '611', 'SOBROSA' ] ] },
        'Calvario' => {
          '4585' => [
            [ '587', '618', 'RECAREI' ] ] },
        'Bouzinde' => {
          '4580' => [
            [ '581', '582', 'LOUREDO PRD' ] ] },
        'Ribeira' => {
          '4580' => [
            [ '682', '682', 'VILELA PRD' ] ] },
        'Pinhal' => {
          '4580' => [
            [ '304', '305', 'BITARES' ] ] },
        'Vilar' => {
          '4580' => [
            [ '392', '394', 'DUAS IGREJAS PRD' ] ] },
        'Baltar' => {
          '4585' => [
            [ '011', '019', 'BALTAR' ],
            [ '021', '023', 'BALTAR' ],
            [ '025', '032', 'BALTAR' ],
            [ '035', '047', 'BALTAR' ],
            [ '050', '055', 'BALTAR' ],
            [ '057', '059', 'BALTAR' ],
            [ '061', '062', 'BALTAR' ] ] },
        'Carvalhal' => {
          '4580' => [
            [ '371', '372', 'DUAS IGREJAS PRD' ] ] },
        'Bairro Campos' => {
          '4580' => [
            [ '641', '641', 'VILELA PRD' ] ] },
        'Sobreira' => {
          '4585' => [
            [ '701', '702', 'SOBREIRA' ],
            [ '704', '713', 'SOBREIRA' ],
            [ '716', '716', 'SOBREIRA' ],
            [ '718', '720', 'SOBREIRA' ] ] },
        'Correcova' => {
          '4585' => [
            [ '255', '255', 'PARADA DE TODEIA' ] ] },
        'Boca' => {
          '4580' => [
            [ '730', '756', 'SOBROSA' ] ] },
        'Senhora da Guia' => {
          '4580' => [
            [ '239', '239', 'PAREDES' ] ] },
        'Rebordosa' => {
          '4585' => [
            [ '305', '305', 'REBORDOSA' ],
            [ '310', '310', 'REBORDOSA' ],
            [ '314', '314', 'REBORDOSA' ],
            [ '318', '318', 'REBORDOSA' ],
            [ '335', '343', 'REBORDOSA' ],
            [ '345', '353', 'REBORDOSA' ],
            [ '356', '360', 'REBORDOSA' ],
            [ '362', '365', 'REBORDOSA' ],
            [ '367', '367', 'REBORDOSA' ],
            [ '369', '370', 'REBORDOSA' ],
            [ '372', '383', 'REBORDOSA' ],
            [ '385', '385', 'REBORDOSA' ],
            [ '387', '387', 'REBORDOSA' ],
            [ '390', '398', 'REBORDOSA' ],
            [ '400', '403', 'REBORDOSA' ],
            [ '405', '411', 'REBORDOSA' ],
            [ '413', '418', 'REBORDOSA' ],
            [ '421', '429', 'REBORDOSA' ],
            [ '431', '433', 'REBORDOSA' ],
            [ '435', '442', 'REBORDOSA' ],
            [ '444', '464', 'REBORDOSA' ],
            [ '466', '468', 'REBORDOSA' ],
            [ '470', '474', 'REBORDOSA' ],
            [ '476', '479', 'REBORDOSA' ],
            [ '482', '485', 'REBORDOSA' ],
            [ '487', '500', 'REBORDOSA' ],
            [ '503', '511', 'REBORDOSA' ],
            [ '513', '513', 'REBORDOSA' ],
            [ '516', '524', 'REBORDOSA' ],
            [ '526', '526', 'REBORDOSA' ],
            [ '528', '538', 'REBORDOSA' ],
            [ '829', '879', 'REBORDOSA' ] ] },
        'Matos' => {
          '4580' => [
            [ '613', '621', 'SOBROSA' ] ] },
        'Vilaqueixe' => {
          '4580' => [
            [ '307', '308', 'BITARES' ] ] },
        'Souto de Maia' => {
          '4580' => [
            [ '688', '690', 'VILELA PRD' ] ] },
        'Recarei' => {
          '4585' => [
            [ '620', '636', 'RECAREI' ] ] },
        'Seixoso' => {
          '4580' => [
            [ '684', '685', 'VILELA PRD' ] ] },
        'Pias' => {
          '4580' => [
            [ '230', '232', 'PAREDES' ] ] },
        'Lordelo' => {
          '4580' => [
            [ '411', '434', 'LORDELO PRD' ],
            [ '436', '437', 'LORDELO PRD' ],
            [ '439', '459', 'LORDELO PRD' ],
            [ '461', '467', 'LORDELO PRD' ],
            [ '469', '472', 'LORDELO PRD' ],
            [ '476', '481', 'LORDELO PRD' ],
            [ '483', '496', 'LORDELO PRD' ],
            [ '505', '505', 'LORDELO PRD' ],
            [ '511', '514', 'LORDELO PRD' ],
            [ '518', '520', 'LORDELO PRD' ],
            [ '524', '528', 'LORDELO PRD' ],
            [ '530', '533', 'LORDELO PRD' ],
            [ '535', '536', 'LORDELO PRD' ],
            [ '538', '540', 'LORDELO PRD' ],
            [ '544', '544', 'LORDELO PRD' ],
            [ '546', '560', 'LORDELO PRD' ],
            [ '562', '562', 'LORDELO PRD' ],
            [ '565', '565', 'LORDELO PRD' ],
            [ '794', '869', 'LORDELO PRD' ] ] },
        'Vila Me' => {
          '4580' => [
            [ '630', '630', 'VILA COVA DE CARROS' ] ] },
        'Bengada' => {
          '4580' => [
            [ '401', '402', 'GONDALES' ] ] },
        'Terronhas' => {
          '4585' => [
            [ '245', '248', 'GANDRA PRD' ],
            [ '640', '645', 'RECAREI' ] ] },
        'Castromil' => {
          '4585' => [
            [ '679', '689', 'SOBREIRA' ] ] },
        'Sedouros' => {
          '4580' => [
            [ '237', '237', 'PAREDES' ] ] },
        'Centies' => {
          '4580' => [
            [ '009', '013', 'PAREDES' ] ] },
        'Casa Boua' => {
          '4580' => [
            [ '648', '655', 'VILELA PRD' ] ] },
        'Astromil' => {
          '4585' => [
            [ '801', '828', 'ASTROMIL' ] ] },
        'Campo' => {
          '4580' => [
            [ '591', '597', 'MOURIZ' ] ] },
        'Longra' => {
          '4580' => [
            [ '351', '368', 'CRISTELO PRD' ] ] },
        'Parada de Todeia' => {
          '4585' => [
            [ '251', '252', 'PARADA DE TODEIA' ],
            [ '261', '268', 'PARADA DE TODEIA' ],
            [ '270', '279', 'PARADA DE TODEIA' ],
            [ '281', '287', 'PARADA DE TODEIA' ],
            [ '289', '295', 'PARADA DE TODEIA' ] ] },
        'Paredes' => {
          '4580' => [
            [ '001', '003', 'PAREDES' ],
            [ '024', '029', 'PAREDES' ],
            [ '032', '035', 'PAREDES' ],
            [ '037', '038', 'PAREDES' ],
            [ '041', '043', 'PAREDES' ],
            [ '045', '046', 'PAREDES' ],
            [ '048', '048', 'PAREDES' ],
            [ '052', '054', 'PAREDES' ],
            [ '056', '058', 'PAREDES' ],
            [ '061', '061', 'PAREDES' ],
            [ '063', '066', 'PAREDES' ],
            [ '069', '070', 'PAREDES' ],
            [ '072', '075', 'PAREDES' ],
            [ '077', '077', 'PAREDES' ],
            [ '083', '086', 'PAREDES' ],
            [ '091', '091', 'PAREDES' ],
            [ '094', '094', 'PAREDES' ],
            [ '100', '100', 'PAREDES' ],
            [ '102', '102', 'PAREDES' ],
            [ '104', '107', 'PAREDES' ],
            [ '109', '114', 'PAREDES' ],
            [ '116', '117', 'PAREDES' ],
            [ '119', '125', 'PAREDES' ],
            [ '127', '133', 'PAREDES' ],
            [ '135', '136', 'PAREDES' ],
            [ '140', '146', 'PAREDES' ],
            [ '149', '149', 'PAREDES' ],
            [ '152', '156', 'PAREDES' ],
            [ '159', '162', 'PAREDES' ],
            [ '165', '165', 'PAREDES' ],
            [ '167', '175', 'PAREDES' ],
            [ '180', '180', 'PAREDES' ],
            [ '186', '186', 'PAREDES' ],
            [ '188', '189', 'PAREDES' ],
            [ '193', '193', 'PAREDES' ],
            [ '195', '196', 'PAREDES' ],
            [ '198', '199', 'PAREDES' ],
            [ '201', '205', 'PAREDES' ],
            [ '207', '211', 'PAREDES' ],
            [ '214', '215', 'PAREDES' ],
            [ '218', '219', 'PAREDES' ],
            [ '223', '223', 'PAREDES' ],
            [ '226', '227', 'PAREDES' ],
            [ '245', '245', 'PAREDES' ],
            [ '247', '255', 'PAREDES' ],
            [ '258', '259', 'PAREDES' ],
            [ '265', '272', 'PAREDES' ] ] },
        'Varzea' => {
          '4585' => [
            [ '771', '772', 'VANDOMA' ] ] },
        'Trigais' => {
          '4580' => [
            [ '243', '243', 'PAREDES' ] ] },
        'Monte da Passagem' => {
          '4580' => [
            [ '016', '021', 'PAREDES' ] ] },
        'Boavista' => {
          '4580' => [
            [ '643', '646', 'VILELA PRD' ],
            [ '757', '776', 'BESTEIROS PRD' ] ] },
        'Agrela' => {
          '4580' => [
            [ '632', '639', 'VILELA PRD' ] ] },
        'Sentiais' => {
          '4580' => [
            [ '241', '241', 'PAREDES' ] ] },
        'Nogueiras' => {
          '4585' => [
            [ '257', '259', 'PARADA DE TODEIA' ] ] },
        'Senande' => {
          '4585' => [
            [ '001', '009', 'AGUIAR DE SOUSA' ] ] },
        'Rebolido' => {
          '4580' => [
            [ '291', '295', 'BITARES' ] ] },
        'Barreiras' => {
          '4580' => [
            [ '623', '629', 'SOBROSA' ] ] },
        'Agrelo' => {
          '4585' => [
            [ '731', '746', 'VANDOMA' ] ] },
        'Pomarinho' => {
          '4580' => [
            [ '719', '720', 'VILELA PRD' ] ] },
        'Portela' => {
          '4580' => [
            [ '777', '793', 'BITARES' ] ] },
        'Quinta das Portas' => {
          '4580' => [
            [ '679', '679', 'VILELA PRD' ] ] }
            },
      'Vila do Conde' => {
        'Azurara' => {
          '4480' => [
            [ '124', '124', 'AZURARA' ],
            [ '150', '188', 'AZURARA' ] ] },
        'Arcos' => {
          '4480' => [
            [ '011', '043', 'ARCOS VCD' ] ] },
        'Parada' => {
          '4480' => [
            [ '340', '340', 'PARADA VCD' ] ] },
        'Guilhabreu' => {
          '4485' => [
            [ '231', '285', 'GUILHABREU' ] ] },
        'Fajozes' => {
          '4485' => [
            [ '065', '105', 'FAJOZES' ] ] },
        'Gio' => {
          '4485' => [
            [ '171', '223', 'GIO VCD' ] ] },
        'Junqueira' => {
          '4480' => [
            [ '251', '311', 'JUNQUEIRA VCD' ],
            [ '313', '329', 'JUNQUEIRA VCD' ],
            [ '331', '335', 'JUNQUEIRA VCD' ] ] },
        'Vila Ch' => {
          '4485' => [
            [ '471', '471', 'VILA CH VCD' ],
            [ '476', '476', 'VILA CH VCD' ],
            [ '518', '518', 'VILA CH VCD' ],
            [ '524', '524', 'VILA CH VCD' ],
            [ '556', '556', 'VILA CH VCD' ],
            [ '689', '702', 'VILA CH VCD' ],
            [ '704', '728', 'VILA CH VCD' ],
            [ '730', '734', 'VILA CH VCD' ],
            [ '736', '748', 'VILA CH VCD' ],
            [ '750', '760', 'VILA CH VCD' ],
            [ '861', '869', 'VILA CH VCD' ],
            [ '872', '879', 'VILA CH VCD' ],
            [ '898', '898', 'VILA CH VCD' ] ] },
        'Mirante' => {
          '4485' => [
            [ '941', '942', 'VILAR DE PINHEIRO' ] ] },
        'Vilar' => {
          '4486' => [
            [ '855', '855', 'VILAR VCD' ] ],
          '4485' => [
            [ '761', '763', 'VILAR VCD' ],
            [ '765', '772', 'VILAR VCD' ],
            [ '776', '781', 'VILAR VCD' ],
            [ '783', '789', 'VILAR VCD' ],
            [ '791', '793', 'VILAR VCD' ],
            [ '795', '802', 'VILAR VCD' ],
            [ '804', '806', 'VILAR VCD' ],
            [ '809', '809', 'VILAR VCD' ],
            [ '811', '812', 'VILAR VCD' ],
            [ '814', '820', 'VILAR VCD' ],
            [ '870', '871', 'VILAR VCD' ] ] },
        'Ferreir' => {
          '4480' => [
            [ '250', '250', 'FERREIR' ] ] },
        'Vilar do Pinheiro' => {
          '4486' => [
            [ '854', '854', 'VILAR DE PINHEIRO' ] ],
          '4485' => [
            [ '016', '016', 'VILAR DE PINHEIRO' ],
            [ '821', '860', 'VILAR DE PINHEIRO' ],
            [ '880', '897', 'VILAR DE PINHEIRO' ] ] },
        'Vila do Conde' => {
          '4480' => [
            [ '001', '003', 'VILA DO CONDE' ],
            [ '650', '657', 'VILA DO CONDE' ],
            [ '659', '665', 'VILA DO CONDE' ],
            [ '667', '744', 'VILA DO CONDE' ],
            [ '746', '755', 'VILA DO CONDE' ],
            [ '757', '763', 'VILA DO CONDE' ],
            [ '765', '800', 'VILA DO CONDE' ],
            [ '802', '903', 'VILA DO CONDE' ],
            [ '905', '962', 'VILA DO CONDE' ] ] },
        'Touguinha' => {
          '4484' => [
            [ '850', '850', 'TOUGUINHA' ] ],
          '4480' => [
            [ '485', '490', 'TOUGUINHA' ],
            [ '492', '503', 'TOUGUINHA' ],
            [ '506', '521', 'TOUGUINHA' ],
            [ '523', '533', 'TOUGUINHA' ],
            [ '535', '541', 'TOUGUINHA' ],
            [ '543', '550', 'TOUGUINHA' ],
            [ '553', '553', 'TOUGUINHA' ] ] },
        'Vairo' => {
          '4485' => [
            [ '635', '638', 'VAIRO' ],
            [ '640', '688', 'VAIRO' ] ] },
        'Retorta' => {
          '4480' => [
            [ '351', '391', 'RETORTA' ] ] },
        'Outeiro Maior' => {
          '4480' => [
            [ '330', '330', 'OUTEIRO MAIOR' ] ] },
        'Labruge' => {
          '4485' => [
            [ '293', '362', 'LABRUGE' ] ] },
        'Touguinh' => {
          '4480' => [
            [ '491', '491', 'TOUGUINH' ],
            [ '504', '504', 'TOUGUINH' ],
            [ '522', '522', 'TOUGUINH' ],
            [ '551', '552', 'TOUGUINH' ],
            [ '554', '574', 'TOUGUINH' ],
            [ '576', '580', 'TOUGUINH' ],
            [ '582', '587', 'TOUGUINH' ],
            [ '589', '597', 'TOUGUINH' ] ] },
        'rvore' => {
          '4480' => [
            [ '051', '123', 'RVORE' ],
            [ '125', '145', 'RVORE' ],
            [ '601', '629', 'RVORE' ] ] },
        'Mosteir' => {
          '4485' => [
            [ '363', '370', 'MOSTEIR VCD' ],
            [ '899', '940', 'MOSTEIR VCD' ],
            [ '943', '944', 'MOSTEIR VCD' ] ] },
        'Bagunte' => {
          '4480' => [
            [ '201', '245', 'BAGUNTE' ] ] },
        'Tougues' => {
          '4480' => [
            [ '480', '480', 'TOUGUES' ] ] },
        'Canidelo' => {
          '4485' => [
            [ '060', '060', 'CANIDELO VCD' ] ] },
        'Rio Mau' => {
          '4480' => [
            [ '401', '425', 'RIO MAU VCD' ],
            [ '427', '459', 'RIO MAU VCD' ],
            [ '461', '478', 'RIO MAU VCD' ] ] },
        'Macieira da Maia' => {
          '4486' => [
            [ '853', '853', 'MACIEIRA DA MAIA' ] ],
          '4485' => [
            [ '371', '427', 'MACIEIRA DA MAIA' ] ] },
        'Modivas' => {
          '4486' => [
            [ '852', '852', 'MODIVAS' ] ],
          '4485' => [
            [ '572', '573', 'MODIVAS' ],
            [ '575', '591', 'MODIVAS' ],
            [ '593', '595', 'MODIVAS' ],
            [ '597', '598', 'MODIVAS' ],
            [ '601', '601', 'MODIVAS' ],
            [ '603', '620', 'MODIVAS' ] ] },
        'Malta' => {
          '4485' => [
            [ '431', '468', 'MALTA' ] ] },
        'Aveleda' => {
          '4485' => [
            [ '006', '015', 'AVELEDA VCD' ],
            [ '017', '021', 'AVELEDA VCD' ],
            [ '023', '041', 'AVELEDA VCD' ] ] },
        'Mindelo' => {
          '4486' => [
            [ '851', '851', 'MINDELO' ] ],
          '4485' => [
            [ '472', '475', 'MINDELO' ],
            [ '477', '517', 'MINDELO' ],
            [ '519', '523', 'MINDELO' ],
            [ '525', '555', 'MINDELO' ],
            [ '557', '571', 'MINDELO' ],
            [ '625', '631', 'MINDELO' ] ] },
        'Fornelo' => {
          '4485' => [
            [ '111', '120', 'FORNELO' ],
            [ '122', '135', 'FORNELO' ],
            [ '137', '143', 'FORNELO' ],
            [ '145', '149', 'FORNELO' ],
            [ '151', '163', 'FORNELO' ] ] }
            },
      'Baio' => {
        'Peso' => {
          '4640' => [
            [ '460', '472', 'SANTA MARINHA DO ZZERE' ] ] },
        'Cortinhas' => {
          '4640' => [
            [ '620', '627', 'VIARIZ' ] ] },
        'Grilo' => {
          '4640' => [
            [ '310', '310', 'GRILO' ] ] },
        'Cimo de Vila' => {
          '4640' => [
            [ '230', '241', 'GESTA' ] ] },
        'Pao' => {
          '4640' => [
            [ '210', '218', 'SO TOM DE COVELAS' ] ] },
        'Reixela' => {
          '4640' => [
            [ '380', '389', 'OVIL' ] ] },
        'Arrabalde' => {
          '5040' => [
            [ '051', '072', 'TEIXEIR' ] ] },
        'Fontainhas' => {
          '4640' => [
            [ '017', '042', 'ANCEDE' ] ] },
        'Tapada' => {
          '4640' => [
            [ '360', '360', 'MESQUINHATA' ] ] },
        'Soutelinho' => {
          '4640' => [
            [ '340', '345', 'LOIVOS DA RIBEIRA' ] ] },
        'Abadia' => {
          '4640' => [
            [ '001', '015', 'ANCEDE' ] ] },
        'Alvaro' => {
          '5040' => [
            [ '011', '039', 'TEIXEIRA BAO' ] ] },
        'Cruzes' => {
          '4640' => [
            [ '400', '404', 'RIBADOURO' ] ] },
        'Almofrela' => {
          '4640' => [
            [ '101', '115', 'BAIO' ] ] },
        'Baio' => {
          '4640' => [
            [ '117', '132', 'BAIO' ],
            [ '134', '181', 'BAIO' ] ] },
        'Outeiro' => {
          '4640' => [
            [ '320', '326', 'LOIVOS DO MONTE' ] ] },
        'Aldeia' => {
          '4640' => [
            [ '580', '582', 'VALADARES BAO' ] ] },
        'Laranjeira' => {
          '4640' => [
            [ '540', '547', 'TRESOURAS' ] ] },
        'Castanheira do Monte' => {
          '4640' => [
            [ '440', '450', 'BAIO (SANTA LEOCDIA)' ] ] },
        'Sobreiral' => {
          '4640' => [
            [ '260', '273', 'GOVE' ] ] },
        'Covela de Cima' => {
          '4640' => [
            [ '584', '594', 'VALADARES BAO' ] ] },
        'Portela' => {
          '4640' => [
            [ '220', '224', 'FRENDE' ] ] },
        'Barreiros' => {
          '4640' => [
            [ '300', '308', 'GRILO' ] ] },
        'Bairro' => {
          '4640' => [
            [ '420', '434', 'SANTA CRUZ DO DOURO' ] ] }
          },
      'Valongo' => {
        'Ermesinde' => {
          '4445' => [
            [ '275', '306', 'ERMESINDE' ],
            [ '308', '315', 'ERMESINDE' ],
            [ '317', '335', 'ERMESINDE' ],
            [ '338', '348', 'ERMESINDE' ],
            [ '350', '365', 'ERMESINDE' ],
            [ '368', '391', 'ERMESINDE' ],
            [ '393', '394', 'ERMESINDE' ],
            [ '396', '413', 'ERMESINDE' ],
            [ '415', '420', 'ERMESINDE' ],
            [ '422', '426', 'ERMESINDE' ],
            [ '428', '429', 'ERMESINDE' ],
            [ '432', '459', 'ERMESINDE' ],
            [ '461', '476', 'ERMESINDE' ],
            [ '478', '520', 'ERMESINDE' ],
            [ '522', '523', 'ERMESINDE' ],
            [ '525', '582', 'ERMESINDE' ],
            [ '584', '584', 'ERMESINDE' ],
            [ '586', '606', 'ERMESINDE' ],
            [ '608', '609', 'ERMESINDE' ],
            [ '611', '618', 'ERMESINDE' ],
            [ '620', '638', 'ERMESINDE' ],
            [ '640', '656', 'ERMESINDE' ],
            [ '658', '662', 'ERMESINDE' ],
            [ '664', '669', 'ERMESINDE' ],
            [ '671', '671', 'ERMESINDE' ] ] },
        'Alfena' => {
          '4445' => [
            [ '001', '001', 'ALFENA' ],
            [ '003', '030', 'ALFENA' ],
            [ '032', '033', 'ALFENA' ],
            [ '035', '094', 'ALFENA' ],
            [ '097', '099', 'ALFENA' ],
            [ '101', '192', 'ALFENA' ],
            [ '194', '261', 'ALFENA' ] ] },
        'Valongo' => {
          '4440' => [
            [ '488', '511', 'VALONGO' ],
            [ '513', '530', 'VALONGO' ],
            [ '532', '534', 'VALONGO' ],
            [ '536', '537', 'VALONGO' ],
            [ '539', '578', 'VALONGO' ],
            [ '580', '600', 'VALONGO' ],
            [ '602', '641', 'VALONGO' ],
            [ '643', '648', 'VALONGO' ],
            [ '650', '655', 'VALONGO' ],
            [ '657', '685', 'VALONGO' ],
            [ '687', '712', 'VALONGO' ],
            [ '714', '729', 'VALONGO' ],
            [ '732', '781', 'VALONGO' ],
            [ '783', '806', 'VALONGO' ],
            [ '808', '812', 'VALONGO' ],
            [ '815', '815', 'VALONGO' ],
            [ '817', '827', 'VALONGO' ],
            [ '829', '834', 'VALONGO' ],
            [ '837', '838', 'VALONGO' ] ] },
        'Sobrado' => {
          '4440' => [
            [ '293', '293', 'SOBRADO' ],
            [ '295', '315', 'SOBRADO' ],
            [ '317', '373', 'SOBRADO' ],
            [ '375', '397', 'SOBRADO' ],
            [ '399', '449', 'SOBRADO' ],
            [ '451', '456', 'SOBRADO' ] ] },
        'Campo' => {
          '4440' => [
            [ '001', '001', 'CAMPO VLG' ],
            [ '003', '220', 'CAMPO VLG' ],
            [ '222', '223', 'CAMPO VLG' ],
            [ '225', '233', 'CAMPO VLG' ] ] }
            },
      'Marco de Canaveses' => {
        'Castelinho' => {
          '4630' => [
            [ '023', '077', 'AVESSADAS' ] ] },
        'gua Levada' => {
          '4635' => [
            [ '063', '078', 'CONSTANCE' ] ] },
        'Poretla de Alfara' => {
          '4630' => [
            [ '717', '719', 'SOALHES' ] ] },
        'Alminhas' => {
          '4625' => [
            [ '001', '014', 'ARIZ MCN' ] ] },
        'Pinheiral' => {
          '4630' => [
            [ '174', '182', 'MANHUNCELOS' ] ] },
        'Constance' => {
          '4635' => [
            [ '080', '160', 'CONSTANCE' ],
            [ '763', '774', 'CONSTANCE' ] ] },
        'Casal' => {
          '4630' => [
            [ '157', '158', 'MANHUNCELOS' ] ] },
        'Alvelo' => {
          '4625' => [
            [ '571', '596', 'VILA BOA DO BISPO' ] ] },
        'Agrocho Alm' => {
          '4635' => [
            [ '352', '365', 'SOBRETMEGA' ] ] },
        'Quinta da Boavista' => {
          '4625' => [
            [ '476', '512', 'SANDE MCN' ] ] },
        'Penha Longa' => {
          '4625' => [
            [ '130', '130', 'PENHA LONGA' ] ] },
        'Torre de Cima' => {
          '4635' => [
            [ '657', '662', 'VRZEA OVELHA E ALIVIADA' ] ] },
        'Ruival' => {
          '4635' => [
            [ '643', '655', 'VRZEA OVELHA E ALIVIADA' ] ] },
        'Campos' => {
          '4635' => [
            [ '503', '528', 'TOUTOSA' ] ] },
        'Recho Forca' => {
          '4630' => [
            [ '731', '776', 'SOALHES' ] ] },
        'Babainho' => {
          '4575' => [
            [ '321', '338', 'TORRO MCN' ] ] },
        'Santo Isidoro' => {
          '4635' => [
            [ '248', '351', 'SANTO ISIDORO MCN' ] ] },
        'Arada' => {
          '4630' => [
            [ '148', '148', 'MANHUNCELOS' ] ] },
        'Agro' => {
          '4625' => [
            [ '136', '188', 'MAGRELOS' ] ] },
        'Abobreira' => {
          '4635' => [
            [ '161', '247', 'FOLHADA' ] ] },
        'Bitetos de Baixo' => {
          '4575' => [
            [ '431', '454', 'VRZEA DO DOURO' ] ] },
        'Bomba' => {
          '4575' => [
            [ '001', '026', 'ALPENDURADA E MATOS' ] ] },
        'Faves' => {
          '4625' => [
            [ '064', '124', 'FAVES' ] ] },
        'Moches' => {
          '4635' => [
            [ '614', '629', 'VRZEA OVELHA E ALIVIADA' ] ] },
        'Caldas de Canaveses' => {
          '4635' => [
            [ '367', '389', 'SOBRETMEGA' ] ] },
        'Quebrada' => {
          '4575' => [
            [ '043', '053', 'ALPENDURADA E MATOS' ] ] },
        'Prado' => {
          '4625' => [
            [ '361', '384', 'PENHA LONGA' ] ] },
        'Aidros' => {
          '4630' => [
            [ '001', '021', 'AVESSADAS' ] ] },
        'Caniva de Baixo' => {
          '4635' => [
            [ '682', '730', 'VILA BOA DE QUIRES' ] ] },
        'Pinheiro' => {
          '4635' => [
            [ '391', '415', 'SOBRETMEGA' ] ] },
        'Portela da Vrzea' => {
          '4630' => [
            [ '721', '729', 'SOALHES' ] ] },
        'Costa' => {
          '4630' => [
            [ '160', '172', 'MANHUNCELOS' ] ] },
        'Aborlide' => {
          '4630' => [
            [ '608', '618', 'SOALHES' ] ] },
        'Vila Boa do Bispo' => {
          '4625' => [
            [ '709', '731', 'VILA BOA DO BISPO' ] ] },
        'Aveleiras' => {
          '4630' => [
            [ '501', '536', 'MAURELES' ] ] },
        'Baloedo' => {
          '4630' => [
            [ '150', '155', 'MANHUNCELOS' ] ] },
        'Lamas' => {
          '4575' => [
            [ '028', '030', 'ALPENDURADA E MATOS' ] ],
          '4630' => [
            [ '690', '715', 'SOALHES' ] ] },
        'Ladario' => {
          '4635' => [
            [ '604', '612', 'VRZEA OVELHA E ALIVIADA' ] ] },
        'Alij' => {
          '4630' => [
            [ '146', '146', 'MANHUNCELOS' ] ] },
        'Casal Velho' => {
          '4635' => [
            [ '564', '593', 'VRZEA OVELHA E ALIVIADA' ] ] },
        'Bairro da Paz' => {
          '4625' => [
            [ '385', '402', 'ROSM' ] ] },
        'Pombal' => {
          '4635' => [
            [ '732', '762', 'VILA BOA DE QUIRES' ] ] },
        'Vilar' => {
          '4575' => [
            [ '060', '073', 'ALPENDURADA E MATOS' ] ] },
        'Memorial' => {
          '4575' => [
            [ '032', '041', 'ALPENDURADA E MATOS' ] ] },
        'Bacelo' => {
          '4630' => [
            [ '620', '688', 'SOALHES' ] ] },
        'Lameira' => {
          '4625' => [
            [ '598', '617', 'VILA BOA DO BISPO' ] ] },
        'Agrito' => {
          '4625' => [
            [ '513', '570', 'SO LOURENO DO DOURO' ] ] },
        'Agrela' => {
          '4635' => [
            [ '664', '680', 'VILA BOA DE QUIRES' ] ],
          '4625' => [
            [ '403', '471', 'SANDE MCN' ] ] },
        'Aldeia' => {
          '4635' => [
            [ '001', '062', 'BANHO E CARVALHOSA' ] ] },
        'Ariz' => {
          '4625' => [
            [ '016', '063', 'ARIZ MCN' ] ] },
        'Marco de Canaveses' => {
          '4630' => [
            [ '084', '084', 'MARCO DE CANAVESES' ],
            [ '087', '088', 'MARCO DE CANAVESES' ],
            [ '090', '102', 'MARCO DE CANAVESES' ],
            [ '104', '124', 'MARCO DE CANAVESES' ],
            [ '126', '127', 'MARCO DE CANAVESES' ],
            [ '142', '143', 'MARCO DE CANAVESES' ],
            [ '192', '192', 'MARCO DE CANAVESES' ],
            [ '195', '210', 'MARCO DE CANAVESES' ],
            [ '212', '212', 'MARCO DE CANAVESES' ],
            [ '214', '217', 'MARCO DE CANAVESES' ],
            [ '219', '252', 'MARCO DE CANAVESES' ],
            [ '254', '266', 'MARCO DE CANAVESES' ],
            [ '268', '268', 'MARCO DE CANAVESES' ],
            [ '270', '276', 'MARCO DE CANAVESES' ],
            [ '278', '281', 'MARCO DE CANAVESES' ],
            [ '283', '297', 'MARCO DE CANAVESES' ],
            [ '299', '303', 'MARCO DE CANAVESES' ],
            [ '305', '331', 'MARCO DE CANAVESES' ],
            [ '333', '333', 'MARCO DE CANAVESES' ],
            [ '335', '335', 'MARCO DE CANAVESES' ],
            [ '337', '337', 'MARCO DE CANAVESES' ],
            [ '339', '341', 'MARCO DE CANAVESES' ],
            [ '343', '346', 'MARCO DE CANAVESES' ],
            [ '350', '351', 'MARCO DE CANAVESES' ],
            [ '353', '353', 'MARCO DE CANAVESES' ],
            [ '355', '355', 'MARCO DE CANAVESES' ],
            [ '363', '363', 'MARCO DE CANAVESES' ],
            [ '371', '371', 'MARCO DE CANAVESES' ],
            [ '376', '377', 'MARCO DE CANAVESES' ],
            [ '389', '391', 'MARCO DE CANAVESES' ],
            [ '393', '393', 'MARCO DE CANAVESES' ],
            [ '401', '401', 'MARCO DE CANAVESES' ],
            [ '405', '405', 'MARCO DE CANAVESES' ],
            [ '408', '408', 'MARCO DE CANAVESES' ],
            [ '410', '412', 'MARCO DE CANAVESES' ],
            [ '415', '416', 'MARCO DE CANAVESES' ],
            [ '419', '420', 'MARCO DE CANAVESES' ],
            [ '423', '424', 'MARCO DE CANAVESES' ],
            [ '426', '426', 'MARCO DE CANAVESES' ],
            [ '428', '429', 'MARCO DE CANAVESES' ],
            [ '431', '431', 'MARCO DE CANAVESES' ],
            [ '438', '440', 'MARCO DE CANAVESES' ],
            [ '444', '444', 'MARCO DE CANAVESES' ],
            [ '446', '446', 'MARCO DE CANAVESES' ],
            [ '448', '448', 'MARCO DE CANAVESES' ],
            [ '450', '450', 'MARCO DE CANAVESES' ],
            [ '455', '455', 'MARCO DE CANAVESES' ],
            [ '457', '459', 'MARCO DE CANAVESES' ],
            [ '461', '461', 'MARCO DE CANAVESES' ],
            [ '463', '467', 'MARCO DE CANAVESES' ],
            [ '469', '479', 'MARCO DE CANAVESES' ],
            [ '482', '490', 'MARCO DE CANAVESES' ],
            [ '493', '497', 'MARCO DE CANAVESES' ],
            [ '500', '500', 'MARCO DE CANAVESES' ],
            [ '777', '787', 'MARCO DE CANAVESES' ] ] },
        'Portela Nova' => {
          '4635' => [
            [ '631', '641', 'VRZEA OVELHA E ALIVIADA' ] ] },
        'Quintas' => {
          '4625' => [
            [ '619', '707', 'VILA BOA DO BISPO' ] ] },
        'Tojal' => {
          '4575' => [
            [ '055', '058', 'ALPENDURADA E MATOS' ] ] },
        'Agrelo' => {
          '4625' => [
            [ '288', '359', 'PENHA LONGA' ] ] },
        'Toutosa' => {
          '4635' => [
            [ '530', '534', 'TOUTOSA' ],
            [ '775', '777', 'TOUTOSA' ] ] },
        'Adro' => {
          '4630' => [
            [ '537', '607', 'PAREDES DE VIADORES' ] ] },
        'Agrocho' => {
          '4635' => [
            [ '535', '562', 'VRZEA OVELHA E ALIVIADA' ] ] },
        'Agradadas' => {
          '4635' => [
            [ '416', '502', 'TABUADO' ] ] },
        'Fraima' => {
          '4635' => [
            [ '595', '602', 'VRZEA OVELHA E ALIVIADA' ] ] },
        'Portela' => {
          '4625' => [
            [ '473', '474', 'SANDE MCN' ] ] },
        'Adega' => {
          '4625' => [
            [ '189', '287', 'PAOS DE GAIOLO' ] ] }
                 },
      'Vila Nova de Gaia' => {
        'So Martinho' => {
          '4415' => [
            [ '754', '769', 'OLIVAL VNG' ] ] },
        'Ameal' => {
          '4415' => [
            [ '686', '718', 'OLIVAL VNG' ] ] },
        'Avintes' => {
          '4430' => [
            [ '285', '285', 'AVINTES' ],
            [ '635', '640', 'AVINTES' ],
            [ '648', '681', 'AVINTES' ],
            [ '687', '687', 'AVINTES' ],
            [ '756', '900', 'AVINTES' ],
            [ '902', '929', 'AVINTES' ],
            [ '932', '932', 'AVINTES' ] ] },
        'Grij' => {
          '4415' => [
            [ '100', '100', 'GRIJ' ],
            [ '111', '111', 'GRIJ' ],
            [ '127', '127', 'GRIJ' ],
            [ '399', '401', 'GRIJ' ],
            [ '416', '417', 'GRIJ' ],
            [ '421', '512', 'GRIJ' ],
            [ '514', '541', 'GRIJ' ],
            [ '543', '545', 'GRIJ' ],
            [ '548', '552', 'GRIJ' ],
            [ '554', '556', 'GRIJ' ],
            [ '558', '558', 'GRIJ' ],
            [ '945', '946', 'GRIJ' ] ] },
        'Olival' => {
          '4415' => [
            [ '414', '414', 'OLIVAL VNG' ],
            [ '725', '727', 'OLIVAL VNG' ],
            [ '729', '732', 'OLIVAL VNG' ] ] },
        'Seixezelo' => {
          '4415' => [
            [ '403', '404', 'SEIXEZELO' ],
            [ '926', '934', 'SEIXEZELO' ],
            [ '936', '944', 'SEIXEZELO' ],
            [ '947', '947', 'SEIXEZELO' ] ] },
        'Moses' => {
          '4415' => [
            [ '720', '722', 'OLIVAL VNG' ] ] },
        'Crestuma' => {
          '4415' => [
            [ '572', '586', 'CRESTUMA' ],
            [ '588', '623', 'CRESTUMA' ] ] },
        'Sermonde' => {
          '4415' => [
            [ '097', '099', 'SERMONDE' ],
            [ '101', '110', 'SERMONDE' ],
            [ '112', '120', 'SERMONDE' ],
            [ '124', '126', 'SERMONDE' ] ] },
        'Lever' => {
          '4415' => [
            [ '402', '402', 'LEVER' ],
            [ '413', '413', 'LEVER' ],
            [ '629', '630', 'LEVER' ],
            [ '632', '644', 'LEVER' ],
            [ '646', '646', 'LEVER' ],
            [ '648', '656', 'LEVER' ],
            [ '659', '677', 'LEVER' ],
            [ '679', '682', 'LEVER' ] ] },
        'Seixo Alvo' => {
          '4415' => [
            [ '735', '741', 'OLIVAL VNG' ],
            [ '744', '752', 'OLIVAL VNG' ] ] },
        'So Flix da Marinha' => {
          '4410' => [
            [ '005', '005', 'SO FELIX DA MARINHA' ],
            [ '095', '221', 'SO FELIX DA MARINHA' ],
            [ '425', '425', 'SO FELIX DA MARINHA' ],
            [ '463', '463', 'SO FELIX DA MARINHA' ] ],
          '4414' => [
            [ '008', '008', 'SO FELIX DA MARINHA' ] ] },
        'Perosinho' => {
          '4415' => [
            [ '001', '044', 'PEROSINHO' ],
            [ '046', '076', 'PEROSINHO' ],
            [ '078', '082', 'PEROSINHO' ],
            [ '084', '086', 'PEROSINHO' ],
            [ '396', '397', 'PEROSINHO' ],
            [ '418', '418', 'PEROSINHO' ] ] },
        'Vila Nova de Gaia' => {
          '4409' => [
            [ '501', '501', 'VILA NOVA DE GAIA' ],
            [ '505', '505', 'VILA NOVA DE GAIA' ],
            [ '512', '515', 'VILA NOVA DE GAIA' ] ],
          '4400' => [
            [ '001', '019', 'VILA NOVA DE GAIA' ],
            [ '021', '023', 'VILA NOVA DE GAIA' ],
            [ '025', '025', 'VILA NOVA DE GAIA' ],
            [ '027', '046', 'VILA NOVA DE GAIA' ],
            [ '048', '103', 'VILA NOVA DE GAIA' ],
            [ '105', '108', 'VILA NOVA DE GAIA' ],
            [ '110', '118', 'VILA NOVA DE GAIA' ],
            [ '120', '143', 'VILA NOVA DE GAIA' ],
            [ '146', '147', 'VILA NOVA DE GAIA' ],
            [ '149', '168', 'VILA NOVA DE GAIA' ],
            [ '170', '188', 'VILA NOVA DE GAIA' ],
            [ '190', '203', 'VILA NOVA DE GAIA' ],
            [ '205', '245', 'VILA NOVA DE GAIA' ],
            [ '247', '319', 'VILA NOVA DE GAIA' ],
            [ '321', '447', 'VILA NOVA DE GAIA' ],
            [ '449', '474', 'VILA NOVA DE GAIA' ],
            [ '476', '536', 'VILA NOVA DE GAIA' ],
            [ '538', '560', 'VILA NOVA DE GAIA' ],
            [ '562', '601', 'VILA NOVA DE GAIA' ],
            [ '603', '616', 'VILA NOVA DE GAIA' ],
            [ '620', '687', 'VILA NOVA DE GAIA' ] ],
          '4434' => [
            [ '501', '502', 'VILA NOVA DE GAIA' ],
            [ '508', '509', 'VILA NOVA DE GAIA' ] ],
          '4404' => [
            [ '501', '504', 'VILA NOVA DE GAIA' ],
            [ '506', '510', 'VILA NOVA DE GAIA' ] ],
          '4405' => [
            [ '520', '523', 'VILA NOVA DE GAIA' ],
            [ '525', '550', 'VILA NOVA DE GAIA' ],
            [ '552', '566', 'VILA NOVA DE GAIA' ],
            [ '568', '575', 'VILA NOVA DE GAIA' ],
            [ '577', '617', 'VILA NOVA DE GAIA' ],
            [ '619', '650', 'VILA NOVA DE GAIA' ],
            [ '652', '668', 'VILA NOVA DE GAIA' ],
            [ '671', '689', 'VILA NOVA DE GAIA' ],
            [ '691', '694', 'VILA NOVA DE GAIA' ],
            [ '696', '697', 'VILA NOVA DE GAIA' ],
            [ '699', '703', 'VILA NOVA DE GAIA' ],
            [ '705', '708', 'VILA NOVA DE GAIA' ],
            [ '712', '737', 'VILA NOVA DE GAIA' ],
            [ '739', '787', 'VILA NOVA DE GAIA' ],
            [ '789', '796', 'VILA NOVA DE GAIA' ],
            [ '798', '801', 'VILA NOVA DE GAIA' ],
            [ '804', '825', 'VILA NOVA DE GAIA' ],
            [ '827', '835', 'VILA NOVA DE GAIA' ],
            [ '837', '883', 'VILA NOVA DE GAIA' ],
            [ '885', '908', 'VILA NOVA DE GAIA' ],
            [ '910', '918', 'VILA NOVA DE GAIA' ],
            [ '931', '933', 'VILA NOVA DE GAIA' ] ],
          '4430' => [
            [ '001', '022', 'VILA NOVA DE GAIA' ],
            [ '024', '031', 'VILA NOVA DE GAIA' ],
            [ '033', '039', 'VILA NOVA DE GAIA' ],
            [ '041', '044', 'VILA NOVA DE GAIA' ],
            [ '046', '048', 'VILA NOVA DE GAIA' ],
            [ '050', '066', 'VILA NOVA DE GAIA' ],
            [ '068', '190', 'VILA NOVA DE GAIA' ],
            [ '192', '201', 'VILA NOVA DE GAIA' ],
            [ '203', '249', 'VILA NOVA DE GAIA' ],
            [ '251', '259', 'VILA NOVA DE GAIA' ],
            [ '265', '273', 'VILA NOVA DE GAIA' ],
            [ '275', '279', 'VILA NOVA DE GAIA' ],
            [ '281', '284', 'VILA NOVA DE GAIA' ],
            [ '286', '293', 'VILA NOVA DE GAIA' ],
            [ '295', '400', 'VILA NOVA DE GAIA' ],
            [ '402', '403', 'VILA NOVA DE GAIA' ],
            [ '405', '410', 'VILA NOVA DE GAIA' ],
            [ '412', '414', 'VILA NOVA DE GAIA' ],
            [ '416', '508', 'VILA NOVA DE GAIA' ],
            [ '510', '536', 'VILA NOVA DE GAIA' ],
            [ '538', '634', 'VILA NOVA DE GAIA' ],
            [ '641', '647', 'VILA NOVA DE GAIA' ],
            [ '682', '686', 'VILA NOVA DE GAIA' ],
            [ '688', '695', 'VILA NOVA DE GAIA' ],
            [ '745', '754', 'VILA NOVA DE GAIA' ],
            [ '933', '940', 'VILA NOVA DE GAIA' ] ] },
        'Pedroso' => {
          '4415' => [
            [ '088', '094', 'PEDROSO' ],
            [ '128', '146', 'PEDROSO' ],
            [ '148', '246', 'PEDROSO' ],
            [ '248', '372', 'PEDROSO' ],
            [ '374', '390', 'PEDROSO' ],
            [ '393', '393', 'PEDROSO' ],
            [ '407', '410', 'PEDROSO' ],
            [ '415', '415', 'PEDROSO' ],
            [ '419', '420', 'PEDROSO' ] ] },
        'Serzedo' => {
          '4410' => [
            [ '002', '004', 'SERZEDO VNG' ],
            [ '013', '094', 'SERZEDO VNG' ] ],
          '4414' => [
            [ '002', '002', 'SERZEDO VNG' ],
            [ '004', '004', 'SERZEDO VNG' ],
            [ '009', '009', 'SERZEDO VNG' ] ] },
        'Arcozelo' => {
          '4410' => [
            [ '001', '001', 'ARCOZELO VNG' ],
            [ '008', '012', 'ARCOZELO VNG' ],
            [ '315', '424', 'ARCOZELO VNG' ],
            [ '426', '462', 'ARCOZELO VNG' ],
            [ '464', '476', 'ARCOZELO VNG' ],
            [ '481', '486', 'ARCOZELO VNG' ] ] },
        'Canelas' => {
          '4410' => [
            [ '006', '007', 'CANELAS VNG' ],
            [ '222', '314', 'CANELAS VNG' ],
            [ '480', '480', 'CANELAS VNG' ],
            [ '487', '488', 'CANELAS VNG' ] ],
          '4414' => [
            [ '001', '001', 'CANELAS VNG' ],
            [ '003', '003', 'CANELAS VNG' ],
            [ '005', '007', 'CANELAS VNG' ] ] },
        'Sandim' => {
          '4415' => [
            [ '405', '405', 'SANDIM' ],
            [ '411', '412', 'SANDIM' ],
            [ '778', '829', 'SANDIM' ],
            [ '831', '917', 'SANDIM' ] ] }
                },
      'Amarante' => {
        'Pao' => {
          '4605' => [
            [ '463', '494', 'TRAVANCA AMT' ] ] },
        'Cortes' => {
          '4600' => [
            [ '710', '710', 'REBORDELO AMT' ] ] },
        'Escola' => {
          '4600' => [
            [ '530', '530', 'BUSTELO AMT' ] ] },
        'Castelo' => {
          '4615' => [
            [ '163', '167', 'FIGUEIR (SANTIAGO)' ] ] },
        'Fontainhas' => {
          '4600' => [
            [ '730', '730', 'SANCHE' ] ] },
        'Guardinhas' => {
          '4615' => [
            [ '011', '014', 'FREIXO DE CIMA' ] ] },
        'Cabana' => {
          '4600' => [
            [ '641', '644', 'GONDAR AMT' ] ] },
        'Igreja' => {
          '4600' => [
            [ '560', '560', 'CARNEIRO' ] ] },
        'Correios' => {
          '4600' => [
            [ '551', '552', 'CANDEMIL AMT' ] ] },
        'Lagarteira' => {
          '4615' => [
            [ '034', '034', 'FREIXO DE CIMA' ] ] },
        'Seixo' => {
          '4605' => [
            [ '024', '025', 'VILA ME' ] ] },
        'Cancela' => {
          '4615' => [
            [ '087', '088', 'FIGUEIR (SANTA CRISTINA)' ] ] },
        'Eira' => {
          '4600' => [
            [ '500', '500', 'ABOADELA' ] ] },
        'Cruzeiro' => {
          '4615' => [
            [ '220', '220', 'TELES AMT' ] ] },
        'Cabo' => {
          '4615' => [
            [ '085', '085', 'FIGUEIR (SANTA CRISTINA)' ] ] },
        'Outeiros' => {
          '4605' => [
            [ '343', '354', 'VILA ME' ] ] },
        'Vila Mea' => {
          '4605' => [
            [ '032', '032', 'VILA ME' ],
            [ '034', '034', 'VILA ME' ],
            [ '375', '375', 'VILA ME' ],
            [ '377', '378', 'VILA ME' ],
            [ '381', '385', 'VILA ME' ] ] },
        'Friande' => {
          '4600' => [
            [ '741', '742', 'SO SIMO GOUVEIA' ] ] },
        'Alm' => {
          '4605' => [
            [ '201', '213', 'VILA ME' ] ] },
        'Aldeia Nova' => {
          '4615' => [
            [ '081', '083', 'FIGUEIR (SANTA CRISTINA)' ] ],
          '4600' => [
            [ '781', '793', 'VILA CAIZ' ] ] },
        'Portas' => {
          '4600' => [
            [ '620', '620', 'FRIDO' ] ] },
        'Salvador' => {
          '4605' => [
            [ '361', '373', 'VILA ME' ] ] },
        'Ribeiro' => {
          '4605' => [
            [ '226', '231', 'VILA ME' ] ] },
        'Barria' => {
          '4605' => [
            [ '007', '014', 'VILA ME' ] ] },
        'Poos' => {
          '4615' => [
            [ '169', '196', 'FIGUEIR (SANTIAGO)' ] ] },
        'Lamas' => {
          '4605' => [
            [ '409', '414', 'TRAVANCA AMT' ] ] },
        'Carvalho' => {
          '4600' => [
            [ '591', '597', 'FREGIM' ] ] },
        'Agras' => {
          '4605' => [
            [ '101', '108', 'MANCELOS' ] ] },
        'Barreiros' => {
          '4605' => [
            [ '416', '461', 'TRAVANCA AMT' ] ] },
        'Rossinho' => {
          '4605' => [
            [ '356', '359', 'VILA ME' ] ] },
        'Meios' => {
          '4615' => [
            [ '040', '051', 'FREIXO DE CIMA' ] ] },
        'Carvalhas' => {
          '4600' => [
            [ '540', '540', 'CANADELO' ] ] },
        'Palmazes' => {
          '4600' => [
            [ '701', '704', 'PADRONELO' ] ] },
        'Parada' => {
          '4605' => [
            [ '218', '224', 'VILA ME' ] ] },
        'Pardieiros' => {
          '4600' => [
            [ '651', '653', 'JAZENTE' ] ] },
        'Sempre Verde' => {
          '4600' => [
            [ '690', '690', 'OLO' ] ] },
        'Aguilho' => {
          '4600' => [
            [ '580', '580', 'CHAPA' ] ] },
        'Terra Negra' => {
          '4600' => [
            [ '720', '720', 'SALVADOR DO MONTE' ] ] },
        'Lama' => {
          '4600' => [
            [ '661', '662', 'LOMBA AMT' ] ] },
        'Soleiro' => {
          '4600' => [
            [ '520', '520', 'ANSIES' ] ] },
        'Boucinhas' => {
          '4605' => [
            [ '110', '176', 'MANCELOS' ] ] },
        'Pereira' => {
          '4615' => [
            [ '198', '198', 'FIGUEIR (SANTIAGO)' ] ] },
        'Tapada' => {
          '4615' => [
            [ '129', '140', 'FIGUEIR (SANTA CRISTINA)' ] ] },
        'Alpes' => {
          '4600' => [
            [ '670', '670', 'LOUREDO AMT' ] ] },
        'Mato' => {
          '4605' => [
            [ '018', '020', 'VILA ME' ] ] },
        'Oliveirinha' => {
          '4605' => [
            [ '215', '216', 'VILA ME' ] ] },
        'Ches' => {
          '4600' => [
            [ '570', '570', 'CARVALHO DE REI' ] ] },
        'Lameirinhas' => {
          '4615' => [
            [ '108', '127', 'FIGUEIR (SANTA CRISTINA)' ] ] },
        'Boavista' => {
          '4615' => [
            [ '016', '030', 'FREIXO DE CIMA' ] ] },
        'Agre Maior' => {
          '4605' => [
            [ '301', '341', 'VILA ME' ] ] },
        'Teles' => {
          '4600' => [
            [ '751', '762', 'TELES AMT' ] ] },
        'Outeiro' => {
          '4600' => [
            [ '801', '802', 'VILA CH DO MARO' ] ] },
        'Cruz das Bouas' => {
          '4600' => [
            [ '810', '810', 'VILA GARCIA AMT' ] ] },
        'Lamaceiro' => {
          '4615' => [
            [ '036', '038', 'FREIXO DE CIMA' ] ] },
        'Fragas' => {
          '4600' => [
            [ '770', '770', 'VRZEA AMT' ] ] },
        'Chamusca' => {
          '4615' => [
            [ '090', '092', 'FIGUEIR (SANTA CRISTINA)' ] ] },
        'Amarante' => {
          '4600' => [
            [ '001', '001', 'AMARANTE' ],
            [ '011', '025', 'AMARANTE' ],
            [ '027', '067', 'AMARANTE' ],
            [ '069', '117', 'AMARANTE' ],
            [ '201', '226', 'AMARANTE' ],
            [ '228', '258', 'AMARANTE' ],
            [ '260', '262', 'AMARANTE' ],
            [ '264', '284', 'AMARANTE' ],
            [ '286', '305', 'AMARANTE' ] ] },
        'Costa da Lage' => {
          '4615' => [
            [ '094', '104', 'FIGUEIR (SANTA CRISTINA)' ] ] },
        'Vrzea' => {
          '4615' => [
            [ '053', '053', 'FREIXO DE CIMA' ] ] },
        'Pascoais Alm' => {
          '4600' => [
            [ '631', '632', 'GATO' ] ] },
        'Presa Velha' => {
          '4605' => [
            [ '022', '022', 'VILA ME' ] ] },
        'So Sebastio' => {
          '4600' => [
            [ '611', '612', 'FREIXO DE BAIXO' ] ] },
        'Outeiro do Moinho' => {
          '4600' => [
            [ '510', '510', 'ABOIM AMT' ] ] },
        'Portela' => {
          '4615' => [
            [ '200', '209', 'FIGUEIR (SANTIAGO)' ] ],
          '4600' => [
            [ '681', '683', 'LUFREI' ] ] }
             },
      'Gondomar' => {
        'Baguim do Monte' => {
          '4435' => [
            [ '593', '596', 'BAGUIM DO MONTE' ],
            [ '601', '615', 'BAGUIM DO MONTE' ],
            [ '617', '617', 'BAGUIM DO MONTE' ],
            [ '619', '622', 'BAGUIM DO MONTE' ],
            [ '624', '652', 'BAGUIM DO MONTE' ],
            [ '654', '692', 'BAGUIM DO MONTE' ],
            [ '694', '727', 'BAGUIM DO MONTE' ],
            [ '729', '744', 'BAGUIM DO MONTE' ],
            [ '746', '756', 'BAGUIM DO MONTE' ],
            [ '758', '771', 'BAGUIM DO MONTE' ],
            [ '773', '779', 'BAGUIM DO MONTE' ],
            [ '781', '798', 'BAGUIM DO MONTE' ],
            [ '801', '814', 'BAGUIM DO MONTE' ],
            [ '816', '839', 'BAGUIM DO MONTE' ] ] },
        'Rio Tinto' => {
          '4435' => [
            [ '001', '029', 'RIO TINTO' ],
            [ '033', '062', 'RIO TINTO' ],
            [ '064', '070', 'RIO TINTO' ],
            [ '072', '102', 'RIO TINTO' ],
            [ '104', '153', 'RIO TINTO' ],
            [ '155', '162', 'RIO TINTO' ],
            [ '164', '235', 'RIO TINTO' ],
            [ '239', '475', 'RIO TINTO' ],
            [ '479', '508', 'RIO TINTO' ],
            [ '597', '600', 'RIO TINTO' ],
            [ '780', '780', 'RIO TINTO' ],
            [ '840', '840', 'RIO TINTO' ] ] },
        'Jovim' => {
          '4510' => [
            [ '001', '130', 'JOVIM' ],
            [ '132', '163', 'JOVIM' ],
            [ '801', '801', 'JOVIM' ],
            [ '807', '809', 'JOVIM' ] ] },
        'Foz Sousa' => {
          '4515' => [
            [ '084', '084', 'FOZ DO SOUSA' ],
            [ '091', '093', 'FOZ DO SOUSA' ],
            [ '095', '107', 'FOZ DO SOUSA' ],
            [ '109', '119', 'FOZ DO SOUSA' ],
            [ '121', '125', 'FOZ DO SOUSA' ],
            [ '127', '127', 'FOZ DO SOUSA' ],
            [ '129', '129', 'FOZ DO SOUSA' ],
            [ '131', '135', 'FOZ DO SOUSA' ],
            [ '138', '138', 'FOZ DO SOUSA' ],
            [ '140', '145', 'FOZ DO SOUSA' ],
            [ '147', '147', 'FOZ DO SOUSA' ],
            [ '151', '158', 'FOZ DO SOUSA' ],
            [ '161', '166', 'FOZ DO SOUSA' ],
            [ '169', '173', 'FOZ DO SOUSA' ],
            [ '175', '176', 'FOZ DO SOUSA' ],
            [ '180', '182', 'FOZ DO SOUSA' ],
            [ '184', '188', 'FOZ DO SOUSA' ],
            [ '190', '197', 'FOZ DO SOUSA' ],
            [ '199', '201', 'FOZ DO SOUSA' ],
            [ '203', '204', 'FOZ DO SOUSA' ],
            [ '206', '208', 'FOZ DO SOUSA' ],
            [ '210', '210', 'FOZ DO SOUSA' ],
            [ '212', '215', 'FOZ DO SOUSA' ],
            [ '217', '217', 'FOZ DO SOUSA' ],
            [ '219', '221', 'FOZ DO SOUSA' ],
            [ '223', '224', 'FOZ DO SOUSA' ],
            [ '227', '227', 'FOZ DO SOUSA' ],
            [ '229', '229', 'FOZ DO SOUSA' ],
            [ '644', '678', 'FOZ DO SOUSA' ],
            [ '681', '693', 'FOZ DO SOUSA' ],
            [ '695', '696', 'FOZ DO SOUSA' ] ] },
        'Medas' => {
          '4515' => [
            [ '344', '347', 'MEDAS' ],
            [ '349', '351', 'MEDAS' ],
            [ '353', '384', 'MEDAS' ],
            [ '386', '404', 'MEDAS' ],
            [ '406', '412', 'MEDAS' ],
            [ '414', '427', 'MEDAS' ],
            [ '429', '431', 'MEDAS' ],
            [ '433', '460', 'MEDAS' ],
            [ '643', '643', 'MEDAS' ],
            [ '704', '704', 'MEDAS' ] ] },
        'Fanzeres' => {
          '4510' => [
            [ '466', '698', 'FNZERES' ],
            [ '802', '806', 'FNZERES' ] ] },
        'So Pedro da Cova' => {
          '4510' => [
            [ '164', '465', 'SO PEDRO DA COVA' ],
            [ '699', '700', 'SO PEDRO DA COVA' ],
            [ '751', '786', 'SO PEDRO DA COVA' ],
            [ '788', '800', 'SO PEDRO DA COVA' ] ] },
        'Gondomar' => {
          '4420' => [
            [ '002', '007', 'GONDOMAR' ],
            [ '009', '020', 'GONDOMAR' ],
            [ '022', '110', 'GONDOMAR' ],
            [ '112', '150', 'GONDOMAR' ],
            [ '152', '177', 'GONDOMAR' ],
            [ '179', '259', 'GONDOMAR' ],
            [ '261', '305', 'GONDOMAR' ],
            [ '307', '359', 'GONDOMAR' ],
            [ '582', '584', 'GONDOMAR' ],
            [ '590', '611', 'GONDOMAR' ] ] },
        'Melres' => {
          '4515' => [
            [ '462', '492', 'MELRES' ],
            [ '494', '642', 'MELRES' ],
            [ '705', '705', 'MELRES' ] ] },
        'Lomba' => {
          '4515' => [
            [ '232', '278', 'LOMBA GDM' ],
            [ '280', '283', 'LOMBA GDM' ],
            [ '286', '343', 'LOMBA GDM' ],
            [ '697', '703', 'LOMBA GDM' ] ] },
        'Valbom' => {
          '4420' => [
            [ '360', '371', 'VALBOM GDM' ],
            [ '373', '404', 'VALBOM GDM' ],
            [ '406', '410', 'VALBOM GDM' ],
            [ '412', '497', 'VALBOM GDM' ],
            [ '499', '581', 'VALBOM GDM' ],
            [ '585', '589', 'VALBOM GDM' ] ] },
        'Covelo' => {
          '4515' => [
            [ '001', '083', 'COVELO GDM' ],
            [ '679', '680', 'COVELO GDM' ] ] }
             },
      'Paos de Ferreira' => {
        'Fonte Velha' => {
          '4590' => [
            [ '273', '286', 'FIGUEIR' ] ] },
        'Alto de Matos' => {
          '4595' => [
            [ '391', '412', 'SEROA' ] ] },
        'Vista Alegre' => {
          '4590' => [
            [ '408', '409', 'FREAMUNDE' ] ] },
        'Mirelo' => {
          '4595' => [
            [ '328', '328', 'PENAMAIOR' ] ] },
        'Palhes' => {
          '4595' => [
            [ '344', '344', 'PENAMAIOR' ] ] },
        'Alto do Pao' => {
          '4595' => [
            [ '213', '225', 'MEIXOMIL' ] ] },
        'Cruz' => {
          '4595' => [
            [ '118', '118', 'FRAZO' ] ] },
        'Pegas' => {
          '4590' => [
            [ '427', '428', 'LAMOSO' ] ] },
        'Igreja' => {
          '4590' => [
            [ '381', '382', 'FREAMUNDE' ] ] },
        'Rossio' => {
          '4595' => [
            [ '181', '182', 'FRAZO' ] ] },
        'Moinhos' => {
          '4595' => [
            [ '151', '169', 'FRAZO' ] ] },
        'Boua' => {
          '4590' => [
            [ '298', '301', 'FREAMUNDE' ] ],
          '4595' => [
            [ '281', '303', 'PENAMAIOR' ] ] },
        'Fontanhas' => {
          '4595' => [
            [ '125', '145', 'FRAZO' ] ] },
        'Fonteirigo' => {
          '4590' => [
            [ '423', '423', 'LAMOSO' ] ] },
        'Cabo' => {
          '4595' => [
            [ '105', '106', 'FRAZO' ] ] },
        'Santa Cruz' => {
          '4590' => [
            [ '403', '406', 'FREAMUNDE' ] ] },
        'Bairros' => {
          '4590' => [
            [ '410', '410', 'LAMOSO' ] ] },
        'Tapadinha' => {
          '4595' => [
            [ '355', '355', 'PENAMAIOR' ] ] },
        'Alm' => {
          '4590' => [
            [ '290', '290', 'FREAMUNDE' ] ],
          '4595' => [
            [ '072', '086', 'EIRIZ' ] ] },
        'Souto' => {
          '4590' => [
            [ '432', '432', 'LAMOSO' ] ] },
        'Aldeia Nova' => {
          '4590' => [
            [ '001', '107', 'CARVALHOSA' ],
            [ '138', '164', 'CODESSOS' ],
            [ '190', '271', 'FIGUEIR' ] ] },
        'Vila Formosa' => {
          '4595' => [
            [ '190', '191', 'FRAZO' ] ] },
        'Agra' => {
          '4590' => [
            [ '631', '705', 'RAIMONDA' ] ],
          '4595' => [
            [ '087', '093', 'FRAZO' ] ] },
        'Mades' => {
          '4590' => [
            [ '388', '399', 'FREAMUNDE' ] ] },
        'Carvalho' => {
          '4595' => [
            [ '108', '108', 'FRAZO' ] ] },
        'Devesa' => {
          '4595' => [
            [ '121', '122', 'FRAZO' ] ] },
        'Cavadinhas' => {
          '4595' => [
            [ '114', '116', 'FRAZO' ] ] },
        'Barreiros' => {
          '4595' => [
            [ '098', '099', 'FRAZO' ] ] },
        'Carvalhosa' => {
          '4590' => [
            [ '109', '137', 'CARVALHOSA' ] ] },
        'Repiade' => {
          '4595' => [
            [ '176', '176', 'FRAZO' ] ] },
        'Repezes' => {
          '4595' => [
            [ '347', '350', 'PENAMAIOR' ] ] },
        'Modelos' => {
          '4595' => [
            [ '336', '338', 'PENAMAIOR' ] ] },
        'Alm do Regato' => {
          '4595' => [
            [ '210', '210', 'MEIXOMIL' ] ] },
        'Alm do Rio' => {
          '4590' => [
            [ '293', '294', 'FREAMUNDE' ] ],
          '4595' => [
            [ '001', '071', 'ARREIGADA' ] ] },
        'Raimonda' => {
          '4590' => [
            [ '707', '749', 'RAIMONDA' ] ] },
        'Inveja' => {
          '4595' => [
            [ '324', '325', 'PENAMAIOR' ] ] },
        'Cavadas' => {
          '4595' => [
            [ '110', '110', 'FRAZO' ] ] },
        'Sobro Pequeno' => {
          '4595' => [
            [ '184', '187', 'FRAZO' ] ] },
        'Boavista' => {
          '4595' => [
            [ '102', '102', 'FRAZO' ] ] },
        'Carvalhal' => {
          '4590' => [
            [ '303', '379', 'FREAMUNDE' ] ] },
        'Abrute' => {
          '4590' => [
            [ '288', '288', 'FREAMUNDE' ] ] },
        'Vila Nova' => {
          '4590' => [
            [ '434', '434', 'LAMOSO' ] ] },
        'Aldeia' => {
          '4590' => [
            [ '437', '503', 'MODELOS' ] ] },
        'So Simo' => {
          '4595' => [
            [ '414', '512', 'SEROA' ] ] },
        'Leigal' => {
          '4590' => [
            [ '384', '384', 'FREAMUNDE' ] ] },
        'Feira' => {
          '4595' => [
            [ '314', '318', 'PENAMAIOR' ] ] },
        'Mirantes' => {
          '4595' => [
            [ '148', '148', 'FRAZO' ] ] },
        'Ferreira' => {
          '4590' => [
            [ '750', '868', 'FERREIRA PFR' ] ] },
        'Barreiras' => {
          '4590' => [
            [ '415', '419', 'LAMOSO' ] ] },
          '3 Horas' => {
          '4595' => [
            [ '197', '197', 'MEIXOMIL' ] ] },
        'Pinheirinhos' => {
          '4595' => [
            [ '171', '174', 'FRAZO' ] ] },
        'Agrelo' => {
          '4595' => [
            [ '361', '390', 'SANFINS DE FERREIRA' ] ] },
        'Paos de Ferreira' => {
          '4590' => [
            [ '165', '168', 'PAOS DE FERREIRA' ],
            [ '170', '175', 'PAOS DE FERREIRA' ],
            [ '177', '182', 'PAOS DE FERREIRA' ],
            [ '504', '509', 'PAOS DE FERREIRA' ],
            [ '511', '522', 'PAOS DE FERREIRA' ],
            [ '524', '565', 'PAOS DE FERREIRA' ],
            [ '567', '567', 'PAOS DE FERREIRA' ],
            [ '569', '630', 'PAOS DE FERREIRA' ] ] },
        'Vale de Sus' => {
          '4595' => [
            [ '358', '360', 'PENAMAIOR' ] ] },
        'Ponte' => {
          '4590' => [
            [ '401', '401', 'FREAMUNDE' ] ] },
        'Meixomil' => {
          '4595' => [
            [ '199', '201', 'MEIXOMIL' ],
            [ '234', '280', 'MEIXOMIL' ] ] }
                },
      'Felgueiras' => {
        'Ameixoeira' => {
          '4610' => [
            [ '561', '592', 'PINHEIRO FLG' ] ] },
        'Alegrete' => {
          '4650' => [
            [ '501', '505', 'SOUSA' ] ] },
        'Figueiredo' => {
          '4610' => [
            [ '100', '219', 'FELGUEIRAS' ] ] },
        'Lordelo' => {
          '4615' => [
            [ '529', '534', 'LIXA' ] ] },
        'Igreja' => {
          '4615' => [
            [ '523', '525', 'LIXA' ] ] },
        'Amarantinha' => {
          '4615' => [
            [ '401', '409', 'MACIEIRA DA LIXA' ] ] },
        'Pinhal Basto' => {
          '4615' => [
            [ '380', '380', 'CARAMOS' ] ] },
        'Crasto' => {
          '4610' => [
            [ '017', '030', 'CARAMOS' ] ] },
        'Padroncelos' => {
          '4615' => [
            [ '536', '551', 'LIXA' ] ] },
        'Alto da Lixa' => {
          '4615' => [
            [ '301', '310', 'BORBA DE GODIM' ] ] },
        'Posmil' => {
          '4610' => [
            [ '300', '333', 'FRIANDE FLG' ] ] },
        'Seivada' => {
          '4650' => [
            [ '827', '834', 'VILA VERDE FLG' ] ] },
        'Padraos' => {
          '4615' => [
            [ '705', '705', 'LIXA' ] ] },
        'Barreira' => {
          '4615' => [
            [ '491', '501', 'LIXA' ] ] },
        'Alto de Unho' => {
          '4650' => [
            [ '651', '671', 'UNHO' ] ] },
        'Alm' => {
          '4610' => [
            [ '611', '655', 'POMBEIRO DE RIBAVIZELA' ] ] },
        'Vinha de Cima' => {
          '4815' => [
            [ '621', '621', 'REGILDE' ] ] },
        'Alto da Cimalha' => {
          '4650' => [
            [ '311', '338', 'RANDE' ] ] },
        'Agra' => {
          '4650' => [
            [ '251', '306', 'PEDREIRA FLG' ],
            [ '361', '377', 'REVINHADE' ],
            [ '451', '468', 'SERNANDE' ],
            [ '801', '811', 'VILA VERDE FLG' ] ],
          '4615' => [
            [ '481', '483', 'LIXA' ] ],
          '4610' => [
            [ '401', '435', 'LAGARES FLG' ],
            [ '671', '716', 'REFONTOURA' ] ] },
        'Trescova' => {
          '4615' => [
            [ '562', '563', 'LIXA' ] ] },
        'Aboim' => {
          '4610' => [
            [ '011', '015', 'CARAMOS' ] ] },
        'Agras' => {
          '4610' => [
            [ '801', '823', 'VRZEA FLG' ] ] },
        'Marco de Simes' => {
          '4615' => [
            [ '435', '435', 'MOURE FLG' ] ] },
        'Outeirinhos' => {
          '4650' => [
            [ '060', '106', 'AIRES' ] ] },
        'Vila Cova de Lixa' => {
          '4615' => [
            [ '352', '352', 'LIXA' ],
            [ '580', '585', 'LIXA' ],
            [ '587', '588', 'LIXA' ],
            [ '592', '593', 'LIXA' ],
            [ '595', '595', 'LIXA' ],
            [ '603', '604', 'LIXA' ],
            [ '606', '606', 'LIXA' ],
            [ '608', '609', 'LIXA' ],
            [ '611', '620', 'LIXA' ],
            [ '625', '627', 'LIXA' ],
            [ '629', '630', 'LIXA' ],
            [ '639', '642', 'LIXA' ],
            [ '646', '659', 'LIXA' ],
            [ '661', '661', 'LIXA' ],
            [ '665', '665', 'LIXA' ],
            [ '667', '667', 'LIXA' ],
            [ '669', '669', 'LIXA' ],
            [ '674', '676', 'LIXA' ],
            [ '678', '680', 'LIXA' ],
            [ '684', '684', 'LIXA' ],
            [ '689', '690', 'LIXA' ],
            [ '693', '701', 'LIXA' ] ] },
        'Gondariz' => {
          '4615' => [
            [ '521', '521', 'LIXA' ] ] },
        'Passarias' => {
          '4615' => [
            [ '419', '421', 'MACIEIRA DA LIXA' ] ] },
        'Fonto' => {
          '4650' => [
            [ '813', '825', 'VILA VERDE FLG' ] ] },
        'Leiras' => {
          '4615' => [
            [ '527', '527', 'LIXA' ] ] },
        'Ribeira' => {
          '4615' => [
            [ '329', '335', 'BORBA DE GODIM' ] ] },
        'Aio' => {
          '4650' => [
            [ '011', '031', 'AIO' ] ] },
        'Bairro de So Miguel' => {
          '4650' => [
            [ '701', '750', 'VARZIELA' ] ] },
        'Maias' => {
          '4615' => [
            [ '703', '703', 'LIXA' ] ] },
        'Cachada' => {
          '4610' => [
            [ '250', '276', 'FELGUEIRAS' ] ] },
        'Eira Vedra' => {
          '4615' => [
            [ '515', '515', 'LIXA' ] ] },
        'Tresborba' => {
          '4615' => [
            [ '337', '343', 'BORBA DE GODIM' ] ] },
        'Borba de Godim' => {
          '4615' => [
            [ '351', '351', 'BORBA DE GODIM' ],
            [ '353', '367', 'BORBA DE GODIM' ],
            [ '589', '591', 'BORBA DE GODIM' ],
            [ '594', '594', 'BORBA DE GODIM' ],
            [ '596', '598', 'BORBA DE GODIM' ],
            [ '601', '602', 'BORBA DE GODIM' ],
            [ '621', '624', 'BORBA DE GODIM' ],
            [ '631', '632', 'BORBA DE GODIM' ],
            [ '634', '638', 'BORBA DE GODIM' ],
            [ '660', '660', 'BORBA DE GODIM' ],
            [ '670', '672', 'BORBA DE GODIM' ],
            [ '677', '677', 'BORBA DE GODIM' ],
            [ '685', '688', 'BORBA DE GODIM' ],
            [ '691', '692', 'BORBA DE GODIM' ] ] },
        'Lameirinho' => {
          '4615' => [
            [ '412', '417', 'MACIEIRA DA LIXA' ] ] },
        'Vila Nova' => {
          '4615' => [
            [ '430', '431', 'MACIEIRA DA LIXA' ] ] },
        'Estrada Nacional' => {
          '4615' => [
            [ '751', '758', 'VILA VERDE FLG' ] ] },
        'Casa da Vila' => {
          '4615' => [
            [ '556', '559', 'LIXA' ] ] },
        'Casarias' => {
          '4615' => [
            [ '507', '513', 'LIXA' ] ] },
        'Aldeia' => {
          '4610' => [
            [ '501', '546', 'PENACOVA FLG' ] ] },
        'Borba' => {
          '4650' => [
            [ '507', '532', 'SOUSA' ] ] },
        'Penegude' => {
          '4615' => [
            [ '440', '440', 'REFONTOURA' ] ] },
        'Ides' => {
          '4650' => [
            [ '120', '187', 'IDES' ] ] },
        'Cto' => {
          '4615' => [
            [ '553', '553', 'LIXA' ] ] },
        'Escalheira' => {
          '4615' => [
            [ '517', '519', 'LIXA' ] ] },
        'Amaras' => {
          '4610' => [
            [ '351', '375', 'JUGUEIROS' ] ] },
        'Cerdeira das Ervas' => {
          '4615' => [
            [ '312', '327', 'BORBA DE GODIM' ] ] },
        'Pinheiro Manso' => {
          '4615' => [
            [ '199', '199', 'MACIEIRA DA LIXA' ] ] },
        'Carvalhinhos' => {
          '4615' => [
            [ '451', '468', 'SANTO' ],
            [ '503', '504', 'LIXA' ] ] },
        'Agrelo' => {
          '4610' => [
            [ '731', '772', 'SENDIM FLG' ] ] },
        'Teixeira' => {
          '4615' => [
            [ '485', '485', 'LIXA' ] ] },
        'Terra Seca' => {
          '4650' => [
            [ '570', '614', 'TORRADOS' ] ] },
        'Barreiro' => {
          '4650' => [
            [ '212', '227', 'LORDELO FLG' ] ] },
        'Macieira  da Lixa' => {
          '4615' => [
            [ '586', '586', 'MACIEIRA DA LIXA' ],
            [ '673', '673', 'MACIEIRA DA LIXA' ] ] },
        'Assento' => {
          '4650' => [
            [ '401', '411', 'SANTO' ] ],
          '4615' => [
            [ '488', '488', 'LIXA' ] ],
          '4610' => [
            [ '841', '864', 'VILA FRIA FLG' ],
            [ '881', '891', 'VIZELA (SO JORGE)' ] ] },
        'Real' => {
          '4615' => [
            [ '423', '428', 'MACIEIRA DA LIXA' ] ] },
        'Argonda' => {
          '4610' => [
            [ '451', '483', 'MOURE FLG' ] ] }
               },
      'Porto' => {
        'Porto' => {
          '4250' => [
            [ '001', '034', 'PORTO' ],
            [ '036', '043', 'PORTO' ],
            [ '045', '066', 'PORTO' ],
            [ '068', '078', 'PORTO' ],
            [ '080', '094', 'PORTO' ],
            [ '096', '096', 'PORTO' ],
            [ '098', '098', 'PORTO' ],
            [ '100', '108', 'PORTO' ],
            [ '110', '118', 'PORTO' ],
            [ '120', '140', 'PORTO' ],
            [ '147', '148', 'PORTO' ],
            [ '150', '173', 'PORTO' ],
            [ '175', '211', 'PORTO' ],
            [ '213', '227', 'PORTO' ],
            [ '229', '229', 'PORTO' ],
            [ '231', '236', 'PORTO' ],
            [ '238', '242', 'PORTO' ],
            [ '244', '244', 'PORTO' ],
            [ '246', '246', 'PORTO' ],
            [ '248', '260', 'PORTO' ],
            [ '262', '316', 'PORTO' ],
            [ '318', '335', 'PORTO' ],
            [ '337', '377', 'PORTO' ],
            [ '379', '385', 'PORTO' ],
            [ '387', '393', 'PORTO' ],
            [ '395', '400', 'PORTO' ],
            [ '403', '415', 'PORTO' ],
            [ '417', '420', 'PORTO' ],
            [ '422', '422', 'PORTO' ],
            [ '424', '446', 'PORTO' ],
            [ '448', '456', 'PORTO' ],
            [ '458', '458', 'PORTO' ],
            [ '460', '483', 'PORTO' ],
            [ '485', '490', 'PORTO' ],
            [ '492', '498', 'PORTO' ],
            [ '500', '507', 'PORTO' ],
            [ '526', '533', 'PORTO' ],
            [ '540', '552', 'PORTO' ] ],
          '4169' => [
            [ '001', '010', 'PORTO' ] ],
          '4149' => [
            [ '001', '017', 'PORTO' ],
            [ '020', '020', 'PORTO' ] ],
          '4369' => [
            [ '001', '002', 'PORTO' ],
            [ '004', '006', 'PORTO' ] ],
          '4050' => [
            [ '008', '018', 'PORTO' ],
            [ '020', '034', 'PORTO' ],
            [ '036', '073', 'PORTO' ],
            [ '075', '116', 'PORTO' ],
            [ '118', '119', 'PORTO' ],
            [ '121', '130', 'PORTO' ],
            [ '132', '155', 'PORTO' ],
            [ '157', '202', 'PORTO' ],
            [ '204', '221', 'PORTO' ],
            [ '223', '225', 'PORTO' ],
            [ '227', '230', 'PORTO' ],
            [ '232', '232', 'PORTO' ],
            [ '234', '234', 'PORTO' ],
            [ '236', '237', 'PORTO' ],
            [ '239', '257', 'PORTO' ],
            [ '259', '298', 'PORTO' ],
            [ '300', '300', 'PORTO' ],
            [ '305', '320', 'PORTO' ],
            [ '322', '330', 'PORTO' ],
            [ '332', '339', 'PORTO' ],
            [ '341', '345', 'PORTO' ],
            [ '347', '348', 'PORTO' ],
            [ '350', '363', 'PORTO' ],
            [ '365', '383', 'PORTO' ],
            [ '385', '394', 'PORTO' ],
            [ '396', '428', 'PORTO' ],
            [ '430', '431', 'PORTO' ],
            [ '433', '456', 'PORTO' ],
            [ '458', '493', 'PORTO' ],
            [ '495', '501', 'PORTO' ],
            [ '503', '504', 'PORTO' ],
            [ '506', '506', 'PORTO' ],
            [ '509', '605', 'PORTO' ],
            [ '607', '640', 'PORTO' ],
            [ '651', '652', 'PORTO' ] ],
          '4199' => [
            [ '001', '005', 'PORTO' ] ],
          '4099' => [
            [ '001', '007', 'PORTO' ],
            [ '009', '010', 'PORTO' ],
            [ '012', '041', 'PORTO' ],
            [ '043', '044', 'PORTO' ],
            [ '062', '062', 'PORTO' ],
            [ '068', '070', 'PORTO' ] ],
          '4350' => [
            [ '004', '014', 'PORTO' ],
            [ '017', '056', 'PORTO' ],
            [ '058', '074', 'PORTO' ],
            [ '076', '076', 'PORTO' ],
            [ '078', '104', 'PORTO' ],
            [ '106', '107', 'PORTO' ],
            [ '111', '114', 'PORTO' ],
            [ '117', '117', 'PORTO' ],
            [ '119', '121', 'PORTO' ],
            [ '123', '124', 'PORTO' ],
            [ '127', '134', 'PORTO' ],
            [ '136', '138', 'PORTO' ],
            [ '140', '141', 'PORTO' ],
            [ '143', '158', 'PORTO' ],
            [ '160', '214', 'PORTO' ],
            [ '216', '231', 'PORTO' ],
            [ '233', '234', 'PORTO' ],
            [ '236', '244', 'PORTO' ],
            [ '246', '267', 'PORTO' ],
            [ '270', '270', 'PORTO' ],
            [ '272', '287', 'PORTO' ],
            [ '289', '294', 'PORTO' ],
            [ '296', '309', 'PORTO' ],
            [ '311', '341', 'PORTO' ],
            [ '343', '343', 'PORTO' ],
            [ '345', '352', 'PORTO' ],
            [ '401', '404', 'PORTO' ] ],
          '4269' => [
            [ '001', '001', 'PORTO' ],
            [ '003', '006', 'PORTO' ] ],
          '4200' => [
            [ '007', '011', 'PORTO' ],
            [ '013', '018', 'PORTO' ],
            [ '020', '033', 'PORTO' ],
            [ '035', '050', 'PORTO' ],
            [ '052', '108', 'PORTO' ],
            [ '110', '116', 'PORTO' ],
            [ '118', '118', 'PORTO' ],
            [ '120', '121', 'PORTO' ],
            [ '123', '134', 'PORTO' ],
            [ '138', '155', 'PORTO' ],
            [ '157', '164', 'PORTO' ],
            [ '170', '171', 'PORTO' ],
            [ '177', '178', 'PORTO' ],
            [ '180', '181', 'PORTO' ],
            [ '185', '186', 'PORTO' ],
            [ '188', '202', 'PORTO' ],
            [ '204', '241', 'PORTO' ],
            [ '243', '259', 'PORTO' ],
            [ '261', '267', 'PORTO' ],
            [ '269', '276', 'PORTO' ],
            [ '278', '289', 'PORTO' ],
            [ '291', '298', 'PORTO' ],
            [ '300', '310', 'PORTO' ],
            [ '312', '323', 'PORTO' ],
            [ '325', '333', 'PORTO' ],
            [ '335', '335', 'PORTO' ],
            [ '337', '387', 'PORTO' ],
            [ '389', '389', 'PORTO' ],
            [ '391', '399', 'PORTO' ],
            [ '401', '406', 'PORTO' ],
            [ '408', '419', 'PORTO' ],
            [ '422', '422', 'PORTO' ],
            [ '427', '427', 'PORTO' ],
            [ '429', '430', 'PORTO' ],
            [ '432', '434', 'PORTO' ],
            [ '437', '456', 'PORTO' ],
            [ '458', '463', 'PORTO' ],
            [ '465', '465', 'PORTO' ],
            [ '469', '470', 'PORTO' ],
            [ '472', '474', 'PORTO' ],
            [ '476', '477', 'PORTO' ],
            [ '479', '491', 'PORTO' ],
            [ '493', '493', 'PORTO' ],
            [ '495', '495', 'PORTO' ],
            [ '497', '526', 'PORTO' ],
            [ '528', '534', 'PORTO' ],
            [ '536', '537', 'PORTO' ],
            [ '548', '551', 'PORTO' ],
            [ '595', '595', 'PORTO' ],
            [ '597', '597', 'PORTO' ],
            [ '601', '604', 'PORTO' ] ],
          '4150' => [
            [ '001', '035', 'PORTO' ],
            [ '038', '039', 'PORTO' ],
            [ '041', '044', 'PORTO' ],
            [ '046', '072', 'PORTO' ],
            [ '074', '084', 'PORTO' ],
            [ '086', '092', 'PORTO' ],
            [ '096', '103', 'PORTO' ],
            [ '105', '122', 'PORTO' ],
            [ '124', '124', 'PORTO' ],
            [ '127', '185', 'PORTO' ],
            [ '187', '188', 'PORTO' ],
            [ '190', '190', 'PORTO' ],
            [ '192', '216', 'PORTO' ],
            [ '219', '232', 'PORTO' ],
            [ '235', '236', 'PORTO' ],
            [ '238', '254', 'PORTO' ],
            [ '260', '260', 'PORTO' ],
            [ '262', '262', 'PORTO' ],
            [ '266', '266', 'PORTO' ],
            [ '269', '282', 'PORTO' ],
            [ '285', '285', 'PORTO' ],
            [ '288', '292', 'PORTO' ],
            [ '294', '297', 'PORTO' ],
            [ '299', '316', 'PORTO' ],
            [ '318', '318', 'PORTO' ],
            [ '320', '335', 'PORTO' ],
            [ '337', '360', 'PORTO' ],
            [ '362', '363', 'PORTO' ],
            [ '365', '369', 'PORTO' ],
            [ '371', '397', 'PORTO' ],
            [ '399', '407', 'PORTO' ],
            [ '409', '412', 'PORTO' ],
            [ '414', '419', 'PORTO' ],
            [ '421', '421', 'PORTO' ],
            [ '425', '453', 'PORTO' ],
            [ '455', '455', 'PORTO' ],
            [ '457', '474', 'PORTO' ],
            [ '476', '477', 'PORTO' ],
            [ '479', '496', 'PORTO' ],
            [ '498', '518', 'PORTO' ],
            [ '520', '521', 'PORTO' ],
            [ '523', '531', 'PORTO' ],
            [ '533', '537', 'PORTO' ],
            [ '540', '547', 'PORTO' ],
            [ '549', '549', 'PORTO' ],
            [ '552', '574', 'PORTO' ],
            [ '579', '581', 'PORTO' ],
            [ '584', '584', 'PORTO' ],
            [ '586', '586', 'PORTO' ],
            [ '589', '589', 'PORTO' ],
            [ '594', '601', 'PORTO' ],
            [ '603', '603', 'PORTO' ],
            [ '605', '605', 'PORTO' ],
            [ '607', '631', 'PORTO' ],
            [ '633', '660', 'PORTO' ],
            [ '662', '699', 'PORTO' ],
            [ '701', '713', 'PORTO' ],
            [ '718', '741', 'PORTO' ],
            [ '743', '749', 'PORTO' ],
            [ '751', '758', 'PORTO' ],
            [ '761', '762', 'PORTO' ],
            [ '785', '792', 'PORTO' ],
            [ '794', '794', 'PORTO' ],
            [ '796', '800', 'PORTO' ] ],
          '4049' => [
            [ '001', '008', 'PORTO' ],
            [ '010', '016', 'PORTO' ],
            [ '019', '052', 'PORTO' ],
            [ '055', '061', 'PORTO' ],
            [ '063', '067', 'PORTO' ],
            [ '069', '070', 'PORTO' ] ],
          '4349' => [
            [ '001', '028', 'PORTO' ],
            [ '030', '030', 'PORTO' ] ],
          '4300' => [
            [ '001', '009', 'PORTO' ],
            [ '011', '013', 'PORTO' ],
            [ '015', '015', 'PORTO' ],
            [ '017', '042', 'PORTO' ],
            [ '045', '075', 'PORTO' ],
            [ '077', '111', 'PORTO' ],
            [ '113', '114', 'PORTO' ],
            [ '117', '119', 'PORTO' ],
            [ '121', '124', 'PORTO' ],
            [ '126', '128', 'PORTO' ],
            [ '137', '137', 'PORTO' ],
            [ '139', '139', 'PORTO' ],
            [ '141', '147', 'PORTO' ],
            [ '149', '153', 'PORTO' ],
            [ '155', '174', 'PORTO' ],
            [ '176', '212', 'PORTO' ],
            [ '214', '223', 'PORTO' ],
            [ '225', '226', 'PORTO' ],
            [ '228', '229', 'PORTO' ],
            [ '231', '231', 'PORTO' ],
            [ '233', '234', 'PORTO' ],
            [ '236', '263', 'PORTO' ],
            [ '265', '281', 'PORTO' ],
            [ '283', '286', 'PORTO' ],
            [ '289', '289', 'PORTO' ],
            [ '293', '314', 'PORTO' ],
            [ '316', '316', 'PORTO' ],
            [ '318', '340', 'PORTO' ],
            [ '342', '354', 'PORTO' ],
            [ '357', '357', 'PORTO' ],
            [ '359', '378', 'PORTO' ],
            [ '380', '383', 'PORTO' ],
            [ '385', '385', 'PORTO' ],
            [ '389', '407', 'PORTO' ],
            [ '409', '410', 'PORTO' ],
            [ '413', '416', 'PORTO' ],
            [ '419', '425', 'PORTO' ],
            [ '427', '428', 'PORTO' ],
            [ '430', '451', 'PORTO' ],
            [ '453', '478', 'PORTO' ],
            [ '480', '499', 'PORTO' ],
            [ '502', '506', 'PORTO' ],
            [ '526', '541', 'PORTO' ] ],
          '4249' => [
            [ '001', '007', 'PORTO' ],
            [ '009', '009', 'PORTO' ],
            [ '012', '012', 'PORTO' ] ],
          '4100' => [
            [ '002', '011', 'PORTO' ],
            [ '014', '020', 'PORTO' ],
            [ '022', '038', 'PORTO' ],
            [ '040', '040', 'PORTO' ],
            [ '042', '047', 'PORTO' ],
            [ '049', '058', 'PORTO' ],
            [ '060', '065', 'PORTO' ],
            [ '067', '068', 'PORTO' ],
            [ '070', '084', 'PORTO' ],
            [ '086', '088', 'PORTO' ],
            [ '090', '093', 'PORTO' ],
            [ '095', '105', 'PORTO' ],
            [ '107', '123', 'PORTO' ],
            [ '125', '132', 'PORTO' ],
            [ '134', '141', 'PORTO' ],
            [ '143', '143', 'PORTO' ],
            [ '145', '148', 'PORTO' ],
            [ '151', '151', 'PORTO' ],
            [ '158', '165', 'PORTO' ],
            [ '167', '176', 'PORTO' ],
            [ '178', '180', 'PORTO' ],
            [ '182', '187', 'PORTO' ],
            [ '189', '189', 'PORTO' ],
            [ '191', '192', 'PORTO' ],
            [ '195', '211', 'PORTO' ],
            [ '214', '226', 'PORTO' ],
            [ '228', '238', 'PORTO' ],
            [ '240', '244', 'PORTO' ],
            [ '246', '248', 'PORTO' ],
            [ '250', '254', 'PORTO' ],
            [ '256', '263', 'PORTO' ],
            [ '265', '267', 'PORTO' ],
            [ '269', '276', 'PORTO' ],
            [ '278', '281', 'PORTO' ],
            [ '283', '293', 'PORTO' ],
            [ '295', '300', 'PORTO' ],
            [ '304', '305', 'PORTO' ],
            [ '307', '313', 'PORTO' ],
            [ '315', '321', 'PORTO' ],
            [ '323', '323', 'PORTO' ],
            [ '325', '327', 'PORTO' ],
            [ '329', '347', 'PORTO' ],
            [ '349', '349', 'PORTO' ],
            [ '351', '360', 'PORTO' ],
            [ '362', '366', 'PORTO' ],
            [ '368', '383', 'PORTO' ],
            [ '385', '398', 'PORTO' ],
            [ '400', '423', 'PORTO' ],
            [ '425', '431', 'PORTO' ],
            [ '434', '462', 'PORTO' ],
            [ '464', '466', 'PORTO' ],
            [ '468', '484', 'PORTO' ],
            [ '486', '491', 'PORTO' ],
            [ '493', '498', 'PORTO' ],
            [ '500', '504', 'PORTO' ],
            [ '506', '506', 'PORTO' ],
            [ '508', '517', 'PORTO' ],
            [ '519', '520', 'PORTO' ],
            [ '530', '531', 'PORTO' ],
            [ '552', '553', 'PORTO' ] ],
          '4000' => [
            [ '008', '016', 'PORTO' ],
            [ '019', '025', 'PORTO' ],
            [ '027', '030', 'PORTO' ],
            [ '032', '035', 'PORTO' ],
            [ '037', '075', 'PORTO' ],
            [ '077', '099', 'PORTO' ],
            [ '101', '126', 'PORTO' ],
            [ '128', '149', 'PORTO' ],
            [ '151', '160', 'PORTO' ],
            [ '162', '181', 'PORTO' ],
            [ '184', '193', 'PORTO' ],
            [ '195', '260', 'PORTO' ],
            [ '263', '274', 'PORTO' ],
            [ '276', '280', 'PORTO' ],
            [ '284', '286', 'PORTO' ],
            [ '288', '288', 'PORTO' ],
            [ '290', '316', 'PORTO' ],
            [ '318', '388', 'PORTO' ],
            [ '390', '416', 'PORTO' ],
            [ '418', '476', 'PORTO' ],
            [ '478', '478', 'PORTO' ],
            [ '480', '539', 'PORTO' ],
            [ '541', '545', 'PORTO' ],
            [ '799', '799', 'PORTO' ] ] }
          },
      'Lousada' => {
        'Soutinho' => {
          '4620' => [
            [ '796', '820', 'TORNO' ] ] },
        'Engenho' => {
          '4620' => [
            [ '207', '215', 'LODARES' ] ] },
        'guas Levadas' => {
          '4620' => [
            [ '906', '914', 'NESPEREIRA LSD' ] ] },
        'Cezes' => {
          '4620' => [
            [ '762', '787', 'TORNO' ] ] },
        'Linhares' => {
          '4620' => [
            [ '217', '230', 'LODARES' ] ] },
        'Mouro' => {
          '4620' => [
            [ '058', '058', 'CADE DE REI' ] ] },
        'Boua' => {
          '4620' => [
            [ '304', '305', 'MACIEIRA LSD' ] ] },
        'Novelho' => {
          '4620' => [
            [ '615', '616', 'SANTO ESTEVO BARROSAS' ] ] },
        'Monte' => {
          '4620' => [
            [ '432', '446', 'NEVOGILDE LSD' ] ] },
        'Pedroso de Baixo' => {
          '4620' => [
            [ '139', '139', 'MACIEIRA LSD' ] ] },
        'Barroco' => {
          '4620' => [
            [ '232', '233', 'LODARES' ] ] },
        'Alm' => {
          '4620' => [
            [ '830', '849', 'MEINEDO' ] ] },
        'Pltano' => {
          '4620' => [
            [ '064', '064', 'CADE DE REI' ] ] },
        'Lugar Novo' => {
          '4620' => [
            [ '177', '199', 'NOGUEIRA LSD' ] ] },
        'Agros' => {
          '4620' => [
            [ '241', '255', 'LUSTOSA' ] ] },
        'Agras' => {
          '4620' => [
            [ '631', '631', 'LOUSADA' ] ] },
        'Santa Isabel' => {
          '4620' => [
            [ '235', '240', 'LODARES' ] ] },
        'Barreiros' => {
          '4620' => [
            [ '040', '047', 'CADE DE REI' ],
            [ '821', '824', 'VILAR DO TORNO E ALENTM' ] ] },
        'Zona Industrial' => {
          '4620' => [
            [ '072', '074', 'CADE DE REI' ] ] },
        'Redolho' => {
          '4620' => [
            [ '503', '520', 'PIAS LSD' ] ] },
        'Penas' => {
          '4620' => [
            [ '258', '268', 'LUSTOSA' ] ] },
        'Afreita' => {
          '4620' => [
            [ '915', '932', 'NEVOGILDE LSD' ] ] },
        'Morgadinhos' => {
          '4620' => [
            [ '324', '330', 'MACIEIRA LSD' ] ] },
        'Arcas' => {
          '4620' => [
            [ '121', '127', 'LOUSADA' ] ] },
        'Eira Vedra' => {
          '4620' => [
            [ '711', '726', 'SOUSELA' ] ] },
        'Aldeia' => {
          '4620' => [
            [ '301', '302', 'MACIEIRA LSD' ] ] },
        'Pomar' => {
          '4620' => [
            [ '728', '734', 'SOUSELA' ] ] },
        'Outeiro Moinhos' => {
          '4620' => [
            [ '061', '061', 'CADE DE REI' ] ] },
        'Alvocovo' => {
          '4620' => [
            [ '200', '206', 'FIGUEIRAS' ] ] },
        'So Mamede' => {
          '4620' => [
            [ '336', '338', 'MEINEDO' ] ] },
        'Sebarelhos' => {
          '4620' => [
            [ '100', '104', 'CERNADELO' ] ] },
        'Estao' => {
          '4620' => [
            [ '051', '052', 'CADE DE REI' ] ] },
        'Alem do Rio' => {
          '4620' => [
            [ '633', '657', 'LOUSADA' ] ] },
        'Ameixoeira' => {
          '4620' => [
            [ '031', '031', 'BOIM' ] ] },
        'Cabo de Vila' => {
          '4620' => [
            [ '400', '408', 'NESPEREIRA LSD' ] ] },
        'Jogo da Bola' => {
          '4620' => [
            [ '460', '468', 'NOGUEIRA LSD' ] ] },
        'Agra de Moreira' => {
          '4620' => [
            [ '701', '709', 'SOUSELA' ] ] },
        'Igreja' => {
          '4620' => [
            [ '427', '430', 'NEVOGILDE LSD' ] ] },
        'Hortezelo' => {
          '4620' => [
            [ '054', '055', 'CADE DE REI' ] ] },
        'Soutelo' => {
          '4620' => [
            [ '736', '744', 'SOUSELA' ] ] },
        'Santa Cruz' => {
          '4620' => [
            [ '111', '112', 'COVAS LSD' ] ] },
        'Covilh' => {
          '4620' => [
            [ '276', '283', 'LUSTOSA' ] ] },
        'Herdade' => {
          '4620' => [
            [ '010', '010', 'ALVARENGA LSD' ] ] },
        'Casas Novas' => {
          '4620' => [
            [ '620', '628', 'LOUSADA (SO MIGUEL)' ] ] },
        'Aidos' => {
          '4620' => [
            [ '411', '424', 'NEVOGILDE LSD' ] ] },
        'Ribeiro' => {
          '4620' => [
            [ '383', '397', 'MEINEDO' ] ] },
        'Heido' => {
          '4620' => [
            [ '307', '320', 'MACIEIRA LSD' ] ] },
        'Boua das Cales' => {
          '4620' => [
            [ '353', '381', 'MEINEDO' ] ] },
        'Vinhas' => {
          '4620' => [
            [ '272', '274', 'LUSTOSA' ] ] },
        'Outeirinhos' => {
          '4620' => [
            [ '341', '346', 'MEINEDO' ] ] },
        'Canto' => {
          '4620' => [
            [ '877', '879', 'LUSTOSA' ] ] },
        'Remanga' => {
          '4620' => [
            [ '448', '452', 'NEVOGILDE LSD' ] ] },
        'Rebordelos' => {
          '4620' => [
            [ '286', '293', 'LUSTOSA' ] ] },
        'Alhares' => {
          '4620' => [
            [ '751', '758', 'TORNO' ] ] },
        'Cortinha' => {
          '4620' => [
            [ '677', '683', 'LOUSADA' ] ] },
        'Cavadinha' => {
          '4620' => [
            [ '322', '322', 'MACIEIRA LSD' ] ] },
        'Surribas' => {
          '4620' => [
            [ '860', '875', 'LUSTOSA' ] ] },
        'Caleires' => {
          '4620' => [
            [ '893', '905', 'VILAR DO TORNO E ALENTM' ] ] },
        'Lama de Baixo' => {
          '4620' => [
            [ '013', '023', 'AVELEDA LSD' ] ] },
        'Bitocas' => {
          '4620' => [
            [ '331', '334', 'MEINEDO' ] ] },
        'Bacelo' => {
          '4620' => [
            [ '481', '501', 'PIAS LSD' ] ] },
        'Marecos' => {
          '4620' => [
            [ '170', '175', 'LOUSADA' ] ] },
        'Alto da Ranha' => {
          '4620' => [
            [ '295', '300', 'FIGUEIRAS' ] ] },
        'Sequeir' => {
          '4620' => [
            [ '270', '270', 'LUSTOSA' ] ] },
        'Carreira' => {
          '4620' => [
            [ '160', '169', 'MACIEIRA LSD' ] ] },
        'Romo' => {
          '4620' => [
            [ '069', '069', 'CADE DE REI' ] ] },
        'Vessada' => {
          '4620' => [
            [ '685', '688', 'LOUSADA' ] ] },
        'Bougega' => {
          '4620' => [
            [ '471', '472', 'ORDEM' ] ] },
        'Agrelo' => {
          '4620' => [
            [ '348', '351', 'MEINEDO' ] ] },
        'Calvelo' => {
          '4620' => [
            [ '610', '610', 'LOUSADA (SANTA MARGARIDA)' ] ] },
        'Cerca' => {
          '4620' => [
            [ '091', '099', 'CASAIS LSD' ] ] },
        'Lousada' => {
          '4620' => [
            [ '130', '132', 'LOUSADA' ],
            [ '134', '138', 'LOUSADA' ],
            [ '140', '140', 'LOUSADA' ],
            [ '142', '155', 'LOUSADA' ],
            [ '157', '159', 'LOUSADA' ],
            [ '659', '675', 'LOUSADA' ],
            [ '692', '697', 'LOUSADA' ] ] }
            }
         },
    'Guarda' => {
      'Meda' => {
        'Aveloso' => {
          '6430' => [
            [ '011', '011', 'AVELOSO' ] ] },
        'Meda' => {
          '6430' => [
            [ '101', '120', 'MEDA' ],
            [ '141', '217', 'MEDA' ] ] },
        'Cornalheira' => {
          '6430' => [
            [ '061', '062', 'FONTE LONGA MDA' ] ] },
        'Vale Flor' => {
          '6430' => [
            [ '371', '371', 'VALE FLOR' ] ] },
        'Rabaal' => {
          '6430' => [
            [ '351', '351', 'RABAAL MDA' ] ] },
        'Longroiva' => {
          '6430' => [
            [ '071', '076', 'LONGROIVA' ] ] },
        'Pai Penela' => {
          '6430' => [
            [ '321', '321', 'PAI PENELA' ] ] },
        'Cancelos' => {
          '6430' => [
            [ '331', '337', 'POO DO CANTO' ] ] },
        'Carvalhal' => {
          '6430' => [
            [ '031', '031', 'CARVALHAL MDA' ] ] },
        'Coriscada' => {
          '6430' => [
            [ '051', '051', 'CORISCADA' ] ] },
        'Casteio' => {
          '6430' => [
            [ '041', '042', 'CASTEIO' ] ] },
        'Barreira' => {
          '6430' => [
            [ '021', '022', 'BARREIRA' ] ] },
        'Marialva' => {
          '6430' => [
            [ '081', '081', 'MARIALVA' ] ] },
        'Areola' => {
          '6430' => [
            [ '311', '313', 'OUTEIRO DE GATOS' ] ] },
        'Prova' => {
          '6430' => [
            [ '341', '342', 'PROVA' ] ] },
        'Alcarva' => {
          '6430' => [
            [ '361', '362', 'RANHADOS MDA' ] ] }
          },
      'Pinhel' => {
        'Atalaia' => {
          '6400' => [
            [ '121', '123', 'ATALAIA PNH' ] ] },
        'Argomil' => {
          '6400' => [
            [ '601', '602', 'POMARES PNH' ] ] },
        'Pinhel' => {
          '6400' => [
            [ '311', '315', 'PINHEL' ],
            [ '317', '349', 'PINHEL' ],
            [ '351', '354', 'PINHEL' ],
            [ '356', '373', 'PINHEL' ],
            [ '376', '384', 'PINHEL' ],
            [ '386', '387', 'PINHEL' ],
            [ '389', '404', 'PINHEL' ],
            [ '406', '420', 'PINHEL' ],
            [ '422', '423', 'PINHEL' ],
            [ '425', '427', 'PINHEL' ],
            [ '429', '434', 'PINHEL' ],
            [ '436', '446', 'PINHEL' ],
            [ '448', '453', 'PINHEL' ],
            [ '455', '455', 'PINHEL' ],
            [ '457', '466', 'PINHEL' ],
            [ '511', '511', 'PINHEL' ],
            [ '513', '516', 'PINHEL' ],
            [ '523', '524', 'PINHEL' ],
            [ '526', '526', 'PINHEL' ],
            [ '528', '529', 'PINHEL' ],
            [ '531', '532', 'PINHEL' ] ] },
        'Azvo' => {
          '6400' => [
            [ '141', '143', 'AZEVO' ] ] },
        'Sorval' => {
          '6400' => [
            [ '641', '641', 'SORVAL' ] ] },
        'Barrego' => {
          '6400' => [
            [ '241', '243', 'LAMEIRAS' ] ] },
        'Arco' => {
          '6400' => [
            [ '221', '224', 'GOUVEIA PNH' ] ] },
        'Juizo' => {
          '6400' => [
            [ '145', '146', 'AZEVO' ] ] },
        'Souropires' => {
          '6400' => [
            [ '651', '651', 'SOURO PIRES' ] ] },
        'Alverca da Beira' => {
          '6400' => [
            [ '101', '101', 'ALVERCA DA BEIRA' ] ] },
        'Ervedosa' => {
          '6400' => [
            [ '201', '202', 'ERVEDOSA PNH' ] ] },
        'Cerejo' => {
          '6400' => [
            [ '181', '181', 'CEREJO' ] ] },
        'Abadia' => {
          '6400' => [
            [ '061', '064', 'PNZIO' ] ] },
        'Santa Eufmia' => {
          '6400' => [
            [ '631', '631', 'SANTA EUFMIA PNH' ] ] },
        'Vale de Madeira' => {
          '6400' => [
            [ '671', '671', 'VALE DE MADEIRA' ] ] },
        'Valbom' => {
          '6400' => [
            [ '661', '661', 'VALBOM PNH' ] ] },
        'Gamelas' => {
          '6400' => [
            [ '271', '273', 'PEREIRO PNH' ] ] },
        'Pnzio' => {
          '6400' => [
            [ '069', '069', 'PNZIO' ] ] },
        'Pvoa de El-Rei' => {
          '6400' => [
            [ '611', '611', 'PVOA DE EL-REI' ] ] },
        'Espedrada' => {
          '6400' => [
            [ '211', '215', 'FREIXEDAS' ] ] },
        'Freixinho' => {
          '6400' => [
            [ '231', '234', 'LAMEGAL' ] ] },
        'Pala' => {
          '6400' => [
            [ '261', '263', 'PALA PNH' ] ] },
        'Cidadelhe' => {
          '6400' => [
            [ '191', '191', 'CIDADELHE PNH' ] ] },
        'Quint dos Bernardos' => {
          '6400' => [
            [ '551', '552', 'PINHEL' ] ] },
        'Manigoto' => {
          '6400' => [
            [ '251', '251', 'MANIGOTO' ] ] },
        'Vascoveiro' => {
          '6400' => [
            [ '681', '681', 'VASCOVEIRO' ] ] },
        'Boua Cova' => {
          '6400' => [
            [ '171', '171', 'BOUA COVA' ] ] },
        'Safurdo' => {
          '6400' => [
            [ '621', '621', 'SAFURDO' ] ] },
        'Malta' => {
          '6400' => [
            [ '301', '301', 'PINHEL' ] ] },
        'Bogalhal' => {
          '6400' => [
            [ '161', '161', 'BOGALHAL' ] ] },
        'Ervas Tenras' => {
          '6400' => [
            [ '192', '192', 'ERVAS TENRAS' ] ] }
            },
      'Gouveia' => {
        'So Paio' => {
          '6290' => [
            [ '411', '417', 'SO PAIO GVA' ] ] },
        'Lagarinhos' => {
          '6290' => [
            [ '091', '094', 'LAGARINHOS' ] ] },
        'Vinh' => {
          '6290' => [
            [ '651', '651', 'VINH' ] ] },
        'Figueir da Serra' => {
          '6290' => [
            [ '071', '071', 'FIGUEIR DA SERRA' ] ] },
        'Melo' => {
          '6290' => [
            [ '121', '123', 'MELO' ] ] },
        'Vila Nova de Tazem' => {
          '6290' => [
            [ '637', '637', 'VILA NOVA DE TAZEM' ] ] },
        'Tazem' => {
          '6290' => [
            [ '631', '635', 'VILA NOVA DE TAZEM' ] ] },
        'Nabais' => {
          '6290' => [
            [ '151', '151', 'NABAIS' ] ] },
        'Nespereira' => {
          '6290' => [
            [ '142', '144', 'NESPEREIRA GVA' ],
            [ '201', '210', 'NESPEREIRA GVA' ] ] },
        'Rio Torto' => {
          '6290' => [
            [ '261', '261', 'RIO TORTO GVA' ] ] },
        'Folgosinho' => {
          '6290' => [
            [ '081', '081', 'FOLGOSINHO' ] ] },
        'Cativelos' => {
          '6290' => [
            [ '061', '063', 'CATIVELOS' ] ] },
        'Freixo da Serra' => {
          '6290' => [
            [ '082', '082', 'FREIXO DA SERRA' ] ] },
        'Paos da Serra' => {
          '6290' => [
            [ '241', '242', 'PAOS DA SERRA' ] ] },
        'Ponte Nova' => {
          '6290' => [
            [ '621', '622', 'VILA FRANCA DA SERRA' ] ] },
        'Ribamondego' => {
          '6290' => [
            [ '251', '251', 'RIBAMONDEGO' ] ] },
        'Mangualde da Serra' => {
          '6290' => [
            [ '111', '111', 'MANGUALDE DA SERRA' ] ] },
        'Moimenta da Serra' => {
          '6290' => [
            [ '141', '141', 'MOIMENTA DA SERRA' ] ] },
        'Gouveia' => {
          '6290' => [
            [ '306', '376', 'GOUVEIA' ],
            [ '511', '556', 'GOUVEIA' ],
            [ '652', '653', 'GOUVEIA' ] ] },
        'Vila Cortez da Serra' => {
          '6290' => [
            [ '611', '611', 'VILA CORTES DA SERRA' ] ] },
        'Arcozelo da Serra' => {
          '6290' => [
            [ '051', '051', 'ARCOZELO GVA' ] ] },
        'Aldeias' => {
          '6290' => [
            [ '012', '013', 'ALDEIAS GVA' ] ] }
             },
      'Sabugal' => {
        'Ruivs' => {
          '6320' => [
            [ '291', '291', 'RUIVS' ] ] },
        'Alfaiates' => {
          '6320' => [
            [ '081', '081', 'ALFAIATES' ] ] },
        'Quinta da Corredoura' => {
          '6320' => [
            [ '530', '534', 'SORTELHA' ] ] },
        'Cerdeira' => {
          '6320' => [
            [ '131', '132', 'CERDEIRA SBG' ] ],
          '6324' => [
            [ '001', '001', 'CERDEIRA SBG' ] ] },
        'Aldeia do Bispo' => {
          '6320' => [
            [ '021', '021', 'ALDEIA DO BISPO SBG' ] ] },
        'Ribeiro Salgado' => {
          '6320' => [
            [ '560', '561', 'VALE DE ESPINHO' ] ] },
        'Quinta da Costa' => {
          '6320' => [
            [ '280', '283', 'RENDO' ] ],
          '6324' => [
            [ '004', '004', 'RENDO' ] ] },
        'Rebolosa' => {
          '6320' => [
            [ '271', '271', 'REBOLOSA' ] ] },
        'Vale das guas' => {
          '6320' => [
            [ '551', '551', 'VALE DAS GUAS' ] ] },
        'Aldeia da Dona' => {
          '6320' => [
            [ '211', '212', 'NAVE' ] ] },
        'Quinta de Santo Amaro' => {
          '6320' => [
            [ '125', '125', 'CASTELEIRO' ] ] },
        'Foios' => {
          '6320' => [
            [ '141', '141', 'FOIOS' ] ] },
        'Santo Estvo' => {
          '6320' => [
            [ '511', '511', 'SANTO ESTEVO SBG' ] ] },
        'Ruvina' => {
          '6320' => [
            [ '301', '301', 'RUVINA' ] ] },
        'Sabugal' => {
          '6320' => [
            [ '311', '312', 'SABUGAL' ],
            [ '315', '315', 'SABUGAL' ],
            [ '317', '318', 'SABUGAL' ],
            [ '326', '328', 'SABUGAL' ],
            [ '330', '330', 'SABUGAL' ],
            [ '332', '333', 'SABUGAL' ],
            [ '336', '336', 'SABUGAL' ],
            [ '338', '339', 'SABUGAL' ],
            [ '342', '345', 'SABUGAL' ],
            [ '347', '347', 'SABUGAL' ],
            [ '349', '356', 'SABUGAL' ],
            [ '358', '365', 'SABUGAL' ],
            [ '368', '373', 'SABUGAL' ],
            [ '376', '378', 'SABUGAL' ],
            [ '380', '386', 'SABUGAL' ],
            [ '389', '392', 'SABUGAL' ],
            [ '394', '394', 'SABUGAL' ],
            [ '399', '402', 'SABUGAL' ],
            [ '406', '411', 'SABUGAL' ],
            [ '413', '417', 'SABUGAL' ],
            [ '420', '422', 'SABUGAL' ],
            [ '424', '427', 'SABUGAL' ],
            [ '432', '432', 'SABUGAL' ],
            [ '441', '459', 'SABUGAL' ],
            [ '501', '506', 'SABUGAL' ],
            [ '508', '508', 'SABUGAL' ] ],
          '6324' => [
            [ '002', '003', 'SABUGAL' ],
            [ '005', '010', 'SABUGAL' ] ] },
        'Souto' => {
          '6320' => [
            [ '170', '171', 'LOMBA SBG' ] ] },
        'Quintas de So Bartolomeu' => {
          '6320' => [
            [ '251', '251', 'QUINTAS DE SO BARTOLOMEU' ] ] },
        'Vilar Maior' => {
          '6320' => [
            [ '601', '602', 'VILAR MAIOR' ] ] },
        'Aldeia Velha' => {
          '6320' => [
            [ '069', '069', 'ALDEIA VELHA SBG' ] ] },
        'Moita' => {
          '6320' => [
            [ '201', '202', 'MOITA SBG' ] ] },
        'Stio do Marco' => {
          '6320' => [
            [ '240', '242', 'QUADRAZAIS' ] ] },
        'Rapoula do Ca' => {
          '6320' => [
            [ '261', '261', 'RAPOULA DO COA' ] ] },
        'Bismula' => {
          '6320' => [
            [ '111', '111', 'BISMULA' ] ] },
        'Malcata' => {
          '6320' => [
            [ '181', '181', 'MALCATA' ] ] },
        'Vila Boa' => {
          '6320' => [
            [ '581', '581', 'VILA BOA SBG' ] ] },
        'Torre' => {
          '6320' => [
            [ '471', '473', 'SABUGAL' ],
            [ '475', '480', 'SABUGAL' ],
            [ '482', '486', 'SABUGAL' ],
            [ '488', '492', 'SABUGAL' ],
            [ '494', '499', 'SABUGAL' ] ] },
        'Bendada' => {
          '6250' => [
            [ '181', '191', 'BENDADA' ] ] },
        'Semideiro' => {
          '6320' => [
            [ '220', '221', 'PENA LOBO' ] ] },
        'Forcalhos' => {
          '6320' => [
            [ '151', '151', 'FORCALHOS' ] ] },
        'guas Belas' => {
          '6320' => [
            [ '011', '014', 'GUAS BELAS SBG' ] ] },
        'Espadanal' => {
          '6320' => [
            [ '620', '626', 'SOITO' ] ] },
        'Abitureira' => {
          '6320' => [
            [ '591', '592', 'VILA DO TOURO' ] ] },
        'Laigitas' => {
          '6320' => [
            [ '160', '161', 'LAJEOSA SBG' ] ] },
        'Vale de So Nicolau' => {
          '6320' => [
            [ '230', '236', 'POUSAFOLES DO BISPO' ] ] },
        'Casteleiro' => {
          '6320' => [
            [ '121', '121', 'CASTELEIRO' ] ] },
        'Covadas' => {
          '6320' => [
            [ '100', '103', 'BARAAL SBG' ] ] },
        'Aldeia da Ribeira' => {
          '6320' => [
            [ '041', '043', 'ALDEIA DA RIBEIRA' ] ] },
        'Soito' => {
          '6320' => [
            [ '628', '707', 'SOITO' ] ] },
        'Aldeia da Ponte' => {
          '6320' => [
            [ '031', '031', 'ALDEIA DA PONTE' ] ] },
        'Badamalos' => {
          '6320' => [
            [ '091', '092', 'BADAMALOS' ] ] },
        'Martim da Pega' => {
          '6320' => [
            [ '521', '523', 'SEIXO DO COA' ] ] },
        'Aldeia de Santo Antnio' => {
          '6320' => [
            [ '050', '054', 'ALDEIA DE SANTO ANTNIO' ] ] },
        'Sortelha' => {
          '6320' => [
            [ '536', '537', 'SORTELHA' ] ] },
        'Colnia Agrcola Martin Rei' => {
          '6320' => [
            [ '461', '461', 'SABUGAL' ] ] },
        'Vale Longo' => {
          '6320' => [
            [ '571', '571', 'VALE LONGO' ] ] }
             },
      'Seia' => {
        'Aldeia da Serra' => {
          '6270' => [
            [ '371', '373', 'SEIA' ] ] },
        'Pvoa Nova' => {
          '6270' => [
            [ '221', '222', 'SO MARTINHO SEI' ] ] },
        'Carvalhal da Loua' => {
          '6270' => [
            [ '131', '134', 'PARANHOS SEI' ] ] },
        'Cabea' => {
          '6270' => [
            [ '021', '021', 'CABEA' ] ] },
        'Maceirinha' => {
          '6270' => [
            [ '191', '192', 'SANTA EULLIA SEI' ] ] },
        'Varzea de Meruje' => {
          '6270' => [
            [ '631', '631', 'VRZEA DE MERUGE' ] ] },
        'Folgosa da Madalena' => {
          '6270' => [
            [ '211', '213', 'SANTIAGO SEI' ] ] },
        'Arcozelo de Varzea' => {
          '6270' => [
            [ '040', '041', 'FOLHADOSA' ] ] },
        'Bairro Novo' => {
          '6270' => [
            [ '581', '588', 'TOURAIS' ] ] },
        'Sameice' => {
          '6270' => [
            [ '161', '161', 'SAMEICE' ] ] },
        'Eir' => {
          '6270' => [
            [ '195', '196', 'SANTA MARINHA SEI' ] ] },
        'So Romo' => {
          '6270' => [
            [ '251', '330', 'SO ROMO SEI' ] ] },
        'Pinhanos' => {
          '6270' => [
            [ '141', '141', 'PINHANOS' ] ] },
        'Santa Comba' => {
          '6270' => [
            [ '184', '186', 'SANTA COMBA SEI' ] ] },
        'Lapa dos Dinheiros' => {
          '6270' => [
            [ '651', '651', 'LAPA DOS DINHEIROS' ] ] },
        'Vide' => {
          '6285' => [
            [ '101', '130', 'VIDE' ] ] },
        'Fonto' => {
          '6270' => [
            [ '071', '104', 'LORIGA' ] ] },
        'Torrozelo' => {
          '6270' => [
            [ '555', '555', 'TORROZELO' ] ] },
        'Carragozela' => {
          '6270' => [
            [ '031', '031', 'CARRAGOZELA' ] ] },
        'Travancinha' => {
          '6270' => [
            [ '601', '606', 'TRAVANCINHA' ] ] },
        'Aldeia de So Miguel' => {
          '6270' => [
            [ '181', '181', 'SANTA COMBA SEI' ] ] },
        'Girabolhos' => {
          '6270' => [
            [ '051', '052', 'GIRABOLHOS' ] ] },
        'Cabea de Eiras' => {
          '6270' => [
            [ '171', '174', 'SANDOMIL' ] ] },
        'Sabugueiro' => {
          '6270' => [
            [ '151', '151', 'SABUGUEIRO SEI' ] ] },
        'Valezim' => {
          '6270' => [
            [ '621', '621', 'VALEZIM' ] ] },
        'Lages' => {
          '6270' => [
            [ '061', '061', 'LAJES SEI' ] ] },
        'Sazes da Beira' => {
          '6270' => [
            [ '351', '352', 'SAZES DA BEIRA' ] ] },
        'Abitureira' => {
          '6285' => [
            [ '053', '075', 'VIDE' ] ] },
        'Aguincho' => {
          '6270' => [
            [ '011', '015', 'ALVOCO DA SERRA' ] ] },
        'Seia' => {
          '6270' => [
            [ '401', '402', 'SEIA' ],
            [ '404', '430', 'SEIA' ],
            [ '432', '541', 'SEIA' ] ] },
        'Teixeira' => {
          '6285' => [
            [ '051', '052', 'TEIXEIRA SEI' ] ] },
        'Vales' => {
          '6270' => [
            [ '551', '554', 'SEIA' ] ] },
        'Barreiro' => {
          '6270' => [
            [ '641', '644', 'VILA COVA  COELHEIRA SEI' ] ] },
        'Loriga' => {
          '6270' => [
            [ '106', '124', 'LORIGA' ] ] }
          },
      'Fornos de Algodres' => {
        'Maceira' => {
          '6370' => [
            [ '341', '341', 'MACEIRA FAG' ] ] },
        'Figueir da Granja' => {
          '6370' => [
            [ '041', '041', 'FIGUEIR DA GRANJA' ] ] },
        'Vila Ch' => {
          '6370' => [
            [ '391', '391', 'VILA CH FAG' ] ] },
        'Fornos de Algodres' => {
          '6370' => [
            [ '101', '106', 'FORNOS DE ALGODRES' ],
            [ '108', '149', 'FORNOS DE ALGODRES' ],
            [ '151', '152', 'FORNOS DE ALGODRES' ],
            [ '154', '163', 'FORNOS DE ALGODRES' ],
            [ '165', '165', 'FORNOS DE ALGODRES' ],
            [ '167', '179', 'FORNOS DE ALGODRES' ],
            [ '183', '183', 'FORNOS DE ALGODRES' ],
            [ '187', '189', 'FORNOS DE ALGODRES' ],
            [ '195', '197', 'FORNOS DE ALGODRES' ],
            [ '199', '220', 'FORNOS DE ALGODRES' ],
            [ '225', '226', 'FORNOS DE ALGODRES' ] ] },
        'Infias' => {
          '6370' => [
            [ '321', '321', 'INFIAS FAG' ] ] },
        'Vila Ruiva' => {
          '6370' => [
            [ '401', '401', 'VILA RUIVA FAG' ] ] },
        'Vila Soeiro do Cho' => {
          '6370' => [
            [ '411', '411', 'VILA SOEIRO DO CHO' ] ] },
        'Fuinhas' => {
          '6370' => [
            [ '311', '311', 'FUINHAS' ] ] },
        'Mata' => {
          '6370' => [
            [ '381', '382', 'SOBRAL PICHORRO' ] ] },
        'Cadoio' => {
          '6370' => [
            [ '331', '333', 'JUNCAIS' ] ] },
        'Algodres' => {
          '6370' => [
            [ '011', '014', 'ALGODRES FAG' ] ] },
        'Muxagata' => {
          '6370' => [
            [ '361', '361', 'MUXAGATA FAG' ] ] },
        'Furtado' => {
          '6370' => [
            [ '228', '234', 'FORNOS DE ALGODRES' ] ] },
        'Casal Vasco' => {
          '6370' => [
            [ '021', '022', 'CASAL VASCO' ] ] },
        'Fonte Fria' => {
          '6370' => [
            [ '351', '353', 'MATANA' ] ] },
        'Aveleiras' => {
          '6370' => [
            [ '371', '374', 'QUEIRIZ' ] ] },
        'Corti' => {
          '6370' => [
            [ '031', '031', 'CORTI' ] ] }
            },
      'Vila Nova de Foz Coa' => {
        'Orgal' => {
          '5150' => [
            [ '145', '145', 'CASTELO MELHOR' ] ] },
        'Cortes da Veiga' => {
          '5150' => [
            [ '501', '504', 'VILA NOVA DE FOZ CA' ] ] },
        'Horta' => {
          '5155' => [
            [ '301', '343', 'HORTA VLF' ] ] },
        'Cedovim' => {
          '5155' => [
            [ '006', '059', 'CEDOVIM' ] ] },
        'Sebadelhe' => {
          '5155' => [
            [ '701', '727', 'SEBADELHE' ] ] },
        'Freixo' => {
          '5155' => [
            [ '401', '432', 'MS VLF' ] ] },
        'Castelo Melhor' => {
          '5150' => [
            [ '101', '103', 'CASTELO MELHOR' ],
            [ '105', '141', 'CASTELO MELHOR' ] ] },
        'Almendra' => {
          '5150' => [
            [ '011', '086', 'ALMENDRA' ] ] },
        'Mura' => {
          '5155' => [
            [ '501', '528', 'MURA VLF' ] ] },
        'Tomadias' => {
          '5150' => [
            [ '450', '450', 'SANTA COMBA VLF' ] ] },
        'Toua' => {
          '5155' => [
            [ '797', '825', 'TOUA' ] ] },
        'Arnozelo' => {
          '5155' => [
            [ '601', '602', 'NUMO' ] ] },
        'Numo' => {
          '5155' => [
            [ '610', '641', 'NUMO' ] ] },
        'Seixas do Douro' => {
          '5155' => [
            [ '745', '779', 'SEIXAS VLF' ] ] },
        'Muxagata' => {
          '5150' => [
            [ '301', '337', 'MUXAGATA VLF' ] ] },
        'Santa Comba' => {
          '5150' => [
            [ '401', '437', 'SANTA COMBA VLF' ] ] },
        'Quinta do Saio' => {
          '5155' => [
            [ '651', '679', 'SANTO AMARO VLF' ] ] },
        'Custias' => {
          '5155' => [
            [ '100', '129', 'CUSTIAS VLF' ] ] },
        'Vila Nova de Foz Ca' => {
          '5150' => [
            [ '520', '567', 'VILA NOVA DE FOZ CA' ],
            [ '569', '569', 'VILA NOVA DE FOZ CA' ],
            [ '571', '583', 'VILA NOVA DE FOZ CA' ],
            [ '586', '707', 'VILA NOVA DE FOZ CA' ],
            [ '709', '719', 'VILA NOVA DE FOZ CA' ] ] },
        'Chs' => {
          '5150' => [
            [ '201', '244', 'CHS' ] ] },
        'Freixo de Numo' => {
          '5155' => [
            [ '201', '264', 'FREIXO DE NUMO' ] ] }
              },
      'Aguiar da Beira' => {
        'Ponte do Abade' => {
          '3570' => [
            [ '191', '192', 'SEQUEIROS AGB' ] ] },
        'Picoto' => {
          '3570' => [
            [ '130', '130', 'DORNELAS AGB' ] ] },
        'Quinta dos Matos' => {
          '3570' => [
            [ '180', '180', 'PINHEIRO AGB' ] ] },
        'Cortiada' => {
          '3570' => [
            [ '110', '110', 'CORTIADA' ] ] },
        'Coja' => {
          '3570' => [
            [ '071', '076', 'AGUIAR DA BEIRA' ] ] },
        'Aguiar da Beira' => {
          '3570' => [
            [ '010', '049', 'AGUIAR DA BEIRA' ] ] },
        'Aores' => {
          '3570' => [
            [ '001', '001', 'AGUIAR DA BEIRA' ] ] },
        'Gradiz' => {
          '3570' => [
            [ '160', '160', 'GRADIZ' ] ] },
        'Monhos' => {
          '3570' => [
            [ '120', '120', 'CORUCHE AGB' ] ] },
        'Souto de Aguiar da Beira' => {
          '3570' => [
            [ '200', '200', 'SOUTO DE AGUIAR DA BEIRA' ] ] },
        'Antela' => {
          '3570' => [
            [ '140', '140', 'EIRADO' ] ] },
        'Urgueira' => {
          '3570' => [
            [ '170', '170', 'PENA VERDE' ] ] },
        'Barraco' => {
          '3570' => [
            [ '211', '212', 'VALVERDE AGB' ] ] },
        'Valagotes' => {
          '3570' => [
            [ '150', '150', 'FORNINHOS' ] ] },
        'Carapito' => {
          '3570' => [
            [ '100', '100', 'CARAPITO' ] ] }
               },
      'Celorico da Beira' => {
        'Casas de Soeiro' => {
          '6360' => [
            [ '220', '221', 'CASAS DO SOEIRO' ] ] },
        'Maal do Cho' => {
          '6360' => [
            [ '090', '090', 'MAAL DO CHO' ] ] },
        'Baraal' => {
          '6360' => [
            [ '020', '022', 'BARAAL CLB' ] ] },
        'Porteira' => {
          '6360' => [
            [ '052', '054', 'CORTI DA SERRA' ] ] },
        'Vide Entre Vinhas' => {
          '6360' => [
            [ '200', '202', 'VIDE ENTRE VINHAS' ] ] },
        'Prados' => {
          '6360' => [
            [ '120', '120', 'PRADOS' ] ] },
        'Minhocal' => {
          '6360' => [
            [ '110', '113', 'MINHOCAL' ] ] },
        'Quinta do Chafariz' => {
          '6360' => [
            [ '355', '371', 'CELORICO DA BEIRA' ] ] },
        'Carrapichana' => {
          '6360' => [
            [ '040', '040', 'CARRAPICHANA' ] ] },
        'Salgueirais' => {
          '6360' => [
            [ '150', '150', 'SALGUEIRAIS' ] ] },
        'Quinta das Poldras' => {
          '6360' => [
            [ '354', '354', 'FORNO TELHEIRO' ] ] },
        'Ratoeira' => {
          '6360' => [
            [ '140', '141', 'RATOEIRA' ] ] },
        'Velosa' => {
          '6360' => [
            [ '190', '190', 'VELOSA' ] ] },
        'Lajeosa do Mondego' => {
          '6360' => [
            [ '492', '532', 'LAJEOSA DO MONDEGO' ] ] },
        'Linhares da Beira' => {
          '6360' => [
            [ '080', '083', 'LINHARES CLB' ] ] },
        'Rapa' => {
          '6360' => [
            [ '130', '130', 'RAPA' ] ] },
        'Vale de Azares (Grichoso)' => {
          '6360' => [
            [ '180', '184', 'VALE DE AZARES' ] ] },
        'Mesquitela' => {
          '6360' => [
            [ '100', '103', 'MESQUITELA CLB' ] ] },
        'Aores' => {
          '6360' => [
            [ '010', '013', 'AORES' ] ] },
        'Celorico da Beira' => {
          '6360' => [
            [ '287', '298', 'CELORICO DA BEIRA' ],
            [ '300', '312', 'CELORICO DA BEIRA' ],
            [ '314', '319', 'CELORICO DA BEIRA' ],
            [ '323', '333', 'CELORICO DA BEIRA' ],
            [ '335', '348', 'CELORICO DA BEIRA' ],
            [ '350', '353', 'CELORICO DA BEIRA' ] ] },
        'Forno Telheiro' => {
          '6360' => [
            [ '060', '067', 'FORNO TELHEIRO' ] ] },
        'Cadafaz' => {
          '6360' => [
            [ '030', '031', 'CADAFAZ CLB' ] ] },
        'Vila Boa do Mondego' => {
          '6360' => [
            [ '210', '211', 'VILA BOA DO MONDEGO' ] ] },
        'Mourela' => {
          '6360' => [
            [ '050', '050', 'CORTI DA SERRA' ] ] }
                 },
      'Trancoso' => {
        'Barrocal' => {
          '6420' => [
            [ '371', '372', 'FIES TCS' ] ] },
        'Souto Maior' => {
          '6420' => [
            [ '613', '615', 'SOUTO MAIOR TCS' ] ] },
        'Terrenho' => {
          '6420' => [
            [ '631', '631', 'TERRENHO' ] ] },
        'Amial' => {
          '6420' => [
            [ '581', '583', 'TRANCOSO' ] ] },
        'Reboleiro' => {
          '6420' => [
            [ '541', '541', 'REBOLEIRO' ] ] },
        'Boco' => {
          '6420' => [
            [ '571', '579', 'TRANCOSO' ] ] },
        'Broca' => {
          '6420' => [
            [ '791', '793', 'VILARES TCS' ] ] },
        'Castanheira' => {
          '6420' => [
            [ '331', '331', 'CASTANHEIRA TCS' ] ] },
        'Alcudra' => {
          '6420' => [
            [ '301', '310', 'ALDEIA NOVA TCS' ] ] },
        'Courelas' => {
          '6420' => [
            [ '586', '589', 'TRANCOSO' ] ] },
        'Freches' => {
          '6420' => [
            [ '401', '448', 'FRECHES' ] ] },
        'Freixial' => {
          '6420' => [
            [ '781', '782', 'VILA GARCIA TCS' ] ] },
        'Carigas' => {
          '6420' => [
            [ '671', '673', 'VALE DO SEIXO' ] ] },
        'Granja' => {
          '6420' => [
            [ '471', '473', 'GRANJA TCS' ] ] },
        'Moitas' => {
          '6420' => [
            [ '661', '662', 'VALDUJO' ] ] },
        'Vila Franca das Naves' => {
          '6420' => [
            [ '692', '700', 'VILA FRANCA DAS NAVES' ],
            [ '702', '743', 'VILA FRANCA DAS NAVES' ],
            [ '746', '749', 'VILA FRANCA DAS NAVES' ] ] },
        'Coras' => {
          '6420' => [
            [ '611', '612', 'SEBADELHE DA SERRA' ] ] },
        'Botocais' => {
          '6420' => [
            [ '551', '553', 'RIO DE MEL' ] ] },
        'A-do-Cavalo' => {
          '6420' => [
            [ '501', '511', 'MOREIRA DE REI' ] ] },
        'Chafariz do Vento' => {
          '6420' => [
            [ '651', '654', 'TORRES' ] ] },
        'Falachos' => {
          '6420' => [
            [ '621', '624', 'TAMANHOS' ] ] },
        'Moimentinha' => {
          '6420' => [
            [ '491', '491', 'MOIMENTINHA' ] ] },
        'Cogula' => {
          '6420' => [
            [ '341', '342', 'COGULA' ] ] },
        'Guilheiro' => {
          '3640' => [
            [ '700', '700', 'GUILHEIRO' ] ] },
        'Mendo Gordo' => {
          '6420' => [
            [ '641', '642', 'TORRE DO TERRENHO' ] ] },
        'A-dos-Ferreiros' => {
          '6420' => [
            [ '351', '351', 'CTIMOS' ] ] },
        'Pvoa do Concelho' => {
          '6420' => [
            [ '531', '531', 'PVOA DO CONCELHO' ] ] },
        'Trancoso' => {
          '6420' => [
            [ '001', '065', 'TRANCOSO' ],
            [ '067', '074', 'TRANCOSO' ],
            [ '076', '083', 'TRANCOSO' ],
            [ '085', '102', 'TRANCOSO' ],
            [ '104', '121', 'TRANCOSO' ],
            [ '123', '123', 'TRANCOSO' ],
            [ '125', '134', 'TRANCOSO' ],
            [ '136', '138', 'TRANCOSO' ],
            [ '140', '141', 'TRANCOSO' ],
            [ '144', '146', 'TRANCOSO' ],
            [ '150', '150', 'TRANCOSO' ],
            [ '152', '153', 'TRANCOSO' ],
            [ '156', '159', 'TRANCOSO' ],
            [ '164', '168', 'TRANCOSO' ],
            [ '170', '173', 'TRANCOSO' ],
            [ '177', '184', 'TRANCOSO' ],
            [ '186', '246', 'TRANCOSO' ] ] },
        'Quinta dos Palheiros' => {
          '6420' => [
            [ '591', '593', 'TRANCOSO' ] ] },
        'Feital' => {
          '6420' => [
            [ '361', '362', 'FEITAL' ] ] },
        'Carnices' => {
          '6420' => [
            [ '321', '321', 'CARNICES' ] ] },
        'Benvende' => {
          '6420' => [
            [ '521', '522', 'PALHAIS TCS' ] ] }
              },
      'Guarda' => {
        'Amial' => {
          '6300' => [
            [ '160', '160', 'PRA DO MOO' ] ] },
        'Aldeia do Bispo' => {
          '6300' => [
            [ '020', '020', 'ALDEIA DO BISPO GRD' ] ] },
        'Cairro' => {
          '6300' => [
            [ '265', '265', 'VILA GARCIA GRD' ] ] },
        'Gagos' => {
          '6300' => [
            [ '110', '110', 'GAGOS GRD' ] ] },
        'Cabriz' => {
          '6300' => [
            [ '245', '245', 'VIDEMONTE' ] ] },
        'Albardo' => {
          '6300' => [
            [ '015', '015', 'ALBARDO' ] ] },
        'Trinta' => {
          '6300' => [
            [ '225', '225', 'TRINTA' ] ] },
        'Balsemo' => {
          '6300' => [
            [ '061', '069', 'JOO ANTO' ] ] },
        'Penedo da S' => {
          '6300' => [
            [ '130', '130', 'MARMELEIRO GRD' ] ] },
        'Ado' => {
          '6300' => [
            [ '010', '010', 'ADO' ] ] },
        'Carvalhal Meo' => {
          '6300' => [
            [ '055', '055', 'CARVALHAL MEO' ] ] },
        'Corujeira' => {
          '6300' => [
            [ '090', '090', 'CORUJEIRA' ] ] },
        'Avels de Ambom' => {
          '6300' => [
            [ '040', '040', 'AVELS DE AMBOM' ] ] },
        'Vila Corts do Mondego' => {
          '6301' => [
            [ '859', '859', 'VILA CORTS DO MONDEGO' ] ],
          '6300' => [
            [ '250', '250', 'VILA CORTS DO MONDEGO' ] ] },
        'Rocamondo' => {
          '6300' => [
            [ '190', '190', 'ROCAMONDO' ] ] },
        'Beta Azul' => {
          '6300' => [
            [ '126', '128', 'MAAINHAS DE BAIXO' ] ] },
        'Amoreiras' => {
          '6300' => [
            [ '220', '220', 'SOBRAL DA SERRA' ] ] },
        'Aldeia Nova' => {
          '6300' => [
            [ '181', '182', 'RAMELA' ] ] },
        'Bogacha' => {
          '6300' => [
            [ '241', '244', 'VELA' ] ] },
        'Pro Soares' => {
          '6300' => [
            [ '165', '165', 'PRO SOARES' ] ] },
        'Pousade' => {
          '6300' => [
            [ '175', '175', 'POUSADA GRD' ] ] },
        'Ferno Joanes' => {
          '6300' => [
            [ '105', '105', 'FERNO JOANES' ] ] },
        'Guarda' => {
          '6301' => [
            [ '851', '858', 'GUARDA' ],
            [ '860', '861', 'GUARDA' ] ],
          '6300' => [
            [ '300', '308', 'GUARDA' ],
            [ '310', '373', 'GUARDA' ],
            [ '375', '417', 'GUARDA' ],
            [ '501', '504', 'GUARDA' ],
            [ '506', '513', 'GUARDA' ],
            [ '515', '516', 'GUARDA' ],
            [ '518', '544', 'GUARDA' ],
            [ '546', '566', 'GUARDA' ],
            [ '568', '575', 'GUARDA' ],
            [ '577', '577', 'GUARDA' ],
            [ '579', '586', 'GUARDA' ],
            [ '588', '622', 'GUARDA' ],
            [ '624', '625', 'GUARDA' ],
            [ '651', '657', 'GUARDA' ],
            [ '659', '659', 'GUARDA' ],
            [ '661', '685', 'GUARDA' ],
            [ '687', '688', 'GUARDA' ],
            [ '690', '691', 'GUARDA' ],
            [ '693', '696', 'GUARDA' ],
            [ '698', '717', 'GUARDA' ],
            [ '719', '721', 'GUARDA' ],
            [ '723', '731', 'GUARDA' ],
            [ '733', '759', 'GUARDA' ],
            [ '761', '780', 'GUARDA' ],
            [ '801', '806', 'GUARDA' ],
            [ '808', '812', 'GUARDA' ],
            [ '814', '821', 'GUARDA' ],
            [ '823', '826', 'GUARDA' ],
            [ '828', '859', 'GUARDA' ],
            [ '861', '870', 'GUARDA' ],
            [ '872', '876', 'GUARDA' ],
            [ '879', '882', 'GUARDA' ] ] },
        'Aldeia Viosa' => {
          '6300' => [
            [ '025', '025', 'ALDEIA VIOSA' ] ] },
        'Barreiros' => {
          '6300' => [
            [ '235', '235', 'VALHELHAS' ] ] },
        'Barraco' => {
          '6300' => [
            [ '151', '152', 'PANOIAS DE CIMA' ] ] },
        'Alvendre' => {
          '6300' => [
            [ '030', '030', 'ALVENDRE' ] ] },
        'Gata' => {
          '6300' => [
           