# pylint: disable=line-too-long,useless-suppression
# ------------------------------------
# Copyright (c) Microsoft Corporation.
# Licensed under the MIT License.
# ------------------------------------

import pytest
from azure.ai.projects import AIProjectClient
from azure.ai.projects.models import AzureAISearchIndex, IndexType
from test_base import TestBase, servicePreparer
from devtools_testutils import recorded_by_proxy


@pytest.mark.skip(
    reason="Skipped until re-enabled and recorded on Foundry endpoint that supports the new versioning schema"
)
class TestIndexes(TestBase):

    # To run this test, use the following command in the \sdk\ai\azure-ai-projects folder:
    # cls & pytest tests\test_indexes.py::TestIndexes::test_indexes -s
    @servicePreparer()
    @recorded_by_proxy
    def test_indexes(self, **kwargs):

        index_name = self.test_indexes_params["index_name"]
        index_version = self.test_indexes_params["index_version"]
        ai_search_connection_name = self.test_indexes_params["ai_search_connection_name"]
        ai_search_index_name = self.test_indexes_params["ai_search_index_name"]

        with self.create_client(**kwargs) as project_client:

            print(
                f"[test_indexes] Create Index `{index_name}` with version `{index_version}`, referencing an existing AI Search resource:"
            )
            index = project_client.indexes.create_or_update(
                name=index_name,
                version=index_version,
                index=AzureAISearchIndex(connection_name=ai_search_connection_name, index_name=ai_search_index_name),
            )
            print(index)
            TestBase.validate_index(
                index,
                expected_index_type=IndexType.AZURE_SEARCH,
                expected_index_name=index_name,
                expected_index_version=index_version,
                expected_ai_search_connection_name=ai_search_connection_name,
                expected_ai_search_index_name=ai_search_index_name,
            )

            print(f"[test_indexes] Get Index `{index_name}` version `{index_version}`:")
            index = project_client.indexes.get(name=index_name, version=index_version)
            print(index)
            TestBase.validate_index(
                index,
                expected_index_type=IndexType.AZURE_SEARCH,
                expected_index_name=index_name,
                expected_index_version=index_version,
                expected_ai_search_connection_name=ai_search_connection_name,
                expected_ai_search_index_name=ai_search_index_name,
            )

            print("[test_indexes] List latest versions of all Indexes:")
            empty = True
            for index in project_client.indexes.list():
                empty = False
                print(index)
                TestBase.validate_index(index)
            assert not empty

            print(f"[test_indexes] Listing all versions of the Index named `{index_name}`:")
            empty = True
            for index in project_client.indexes.list_versions(name=index_name):
                empty = False
                print(index)
                TestBase.validate_index(index)
            assert not empty

            print(f"[test_indexes] Delete Index `{index_name}` version `{index_version}`.")
            project_client.indexes.delete(name=index_name, version=index_version)

            print(
                f"[test_indexes] Again delete Index `{index_name}` version `{index_version}`. Since it does not exist, the REST API should return 204 (No content). This call should NOT throw an exception."
            )
            project_client.indexes.delete(name=index_name, version=index_version)
